(* IsageoCoq - Tarski_Neutral.thy
Port part of GeoCoq 3.4.0 (https://geocoq.github.io/GeoCoq/)

Version 2.0.0 IsaGeoCoq
Copyright (C) 2021-2025 Roland Coghetto roland.coghetto ( a t ) cafr-msa2p.be

History
Version 1.0.0 IsaGeoCoq
Port part of GeoCoq 3.4.0 (https://geocoq.github.io/GeoCoq/) in Isabelle/Hol (Isabelle2021)
Copyright (C) 2021  Roland Coghetto roland_coghetto (at) hotmail.com

License: LGPL

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*)

theory Tarski_Neutral

imports
  Main

begin

section "Neutral geometry dimensionless"

subsection "Tarski's axiom system for neutral geometry: dimensionless"

locale Tarski_neutral_dimensionless =
  fixes Bet  :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool" ("(_ \<midarrow> _ \<midarrow> _) ")
    and Cong :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool"
    and TPA TPB TPC :: 'p
  assumes cong_pseudo_reflexivity: "\<forall> a b. 

Cong a b b a"

and   cong_inner_transitivity: "\<forall> a b p q r s.

Cong a b p q \<and>
Cong a b r s
\<longrightarrow>
Cong p q r s"

and   cong_identity: "\<forall> a b c.

Cong a b c c
\<longrightarrow>
a = b"

and   segment_construction: "\<forall> a b c q.

\<exists>x. (Bet q a x \<and> Cong a x b c)"

and   five_segment: "\<forall> a b c d a' b' c' d'.

a \<noteq> b \<and>
Bet a b c \<and>
Bet a' b' c'\<and>
Cong a b a' b' \<and>
Cong b c b' c' \<and>
Cong a d a' d' \<and>
Cong b d b' d'
\<longrightarrow>
Cong c d c' d'"

and   between_identity: "\<forall> a b.

Bet a b a
\<longrightarrow>
a = b"

and   inner_pasch: "\<forall> a b c p q.

Bet a p c \<and>
Bet b q c
\<longrightarrow>
(\<exists> x. Bet p x b \<and> Bet q x a)"

and   lower_dim:  "\<not> Bet TPA TPB TPC \<and> \<not> Bet TPB TPC TPA \<and> \<not> Bet TPC TPA TPB"

context Tarski_neutral_dimensionless

begin

subsection "Definitions"

definition OFSC ::
  "['p,'p,'p,'p,'p,'p,'p,'p] \<Rightarrow> bool"
  ("_ _ _ _ OFSC _ _ _ _" [99,99,99,99,99,99,99,99] 50)
  where
    "A B C D OFSC A' B' C' D' 
\<equiv>
Bet A B C \<and>
Bet A' B' C' \<and>
Cong A B A' B' \<and>
Cong B C B' C' \<and>
Cong A D A' D' \<and>
Cong B D B' D'"

definition Cong3 ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool"
  ("_ _ _ Cong3 _ _ _" [99,99,99,99,99,99] 50)
  where
    "A B C Cong3 A' B' C' 
\<equiv>
Cong A B A' B' \<and>
Cong A C A' C' \<and>
Cong B C B' C'"

definition Col ::
  "['p,'p,'p] \<Rightarrow> bool"
  ("Col _ _ _" [99,99,99] 50)
  where
    "Col A B C 
\<equiv>
Bet A B C \<or> Bet B C A \<or> Bet C A B"

definition Bet4 ::
  "['p,'p,'p,'p] \<Rightarrow> bool"
  ("Bet4 _ _ _ _" [99,99,99,99] 50)
  where
    "Bet4 A1 A2 A3 A4 
\<equiv>
Bet A1 A2 A3 \<and>
Bet A2 A3 A4 \<and>
Bet A1 A3 A4 \<and>
Bet A1 A2 A4"

definition BetS ::
  "['p,'p,'p] \<Rightarrow> bool" ("BetS _ _ _" [99,99,99] 50)
  where
    "BetS A B C 
\<equiv>
Bet A B C \<and>
A \<noteq> B \<and>
B \<noteq> C"

(** Definition of the sum of segments.
SumS A B C D E F means that AB + CD = EF. *)

definition SumS ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ _ _ SumS _ _" [99,99,99,99,99,99] 50)
  where
    "A B C D SumS E F 
\<equiv>
\<exists> P Q R. Bet P Q R \<and> Cong P Q A B \<and> Cong Q R C D \<and> Cong P R E F"

definition FSC ::
  "['p,'p,'p,'p,'p,'p,'p,'p] \<Rightarrow> bool"
  ("_ _ _ _ FSC _ _ _ _" [99,99,99,99,99,99,99,99] 50)
  where
    "A B C D FSC A' B' C' D' 
\<equiv>
Col A B C \<and>
A B C Cong3 A' B' C' \<and>
Cong A D A' D' \<and>
Cong B D B' D'"

definition IFSC ::
  "['p,'p,'p,'p,'p,'p,'p,'p] \<Rightarrow> bool"
  ("_ _ _ _ IFSC _ _ _ _" [99,99,99,99,99,99,99,99] 50)
  where
    "A B C D IFSC A' B' C' D' 
\<equiv>
Bet A B C \<and>
Bet A' B' C' \<and>
Cong A C A' C' \<and>
Cong B C B' C' \<and>
Cong A D A' D' \<and>
Cong C D C' D'"

definition Le ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ Le _ _" [99,99,99,99] 50)
  where 
    "A B Le C D 
\<equiv>
\<exists> E. (Bet C E D \<and> Cong A B C E)"

definition Lt ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ Lt _ _" [99,99,99,99] 50)
  where 
    "A B Lt C D 
\<equiv>
A B Le C D \<and> \<not> Cong A B C D"

definition Ge ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _Ge _ _" [99,99,99,99] 50)
  where 
    "A B Ge C D 
\<equiv>
C D Le A B"

definition Gt ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ Gt _ _" [99,99,99,99] 50)
  where 
    "A B Gt C D 
\<equiv>
C D Lt A B"

definition Out ::
  "['p,'p,'p] \<Rightarrow> bool" 
  ("_ Out _ _" [99,99,99] 50)
  where 
    "P Out A B 
\<equiv>
A \<noteq> P \<and>
B \<noteq> P \<and>
(Bet P A B \<or> Bet P B A)"

definition Midpoint ::
  "['p,'p,'p] \<Rightarrow> bool" 
  ("_ Midpoint _ _" [99,99,99] 50)
  where 
    "M Midpoint A B 
\<equiv>
Bet A M B \<and>
Cong A M M B"

definition Per ::
  "['p,'p,'p] \<Rightarrow> bool" 
  ("Per _ _ _" [99,99,99] 50)
  where 
    "Per A B C 
\<equiv>
\<exists> C'. (B Midpoint C C' \<and> Cong A C A C')"

definition PerpAt ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ PerpAt _ _ _ _ " [99,99,99,99,99] 50)
  where 
    "X PerpAt A B C D 
\<equiv>
A \<noteq> B \<and>
C \<noteq> D \<and>
Col X A B \<and>
Col X C D \<and>
(\<forall> U V. ((Col U A B \<and> Col V C D) \<longrightarrow> Per U X V))"

definition Perp ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ Perp _ _" [99,99,99,99] 50)
  where 
    "A B Perp C D 
\<equiv>
\<exists> X::'p. X PerpAt A B C D"

definition Coplanar ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("Coplanar _ _ _ _" [99,99,99,99] 50)
  where 
    "Coplanar A B C D 
\<equiv>
\<exists> X. (Col A B X \<and> Col C D X) \<or>
(Col A C X \<and> Col B D X) \<or>
(Col A D X \<and> Col B C X)"

definition TS ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ TS _ _" [99,99,99,99] 50)
  where 
    "A B TS P Q 
\<equiv>
\<not> Col P A B \<and> \<not> Col Q A B \<and> (\<exists> T::'p. Col T A B \<and> Bet P T Q)"

definition ReflectL ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ ReflectL _ _" [99,99,99,99] 50)
  where 
    "P' P ReflectL A B 
\<equiv>
(\<exists> X. X Midpoint P P' \<and> Col A B X) \<and> (A B Perp P P' \<or> P = P')"

definition Reflect ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ Reflect _ _" [99,99,99,99] 50)
  where 
    "P' P Reflect A B 
\<equiv>
(A \<noteq> B \<and> P' P ReflectL A B) \<or> (A = B \<and> A Midpoint P P')"

definition InAngle ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ InAngle _ _ _" [99,99,99,99] 50)
  where 
    "P InAngle A B C 
\<equiv>
A \<noteq> B \<and> C \<noteq> B \<and> P \<noteq> B \<and>
(\<exists> X. Bet A X C \<and> (X = B \<or> B Out X P))"

definition ParStrict::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ ParStrict _ _" [99,99,99,99] 50)
  where 
    "A B ParStrict C D 
\<equiv>  
Coplanar A B C D \<and> 
\<not> (\<exists> X. Col X A B \<and> Col X C D)"

definition Par::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ Par _ _" [99,99,99,99] 50)
  where 
    "A B Par C D 
\<equiv>
A B ParStrict C D \<or> (A \<noteq> B \<and> C \<noteq> D \<and> Col A C D \<and> Col B C D)"

definition Plg::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("Plg _ _ _ _" [99,99,99,99] 50)
  where 
    "Plg  A B C D 
\<equiv>
(A \<noteq> C \<or> B \<noteq> D) \<and> (\<exists> M. M Midpoint A C \<and> M Midpoint B D)"

definition ParallelogramStrict::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("ParallelogramStrict _ _ _ _" [99,99,99,99] 50)
  where 
    "ParallelogramStrict A B A' B' 
\<equiv>
A A' TS B B' \<and> 
A B Par A' B' \<and> 
Cong A B A' B'"

definition ParallelogramFlat::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("ParallelogramFlat _ _ _ _" [99,99,99,99] 50)
  where
    "ParallelogramFlat A B A' B' 
\<equiv>
Col A B A' \<and> 
Col A B B' \<and>
Cong A B A' B' \<and> 
Cong A B' A' B \<and>
(A \<noteq> A' \<or> B \<noteq> B')"

definition Parallelogram::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("Parallelogram _ _ _ _" [99,99,99,99] 50)
  where 
    "Parallelogram A B A' B' 
\<equiv>
ParallelogramStrict A B A' B' \<or> ParallelogramFlat A B A' B'"

definition Rhombus::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("Rhombus _ _ _ _" [99,99,99,99] 50)
  where 
    "Rhombus A B C D 
\<equiv> 
Plg A B C D \<and> Cong A B B C"

definition Rectangle::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("Rectangle _ _ _ _" [99,99,99,99] 50)
  where 
    "Rectangle A B C D 
\<equiv> 
Plg A B C D \<and> Cong A C B D"

definition Square::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("Square _ _ _ _" [99,99,99,99] 50)
  where 
    "Square A B C D 
\<equiv> 
Rectangle A B C D \<and> Cong A B B C"

definition Kite::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("Kite _ _ _ _" [99,99,99,99] 50)
  where 
    "Kite A B C D 
\<equiv> 
Cong B C C D \<and> Cong D A A B"

definition Lambert::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("Lambert _ _ _ _" [99,99,99,99] 50)
  where 
    "Lambert A B C D 
\<equiv>
A \<noteq> B \<and> B \<noteq> C \<and> C \<noteq> D \<and> A \<noteq> D \<and> 
Per B A D \<and> 
Per A D C \<and> 
Per A B C \<and> 
Coplanar A B C D"

definition OS ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ OS _ _" [99,99,99,99] 50)
  where 
    "A B OS P Q 
\<equiv>
\<exists> R::'p. A B TS P R \<and> A B TS Q R"

definition TSP ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ _ TSP _ _" [99,99,99,99,99] 50)
  where 
    "A B C TSP P Q 
\<equiv>
(\<not> Coplanar A B C P) \<and> (\<not> Coplanar A B C Q) \<and>
(\<exists> T. Coplanar A B C T \<and> Bet P T Q)"

definition OSP ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ _ OSP _ _" [99,99,99,99,99] 50)
  where 
    "A B C OSP P Q 
\<equiv>
\<exists> R. ((A B C TSP P R) \<and> (A B C TSP Q R))"

definition Saccheri::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("Saccheri _ _ _ _" [99,99,99,99] 50)
  where 
    "Saccheri A B C D 
\<equiv>
Per B A D \<and> 
Per A D C \<and> 
Cong A B C D \<and> A D OS B C"

definition ReflectLAt ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ ReflectLAt _ _ _ _" [99,99,99,99,99] 50)
  where 
    "M ReflectLAt P' P A B 
\<equiv>
(M Midpoint P P' \<and> Col A B M) \<and> (A B Perp P P' \<or> P = P')"

definition ReflectAt ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ ReflectAt _ _ _ _" [99,99,99,99,99] 50)
  where 
    "M ReflectAt P' P A B 
\<equiv>
(A \<noteq> B \<and> M ReflectLAt P' P A B) \<or> (A = B \<and> A = M \<and> M Midpoint P P')"

definition upper_dim_axiom ::
  "bool" 
  ("UpperDimAxiom" [] 50)
  where
    "upper_dim_axiom 
\<equiv> 
\<forall> A B C P Q.
P \<noteq> Q \<and>
Cong A P A Q \<and>
Cong B P B Q \<and>
Cong C P C Q
\<longrightarrow>
(Bet A B C \<or> Bet B C A \<or> Bet C A B)"

definition all_coplanar_axiom ::
  "bool" 
  ("AllCoplanarAxiom" [] 50)
  where
    "AllCoplanarAxiom 
\<equiv> 
\<forall> A B C P Q.
P \<noteq> Q \<and>
Cong A P A Q \<and>
Cong B P B Q \<and>
Cong C P C Q
\<longrightarrow>
(Bet A B C \<or> Bet B C A \<or> Bet C A B)"

(** If three points A, B and C are equidistant to three distinct points P, Q and R,
    then A, B and C are collinear. *)

definition upper_dim_3_axiom :: 
  "bool" 
  where
    "upper_dim_3_axiom 
\<equiv> 
\<forall> A B C P Q R. P \<noteq> Q \<and> Q \<noteq> R \<and> P \<noteq> R \<and>
  Cong A P A Q \<and> Cong B P B Q \<and> Cong C P C Q \<and>
  Cong A P A R \<and> Cong B P B R \<and> Cong C P C R \<longrightarrow>
  (Bet A B C \<or> Bet B C A \<or> Bet C A B)"

(** If four points are equidistant to two distinct points, then they are coplanar. *)

definition median_planes_axiom :: 
  "bool" 
  where
    "median_planes_axiom 
\<equiv> 
\<forall> A B C D P Q. P \<noteq> Q \<and>
  Cong A P A Q \<and> Cong B P B Q \<and> Cong C P C Q \<and> Cong D P D Q \<longrightarrow>
  Coplanar A B C D"

(** If two planes meet in some point, then they also meet in another point. *)

definition plane_intersection_axiom :: 
  "bool" 
  where
    "plane_intersection_axiom 
\<equiv> 
\<forall> A B C D E F P.
  Coplanar A B C P \<and> Coplanar D E F P \<longrightarrow>
(\<exists> Q. Coplanar A B C Q \<and> Coplanar D E F Q \<and> P \<noteq> Q)"

(** If two points do not lie on a plane, then they are either
    on opposite sides or on the same side of the plane. *)

definition space_separation_axiom :: 
  "bool" 
  where 
    "space_separation_axiom 
\<equiv> 
\<forall> A B C P Q.
  \<not> Coplanar A B C P \<and> \<not> Coplanar A B C Q \<longrightarrow>
  (A B C TSP P Q \<or> A B C OSP P Q)"

(** The line segments SU1, SU2, SU3 and SU4 can not form an orthonormal family *)

definition orthonormal_family_axiom :: 
  "bool" 
  where
    "orthonormal_family_axiom 
\<equiv> 
\<forall> S U1' U1 U2 U3 U4.
  \<not> (S \<noteq> U1' \<and> Bet U1 S U1' \<and>
     Cong S U1 S U1' \<and> Cong S U2 S U1' \<and> Cong S U3 S U1' \<and> Cong S U4 S U1' \<and>
     Cong U1 U2 U1' U2 \<and> Cong U1 U3 U1' U2 \<and> Cong U1 U4 U1' U2 \<and>
     Cong U2 U3 U1' U2 \<and> Cong U2 U4 U1' U2 \<and> Cong U3 U4 U1' U2)"

definition CongA ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ _ CongA _ _ _" [99,99,99,99,99,99] 50)
  where 
    "A B C CongA D E F 
\<equiv>
A \<noteq> B \<and> C \<noteq> B \<and> D \<noteq> E \<and> F \<noteq> E \<and>
(\<exists> A' C' D' F'. Bet B A A' \<and> Cong A A' E D \<and> Bet B C C' \<and> Cong C C' E F \<and>
Bet E D D' \<and> Cong D D' B A \<and> Bet E F F' \<and> Cong F F' B C \<and>
Cong A' C' D' F')"

definition LeA ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ _ LeA _ _ _" [99,99,99,99,99,99] 50)
  where 
    "A B C LeA D E F 
\<equiv>
\<exists> P. (P InAngle D E F \<and> A B C CongA D E P)"

definition LtA ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ _ LtA _ _ _" [99,99,99,99,99,99] 50)
  where 
    "A B C LtA D E F 
\<equiv> 
A B C LeA D E F \<and> \<not> A B C CongA D E F"

definition GtA ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ _ GtA _ _ _" [99,99,99,99,99,99] 50)
  where 
    "A B C GtA D E F 
\<equiv> 
D E F LtA A B C"

definition Acute ::
  "['p,'p,'p] \<Rightarrow> bool" 
  ("Acute _ _ _" [99,99,99] 50)
  where 
    "Acute A B C 
\<equiv>
\<exists> A' B' C'. (Per A' B' C' \<and> A B C LtA A' B' C')"

definition Obtuse ::
  "['p,'p,'p] \<Rightarrow> bool" 
  ("Obtuse _ _ _" [99,99,99] 50)
  where 
    "Obtuse A B C 
\<equiv>
\<exists> A' B' C'. (Per A' B' C' \<and> A' B' C' LtA A B C)"

definition OrthAt ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ OrthAt _ _ _ _ _" [99,99,99,99,99,99] 50)
  where 
    "X OrthAt A B C U V 
\<equiv>
\<not> Col A B C \<and> U \<noteq> V \<and> Coplanar A B C X \<and> Col U V X \<and>
(\<forall> P Q. (Coplanar A B C P \<and> Col U V Q) \<longrightarrow> Per P X Q)"

definition Orth ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ _ Orth _ _" [99,99,99,99,99] 50)
  where 
    "A B C Orth U V 
\<equiv>
\<exists> X. X OrthAt A B C U V"

definition SuppA ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool"
  ("_ _ _ SuppA _ _ _ " [99,99,99,99,99,99] 50)
  where
    "A B C SuppA D E F 
\<equiv>
A \<noteq> B \<and> (\<exists> A'. Bet A B A' \<and>  D E F CongA C B A')"

definition SumA ::
  "['p,'p,'p,'p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ _ _ _ _ SumA _ _ _" [99,99,99,99,99,99,99,99,99] 50)
  where
    "A B C D E F SumA G H I 
\<equiv>
\<exists> J. (C B J CongA D E F \<and> \<not> B C OS A J \<and> Coplanar A B C J \<and> A B J CongA G H I)"

definition TriSumA ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ _ TriSumA _ _ _" [99,99,99,99,99,99] 50)
  where
    "A B C TriSumA D E F 
\<equiv>
\<exists> G H I. (A B C B C A SumA G H I \<and> G H I C A B SumA D E F)"

definition SAMS ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("SAMS _ _ _ _ _ _" [99,99,99,99,99,99] 50)
  where
    "SAMS A B C D E F 
\<equiv>
(A \<noteq> B \<and>
(E Out D F \<or> \<not> Bet A B C)) \<and>
(\<exists> J. (C B J CongA D E F \<and> \<not> (B C OS A J) \<and> \<not> (A B TS C J) \<and> Coplanar A B C J))"

definition Inter ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ Inter _ _ _ _" [99,99,99,99,99] 50)
  where 
    "X Inter A1 A2 B1 B2 
\<equiv>
B1 \<noteq> B2 \<and>
(\<exists> P::'p. (Col P B1 B2 \<and> \<not> Col P A1 A2)) \<and>
Col A1 A2 X \<and> Col B1 B2 X"

definition Perp2 :: (*GeoCoq 'Perp2 A B C D E' = IsaGeoCoq 'E Perp2 A B C D'*)
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ Perp2 _ _ _ _" [99,99,99,99,99] 50)
  where
    "P Perp2 A B C D 
\<equiv>
\<exists> X Y. (Col P X Y \<and> X Y Perp A B \<and> X Y Perp C D)"

definition Perp_bisect ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ PerpBisect _ _" [99,99,99,99] 50)
  where 
    "P Q PerpBisect A B 
\<equiv>
A B ReflectL P Q \<and> A \<noteq> B"

definition Perp_bisect_bis ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ PerpBisectBis _ _" [99,99,99,99] 50)
  where 
    "P Q PerpBisectBis A B 
\<equiv>
\<exists> I. I PerpAt P Q A B \<and> I Midpoint A B"

definition Is_on_perp_bisect ::
  "['p,'p,'p] \<Rightarrow> bool" 
  ("_ IsOnPerpBisect _ _" [99,99,99] 50)
  where 
    "P IsOnPerpBisect A B 
\<equiv>
Cong A P P B"

definition isosceles::
  "['p,'p,'p] \<Rightarrow> bool"
  ("_ _ _ isosceles" [99,99,99] 50)
  where
    "A B C isosceles 
\<equiv>
Cong A B B C"

definition equilateral::
  "['p,'p,'p] \<Rightarrow> bool"
  ("_ _ _ equilateral" [99,99,99] 50)
  where
    "A B C equilateral 
\<equiv>
Cong A B B C \<and> Cong B C C A"

definition equilateralStrict::
  "['p,'p,'p] \<Rightarrow> bool"
  ("_ _ _ equilateralStrict" [99,99,99] 50)
  where
    "A B C equilateralStrict 
\<equiv>
A B C equilateral \<and> A \<noteq> B"

definition QCong::
  "(['p,'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("QCong _" [99] 50)
  where
    "QCong l 
\<equiv>
\<exists> A B. (\<forall> X Y. (Cong A B X Y \<longleftrightarrow> l X Y))"

definition TarskiLen::
  "['p,'p,(['p,'p] \<Rightarrow> bool)] \<Rightarrow> bool" 
  ("TarskiLen _ _ _" [99,99,99] 50)
  where
    "TarskiLen A B l 
\<equiv>
QCong l \<and> l A B"

definition QCongNull ::
  "(['p,'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("QCongNull _" [99] 50)
  where
    "QCongNull l 
\<equiv>
QCong l \<and> (\<exists> A. l A A)"

definition QCongA ::
  "(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("QCongA _" [99] 50)
  where
    "QCongA a 
\<equiv>
\<exists> A B C. (A \<noteq> B \<and> C \<noteq> B \<and> (\<forall> X Y Z. A B C CongA X Y Z \<longleftrightarrow> a X Y Z))"

definition Ang ::
  "['p,'p,'p, (['p, 'p, 'p] \<Rightarrow> bool) ] \<Rightarrow> bool" 
  ("_ _ _ Ang _" [99,99,99,99] 50)
  where
    "A B C Ang a 
\<equiv>
QCongA a \<and>
a A B C"

definition QCongAAcute ::
  "(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("QCongAACute _" [99] 50)
  where
    "QCongAAcute a 
\<equiv>
\<exists> A B C. (Acute A B C \<and> (\<forall> X Y Z. (A B C CongA X Y Z \<longleftrightarrow> a X Y Z)))"

definition AngAcute ::
  "['p,'p,'p, (['p,'p,'p] \<Rightarrow> bool)] \<Rightarrow> bool" 
  ("_ _ _ AngAcute _" [99,99,99,99] 50)
  where
    "A B C AngAcute a 
\<equiv>
((QCongAAcute a) \<and> (a A B C))"

definition QCongANullAcute ::
  "(['p,'p,'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("QCongANullAcute _" [99] 50)
  where
    "QCongANullAcute a 
\<equiv>
QCongAAcute a \<and>
(\<forall> A B C. (a A B C \<longrightarrow> B Out A C))"

definition QCongAnNull ::
  "(['p,'p,'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("QCongAnNull _" [99] 50)
  where
    "QCongAnNull a 
\<equiv>
QCongA a \<and>
(\<forall> A B C. (a A B C \<longrightarrow> \<not> B Out A C))"

definition QCongAnFlat ::
  "(['p,'p,'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("QCongAnFlat _" [99] 50)
  where
    "QCongAnFlat a 
\<equiv>
QCongA a \<and>
(\<forall> A B C. (a A B C \<longrightarrow> \<not> Bet A B C))"

definition IsNullAngaP ::
  "(['p,'p,'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("IsNullAngaP _" [99] 50)
  where
    "IsNullAngaP a
\<equiv>
QCongAAcute a \<and>
(\<exists> A B C. (a A B C \<and> B Out A C))"

definition QCongANull ::
  "(['p,'p,'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("QCongANull _" [99] 50)
  where
    "QCongANull a 
\<equiv>
QCongA a \<and>
(\<forall> A B C. (a A B C \<longrightarrow> B Out A C))"

definition AngFlat ::
  "(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("AngFlat _" [99] 50)
  where
    "AngFlat a 
\<equiv>
QCongA a \<and>
(\<forall> A B C. (a A B C \<longrightarrow> Bet A B C))"

definition EqLTarski ::
  "(['p, 'p] \<Rightarrow> bool) \<Rightarrow> (['p, 'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("_ EqLTarski _" [99,99] 50)
  where
    "l1 EqLTarski l2 
\<equiv> 
\<forall> A B. l1 A B \<longleftrightarrow> l2 A B"

definition EqA ::
  "(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> (['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> bool" 
  ("_ EqA _" [99,99] 50)
  where
    "a1 EqA a2 
\<equiv> 
\<forall> A B C. a1 A B C \<longleftrightarrow> a2 A B C"

definition hypothesis_of_right_saccheri_quadrilaterals ::
  "bool"
  ("HypothesisRightSaccheriQuadrilaterals")
  where
    "hypothesis_of_right_saccheri_quadrilaterals 
\<equiv>
\<forall> A B C D. Saccheri A B C D \<longrightarrow> Per A B C"

definition hypothesis_of_acute_saccheri_quadrilaterals ::
  "bool"
  ("HypothesisAcuteSaccheriQuadrilaterals")
  where
    "hypothesis_of_acute_saccheri_quadrilaterals 
\<equiv>
\<forall> A B C D. Saccheri A B C D \<longrightarrow> Acute A B C"

definition hypothesis_of_obtuse_saccheri_quadrilaterals ::
  "bool"
  ("HypothesisObtuseSaccheriQuadrilaterals")
  where
    "hypothesis_of_obtuse_saccheri_quadrilaterals 
\<equiv>
\<forall> A B C D. Saccheri A B C D \<longrightarrow> Obtuse A B C"

definition Defect ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Defect _ _ _ _ _ _ " [99,99,99,99,99,99] 50)
  where
    "Defect A B C D E F
\<equiv>
(\<exists> G H I. (A B C TriSumA G H I \<and> G H I SuppA D E F))"

(** Definition 14.1 **)

definition Ar1 ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Ar1 _ _ _ _ _" [99,99,99,99,99] 50)
  where 
    "Ar1 PO E A B C 
\<equiv>
PO \<noteq> E \<and> Col PO E A \<and> Col PO E B \<and> Col PO E C"

definition Ar2 ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Ar2 _ _ _ _ _ _" [99,99,99,99,99,99] 50)
  where 
    "Ar2 PO E E' A B C 
\<equiv>
\<not> Col PO E E' \<and> Col PO E A \<and> Col PO E B \<and> Col PO E C"

(** Definition 14.2 **)

definition Pj ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ Pj _ _" [99,99,99,99] 50)
  where 
    "A B Pj C D
\<equiv>
A B Par C D \<or> C = D"

(** Definition 14.3. **)

definition Sum ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Sum _ _ _ _ _ _" [99,99,99,99,99,99] 50)
  where 
    "Sum PO E E' A B C 
\<equiv>
Ar2 PO E E' A B C \<and>
(\<exists> A' C'. E E' Pj A  A' \<and> Col PO E' A' \<and> PO E Pj A' C' \<and> 
         PO E' Pj B  C' \<and> E' E Pj C' C)"

definition Proj ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("_ _ Proj _ _ _ _" [99,99,99,99,99,99] 50)
  where
    "P Q Proj A B X Y
\<equiv>
A \<noteq> B \<and> X \<noteq> Y \<and> \<not> A B Par X Y \<and> Col A B Q \<and> (P Q Par X Y \<or> P = Q)"

definition Sump ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Sump _ _ _ _ _ _" [99,99,99,99,99,99] 50)
  where
    "Sump PO E E' A B C
\<equiv>  
Col PO E A \<and> Col PO E B \<and>
(\<exists> A' C' P'. A A' Proj PO E' E E' \<and> PO E Par A' P' \<and>
             B C' Proj A' P' PO E' \<and> C' C Proj PO E E E')"

(** Definition 14.4. *)

definition Prod ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Prod _ _ _ _ _ _" [99,99,99,99,99,99] 50)
  where
    "Prod PO E E' A B C 
\<equiv>
Ar2 PO E E' A B C \<and>
 (\<exists> B'. E E' Pj B B' \<and> Col PO E' B' \<and> E' A Pj B' C)"

definition Prodp ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Prodp _ _ _ _ _ _" [99,99,99,99,99,99] 50)
  where
    "Prodp PO E E' A B C 
\<equiv>
Col PO E A \<and> Col PO E B \<and>
 (\<exists> B'. B B' Proj PO E' E E' \<and> B' C Proj PO E A E')"

(** Definition 14.8. *)

definition Opp ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Opp _ _ _ _ _" [99,99,99,99,99] 50)
  where 
    "Opp PO E E' A B 
\<equiv>
Sum PO E E' B A PO"

(** Definition 14.38. *)

definition Diff ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Diff _ _ _ _ _ _" [99,99,99,99,99,99] 50)
  where
    "Diff PO E E' A B C
\<equiv>
\<exists> B'. Opp PO E E' B B' \<and> Sum PO E E' A B' C"

definition sum3 ::
  "['p,'p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("sum3 _ _ _ _ _ _ _" [99,99,99,99,99,99,99] 50)
  where
    "sum3 PO E E' A B C S
\<equiv>
\<exists> AB. Sum PO E E' A B AB \<and> Sum PO E E' AB C S"

definition sum4 ::
  "['p,'p,'p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Sum4 _ _ _ _ _ _ _ _" [99,99,99,99,99,99,99,99] 50)
  where
    "Sum4 PO E E' A B C D S 
\<equiv>
\<exists> ABC. sum3 PO E E' A B C ABC \<and> Sum PO E E' ABC D S"

definition sum22 ::
  "['p,'p,'p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("sum22 _ _ _ _ _ _ _ _" [99,99,99,99,99,99,99,99] 50)
  where
    "sum22 PO E E' A B C D S 
\<equiv>
\<exists> AB CD. Sum PO E E' A B AB \<and> Sum PO E E' C D CD \<and> Sum PO E E' AB CD S"

definition Ar2p4 :: (* Ar2_4 *)
  "['p,'p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Ar2p4 _ _ _ _ _ _ _" [99,99,99,99,99,99,99] 50)
  where
    "Ar2p4 PO E E' A B C D
\<equiv>
\<not> Col PO E E' \<and> Col PO E A \<and> Col PO E B \<and> Col PO E C \<and> Col PO E D" 

(** Definition 14.34. *)

definition Ps ::
  "['p,'p,'p] \<Rightarrow> bool" 
  ("Ps _ _ _" [99,99,99] 50)
  where 
    "Ps X E A 
\<equiv> 
 X Out A E"

definition Ng ::
  "['p,'p,'p] \<Rightarrow> bool" 
  ("Ng _ _ _" [99,99,99] 50)
  where 
    "Ng X E A 
\<equiv>
 A \<noteq> X \<and> E \<noteq> X \<and> Bet A X E"

(** Definition 14.38. *)

definition LtP ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("LtP _ _ _ _ _ " [99,99,99,99,99] 50)
  where 
    "LtP X E E' A B
\<equiv>
\<exists> D. Diff X E E' B A D \<and> Ps X E D"

definition LeP :: 
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("LeP _ _ _ _ _" [99,99,99,99,99] 50)
  where 
    "LeP X E E' A B
\<equiv> 
LtP X E E' A B \<or> A = B"

definition Length ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Length _ _ _ _ _ _ " [99,99,99,99,99,99] 50)
  where 
    "Length X E E' A B L
\<equiv>
 X \<noteq> E \<and> Col X E L \<and> LeP X E E' X L \<and> Cong X L A B"

(** Definition 15.1. *)

definition IsLength ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("IsLength _ _ _ _ _ _ " [99,99,99,99,99,99] 50)
  where 
    "IsLength X E E' A B L 
\<equiv>  
Length X E E' A B L \<or> (X = E \<and> X = L)"

definition Sumg ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Sumg _ _ _ _ _ _ " [99,99,99,99,99,99] 50)
  where 
    "Sumg X E E' A B C 
\<equiv>
Sum X E E' A B C \<or> (\<not> Ar2 X E E' A B B \<and> C = X)"

definition Prodg ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("Prodg _ _ _ _ _ _ " [99,99,99,99,99,99] 50)
  where 
    "Prodg X E E' A B C 
\<equiv>
Prod X E E' A B C \<or> (\<not> Ar2 X E E' A B B \<and> C = X)"

definition PythRel ::
  "['p,'p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("PythRel _ _ _ _ _ _ " [99,99,99,99,99,99] 50)
  where 
    "PythRel X E E' A B C 
\<equiv>
Ar2 X E E' A B C \<and>
((X = B \<and> (A = C \<or> Opp X E E' A C)) \<or> (\<exists> B'. X B' Perp X B \<and> Cong X B' X B \<and> Cong X C A B'))"

definition SignEq ::
  "['p,'p,'p,'p] \<Rightarrow> bool" 
  ("SignEq _ _ _ _ " [99,99,99,99] 50)
  where 
    "SignEq X E A B 
\<equiv>
Ps X E A \<and> Ps X E B \<or> Ng X E A \<and> Ng X E B" 

definition LtPs ::
  "['p,'p,'p,'p,'p] \<Rightarrow> bool" 
  ("LtPs _ _ _ _ _" [99,99,99,99,99] 50)
  where 
    "LtPs X E E' A B 
\<equiv>
\<exists> D. Ps X E D \<and> Sum X E E' A D B" 

definition IsOrthocenter ::
  "['p,'p,'p,'p] \<Rightarrow> bool"
  ("_ IsOrthocenter _ _ _" [99,99,99,99] 50)
  where
    "H IsOrthocenter A B C \<equiv> \<not> Col A B C \<and> 
                              A H Perp B C \<and> 
                              B H Perp A C \<and> 
                              C H Perp A B"

definition IsCircumcenter ::
  "['p,'p,'p,'p] \<Rightarrow> bool"
  ("_ IsCircumcenter _ _ _" [99,99,99,99] 50)
  where
    "G IsCircumcenter A B C \<equiv>
Cong A G B G \<and> 
Cong B G C G \<and> 
Coplanar G A B C"

definition IsGravityCenter ::
  "['p,'p,'p,'p] \<Rightarrow> bool"
  ("_ IsGravityCenter _ _ _" [99,99,99,99] 50)
  where
    "G IsGravityCenter A B C \<equiv> \<not> Col A B C \<and> 
                               (\<exists> I J. I Midpoint B C \<and> 
                                       J Midpoint A C \<and> 
                                       Col G A I \<and> 
                                       Col G B J)"

(** Definition 13.9. *)

definition Lcos :: "(['p,'p] \<Rightarrow> bool) \<Rightarrow> 
(['p,'p] \<Rightarrow> bool) \<Rightarrow>
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
bool"
  where
    "Lcos lb lc a \<equiv>
  QCong lb \<and> QCong lc \<and> QCongAAcute a \<and>
  (\<exists> A B C. (Per C B A \<and> lb A B \<and> lc A C \<and> a B A C))"

definition EqLcos :: "(['p,'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p,'p] \<Rightarrow> bool) \<Rightarrow>
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
bool"
  where
    "EqLcos la a lb b \<equiv> (\<exists> lp. Lcos lp la a \<and> Lcos lp lb b)"

definition Lcos2 :: "(['p,'p] \<Rightarrow> bool) \<Rightarrow> 
(['p,'p] \<Rightarrow> bool) \<Rightarrow>
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
bool"
  where
    "Lcos2 lp l a b \<equiv> \<exists> la. Lcos la l a \<and> Lcos lp la b" 

definition EqLcos2 :: "(['p,'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p,'p] \<Rightarrow> bool) \<Rightarrow>
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
bool"
  where
    "EqLcos2 l1 a b l2 c d \<equiv> (\<exists> lp. Lcos2 lp l1 a b \<and> Lcos2 lp l2 c d)" 

definition Lcos3 :: "(['p,'p] \<Rightarrow> bool) \<Rightarrow> 
(['p,'p] \<Rightarrow> bool) \<Rightarrow>
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
bool"
  where
    "Lcos3 lp l a b c \<equiv> \<exists> la lab. Lcos la l a \<and>
Lcos lab la b \<and> Lcos lp lab c"

definition EqLcos3 :: "(['p,'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p,'p] \<Rightarrow> bool) \<Rightarrow>
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
(['p, 'p, 'p] \<Rightarrow> bool) \<Rightarrow> 
bool"
  where
    "EqLcos3 l1 a b c l2 d e f \<equiv> (\<exists> lp. Lcos3 lp l1 a b c \<and> Lcos3 lp l2 d e f)" 

(** Vector *)

definition EqV :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool"
  ("_ _ EqV _ _ " [99,99,99,99] 50)
  where
    "A B EqV C D \<equiv> Parallelogram A B D C \<or> (A = B \<and> C = D)"

definition SumV :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool"
  ("_ _ _ _ SumV _ _ " [99,99,99,99,99,99] 50)
  where
    "A B C D SumV E F \<equiv> \<forall> D'. C D EqV B D' \<longrightarrow> A D' EqV E F"

definition SumVExists :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool"
  ("_ _ _ _ SumVExists _ _ " [99,99,99,99,99,99] 50)
  where
    "A B C D SumVExists E F \<equiv> (\<exists> D'. B D' EqV C D \<and> A D' EqV E F)"

definition SameDir :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool"
  ("_ _ SameDir _ _ " [99,99,99,99] 50)
  where
    "A B SameDir C D  \<equiv> 
(A = B \<and> C = D) \<or> (\<exists> D'. C Out D D' \<and> A B EqV C D')" 

definition OppDir :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool"
  ("_ _ OppDir _ _ " [99,99,99,99] 50)
  where
    "A B OppDir C D  \<equiv> A B SameDir D C"

(* definition Proj :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool"
  ("_ _ Proj _ _ _ _" [99,99,99,99,99,99] 50)
  where
    "P Q Proj A B X Y \<equiv>

  A \<noteq> B \<and> X \<noteq> Y \<and> \<not> A B Par X Y \<and> Col A B Q \<and> (P Q Par X Y \<or> P = Q)"
*)

definition CongA3 :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool"
  ("_ _ _ CongA3 _ _ _" [99,99,99,99,99,99] 50)
  where
    "A B C CongA3 A' B' C' \<equiv>
  A B C CongA A' B' C' \<and> B C A CongA B' C' A' \<and> C A B CongA C' A' B'" 

(** Q is the orthogonal projection of P on the line AB. *)
definition Projp :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool"
  ("_ _ Projp _ _" [99,99,99,99] 50)
  where
    "P Q Projp A B \<equiv>
  A \<noteq> B \<and> ((Col A B Q \<and> A B Perp P Q) \<or> (Col A B P \<and> P = Q))"

subsection "Propositions"

lemma cong_reflexivity:
  shows "Cong A B A B"
  using cong_inner_transitivity cong_pseudo_reflexivity by blast

lemma cong_symmetry:
  assumes "Cong A B C D"
  shows "Cong C D A B"
  using assms cong_inner_transitivity cong_reflexivity by blast

lemma cong_transitivity:
  assumes "Cong A B C D" and "Cong C D E F"
  shows "Cong A B E F"
  by (meson assms(1) assms(2) cong_inner_transitivity cong_pseudo_reflexivity)

lemma cong_left_commutativity:
  assumes "Cong A B C D"
  shows "Cong B A C D"
  using assms cong_inner_transitivity cong_pseudo_reflexivity by blast

lemma cong_right_commutativity:
  assumes "Cong A B C D"
  shows "Cong A B D C"
  using assms cong_left_commutativity cong_symmetry by blast

lemma cong_3421:
  assumes "Cong A B C D"
  shows "Cong C D B A"
  using assms cong_left_commutativity cong_symmetry by blast

lemma cong_4312:
  assumes "Cong A B C D"
  shows "Cong D C A B"
  using assms cong_left_commutativity cong_symmetry by blast

lemma cong_4321:
  assumes "Cong A B C D"
  shows "Cong D C B A"
  using assms cong_3421 cong_left_commutativity by blast

lemma cong_trivial_identity:
  shows "Cong A A B B"
  using cong_identity segment_construction by blast

lemma cong_reverse_identity:
  assumes "Cong A A C D"
  shows "C = D"
  using assms cong_3421 cong_identity by blast

lemma cong_commutativity:
  assumes "Cong A B C D"
  shows "Cong B A D C"
  using assms cong_3421 by blast

lemma not_cong_2134:
  assumes " \<not> Cong A B C D"
  shows "\<not> Cong B A C D"
  using assms cong_left_commutativity by blast

lemma not_cong_1243:
  assumes "\<not> Cong A B C D"
  shows "\<not> Cong A B D C"
  using assms cong_right_commutativity by blast

lemma not_cong_2143:
  assumes "\<not> Cong A B C D"
  shows "\<not> Cong B A D C"
  using assms cong_commutativity by blast

lemma not_cong_3412:
  assumes "\<not> Cong A B C D"
  shows "\<not> Cong C D A B"
  using assms cong_symmetry by blast

lemma not_cong_4312:
  assumes "\<not> Cong A B C D"
  shows "\<not> Cong D C A B"
  using assms cong_3421 by blast

lemma not_cong_3421:
  assumes "\<not> Cong A B C D"
  shows "\<not> Cong C D B A"
  using assms cong_4312 by blast

lemma not_cong_4321:
  assumes "\<not> Cong A B C D"
  shows "\<not> Cong D C B A"
  using assms cong_4321 by blast

lemma five_segment_with_def:
  assumes "A B C D OFSC A' B' C' D'" and "A \<noteq> B"
  shows "Cong C D C' D'"
  using assms(1) assms(2) OFSC_def five_segment by blast

lemma cong_diff:
  assumes "A \<noteq> B" and "Cong A B C D"
  shows "C \<noteq> D"
  using assms(1) assms(2) cong_identity by blast

lemma cong_diff_2:
  assumes "B \<noteq> A" and "Cong A B C D"
  shows "C \<noteq> D"
  using assms(1) assms(2) cong_identity by blast

lemma cong_diff_3:
  assumes "C \<noteq> D" and "Cong A B C D"
  shows "A \<noteq> B"
  using assms(1) assms(2) cong_reverse_identity by blast

lemma cong_diff_4:
  assumes "D \<noteq> C" and "Cong A B C D"
  shows "A \<noteq> B"
  using assms(1) assms(2) cong_reverse_identity by blast

lemma cong_3_sym:
  assumes "A B C Cong3 A' B' C'"
  shows "A' B' C' Cong3 A B C"
  using assms Cong3_def not_cong_3412 by blast

lemma cong_3_swap:
  assumes "A B C Cong3 A' B' C'"
  shows "B A C Cong3 B' A' C'"
  using assms Cong3_def cong_commutativity by blast

lemma cong_3_swap_2:
  assumes "A B C Cong3 A' B' C'"
  shows "A C B Cong3 A' C' B'"
  using assms Cong3_def cong_commutativity by blast

lemma cong3_transitivity:
  assumes "A0 B0 C0 Cong3 A1 B1 C1" and
    "A1 B1 C1 Cong3 A2 B2 C2"
  shows "A0 B0 C0 Cong3 A2 B2 C2"
  by (meson assms(1) assms(2) Cong3_def cong_inner_transitivity not_cong_3412)

lemma eq_dec_points:
  shows "A = B \<or> \<not> A = B"
  by simp

lemma distinct:
  assumes "P \<noteq> Q"
  shows "R \<noteq> P \<or> R \<noteq> Q"
  using assms by simp

lemma l2_11:
  assumes "Bet A B C" and
    "Bet A' B' C'" and
    "Cong A B A' B'" and
    "Cong B C B' C'"
  shows "Cong A C A' C'"
proof cases
  assume "A = B"
  thus ?thesis 
    using assms(3) assms(4) cong_reverse_identity by blast
next
  assume "A \<noteq> B"
  thus ?thesis 
    using five_segment Tarski_neutral_dimensionless_axioms assms(1) assms(2) assms(3) assms(4) 
      cong_commutativity cong_trivial_identity by blast
qed

lemma bet_cong3:
  assumes "Bet A B C" and
    "Cong A B A' B'"
  shows "\<exists> C'. A B C Cong3 A' B' C'"
  by (meson assms(1) assms(2) Cong3_def l2_11 not_cong_3412 segment_construction)

lemma construction_uniqueness:
  assumes "Q \<noteq> A" and
    "Bet Q A X" and
    "Cong A X B C" and
    "Bet Q A Y" and
    "Cong A Y B C"
  shows "X = Y"
  by (meson assms(1) assms(2) assms(3) assms(4) assms(5) cong_identity cong_inner_transitivity 
      cong_reflexivity five_segment)

lemma Cong_cases:
  assumes "Cong A B C D \<or> Cong A B D C \<or> Cong B A C D \<or> Cong B A D C \<or> Cong C D A B \<or> 
Cong C D B A \<or> Cong D C A B \<or> Cong D C B A"
  shows "Cong A B C D"
  using assms not_cong_3421 not_cong_4321 by blast

lemma Cong_perm :
  assumes "Cong A B C D"
  shows "Cong A B C D \<and> Cong A B D C \<and> Cong B A C D \<and> Cong B A D C \<and> Cong C D A B \<and> 
Cong C D B A \<and> Cong D C A B \<and> Cong D C B A"
  using assms not_cong_1243 not_cong_3412 by blast

lemma bet_col:
  assumes "Bet A B C"
  shows "Col A B C"
  by (simp add: assms Col_def)

lemma between_trivial:
  shows "Bet A B B"
  using cong_identity segment_construction by blast

lemma between_symmetry:
  assumes "Bet A B C"
  shows "Bet C B A"
  using assms between_identity between_trivial inner_pasch by blast

lemma Bet_cases:
  assumes "Bet A B C \<or> Bet C B A"
  shows "Bet A B C"
  using assms between_symmetry by blast

lemma Bet_perm:
  assumes "Bet A B C"
  shows "Bet A B C \<and> Bet C B A"
  using assms Bet_cases by blast

lemma between_trivial2:
  shows "Bet A A B"
  using Bet_perm between_trivial by blast

lemma between_equality:
  assumes "Bet A B C" and "Bet B A C"
  shows "A = B"
  using assms(1) assms(2) between_identity inner_pasch by blast

lemma between_equality_2:
  assumes "Bet A B C" and
    "Bet A C B"
  shows "B = C"
  using assms(1) assms(2) between_equality between_symmetry by blast

lemma between_exchange3:
  assumes "Bet A B C" and
    "Bet A C D"
  shows "Bet B C D"
  by (metis Bet_perm assms(1) assms(2) between_identity inner_pasch)

lemma bet_neq12__neq:
  assumes "Bet A B C" and
    "A \<noteq> B"
  shows "A \<noteq> C"
  using assms(1) assms(2) between_identity by blast

lemma bet_neq21__neq:
  assumes "Bet A B C" and
    "B \<noteq> A"
  shows "A \<noteq> C"
  using assms(1) assms(2) between_identity by blast

lemma bet_neq23__neq:
  assumes "Bet A B C" and
    "B \<noteq> C"
  shows "A \<noteq> C"
  using assms(1) assms(2) between_identity by blast

lemma bet_neq32__neq:
  assumes "Bet A B C" and
    "C \<noteq> B"
  shows "A \<noteq> C"
  using assms(1) assms(2) between_identity by blast

lemma not_bet_distincts:
  assumes "\<not> Bet A B C"
  shows "A \<noteq> B \<and> B \<noteq> C"
  using assms between_trivial between_trivial2 by blast

lemma between_inner_transitivity:
  assumes "Bet A B D" and
    "Bet B C D"
  shows "Bet A B C"
  using assms(1) assms(2) Bet_perm between_exchange3 by blast

lemma outer_transitivity_between2:
  assumes "Bet A B C" and
    "Bet B C D" and
    "B \<noteq> C"
  shows "Bet A C D"
proof -
  obtain X where "Bet A C X" and "Cong C X C D"
    using segment_construction by blast
  thus ?thesis
    using assms(1) assms(2) assms(3) between_exchange3 cong_inner_transitivity 
      construction_uniqueness by blast
qed

lemma between_exchange2:
  assumes "Bet A B D" and
    "Bet B C D"
  shows "Bet A C D"
  using assms(1) assms(2) between_inner_transitivity outer_transitivity_between2 by blast

lemma outer_transitivity_between:
  assumes "Bet A B C" and
    "Bet B C D" and
    "B \<noteq> C"
  shows "Bet A B D"
  using assms(1) assms(2) assms(3) between_symmetry outer_transitivity_between2 by blast

lemma between_exchange4:
  assumes "Bet A B C" and
    "Bet A C D"
  shows "Bet A B D"
  using assms(1) assms(2) between_exchange2 between_symmetry by blast

lemma l3_9_4:
  assumes "Bet4 A1 A2 A3 A4"
  shows "Bet4 A4 A3 A2 A1"
  using assms Bet4_def Bet_cases by blast

lemma l3_17:
  assumes "Bet A B C" and
    "Bet A' B' C" and
    "Bet A P A'"
  shows "\<exists> Q. Bet P Q C \<and> Bet B Q B'"
proof -
  obtain X where "Bet B' X A" and "Bet P X C"
    using Bet_perm assms(2) assms(3) inner_pasch by blast
  moreover then obtain Y where "Bet X Y C" and "Bet B Y B'"
    using Bet_perm assms(1) inner_pasch by blast
  ultimately show ?thesis
    using between_exchange2 by blast
qed

lemma lower_dim_ex:
  "\<exists> A B C. \<not> (Bet A B C \<or> Bet B C A \<or> Bet C A B)"
  using lower_dim by auto

lemma two_distinct_points:
  "\<exists> X::'p. \<exists> Y::'p. X \<noteq> Y"
  using lower_dim_ex not_bet_distincts by blast

lemma point_construction_different:
  "\<exists> C. Bet A B C \<and> B \<noteq> C"
  using two_distinct_points Tarski_neutral_dimensionless_axioms 
    cong_reverse_identity segment_construction by blast

lemma another_point:
  "\<exists> B::'p. A \<noteq> B"
  using point_construction_different by blast

lemma Cong_stability:
  assumes "\<not> \<not> Cong A B C D"
  shows "Cong A B C D"
  using assms by simp

lemma l2_11_b:
  assumes "Bet A B C" and
    "Bet A' B' C'" and
    "Cong A B A' B'" and
    "Cong B C B' C'"
  shows "Cong A C A' C'"
  using assms(1) assms(2) assms(3) assms(4) l2_11 by auto

lemma cong_dec_eq_dec_b:
  assumes "\<not> A \<noteq> B"
  shows "A = B"
  using assms(1) by simp

lemma BetSEq:
  assumes "BetS A B C"
  shows "Bet A B C \<and> A \<noteq> B \<and> A \<noteq> C \<and> B \<noteq> C"
  using assms BetS_def between_identity by auto

lemma l4_2:
  assumes "A B C D IFSC A' B' C' D'"
  shows "Cong B D B' D'"
proof cases
  assume "A = C"
  thus ?thesis
    by (metis IFSC_def between_identity assms cong_diff_3)
next
  assume "A \<noteq> C"
  have "Bet A B C" and "Bet A' B' C'" and
    "Cong A C A' C'" and "Cong B C B' C'"
    "Cong A D A' D'" and "Cong C D C' D'"
    using IFSC_def assms by auto
  obtain E where "Bet A C E" and "Cong C E A C"
    using segment_construction by blast
  obtain E' where "Bet A' C' E'" and "Cong C' E' C E"
    using segment_construction by blast
  hence "Cong C E C' E'" 
    using Cong_cases by blast
  hence "Cong E D E' D'" 
    using \<open>A \<noteq> C\<close> \<open>Bet A C E\<close> \<open>Bet A' C' E'\<close> \<open>Cong A C A' C'\<close> 
      \<open>Cong A D A' D'\<close> \<open>Cong C D C' D'\<close> 
      five_segment by blast
  moreover
  have "E \<noteq> C" 
    using \<open>A \<noteq> C\<close> \<open>Cong C E A C\<close> cong_reverse_identity by blast
  moreover
  have "Bet E C B" 
    using \<open>Bet A B C\<close> \<open>Bet A C E\<close> between_exchange3 between_symmetry by blast
  moreover
  have "Bet E' C' B'" 
    using \<open>Bet A' B' C'\<close> \<open>Bet A' C' E'\<close> between_exchange3 between_symmetry by blast
  moreover
  have "Cong E C E' C'" 
    by (simp add: \<open>Cong C E C' E'\<close> cong_commutativity)
  moreover
  have "Cong C B C' B' " 
    using Cong_cases \<open>Cong B C B' C'\<close> by blast
  ultimately show ?thesis
    using \<open>Cong C D C' D'\<close> five_segment by blast
qed

lemma l4_3:
  assumes "Bet A B C" and
    "Bet A' B' C'" and
    "Cong A C A' C'"
    and "Cong B C B' C'"
  shows "Cong A B A' B'"
proof -
  have "A B C A IFSC A' B' C' A'"
    using IFSC_def assms(1) assms(2) assms(3) assms(4) cong_trivial_identity 
      not_cong_2143 by blast
  thus ?thesis
    using l4_2 not_cong_2143 by blast
qed


lemma l4_3_1:
  assumes "Bet A B C" and
    "Bet A' B' C'" and
    "Cong A B A' B'" and
    "Cong A C A' C'"
  shows "Cong B C B' C'"
  by (meson assms(1) assms(2) assms(3) assms(4) between_symmetry cong_4321 l4_3)

lemma l4_5:
  assumes "Bet A B C" and
    "Cong A C A' C'"
  shows  "\<exists> B'. (Bet A' B' C' \<and> A B C Cong3 A' B' C')"
proof -
  obtain X' where "Bet C' A' X'" and "A' \<noteq> X'"
    using point_construction_different by auto
  obtain B' where "Bet X' A' B'" and "Cong A' B' A B"
    using segment_construction by blast
  obtain C'' where "Bet X' B' C''" and "Cong B' C'' B C"
    using segment_construction by blast
  hence "Bet A' B' C''"
    using \<open>Bet X' A' B'\<close> between_exchange3 by blast
  moreover
  have "A B C Cong3 A' B' C''" 
    using Cong3_def \<open>Cong A' B' A B\<close> \<open>Cong B' C'' B C\<close> assms(1) calculation cong_symmetry l2_11_b 
    by blast
  moreover
  have "C'' = C'"
  proof -
    have "Bet X' A' C''" 
      using \<open>Bet X' A' B'\<close> \<open>Bet X' B' C''\<close> between_exchange4 by blast
    moreover have "Bet X' A' C'" 
      using Bet_cases \<open>Bet C' A' X'\<close> by auto
    moreover have "Cong A' C'' A C" 
      using \<open>Bet A' B' C''\<close> \<open>Cong A' B' A B\<close> \<open>Cong B' C'' B C\<close> assms(1) l2_11_b by blast
    ultimately show ?thesis 
      by (metis \<open>A' \<noteq> X'\<close> assms(2) cong_symmetry construction_uniqueness)
  qed
  ultimately show ?thesis
    by auto
qed

lemma l4_6:
  assumes "Bet A B C" and
    "A B C Cong3 A' B' C'"
  shows "Bet A' B' C'"
proof -
  obtain x where P1: "Bet A' x C' \<and> A B C Cong3 A' x C'"
    using Cong3_def assms(1) assms(2) l4_5 by blast
  hence "A' x C' Cong3 A' B' C'"
    using assms(2) cong3_transitivity cong_3_sym by blast
  hence "A' x C' x IFSC A' x C' B'"
    by (meson Cong3_def Cong_perm IFSC_def P1 cong_reflexivity)
  hence "Cong x x x B'"
    using l4_2 by auto
  thus ?thesis
    using P1 cong_reverse_identity by blast
qed

lemma cong3_bet_eq:
  assumes "Bet A B C" and
    "A B C Cong3 A X C"
  shows "X = B"
proof -
  have "A B C B IFSC A B C X"
    by (meson Cong3_def Cong_perm IFSC_def assms(1) assms(2) cong_reflexivity)
  thus ?thesis
    using cong_reverse_identity l4_2 by blast
qed

lemma col_permutation_1:
  assumes "Col A B C"
  shows "Col B C A"
  using assms(1) Col_def by blast

lemma col_permutation_2:
  assumes "Col A B C"
  shows "Col C A B"
  using assms(1) col_permutation_1 by blast

lemma col_permutation_3:
  assumes "Col A B C"
  shows "Col C B A"
  using assms(1) Bet_cases Col_def by auto

lemma col_permutation_4:
  assumes "Col A B C"
  shows "Col B A C"
  using assms(1) Bet_perm Col_def by blast

lemma col_permutation_5:
  assumes "Col A B C"
  shows "Col A C B"
  using assms(1) col_permutation_1 col_permutation_3 by blast

lemma not_col_permutation_1:
  assumes "\<not> Col A B C"
  shows "\<not> Col B C A"
  using assms col_permutation_2 by blast

lemma not_col_permutation_2:
  assumes "\<not> Col A B C"
  shows  "\<not> Col C A B"
  using assms col_permutation_1 by blast

lemma not_col_permutation_3:
  assumes "\<not> Col A B C"
  shows "\<not> Col C B A"
  using assms col_permutation_3 by blast

lemma not_col_permutation_4:
  assumes "\<not> Col A B C"
  shows "\<not> Col B A C"
  using assms col_permutation_4 by blast

lemma not_col_permutation_5:
  assumes "\<not> Col A B C"
  shows "\<not> Col A C B"
  using assms col_permutation_5 by blast

lemma Col_cases:
  assumes "Col A B C \<or> Col A C B \<or> Col B A C \<or> Col B C A \<or> Col C A B \<or> Col C B A"
  shows "Col A B C"
  using assms not_col_permutation_4 not_col_permutation_5 by blast

lemma Col_perm:
  assumes "Col A B C"
  shows "Col A B C \<and> Col A C B \<and> Col B A C \<and> Col B C A \<and> Col C A B \<and> Col C B A"
  using Col_cases assms by blast

lemma col_trivial_1:
  "Col A A B"
  using bet_col not_bet_distincts by blast

lemma col_trivial_2:
  "Col A B B"
  by (simp add: Col_def between_trivial2)

lemma col_trivial_3:
  "Col A B A"
  by (simp add: Col_def between_trivial2)

lemma l4_13:
  assumes "Col A B C" and
    "A B C Cong3 A' B' C'"
  shows "Col A' B' C'"
  by (metis Col_def cong_3_swap cong_3_swap_2 assms(1) assms(2) l4_6)

lemma l4_14R1:
  assumes "Bet A B C" and
    "Cong A B A' B'"
  shows "\<exists> C'. A B C Cong3 A' B' C'"
  by (simp add: assms(1) assms(2) bet_cong3)

lemma l4_14R2:
  assumes "Bet B C A" and
    "Cong A B A' B'"
  shows "\<exists> C'. A B C Cong3 A' B' C'"
  by (meson assms(1) assms(2) between_symmetry cong_3_swap_2 l4_5)

lemma l4_14R3:
  assumes "Bet C A B" and
    "Cong A B A' B'"
  shows "\<exists> C'. A B C Cong3 A' B' C'"
  by (meson assms(1) assms(2) between_symmetry cong_3_swap l4_14R1 not_cong_2143)

lemma l4_14:
  assumes "Col A B C" and
    "Cong A B A' B'"
  shows "\<exists> C'. A B C Cong3 A' B' C'"
  using Col_def assms(1) assms(2) l4_14R1 l4_14R2 l4_14R3 by blast

lemma l4_16R1:
  assumes "A B C D FSC A' B' C' D'" and
    "A \<noteq> B" and
    "Bet A B C"
  shows "Cong C D C' D'"
proof -
  have "A B C Cong3 A' B' C'"
    using FSC_def assms(1) by blast
  hence "Bet A' B' C'"
    using assms(3) l4_6 by blast
  hence "A B C D OFSC A' B' C' D'"
    by (meson Cong3_def FSC_def OFSC_def assms(1) cong_3_sym l4_6)
  thus ?thesis
    using assms(2) five_segment_with_def by blast
qed

lemma l4_16R2:
  assumes "A B C D FSC A' B' C' D'"
    and "Bet B C A"
  shows "Cong C D C' D'"
proof -
  have "A B C Cong3 A' B' C'"
    using FSC_def assms(1) by blast
  hence "Bet B' C' A'"
    using Bet_perm assms(2) cong_3_swap_2 l4_6 by blast
  hence "B C A D IFSC B' C' A' D'"
    by (meson Cong3_def FSC_def IFSC_def assms(1) assms(2) not_cong_2143)
  thus ?thesis
    using l4_2 by auto
qed

lemma l4_16R3:
  assumes "A B C D FSC A' B' C' D'" and 
    "A \<noteq> B" and 
    "Bet C A B"
  shows "Cong C D C' D'"
proof -
  have "A B C Cong3 A' B' C'"
    using FSC_def assms(1) by blast
  hence "Bet C' A' B'"
    using assms(3) between_symmetry cong_3_swap l4_6 by blast
  thus ?thesis 
    by (metis Bet_cases Col_def FSC_def cong_3_swap assms(1) assms(2) assms(3) l4_16R1)
qed

lemma l4_16:
  assumes "A B C D FSC A' B' C' D'" and
    "A \<noteq> B"
  shows "Cong C D C' D'"
  by (meson Col_def FSC_def assms(1) assms(2) l4_16R1 l4_16R2 l4_16R3)

lemma l4_17:
  assumes "A \<noteq> B" and
    "Col A B C" and
    "Cong A P A Q" and
    "Cong B P B Q"
  shows "Cong C P C Q"
proof -
  {
    assume "\<not> Bet B C A"
    hence "\<exists>p pa. Bet p pa C \<and> Cong pa P pa Q \<and> Cong p P p Q \<and> p \<noteq> pa"
      using Col_def assms(1) assms(2) assms(3) assms(4) between_symmetry by blast
    hence ?thesis
      using cong_reflexivity five_segment by blast
  }
  thus ?thesis
    by (meson IFSC_def assms(3) assms(4) cong_reflexivity l4_2)
qed

lemma l4_18:
  assumes "A \<noteq> B" and
    "Col A B C" and
    "Cong A C A C'" and
    "Cong B C B C'"
  shows "C = C'"
  using assms(1) assms(2) assms(3) assms(4) cong_diff_3 l4_17 by blast

lemma l4_19:
  assumes "Bet A C B" and
    "Cong A C A C'" and
    "Cong B C B C'"
  shows "C = C'"
  by (metis Col_def assms(1) assms(2) assms(3) between_equality between_trivial cong_identity 
      l4_18 not_cong_3421)

lemma not_col_distincts:
  assumes "\<not> Col A B C"
  shows "\<not> Col A B C \<and> A \<noteq> B \<and> B \<noteq> C \<and> A \<noteq> C"
  using Col_def assms between_trivial by blast

lemma NCol_cases:
  assumes "\<not> Col A B C \<or> \<not> Col A C B \<or> \<not> Col B A C \<or> \<not> Col B C A \<or> \<not> Col C A B \<or> \<not> Col C B A"
  shows  "\<not> Col A B C"
  using assms not_col_permutation_2 not_col_permutation_3 by blast

lemma NCol_perm:
  assumes "\<not> Col A B C"
  shows "\<not> Col A B C \<and> \<not> Col A C B \<and> \<not> Col B A C \<and> \<not> Col B C A \<and> \<not> Col C A B \<and> \<not> Col C B A"
  using NCol_cases assms by blast

lemma col_cong_3_cong_3_eq:
  assumes "A \<noteq> B"
    and "Col A B C"
    and "A B C Cong3 A' B' C1"
    and  "A B C Cong3 A' B' C2"
  shows  "C1 = C2"
  using Cong3_def cong_diff l4_18 assms(1) assms(2) assms(3) assms(4) cong_inner_transitivity 
    l4_13 by meson

lemma l5_1:
  assumes "A \<noteq> B" and
    "Bet A B C" and
    "Bet A B D"
  shows "Bet A C D \<or> Bet A D C"
proof -
  obtain C' where "Bet A D C'" and "Cong D C' C D"
    using segment_construction by blast
  obtain D' where "Bet A C D'" and "Cong C D' C D"
    using segment_construction by blast
  obtain B' where "Bet A C' B'" and "Cong C' B' C B"
    using segment_construction by blast
  obtain B'' where "Bet A D' B''" and "Cong D' B'' D B"
    using segment_construction by blast
  hence "Cong B C' B'' C"
    using assms(3) between_exchange3 between_symmetry cong_4312 cong_inner_transitivity l2_11_b
      \<open>Bet A C D'\<close> \<open>Bet A D C'\<close> \<open>Cong C D' C D\<close> \<open>Cong D C' C D\<close> by meson
  hence "Cong B B' B'' B"
    by (meson Bet_cases assms(2) assms(3) between_exchange4 between_inner_transitivity l2_11_b
        \<open>Bet A C D'\<close> \<open>Bet A C' B'\<close> \<open>Bet A D C'\<close> \<open>Bet A D' B''\<close> \<open>Cong C' B' C B\<close>)
  hence "B'' = B'"
    by (meson assms(1) assms(2) assms(3) between_exchange4 cong_inner_transitivity 
        construction_uniqueness not_cong_2134 \<open>Bet A C D'\<close> \<open>Bet A C' B'\<close> \<open>Bet A D C'\<close> 
        \<open>Bet A D' B''\<close>)
  have "B C D' Cong3 B' C' D" 
  proof -
    have "Cong B C B' C'" 
      using \<open>Cong C' B' C B\<close> not_cong_4321 by blast
    moreover
    have "Bet B' C' D" 
      using \<open>Bet A C' B'\<close> \<open>Bet A D C'\<close> between_exchange3 between_symmetry by blast
    have "Bet B C D'" 
      using \<open>Bet A C D'\<close> assms(2) between_exchange3 by blast
    moreover have "Cong B D' B' D" 
      using l2_11  Cong_cases \<open>Bet B C D'\<close> \<open>Bet B' C' D\<close> \<open>Cong C D' C D\<close> \<open>Cong D C' C D\<close> 
        cong_transitivity \<open>Cong B C B' C'\<close> by blast
    moreover have "Cong C D' C' D" 
      by (metis Cong_cases \<open>Cong C D' C D\<close> \<open>Cong D C' C D\<close> cong_transitivity)
    ultimately show ?thesis 
      using Cong3_def by blast
  qed
  have "Bet B C D'"
    using assms(2) between_exchange3 \<open>Bet A C D'\<close> by blast
  have "Cong C D' C' D" 
    by (metis Cong_perm cong_transitivity \<open>Cong C D' C D\<close> \<open>Cong D C' C D\<close>)
  hence "B C D' C' FSC B' C' D C" 
    using FSC_def \<open>B C D' Cong3 B' C' D\<close> \<open>B'' = B'\<close> \<open>Bet B C D'\<close> \<open>Cong B C' B'' C\<close> 
      bet_col cong_pseudo_reflexivity by presburger
  hence "Cong D' C' D C"
    using cong_identity l4_16 \<open>B'' = B'\<close> \<open>Cong C' B' C B\<close> \<open>Cong D' B'' D B\<close> by blast
  obtain E where "Bet C E C'" and "Bet D E D'"
    using between_trivial2 l3_17 \<open>Bet A C D'\<close> \<open>Bet A D C'\<close> by blast
  hence "D E D' C IFSC D E D' C'"
    by (meson IFSC_def cong_reflexivity cong_3421 cong_inner_transitivity \<open>Cong C D' C D\<close> 
        \<open>Cong D C' C D\<close> \<open>Cong D' C' D C\<close>)
  hence "Cong E C E C'"
    using l4_2 by auto
  have "C E C' D IFSC C E C' D'"
    using IFSC_def cong_reflexivity cong_3421 cong_inner_transitivity
    by (meson \<open>Bet C E C'\<close> \<open>Cong C D' C' D\<close> \<open>D E D' C IFSC D E D' C'\<close>)
  hence "Cong E D E D'"
    using l4_2 by auto
  obtain P where "Bet C' C P" and "Cong C P C D'"
    using segment_construction by blast
  obtain R where "Bet D' C R" and "Cong C R C E"
    using segment_construction by blast
  obtain Q where "Bet P R Q" and "Cong R Q R P"
    using segment_construction by blast
  have "D' C R P FSC P C E D'"
    by (meson Bet_perm Cong3_def FSC_def \<open>Bet C E C'\<close> \<open>Bet C' C P\<close> \<open>Cong C P C D'\<close>  \<open>Bet D' C R\<close> 
        \<open>Cong C R C E\<close> bet_col between_exchange3 cong_pseudo_reflexivity l2_11_b not_cong_4321)
  have "Cong R P E D'" 
    by (metis \<open>Cong C P C D'\<close> \<open>Cong C R C E\<close> \<open>D' C R P FSC P C E D'\<close> cong_commutativity 
        cong_diff l4_16)
  have "Cong R Q E D" 
    by (metis cong_symmetry cong_transitivity \<open>Cong E D E D'\<close> \<open>Cong R P E D'\<close> \<open>Cong R Q R P\<close>)
  have "D' E D C FSC P R Q C" 
    by (meson Cong3_def FSC_def \<open>Bet D E D'\<close> \<open>Bet P R Q\<close> \<open>Cong C P C D'\<close> \<open>Cong C R C E\<close> 
        \<open>Cong R P E D'\<close> \<open>Cong R Q E D\<close> bet_col between_symmetry l2_11_b not_cong_3412 
        not_cong_4321)
  have "Cong D C Q C" 
    by (metis \<open>Cong E D E D'\<close> \<open>D' E D C FSC P R Q C\<close> between_trivial2 cong_identity l4_16 l4_16R2)
  have "Cong C P C Q" 
    by (meson \<open>Cong C D' C D\<close> \<open>Cong C P C D'\<close> \<open>Cong D C Q C\<close> cong_transitivity not_cong_2143)
  have "Bet A C D \<or> Bet A D C"
  proof cases
    assume "R = C"
    thus ?thesis 
      using \<open>Bet A D C'\<close> \<open>Cong C R C E\<close> \<open>Cong E C E C'\<close> cong_reverse_identity by blast
  next
    assume "R \<noteq> C"
    {
      have "Cong D' P D' Q"
      proof -
        have "Col R C D'"
          by (simp add: \<open>Bet D' C R\<close> \<open>Cong C R C E\<close> bet_col between_symmetry)
        have "Cong R P R Q" 
          by (metis Cong_cases \<open>Cong R Q R P\<close>)
        have "Cong C P C Q"
          by (simp add: \<open>Cong C P C Q\<close>)
        thus ?thesis
          using \<open>Col R C D'\<close> \<open>Cong R P R Q\<close> \<open>R \<noteq> C\<close> l4_17 by blast
      qed
      hence "Cong B P B Q"  
        by (metis Col_def \<open>Bet B C D'\<close> \<open>Cong C P C D'\<close> \<open>Cong C P C Q\<close> cong_identity 
            cong_reflexivity l4_17 l4_19 not_bet_distincts)
      have "Cong B' P B' Q" 
        by (metis cong_diff_2 cong_diff_4 \<open>B'' = B'\<close> \<open>Bet A C D'\<close> \<open>Bet A D' B''\<close> \<open>Bet C E C'\<close> 
            \<open>Cong C D' C D\<close> \<open>Cong C D' C' D\<close> \<open>Cong C P C Q\<close> \<open>Cong C R C E\<close> \<open>Cong D' P D' Q\<close> \<open>R \<noteq> C\<close> 
            between_exchange3 between_identity cong_reflexivity five_segment)
      have "Cong C' P C' Q"
      proof -
        have "Bet B C' B'" 
          using \<open>Bet A C' B'\<close> \<open>Bet A D C'\<close> assms(3) between_exchange3 between_exchange4 by blast
        thus ?thesis
          by (metis Col_def \<open>Cong B P B Q\<close> \<open>Cong B' P B' Q\<close> between_equality l4_17 
              not_bet_distincts)
      qed
      have "Cong P P P Q" 
        by (metis cong_diff \<open>Bet C E C'\<close> \<open>Bet C' C P\<close> \<open>Cong C P C Q\<close> \<open>Cong C R C E\<close> 
            \<open>Cong C' P C' Q\<close> \<open>R \<noteq> C\<close> bet_col between_equality_2 between_trivial2 l4_17)
      thus ?thesis 
        using \<open>Bet A C D'\<close> \<open>Bet P R Q\<close> \<open>Cong R P E D'\<close> \<open>Cong R Q E D\<close> between_identity 
          cong_reverse_identity by blast
    }
    hence "R \<noteq> C \<longrightarrow> Bet A C D \<or> Bet A D C" by blast
  qed
  thus ?thesis
    by simp
qed

lemma l5_2:
  assumes "A \<noteq> B" and
    "Bet A B C" and
    "Bet A B D"
  shows  "Bet B C D \<or> Bet B D C"
  using assms(1) assms(2) assms(3) between_exchange3 l5_1 by blast

lemma segment_construction_2:
  assumes  "A \<noteq> Q"
  shows "\<exists> X. ((Bet Q A X \<or> Bet Q X A) \<and> Cong Q X B C)"
proof -
  obtain A' where "Bet A Q A'" and "Cong Q A' A Q"
    using segment_construction by blast
  obtain X where "Bet A' Q X" and "Cong Q X B C"
    using segment_construction by blast
  thus ?thesis
    by (metis \<open>Bet A Q A'\<close> \<open>Cong Q A' A Q\<close> cong_diff_4 between_symmetry l5_2)
qed

lemma l5_3:
  assumes "Bet A B D" and
    "Bet A C D"
  shows "Bet A B C \<or> Bet A C B"
  by (metis Bet_perm assms(1) assms(2) between_inner_transitivity l5_2 
      point_construction_different)

lemma bet3__bet:
  assumes "Bet A B E" and
    "Bet A D E" and
    "Bet B C D"
  shows "Bet A C E"
  by (meson assms(1) assms(2) assms(3) between_exchange2 between_symmetry l5_3)

lemma le_bet:
  assumes "C D Le A B"
  shows "\<exists> X. (Bet A X B \<and> Cong A X C D)"
  by (meson Le_def assms cong_symmetry)

lemma l5_5_1:
  assumes "A B Le C D"
  shows "\<exists> X. (Bet A B X \<and> Cong A X C D)"
proof -
  obtain P where "Bet C P D" and "Cong A B C P"
    using Le_def assms by blast
  obtain X where "Bet A B X" and "Cong B X P D"
    using segment_construction by blast
  thus ?thesis 
    by (meson \<open>Bet C P D\<close> \<open>Cong A B C P\<close> l2_11_b)
qed

lemma l5_5_2:
  assumes "\<exists> X. (Bet A B X \<and> Cong A X C D)"
  shows "A B Le C D"
proof -
  obtain P where "Bet A B P" and "Cong A P C D"
    using assms by blast
  then obtain B' where "Bet C B' D" and "A B P Cong3 C B' D"
    using l4_5 by blast
  thus ?thesis
    using Cong3_def Le_def by blast
qed

lemma l5_6:
  assumes "A B Le C D" and
    "Cong A B A' B'" and
    "Cong C D C' D'"
  shows "A' B' Le C' D'"
  by (meson Cong3_def Le_def assms(1) assms(2) assms(3) cong_inner_transitivity l4_5)

lemma le_reflexivity:
  shows "A B Le A B"
  using between_trivial cong_reflexivity l5_5_2 by blast

lemma le_transitivity:
  assumes "A B Le C D" and
    "C D Le E F"
  shows "A B Le E F"
  by (meson assms(1) assms(2) between_exchange4 cong_reflexivity l5_5_1 l5_5_2 l5_6 le_bet)

lemma between_cong:
  assumes "Bet A C B" and
    "Cong A C A B"
  shows "C = B" 
  by (metis assms(1) assms(2) between_trivial cong_diff_2 cong_reflexivity l4_3_1)

lemma cong3_symmetry:
  assumes "A B C Cong3 A' B' C'"
  shows "A' B' C' Cong3 A B C"
  by (simp add: assms cong_3_sym)

lemma between_cong_2:
  assumes "Bet A D B" and
    "Bet A E B" and 
    "Cong A D A E"
  shows "D = E" 
  using l5_3 by (metis Cong_cases assms(1) assms(2) assms(3) between_cong)

lemma between_cong_3:
  assumes "A \<noteq> B"
    and "Bet A B D"
    and "Bet A B E"
    and "Cong B D B E"
  shows "D = E"
  by (meson assms(1) assms(2) assms(3) assms(4) cong_reflexivity construction_uniqueness)

lemma le_anti_symmetry:
  assumes "A B Le C D" and
    "C D Le A B"
  shows "Cong A B C D" 
proof -
  obtain Y where "Bet C Y D" and "Cong A B C Y" 
    using Le_def assms(1) by blast
  obtain T where "Bet C D T" and "Cong C T A B" 
    using assms(2) l5_5_1 by blast
  have "Cong C Y C T" 
    by (metis cong_transitivity \<open>Cong A B C Y\<close> \<open>Cong C T A B\<close> cong_symmetry)
  have "Bet C Y T" 
    using \<open>Bet C D T\<close> \<open>Bet C Y D\<close> between_exchange4 by blast
  have "Y = T" 
    using \<open>Bet C Y T\<close> \<open>Cong C Y C T\<close> between_cong by auto
  moreover have "T = D" 
    using \<open>Bet C D T\<close> \<open>Bet C Y D\<close> between_equality_2 calculation by auto
  ultimately show ?thesis 
    using \<open>Cong A B C Y\<close> by auto
qed

lemma cong_dec:
  shows "Cong A B C D \<or> \<not> Cong A B C D"
  by simp

lemma bet_dec:
  shows "Bet A B C  \<or> \<not> Bet A B C"
  by simp

lemma col_dec:
  shows "Col A B C \<or> \<not> Col A B C"
  by simp

lemma le_trivial:
  shows "A A Le C D"
  using Le_def between_trivial2 cong_trivial_identity by blast

lemma le_cases:
  shows "A B Le C D \<or> C D Le A B" 
  by (metis Cong_cases Le_def l5_5_2 le_trivial segment_construction_2)

lemma le_zero:
  assumes "A B Le C C"
  shows "A = B"
  by (metis assms cong_diff_4 le_anti_symmetry le_trivial)

lemma le_diff:
  assumes "A \<noteq> B" and "A B Le C D"
  shows "C \<noteq> D"
  using assms(1) assms(2) le_zero by blast

lemma lt_diff:
  assumes "A B Lt C D"
  shows "C \<noteq> D"
  using Lt_def assms cong_trivial_identity le_zero by blast

lemma bet_cong_eq:
  assumes "Bet A B C" and
    "Bet A C D" and
    "Cong B C A D"
  shows "C = D \<and> A = B"
proof -
  have "Bet C B A"
    using Bet_perm assms(1) by blast
  thus ?thesis
    by (metis Cong_perm Le_def assms(2) assms(3) between_cong cong_pseudo_reflexivity 
        le_anti_symmetry)
qed

lemma cong__le:
  assumes "Cong A B C D"
  shows "A B Le C D"
  using Le_def assms between_trivial by blast

lemma cong__le3412:
  assumes "Cong A B C D"
  shows "C D Le A B"
  using assms cong__le cong_symmetry by blast

lemma le1221:
  shows "A B Le B A"
  by (simp add: cong__le cong_pseudo_reflexivity)

lemma le_left_comm:
  assumes "A B Le C D"
  shows "B A Le C D"
  using assms le1221 le_transitivity by blast

lemma le_right_comm:
  assumes "A B Le C D"
  shows "A B Le D C"
  by (meson assms cong_right_commutativity l5_5_1 l5_5_2)

lemma le_comm:
  assumes "A B Le C D"
  shows "B A Le D C"
  using assms le_left_comm le_right_comm by blast

lemma ge_left_comm:
  assumes "A B Ge C D"
  shows "B A Ge C D"
  by (meson Ge_def assms le_right_comm)

lemma ge_right_comm:
  assumes "A B Ge C D"
  shows "A B Ge D C"
  using Ge_def assms le_left_comm by presburger

lemma ge_comm0:
  assumes "A B Ge C D"
  shows "B A Ge D C"
  by (meson assms ge_left_comm ge_right_comm)

lemma lt_right_comm:
  assumes "A B Lt C D"
  shows "A B Lt D C"
  using Lt_def assms le_right_comm not_cong_1243 by blast

lemma lt_left_comm:
  assumes "A B Lt C D"
  shows "B A Lt C D"
  using Lt_def assms le_comm lt_right_comm not_cong_2143 by blast

lemma lt_comm:
  assumes "A B Lt C D"
  shows "B A Lt D C"
  using assms lt_left_comm lt_right_comm by blast

lemma gt_left_comm0:
  assumes "A B Gt C D"
  shows "B A Gt C D"
  by (meson Gt_def assms lt_right_comm)

lemma gt_right_comm:
  assumes "A B Gt C D"
  shows "A B Gt D C"
  using Gt_def assms lt_left_comm by presburger

lemma gt_comm:
  assumes "A B Gt C D"
  shows "B A Gt D C"
  by (meson assms gt_left_comm0 gt_right_comm)

lemma cong2_lt__lt:
  assumes "A B Lt C D" and
    "Cong A B A' B'" and
    "Cong C D C' D'"
  shows "A' B' Lt C' D'"
  by (meson Lt_def assms(1) assms(2) assms(3) l5_6 le_anti_symmetry not_cong_3412)

lemma fourth_point:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Col A B P" and
    "Bet A B C"
  shows "Bet P A B \<or> Bet A P B \<or> Bet B P C \<or> Bet B C P"
  by (metis Col_def l5_2 assms(3) assms(4) between_symmetry)

lemma third_point:
  assumes "Col A B P"
  shows "Bet P A B \<or> Bet A P B \<or> Bet A B P"
  using Col_def assms between_symmetry by blast

lemma l5_12_a:
  assumes "Bet A B C"
  shows "A B Le A C \<and> B C Le A C"
  using assms between_symmetry cong_left_commutativity cong_reflexivity l5_5_2 le_left_comm 
  by blast

lemma bet__le1213:
  assumes "Bet A B C"
  shows "A B Le A C"
  using assms l5_12_a by blast

lemma bet__le2313:
  assumes "Bet A B C"
  shows "B C Le A C"
  by (simp add: assms l5_12_a)

lemma bet__lt1213:
  assumes "B \<noteq> C" and
    "Bet A B C"
  shows "A B Lt A C"
  using Lt_def assms(1) assms(2) bet__le1213 between_cong by blast

lemma bet__lt2313:
  assumes "A \<noteq> B" and
    "Bet A B C"
  shows "B C Lt A C"
  using Lt_def assms(1) assms(2) bet__le2313 bet_cong_eq l5_1 by blast

lemma l5_12_b:
  assumes "Col A B C" and
    "A B Le A C" and
    "B C Le A C"
  shows "Bet A B C"
proof -
  {
    assume "Bet B C A"
    hence ?thesis 
      using assms(2) between_cong between_symmetry l5_12_a le_anti_symmetry by blast
  }
  moreover
  {
    assume "Bet C A B"
    hence ?thesis 
      by (metis assms(3) bet_cong_eq between_trivial2 l5_12_a le_anti_symmetry le_comm)
  }
  ultimately show ?thesis 
    using Col_def assms(1) by blast
qed

lemma bet_le_eq:
  assumes "Bet A B C"
    and "A C Le B C"
  shows "A = B"
  by (meson assms(1) assms(2) bet__le2313 bet_cong_eq l5_1 le_anti_symmetry)

lemma or_lt_cong_gt:
  "A B Lt C D \<or> A B Gt C D \<or> Cong A B C D"
  by (meson Gt_def Lt_def cong_symmetry local.le_cases)

lemma lt__le:
  assumes "A B Lt C D"
  shows "A B Le C D"
  using Lt_def assms by blast

lemma le1234_lt__lt:
  assumes "A B Le C D" and
    "C D Lt E F"
  shows "A B Lt E F"
  by (meson Lt_def assms(1) assms(2) cong__le3412 le_anti_symmetry le_transitivity)

lemma le3456_lt__lt:
  assumes "A B Lt C D" and
    "C D Le E F"
  shows "A B Lt E F"
  by (meson Lt_def assms(1) assms(2) cong2_lt__lt cong_reflexivity le1234_lt__lt)

lemma lt_transitivity:
  assumes "A B Lt C D" and
    "C D Lt E F"
  shows "A B Lt E F"
  using Lt_def assms(1) assms(2) le1234_lt__lt by blast

lemma not_and_lt:
  "\<not> (A B Lt C D \<and> C D Lt A B)"
  by (simp add: Lt_def le_anti_symmetry)

lemma nlt:
  "\<not> A B Lt A B"
  using not_and_lt by blast

lemma le__nlt:
  assumes "A B Le C D"
  shows "\<not> C D Lt A B"
  using assms le3456_lt__lt nlt by blast

lemma cong__nlt:
  assumes "Cong A B C D"
  shows "\<not> A B Lt C D"
  by (simp add: Lt_def assms)

lemma nlt__le:
  assumes "\<not> A B Lt C D"
  shows "C D Le A B"
  using Lt_def assms cong__le3412 local.le_cases by blast

lemma lt__nle:
  assumes "A B Lt C D"
  shows "\<not> C D Le A B"
  using assms le__nlt by blast

lemma nle__lt:
  assumes "\<not> A B Le C D"
  shows "C D Lt A B"
  using assms nlt__le by blast

lemma lt1123:
  assumes "B \<noteq> C"
  shows "A A Lt B C"
  using assms le_diff nle__lt by blast

lemma bet2_le2__le_R1:
  assumes "Bet a P b" and
    "Bet A Q B" and
    "P a Le Q A" and
    "P b Le Q B" and
    "B = Q"
  shows "a b Le A B"
  by (metis assms(3) assms(4) assms(5) le_comm le_diff)

lemma bet2_le2__le_R2:
  assumes "Bet a Po b" and
    "Bet A PO B" and
    "Po a Le PO A" and
    "Po b Le PO B" and
    "A \<noteq> PO" and
    "B \<noteq> PO"
  shows "a b Le A B"
proof -
  obtain b' where "Bet A PO b'" and "Cong PO b' b Po"
    using segment_construction by blast
  obtain a' where "Bet B PO a'" and "Cong PO a' a Po"
    using segment_construction by blast
  obtain a'' where "Bet PO a'' A" and "Cong Po a PO a''"
    using Le_def assms(3) by blast
  have "Cong PO a'' a Po" 
    using Cong_cases \<open>Cong Po a PO a''\<close> by blast
  hence "a' = a''" 
    by (meson Bet_perm \<open>Bet B PO a'\<close> \<open>Bet PO a'' A\<close> \<open>Cong PO a' a Po\<close> assms(2) assms(6) 
        between_inner_transitivity construction_uniqueness)
  have "B a' Le B A" 
    by (metis \<open>Bet B PO a'\<close> \<open>Bet PO a'' A\<close> \<open>a' = a''\<close> assms(2) bet__le1213 bet_le_eq le_comm 
        le_cases outer_transitivity_between2)
  obtain b'' where "Bet PO b'' B" and "Cong Po b PO b''"
    using Le_def assms(4) by blast
  hence "b' = b''"
    using assms(2) assms(5) between_inner_transitivity cong_right_commutativity 
      construction_uniqueness not_cong_3412 by (meson \<open>Bet A PO b'\<close> \<open>Cong PO b' b Po\<close>)
  hence "a' b' Le a' B"
    using Bet_cases bet__le1213 between_exchange2 \<open>Bet B PO a'\<close> \<open>Bet PO b'' B\<close> by blast
  hence "a' b' Le A B"
    using le_comm le_transitivity by (meson \<open>B a' Le B A\<close>)
  have "Cong a' b' a b" 
  proof -
    have "Bet a' PO b'" 
      using Bet_cases \<open>Bet A PO b'\<close> \<open>Bet PO a'' A\<close> \<open>a' = a''\<close> between_exchange3 by blast
    moreover have "Cong a' PO a Po" 
      using Cong_cases \<open>Cong PO a' a Po\<close> by auto
    moreover have "Cong PO b' Po b" 
      using \<open>Cong PO b' b Po\<close> not_cong_1243 by blast
    ultimately show ?thesis 
      using assms(1) l2_11_b by blast
  qed
  thus ?thesis
    using \<open>a' b' Le A B\<close> cong_reflexivity l5_6 by blast
qed

lemma bet2_le2__le:
  assumes "Bet a P b" and
    "Bet A Q B" and
    "P a Le Q A" and
    "P b Le Q B"
  shows "a b Le A B"
proof cases
  assume "A = Q"
  thus ?thesis
    using assms(3) assms(4) le_diff by force
next
  assume "\<not> A = Q"
  thus ?thesis
    using assms(1) assms(2) assms(3) assms(4) bet2_le2__le_R1 bet2_le2__le_R2 by blast
qed

lemma Le_cases:
  assumes "A B Le C D \<or> B A Le C D \<or> A B Le D C \<or> B A Le D C"
  shows "A B Le C D"
  using assms le_left_comm le_right_comm by blast

lemma Lt_cases:
  assumes "A B Lt C D \<or> B A Lt C D \<or> A B Lt D C \<or> B A Lt D C"
  shows "A B Lt C D"
  using assms lt_comm lt_left_comm by blast

lemma bet_out:
  assumes "B \<noteq> A" and
    "Bet A B C"
  shows  "A Out B C"
  using Out_def assms(1) assms(2) bet_neq12__neq by fastforce

lemma bet_out_1:
  assumes "B \<noteq> A" and
    "Bet C B A"
  shows "A Out B C"
  by (simp add: assms(1) assms(2) bet_out between_symmetry)

lemma out_dec:
  shows "P Out A B \<or> \<not> P Out A B"
  by simp

lemma out_diff1:
  assumes "A Out B C"
  shows "B \<noteq> A"
  using Out_def assms by auto

lemma out_diff2:
  assumes "A Out B C"
  shows "C \<noteq> A"
  using Out_def assms by auto

lemma out_distinct:
  assumes "A Out B C"
  shows  "B \<noteq> A \<and> C \<noteq> A"
  using assms out_diff1 out_diff2 by auto

lemma out_col:
  assumes "A Out B C"
  shows "Col A B C"
  using Col_def Out_def assms between_symmetry by auto

lemma l6_2:
  assumes "A \<noteq> P" and
    "B \<noteq> P" and
    "C \<noteq> P" and
    "Bet A P C"
  shows "Bet B P C \<longleftrightarrow> P Out A B"
proof -
  {
    assume "Bet B P C"
    hence "P Out A B" 
      using Out_def assms(1) assms(2) assms(3) assms(4) between_symmetry l5_2 by presburger
  }
  moreover
  {
    assume "P Out A B" 
    hence "Bet B P C" 
      by (metis Bet_perm Out_def between_exchange3 outer_transitivity_between2 assms(4))
  }
  ultimately show ?thesis 
    by auto
qed

lemma bet_out__bet:
  assumes "Bet A P C" and
    "P Out A B"
  shows "Bet B P C"
  by (metis l6_2 assms(1) assms(2) not_bet_distincts out_diff1)

lemma l6_3_1:
  assumes "P Out A B"
  shows "A \<noteq> P \<and> B \<noteq> P \<and> (\<exists> C. (C \<noteq> P \<and> Bet A P C \<and> Bet B P C))"
  using assms bet_out__bet out_diff1 out_diff2 point_construction_different by fastforce

lemma l6_3_2:
  assumes "A \<noteq> P" and
    "B \<noteq> P" and
    "\<exists> C. (C \<noteq> P \<and> Bet A P C \<and> Bet B P C)"
  shows "P Out A B"
  using assms(1) assms(2) assms(3) l6_2 by blast

lemma l6_4_1:
  assumes "P Out A B" and
    "Col A P B"
  shows "\<not> Bet A P B"
  using Out_def assms(1) between_equality between_symmetry by fastforce

lemma l6_4_2:
  assumes "Col A P B"
    and "\<not> Bet A P B"
  shows "P Out A B"
  by (metis Out_def assms(1) assms(2) bet_out col_permutation_1 third_point)

lemma out_trivial:
  assumes "A \<noteq> P"
  shows "P Out A A"
  by (simp add: assms bet_out_1 between_trivial2)

lemma l6_6:
  assumes "P Out A B"
  shows "P Out B A"
  using Out_def assms by auto

lemma l6_7:
  assumes "P Out A B" and
    "P Out B C"
  shows "P Out A C" 
  by (metis assms(1) assms(2) l6_2 l6_6 out_diff2 point_construction_different)

lemma bet_out_out_bet:
  assumes "Bet A B C" and
    "B Out A A'" and
    "B Out C C'"
  shows "Bet A' B C'"
  by (metis Out_def assms(1) assms(2) assms(3) bet_out__bet between_inner_transitivity 
      outer_transitivity_between)

lemma out2_bet_out:
  assumes "B Out A C" and
    "B Out X P" and
    "Bet A X C"
  shows "B Out A P \<and> B Out C P"
proof -
  have "Bet B A C \<or> Bet B C A" 
    using Out_def assms(1) by auto
  {
    assume "Bet B A C"
    have "B Out A P \<and> B Out C P" 
      by (metis Out_def between_inner_transitivity l6_7 \<open>Bet B A C\<close> assms(1) assms(2) assms(3))
  }
  moreover
  {
    assume "Bet B C A" 
    hence "B Out A P" 
      by (metis Bet_perm bet3__bet bet_out_1 l5_1 l6_6 l6_7 out_distinct assms(2) assms(3))
  }
  moreover
  {
    assume "Bet B C A" 
    hence "B Out C P" 
      using \<open>Bet B C A\<close> assms(1) calculation(2) l6_6 l6_7 by blast
  }
  ultimately show ?thesis 
    using \<open>Bet B A C \<or> Bet B C A\<close> by blast
qed

lemma l6_11_uniqueness:
  assumes "A Out X R" and
    "Cong A X B C" and
    "A Out Y R" and
    "Cong A Y B C"
  shows "X = Y"
  by (metis Out_def assms(1) assms(2) assms(3) assms(4) between_cong cong_symmetry 
      cong_transitivity l6_6 l6_7)

lemma l6_11_existence:
  assumes "R \<noteq> A" and
    "B \<noteq> C"
  shows "\<exists> X. (A Out X R \<and> Cong A X B C)"
  by (metis Out_def assms(1) assms(2) cong_reverse_identity segment_construction_2)


lemma segment_construction_3:
  assumes "A \<noteq> B" and
    "X \<noteq> Y"
  shows "\<exists> C. (A Out B C \<and> Cong A C X Y)"
  by (metis assms(1) assms(2) l6_11_existence l6_6)

lemma l6_13_1:
  assumes "P Out A B" and
    "P A Le P B"
  shows "Bet P A B"
  by (metis Out_def assms(1) assms(2) bet__lt1213 le__nlt)

lemma l6_13_2:
  assumes "P Out A B" and
    "Bet P A B"
  shows "P A Le P B"
  by (simp add: assms(2) bet__le1213)

lemma l6_16_1:
  assumes "P \<noteq> Q" and
    "Col S P Q" and
    "Col X P Q"
  shows "Col X P S"
proof cases
  assume "S = P"
  thus ?thesis 
    using col_trivial_2 by blast
next
  assume "S \<noteq> P" 
  {
    assume "Bet S P Q" and "Bet X P Q"  
    hence "Col X P S" 
      by (metis Col_cases assms(1) col_trivial_1 l6_2 out_col)
  }
  moreover
  {
    assume "Bet P Q S" and "Bet X P Q"  
    hence "Col X P S" 
      using assms(1) bet_col outer_transitivity_between by blast
  }
  moreover
  {
    assume "Bet Q S P" and "Bet X P Q"  
    hence "Col X P S" 
      using bet_col between_inner_transitivity between_symmetry by blast
  }
  moreover
  {
    assume "Bet P Q S" and "Bet P Q X"  
    hence "Col X P S" 
      by (meson Col_def assms(1) l5_1 not_col_permutation_5)
  }
  moreover
  {
    assume "Bet Q S P" and "Bet P Q X"  
    hence "Col X P S" 
      using Col_def between_exchange4 between_symmetry by blast
  }
  moreover
  {
    assume "Bet S P Q" and "Bet P Q X"  
    hence "Col X P S" 
      using Bet_cases assms(1) bet_col outer_transitivity_between by blast
  }
  moreover
  {
    assume "Bet P Q S" and "Bet Q X P"  
    hence "Col X P S" 
      using Bet_cases Col_def between_exchange2 by blast
  }
  moreover
  {
    assume "Bet Q S P" and "Bet Q X P"  
    hence "Col X P S"
      by (meson Col_def between_exchange3 l5_3 not_col_permutation_3)
  }
  moreover
  {
    assume "Bet S P Q" and "Bet Q X P"  
    hence "Col X P S" 
      using bet_col between_exchange3 between_symmetry by blast
  }
  ultimately show ?thesis 
    by (metis Col_def assms(2) assms(3))
qed

lemma col_transitivity_1:
  assumes "P \<noteq> Q" and
    "Col P Q A" and
    "Col P Q B"
  shows "Col P A B"
  by (meson l6_16_1 assms(1) assms(2) assms(3) not_col_permutation_2)

lemma col_transitivity_2:
  assumes "P \<noteq> Q" and
    "Col P Q A" and
    "Col P Q B"
  shows "Col Q A B"
  by (metis col_transitivity_1 assms(1) assms(2) assms(3) not_col_permutation_4)

lemma l6_21:
  assumes "\<not> Col A B C" and
    "C \<noteq> D" and
    "Col A B P" and
    "Col A B Q" and
    "Col C D P" and
    "Col C D Q"
  shows "P = Q"
  by (metis assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) col_transitivity_1 l6_16_1 
      not_col_distincts)

lemma col2__eq:
  assumes "Col A X Y" and
    "Col B X Y" and
    "\<not> Col A X B"
  shows  "X = Y"
  using assms(1) assms(2) assms(3) l6_16_1 by blast

lemma not_col_exists:
  assumes  "A \<noteq> B"
  shows "\<exists> C. \<not> Col A B C"
  by (metis Col_def assms col_transitivity_2 lower_dim_ex)

lemma col3:
  assumes "X \<noteq> Y" and
    "Col X Y A" and
    "Col X Y B" and
    "Col X Y C"
  shows "Col A B C"
  by (metis assms(1) assms(2) assms(3) assms(4) col_transitivity_2)

lemma colx:
  assumes "A \<noteq> B" and
    "Col X Y A" and
    "Col X Y B" and
    "Col A B C"
  shows "Col X Y C"
  by (metis assms(1) assms(2) assms(3) assms(4) l6_21 not_col_distincts)

lemma out2__bet:
  assumes "A Out B C" and
    "C Out A B"
  shows "Bet A B C"
  by (metis Out_def assms(1) assms(2) between_equality between_symmetry)

lemma bet2_le2__le1346:
  assumes "Bet A B C" and
    "Bet A' B' C'" and
    "A B Le A' B'" and
    "B C Le B' C'"
  shows "A C Le A' C'"
  using Le_cases assms(1) assms(2) assms(3) assms(4) bet2_le2__le by blast

lemma bet2_le2__le2356_R1:
  assumes "Bet A A C" and
    "Bet A' B' C'" and
    "A A Le A' B'" and
    "A' C' Le A C"
  shows "B' C' Le A C"
  using assms(2) assms(4) bet__le2313 le3456_lt__lt lt__nle nlt__le by blast

lemma bet2_le2__le2356_R2:
  assumes "A \<noteq> B" and
    "Bet A B C" and
    "Bet A' B' C'" and
    "A B Le A' B'" and
    "A' C' Le A C"
  shows "B' C' Le B C"
proof -
  obtain B0 where "Bet A B B0" and "Cong A B0 A' B'"
    using assms(4) l5_5_1 by blast
  hence "A \<noteq> B0"
    using assms(1) bet_neq12__neq by blast
  obtain C0 where "Bet A C0 C" and "Cong A' C' A C0"
    using Le_def assms(5) by blast
  hence "A \<noteq> C0"
    using assms(1) assms(3) assms(4) bet_neq12__neq cong_diff le_diff by blast
  hence "Bet A B0 C0" 
    by (metis Bet_cases Cong_cases \<open>A \<noteq> B0\<close> \<open>Bet A B B0\<close> \<open>Bet A C0 C\<close> \<open>Cong A B0 A' B'\<close> 
        \<open>Cong A' C' A C0\<close> assms(1) assms(2) assms(3) bet__le1213 between_inner_transitivity 
        cong__nlt l6_13_1 l6_2 le_transitivity nlt__le outer_transitivity_between 
        point_construction_different)
  have "B0 C0 Le B C0" 
    using \<open>Bet A B B0\<close> \<open>Bet A B0 C0\<close> bet__le2313 between_exchange3 by blast
  moreover have "B C0 Le B C" 
    by (meson \<open>Bet A B B0\<close> \<open>Bet A B0 C0\<close> \<open>Bet A C0 C\<close> bet__le1213 between_exchange3 
        between_exchange4)
  moreover have "Cong B0 C0 B' C'" 
    using \<open>Bet A B0 C0\<close> \<open>Cong A B0 A' B'\<close> \<open>Cong A' C' A C0\<close> assms(3) l4_3_1 not_cong_3412 by blast
  ultimately show ?thesis 
    using cong__le3412 le_transitivity by blast
qed

lemma bet2_le2__le2356:
  assumes "Bet A B C" and
    "Bet A' B' C'" and
    "A B Le A' B'" and
    "A' C' Le A C"
  shows "B' C' Le B C"
proof (cases)
  assume "A = B"
  thus ?thesis
    using assms(1) assms(2) assms(3) assms(4) bet2_le2__le2356_R1 by blast
next
  assume "\<not> A = B"
  thus ?thesis
    using assms(1) assms(2) assms(3) assms(4) bet2_le2__le2356_R2 by blast
qed

lemma bet2_le2__le1245:
  assumes "Bet A B C" and
    "Bet A' B' C'" and
    "B C Le B' C'" and
    "A' C' Le A C"
  shows "A' B' Le A B"
  using assms(1) assms(2) assms(3) assms(4) bet2_le2__le2356 between_symmetry le_comm by blast

lemma cong_preserves_bet:
  assumes "Bet B A' A0" and
    "Cong B A' E D'" and
    "Cong B A0 E D0" and
    "E Out D' D0"
  shows "Bet E D' D0"
  using l6_13_1 Tarski_neutral_dimensionless_axioms assms(1) assms(2) assms(3) assms(4)
    bet__le1213 l5_6 by fastforce

lemma out_cong_cong:
  assumes "B Out A A0" and
    "E Out D D0" and
    "Cong B A E D" and
    "Cong B A0 E D0"
  shows "Cong A A0 D D0"
  by (meson Out_def assms(1) assms(2) assms(3) assms(4) cong_4321 cong_symmetry l4_3_1 l5_6 
      l6_13_1 l6_13_2)

lemma not_out_bet:
  assumes "Col A B C" and
    "\<not> B Out A C"
  shows "Bet A B C"
  using assms(1) assms(2) l6_4_2 by blast

lemma or_bet_out:
  shows "Bet A B C \<or> B Out A C \<or> \<not> Col A B C"
  using not_out_bet by blast

lemma not_bet_out:
  assumes "Col A B C" and
    "\<not> Bet A B C"
  shows "B Out A C"
  by (simp add: assms(1) assms(2) l6_4_2)

lemma not_bet_and_out:
  shows  "\<not> (Bet A B C \<and> B Out A C)"
  using bet_col l6_4_1 by blast

lemma out_to_bet:
  assumes "Col A' B' C'" and
    "B Out A C \<longleftrightarrow> B' Out A' C'" and
    "Bet A B C"
  shows "Bet A' B' C'"
  using assms(1) assms(2) assms(3) not_bet_and_out or_bet_out by blast

lemma col_out2_col:
  assumes "Col A B C" and
    "B Out A AA" and
    "B Out C CC"
  shows "Col AA B CC" 
  by (metis Col_cases assms(1) assms(2) assms(3) col_transitivity_1 out_col out_diff1)

lemma bet2_out_out:
  assumes "B \<noteq> A" and
    "B' \<noteq> A" and
    "A Out C C'" and
    "Bet A B C" and
    "Bet A B' C'"
  shows "A Out B B'"
  by (meson assms(1) assms(2) assms(3) assms(4) assms(5) bet_out l6_6 l6_7)

lemma bet2__out:
  assumes "A \<noteq> B" and
    "A \<noteq> B'" and
    "Bet A B C"
    and "Bet A B' C"
  shows "A Out B B'"
  using Out_def assms(1) assms(2) assms(3) assms(4) l5_3 by auto

lemma out_bet_out_1:
  assumes "P Out A C" and
    "Bet A B C"
  shows "P Out A B"
  by (metis assms(1) assms(2) not_bet_and_out out2_bet_out out_trivial)

lemma out_bet_out_2:
  assumes "P Out A C" and
    "Bet A B C"
  shows "P Out B C"
  using assms(1) assms(2) l6_6 l6_7 out_bet_out_1 by blast

lemma out_bet__out:
  assumes "Bet P Q A" and
    "Q Out A B"
  shows "P Out A B"
  by (smt (verit, best) Out_def assms(1,2) bet3__bet l5_1 not_bet_and_out
      outer_transitivity_between2) 

lemma segment_reverse:
  assumes "Bet A B C "
  shows "\<exists> B'. Bet A B' C \<and> Cong C B' A B"
  by (metis Bet_perm Cong_perm assms bet_cong_eq cong_reflexivity segment_construction_2)

lemma diff_col_ex:
  shows "\<exists> C. A \<noteq> C \<and> B \<noteq> C \<and> Col A B C"
  by (metis bet_col bet_neq12__neq point_construction_different)

lemma diff_bet_ex3:
  assumes "Bet A B C"
  shows "\<exists> D. A \<noteq> D \<and> B \<noteq> D \<and> C \<noteq> D \<and> Col A B D"
  by (metis Col_def bet_out_1 between_trivial2 col_transitivity_1 l6_4_1 
      point_construction_different)

lemma diff_col_ex3:
  assumes "Col A B C"
  shows "\<exists> D. A \<noteq> D \<and> B \<noteq> D \<and> C \<noteq> D \<and> Col A B D"
  by (metis Bet_perm Col_def between_equality between_trivial2 point_construction_different)

lemma Out_cases:
  assumes "A Out B C \<or> A Out C B"
  shows "A Out B C"
  using assms l6_6 by blast

(** Existence of the sum *)

lemma ex_sums: 
  shows "\<exists> E F. A B C D SumS E F" 
proof -
  obtain R where "Bet A B R" and "Cong B R C D" 
    using segment_construction by blast
  thus ?thesis 
    using SumS_def cong_reflexivity by blast
qed

(** Commutativity of the sum. *)

lemma sums_sym:
  assumes "A B C D SumS E F" 
  shows "C D A B SumS E F" 
proof -
  obtain P Q R where "Bet P Q R" and "Cong P Q A B" and "Cong Q R C D" and "Cong P R E F" 
    using SumS_def assms by auto
  thus ?thesis 
    by (meson Cong_cases SumS_def between_symmetry)
qed

(** Unicity of the sum. *)

lemma sums2__cong56:
  assumes "A B C D SumS E F" and
    "A B C D SumS E' F'"
  shows "Cong E F E' F'" 
proof -
  obtain P Q R where "Bet P Q R" and "Cong P Q A B" and "Cong Q R C D" and "Cong P R E F"
    using SumS_def assms(1) by blast
  obtain P' Q' R' where "Bet P' Q' R'" and "Cong P' Q' A B" and "Cong Q' R' C D" and "Cong P' R' E' F'"
    using SumS_def assms(2) by blast
  have "Cong P Q P' Q'" 
    using Cong_cases \<open>Cong P Q A B\<close> \<open>Cong P' Q' A B\<close> cong_transitivity by blast
  have "Cong Q R Q' R'" 
    using Cong_cases \<open>Cong Q R C D\<close> \<open>Cong Q' R' C D\<close> cong_transitivity by blast
  hence "Cong P R P' R'" 
    using \<open>Bet P Q R\<close> \<open>Bet P' Q' R'\<close> \<open>Cong P Q P' Q'\<close> l2_11_b by blast
  hence "Cong P R E' F'" 
    using \<open>Cong P' R' E' F'\<close> cong_transitivity by blast
  thus ?thesis 
    using \<open>Cong P R E F\<close> cong_inner_transitivity by blast
qed
  (** Unicity of the difference of segments. *)

lemma sums2__cong12: 
  assumes "A B C D SumS E F" 
    and "A' B' C D SumS E F" 
  shows "Cong A B A' B'" 
proof -
  obtain P Q R where "Bet P Q R" and "Cong P Q A B" and "Cong Q R C D" and "Cong P R E F"
    using SumS_def assms(1) by blast
  obtain P' Q' R' where "Bet P' Q' R'" and "Cong P' Q' A' B'" and "Cong Q' R' C D" and "Cong P' R' E F"
    using SumS_def assms(2) by blast
  have "Cong P R P' R'" 
    using Cong_cases \<open>Cong P R E F\<close> \<open>Cong P' R' E F\<close> cong_transitivity by blast
  moreover have "Cong Q R Q' R'" 
    using \<open>Cong Q R C D\<close> \<open>Cong Q' R' C D\<close> cong_inner_transitivity cong_symmetry by blast
  ultimately have "Cong P Q P' Q'" 
    using \<open>Bet P Q R\<close> \<open>Bet P' Q' R'\<close> l4_3 by blast
  hence "Cong P Q A' B'" 
    using \<open>Cong P' Q' A' B'\<close> cong_transitivity by blast
  thus ?thesis 
    using \<open>Cong P Q A B\<close> cong_inner_transitivity by blast
qed
  (** Unicity of the difference of segments on the right. *)

lemma sums2__cong34: 
  assumes "A B C D SumS E F" and
    "A B C' D' SumS E F"
  shows "Cong C D C' D'" 
  using assms(1) assms(2) sums2__cong12 sums_sym by blast

(** Cong preserves SumS *)

lemma cong3_sums__sums:
  assumes "Cong A B A' B'" and
    "Cong C D C' D'" and 
    "Cong E F E' F'" and 
    "A B C D SumS E F" 
  shows "A' B' C' D' SumS E' F'" 
  by (meson SumS_def assms(1) assms(2) assms(3) assms(4) cong_inner_transitivity cong_symmetry)

(** The degenerate segments represent the additive identity *)

lemma sums123312:
  shows "A B C C SumS A B" 
  using SumS_def between_trivial cong_reflexivity cong_trivial_identity by blast

lemma sums__cong1245: 
  assumes "A B C C SumS D E"
  shows "Cong A B D E" 
  using assms sums123312 sums2__cong56 by blast

lemma sums__eq34: 
  assumes "A B C D SumS A B"
  shows "C = D" 
  using assms cong_reverse_identity sums123312 sums2__cong34 by blast

lemma sums112323: 
  shows "A A B C SumS B C" 
  by (simp add: sums123312 sums_sym)

lemma sums__cong2345: 
  assumes "A A B C SumS D E"
  shows "Cong B C D E" 
  using assms sums112323 sums2__cong56 by blast

lemma sums__eq12: 
  assumes "A B C D SumS C D" 
  shows "A = B" 
  using assms sums__eq34 sums_sym by blast

(** Some permutation properties *)

lemma sums_left_comm:
  assumes "A B C D SumS E F"
  shows "B A C D SumS E F" 
  using assms cong3_sums__sums cong_pseudo_reflexivity cong_reflexivity by blast

lemma sums_middle_comm: 
  assumes "A B C D SumS E F"
  shows "A B D C SumS E F" 
  using assms sums_left_comm sums_sym by blast

lemma sums_right_comm: 
  assumes "A B C D SumS E F" 
  shows "A B C D SumS F E" 
  using assms cong3_sums__sums cong_pseudo_reflexivity cong_reflexivity by blast

lemma sums_comm: 
  assumes "A B C D SumS E F"
  shows "B A D C SumS F E" 
  using assms cong3_sums__sums cong_pseudo_reflexivity by blast

(** Basic case of sum *)

lemma bet__sums: 
  assumes "Bet A B C" 
  shows "A B B C SumS A C"
  using SumS_def assms cong_reflexivity by blast

lemma sums_assoc_1:
  assumes "A B C D SumS G H" and
    "C D E F SumS I J" and
    "G H E F SumS K L" 
  shows "A B I J SumS K L" 
proof -
  obtain P Q R where "Bet P Q R" and "Cong P Q A B" and 
    "Cong Q R C D" and "Cong P R G H"
    using assms(1) SumS_def by fastforce
  obtain S where "Bet P R S" and "Cong R S E F" 
    using segment_construction by blast
  hence "Bet P Q S" 
    using \<open>Bet P Q R\<close> between_exchange4 by blast
  moreover have "Cong Q S I J" 
    using SumS_def \<open>Bet P Q R\<close> \<open>Bet P R S\<close> \<open>Cong Q R C D\<close> \<open>Cong R S E F\<close> assms(2) between_exchange3 cong_reflexivity sums2__cong56 by blast
  moreover have "Cong P S K L" 
    using SumS_def \<open>Bet P R S\<close> \<open>Cong P R G H\<close> \<open>Cong R S E F\<close> assms(3) cong_reflexivity sums2__cong56 by blast
  ultimately show ?thesis 
    using SumS_def \<open>Cong P Q A B\<close> by blast
qed

lemma sums_assoc_2:
  assumes "A B C D SumS G H" and
    "C D E F SumS I J" and
    "A B I J SumS K L"
  shows "G H E F SumS K L"
proof -
  have "E F G H SumS K L"
  proof -
    have "E F C D SumS I J" 
      by (simp add: assms(2) sums_sym)
    moreover have "C D A B SumS G H" 
      by (simp add: assms(1) sums_sym)
    moreover have "I J A B SumS K L" 
      using assms(3) sums_sym by blast
    ultimately show ?thesis
      using sums_assoc_1 by blast
  qed
  thus ?thesis     
    using sums_sym by blast
qed

(** Associativity of the sum. *)

lemma sums_assoc:
  assumes "A B C D SumS G H" and
    "C D E F SumS I J"
  shows "G H E F SumS K L \<longleftrightarrow> A B I J SumS K L" 
  by (meson assms(1) assms(2) sums_assoc_1 sums_assoc_2)

(** AB <= AB + CD *)

lemma sums__le1256: 
  assumes "A B C D SumS E F"
  shows "A B Le E F" 
proof -
  obtain P Q R where "Bet P Q R" and "Cong P Q A B" and 
    "Cong Q R C D" and "Cong P R E F" 
    using SumS_def assms by blast
  thus ?thesis 
    using bet__le1213 l5_6 by blast
qed

(** CD <= AB + CD *)

lemma sums__le3456: 
  assumes "A B C D SumS E F" 
  shows "C D Le E F" 
  using assms sums__le1256 sums_sym by blast

(** If the sum of two segments is degenerate, then the segments are degenerate *)

lemma eq_sums__eq: 
  assumes "A B C D SumS E E"
  shows "A = B \<and> C = D" 
  by (metis assms cong_identity le_diff sums__cong1245 sums__le3456)

lemma sums_diff_1: 
  assumes "A \<noteq> B" and
    "A B C D SumS E F"
  shows "E \<noteq> F" 
  using assms(1) assms(2) eq_sums__eq by force

lemma sums_diff_2: 
  assumes "C \<noteq> D" and
    "A B C D SumS E F" 
  shows "E \<noteq> F" 
  using assms(1) assms(2) eq_sums__eq by blast

(** SumS preserves Le *)

lemma le2_sums2__le:
  assumes "A B Le A' B'" and
    "C D Le C' D'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "E F Le E' F'"
proof -
  obtain P Q R where "Bet P Q R" and "Cong P Q A B" and 
    "Cong Q R C D" and "Cong P R E F" 
    using SumS_def assms(3) by blast
  obtain P' Q' R' where "Bet P' Q' R'" and "Cong P' Q' A' B'" and 
    "Cong Q' R' C' D'" and "Cong P' R' E' F'" 
    using SumS_def assms(4) by blast
  have "P Q Le P' Q'" 
    by (meson \<open>Cong P Q A B\<close> \<open>Cong P' Q' A' B'\<close> assms(1) cong_symmetry l5_6)
  moreover have "Q R Le Q' R'" 
    by (meson Cong_perm \<open>Cong Q R C D\<close> \<open>Cong Q' R' C' D'\<close> assms(2) l5_6)
  ultimately have "P R Le P' R'" 
    using \<open>Bet P Q R\<close> \<open>Bet P' Q' R'\<close> bet2_le2__le1346 by blast
  thus ?thesis 
    using \<open>Cong P R E F\<close> \<open>Cong P' R' E' F'\<close> l5_6 by blast
qed

(** If AB <= A'B', CD <= C'D' and AB + CD = A'B' + C'D', then AB = A'B' and CD = C'D' *)

lemma le2_sums2__cong12:
  assumes "A B Le A' B'" and
    "C D Le C' D'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E F" 
  shows "Cong A B A' B'" 
proof -
  obtain E' F' where "A' B' C D SumS E' F'" 
    using ex_sums by blast
  hence "Cong E' F' E F" 
    by (meson assms(1) assms(2) assms(3) assms(4) le2_sums2__le le_anti_symmetry le_reflexivity)
  hence "A' B' C D SumS E F" 
    using \<open>A' B' C D SumS E' F'\<close> cong3_sums__sums cong_reflexivity by blast
  thus ?thesis
    using assms(3) sums2__cong12 by blast
qed

lemma le2_sums2__cong34:
  assumes "A B Le A' B'" and
    "C D Le C' D'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E F" 
  shows "Cong C D C' D'" 
  by (meson assms(1) assms(2) assms(3) assms(4) cong3_sums__sums cong_reflexivity 
      le2_sums2__cong12 sums2__cong34)

(** If AB < A'B' and CD <= C'D', then AB + CD < A'B' + C'D' *)

lemma le_lt12_sums2__lt:
  assumes "A B Lt A' B'" and 
    "C D Le C' D'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "E F Lt E' F'" 
proof -
  have "E F Le E' F'" 
    using Lt_def assms(1) assms(2) assms(3) assms(4) le2_sums2__le by blast
  moreover {
    assume "Cong E F E' F'" 
    hence "A' B' C' D' SumS E F" 
      by (meson assms(4) cong3_sums__sums cong_reflexivity not_cong_3412)
    hence "Cong A B A' B'" 
      using assms(1) assms(2) assms(3) le2_sums2__cong12 lt__le by blast
    hence False 
      using assms(1) cong__nlt lt__le by blast 
  }
  ultimately show ?thesis 
    using Lt_def by blast
qed

lemma le_lt34_sums2__lt:
  assumes "A B Le A' B'" and 
    "C D Lt C' D'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "E F Lt E' F'" 
  using assms(1) assms(2) assms(3) assms(4) le_lt12_sums2__lt sums_sym by blast

lemma lt2_sums2__lt:
  assumes "A B Lt A' B'" and
    "C D Lt C' D'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "E F Lt E' F'" 
  using assms(1) assms(2) assms(3) assms(4) le_lt12_sums2__lt lt__le by blast

(** If CD >= C'D' and AB + CD <= A'B' + C'D', then AB <= A'B' *)

lemma le2_sums2__le12:
  assumes "C' D' Le C D" and
    "E F Le E' F'" and 
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "A B Le A' B'" 
  using assms(1) assms(2) assms(3) assms(4) le_lt12_sums2__lt lt__nle nlt__le by blast

lemma le2_sums2__le34:
  assumes "A' B' Le A B" and
    "E F Le E' F'" and
    "A B C D SumS E F" and 
    "A' B' C' D' SumS E' F'" 
  shows "C D Le C' D'" 
  by (meson assms(1) assms(2) assms(3) assms(4) le_lt12_sums2__lt lt__nle nlt__le sums_sym)

(** If CD > C'D' and AB + CD <= A'B' + C'D', then AB < A'B' *)

lemma le_lt34_sums2__lt12:
  assumes "C' D' Lt C D" and
    "E F Le E' F'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "A B Lt A' B'" 
  by (meson assms(1) assms(2) assms(3) assms(4) le3456_lt__lt le_lt34_sums2__lt nle__lt nlt)

lemma le_lt12_sums2__lt34:
  assumes "A' B' Lt A B" and
    "E F Le E' F'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "C D Lt C' D'" 
  using assms(1) assms(2) assms(3) assms(4) le_lt12_sums2__lt lt__nle nlt__le by blast

(** If CD >= C'D' and AB + CD < A'B' + C'D', then AB < A'B' *)

lemma le_lt56_sums2__lt12:
  assumes "C' D' Le C D" and
    "E F Lt E' F'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "A B Lt A' B'" 
  by (meson assms(1) assms(2) assms(3) assms(4) le2_sums2__le le__nlt nle__lt)

lemma le_lt56_sums2__lt34:
  assumes "A' B' Le A B" and
    "E F Lt E' F'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "C D Lt C' D'" 
  using assms(1) assms(2) assms(3) assms(4) le2_sums2__le lt__nle nlt__le by blast

lemma lt2_sums2__lt12:
  assumes "C' D' Lt C D" and
    "E F Lt E' F'" and 
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "A B Lt A' B'" 
  using assms(1) assms(2) assms(3) assms(4) le_lt34_sums2__lt nle__lt not_and_lt by blast

lemma lt2_sums2__lt34: 
  assumes "A' B' Lt A B" and
    "E F Lt E' F'" and
    "A B C D SumS E F" and
    "A' B' C' D' SumS E' F'" 
  shows "C D Lt C' D'" 
  by (meson assms(1) assms(2) assms(3) assms(4) le_lt12_sums2__lt nlt__le not_and_lt)

lemma midpoint_dec:
  "I Midpoint A B \<or> \<not> I Midpoint A B"
  by simp

lemma is_midpoint_id:
  assumes "A Midpoint A B"
  shows "A = B"
  using Midpoint_def assms between_cong by blast

lemma is_midpoint_id_2:
  assumes "A Midpoint B A"
  shows "A = B"
  using Midpoint_def assms cong_diff_2 by blast

lemma l7_2:
  assumes "M Midpoint A B"
  shows "M Midpoint B A"
  using Bet_perm Cong_perm Midpoint_def assms by blast

lemma l7_3:
  assumes "M Midpoint A A"
  shows "M = A"
  using Midpoint_def assms bet_neq23__neq by blast

lemma l7_3_2:
  "A Midpoint A A"
  by (simp add: Midpoint_def between_trivial2 cong_reflexivity)

lemma symmetric_point_construction:
  "\<exists> P'. A Midpoint P P'"
  by (meson Midpoint_def cong__le cong__le3412 le_anti_symmetry segment_construction)

lemma symmetric_point_uniqueness:
  assumes "P Midpoint A P1" and
    "P Midpoint A P2"
  shows "P1 = P2"
  by (metis Midpoint_def assms(1) assms(2) between_cong_3 cong_diff_4 cong_inner_transitivity)

lemma l7_9:
  assumes "A Midpoint P X" and
    "A Midpoint Q X"
  shows "P = Q"
  using assms(1) assms(2) l7_2 symmetric_point_uniqueness by blast

lemma l7_9_bis:
  assumes "A Midpoint P X" and
    "A Midpoint X Q"
  shows "P = Q"
  using assms(1) assms(2) l7_2 symmetric_point_uniqueness by blast

lemma l7_13_R1:
  assumes "A \<noteq> P" and
    "A Midpoint P' P" and
    "A Midpoint Q' Q"
  shows "Cong P Q P' Q'"
proof -
  obtain X where "Bet P' P X" and "Cong P X Q A"
    using segment_construction by blast
  obtain X' where "Bet X P' X'" and "Cong P' X' Q A"
    using segment_construction by blast
  obtain Y where "Bet Q' Q Y" and "Cong Q Y P A"
    using segment_construction by blast
  obtain Y' where "Bet Y Q' Y'" and "Cong Q' Y' P A"
    using segment_construction by blast
  have "Bet Y A Q'"
    using Bet_cases Midpoint_def \<open>Bet Q' Q Y\<close> assms(3) between_exchange4 by blast
  have "Bet P' A X"
    using Midpoint_def \<open>Bet P' P X\<close> assms(2) between_exchange4 by blast
  have "Bet A P X" 
    using Midpoint_def \<open>Bet P' P X\<close> assms(2) between_exchange3 by blast
  have "Bet Y Q A" 
    using Midpoint_def \<open>Bet Q' Q Y\<close> assms(3) between_exchange3 between_symmetry by blast
  have "Bet A Q' Y'"
    using \<open>Bet Y A Q'\<close> \<open>Bet Y Q' Y'\<close> between_exchange3 by blast
  have "Bet X' P' A" 
    using \<open>Bet P' A X\<close> \<open>Bet X P' X'\<close> between_exchange3 between_symmetry by blast
  hence "Bet X A X'"
    using \<open>Bet P' A X\<close> \<open>Bet X P' X'\<close> between_symmetry outer_transitivity_between2 by blast
  have "Bet Y A Y'" 
    using \<open>Bet Y A Q'\<close> \<open>Bet Y Q' Y'\<close> between_exchange4 by blast
  have "Cong A X Y A"
    using \<open>Bet A P X\<close> \<open>Bet Y Q A\<close> \<open>Cong P X Q A\<close> \<open>Cong Q Y P A\<close> l2_11_b not_cong_4321 by blast
  have "Cong A Y' X' A"
  proof -
    have "Cong Q' Y' P' A"
      using Midpoint_def \<open>Cong Q' Y' P A\<close> assms(2) cong_4312 cong_transitivity by blast
    have "Cong A Q' X' P'"
      by (metis Cong_cases Midpoint_def \<open>Cong P' X' Q A\<close> assms(3) cong_transitivity)
    thus ?thesis 
      using \<open>Bet A Q' Y'\<close> \<open>Bet X' P' A\<close> \<open>Cong Q' Y' P' A\<close> l2_11_b by force
  qed
  have "Cong A Y A Y'"
  proof -
    have "Cong Q Y Q' Y'" 
      using \<open>Cong Q Y P A\<close> \<open>Cong Q' Y' P A\<close> cong_inner_transitivity cong_symmetry by blast
    thus ?thesis 
      by (meson Midpoint_def \<open>Bet A Q' Y'\<close> \<open>Bet Q' Q Y\<close> assms(3) between_exchange3 
          cong_left_commutativity cong_symmetry l2_11_b)
  qed
  have "Cong X A Y' A" 
    by (metis cong_inner_transitivity \<open>Cong A X Y A\<close> \<open>Cong A Y A Y'\<close> cong_4312)
  have "Cong A X' A Y" 
    using \<open>Cong A Y A Y'\<close> \<open>Cong A Y' X' A\<close> cong_4312 cong_transitivity by blast
  have "Cong A X A X'"
    using Cong_cases \<open>Cong A X Y A\<close> \<open>Cong A X' A Y\<close> cong_transitivity by blast
  have "X A X' Y' FSC Y' A Y X" 
  proof -
    have "Col X A X'"
      using Col_def \<open>Bet X A X'\<close> by blast
    have "Cong X X' Y' Y" 
      using Cong_cases \<open>Bet X A X'\<close> \<open>Bet Y A Y'\<close> \<open>Cong A X Y A\<close> \<open>Cong A Y' X' A\<close> l2_11_b by blast
    thus ?thesis 
      using Cong3_def FSC_def \<open>Col X A X'\<close> \<open>Cong A X' A Y\<close> \<open>Cong X A Y' A\<close> 
        cong_pseudo_reflexivity not_cong_4321 by blast
  qed
  hence "Y Q A X IFSC Y' Q' A X'" 
    by (metis Bet_cases Cong_cases IFSC_def Midpoint_def bet_neq23__neq 
        \<open>Bet A P X\<close> \<open>Bet A Q' Y'\<close> \<open>Bet X A X'\<close> \<open>Bet Y Q A\<close> \<open>Cong A X A X'\<close> \<open>Cong A Y A Y'\<close> 
        assms(1) assms(3) l4_16R1)
  hence "X P A Q IFSC X' P' A Q'" 
    by (metis Cong_cases IFSC_def Midpoint_def \<open>Bet A P X\<close> \<open>Bet X' P' A\<close> assms(2) 
        between_symmetry l4_2)
  thus ?thesis 
    using l4_2 by force
qed

lemma l7_13:
  assumes "A Midpoint P' P" and
    "A Midpoint Q' Q"
  shows "Cong P Q P' Q'"
proof (cases)
  assume "A = P"
  thus ?thesis
    using Midpoint_def assms(1) assms(2) cong_3421 is_midpoint_id_2 by blast
next
  show ?thesis
    by (metis l7_13_R1 assms(1) assms(2) cong_trivial_identity is_midpoint_id_2 not_cong_2143)
qed

lemma l7_15:
  assumes "A Midpoint P P'" and
    "A Midpoint Q Q'" and
    "A Midpoint R R'" and
    "Bet P Q R"
  shows "Bet P' Q' R'"
proof -
  have "P Q R Cong3 P' Q' R'"
    using Cong3_def assms(1) assms(2) assms(3) l7_13 l7_2 by blast
  thus ?thesis
    using assms(4) l4_6 by blast
qed

lemma l7_16:
  assumes "A Midpoint P P'" and
    "A Midpoint Q Q'" and
    "A Midpoint R R'" and
    "A Midpoint S S'" and
    "Cong P Q R S"
  shows "Cong P' Q' R' S'"
  by (meson assms(1) assms(2) assms(3) assms(4) assms(5) cong_transitivity l7_13 not_cong_3412)

lemma symmetry_preserves_midpoint:
  assumes "Z Midpoint A D" and
    "Z Midpoint B E" and
    "Z Midpoint C F" and
    "B Midpoint A C"
  shows "E Midpoint D F"
  by (meson Midpoint_def assms(1) assms(2) assms(3) assms(4) l7_15 l7_16)

lemma Mid_cases:
  assumes "A Midpoint B C \<or> A Midpoint C B"
  shows "A Midpoint B C"
  using assms l7_2 by blast

lemma Mid_perm:
  assumes "A Midpoint B C"
  shows "A Midpoint B C \<and> A Midpoint C B"
  by (simp add: assms l7_2)

lemma l7_17:
  assumes "A Midpoint P P'" and
    "B Midpoint P P'"
  shows "A = B"
proof -
  have "Cong P B P' B" 
    using Cong_cases Midpoint_def assms(2) by blast
  obtain x where "A Midpoint B x" 
    using symmetric_point_construction by presburger
  hence "Cong P' B P x" 
    using assms(1) l7_13 l7_2 by blast
  hence "Cong P B P x" 
    using \<open>Cong P B P' B\<close> cong_transitivity by blast
  have "Cong P B P' x" 
    using \<open>A Midpoint B x\<close> assms(1) cong_4321 l7_13 by blast
  have "Cong P' B P' x" 
    using cong_inner_transitivity \<open>Cong P B P' B\<close> \<open>Cong P B P' x\<close> by blast
  have "Bet P B P'" 
    using Midpoint_def assms(2) by blast
  hence "B = x" 
    using \<open>Cong P B P x\<close> \<open>Cong P' B P' x\<close> l4_19 by auto
  thus ?thesis 
    using \<open>A Midpoint B x\<close> l7_3 by blast
qed

lemma l7_17_bis:
  assumes "A Midpoint P P'" and
    "B Midpoint P' P"
  shows "A = B"
  by (meson l7_17 l7_2 Tarski_neutral_dimensionless_axioms assms(1) assms(2))

lemma l7_20:
  assumes "Col A M B" and
    "Cong M A M B"
  shows "A = B \<or> M Midpoint A B"
  by (metis Bet_cases Col_def Midpoint_def assms(1) assms(2) between_cong 
      cong_left_commutativity not_cong_3412)

lemma l7_20_bis:
  assumes "A \<noteq> B" and
    "Col A M B" and
    "Cong M A M B"
  shows "M Midpoint A B"
  using assms(1) assms(2) assms(3) l7_20 by blast

lemma cong_col_mid:
  assumes "A \<noteq> C" and
    "Col A B C" and
    "Cong A B B C"
  shows "B Midpoint A C"
  using assms(1) assms(2) assms(3) cong_left_commutativity l7_20 by blast

lemma l7_21_R1:
  assumes "\<not> Col A B C" and
    "B \<noteq> D" and
    "Cong A B C D" and
    "Cong B C D A" and
    "Col A P C" and
    "Col B P D"
  shows "P Midpoint A C"
proof -
  obtain X where "B D P Cong3 D B X"
    using Col_perm assms(6) cong_pseudo_reflexivity l4_14 by blast
  hence "Col D B X"
    using assms(6) l4_13 not_col_permutation_5 by blast
  have "B D P A FSC D B X C" 
    by (meson Col_cases Cong_cases FSC_def \<open>B D P Cong3 D B X\<close> assms(3) assms(4) assms(6))
  have "B D P C FSC D B X A" 
    using Col_cases Cong_perm FSC_def \<open>B D P Cong3 D B X\<close> assms(3) assms(4) assms(6) by blast
  hence "A P C Cong3 C X A" 
    using Cong3_def Cong_cases \<open>B D P A FSC D B X C\<close> assms(2) cong_pseudo_reflexivity l4_16 
    by blast
  hence "Col C X A" 
    using assms(5) l4_13 by blast
  hence "P = X" 
    using \<open>Col D B X\<close> assms(1) assms(2) assms(5) assms(6) l6_21 not_col_permutation_1 
      not_col_permutation_5 by blast
  thus ?thesis 
    by (metis Col_perm \<open>B D P A FSC D B X C\<close> \<open>Col C X A\<close> assms(1) assms(2) l4_16 l7_20_bis 
        not_col_distincts)
qed

lemma l7_21:
  assumes "\<not> Col A B C" and
    "B \<noteq> D" and
    "Cong A B C D" and
    "Cong B C D A" and
    "Col A P C" and
    "Col B P D"
  shows "P Midpoint A C \<and> P Midpoint B D" 
  by (metis Cong_cases Midpoint_def assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) 
      cong_reverse_identity l7_21_R1 not_col_distincts)

lemma l7_22_aux_R1:
  assumes "Bet A1 C C" and
    "Bet B1 C B2" and
    "Cong C A1 C B1" and
    "Cong C C C B2" and
    "M1 Midpoint A1 B1" and
    "M2 Midpoint A2 B2"and
    "C A1 Le C C"
  shows "Bet M1 C M2"
  by (metis assms(3) assms(5) assms(7) cong_diff_3 l7_3 le_diff not_bet_distincts)

lemma l7_22_aux_R2:
  assumes "A2 \<noteq> C" and
    "Bet A1 C A2" and
    "Bet B1 C B2" and
    "Cong C A1 C B1" and
    "Cong C A2 C B2" and
    "M1 Midpoint A1 B1" and
    "M2 Midpoint A2 B2" and
    "C A1 Le C A2"
  shows "Bet M1 C M2"
proof -
  obtain X where "C Midpoint A2 X"
    using symmetric_point_construction by blast
  obtain X0 where "C Midpoint B2 X0"
    using symmetric_point_construction by blast
  obtain X1 where "C Midpoint M2 X1"
    using symmetric_point_construction by blast
  hence "X1 Midpoint X X0"
    using \<open>C Midpoint A2 X\<close> \<open>C Midpoint B2 X0\<close> assms(7) symmetry_preserves_midpoint by blast
  have "C A1 Le C X" 
    by (metis Midpoint_def cong_reflexivity l5_6 \<open>C Midpoint A2 X\<close> assms(8) le_right_comm)
  hence "Bet C A1 X"
    by (metis (full_types) Bet_cases Le_cases Midpoint_def \<open>C Midpoint A2 X\<close> assms(1,2)
        bet_le_eq l5_2) 
  have "Cong C X C X0" 
    by (meson l7_3_2 \<open>C Midpoint A2 X\<close> \<open>C Midpoint B2 X0\<close> assms(5) l7_16)
  hence "C B1 Le C X0"
    using \<open>C A1 Le C X\<close> assms(4) l5_6 by blast
  have "Bet C B1 X0" 
  proof cases
    assume "B1 = C"
    thus ?thesis 
      using between_trivial2 by auto
  next
    assume "B1 \<noteq> C"
    thus ?thesis
      by (metis (full_types) Bet_cases Le_cases Midpoint_def \<open>C B1 Le C X0\<close> \<open>C Midpoint B2 X0\<close> assms(3)
          bet_cong_eq bet_le_eq l5_2)
  qed
  hence "Bet X0 B1 C" 
    using Bet_cases by blast
  have "\<exists> Q. Bet X1 Q C \<and> Bet A1 Q B1" 
  proof -
    have "Bet X A1 C" 
      using Bet_cases \<open>Bet C A1 X\<close> by blast
    moreover have "Bet X X1 X0" 
      using Midpoint_def \<open>X1 Midpoint X X0\<close> by auto
    ultimately show ?thesis 
      using \<open>Bet X0 B1 C\<close> l3_17 by blast
  qed
  then obtain Q where "Bet X1 Q C" and "Bet A1 Q B1" 
    by blast
  have "X A1 C X1 IFSC X0 B1 C X1" 
    by (metis Bet_cases Cong_cases IFSC_def Midpoint_def \<open>Bet C A1 X\<close> \<open>Bet X0 B1 C\<close> 
        \<open>Cong C X C X0\<close> \<open>X1 Midpoint X X0\<close> assms(4) cong_reflexivity)
  hence "Cong A1 X1 B1 X1" 
    using l4_2 by auto
  have  "Cong Q A1 Q B1"
  proof cases
    assume "C = X1"
    thus ?thesis 
      using between_identity \<open>Bet X1 Q C\<close> assms(4) by blast
  next
    assume "\<not> C = X1"
    moreover
    have "Col C X1 Q" 
      by (simp add: Col_def \<open>Bet X1 Q C\<close>)
    moreover
    have "Cong X1 A1 X1 B1" 
      using Cong_cases \<open>Cong A1 X1 B1 X1\<close> by auto
    ultimately show ?thesis 
      by (simp add: assms(4) l4_17)
  qed
  have "Q Midpoint A1 B1" 
    using Midpoint_def \<open>Bet A1 Q B1\<close> \<open>Cong Q A1 Q B1\<close> not_cong_2134 by blast
  thus ?thesis 
    by (metis Midpoint_def \<open>Bet X1 Q C\<close> \<open>C Midpoint M2 X1\<close> assms(6) between_inner_transitivity 
        between_symmetry l7_17)
qed

lemma l7_22_aux:
  assumes "Bet A1 C A2" and
    "Bet B1 C B2" and
    "Cong C A1 C B1" and
    "Cong C A2 C B2" and
    "M1 Midpoint A1 B1" and
    "M2 Midpoint A2 B2" and
    "C A1 Le C A2"
  shows "Bet M1 C M2" 
  by (metis assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) assms(7) between_trivial 
      cong_diff_4 l7_22_aux_R2 l7_3)

lemma l7_22:
  assumes "Bet A1 C A2" and
    "Bet B1 C B2" and
    "Cong C A1 C B1" and
    "Cong C A2 C B2" and
    "M1 Midpoint A1 B1" and
    "M2 Midpoint A2 B2"
  shows "Bet M1 C M2"
  by (meson assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) between_symmetry l7_22_aux 
      le_cases)

lemma bet_col1:
  assumes "Bet A B D" and
    "Bet A C D"
  shows "Col A B C"
  using Bet_perm Col_def assms(1) assms(2) l5_3 by blast

lemma l7_25_R1:
  assumes "Cong C A C B" and
    "Col A B C"
  shows "\<exists> X. X Midpoint A B"
  using assms(1) assms(2) l7_20 l7_3_2 not_col_permutation_5 by blast

lemma l7_25_R2:
  assumes "Cong C A C B" and
    "\<not> Col A B C"
  shows "\<exists> X. X Midpoint A B"
proof -
  obtain P where "Bet C A P" and "A \<noteq> P"
    using point_construction_different by auto
  obtain Q where "Bet C B Q" and "Cong B Q A P"
    using segment_construction by blast
  obtain R where "Bet A R Q" and "Bet B R P" 
    by (meson \<open>Bet C A P\<close> \<open>Bet C B Q\<close> l3_17 not_bet_distincts)
  obtain X where "Bet A X B" and "Bet R X C" 
    using \<open>Bet B R P\<close> \<open>Bet C A P\<close> inner_pasch by blast
  have "Cong X A X B"
  proof -
    have "Cong R A R B \<longrightarrow> Cong X A X B"
    proof cases
      assume "R = C"
      thus ?thesis 
        using \<open>Bet R X C\<close> between_identity by blast
    next
      assume "\<not> R = C"
      have "Col R C X" 
        using \<open>Bet R X C\<close> bet_col1 between_trivial by blast
      thus ?thesis 
        using \<open>R \<noteq> C\<close> assms(1) l4_17 by blast
    qed
    have "Cong R A R B"
    proof -
      have "C A P B OFSC C B Q A" 
        by (metis OFSC_def \<open>Bet C A P\<close> \<open>Bet C B Q\<close> \<open>Cong B Q A P\<close> assms(1) 
            cong_pseudo_reflexivity not_cong_3412)
      hence "Cong P B Q A" 
        using assms(2) col_trivial_3 five_segment_with_def by blast
      hence "Cong B P A Q" 
        using Cong_cases by blast
      then obtain R' where "Bet A R' Q" and "B R P Cong3 A R' Q" 
        using \<open>Bet B R P\<close> l4_5 by blast
      have "B R P A IFSC A R' Q B" 
        using Cong3_def IFSC_def \<open>B R P Cong3 A R' Q\<close> \<open>Bet A R' Q\<close> \<open>Bet B R P\<close> \<open>Cong B Q A P\<close> 
          cong_pseudo_reflexivity not_cong_4321 by blast
      hence "Cong R A R' B" 
        using l4_2 by auto
      have "B R P Q IFSC A R' Q P" 
        using Cong3_def IFSC_def \<open>B R P Cong3 A R' Q\<close> \<open>Bet A R' Q\<close> \<open>Bet B R P\<close> \<open>Cong B Q A P\<close> 
          cong_pseudo_reflexivity by blast
      hence "Cong R Q R' P" 
        using l4_2 by blast
      hence "A R Q Cong3 B R' P" 
        using Cong3_def Cong_cases \<open>Cong B P A Q\<close> \<open>Cong R A R' B\<close> by blast
      have "Col B R' P" 
        using \<open>A R Q Cong3 B R' P\<close> \<open>Bet A R Q\<close> bet_col l4_13 by blast
      have "R = R'"
      proof -
        have "B \<noteq> P" 
          using Col_def \<open>Bet C A P\<close> assms(2) by blast
        moreover
        have "B \<noteq> Q" 
          using \<open>A \<noteq> P\<close> \<open>Cong B Q A P\<close> cong_reverse_identity by force
        hence "\<not> Col A Q B" 
          using \<open>Bet C B Q\<close> assms(2) bet_col col2__eq not_col_permutation_5 by blast
        moreover have "Col A Q R" 
          using Bet_cases Col_def \<open>Bet A R Q\<close> by auto
        moreover have "Col A Q R'" 
          using \<open>Bet A R' Q\<close> bet_col not_col_permutation_5 by blast
        moreover have "Col B P R" 
          using Bet_cases Col_def \<open>Bet B R P\<close> by auto
        moreover have "Col B P R'" 
          using \<open>Col B R' P\<close> not_col_permutation_5 by blast
        ultimately show ?thesis 
          using \<open>B \<noteq> P\<close> \<open>\<not> Col A Q B\<close> l6_21 by blast
      qed
      thus ?thesis 
        using \<open>Cong R A R' B\<close> by auto
    qed
    thus ?thesis 
      by (simp add: \<open>Cong R A R B \<longrightarrow> Cong X A X B\<close>)
  qed
  thus ?thesis 
    using Col_def \<open>Bet A X B\<close> between_symmetry l7_25_R1 by blast
qed

lemma l7_25:
  assumes "Cong C A C B"
  shows "\<exists> X. X Midpoint A B"
  using assms l7_25_R1 l7_25_R2 by blast

lemma midpoint_distinct_1:
  assumes "A \<noteq> B" and
    "I Midpoint A B"
  shows "I \<noteq> A \<and> I \<noteq> B"
  using assms(1) assms(2) is_midpoint_id is_midpoint_id_2 by blast

lemma midpoint_distinct_2:
  assumes "I \<noteq> A" and
    "I Midpoint A B"
  shows "A \<noteq> B \<and> I \<noteq> B"
  using assms(1) assms(2) is_midpoint_id_2 l7_3 by blast

lemma midpoint_distinct_3:
  assumes "I \<noteq> B" and
    "I Midpoint A B"
  shows "A \<noteq> B \<and> I \<noteq> A"
  using assms(1) assms(2) is_midpoint_id l7_3 by blast

lemma midpoint_def:
  assumes "Bet A B C" and
    "Cong A B B C"
  shows "B Midpoint A C"
  using Midpoint_def assms(1) assms(2) by blast

lemma midpoint_bet:
  assumes "B Midpoint A C"
  shows "Bet A B C"
  using Midpoint_def assms by blast

lemma midpoint_col:
  assumes "M Midpoint A B"
  shows "Col M A B"
  using assms bet_col col_permutation_4 midpoint_bet by blast

lemma midpoint_cong:
  assumes "B Midpoint A C"
  shows "Cong A B B C"
  using Midpoint_def assms by blast

lemma midpoint_out:
  assumes "A \<noteq> C" and
    "B Midpoint A C"
  shows "A Out B C"
  using assms(1) assms(2) bet_out midpoint_bet midpoint_distinct_1 by blast

lemma midpoint_out_1:
  assumes "A \<noteq> C" and
    "B Midpoint A C"
  shows "C Out A B"
  by (metis midpoint_bet midpoint_distinct_1 assms(1) assms(2) bet_out_1 l6_6)

lemma midpoint_not_midpoint:
  assumes "A \<noteq> B" and
    "I Midpoint A B"
  shows "\<not> B Midpoint A I"
  using assms(1) assms(2) between_equality_2 midpoint_bet midpoint_distinct_1 by blast

lemma swap_diff:
  assumes "A \<noteq> B"
  shows "B \<noteq> A"
  using assms by auto

lemma cong_cong_half_1:
  assumes "M Midpoint A B" and
    "M' Midpoint A' B'" and
    "Cong A B A' B'"
  shows "Cong A M A' M'"
proof -
  obtain M'' where "Bet A' M'' B'" and "A M B Cong3 A' M'' B'"
    using assms(1) assms(3) l4_5 midpoint_bet by blast
  hence "M'' Midpoint A' B'"
    by (meson Cong3_def assms(1) cong_inner_transitivity midpoint_cong midpoint_def)
  hence "M' = M''"
    using assms(2) l7_17 by auto
  thus ?thesis 
    using Cong3_def \<open>A M B Cong3 A' M'' B'\<close> by blast
qed

lemma cong_cong_half_2:
  assumes "M Midpoint A B" and
    "M' Midpoint A' B'" and
    "Cong A B A' B'"
  shows "Cong B M B' M'"
  using assms(1) assms(2) assms(3) cong_cong_half_1 l7_2 not_cong_2143 by blast

lemma cong_mid2__cong:
  assumes "M Midpoint A B" and
    "M' Midpoint A' B'" and
    "Cong A M A' M'"
  shows "Cong A B A' B'"
  by (meson assms(1) assms(2) assms(3) cong_inner_transitivity l2_11_b midpoint_bet midpoint_cong)

lemma mid__lt:
  assumes "A \<noteq> B" and
    "M Midpoint A B"
  shows "A M Lt A B"
  using assms(1) assms(2) bet__lt1213 midpoint_bet midpoint_distinct_1 by blast

lemma le_mid2__le13:
  assumes "M Midpoint A B" and
    "M' Midpoint A' B'" and
    "A M Le A' M'"
  shows "A B Le A' B'" 
  by (meson Midpoint_def l5_6 assms(1) assms(2) assms(3) bet2_le2__le1346)

lemma le_mid2__le12:
  assumes "M Midpoint A B" and
    "M' Midpoint A' B'"
    and "A B Le A' B'"
  shows "A M Le A' M'"
  by (meson assms(1) assms(2) assms(3) cong__le3412 cong_cong_half_1 le_anti_symmetry 
      le_mid2__le13 local.le_cases)

lemma lt_mid2__lt13:
  assumes "M Midpoint A B" and
    "M' Midpoint A' B'" and
    "A M Lt A' M'"
  shows "A B Lt A' B'"
  by (meson le_mid2__le12 Tarski_neutral_dimensionless_axioms assms(1) assms(2) assms(3) 
      lt__nle nlt__le)

lemma lt_mid2__lt12:
  assumes "M Midpoint A B" and
    "M' Midpoint A' B'" and
    "A B Lt A' B'"
  shows "A M Lt A' M'"
  by (meson le_mid2__le13 Tarski_neutral_dimensionless_axioms assms(1) assms(2) assms(3) 
      le__nlt nle__lt)

lemma midpoint_preserves_out:
  assumes "A Out B C" and
    "M Midpoint A A'" and
    "M Midpoint B B'" and
    "M Midpoint C C'"
  shows "A' Out B' C'" 
  using Out_def assms(1) assms(2) assms(3) assms(4) l7_15 l7_9 by fastforce

lemma col_cong_bet:
  assumes "Col A B D" and
    "Cong A B C D" and
    "Bet A C B"
  shows "Bet C A D \<or> Bet C B D" 
proof -
  obtain D1 where "Bet B A D1" and "Cong A D1 B C" 
    using segment_construction by blast
  obtain D2 where "Bet A B D2" and "Cong B D2  A C" 
    using segment_construction by blast
  have "Cong A B C D1" 
    by (meson Bet_cases \<open>Bet B A D1\<close> \<open>Cong A D1 B C\<close> assms(3) between_exchange4 
        cong_pseudo_reflexivity l4_3_1 not_cong_2134)
  have "D = D1 \<or> C Midpoint D D1" 
  proof -
    have "Col D C D1" 
    proof cases
      assume "A = B"
      thus ?thesis 
        by (metis assms(2) col_trivial_1 cong_diff_3)
    next
      assume "A \<noteq> B"
      thus ?thesis 
        by (meson Col_def \<open>Bet B A D1\<close> assms(1) assms(3) col3 not_col_permutation_4)
    qed
    moreover have "Cong C D C D1" 
      using \<open>Cong A B C D1\<close> assms(2) cong_inner_transitivity by blast
    ultimately show ?thesis 
      using l7_20 by auto
  qed
  {
    assume "D = D1"
    hence "Bet C A D \<or> Bet C B D" 
      using \<open>Bet B A D1\<close> assms(3) between_exchange3 between_symmetry by blast
  }
  moreover
  {
    assume "C Midpoint D D1"
    have "Cong B A C D2" 
    proof -
      have "Bet B C A" 
        using Bet_cases assms(3) by blast
      moreover have "Bet C B D2" 
        using \<open>Bet A B D2\<close> assms(3) between_exchange3 by blast
      moreover have "Cong B C C B" 
        using cong_pseudo_reflexivity by blast
      have "Cong C A B D2" 
        using Cong_cases \<open>Cong B D2 A C\<close> by blast
      ultimately show ?thesis 
        using \<open>Cong B C C B\<close> l2_11_b by blast
    qed
    have "C Midpoint D2 D1" 
    proof cases
      assume "A = B"
      thus ?thesis 
        by (metis \<open>C Midpoint D D1\<close> \<open>Cong B A C D2\<close> assms(2) cong_diff_3)
    next
      assume "A \<noteq> B"
      show ?thesis 
      proof cases
        assume "B = C"
        thus ?thesis 
          using Midpoint_def \<open>Bet A B D2\<close> \<open>Cong A D1 B C\<close> \<open>Cong B D2 A C\<close> between_symmetry 
            cong_commutativity cong_identity by blast
      next
        assume "B \<noteq> C"
        have "Bet D1 C B" 
          using \<open>Bet B A D1\<close> assms(3) between_exchange4 between_symmetry by blast
        have "Bet C B D2" 
          using \<open>Bet A B D2\<close> assms(3) between_exchange3 by blast
        thus ?thesis 
          by (metis Midpoint_def cong_inner_transitivity \<open>B \<noteq> C\<close> \<open>Bet D1 C B\<close> 
              \<open>Cong A B C D1\<close> \<open>Cong B A C D2\<close> between_symmetry not_cong_2134 
              outer_transitivity_between2)
      qed
    qed
    have "Bet C A D \<or> Bet C B D" 
      using \<open>Bet A B D2\<close> \<open>C Midpoint D2 D1\<close> \<open>D = D1 \<or> C Midpoint D D1\<close> assms(3) 
        between_exchange3 calculation l7_9 by blast
  }
  ultimately show ?thesis 
    using \<open>D = D1 \<or> C Midpoint D D1\<close> by blast
qed

lemma col_cong2_bet1:
  assumes "Col A B D" and
    "Bet A C B" and
    "Cong A B C D" and
    "Cong A C B D"
  shows "Bet C B D"
  by (metis assms(1) assms(2) assms(3) assms(4) bet__le1213 bet_cong_eq between_symmetry 
      col_cong_bet cong__le cong_left_commutativity l5_12_b l5_6 outer_transitivity_between2)

lemma col_cong2_bet2:
  assumes "Col A B D" and
    "Bet A C B" and
    "Cong A B C D" and
    "Cong A D B C"
  shows "Bet C A D"
  by (metis assms(1) assms(2) assms(3) assms(4) bet_cong_eq col_cong_bet 
      cong_identity not_bet_distincts not_cong_3421 outer_transitivity_between2)

lemma col_cong2_bet3:
  assumes "Col A B D" and
    "Bet A B C" and
    "Cong A B C D" and
    "Cong A C B D"
  shows "Bet B C D"
  by (metis assms(1) assms(2) assms(3) assms(4) bet__le1213 bet__le2313 
      bet_col col_transitivity_2 cong_diff_3 cong_reflexivity l5_12_b l5_6 not_bet_distincts)

lemma col_cong2_bet4:
  assumes "Col A B C" and
    "Bet A B D" and
    "Cong A B C D" and
    "Cong A D B C"
  shows "Bet B D C"
  using assms(1) assms(2) assms(3) assms(4) col_cong2_bet3 cong_right_commutativity by blast

lemma col_bet2_cong1:
  assumes "Col A B D" and
    "Bet A C B" and
    "Cong A B C D" and
    "Bet C B D"
  shows "Cong A C D B"
  by (meson assms(2) assms(3) assms(4) between_symmetry cong_pseudo_reflexivity 
      cong_right_commutativity l4_3)

lemma col_bet2_cong2:
  assumes "Col A B D" and
    "Bet A C B" and
    "Cong A B C D" and
    "Bet C A D"
  shows "Cong D A B C"
  by (meson assms(2) assms(3) assms(4) between_symmetry cong_commutativity 
      cong_pseudo_reflexivity cong_symmetry l4_3)

lemma bet2_lt2__lt:
  assumes "Bet a Po b" and
    "Bet A PO B" and
    "Po a Lt PO A" and
    "Po b Lt PO B"
  shows "a b Lt A B"
  by (metis Lt_cases nle__lt assms(1) assms(2) assms(3) assms(4) bet2_le2__le1245 le__nlt lt__le)

lemma bet2_lt_le__lt:
  assumes "Bet a Po b" and
    "Bet A PO B" and
    "Cong Po a PO A" and
    "Po b Lt PO B"
  shows "a b Lt A B" 
proof -
  have "Po a Le PO A" 
    using assms(3) cong__le by blast
  thus ?thesis
    by (meson Le_cases nlt__le assms(1) assms(2) assms(4) bet2_le2__le2356 lt__nle)
qed

lemma per_dec:
  "Per A B C \<or> \<not> Per A B C"
  by simp

lemma l8_2:
  assumes "Per A B C"
  shows "Per C B A"
proof -
  obtain C' where "B Midpoint C C'" and "Cong A C A C'"
    using Per_def assms by blast
  obtain A' where "B Midpoint A A'"
    using symmetric_point_construction by blast
  hence "Cong C' A C A'" 
    using Mid_cases \<open>B Midpoint C C'\<close> l7_13 by blast
  thus ?thesis 
    using Per_def \<open>B Midpoint A A'\<close> \<open>Cong A C A C'\<close> cong_transitivity not_cong_2143 by blast
qed

lemma Per_cases:
  assumes "Per A B C \<or> Per C B A"
  shows "Per A B C"
  using assms l8_2 by blast

lemma Per_perm :
  assumes "Per A B C"
  shows "Per A B C \<and> Per C B A"
  by (simp add: assms l8_2)

lemma l8_3 :
  assumes "Per A B C" and
    "A \<noteq> B" and
    "Col B A A'"
  shows "Per A' B C" 
  by (metis Per_def cong_left_commutativity assms(1) assms(2) assms(3) l4_17 midpoint_cong)

lemma l8_4:
  assumes "Per A B C" and
    "B Midpoint C C'"
  shows "Per A B C'"
  by (metis l8_2 assms(1) assms(2) l8_3 midpoint_col midpoint_distinct_1)

lemma l8_5:
  shows "Per A B B"
  using Per_def cong_reflexivity l7_3_2 by blast

lemma l8_6:
  assumes "Per A B C" and
    "Per A' B C" and
    "Bet A C A'"
  shows "B = C"
  by (metis Per_def assms(1) assms(2) assms(3) l4_19 midpoint_distinct_3 
      symmetric_point_uniqueness)

lemma l8_7:
  assumes "Per A B C" and
    "Per A C B"
  shows "B = C"
proof -
  obtain C' where P1: "B Midpoint C C' \<and> Cong A C A C'"
    using Per_def assms(1) by blast
  obtain A' where P2: "C Midpoint A A'"
    using Per_def assms(2) l8_2 by blast
  have "Per C' C A"
    by (metis P1 l8_3 assms(2) bet_col l8_2 midpoint_bet midpoint_distinct_3)
  hence "Cong A C' A' C'"
    using Cong_perm P2 Per_def symmetric_point_uniqueness by blast
  hence "Cong A' C A' C'"
    using P1 P2 cong_inner_transitivity midpoint_cong not_cong_2134 by blast
  hence Q4: "Per A' B C"
    using P1 Per_def by blast
  have "Bet A' C A"
    using Mid_perm P2 midpoint_bet by blast
  thus ?thesis
    using Q4 assms(1) l8_6 by blast
qed

lemma l8_8:
  assumes "Per A B A"
  shows "A = B"
  using l8_6 Tarski_neutral_dimensionless_axioms assms between_trivial2 by fastforce

lemma per_distinct:
  assumes "Per A B C" and
    "A \<noteq> B"
  shows "A \<noteq> C"
  using assms(1) assms(2) l8_8 by blast

lemma per_distinct_1:
  assumes "Per A B C" and
    "B \<noteq> C"
  shows "A \<noteq> C"
  using assms(1) assms(2) l8_8 by blast

lemma l8_9:
  assumes "Per A B C" and
    "Col A B C"
  shows "A = B \<or> C = B"
  using Col_cases assms(1) assms(2) l8_3 l8_8 by blast

lemma l8_10:
  assumes "Per A B C" and
    "A B C Cong3 A' B' C'"
  shows "Per A' B' C'"
proof -
  obtain D where "B Midpoint C D" and "Cong A C A D"
    using Per_def assms(1) by blast
  obtain D' where "Bet C' B' D'" and "Cong B' D' B' C'"
    using segment_construction by blast
  hence "B' Midpoint C' D'"
    by (simp add: Midpoint_def cong_4312)
  have "Cong A' C' A' D'"
  proof cases
    assume "C = B"
    thus ?thesis 
      by (metis Cong3_def Cong_cases \<open>Cong B' D' B' C'\<close> assms(2) cong_reflexivity 
          cong_reverse_identity)
  next
    assume "\<not> C = B"
    hence "C B D A OFSC C' B' D' A'"
      by (meson Cong3_def OFSC_def \<open>B Midpoint C D\<close> \<open>B' Midpoint C' D'\<close> assms(2) cong_commutativity
          cong_cong_half_2 cong_mid2__cong midpoint_bet) 
    thus ?thesis 
      by (meson Cong3_def \<open>C \<noteq> B\<close> \<open>Cong A C A D\<close> assms(2) cong_inner_transitivity 
          five_segment_with_def not_cong_2143)
  qed
  thus ?thesis 
    using Per_def \<open>B' Midpoint C' D'\<close> by blast
qed

lemma col_col_per_per:
  assumes "A \<noteq> X" and
    "C \<noteq> X" and
    "Col U A X" and
    "Col V C X" and
    "Per A X C"
  shows "Per U X V"
  by (meson l8_2 l8_3 Tarski_neutral_dimensionless_axioms assms(1) assms(2) assms(3) assms(4) 
      assms(5) not_col_permutation_3)

lemma perp_in_dec:
  "X PerpAt A B C D \<or> \<not> X PerpAt A B C D"
  by simp

lemma perp_distinct:
  assumes "A B Perp C D"
  shows "A \<noteq> B \<and> C \<noteq> D"
  using PerpAt_def Perp_def assms by auto

lemma l8_12:
  assumes "X PerpAt A B C D"
  shows "X PerpAt C D A B"
  using Per_perm PerpAt_def assms by auto

lemma per_col:
  assumes "B \<noteq> C" and
    "Per A B C" and
    "Col B C D"
  shows "Per A B D"
  by (metis l8_3 assms(1) assms(2) assms(3) l8_2)

lemma l8_13_2:
  assumes "A \<noteq> B" and
    "C \<noteq> D" and
    "Col X A B" and
    "Col X C D" and
    "\<exists> U. \<exists> V. Col U A B \<and> Col V C D \<and> U \<noteq> X \<and> V \<noteq> X \<and> Per U X V"
  shows "X PerpAt A B C D"
proof -
  obtain U V where "Col U A B" and "Col V C D" and "U \<noteq> X" and "V \<noteq> X" and "Per U X V" 
    using assms(5) by blast
  {
    fix U0 V0
    assume "Col U0 A B" and "Col V0 C D"
    have "Col X U U0" 
      using \<open>Col U A B\<close> \<open>Col U0 A B\<close> assms(1) assms(3) col3 not_col_permutation_2 by blast
    hence "Per U0 X V" 
      using \<open>Per U X V\<close> \<open>U \<noteq> X\<close> l8_3 by blast
    hence "Per V X U0" 
      using l8_2 by blast
    hence "Per U0 X V0" 
      by (metis NCol_perm 
          \<open>Col V C D\<close> \<open>Col V0 C D\<close> \<open>Per U0 X V\<close> \<open>V \<noteq> X\<close> assms(2) assms(4) l6_16_1 per_col)
  }
  thus ?thesis 
    by (simp add: PerpAt_def assms(1) assms(2) assms(3) assms(4))
qed

lemma l8_14_1:
  "\<not> A B Perp A B"
  by (metis PerpAt_def Perp_def col_trivial_1 col_trivial_3 l8_8)

lemma l8_14_2_1a:
  assumes "X PerpAt A B C D"
  shows "A B Perp C D"
  using Perp_def assms by blast

lemma perp_in_distinct:
  assumes "X PerpAt A B C D"
  shows "A \<noteq> B \<and> C \<noteq> D"
  using PerpAt_def assms by blast

lemma l8_14_2_1b:
  assumes "X PerpAt A B C D" and
    "Col Y A B" and
    "Col Y C D"
  shows "X = Y"
  by (metis PerpAt_def assms(1) assms(2) assms(3) l8_13_2 l8_14_1 l8_14_2_1a)

lemma l8_14_2_1b_bis:
  assumes "A B Perp C D" and
    "Col X A B" and
    "Col X C D"
  shows "X PerpAt A B C D"
  using Perp_def assms(1) assms(2) assms(3) l8_14_2_1b by blast

lemma l8_14_2_2:
  assumes "A B Perp C D" and
    "\<forall> Y. (Col Y A B \<and> Col Y C D) \<longrightarrow> X = Y"
  shows "X PerpAt A B C D"
  by (metis PerpAt_def Perp_def assms(1) assms(2))

lemma l8_14_3:
  assumes "X PerpAt A B C D" and
    "Y PerpAt A B C D"
  shows "X = Y"
  by (meson PerpAt_def assms(1) assms(2) l8_14_2_1b)

lemma l8_15_1:
  assumes "Col A B X" and
    "A B Perp C X"
  shows "X PerpAt A B C X"
  using NCol_perm assms(1) assms(2) col_trivial_3 l8_14_2_1b_bis by blast

lemma l8_15_2:
  assumes "Col A B X" and
    "X PerpAt A B C X"
  shows "A B Perp C X"
  using assms(2) l8_14_2_1a by blast

lemma perp_in_per:
  assumes "B PerpAt A B B C"
  shows "Per A B C"
  by (meson NCol_cases PerpAt_def assms col_trivial_3)

lemma perp_sym:
  assumes "A B Perp A B"
  shows "C D Perp C D"
  using assms l8_14_1 by auto

lemma perp_col0:
  assumes "A B Perp C D" and
    "X \<noteq> Y" and
    "Col A B X" and
    "Col A B Y"
  shows "C D Perp X Y"
proof -
  obtain X0 where "X0 PerpAt A B C D"
    using Perp_def assms(1) by blast
  hence " A \<noteq> B \<and> C \<noteq> D \<and> Col X0 A B \<and> Col X0 C D \<and>
((Col U A B \<and> Col V C D) \<longrightarrow> Per U X0 V)" 
    using PerpAt_def by blast
  have "C \<noteq> D" 
    by (simp add: \<open>A \<noteq> B \<and> C \<noteq> D \<and> Col X0 A B \<and> Col X0 C D \<and> 
    (Col U A B \<and> Col V C D \<longrightarrow> Per U X0 V)\<close>)
  have "X \<noteq> Y"
    by (simp add: assms(2))
  have "Col X0 C D" 
    using \<open>A \<noteq> B \<and> C \<noteq> D \<and> Col X0 A B \<and> Col X0 C D \<and> (Col U A B \<and> Col V C D \<longrightarrow> Per U X0 V)\<close> 
    by blast
  have "Col X0 X Y" 
    by (meson \<open>A \<noteq> B \<and> C \<noteq> D \<and> Col X0 A B \<and> Col X0 C D \<and> (Col U A B \<and> Col V C D \<longrightarrow> Per U X0 V)\<close> 
        assms(3) assms(4) col3 not_col_permutation_2)
  have "X0 PerpAt C D X Y"
  proof -
    have "\<forall> U V. (Col U C D \<and> Col V X Y) \<longrightarrow> Per U X0 V" 
      by (metis Per_perm PerpAt_def col_trivial_2 \<open>X0 PerpAt A B C D\<close> assms(2) assms(3) 
          assms(4) l6_21 not_col_permutation_1)
    thus ?thesis 
      by (simp add: PerpAt_def \<open>C \<noteq> D\<close> \<open>Col X0 C D\<close> \<open>Col X0 X Y\<close> assms(2))
  qed
  thus ?thesis
    using Perp_def by auto
qed

lemma per_perp_in:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Per A B C"
  shows "B PerpAt A B B C"
  by (metis Col_def assms(1) assms(2) assms(3) between_trivial2 l8_13_2)

lemma per_perp:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Per A B C"
  shows "A B Perp B C"
  using Perp_def assms(1) assms(2) assms(3) per_perp_in by blast

lemma perp_left_comm:
  assumes "A B Perp C D"
  shows "B A Perp C D"
proof -
  obtain X where "X PerpAt A B C D"
    using Perp_def assms by blast
  hence "X PerpAt B A C D"
    using PerpAt_def col_permutation_5 by auto
  thus ?thesis
    using Perp_def by blast
qed

lemma perp_right_comm:
  assumes "A B Perp C D"
  shows "A B Perp D C"
  by (meson Perp_def assms l8_12 perp_left_comm)

lemma perp_comm:
  assumes "A B Perp C D"
  shows "B A Perp D C"
  by (simp add: assms perp_left_comm perp_right_comm)

lemma perp_in_sym:
  assumes  "X PerpAt A B C D"
  shows "X PerpAt C D A B"
  by (simp add: assms l8_12)

lemma perp_in_left_comm:
  assumes "X PerpAt A B C D"
  shows "X PerpAt B A C D"
  by (metis Col_cases PerpAt_def assms)

lemma perp_in_right_comm:
  assumes "X PerpAt A B C D"
  shows "X PerpAt A B D C"
  using assms perp_in_left_comm perp_in_sym by blast

lemma perp_in_comm:
  assumes "X PerpAt A B C D"
  shows "X PerpAt B A D C"
  by (simp add: assms perp_in_left_comm perp_in_right_comm)

lemma Perp_cases:
  assumes "A B Perp C D \<or> B A Perp C D \<or> A B Perp D C \<or> B A Perp D C \<or> C D Perp A B \<or> 
C D Perp B A \<or> D C Perp A B \<or> D C Perp B A"
  shows "A B Perp C D"
  by (meson Perp_def assms perp_in_sym perp_left_comm)

lemma Perp_perm :
  assumes "A B Perp C D"
  shows "A B Perp C D \<and> B A Perp C D \<and> A B Perp D C \<and> B A Perp D C \<and> C D Perp A B \<and> 
C D Perp B A \<and> D C Perp A B \<and> D C Perp B A"
  by (meson Perp_def assms perp_in_sym perp_left_comm)

lemma Perp_in_cases:
  assumes "X PerpAt A B C D \<or> X PerpAt B A C D \<or> X PerpAt A B D C \<or> X PerpAt B A D C \<or> 
X PerpAt C D A B \<or> X PerpAt C D B A \<or> X PerpAt D C A B \<or> X PerpAt D C B A"
  shows "X PerpAt A B C D"
  using assms perp_in_left_comm perp_in_sym by blast

lemma Perp_in_perm:
  assumes "X PerpAt A B C D"
  shows "X PerpAt A B C D \<and> X PerpAt B A C D \<and> X PerpAt A B D C \<and> X PerpAt B A D C \<and> 
X PerpAt C D A B \<and> X PerpAt C D B A \<and> X PerpAt D C A B \<and> X PerpAt D C B A"
  using Perp_in_cases assms by blast

lemma perp_in_col:
  assumes "X PerpAt A B C D"
  shows "Col A B X \<and> Col C D X"
  using PerpAt_def assms col_permutation_2 by presburger

lemma perp_perp_in:
  assumes "A B Perp C A"
  shows "A PerpAt A B C A"
  using assms l8_15_1 not_col_distincts by blast

lemma perp_per_1:
  assumes "A B Perp C A"
  shows "Per B A C"
  using Perp_in_cases assms perp_in_per perp_perp_in by blast

lemma perp_per_2:
  assumes "A B Perp A C"
  shows "Per B A C"
  by (simp add: Perp_perm assms perp_per_1)

lemma perp_col:
  assumes "A \<noteq> E" and
    "A B Perp C D" and
    "Col A B E"
  shows "A E Perp C D"
  using Perp_perm assms(1) assms(2) assms(3) col_trivial_3 perp_col0 by blast

lemma perp_col2:
  assumes "A B Perp X Y" and
    "C \<noteq> D" and
    "Col A B C" and
    "Col A B D"
  shows "C D Perp X Y"
  using Perp_perm assms(1) assms(2) assms(3) assms(4) perp_col0 by blast

lemma perp_col4:
  assumes "P \<noteq> Q" and
    "R \<noteq> S" and
    "Col A B P" and
    "Col A B Q" and
    "Col C D R" and
    "Col C D S" and
    "A B Perp C D"
  shows "P Q Perp R S"
  using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) assms(7) perp_col0 by blast

lemma perp_not_eq_1:
  assumes "A B Perp C D"
  shows "A \<noteq> B"
  using assms perp_distinct by auto

lemma perp_not_eq_2:
  assumes "A B Perp C D"
  shows "C \<noteq> D"
  using assms perp_distinct by auto

lemma diff_per_diff:
  assumes "A \<noteq> B" and
    "Cong A P B R" and
    "Per B A P"
    and "Per A B R"
  shows "P \<noteq> R"
  using assms(1) assms(3) assms(4) l8_2 l8_7 by blast

lemma per_not_colp:
  assumes "A \<noteq> B" and
    "A \<noteq> P" and
    "B \<noteq> R" and
    "Per B A P"
    and "Per A B R"
  shows "\<not> Col P A R"
  by (metis Per_cases col_permutation_4 assms(1) assms(2) assms(4) assms(5) l8_3 l8_7)

lemma per_not_col:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Per A B C"
  shows "\<not> Col A B C"
  using assms(1) assms(2) assms(3) l8_9 by auto

lemma perp_not_col2:
  assumes "A B Perp C D"
  shows "\<not> Col A B C \<or> \<not> Col A B D"
  using assms l8_14_1 perp_col2 perp_distinct by blast

lemma perp_not_col:
  assumes "A B Perp P A"
  shows "\<not> Col A B P"
proof -
  have "A PerpAt A B P A"
    using assms perp_perp_in by auto
  hence "Per P A B"
    by (simp add: perp_in_per perp_in_sym)
  hence "\<not> Col B A P"
    by (metis NCol_perm perp_not_eq_1 perp_not_eq_2 assms per_not_col)
  thus ?thesis
    using Col_perm by blast
qed

lemma perp_in_col_perp_in:
  assumes "C \<noteq> E" and
    "Col C D E" and
    "P PerpAt A B C D"
  shows "P PerpAt A B C E"
proof -
  have "C \<noteq> D"
    using assms(3) perp_in_distinct by blast
  have "Col P C D"
    using PerpAt_def assms(3) by blast
  hence "Col P C E" 
    using \<open>C \<noteq> D\<close> assms(2) col_trivial_2 colx by blast
  moreover
  {
    fix U V 
    assume "Col U A B" and "Col V C E" 
    hence "Per U P V"
      by (metis PerpAt_def assms(1) assms(2) assms(3) col_permutation_1 col_trivial_2 colx)
  }
  ultimately
  show ?thesis 
    using PerpAt_def assms(1) assms(3) by presburger
qed

lemma perp_col2_bis:
  assumes "A B Perp C D" and
    "Col C D P" and
    "Col C D Q" and
    "P \<noteq> Q"
  shows "A B Perp P Q"
  using Perp_cases assms(1) assms(2) assms(3) assms(4) perp_col0 by blast

lemma perp_in_perp_bis_R1:
  assumes "X \<noteq> A" and
    "X PerpAt A B C D"
  shows "X B Perp C D \<or> A X Perp C D"
  by (metis assms(2) l8_14_2_1a perp_col perp_in_col)

lemma perp_in_perp_bis:
  assumes "X PerpAt A B C D"
  shows "X B Perp C D \<or> A X Perp C D"
  by (metis assms l8_14_2_1a perp_in_perp_bis_R1)

lemma col_per_perp:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    (*   "D \<noteq> B" and  *)
    "D \<noteq> C" and
    "Col B C D" and
    "Per A B C"
  shows "C D Perp A B"
  by (metis Perp_cases assms(1) assms(2) assms(3) assms(4) assms(5) col_trivial_2 
      per_perp perp_col2_bis)

lemma per_cong_mid_R1:
  assumes "B = H" and
    (*  "B \<noteq> C" and *)
    "Bet A B C" and
    "Cong A H C H" and
    "Per H B C"
  shows "B Midpoint A C"
  using assms(1) assms(2) assms(3) midpoint_def not_cong_1243 by blast

lemma per_cong_mid_R2:
  assumes (*"B \<noteq> H" and *)
    "B \<noteq> C" and
    "Bet A B C" and
    "Cong A H C H" and
    "Per H B C"
  shows "B Midpoint A C"
proof -
  have P1: "Per C B H"
    using Per_cases assms(4) by blast
  have P2: "Per H B A"
    using assms(1) assms(2) assms(4) bet_col col_permutation_1 per_col by blast
  hence P3: "Per A B H"
    using Per_cases by blast
  obtain C' where P4: "B Midpoint C C' \<and> Cong H C H C'"
    using Per_def assms(4) by blast
  obtain H' where P5: "B Midpoint H H' \<and> Cong C H C H'"
    using P1 Per_def by blast
  obtain A' where P6: "B Midpoint A A' \<and> Cong H A H A'"
    using P2 Per_def by blast
  obtain H'' where P7: "B Midpoint H H'' \<and> Cong A H A H'"
    using P3 P5 Per_def Tarski_neutral_dimensionless_axioms symmetric_point_uniqueness 
    by fastforce
  hence P8: "H' = H''"
    using P5 symmetric_point_uniqueness by blast
  have "H B H' A IFSC H B H' C"
  proof -
    have Q1: "Bet H B H'"
      by (simp add: P7 P8 midpoint_bet)
    have Q2: "Cong H H' H H'"
      by (simp add: cong_reflexivity)
    have Q3: "Cong B H' B H'"
      by (simp add: cong_reflexivity)
    have Q4: "Cong H A H C"
      using assms(3) not_cong_2143 by blast
    have "Cong H' A H' C"
      using P5 P7 assms(3) cong_commutativity cong_inner_transitivity by blast
    thus ?thesis
      by (simp add: IFSC_def Q1 Q2 Q3 Q4)
  qed
  thus ?thesis
    using assms(1) assms(2) bet_col bet_neq23__neq l4_2 l7_20_bis by auto
qed

lemma per_cong_mid:
  assumes "B \<noteq> C" and
    "Bet A B C" and
    "Cong A H C H" and
    "Per H B C"
  shows "B Midpoint A C"
  using assms(1) assms(2) assms(3) assms(4) per_cong_mid_R1 per_cong_mid_R2 by blast

lemma per_double_cong:
  assumes "Per A B C" and
    "B Midpoint C C'"
  shows "Cong A C A C'"
  using Mid_cases Per_def assms(1) assms(2) l7_9_bis by blast

lemma cong_perp_or_mid_R1:
  assumes "Col A B X" and
    "A \<noteq> B" and
    "M Midpoint A B" and
    "Cong A X B X"
  shows "X = M \<or> \<not> Col A B X \<and> M PerpAt X M A B"
  using assms(1) assms(2) assms(3) assms(4) col_permutation_5 cong_commutativity l7_17_bis 
    l7_2 l7_20 by blast

lemma cong_perp_or_mid_R2:
  assumes "\<not> Col A B X" and
    "A \<noteq> B" and
    "M Midpoint A B" and
    "Cong A X B X"
  shows "X = M \<or> \<not> Col A B X \<and> M PerpAt X M A B"
proof -
  have P1: "Col M A B"
    by (simp add: assms(3) midpoint_col)
  have "Per X M A"
    using Per_def assms(3) assms(4) cong_commutativity by blast
  thus ?thesis
    by (metis P1 assms(1) assms(2) assms(3) midpoint_distinct_1 not_col_permutation_4 
        per_perp_in perp_in_col_perp_in perp_in_right_comm)
qed

lemma cong_perp_or_mid:
  assumes "A \<noteq> B" and
    "M Midpoint A B" and
    "Cong A X B X"
  shows "X = M \<or> \<not> Col A B X \<and> M PerpAt X M A B"
  using assms(1) assms(2) assms(3) cong_perp_or_mid_R1 cong_perp_or_mid_R2 by blast

lemma col_per2_cases:
  assumes "B \<noteq> C" and
    "B' \<noteq> C" and
    "C \<noteq> D" and
    "Col B C D" and
    "Per A B C" and
    "Per A B' C"
  shows  "B = B' \<or> \<not> Col B' C D"
  by (meson l8_7 Tarski_neutral_dimensionless_axioms assms(1) assms(2) assms(3) assms(4) 
      assms(5) assms(6) l6_16_1 per_col)

lemma l8_16_1:
  assumes "Col A B X" and
    "Col A B U" and
    "A B Perp C X"
  shows "\<not> Col A B C \<and> Per C X U"
  by (metis assms(1) assms(2) assms(3) l8_5 perp_col0 perp_left_comm perp_not_col2 perp_per_2)

lemma l8_16_2:
  assumes "Col A B X" and
    "Col A B U" and 
    "U \<noteq> X" and
    "\<not> Col A B C" and
    "Per C X U"
  shows "A B Perp C X"
proof -
  obtain X where "X PerpAt A B C X"
    by (metis NCol_perm assms(1) assms(2) assms(3) assms(4) assms(5) l8_13_2 l8_2 
        not_col_distincts)
  thus ?thesis 
    by (metis Perp_perm per_col assms(1) assms(2) assms(3) assms(4) assms(5) col3 col_per_perp
        not_col_distincts)
qed

lemma l8_18_uniqueness:
  assumes (*"\<not> Col A B C" and *)
    "Col A B X" and
    "A B Perp C X" and
    "Col A B Y" and
    "A B Perp C Y"
  shows "X = Y"
  using assms(1) assms(2) assms(3) assms(4) l8_16_1 l8_7 by blast

lemma midpoint_distinct:
  assumes "\<not> Col A B C" and
    "Col A B X" and
    "X Midpoint C C'"
  shows "C \<noteq> C'"
  using assms(1) assms(2) assms(3) l7_3 by auto

lemma l8_20_1_R1:
  assumes "A = B"
  shows "Per B A P"
  by (simp add: assms l8_2 l8_5)

lemma l8_20_1_R2:
  assumes "A \<noteq> B" and
    "Per A B C" and
    "P Midpoint C' D" and
    "A Midpoint C' C" and
    "B Midpoint D C"
  shows "Per B A P"
proof -
  obtain B' where P1: "A Midpoint B B'"
    using symmetric_point_construction by blast
  obtain D' where P2: "A Midpoint D D'"
    using symmetric_point_construction by blast
  obtain P' where P3: "A Midpoint P P'"
    using symmetric_point_construction by blast
  have P4: "Per B' B C"
    by (metis P1 Per_cases per_col assms(1) assms(2) midpoint_col not_col_permutation_4)
  have P5: "Per B B' C'"
  proof -
    have "Per B' B C"
      by (simp add: P4)
    have "B' B C Cong3 B B' C'"
      by (meson Cong3_def P1 assms(4) l7_13 l7_2)
    thus ?thesis
      using P4 l8_10 by blast
  qed
  have P6: "B' Midpoint D' C'"
    by (meson P1 P2 assms(4) assms(5) l7_15 l7_16 l7_2 midpoint_bet midpoint_cong midpoint_def)
  have P7: "P' Midpoint C D'"
    using P2 P3 assms(3) assms(4) symmetry_preserves_midpoint by blast
  have P8: "A Midpoint P P'"
    by (simp add: P3)
  obtain D'' where P9: "B Midpoint C D'' \<and> Cong B' C B' D"
    using P4 assms(5) l7_2 per_double_cong by blast
  have P10: "D'' = D"
    using P9 assms(5) l7_9_bis by blast
  obtain D'' where P11: "B' Midpoint C' D'' \<and> Cong B C' B D''"
    using P5 Per_def by blast
  have P12: "D' = D''"
    by (meson P11 P6 l7_9_bis Tarski_neutral_dimensionless_axioms)
  have P13: "P Midpoint C' D"
    using assms(3) by blast
  have P14: "Cong C D C' D'"
    using P2 assms(4) l7_13 l7_2 by blast
  have P15: "Cong C' D C D'"
    using P2 assms(4) cong_4321 l7_13 by blast
  have P16: "Cong P D P' D'"
    using P2 P8 cong_symmetry l7_13 by blast
  have P17: "Cong P D P' C"
    using P16 P7 cong_3421 cong_transitivity midpoint_cong by blast
  have P18: "C' P D B IFSC D' P' C B"
    by (metis Bet_cases IFSC_def P10 P11 P12 P13 P15 P17 P7 P9 cong_commutativity 
        cong_right_commutativity l7_13 l7_3_2 midpoint_bet)
  hence "Cong B P B P'"
    using l4_2 Tarski_neutral_dimensionless_axioms not_cong_2143 by fastforce
  thus ?thesis
    using P8 Per_def by blast
qed

lemma l8_20_1:
  assumes "Per A B C" and
    "P Midpoint C' D" and
    "A Midpoint C' C" and
    "B Midpoint D C"
  shows "Per B A P"
  using assms(1) assms(2) assms(3) assms(4) l8_20_1_R1 l8_20_1_R2 by fastforce

lemma l8_20_2:
  assumes "P Midpoint C' D" and
    "A Midpoint C' C" and
    "B Midpoint D C" and
    "B \<noteq> C"
  shows "A \<noteq> P"
  using assms(1) assms(2) assms(3) assms(4) l7_3 symmetric_point_uniqueness by blast

lemma perp_col1:
  assumes "C \<noteq> X" and
    "A B Perp C D" and
    "Col C D X"
  shows "A B Perp C X"
  using assms(1) assms(2) assms(3) col_trivial_3 perp_col2_bis by blast

lemma l8_18_existence:
  assumes "\<not> Col A B C"
  shows "\<exists> X. Col A B X \<and> A B Perp C X"
proof -
  obtain Y where "Bet B A Y" and "Cong A Y A C"
    using segment_construction by blast
  then obtain P where "P Midpoint C Y"
    using Mid_cases l7_25 by blast
  hence "Per A P Y" 
    using Per_def \<open>Cong A Y A C\<close> l7_2 by blast
  obtain Z where "Bet A Y Z" and "Cong Y Z Y P" 
    using segment_construction by blast
  obtain Q where "Bet P Y Q" and "Cong Y Q Y A"
    using segment_construction by blast
  obtain Q' where "Bet Q Z Q'" and "Cong Z Q' Q Z"
    using segment_construction by blast
  hence "Z Midpoint Q Q'"
    using midpoint_def not_cong_3412 by blast
  obtain C' where "Bet Q' Y C'" and "Cong Y C' Y C"
    using segment_construction by blast
  obtain X where "X Midpoint C C'" 
    using \<open>Cong Y C' Y C\<close> l7_2 l7_25 by blast
  have "A Y Z Q OFSC Q Y P A" 
    by (simp add: OFSC_def \<open>Bet A Y Z\<close> \<open>Bet P Y Q\<close> \<open>Cong Y Q Y A\<close> \<open>Cong Y Z Y P\<close> 
        between_symmetry cong_4321 cong_pseudo_reflexivity)
  have "A \<noteq> Y" 
    using \<open>Cong A Y A C\<close> assms is_midpoint_id l7_20 not_col_distincts by blast
  hence "Cong Z Q P A" 
    using \<open>A Y Z Q OFSC Q Y P A\<close> five_segment_with_def by auto
  hence "A P Y Cong3 Q Z Y" 
    using Cong3_def Cong_cases \<open>Cong Y Q Y A\<close> \<open>Cong Y Z Y P\<close> by blast
  hence "Per Q Z Y" 
    using \<open>Per A P Y\<close> l8_10 by blast
  hence "Per Y Z Q" 
    using l8_2 by blast
  have "P \<noteq> Y" 
    by (metis Col_def \<open>Bet B A Y\<close> \<open>P Midpoint C Y\<close> assms between_symmetry midpoint_not_midpoint)
  obtain Q'' where "Z Midpoint Q Q''" and "Cong Y Q Y Q'" 
    using \<open>Per Y Z Q\<close> \<open>Z Midpoint Q Q'\<close> per_double_cong by force
  hence "Q' = Q''" 
    by (meson symmetric_point_uniqueness \<open>Z Midpoint Q Q'\<close>)
  have "Bet Q Y C" 
    by (metis midpoint_bet \<open>Bet P Y Q\<close> \<open>P Midpoint C Y\<close> \<open>P \<noteq> Y\<close> between_symmetry 
        outer_transitivity_between2)
  hence "Bet Z Y X"
    by (meson l7_22 \<open>Bet Q' Y C'\<close> \<open>Cong Y C' Y C\<close> \<open>Cong Y Q Y Q'\<close> \<open>X Midpoint C C'\<close> 
        \<open>Z Midpoint Q Q'\<close> cong_symmetry)
  have "Q \<noteq> Y" 
    using \<open>A \<noteq> Y\<close> \<open>Cong Y Q Y A\<close> cong_reverse_identity by blast
  have "Bet C P Y" 
    using Midpoint_def \<open>P Midpoint C Y\<close> by auto
  hence "Per Y X C"
    using Per_def \<open>Cong Y C' Y C\<close> \<open>X Midpoint C C'\<close> cong_inner_transitivity 
      cong_reflexivity by blast
  moreover
  have "Col P Y Q" 
    by (simp add: Col_def \<open>Bet P Y Q\<close>)
  have "Col P Y C" 
    by (simp add: Col_def \<open>Bet C P Y\<close>)
  have "Col P Q C" 
    using \<open>Col P Y C\<close> \<open>Col P Y Q\<close> \<open>P \<noteq> Y\<close> col_transitivity_1 by blast
  have "Col Y Q C" 
    using Bet_cases Col_def \<open>Bet Q Y C\<close> by auto
  have "Col A Y B" 
    by (simp add: Col_def \<open>Bet B A Y\<close>)
  moreover
  have "Col A Y Z" 
    using Col_def \<open>Bet A Y Z\<close> by blast
  have "Col A B Z" 
    using \<open>A \<noteq> Y\<close> \<open>Col A Y B\<close> \<open>Col A Y Z\<close> col_transitivity_1 by blast
  have "Col Y B Z" 
    using \<open>A \<noteq> Y\<close> \<open>Col A Y B\<close> \<open>Col A Y Z\<close> col_transitivity_2 by blast
  have "Col Q Y P" 
    using Col_cases \<open>Col P Y Q\<close> by blast
  have "Q \<noteq> C" 
    using \<open>Bet Q Y C\<close> \<open>Q \<noteq> Y\<close> between_identity by blast
  have "Col Y Q' C'" 
    using Col_cases Col_def \<open>Bet Q' Y C'\<close> by blast
  {
    assume "Q = Q'"
    hence "Col P B C" 
      by (metis cong_reverse_identity \<open>Col A Y B\<close> \<open>Col P Y C\<close> \<open>Cong Z Q P A\<close> \<open>P \<noteq> Y\<close> 
          \<open>Q' = Q''\<close> \<open>Z Midpoint Q Q''\<close> col_transitivity_1 midpoint_distinct_2)
    moreover have "\<not> Col P B C" 
      by (metis \<open>Q = Q'\<close> \<open>Bet Q Z Q'\<close> \<open>Cong Z Q P A\<close> assms bet_neq12__neq cong_diff_3)
    ultimately have False 
      by blast
  }
  hence "C \<noteq> C'" 
    by (metis between_cong_3 \<open>Bet Q Y C\<close> \<open>Bet Q' Y C'\<close> \<open>Cong Y Q Y Q'\<close> \<open>P Midpoint C Y\<close> 
        \<open>P \<noteq> Y\<close> between_symmetry midpoint_distinct_3)
  have "Q Y C Z OFSC Q' Y C' Z" 
    by (simp add: OFSC_def \<open>Bet Q Y C\<close> \<open>Bet Q' Y C'\<close> \<open>Cong Y C' Y C\<close> \<open>Cong Y Q Y Q'\<close> 
        \<open>Cong Z Q' Q Z\<close> cong_3421 cong_reflexivity cong_symmetry)
  hence "Cong C Z C' Z" 
    using \<open>Q \<noteq> Y\<close> five_segment_with_def by force
  have "Col Z Y X" 
    using Col_def \<open>Bet Z Y X\<close> by blast
  have "Y \<noteq> Z" 
    using \<open>Cong Y Z Y P\<close> \<open>P \<noteq> Y\<close> cong_diff_4 by blast
  {
    assume "X = Y"
    hence "C' \<noteq> Y" 
      using \<open>C \<noteq> C'\<close> \<open>Cong Y C' Y C\<close> cong_reverse_identity by blast
    have "Col Y C' P" 
      by (metis Col_def Midpoint_def \<open>Bet C P Y\<close> \<open>X = Y\<close> \<open>X Midpoint C C'\<close> between_equality_2 
          col_transitivity_1 not_bet_distincts)
    hence "Col Y P Q'" 
      by (metis \<open>C' \<noteq> Y\<close> \<open>Col Y Q' C'\<close> col_trivial_3 colx not_col_permutation_5)
    hence "Col Y Q Q'" 
      by (meson \<open>Col P Y Q\<close> \<open>P \<noteq> Y\<close> colx not_col_distincts not_col_permutation_5)
    hence False 
      using l7_20 \<open>Cong Y Q Y Q'\<close> \<open>Q = Q' \<Longrightarrow> False\<close> \<open>Q' = Q''\<close> \<open>Y \<noteq> Z\<close> 
        \<open>Z Midpoint Q Q''\<close> col_permutation_4 l7_17 by blast
  }
  hence "X \<noteq> Y" 
    by auto
  moreover have "Col A B X" 
    by (meson \<open>Col A Y Z\<close> \<open>Col Y B Z\<close> \<open>Col Z Y X\<close> \<open>Y \<noteq> Z\<close> col3 col_permutation_3 
        not_col_permutation_1)
  ultimately show ?thesis 
    by (metis Col_cases l8_2 assms l8_16_2)
qed

lemma l8_21_aux:
  assumes "\<not> Col A B C"
  shows "\<exists> P. \<exists> T. (A B Perp P A \<and> Col A B T \<and> Bet C T P)"
proof -
  obtain X where "Col A B X" and "A B Perp C X"
    using assms l8_18_existence by blast
  hence "X PerpAt A B C X"
    by (simp add: l8_15_1)
  have "Per A X C" 
    using PerpAt_def \<open>X PerpAt A B C X\<close> col_trivial_1 by presburger
  obtain C' where "X Midpoint C C'" and "Cong A C A C'" 
    using Per_def \<open>Per A X C\<close> by auto
  obtain C'' where "A Midpoint C C''"
    using symmetric_point_construction by blast
  obtain P where "P Midpoint C' C''" 
    by (metis Cong_cases \<open>A Midpoint C C''\<close> \<open>Cong A C A C'\<close> cong_inner_transitivity 
        l7_25 midpoint_cong)
  have "Per X A P" 
  proof -
    have "P Midpoint C'' C'" 
      using \<open>P Midpoint C' C''\<close> l7_2 by blast
    moreover have "A Midpoint C'' C" 
      using \<open>A Midpoint C C''\<close> l7_2 by blast
    moreover have "X Midpoint C' C" 
      by (simp add: \<open>X Midpoint C C'\<close> l7_2)
    ultimately show ?thesis 
      using \<open>Per A X C\<close> l8_20_1 by blast
  qed
  have "X \<noteq> C" 
    using \<open>Col A B X\<close> assms by blast
  hence "A \<noteq> P" 
    using \<open>A Midpoint C C''\<close> \<open>P Midpoint C' C''\<close> \<open>X Midpoint C C'\<close> l7_9 midpoint_distinct_2 
    by blast
  have "\<exists> T. Bet P T C \<and> Bet A T X"
  proof -
    have "Bet C'' A C" 
      using Mid_cases \<open>A Midpoint C C''\<close> midpoint_bet by blast
    moreover have "Bet C' X C" 
      using Bet_cases Midpoint_def \<open>X Midpoint C C'\<close> by auto
    moreover have "Bet C'' P C'" 
      using \<open>P Midpoint C' C''\<close> between_symmetry midpoint_bet by blast
    ultimately show ?thesis 
      using l3_17 by fastforce
  qed
  then obtain T where "Bet P T C" and "Bet A T X" 
    by blast
  show ?thesis
  proof cases
    assume "A = X"
    thus ?thesis 
      by (metis Bet_perm between_identity midpoint_col midpoint_not_midpoint perp_col1 
          \<open>A B Perp C X\<close> \<open>A Midpoint C C''\<close> \<open>Bet A T X\<close> \<open>Bet P T C\<close> \<open>Col A B X\<close> 
          \<open>P Midpoint C' C''\<close> \<open>X Midpoint C C'\<close> l8_20_2 perp_comm)
  next
    assume "A \<noteq> X"
    have "A B Perp P A" 
      by (metis col3 l8_2 per_not_col \<open>A \<noteq> P\<close> \<open>A \<noteq> X\<close> \<open>Col A B X\<close> \<open>Per X A P\<close> 
          assms l8_16_2 not_col_distincts)
    moreover have "Col A B T" 
      by (metis Col_def \<open>A \<noteq> X\<close> \<open>Bet A T X\<close> \<open>Col A B X\<close> col_transitivity_2)
    moreover have "Bet C T P" 
      using Bet_cases \<open>Bet P T C\<close> by blast
    ultimately show ?thesis 
      by blast
  qed
qed

lemma l8_21:
  assumes "A \<noteq> B"
  shows "\<exists> P T. A B Perp P A \<and> Col A B T \<and> Bet C T P"
  by (meson assms between_trivial2 l8_21_aux not_col_exists)

lemma per_cong:
  assumes "A \<noteq> B" and
    "A \<noteq> P" and
    "Per B A P" and
    "Per A B R" and
    "Cong A P B R" and
    "Col A B X" and
    "Bet P X R"
  shows "Cong A R P B"
proof -
  have "Per P A B"
    using Per_cases assms(3) by blast
  obtain Q where "R Midpoint B Q"
    using symmetric_point_construction by auto
  have "B \<noteq> R"
    using assms(2) assms(5) cong_identity by blast
  hence "Per A B Q" 
    using Col_def Midpoint_def \<open>R Midpoint B Q\<close> assms(4) per_col by blast
  have "Per P A X" 
    using \<open>Per P A B\<close> assms(1) assms(6) per_col by blast
  have "B \<noteq> Q" 
    using \<open>B \<noteq> R\<close> \<open>R Midpoint B Q\<close> l7_3 by blast
  have "Per R B X" 
    by (metis col_permutation_4 per_col assms(1) assms(4) assms(6) l8_2)
  have "X \<noteq> A" 
    using \<open>B \<noteq> R\<close> assms(1) assms(2) assms(3) assms(4) assms(7) bet_col per_not_colp by blast
  obtain P' where "A Midpoint P P'" 
    using Per_def assms(3) by blast
  obtain R' where "Bet P' X R'" and "Cong X R' X R"   
    using segment_construction by blast
  obtain M where "M Midpoint R R'"
    using \<open>Cong X R' X R\<close> l7_2 l7_25 by blast
  have "Per X M R" 
    by (metis Per_def \<open>Cong X R' X R\<close> \<open>M Midpoint R R'\<close> cong_symmetry)
  have "Cong X P X P'" 
    using Per_cases \<open>A Midpoint P P'\<close> \<open>Per P A X\<close> per_double_cong by blast
  have "X \<noteq> P'" 
    using \<open>Cong X P X P'\<close> \<open>Per P A X\<close> assms(2) cong_identity l8_8 by blast
  have "P \<noteq> P'" 
    using \<open>A Midpoint P P'\<close> assms(2) l7_3 by blast
  have "\<not> Col X P P'" 
    using \<open>A Midpoint P P'\<close> \<open>Cong X P X P'\<close> \<open>P \<noteq> P'\<close> \<open>X \<noteq> A\<close> col_permutation_4 l7_17 l7_20 
    by blast
  have "Bet A X M" 
    by (meson \<open>A Midpoint P P'\<close> \<open>Bet P' X R'\<close> \<open>Cong X P X P'\<close> \<open>Cong X R' X R\<close> \<open>M Midpoint R R'\<close> 
        assms(7) cong_symmetry l7_22)
  have "X \<noteq> R" 
    using \<open>B \<noteq> R\<close> \<open>Per R B X\<close> l8_8 by blast
  have "X \<noteq> R'" 
    using \<open>Cong X R' X R\<close> \<open>X \<noteq> R\<close> cong_diff_3 by blast
  {
    assume "X = M"
    have "Col X P P'" 
    proof -
      have "Col X R P" 
        using Col_def assms(7) by blast
      moreover have "Col X R P'" 
        by (metis Col_def \<open>Bet P' X R'\<close> \<open>M Midpoint R R'\<close> \<open>X = M\<close> \<open>X \<noteq> R'\<close> 
            col_transitivity_1 midpoint_col)
      ultimately show ?thesis 
        using \<open>X \<noteq> R\<close> col_transitivity_1 by blast
    qed
    hence False 
      by (simp add: \<open>\<not> Col X P P'\<close>)
  }
  have "M = B" 
  proof -
    have "\<not> Col A X R" 
      by (metis \<open>B \<noteq> R\<close> \<open>X \<noteq> A\<close> assms(1) assms(4) assms(6) col_trivial_3 colx per_not_col)
    moreover have "Col A X M" 
      using Col_def \<open>Bet A X M\<close> by auto
    moreover have "A X Perp R M" 
      by (metis Col_cases Per_cases \<open>Per X M R\<close> \<open>X = M \<Longrightarrow> False\<close> \<open>X \<noteq> A\<close> calculation(1) 
          calculation(2) col_per_perp perp_left_comm)
    moreover have "Col A X B" 
      using Col_cases assms(6) by blast
    moreover have "A X Perp R B" 
      by (metis Col_cases Per_cases \<open>B \<noteq> R\<close> \<open>X \<noteq> A\<close> assms(1) assms(4) assms(6) col_per_perp)
    ultimately show ?thesis 
      using l8_18_uniqueness by blast
  qed
  have "P X R P' OFSC P' X R' P" 
    using Cong_cases OFSC_def \<open>Bet P' X R'\<close> \<open>Cong X P X P'\<close> \<open>Cong X R' X R\<close> assms(7) 
      cong_pseudo_reflexivity by auto
  hence "Cong R P' R' P" 
    using \<open>\<not> Col X P P'\<close> five_segment_with_def not_col_distincts by blast
  have "P' A P R IFSC R' B R P"
  proof -
    have "M = B"
      using \<open>M = B\<close> by auto
    then have "B Midpoint R R'"
      using \<open>M Midpoint R R'\<close> by blast
    then show ?thesis
      by (smt (z3) Bet_cases Cong_cases IFSC_def Midpoint_def \<open>A Midpoint P P'\<close> 
          \<open>Cong R P' R' P\<close> assms(5) cong_mid2__cong cong_pseudo_reflexivity)
  qed
  thus ?thesis
    using l4_2 not_cong_1243 by blast
qed

lemma perp_cong:
  assumes "A \<noteq> B" and
    "A \<noteq> P" and
    "A B Perp P A" and
    "A B Perp R B" and
    "Cong A P B R" and
    "Col A B X" and
    "Bet P X R"
  shows "Cong A R P B"
  using Perp_cases assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) assms(7) per_cong 
    perp_per_1 by blast

lemma perp_exists:
  assumes "A \<noteq> B"
  shows "\<exists> X. PO X Perp A B"
proof cases
  assume "Col A B PO"
  then obtain C where "A \<noteq> C" and "B \<noteq> C" and "PO \<noteq> C" and "Col A B C"
    using diff_col_ex3 by blast
  then obtain P T where "PO C Perp P PO" and "Col PO C T" and "Bet PO T P" 
    using l8_21 by blast
  hence "PO P Perp A B" 
    by (metis Perp_perm col_transitivity_2 \<open>Col A B C\<close> \<open>Col A B PO\<close> assms 
        not_col_permutation_2 perp_col0)
  thus ?thesis
    by blast
next
  assume "\<not> Col A B PO"
  thus ?thesis 
    using l8_18_existence assms col_trivial_2 col_trivial_3 l8_18_existence perp_col0 by blast
qed

lemma perp_vector:
  assumes "A \<noteq> B"
  shows "\<exists> X Y. A B Perp X Y"
  using assms l8_21 by blast

lemma midpoint_existence_aux:
  assumes "A \<noteq> B" and
    "A B Perp Q B" and
    "A B Perp P A" and
    "Col A B T" and
    "Bet Q T P" and
    "A P Le B Q"
  shows "\<exists> X. X Midpoint A B"
proof -
  obtain R where "Bet B R Q" and "Cong A P B R"
    using Le_def assms(6) by blast
  obtain X where "Bet T X B" and "Bet R X P" 
    by (meson \<open>Bet B R Q\<close> assms(5) between_symmetry inner_pasch)
  have "Col A B X" 
    by (metis Col_def \<open>Bet T X B\<close> assms(4) between_equality_2 between_trivial2 col2__eq)
  have "B \<noteq> R"
    using \<open>Cong A P B R\<close> assms(3) cong_identity perp_not_eq_2 by blast
  have "\<not> Col A B Q" 
    using Col_cases Perp_cases assms(2) perp_not_col by blast
  have "\<not> Col A B R" 
    using Col_def \<open>B \<noteq> R\<close> \<open>Bet B R Q\<close> \<open>\<not> Col A B Q\<close> l6_16_1 by blast
  have "P \<noteq> R" 
    using \<open>Bet R X P\<close> \<open>Col A B X\<close> \<open>\<not> Col A B R\<close> between_identity by blast
  have "\<exists> X. X Midpoint A B"
  proof cases
    assume "A = P"
    thus ?thesis
      using assms(3) col_trivial_3 perp_not_col2 by blast
  next
    assume "\<not> A = P"
    have "A B Perp R B" 
      by (metis Col_def \<open>B \<noteq> R\<close> \<open>Bet B R Q\<close> assms(2) not_col_distincts perp_col2_bis)
    hence "Cong A R P B" 
      using \<open>A \<noteq> P\<close> \<open>Bet R X P\<close> \<open>Col A B X\<close> \<open>Cong A P B R\<close> assms(1) assms(3) 
        between_symmetry perp_cong by blast
    hence "X Midpoint A B \<and> X Midpoint P R" 
      by (meson \<open>Bet R X P\<close> \<open>Col A B X\<close> \<open>Cong A P B R\<close> \<open>P \<noteq> R\<close> assms(3) bet_col 
          between_symmetry cong_4312 l7_2 l7_21 not_col_permutation_2 perp_not_col)
    thus ?thesis
      by blast
  qed
  thus ?thesis by blast
qed

lemma midpoint_existence:
  "\<exists> X. X Midpoint A B"
proof cases
  assume "A = B"
  thus ?thesis
    using l7_3_2 by blast
next
  assume P1: "\<not> A = B"
  obtain Q where P2: "A B Perp B Q"
    by (metis P1 l8_21 perp_comm)
  obtain P T where P3: "A B Perp P A \<and> Col A B T \<and> Bet Q T P"
    using P2 l8_21_aux not_col_distincts perp_not_col2 by blast
  have P4: "A P Le B Q \<or> B Q Le A P"
    by (simp add: local.le_cases)
  have P5: "A P Le B Q \<longrightarrow> (\<exists> X. X Midpoint A B)"
    by (meson P1 P2 P3 Perp_cases midpoint_existence_aux Tarski_neutral_dimensionless_axioms)
  have P6: "B Q Le A P \<longrightarrow> (\<exists> X. X Midpoint A B)"
  proof -
    {
      assume H1: "B Q Le A P"
      have Q6: "B \<noteq> A"
        using P1 by auto
      have Q2: "B A Perp P A"
        by (simp add: P3 perp_left_comm)
      have Q3: "B A Perp Q B"
        using P2 Perp_perm by blast
      have Q4: "Col B A T"
        using Col_perm P3 by blast
      have Q5: "Bet P T Q"
        using Bet_perm P3 by blast
      obtain X where "X Midpoint B A"
        using H1 Q2 Q3 Q4 Q5 Q6 midpoint_existence_aux by blast
      hence "\<exists> X. X Midpoint A B"
        using l7_2 by blast
    }
    thus ?thesis
      by simp
  qed
  thus ?thesis
    using P4 P5 by blast
qed

lemma MidR_uniq_aux:
  shows "\<exists>!x. x Midpoint A B"
  using l7_17_bis midpoint_existence by blast

lemma SymR_uniq_aux:
  assumes "B Midpoint A x" and
    "B Midpoint A y"
  shows "x = y"
  using assms(1) assms(2) symmetric_point_uniqueness by auto

lemma perp_in_id:
  assumes "X PerpAt A B C A"
  shows "X = A"
  by (meson Col_cases assms col_trivial_3 l8_14_2_1b)

lemma l8_22:
  assumes "A \<noteq> B" and
    "A \<noteq> P" and
    "Per B A P" and
    "Per A B R" and
    "Cong A P B R" and
    "Col A B X" and
    "Bet P X R" and
    "Cong A R P B"
  shows "X Midpoint A B \<and> X Midpoint P R"
  by (metis assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) assms(7) assms(8) 
      bet_col cong_commutativity cong_diff cong_right_commutativity l7_21 
      not_col_permutation_5 per_not_colp)

lemma l8_22_bis:
  assumes "A \<noteq> B" and
    "A \<noteq> P" and
    "A B Perp P A" and
    "A B Perp R B" and
    "Cong A P B R" and
    "Col A B X" and
    "Bet P X R"
  shows "Cong A R P B \<and> X Midpoint A B \<and> X Midpoint P R"
  by (metis l8_22 Perp_cases assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) 
      assms(7) perp_cong perp_per_2)

lemma perp_in_perp:
  assumes "X PerpAt A B C D"
  shows "A B Perp C D"
  using assms l8_14_2_1a by auto

lemma perp_proj:
  assumes "A B Perp C D" and
    "\<not> Col A C D"
  shows "\<exists> X. Col A B X \<and> A X Perp C D"
  using assms(1) not_col_distincts by auto

lemma l8_24 :
  assumes "P A Perp A B" and
    "Q B Perp A B" and
    "Col A B T" and
    "Bet P T Q" and
    "Bet B R Q" and
    "Cong A P B R"
  shows "\<exists> X. X Midpoint A B \<and> X Midpoint P R"
proof -
  obtain X where P1: "Bet T X B \<and> Bet R X P"
    using assms(4) assms(5) inner_pasch by blast
  have P2: "Col A B X"
    by (metis Out_cases P1 assms(3) bet_out_1 col_out2_col not_col_distincts out_trivial)
  have P3: "A \<noteq> B"
    using assms(1) col_trivial_2 l8_16_1 by blast
  have P4: "A \<noteq> P"
    using assms(1) col_trivial_1 l8_16_1 by blast
  have "\<exists> X. X Midpoint A B \<and> X Midpoint P R"
  proof cases
    assume "Col A B P"
    thus ?thesis
      using Perp_perm assms(1) perp_not_col by blast
  next
    assume Q1: "\<not> Col A B P"
    have Q2: "B \<noteq> R"
      using P4 assms(6) cong_diff by blast
    have Q3: "Q \<noteq> B"
      using Q2 assms(5) between_identity by blast
    have Q4: "\<not> Col A B Q"
      by (metis assms(2) col_permutation_3 l8_14_1 perp_col1 perp_not_col)
    have Q5: "\<not> Col A B R"
      by (meson Q2 Q4 assms(5) bet_col col_transitivity_1 not_col_permutation_2)
    have Q6: "P \<noteq> R"
      using P1 P2 Q5 between_identity by blast
    have "\<exists> X. X Midpoint A B \<and> X Midpoint P R"
    proof cases
      assume "A = P"
      thus ?thesis
        using P4 by blast
    next
      assume R0: "\<not> A = P"
      have R1: "A B Perp R B"
        by (metis Perp_cases Q2 bet_col1 assms(2) assms(5) bet_col col_transitivity_1 perp_col1)
      have R2: "Cong A R P B"
        using P1 P2 P3 Perp_perm R0 R1 assms(1) assms(6) between_symmetry perp_cong by blast
      have R3: "\<not> Col A P B"
        using Col_perm Q1 by blast
      have R4: "P \<noteq> R"
        by (simp add: Q6)
      have R5: "Cong A P B R"
        by (simp add: assms(6))
      have R6: "Cong P B R A"
        using R2 not_cong_4312 by blast
      have R7: "Col A X B"
        using Col_perm P2 by blast
      have R8: "Col P X R"
        by (simp add: P1 bet_col between_symmetry)
      thus ?thesis using l7_21
        using R3 R4 R5 R6 R7 by blast
    qed
    thus ?thesis by simp
  qed
  thus ?thesis
    by simp
qed

lemma col_per2__per:
  assumes "A \<noteq> B" and
    "Col A B C" and
    "Per A X P" and
    "Per B X P"
  shows "Per C X P"
  by (meson Per_def assms(1) assms(2) assms(3) assms(4) l4_17 per_double_cong)

lemma perp_in_per_1:
  assumes "X PerpAt A B C D"
  shows "Per A X C"
  using PerpAt_def assms col_trivial_1 by auto

lemma perp_in_per_2:
  assumes "X PerpAt A B C D"
  shows "Per A X D"
  using assms perp_in_per_1 perp_in_right_comm by blast

lemma perp_in_per_3:
  assumes "X PerpAt A B C D"
  shows "Per B X C"
  using assms perp_in_comm perp_in_per_2 by blast

lemma perp_in_per_4:
  assumes "X PerpAt A B C D"
  shows "Per B X D"
  using assms perp_in_per_3 perp_in_right_comm by blast

lemma coplanar_perm_1:
  assumes "Coplanar A B C D"
  shows "Coplanar A B D C"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_2:
  assumes "Coplanar A B C D"
  shows "Coplanar A C B D"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_3:
  assumes "Coplanar A B C D"
  shows "Coplanar A C D B"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_4:
  assumes "Coplanar A B C D"
  shows "Coplanar A D B C"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_5:
  assumes "Coplanar A B C D"
  shows "Coplanar A D C B"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_6:
  assumes "Coplanar A B C D"
  shows "Coplanar B A C D"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_7:
  assumes "Coplanar A B C D"
  shows "Coplanar B A D C"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_8:
  assumes "Coplanar A B C D"
  shows "Coplanar B C A D"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_9:
  assumes "Coplanar A B C D"
  shows "Coplanar B C D A"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_10:
  assumes "Coplanar A B C D"
  shows "Coplanar B D A C"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_11:
  assumes "Coplanar A B C D"
  shows "Coplanar B D C A"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_12:
  assumes "Coplanar A B C D"
  shows "Coplanar C A B D"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_13:
  assumes "Coplanar A B C D"
  shows "Coplanar C A D B"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_14:
  assumes "Coplanar A B C D"
  shows "Coplanar C B A D"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_15:
  assumes "Coplanar A B C D"
  shows "Coplanar C B D A"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_16:
  assumes "Coplanar A B C D"
  shows "Coplanar C D A B"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_17:
  assumes "Coplanar A B C D"
  shows "Coplanar C D B A"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_18:
  assumes "Coplanar A B C D"
  shows "Coplanar D A B C"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_19:
  assumes "Coplanar A B C D"
  shows "Coplanar D A C B"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_20:
  assumes "Coplanar A B C D"
  shows "Coplanar D B A C"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_21:
  assumes "Coplanar A B C D"
  shows "Coplanar D B C A"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_22:
  assumes "Coplanar A B C D"
  shows "Coplanar D C A B"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma coplanar_perm_23:
  assumes "Coplanar A B C D"
  shows "Coplanar D C B A"
proof -
  obtain X where "(Col A B X \<and> Col C D X) \<or> (Col A C X \<and> Col B D X) \<or> (Col A D X \<and> Col B C X)"
    using Coplanar_def assms by blast
  thus ?thesis
    using Coplanar_def col_permutation_4 by blast
qed

lemma ncoplanar_perm_1:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar A B D C"
  using assms coplanar_perm_1 by blast

lemma ncoplanar_perm_2:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar A C B D"
  using assms coplanar_perm_2 by blast

lemma ncoplanar_perm_3:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar A C D B"
  using assms coplanar_perm_4 by blast

lemma ncoplanar_perm_4:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar A D B C"
  using assms coplanar_perm_3 by blast

lemma ncoplanar_perm_5:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar A D C B"
  using assms coplanar_perm_5 by blast

lemma ncoplanar_perm_6:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar B A C D"
  using assms coplanar_perm_6 by blast

lemma ncoplanar_perm_7:
  assumes "\<not>  Coplanar A B C D"
  shows "\<not> Coplanar B A D C"
  using assms coplanar_perm_7 by blast

lemma ncoplanar_perm_8:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar B C A D"
  using assms coplanar_perm_12 by blast

lemma ncoplanar_perm_9:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar B C D A"
  using assms coplanar_perm_18 by blast

lemma ncoplanar_perm_10:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar B D A C"
  using assms coplanar_perm_13 by blast

lemma ncoplanar_perm_11:
  assumes "\<not> Coplanar A B C D"
  shows "\<not>  Coplanar B D C A"
  using assms coplanar_perm_19 by blast

lemma ncoplanar_perm_12:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar C A B D"
  using assms coplanar_perm_8 by blast

lemma ncoplanar_perm_13:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar C A D B"
  using assms coplanar_perm_10 by blast

lemma ncoplanar_perm_14:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar C B A D"
  using assms coplanar_perm_14 by blast

lemma ncoplanar_perm_15:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar C B D A"
  using assms coplanar_perm_20 by blast

lemma ncoplanar_perm_16:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar C D A B"
  using assms coplanar_perm_16 by blast

lemma ncoplanar_perm_17:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar C D B A"
  using assms coplanar_perm_22 by blast

lemma ncoplanar_perm_18:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar D A B C"
  using assms coplanar_perm_9 by blast

lemma ncoplanar_perm_19:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar D A C B"
  using assms coplanar_perm_11 by blast

lemma ncoplanar_perm_20:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar D B A C"
  using assms coplanar_perm_15 by blast

lemma ncoplanar_perm_21:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar D B C A"
  using assms coplanar_perm_21 by blast

lemma ncoplanar_perm_22:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Coplanar D C A B"
  using assms coplanar_perm_17 by blast

lemma ncoplanar_perm_23:
  assumes "\<not>  Coplanar A B C D"
  shows "\<not> Coplanar D C B A"
  using assms coplanar_perm_23 by blast

lemma coplanar_trivial:
  shows "Coplanar A A B C"
  using Coplanar_def NCol_cases col_trivial_1 by blast

lemma col__coplanar:
  assumes "Col A B C"
  shows "Coplanar A B C D"
  using Coplanar_def assms not_col_distincts by blast

lemma ncop__ncol:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Col A B C"
  using assms col__coplanar by blast

lemma ncop__ncols:
  assumes "\<not> Coplanar A B C D"
  shows "\<not> Col A B C \<and> \<not> Col A B D \<and> \<not> Col A C D \<and> \<not> Col B C D"
  by (meson assms col__coplanar coplanar_perm_4 ncoplanar_perm_9)

lemma bet__coplanar:
  assumes "Bet A B C"
  shows "Coplanar A B C D"
  using assms bet_col ncop__ncol by blast

lemma out__coplanar:
  assumes "A Out B C"
  shows "Coplanar A B C D"
  using assms col__coplanar out_col by blast

lemma midpoint__coplanar:
  assumes "A Midpoint B C"
  shows "Coplanar A B C D"
  using assms midpoint_col ncop__ncol by blast

lemma perp__coplanar:
  assumes "A B Perp C D"
  shows "Coplanar A B C D"
proof -
  obtain P where "P PerpAt A B C D"
    using Perp_def assms by blast
  thus ?thesis
    using Coplanar_def perp_in_col by blast
qed

lemma ts__coplanar:
  assumes "A B TS C D"
  shows "Coplanar A B C D"
  by (metis Coplanar_def TS_def assms bet_col col_permutation_2 col_permutation_3)

lemma reflectl__coplanar:
  assumes "A B ReflectL C D"
  shows "Coplanar A B C D"
  by (metis ReflectL_def perp__coplanar assms col__coplanar col_trivial_1 ncoplanar_perm_17)

lemma reflect__coplanar:
  assumes "A B Reflect C D"
  shows "Coplanar A B C D"
  by (metis Reflect_def reflectl__coplanar assms col_trivial_2 ncop__ncols)

lemma inangle__coplanar:
  assumes "A InAngle B C D"
  shows "Coplanar A B C D"
proof -
  obtain X where "Bet B X D \<and> (X = C \<or> C Out X A)"
    using InAngle_def assms by auto
  thus ?thesis
    by (meson Col_cases Coplanar_def bet_col ncop__ncols out_col)
qed

lemma pars__coplanar:
  assumes "A B ParStrict C D"
  shows "Coplanar A B C D"
  using ParStrict_def assms by auto

lemma par__coplanar:
  assumes "A B Par C D"
  shows "Coplanar A B C D"
  using Par_def assms ncop__ncols pars__coplanar by blast

lemma plg__coplanar:
  assumes "Plg A B C D"
  shows "Coplanar A B C D"
proof -
  obtain M where "Bet A M C \<and> Bet B M D"
    by (meson Plg_def assms midpoint_bet)
  thus ?thesis
    by (metis InAngle_def bet_out_1 inangle__coplanar ncop__ncols not_col_distincts)
qed

lemma plgs__coplanar:
  assumes "ParallelogramStrict A B C D"
  shows "Coplanar A B C D"
  using ParallelogramStrict_def assms par__coplanar by blast

lemma plgf__coplanar:
  assumes "ParallelogramFlat A B C D"
  shows "Coplanar A B C D"
  using ParallelogramFlat_def assms col__coplanar by auto

lemma parallelogram__coplanar:
  assumes "Parallelogram A B C D"
  shows "Coplanar A B C D"
  using Parallelogram_def assms plgf__coplanar plgs__coplanar by auto

lemma rhombus__coplanar:
  assumes "Rhombus A B C D"
  shows "Coplanar A B C D"
  using Rhombus_def assms plg__coplanar by blast

lemma rectangle__coplanar:
  assumes "Rectangle A B C D"
  shows "Coplanar A B C D"
  using Rectangle_def assms plg__coplanar by blast

lemma square__coplanar:
  assumes "Square A B C D"
  shows "Coplanar A B C D"
  using Square_def assms rectangle__coplanar by blast

lemma lambert__coplanar:
  assumes "Lambert A B C D"
  shows "Coplanar A B C D"
  using Lambert_def assms by presburger

lemma ts_distincts:
  assumes "A B TS P Q"
  shows "A \<noteq> B \<and> A \<noteq> P \<and> A \<noteq> Q \<and> B \<noteq> P \<and> B \<noteq> Q \<and> P \<noteq> Q"
  using TS_def assms bet_neq12__neq not_col_distincts by blast

lemma l9_2:
  assumes "A B TS P Q"
  shows "A B TS Q P"
  using TS_def assms between_symmetry by blast

lemma invert_two_sides:
  assumes "A B TS P Q"
  shows "B A TS P Q"
  using TS_def assms not_col_permutation_5 by blast

lemma l9_3:
  assumes "P Q TS A C" and
    "Col M P Q" and
    "M Midpoint A C" and
    "Col R P Q" and
    "R Out A B"
  shows "P Q TS B C"
proof -
  have "\<not> Col A P Q"
    using TS_def assms(1) by blast
  hence "P \<noteq> Q"
    using not_col_distincts by auto
  obtain T where "Col T P Q" and "Bet A T C"
    using assms(2) assms(3) midpoint_bet by blast
  have "A \<noteq> C"
    using assms(1) ts_distincts by blast
  have "T = M" 
  proof -
    have "Bet A M C" 
      using assms(3) midpoint_bet by blast
    hence "Col A M C" 
      using Col_def by force
    moreover have "Col A T C" 
      using \<open>Bet A T C\<close> bet_col by blast
    moreover have "Col R A B" 
      using assms(5) out_col by auto
    ultimately show ?thesis
      by (meson l6_21 \<open>A \<noteq> C\<close> \<open>Col T P Q\<close> \<open>\<not> Col A P Q\<close> assms(2) col_permutation_3 
          col_permutation_5)
  qed
  have "P Q TS B C"
  proof cases
    assume "C = M"
    thus ?thesis 
      using \<open>A \<noteq> C\<close> assms(3) is_midpoint_id_2 by blast
  next
    assume "\<not> C = M"
    have "\<not> Col B P Q" 
      by (metis \<open>\<not> Col A P Q\<close> assms(4) assms(5) col_permutation_2 colx out_col out_diff2)
    have "Bet R A B \<or> Bet R B A"
      using Out_def assms(5) by auto
    {
      assume "Bet R A B"
      obtain B' where "M Midpoint B B'"
        using symmetric_point_construction by blast
      obtain R' where "M Midpoint R R'"
        using symmetric_point_construction by blast
      have "Bet B' C R'" 
        using l7_15 \<open>Bet R A B\<close> \<open>M Midpoint B B'\<close> \<open>M Midpoint R R'\<close> assms(3) between_symmetry 
        by fastforce
      have "\<exists> X. Bet M X R' \<and> Bet C X B" 
      proof -
        have "Bet B M B'" 
          using Midpoint_def \<open>M Midpoint B B'\<close> by auto
        moreover have "Bet R' C B'" 
          using Bet_cases \<open>Bet B' C R'\<close> by auto
        ultimately show ?thesis 
          by (simp add: inner_pasch)
      qed
      then obtain X where "Bet M X R'" and "Bet C X B" 
        by blast
      have "Col X P Q"
      proof -
        have "Col P M R" 
          using \<open>P \<noteq> Q\<close> assms(2) assms(4) l6_16_1 not_col_permutation_2 by blast
        have "Col Q M R" 
          by (metis l6_16_1 \<open>Col P M R\<close> assms(2) assms(4) col_permutation_2)
        {
          assume "M = X"
          hence "Col X P Q"
            using assms(2) by blast
        }
        hence "M = X \<longrightarrow> Col X P Q" by simp
        {
          assume "M \<noteq> X"
          hence "M \<noteq> R'" 
            using \<open>Bet M X R'\<close> between_identity by blast
          hence "M \<noteq> R" 
            using \<open>M Midpoint R R'\<close> is_midpoint_id by blast
          hence "Col X P Q" 
            by (metis \<open>Bet M X R'\<close> \<open>M Midpoint R R'\<close> \<open>M \<noteq> R'\<close> assms(2) assms(4) 
                bet_col col_permutation_4 col_permutation_5 colx midpoint_col)
        }
        hence "M \<noteq> X \<longrightarrow> Col X P Q" by simp
        thus ?thesis 
          using assms(2) by blast
      qed
      have "Bet B X C" 
        using Bet_cases \<open>Bet C X B\<close> by blast
      hence "P Q TS B C" 
        using TS_def \<open>Col X P Q\<close> \<open>\<not> Col B P Q\<close> assms(1) by blast
    }
    hence "Bet R A B \<longrightarrow> P Q TS B C" by simp
    {
      assume "Bet R B A"
      have "Bet C M A" 
        using Bet_cases \<open>Bet A T C\<close> \<open>T = M\<close> by blast
      then obtain X where "Bet B X C" and "Bet M X R" 
        using \<open>Bet R B A\<close> inner_pasch by blast
      have "Col X P Q"  
        by (metis Col_def \<open>Bet M X R\<close> assms(2) assms(4) between_equality_2 
            between_trivial2 col_transitivity_1)
      hence "P Q TS B C" 
        using TS_def \<open>Bet B X C\<close> \<open>\<not> Col B P Q\<close> assms(1) by blast
    }
    hence "Bet R B A \<longrightarrow> P Q TS B C" by simp
    thus ?thesis 
      using \<open>Bet R A B \<Longrightarrow> P Q TS B C\<close> \<open>Bet R A B \<or> Bet R B A\<close> by blast
  qed
  thus ?thesis by blast
qed

lemma mid_preserves_col:
  assumes "Col A B C" and
    "M Midpoint A A'" and
    "M Midpoint B B'" and
    "M Midpoint C C'"
  shows "Col A' B' C'"
  using Col_def assms(1) assms(2) assms(3) assms(4) l7_15 by auto

lemma per_mid_per:
  assumes (*"A \<noteq> B" and*)
    "Per X A B" and
    "M Midpoint A B" and
    "M Midpoint X Y"
  shows "Cong A X B Y \<and> Per Y B A"
  by (meson Cong3_def Mid_perm assms(1) assms(2) assms(3) l7_13 l8_10)

lemma sym_preserve_diff:
  assumes "A \<noteq> B" and
    "M Midpoint A A'" and
    "M Midpoint B B'"
  shows "A'\<noteq> B'"
  using assms(1) assms(2) assms(3) l7_9 by blast

lemma l9_4_1_aux_R1:
  assumes "R = S" and
    "S C Le R A" and
    "P Q TS A C" and
    "Col R P Q" and
    "P Q Perp A R" and
    "Col S P Q" and
    "P Q Perp C S" and
    "M Midpoint R S"
  shows "\<forall> U C'. M Midpoint U C' \<longrightarrow> (R Out U A \<longleftrightarrow> S Out C C')"
proof -
  have "M = R"
    using assms(1) assms(8) l7_3 by blast
  have "\<not> Col A P Q"
    using TS_def assms(3) by auto
  hence "P \<noteq> Q"
    using not_col_distincts by blast
  obtain T where "Col T P Q" and "Bet A T C"
    using TS_def assms(3) by blast
  {
    assume "\<not> M = T"
    hence "M PerpAt M T A M" 
      using perp_col2 assms(4) assms(5) not_col_permutation_3 perp_left_comm perp_perp_in
      by (metis \<open>Col T P Q\<close> \<open>M = R\<close>)
    hence "M T Perp C M"
      using \<open>M \<noteq> T\<close> assms(1) assms(4) assms(7) col_permutation_1 perp_col2 \<open>Col T P Q\<close> \<open>M = R\<close> 
      by blast
    hence "Per T M A"
      using \<open>M PerpAt M T A M\<close> perp_in_per_3 by blast
    have "Per T M C"
      by (simp add: \<open>M T Perp C M\<close> perp_per_1)
    have "M = T"
    proof -
      have "Per C M T"
        by (simp add: \<open>Per T M C\<close> l8_2)
      thus ?thesis 
        using l8_6 l8_2 \<open>Bet A T C\<close> \<open>Per T M A\<close> by blast
    qed
    hence "False"
      using \<open>M \<noteq> T\<close> by blast
  }
  hence "M = T" by blast
  have "\<forall> U C'. ((M Midpoint U C' \<and> M Out U A) \<longrightarrow> M Out C C')"
  proof -
    {
      fix U C'
      assume "M Midpoint U C'" and "M Out U A"
      have "C \<noteq> M"
        using assms(1) assms(7) perp_not_eq_2 \<open>M = R\<close> by blast
      have "C' \<noteq> M"
        using midpoint_not_midpoint out_diff1 \<open>M Midpoint U C'\<close> \<open>M Out U A\<close> by blast
      have "Bet U M C"
        using bet_out__bet l6_6 \<open>Bet A T C\<close> \<open>M = T\<close> \<open>M Out U A\<close> by blast
      hence "M Out C C'" 
        by (metis Out_def midpoint_bet \<open>C \<noteq> M\<close> \<open>C' \<noteq> M\<close> \<open>M Midpoint U C'\<close> \<open>M Out U A\<close> l5_2)
    }
    thus ?thesis by blast
  qed
  have "\<forall> U C'. ((M Midpoint U C' \<and> M Out C C') \<longrightarrow> M Out U A)"
  proof -
    {
      fix U C'
      assume "M Midpoint U C'" and "M Out C C'"
      have "C \<noteq> M"
        using assms(1) assms(7) perp_not_eq_2 \<open>M = R\<close> by blast
      have "C' \<noteq> M"
        using l6_3_1 \<open>M Out C C'\<close> by blast
      have "Bet U M C"
        using Out_def between_inner_transitivity midpoint_bet outer_transitivity_between 
        by (metis \<open>M Midpoint U C'\<close> \<open>M Out C C'\<close>)
      hence "M Out U A"
        using l6_2 midpoint_distinct_1 
        by (metis \<open>Bet A T C\<close> \<open>C \<noteq> M\<close> \<open>C' \<noteq> M\<close> \<open>M = R\<close> \<open>M Midpoint U C'\<close> 
            \<open>M \<noteq> T \<Longrightarrow> False\<close> \<open>\<not> Col A P Q\<close> assms(4))
    }
    thus ?thesis by blast
  qed
  thus ?thesis 
    using \<open>M = R\<close> \<open>\<forall>U C'. M Midpoint U C' \<and> M Out U A \<longrightarrow> M Out C C'\<close> assms(1) by blast
qed

lemma l9_4_1_aux_R21:
  assumes "R \<noteq> S" and
    "S C Le R A" and
    "P Q TS A C" and
    "Col R P Q" and
    "P Q Perp A R" and
    "Col S P Q" and
    "P Q Perp C S" and
    "M Midpoint R S"
  shows "\<forall> U C'. M Midpoint U C' \<longrightarrow>  (R Out U A \<longleftrightarrow> S Out C C')"
proof -
  obtain D where "Bet R D A" and "Cong S C R D"
    using Le_def assms(2) by blast
  have "C \<noteq> S"
    using assms(7) perp_not_eq_2 by auto
  have "R \<noteq> D"
    using cong_identity \<open>C \<noteq> S\<close> \<open>Cong S C R D\<close> by blast
  have "R S Perp A R"
    using assms(1) assms(4) assms(5) assms(6) not_col_permutation_2 perp_col2 by blast
  have "\<exists> M. (M Midpoint S R \<and> M Midpoint C D)"
  proof -
    have "\<not> Col A P Q"
      using TS_def assms(3) by blast
    have "P \<noteq> Q"
      using not_col_distincts assms(3) ts_distincts by blast
    obtain T where "Col T P Q" and "Bet A T C"
      using TS_def assms(3) by blast
    have "C S Perp S R"
      by (metis NCol_perm assms(1) assms(4) assms(6) assms(7) perp_col0)
    have "A R Perp S R"
      using Perp_perm \<open>R S Perp A R\<close> by blast
    have "Col S R T"
      using Col_cases assms(4) assms(6) col3 \<open>Col T P Q\<close> \<open>P \<noteq> Q\<close> by blast
    have "Bet C T A"
      using Bet_perm \<open>Bet A T C\<close> by blast
    thus ?thesis 
      using l8_24 \<open>A R Perp S R\<close> \<open>Bet R D A\<close> \<open>C S Perp S R\<close> \<open>Col S R T\<close> \<open>Cong S C R D\<close> by blast
  qed
  then obtain M' where "M' Midpoint S R" and "M' Midpoint C D" by blast
  have "M = M'" 
    using assms(8) l7_17_bis \<open>M' Midpoint S R\<close> by blast
  have "\<forall> U C'. (M Midpoint U C' \<and> R Out U A) \<longrightarrow> S Out C C'"
  proof -
    {
      fix U C'
      assume "M Midpoint U C'" and "R Out U A"
      have "C' \<noteq> S"
        using \<open>M Midpoint U C'\<close> \<open>R Out U A\<close> assms(8) l7_9 out_diff1 by blast
      have "Bet S C C' \<or> Bet S C' C"
      proof -
        have "Bet R U A \<or> Bet R A U"
          using Out_def \<open>R Out U A\<close> by auto
        {
          assume "Bet R U A"
          hence "Bet R U D \<or> Bet R D U"
            by (simp add: \<open>Bet R D A\<close> l5_3)
          hence "Bet S C C' \<or> Bet S C' C"
            using l7_15 l7_2 \<open>M = M'\<close> \<open>M Midpoint U C'\<close> \<open>M' Midpoint C D\<close> assms(8) by blast
        }
        hence "Bet R U A \<longrightarrow> Bet S C C' \<or> Bet S C' C" by simp
        have "Bet R A U \<longrightarrow> Bet S C C' \<or> Bet S C' C"
          using l7_15 l7_2 \<open>Bet R D A\<close> \<open>M = M'\<close> \<open>M Midpoint U C'\<close> \<open>M' Midpoint C D\<close> 
            assms(8) between_exchange4 by blast
        thus ?thesis 
          using \<open>Bet R U A \<Longrightarrow> Bet S C C' \<or> Bet S C' C\<close> \<open>Bet R U A \<or> Bet R A U\<close> by blast
      qed
      hence "S Out C C'" 
        using Out_def \<open>C \<noteq> S\<close> \<open>C' \<noteq> S\<close> by auto
    }
    thus ?thesis
      by blast
  qed
  have "\<forall> U C'. (M Midpoint U C' \<and> S Out C C') \<longrightarrow> R Out U A"
  proof -
    {
      fix U C'
      assume "M Midpoint U C'" and "S Out C C'"
      hence "U \<noteq> R" 
        using l7_9_bis \<open>M = M'\<close> \<open>M' Midpoint S R\<close> out_distinct by blast
      have "A \<noteq> R"
        using assms(5) perp_distinct by auto
      have "Bet S C C' \<or> Bet S C' C"
        using Out_def \<open>S Out C C'\<close> by fastforce
      {
        assume "Bet S C C'"
        have "Bet R D U"
        proof -
          have "M Midpoint S R" 
            by (simp add: \<open>M = M'\<close> \<open>M' Midpoint S R\<close>)
          moreover have "M Midpoint C D" 
            by (simp add: \<open>M = M'\<close> \<open>M' Midpoint C D\<close>)
          moreover have "M Midpoint C' U" 
            by (simp add: \<open>M Midpoint U C'\<close> l7_2)
          ultimately show ?thesis 
            by (simp add: \<open>Bet S C C'\<close> l7_15)
        qed
        hence "Bet R U A \<or> Bet R A U" 
          using \<open>Bet R D A\<close> \<open>R \<noteq> D\<close> l5_1 by auto
      }
      hence "Bet S C C' \<longrightarrow> Bet R U A \<or> Bet R A U" by simp
      {
        assume "Bet S C' C"
        have "Bet R U A" 
          using l7_15 l7_2 between_exchange4 \<open>Bet R D A\<close> \<open>M = M'\<close> \<open>M Midpoint U C'\<close> 
          by (meson \<open>Bet S C' C\<close> \<open>M' Midpoint C D\<close> \<open>M' Midpoint S R\<close>)
      }
      hence "Bet S C' C \<longrightarrow> Bet R U A \<or> Bet R A U" by simp
      hence "Bet R U A \<or> Bet R A U" 
        using \<open>Bet S C C' \<Longrightarrow> Bet R U A \<or> Bet R A U\<close> \<open>Bet S C C' \<or> Bet S C' C\<close> by blast
      hence "R Out U A" 
        by (simp add: Out_def \<open>A \<noteq> R\<close> \<open>U \<noteq> R\<close>)
    }
    thus ?thesis by blast
  qed
  thus ?thesis 
    using \<open>\<forall>U C'. M Midpoint U C' \<and> R Out U A \<longrightarrow> S Out C C'\<close> by blast
qed

lemma l9_4_1_aux:
  assumes "S C Le R A" and
    "P Q TS A C" and
    "Col R P Q" and
    "P Q Perp A R" and
    "Col S P Q" and
    "P Q Perp C S" and
    "M Midpoint R S"
  shows "\<forall> U C'. (M Midpoint U C' \<longrightarrow> (R Out U A \<longleftrightarrow> S Out C C'))"
  using l9_4_1_aux_R1 l9_4_1_aux_R21 assms by fast


lemma per_col_eq:
  assumes "Per A B C" and
    "Col A B C" and
    "B \<noteq> C"
  shows "A = B"
  using assms(1) assms(2) assms(3) l8_9 by blast

lemma l9_4_1:
  assumes "P Q TS A C" and
    "Col R P Q" and
    "P Q Perp A R" and
    "Col S P Q" and
    "P Q Perp C S" and
    "M Midpoint R S"
  shows "\<forall> U C'. M Midpoint U C' \<longrightarrow> (R Out U A \<longleftrightarrow> S Out C C')"
proof -
  have "S C Le R A \<or> R A  Le S C"
    using le_cases by blast
  {
    assume "S C Le R A"
    {
      fix U C'
      assume "M Midpoint U C'"
      hence "(R Out U A \<longleftrightarrow> S Out C C')"
        using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) l9_4_1_aux \<open>S C Le R A\<close> 
        by blast
    }
    hence "\<forall> U C'. M Midpoint U C' \<longrightarrow> (R Out U A \<longleftrightarrow> S Out C C')" by simp
  }
  hence "S C Le R A \<longrightarrow> (\<forall> U C'. M Midpoint U C' \<longrightarrow> (R Out U A \<longleftrightarrow> S Out C C'))" by simp
  moreover
  {
    assume " R A Le S C"
    {
      fix U C'
      assume "M Midpoint U C'"
      hence "(R Out A U \<longleftrightarrow> S Out C' C)"
        using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) l7_2 l9_2 l9_4_1_aux 
        by (meson Out_cases le_cases)
      hence "(R Out U A \<longleftrightarrow> S Out C C')"
        using l6_6 by blast
    }
    hence "\<forall> U C'. M Midpoint U C' \<longrightarrow> (R Out U A \<longleftrightarrow> S Out C C')" by simp
  }
  hence "R A Le S C \<longrightarrow> (\<forall> U C'. M Midpoint U C' \<longrightarrow> (R Out U A \<longleftrightarrow> S Out C C'))" by simp
  thus ?thesis 
    using \<open>S C Le R A \<or> R A Le S C\<close> calculation by presburger
qed

lemma mid_two_sides:
  assumes "M Midpoint A B" and
    "\<not> Col A B X" and
    "M Midpoint X Y"
  shows "A B TS X Y"
proof -
  have "\<not> Col Y A B"
    by (meson Mid_cases mid_preserves_col assms(1) assms(2) assms(3) col_permutation_3)
  moreover have "Bet X M Y"
    using assms(3) midpoint_bet by blast
  ultimately show ?thesis 
    by (metis TS_def not_col_permutation_2 assms(1) assms(2) midpoint_col)
qed

lemma col_preserves_two_sides:
  assumes "C \<noteq> D" and
    "Col A B C" and
    "Col A B D" and
    "A B TS X Y"
  shows "C D TS X Y"
proof -
  have "\<not> Col X A B"
    using TS_def assms(4) by blast
  hence "A \<noteq> B"
    using not_col_distincts by blast
  have "\<not> Col X C D"
    using colx assms(1) assms(2) assms(3)
    by (meson \<open>\<not> Col X A B\<close> not_col_permutation_2)
  have "\<not> Col Y C D"
    by (metis Col_cases TS_def colx assms(1) assms(2) assms(3) assms(4))
  thus ?thesis
  proof -
    obtain PP where "\<not> Col X A B" and "\<not> Col Y A B" and "Col PP A B" and "Bet X PP Y" 
      using TS_def assms(4) by blast
    hence "Col PP C D" 
      using \<open>A \<noteq> B\<close> assms(2) assms(3) col3 col_permutation_1 by blast
    thus ?thesis 
      using TS_def \<open>Bet X PP Y\<close> \<open>\<not> Col X C D\<close> \<open>\<not> Col Y C D\<close> by blast
  qed
qed

lemma out_out_two_sides:
  assumes "A \<noteq> B" and
    "A B TS X Y" and
    "Col I A B" and
    "Col I X Y" and
    "I Out X U" and
    "I Out Y V"
  shows "A B TS U V"
proof -
  have "\<not> Col X A B"
    using TS_def assms(2) by blast
  {
    assume "Col V A B"
    hence "Col Y A B" 
      by (metis out_distinct assms(3) assms(6) col_permutation_2 colx out_col)
    have False 
      using TS_def \<open>Col Y A B\<close> assms(2) by blast
  }
  moreover
  have "\<not> Col U A B" 
    using assms(3) assms(5) col_permutation_2 colx out_col out_distinct by (meson \<open>\<not> Col X A B\<close>)
  moreover
  obtain T where "Col T A B" and "Bet X T Y"
    using TS_def assms(2) by blast
  have "I = T"
  proof -
    have "Col X Y I" 
      using assms(4) Col_cases by blast
    moreover have "Col B A I"
      using assms(3) Col_cases by blast
    moreover have "Col B A T" 
      using Col_cases \<open>Col T A B\<close> by auto
    moreover have "\<not> Col X A B \<and> \<not> Col X B A \<and> \<not> Col A X B \<and> \<not> Col A B X \<and> \<not> Col B X A \<and> \<not> Col B A X"
      using \<open>\<not> Col X A B\<close> Col_cases by blast
    moreover have "A \<noteq> B \<and> A \<noteq> X \<and> A \<noteq> Y \<and> B \<noteq> X \<and> B \<noteq> Y \<and> X \<noteq> Y"
      using assms(2) ts_distincts by presburger
    moreover have "Col X Y T" 
      using \<open>Bet X T Y\<close> bet_col1 between_trivial by blast
    ultimately show ?thesis 
      using l6_21 by blast
  qed
  hence "Bet U T V"
    using assms(5) assms(6) bet_out_out_bet \<open>Bet X T Y\<close> by blast
  ultimately show ?thesis 
    using TS_def \<open>Col T A B\<close> by blast
qed

lemma l9_4_2_aux_R1:
  assumes "R = S " and
    "S C Le R A" and
    "P Q TS A C" and
    "Col R P Q" and
    "P Q Perp A R" and
    "Col S P Q" and
    "P Q Perp C S" and
    "R Out U A" and
    "S Out V C"
  shows "P Q TS U V"
proof -
  have "\<not> Col A P Q"
    using TS_def assms(3) by auto
  hence "P \<noteq> Q"
    using not_col_distincts by blast
  obtain T where "Col T P Q" and "Bet A T C"
    using TS_def assms(3) by blast
  have "R = T" 
    using assms(1) assms(5) assms(6) assms(7) col_permutation_1 l8_16_1 l8_6 
    by (meson \<open>Bet A T C\<close> \<open>Col T P Q\<close>)
  thus ?thesis 
    by (metis \<open>Bet A T C\<close> \<open>Col T P Q\<close> \<open>P \<noteq> Q\<close> assms(1) assms(3) assms(8) assms(9) 
        bet_col col_permutation_4 l6_6 out_out_two_sides)
qed

lemma l9_4_2_aux_R2:
  assumes "R \<noteq> S" and
    "S C Le R A" and
    "P Q TS A C" and
    "Col R P Q" and
    "P Q Perp A R" and
    "Col S P Q" and
    "P Q Perp C S" and
    "R Out U A" and
    "S Out V C"
  shows "P Q TS U V"
proof -
  have "P \<noteq> Q"
    using assms(7) perp_distinct by auto
  have "R S TS A C"
    using assms(1) assms(3) assms(4) assms(6) col_permutation_1 col_preserves_two_sides by blast
  have "Col R S P"
    using assms(4) assms(6) col2__eq not_col_permutation_1 \<open>P \<noteq> Q\<close> by blast
  have "Col R S Q"
    using colx Tarski_neutral_dimensionless_axioms assms(4) assms(6) col_trivial_2 
    by (metis \<open>Col R S P\<close>)
  have "R S Perp A R"
    using NCol_perm assms(1) assms(4) assms(5) assms(6) perp_col2 by blast
  have "R S Perp C S"
    using assms(1) assms(4) assms(6) assms(7) col_permutation_1 perp_col2 by blast
  have "\<not> Col A R S" 
    using TS_def \<open>R S TS A C\<close> by force
  obtain T where "Col T R S" and "Bet A T C" 
    using TS_def \<open>R S TS A C\<close> by blast
  obtain C' where "Bet R C' A" and "Cong S C R C'" 
    using Le_def assms(2) by blast
  have "\<exists> X. X Midpoint S R \<and> X Midpoint C C'"
  proof -
    have "C S Perp S R" 
      using Perp_perm \<open>R S Perp C S\<close> by blast
    moreover have "A R Perp S R" 
      using Perp_perm \<open>R S Perp A R\<close> by blast
    moreover have "Col S R T" 
      using Col_cases \<open>Col T R S\<close> by auto
    moreover have "Bet C T A" 
      using Bet_cases \<open>Bet A T C\<close> by blast
    ultimately show ?thesis 
      using l8_24 \<open>Bet R C' A\<close> \<open>Cong S C R C'\<close> by blast
  qed
  then obtain M where "M Midpoint S R" and "M Midpoint C C'" 
    by blast
  obtain U' where "M Midpoint U U'"
    using symmetric_point_construction by blast
  have "R \<noteq> U"
    using assms(8) out_diff1 by blast
  have "R S TS U U'" 
    by (metis Col_def Out_def invert_two_sides l6_16_1 mid_two_sides \<open>M Midpoint S R\<close> 
        \<open>M Midpoint U U'\<close> \<open>\<not> Col A R S\<close> assms(8))
  have "R S TS V U"
  proof -
    have "Col M R S" 
      using Col_def Midpoint_def \<open>M Midpoint S R\<close> by force
    moreover have "M Midpoint U' U" 
      by (simp add: \<open>M Midpoint U U'\<close> l7_2)
    moreover have "S Out U' V" 
      by (meson l6_7 \<open>M Midpoint S R\<close> \<open>M Midpoint U U'\<close> assms(2) assms(3) assms(4) 
          assms(5) assms(6) assms(7) assms(8) assms(9) l6_6 l7_2 l9_4_1_aux)
    ultimately show ?thesis 
      using \<open>R S TS U U'\<close> col_trivial_3 l9_2 l9_3 by blast
  qed
  thus ?thesis 
    using \<open>Col R S P\<close> \<open>Col R S Q\<close> \<open>P \<noteq> Q\<close> col_preserves_two_sides l9_2 by presburger
qed

lemma l9_4_2_aux:
  assumes "S C Le R A" and
    "P Q TS A C" and
    "Col R P Q" and
    "P Q Perp A R" and
    "Col S P Q" and
    "P Q Perp C S" and
    "R Out U A" and
    "S Out V C"
  shows "P Q TS U V"
  using l9_4_2_aux_R1 l9_4_2_aux_R2 
  by (metis assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) assms(7) assms(8))

lemma l9_4_2:
  assumes "P Q TS A C" and
    "Col R P Q" and
    "P Q Perp A R" and
    "Col S P Q" and
    "P Q Perp C S" and
    "R Out U A" and
    "S Out V C"
  shows "P Q TS U V"
proof -
  have "S C Le R A \<or> R A Le S C"
    by (simp add: local.le_cases)
  moreover have "S C Le R A \<longrightarrow> P Q TS U V"
    by (simp add: assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) assms(7) l9_4_2_aux)
  moreover have "R A Le S C \<longrightarrow> P Q TS U V"
    by (simp add: assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) assms(7) l9_2 l9_4_2_aux)
  ultimately show ?thesis
    by blast
qed

lemma l9_5:
  assumes "P Q TS A C" and
    "Col R P Q" and
    "R Out A B"
  shows "P Q TS B C"
proof -
  have "P \<noteq> Q"
    using assms(1) ts_distincts by blast
  obtain A' where "Col P Q A'" and "P Q Perp A A'"
    by (metis NCol_perm TS_def assms(1) l8_18_existence)
  obtain C' where "Col P Q C'" and "P Q Perp C C'"
    using Col_perm TS_def assms(1) l8_18_existence by blast
  obtain M where "M Midpoint A' C'"
    using midpoint_existence by blast
  obtain D where "M Midpoint A D"
    using symmetric_point_construction by auto
  have "\<exists> B0. Col P Q B0 \<and> P Q Perp B B0"
  proof -
    have "\<not> Col P Q B"
      using colx perp_not_col2 Tarski_neutral_dimensionless_axioms assms(2) assms(3) 
        col_permutation_1 l6_3_1 out_col by (meson \<open>Col P Q A'\<close> \<open>P Q Perp A A'\<close>)
    thus ?thesis
      by (simp add: l8_18_existence)
  qed
  then obtain B' where "Col P Q B'" and "P Q Perp B B'" by blast
  have "P Q TS B C"
  proof -
    have "C' Out D C \<longleftrightarrow> A' Out A A"
      using Out_cases assms(1) l9_4_1 not_col_permutation_1 \<open>Col P Q A'\<close> \<open>Col P Q C'\<close> 
        \<open>M Midpoint A D\<close> \<open>M Midpoint A' C'\<close> \<open>P Q Perp A A'\<close> \<open>P Q Perp C C'\<close> by blast
    hence "C' Out D C"
      using perp_not_eq_2 Tarski_neutral_dimensionless_axioms out_trivial \<open>P Q Perp A A'\<close> by blast
    have "P Q TS A D"
      using assms(1) col_permutation_2 l9_4_2 
      by (meson \<open>(C' Out D C) = (A' Out A A)\<close> \<open>C' Out D C\<close> \<open>Col P Q A'\<close> \<open>Col P Q C'\<close> 
          \<open>P Q Perp A A'\<close> \<open>P Q Perp C C'\<close>)
    {
      assume "A' \<noteq> C'"
      hence "Col M P Q"
        using col_trivial_2 l6_21 midpoint_col not_col_permutation_1 
          \<open>Col P Q A'\<close> \<open>Col P Q C'\<close> \<open>M Midpoint A' C'\<close> by fast
      hence "P Q TS B D"
        using assms(2) assms(3) l9_3 \<open>M Midpoint A D\<close> \<open>P Q TS A D\<close> by blast
    }
    hence "A' \<noteq> C' \<longrightarrow> P Q TS B D" by simp
    hence "P Q TS B D"
      using assms(2) assms(3) l9_3 midpoint_distinct_2 not_col_permutation_1 
      by (metis \<open>Col P Q A'\<close> \<open>M Midpoint A D\<close> \<open>M Midpoint A' C'\<close> \<open>P Q TS A D\<close>)
    moreover have "Col B' P Q" 
      using \<open>Col P Q B'\<close> not_col_permutation_1 by blast
    moreover have "Col C' P Q" 
      using \<open>Col P Q C'\<close> col_permutation_2 by blast
    moreover have "P Q Perp D C'"
      using Col_perm l6_3_1 out_col perp_col1 perp_right_comm 
      by (metis \<open>C' Out D C\<close> \<open>P Q Perp C C'\<close>)
    moreover have "B' Out B B" 
      using \<open>P Q Perp B B'\<close> out_trivial perp_not_eq_2 by auto
    moreover have "C' Out C D" 
      using Out_cases \<open>C' Out D C\<close> by auto
    ultimately show ?thesis 
      using l9_4_2 \<open>P Q Perp B B'\<close> by blast
  qed
  thus ?thesis using l8_18_existence by blast
qed

lemma outer_pasch_R1:
  assumes "Col P Q C" and
    "Bet A C P" and
    "Bet B Q C"
  shows "\<exists> X. Bet A X B \<and> Bet P Q X"
proof cases
  assume "Bet P Q C"
  thus ?thesis 
    using Bet_cases assms(2) between_exchange4 between_trivial2 by blast
next
  assume "\<not> Bet P Q C"
  thus ?thesis 
    by (metis Bet_cases assms(1) assms(3) between_exchange3 between_trivial 
        outer_transitivity_between2 third_point)
qed

lemma outer_pasch_R2:
  assumes "\<not> Col P Q C" and
    "Bet A C P" and
    "Bet B Q C"
  shows "\<exists> X. Bet A X B \<and> Bet P Q X"
proof cases
  assume "B = Q"
  thus ?thesis
    using between_trivial by blast
next
  assume "B \<noteq> Q"
  have "A \<noteq> P"
    using assms(1) assms(2) between_identity col_trivial_3 by blast
  have "P \<noteq> Q"
    using assms(1) col_trivial_1 by blast
  have "P \<noteq> B"
    using assms(1) assms(3) bet_col by blast
  have "P Q TS C B"
  proof -
    have "\<not> Col C P Q"
      using Col_cases assms(1) by blast
    moreover have "\<not> Col B P Q"
      using Col_cases colx Tarski_neutral_dimensionless_axioms assms(1) assms(3) bet_col 
        col_trivial_2 \<open>B \<noteq> Q\<close> by fast
    moreover have "\<exists> T. Col T P Q \<and> Bet C T B"
      using Col_cases assms(3) between_symmetry col_trivial_2 by blast
    ultimately show ?thesis 
      using TS_def by blast
  qed
  have "P Q TS A B"
    using assms(1) assms(2) bet_out_1 l9_5 not_col_distincts \<open>P Q TS C B\<close> by metis
  obtain X where "Col X P Q" and "Bet A X B" 
    using TS_def \<open>P Q TS A B\<close> by blast
  have "Bet P Q X"
  proof -
    obtain T where "Bet X T P" and "Bet C T B"
      using assms(2) between_symmetry inner_pasch by (meson \<open>Bet A X B\<close>)
    have "B \<noteq> C"
      using assms(3) bet_neq12__neq \<open>B \<noteq> Q\<close> by auto
    have "T = Q"
    proof -
      have "Col Q C B"
        using NCol_cases assms(3) Col_def by blast
      have "Col T C B" 
        using \<open>Bet C T B\<close> bet_col col_permutation_4 by presburger
      have "\<not> Col X P B" 
        by (metis Bet_cases TS_def \<open>Bet C T B\<close> \<open>Bet X T P\<close> \<open>Col X P Q\<close> \<open>P Q TS C B\<close> 
            assms(3) bet_col1 between_equality_2 between_trivial2 col_transitivity_2)
      thus ?thesis
        by (metis Col_def \<open>B \<noteq> C\<close> \<open>Bet C T B\<close> \<open>Bet X T P\<close> \<open>Col Q C B\<close> \<open>Col X P Q\<close> 
            between_symmetry l6_21)
    qed
    thus ?thesis 
      using \<open>Bet X T P\<close> between_symmetry by blast
  qed
  thus ?thesis 
    using \<open>Bet A X B\<close> by blast
qed

(** GeoCoq: Ch09_planve.v **)
(** This lemma used to be an axiom in previous versions of Tarski's axiom system.
It is a been shown to a theorem by Gupta in his Phd 1965. *)
(** This corresponds to l9_6 in Tarski's book. *)
lemma outer_pasch:
  assumes "Bet A C P" and
    "Bet B Q C"
  shows "\<exists> X. Bet A X B \<and> Bet P Q X"
  using assms(1) assms(2) outer_pasch_R1 outer_pasch_R2 by blast

lemma os_distincts:
  assumes "A B OS X Y"
  shows "A \<noteq> B \<and> A \<noteq> X \<and> A \<noteq> Y \<and> B \<noteq> X \<and> B \<noteq> Y"
  using OS_def assms ts_distincts by blast

lemma invert_one_side:
  assumes "A B OS P Q"
  shows "B A OS P Q"
proof -
  obtain T where "A B TS P T \<and> A B TS Q T"
    using OS_def assms by blast
  hence "B A TS P T \<and> B A TS Q T"
    using invert_two_sides by blast
  thus ?thesis
    using OS_def by blast
qed

lemma l9_8_1:
  assumes "P Q TS A C" and
    "P Q TS B C"
  shows "P Q OS A B"
proof -
  have "\<exists> R::'p. (P Q TS A R \<and> P Q TS B R)"
    using assms(1) assms(2) by blast
  thus ?thesis
    using OS_def by blast
qed

lemma not_two_sides_id:
  shows "\<not> P Q TS A A"
  using ts_distincts by blast

lemma l9_8_2:
  assumes "P Q TS A C" and
    "P Q OS A B"
  shows "P Q TS B C"
proof -
  obtain D where P1: "P Q TS A D \<and> P Q TS B D"
    using assms(2) OS_def by blast
  hence "P \<noteq> Q"
    using ts_distincts by blast
  obtain T where P2: "Col T P Q \<and> Bet A T C"
    using TS_def assms(1) by blast
  obtain X where P3: "Col X P Q \<and> Bet A X D"
    using TS_def P1 by blast
  obtain Y where P4: "Col Y P Q \<and> Bet B Y D"
    using TS_def P1 by blast
  then obtain M where P5: "Bet Y M A \<and> Bet X M B" using P3 inner_pasch by blast
  have P6: "A \<noteq> D"
    using P1 ts_distincts by blast
  have P7: "B \<noteq> D"
    using P1 not_two_sides_id by blast
  {
    assume Q0: "Col A B D"
    have "P Q TS B C"
    proof cases
      assume Q1: "M = Y"
      have "X = Y"
      proof -
        have S1: "\<not> Col P Q A"
          using TS_def assms(1) not_col_permutation_1 by blast
        have S3: "Col P Q X"
          using Col_perm P3 by blast
        have S4: "Col P Q Y"
          using Col_perm P4 by blast
        have S5: "Col A D X"
          by (simp add: P3 bet_col col_permutation_5)
        have "Col A D Y"
          by (metis Col_def P5 Q1 S5 Q0 between_equality between_trivial l6_16_1)
        thus ?thesis using S1 S3 S4 S5 P6 l6_21
          by blast
      qed
      hence "X Out A B"
        by (metis P1 P3 P4 TS_def l6_2)
      thus ?thesis using assms(1) P3 l9_5 by blast
    next
      assume Z1: "\<not> M = Y"
      have "X = Y"
      proof -
        have S1: "\<not> Col P Q A"
          using TS_def assms(1) not_col_permutation_1 by blast
        have S3: "Col P Q X"
          using Col_perm P3 by blast
        have S4: "Col P Q Y"
          using Col_perm P4 by blast
        have S5: "Col A D X"
          by (simp add: P3 bet_col col_permutation_5)
        have "Col A D Y"
          by (metis Col_def P4 Q0 P7 l6_16_1)
        thus ?thesis using S1 S3 S4 S5 P6 l6_21
          by blast
      qed
      hence Z3: "M \<noteq> X" using Z1 by blast
      have Z4: "P Q TS M C"
        by (meson Out_cases P4 P5 l9_5 Tarski_neutral_dimensionless_axioms Z1 assms(1) bet_out)
      have "X Out M B"
        using P5 Z3 bet_out by auto
      thus ?thesis using Z4 P3 l9_5 by blast
    qed
  }
  hence Z99: "Col A B D \<longrightarrow> P Q TS B C" by blast
  {
    assume Q0: "\<not> Col A B D"
    have Q1: "P Q TS M C"
    proof -
      have S3: "Y Out A M"
      proof -
        have T1: "A \<noteq> Y"
          using Col_def P4 Q0 col_permutation_4 by blast
        have T2: "M \<noteq> Y"
        proof -
          {
            assume T3: "M = Y"
            have "Col B D X"
            proof -
              have U1: "B \<noteq> M"
                using P1 P4 T3 TS_def by blast
              have U2: "Col B M D"
                by (simp add: P4 T3 bet_col)
              have "Col B M X"
                by (simp add: P5 bet_col between_symmetry)
              thus ?thesis using U1 U2
                using col_transitivity_1 by blast
            qed
            have "False"
              by (metis NCol_cases P1 P3 TS_def \<open>Col B D X\<close> Q0 bet_col col_trivial_2 l6_21)
          }
          thus ?thesis by blast
        qed
        have "Bet Y A M \<or> Bet Y M A" using P5 by blast
        thus ?thesis using T1 T2
          by (simp add: Out_def)
      qed
      hence "X Out M B"
        by (metis P1 P3 P4 P5 TS_def bet_out l9_5)
      thus ?thesis using assms(1) S3 l9_5 P3 P4 by blast
    qed
    have "X Out M B"
      by (metis P3 P5 Q1 TS_def bet_out)
    hence "P Q TS B C" using Q1 P3 l9_5 by blast
  }
  hence "\<not> Col A B D \<longrightarrow> P Q TS B C" by blast
  thus ?thesis using Z99 by blast
qed

lemma l9_9:
  assumes "P Q TS A B"
  shows "\<not> P Q OS A B"
  using assms l9_8_2 not_two_sides_id by blast

lemma l9_9_bis:
  assumes "P Q OS A B"
  shows "\<not> P Q TS A B"
  using assms l9_9 by blast

lemma one_side_chara:
  assumes "P Q OS A B"
  shows "\<forall> X. Col X P Q \<longrightarrow> \<not> Bet A X B"
proof -
  have "\<not> Col A P Q \<and> \<not> Col B P Q"
    using OS_def TS_def assms by auto
  thus ?thesis
    using l9_9_bis TS_def assms by blast
qed

lemma l9_10:
  assumes "\<not> Col A P Q"
  shows "\<exists> C. P Q TS A C"
  by (meson Col_perm assms mid_two_sides midpoint_existence symmetric_point_construction)

lemma one_side_reflexivity:
  assumes "\<not> Col A P Q"
  shows "P Q OS A A"
  using assms l9_10 l9_8_1 by blast

lemma one_side_symmetry:
  assumes "P Q OS A B"
  shows "P Q OS B A"
  by (meson OS_def Tarski_neutral_dimensionless_axioms assms invert_two_sides)

lemma one_side_transitivity:
  assumes "P Q OS A B" and
    "P Q OS B C"
  shows "P Q OS A C"
  by (meson OS_def l9_8_2 Tarski_neutral_dimensionless_axioms assms(1) assms(2))

lemma l9_17:
  assumes "P Q OS A C" and
    "Bet A B C"
  shows "P Q OS A B"
proof cases
  assume "A = C"
  thus ?thesis
    using assms(1) assms(2) between_identity by blast
next
  assume P1: "\<not> A = C"
  obtain D where P2: "P Q TS A D \<and> P Q TS C D"
    using OS_def assms(1) by blast
  hence P3: "P \<noteq> Q"
    using ts_distincts by blast
  obtain X where P4: "Col X P Q \<and> Bet A X D"
    using P2 TS_def by blast
  obtain Y where P5: "Col Y P Q \<and> Bet C Y D"
    using P2 TS_def by blast
  obtain T where P6: "Bet B T D \<and> Bet X T Y"
    using P4 P5 assms(2) l3_17 by blast
  have P7: "P Q TS A D"
    by (simp add: P2)
  have "P Q TS B D"
  proof -
    have Q1: "\<not> Col B P Q"
      using assms(1) assms(2) one_side_chara by blast
    have Q2: "\<not> Col D P Q"
      using P2 TS_def by blast
    obtain T0 where "Col T0 P Q \<and> Bet B T0 D"
    proof -
      assume a1: "\<And>T0. Col T0 P Q \<and> Bet B T0 D \<Longrightarrow> thesis"
      obtain pp :: 'p where
        f2: "Bet B pp D \<and> Bet X pp Y"
        using \<open>\<And>thesis. (\<And>T. Bet B T D \<and> Bet X T Y \<Longrightarrow> thesis) \<Longrightarrow> thesis\<close> by blast
      have "Col P Q Y"
        using Col_def P5 by blast
      hence "Y = X \<or> Col P Q pp"
        using f2 Col_def P4 colx by blast
      thus ?thesis
        using f2 a1 by (metis BetSEq BetS_def Col_def P4)
    qed
    thus ?thesis 
      using Q1 Q2 TS_def by blast
  qed
  thus ?thesis using P7
    using OS_def by blast
qed

lemma l9_18_R1:
  assumes "Col X Y P" and
    "Col A B P"
    and "X Y TS A B"
  shows "Bet A P B \<and> \<not> Col X Y A \<and> \<not> Col X Y B"
  by (meson TS_def assms(1) assms(2) assms(3) col_permutation_5 l9_5 not_col_permutation_1 
      not_out_bet not_two_sides_id)

lemma l9_18_R2:
  assumes "Col X Y P" and
    "Col A B P" and
    "Bet A P B" and
    "\<not> Col X Y A" and
    "\<not> Col X Y B"
  shows "X Y TS A B"
  using Col_perm TS_def assms(1) assms(3) assms(4) assms(5) by blast

lemma l9_18:
  assumes "Col X Y P" and
    "Col A B P"
  shows "X Y TS A B \<longleftrightarrow> (Bet A P B \<and> \<not> Col X Y A \<and> \<not> Col X Y B)"
  using l9_18_R1 l9_18_R2 assms(1) assms(2) by blast

lemma l9_19_R1:
  assumes "Col X Y P" and
    "Col A B P" and
    "X Y OS A B"
  shows "P Out A B \<and> \<not> Col X Y A"
  by (meson OS_def TS_def assms(1) assms(2) assms(3) col_permutation_5 not_col_permutation_1 
      not_out_bet one_side_chara)

lemma l9_19_R2:
  assumes "Col X Y P" and
    (*    "Col A B P" and *)
    "P Out A B" and
    "\<not> Col X Y A"
  shows "X Y OS A B"
proof -
  obtain D where "X Y TS A D"
    using Col_perm assms(3) l9_10 by blast
  thus ?thesis
    using OS_def assms(1) assms(2) l9_5 not_col_permutation_1 by blast
qed

lemma l9_19:
  assumes "Col X Y P" and
    "Col A B P"
  shows "X Y OS A B \<longleftrightarrow> (P Out A B \<and> \<not> Col X Y A)"
  using l9_19_R1 l9_19_R2 assms(1) assms(2) by blast

lemma one_side_not_col123:
  assumes "A B OS X Y"
  shows "\<not> Col A B X"
  using assms col_trivial_3 l9_19 by blast

lemma one_side_not_col124:
  assumes "A B OS X Y"
  shows "\<not> Col A B Y"
  using assms one_side_not_col123 one_side_symmetry by blast

lemma col_two_sides:
  assumes "Col A B C" and
    "A \<noteq> C" and
    "A B TS P Q"
  shows "A C TS P Q"
  using assms(1) assms(2) assms(3) col_preserves_two_sides col_trivial_3 by blast

lemma col_one_side:
  assumes "Col A B C" and
    "A \<noteq> C" and
    "A B OS P Q"
  shows "A C OS P Q"
proof -
  obtain T where "A B TS P T \<and> A B TS Q T" using assms(1) assms(2) assms(3) OS_def by blast
  thus ?thesis
    using col_two_sides OS_def assms(1) assms(2) by blast
qed

lemma out_out_one_side:
  assumes "A B OS X Y" and
    "A Out Y Z"
  shows "A B OS X Z"
  by (meson Col_cases OS_def Tarski_neutral_dimensionless_axioms assms(1) assms(2) 
      col_trivial_3 l9_5)

lemma out_one_side:
  assumes "\<not> Col A B X \<or> \<not> Col A B Y" and
    "A Out X Y"
  shows "A B OS X Y"
  using assms(1) assms(2) l6_6 not_col_permutation_2 one_side_reflexivity one_side_symmetry 
    out_out_one_side by blast

lemma bet__ts:
  assumes "A \<noteq> Y" and
    "\<not> Col A B X" and
    "Bet X A Y"
  shows "A B TS X Y"
proof -
  have "\<not> Col Y A B"
    using NCol_cases assms(1) assms(2) assms(3) bet_col col2__eq by blast
  thus ?thesis
    by (meson TS_def assms(2) assms(3) col_permutation_3 col_permutation_5 col_trivial_3)
qed

lemma bet_ts__ts:
  assumes "A B TS X Y" and
    "Bet X Y Z"
  shows "A B TS X Z"
proof -
  have "\<not> Col Z A B"
    using assms(1) assms(2) bet_col between_equality_2 col_permutation_1 l9_18 by blast
  thus ?thesis
    using TS_def assms(1) assms(2) between_exchange4 by blast
qed

lemma bet_ts__os:
  assumes "A B TS X Y" and
    "Bet X Y Z"
  shows "A B OS Y Z"
  using OS_def assms(1) assms(2) bet_ts__ts l9_2 by blast

lemma l9_31 :
  assumes "A X OS Y Z" and
    "A Z OS Y X"
  shows "A Y TS X Z"
proof -
  have "A \<noteq> X \<and> A \<noteq> Z \<and> \<not> Col Y A X \<and> \<not> Col Z A X \<and> \<not> Col Y A Z"
    using assms(1) assms(2) col_permutation_1 one_side_not_col123 one_side_not_col124 
      os_distincts by blast
  obtain Z' where "Bet Z A Z'" and "Cong A Z' Z A" 
    using segment_construction by blast
  have "Z' \<noteq> A" 
    by (metis \<open>Cong A Z' Z A\<close> assms(1) cong_diff_3 os_distincts)
  have "A X TS Y Z'" 
    by (metis l9_8_2 one_side_symmetry \<open>Bet Z A Z'\<close> \<open>Z' \<noteq> A\<close> assms(1) bet__ts 
        one_side_not_col124)
  have "\<not> Col Y A X" 
    using \<open>A \<noteq> X \<and> A \<noteq> Z \<and> \<not> Col Y A X \<and> \<not> Col Z A X \<and> \<not> Col Y A Z\<close> by auto
  obtain T where "Col A T X" and "Bet Y T Z'" 
    by (meson TS_def col_permutation_4 \<open>A X TS Y Z'\<close>)
  hence "T \<noteq> A"
  proof -
    have "\<not> A Out Z Y" 
      using Col_cases \<open>A \<noteq> X \<and> A \<noteq> Z \<and> \<not> Col Y A X \<and> \<not> Col Z A X \<and> \<not> Col Y A Z\<close> out_col by blast
    have "A \<noteq> Z'" 
      using \<open>Z' \<noteq> A\<close> by auto
    thus ?thesis 
      by (metis \<open>A \<noteq> X \<and> A \<noteq> Z \<and> \<not> Col Y A X \<and> \<not> Col Z A X \<and> \<not> Col Y A Z\<close> \<open>Bet Y T Z'\<close> 
          \<open>Bet Z A Z'\<close> bet_col l6_21 not_col_distincts)
  qed
  have "Y A OS Z' T" 
    by (metis NCol_cases Out_def col3 col_trivial_3 out_one_side \<open>A X TS Y Z'\<close> \<open>Bet Y T Z'\<close> 
        \<open>Col A T X\<close> \<open>T \<noteq> A\<close> \<open>\<not> Col Y A X\<close> ts_distincts)
  have "A Y TS Z' Z" 
    by (metis Bet_cases Col_cases \<open>A \<noteq> X \<and> A \<noteq> Z \<and> \<not> Col Y A X \<and> \<not> Col Z A X \<and> \<not> Col Y A Z\<close> 
        \<open>Bet Z A Z'\<close> \<open>Y A OS Z' T\<close> bet__ts one_side_not_col123)
  {
    assume "Bet T A X"
    have "Z' Z OS Y T" 
      using BetSEq BetS_def TS_def l6_6 bet_col bet_out_1 col_trivial_3 
        colx not_col_permutation_3 not_col_permutation_4 out_one_side 
        \<open>A Y TS Z' Z\<close> \<open>A \<noteq> X \<and> A \<noteq> Z \<and> \<not> Col Y A X \<and> \<not> Col Z A X \<and> \<not> Col Y A Z\<close> 
        \<open>Bet Y T Z'\<close> \<open>Bet Z A Z'\<close> \<open>Col A T X\<close> by metis
    hence "Z' Out T Y" 
      by (metis \<open>Bet Y T Z'\<close> bet_out_1 os_distincts)
    hence "A Z OS Y T" 
      using \<open>A \<noteq> X \<and> A \<noteq> Z \<and> \<not> Col Y A X \<and> \<not> Col Z A X \<and> \<not> Col Y A Z\<close> \<open>Bet Z A Z'\<close> 
        bet_col invert_one_side l6_6 l9_19_R2 not_col_permutation_3 by blast
    have "A Z TS X T"
    proof -
      have "\<not> Col X A Z" 
        using Col_cases \<open>A \<noteq> X \<and> A \<noteq> Z \<and> \<not> Col Y A X \<and> \<not> Col Z A X \<and> \<not> Col Y A Z\<close> by blast
      have "\<not> Col T A Z" 
        using \<open>A Z OS Y T\<close> col_permutation_1 one_side_not_col124 by blast
      have "\<exists> T0. Col T0 A Z \<and> Bet X T0 T"
      proof -
        have "Col A A Z" 
          by (simp add: col_trivial_1)
        moreover have "Bet X A T" 
          using Bet_cases \<open>Bet T A X\<close> by blast
        ultimately show ?thesis 
          by auto
      qed
      thus ?thesis 
        using TS_def \<open>\<not> Col T A Z\<close> \<open>\<not> Col X A Z\<close> by presburger
    qed
    have "A Y TS X Z" 
      by (meson l9_8_2 \<open>A Z OS Y T\<close> \<open>A Z TS X T\<close> assms(2) l9_9 one_side_symmetry)
  }
  moreover
  {
    assume "Bet A X T"
    hence "A Y OS Z' X" 
      using Bet_cases between_equality invert_one_side not_col_permutation_4 
        not_out_bet out_out_one_side 
      by (meson OS_def \<open>A Y TS Z' Z\<close> \<open>Col A T X\<close> \<open>Y A OS Z' T\<close> calculation)
    have "A Y TS X Z" 
      using \<open>A Y OS Z' X\<close> \<open>A Y TS Z' Z\<close> l9_8_2 by blast
  }
  moreover
  {
    assume "Bet X T A"
    hence "A Y OS T X" 
      using \<open>A \<noteq> X \<and> A \<noteq> Z \<and> \<not> Col Y A X \<and> \<not> Col Z A X \<and> \<not> Col Y A Z\<close> \<open>T \<noteq> A\<close> 
        bet_out_1 not_col_permutation_4 out_one_side by presburger
    hence "A Y TS X Z" 
      using \<open>A Y TS Z' Z\<close> \<open>Y A OS Z' T\<close> invert_two_sides l9_8_2 by blast
  }
  ultimately show ?thesis 
    using Bet_cases \<open>Col A T X\<close> third_point by blast
qed

lemma col123__nos:
  assumes "Col P Q A"
  shows "\<not> P Q OS A B"
  using assms one_side_not_col123 by blast

lemma col124__nos:
  assumes "Col P Q B"
  shows "\<not> P Q OS A B"
  using assms one_side_not_col124 by blast

lemma col2_os__os:
  assumes "C \<noteq> D" and
    "Col A B C" and
    "Col A B D" and
    "A B OS X Y"
  shows "C D OS X Y"
  by (metis assms(1) assms(2) assms(3) assms(4) col3 col_one_side col_trivial_3 
      invert_one_side os_distincts)

lemma os_out_os:
  assumes "Col A B P" and
    "A B OS C D" and
    "P Out C C'"
  shows "A B OS C' D"
  using OS_def assms(1) assms(2) assms(3) l9_5 not_col_permutation_1 by blast

lemma ts_ts_os:
  assumes "A B TS C D" and
    "C D TS A B"
  shows "A C OS B D"
proof -
  obtain T1 where P1: "Col T1 A B \<and> Bet C T1 D"
    using TS_def assms(1) by blast
  obtain T where P2: "Col T C D \<and> Bet A T B"
    using TS_def assms(2) by blast
  have P3: "T1 = T"
  proof -
    have "A \<noteq> B"
      using assms(2) ts_distincts by blast
    thus ?thesis
    proof -
      have "Col T1 D C"
        using Col_def P1 by blast
      hence f1: "\<forall>p. (C = T1 \<or> Col C p T1) \<or> \<not> Col C T1 p"
        by (metis assms(1) col_transitivity_1 l6_16_1 ts_distincts)
      have f2: "\<not> Col C A B"
        using TS_def assms(1) by presburger
      have f3: "(Bet B T1 A \<or> Bet T1 A B) \<or> Bet A B T1"
        using Col_def P1 by blast
      {
        assume "T1 \<noteq> B"
        hence "C \<noteq> T1 \<and> \<not> Col C T1 B \<or> (\<exists>p. \<not> Col p T1 B \<and> Col p T1 T) \<or> T \<noteq> A \<and> T \<noteq> B"
          using f3 f2 by (metis (no_types) Col_def col_transitivity_1 l6_16_1)
        hence "T \<noteq> A \<and> T \<noteq> B \<or> C \<noteq> T1 \<and> \<not> Col C T1 T \<or> T1 = T"
          using f3 by (meson Col_def l6_16_1)
      }
      moreover
      {
        assume "T \<noteq> A \<and> T \<noteq> B"
        hence "C \<noteq> T1 \<and> \<not> Col C T1 T \<or> T1 = T"
          using f2 by (metis (no_types) Col_def P1 P2 \<open>A \<noteq> B\<close> col_transitivity_1 l6_16_1)
      }
      ultimately have "C \<noteq> T1 \<and> \<not> Col C T1 T \<or> T1 = T"
        using f2 f1 assms(1) ts_distincts by blast
      thus ?thesis
        by (metis (no_types) Col_def P1 P2 assms(1) l6_16_1 ts_distincts)
    qed
  qed
  have P4: "A C OS T B"
    by (metis Col_cases P2 TS_def assms(1) assms(2) bet_out out_one_side)
  hence "C A OS T D"
    by (metis Col_cases P1 TS_def P3 assms(2) bet_out os_distincts out_one_side)
  thus ?thesis
    by (meson P4 invert_one_side one_side_symmetry Tarski_neutral_dimensionless_axioms 
        one_side_transitivity)
qed

lemma col_one_side_out:
  assumes "Col A X Y" and
    "A B OS X Y"
  shows "A Out X Y"
  by (meson assms(1) assms(2) l6_4_2 not_col_distincts not_col_permutation_4 one_side_chara)

lemma col_two_sides_bet:
  assumes "Col A X Y" and
    "A B TS X Y"
  shows "Bet X A Y"
  using Col_cases assms(1) assms(2) l9_8_1 l9_9 or_bet_out out_out_one_side by blast

lemma os_ts1324__os:
  assumes "A X OS Y Z" and
    "A Y TS X Z"
  shows "A Z OS X Y"
proof -
  obtain P where P1: "Col P A Y \<and> Bet X P Z"
    using TS_def assms(2) by blast
  have P2: "A Z OS X P"
    by (metis Col_cases P1 TS_def assms(1) assms(2) bet_col bet_out_1 col124__nos 
        col_trivial_2 l6_6 l9_19)
  have "A Z OS P Y"
  proof -
    have "\<not> Col A Z P \<or> \<not> Col A Z Y"
      using P2 col124__nos by blast
    moreover have "A Out P Y"
    proof -
      have "X A OS P Z"
        by (metis Col_cases P1 P2 assms(1) bet_out col123__nos out_one_side)
      hence "A X OS P Y"
        by (meson invert_one_side one_side_symmetry Tarski_neutral_dimensionless_axioms 
            assms(1) one_side_transitivity)
      thus ?thesis
        using P1 col_one_side_out not_col_permutation_4 by blast
    qed
    ultimately show ?thesis
      by (simp add: out_one_side)
  qed
  thus ?thesis
    using P2 one_side_transitivity by blast
qed

lemma ts2__ex_bet2:
  assumes "A C TS B D" and
    "B D TS A C"
  shows "\<exists> X. Bet A X C \<and> Bet B X D"
  by (metis TS_def assms(1) assms(2) bet_col col_permutation_5 l9_18_R1 not_col_permutation_2)

lemma out_one_side_1:
  assumes "\<not> Col A B C" and
    "Col A B X" and
    "X Out C D"
  shows "A B OS C D"
  using assms(1) assms(2) assms(3) not_col_permutation_2 one_side_reflexivity 
    one_side_symmetry os_out_os by blast

lemma out_two_sides_two_sides:
  assumes (*"A \<noteq> PX" and *)
    "Col A B PX" and
    "PX Out X P" and
    "A B TS P Y"
  shows "A B TS X Y"
  using assms(1) assms(2) assms(3) l6_6 l9_5 not_col_permutation_1 by blast

lemma l8_21_bis:
  assumes "X \<noteq> Y" and
    "\<not> Col C A B"
  shows "\<exists> P. Cong A P X Y \<and> A B Perp P A \<and> A B TS C P"
proof -
  have "A \<noteq> B"
    using assms(2) not_col_distincts by blast
  hence "\<exists> P T. A B Perp P A \<and> Col A B T \<and> Bet C T P"
    using l8_21 by auto
  then obtain P T where "A B Perp P A" and "Col A B T" and "Bet C T P" by blast
  have "A B TS C P"
  proof -
    have "\<not> Col P A B"
      using col_permutation_1 perp_not_col \<open>A B Perp P A\<close> by blast
    thus ?thesis 
      using Col_cases TS_def \<open>Bet C T P\<close> \<open>Col A B T\<close> assms(2) by blast
  qed
  have "P \<noteq> A" 
    using \<open>A B Perp P A\<close> perp_distinct by auto
  obtain P' where "(Bet A P P' \<or> Bet A P' P)" and "Cong A P' X Y"
    using segment_construction_2 \<open>P \<noteq> A\<close> by blast
  have "A B Perp P' A" 
    by (metis Bet_cases Col_def cong_identity perp_col1 \<open>A B Perp P A\<close> 
        \<open>Bet A P P' \<or> Bet A P' P\<close> \<open>Cong A P' X Y\<close> assms(1) cong_symmetry perp_comm)
  have "\<not> Col P' A B" 
    using \<open>A B Perp P' A\<close> not_col_permutation_2 perp_not_col by blast
  hence "A B OS P P'" 
    using Out_def \<open>A B Perp P A\<close> \<open>A B Perp P' A\<close> \<open>Bet A P P' \<or> Bet A P' P\<close> 
      out_one_side perp_not_col perp_not_eq_2 by presburger
  hence "A B TS C P'" 
    using \<open>A B TS C P\<close> l9_2 l9_8_2 by blast
  thus ?thesis 
    using \<open>A B Perp P' A\<close> \<open>Cong A P' X Y\<close> by blast
qed

lemma ts__ncol:
  assumes "A B TS X Y"
  shows  "\<not> Col A X Y \<or> \<not> Col B X Y"
  by (metis TS_def assms col_permutation_1 col_transitivity_2 ts_distincts)

lemma one_or_two_sides_aux:
  assumes "\<not> Col C A B" and
    "\<not> Col D A B" and
    "Col A C X"
    and "Col B D X"
  shows "A B TS C D \<or> A B OS C D"
proof -
  have P1: "A \<noteq> X"
    using assms(2) assms(4) col_permutation_2 by blast
  have P2: "B \<noteq> X"
    using assms(1) assms(3) col_permutation_4 by blast
  have P3: "\<not> Col X A B"
    using P1 assms(1) assms(3) col_permutation_5 col_transitivity_1 not_col_permutation_4 
    by blast
  {
    assume Q0: "Bet A C X \<and> Bet B D X"
    hence Q1: "A B OS C X"
      using assms(1) bet_out not_col_distincts not_col_permutation_1 out_one_side by blast
    hence "A B OS X D"
      by (metis Q0 assms(2) assms(4) bet_out_1 col_permutation_2 col_permutation_3 
          invert_one_side l6_4_2 not_bet_and_out not_col_distincts out_one_side)
    hence "A B OS C D"
      using Q1 one_side_transitivity by blast
  }
  hence P4: "Bet A C X \<and> Bet B D X \<longrightarrow> A B OS C D" by blast
  {
    assume "Bet A C X \<and> Bet D X B"
    hence "A B OS C D" 
      by (metis P2 assms(1) bet_out bet_out_1 not_col_distincts one_side_reflexivity 
          one_side_symmetry os_out_os)
  }
  hence P5: "Bet A C X \<and> Bet D X B \<longrightarrow> A B OS C D " by blast
  {
    assume Q0: "Bet A C X \<and> Bet X B D"
    have Q1: "A B TS X D"
      using P3 Q0 TS_def assms(2) col_trivial_3 by blast
    have "A B OS X C"
      using Q0 assms(1) bet_out not_col_distincts one_side_reflexivity one_side_symmetry 
        out_out_one_side by blast
    hence "A B TS C D"
      using Q1 l9_8_2 by blast
  }
  hence P6: "Bet A C X \<and> Bet X B D \<longrightarrow> A B TS C D" by blast
  {
    assume Q1: "Bet C X A \<and> Bet B D X"
    hence Q2: "A B OS C X"
      using P1 assms(1) assms(3) between_equality_2 l6_4_2 not_col_permutation_1 
        not_col_permutation_4 out_one_side by blast
    have "A B OS X D"
      using Q1 assms(2) bet_out not_col_distincts one_side_reflexivity os_out_os by blast
    hence "A B OS C D" using Q2
      using one_side_transitivity by blast
  }
  hence P7: "Bet C X A \<and> Bet B D X \<longrightarrow> A B OS C D" by blast
  {
    assume "Bet C X A \<and> Bet D X B"
    hence "A B OS C D" 
      by (metis Out_def P1 P2 assms(1) bet_out between_symmetry col2__eq invert_one_side 
          l9_19_R2 not_col_distincts out_out_one_side)
  }
  hence P8: "Bet C X A \<and> Bet D X B \<longrightarrow> A B OS C D" by blast
  {
    assume Q1: "Bet C X A \<and> Bet X B D"
    have Q2: "A B TS X D"
      by (metis P3 Q1 assms(2) bet__ts invert_two_sides not_col_distincts not_col_permutation_3)
    have Q3: "A B OS X C"
      using P1 Q1 assms(1) bet_out_1 not_col_permutation_1 out_one_side by auto
    hence "A B TS C D"
      using Q2 l9_8_2 by blast
  }
  hence P9: "Bet C X A \<and> Bet X B D \<longrightarrow> A B TS C D" by blast
  {
    assume Q0: "Bet X A C \<and> Bet B D X"
    have Q1: "A B TS X C"
      by (metis P3 Q0 assms(1) bet__ts col_permutation_2 not_col_distincts)
    have "A B OS X D"
      by (metis NCol_cases Q0 out_one_side assms(2) assms(4) bet_out_1 invert_one_side 
          l6_4_1 not_col_distincts not_out_bet)
    hence "A B TS C D"
      using Q1 l9_2 l9_8_2 by blast
  }
  hence P10: "Bet X A C \<and> Bet B D X \<longrightarrow> A B TS C D" by blast
  {
    assume Q0: "Bet X A C \<and> Bet D X B"
    have Q1: "A B TS X C"
      by (metis NCol_cases P3 Q0 assms(1) bet__ts not_col_distincts)
    have "A B OS X D"
      by (metis P2 P3 Q0 bet_out_1 col_permutation_3 invert_one_side out_one_side)
    hence "A B TS C D"
      using Q1 l9_2 l9_8_2 by blast
  }
  hence P11: "Bet X A C \<and> Bet D X B \<longrightarrow> A B TS C D"
    by blast
  {
    assume Q0: "Bet X A C \<and> Bet X B D"
    hence Q1: "A B TS C X"
      by (simp add: P1 Q0 assms(1) bet__ts between_symmetry not_col_permutation_1)
    have "A B TS D X"
      by (simp add: P2 Q0 assms(2) bet__ts between_symmetry invert_two_sides 
          not_col_permutation_3)
    hence "A B OS C D"
      using Q1 l9_8_1 by blast
  }
  hence P12: "Bet X A C \<and> Bet X B D \<longrightarrow> A B OS C D" by blast
  thus ?thesis using P4 P5 P6 P7 P8 P9 P10 P11
    using Col_def assms(3) assms(4) by auto
qed

lemma cop__one_or_two_sides:
  assumes "Coplanar A B C D" and
    "\<not> Col C A B" and
    "\<not> Col D A B"
  shows "A B TS C D \<or> A B OS C D"
proof -
  obtain X where P1: "Col A B X \<and> Col C D X \<or> Col A C X \<and> Col B D X \<or> Col A D X \<and> Col B C X"
    using Coplanar_def assms(1) by auto
  have P2: "Col A B X \<and> Col C D X \<longrightarrow> A B TS C D \<or> A B OS C D"
    by (metis TS_def l9_19_R2 assms(2) assms(3) not_col_permutation_3 not_col_permutation_5 
        not_out_bet)
  have P3: "Col A C X \<and> Col B D X \<longrightarrow>  A B TS C D \<or> A B OS C D"
    using assms(2) assms(3) one_or_two_sides_aux by blast
  have "Col A D X \<and> Col B C X \<longrightarrow>  A B TS C D \<or> A B OS C D"
    using assms(2) assms(3) l9_2 one_or_two_sides_aux one_side_symmetry by blast
  thus ?thesis
    using P1 P2 P3 by blast
qed

lemma os__coplanar:
  assumes "A B OS C D"
  shows "Coplanar A B C D"
proof -
  have P1: "\<not> Col A B C"
    using assms one_side_not_col123 by blast
  obtain C' where P2: "Bet C B C' \<and> Cong B C' B C"
    using segment_construction by presburger
  have P3: "A B TS D C'"
    by (metis Cong_perm OS_def P2 TS_def assms bet__ts bet_cong_eq invert_one_side l9_10 
        l9_8_2 one_side_not_col123 ts_distincts)
  obtain T where P4: "Col T A B \<and> Bet D T C'"
    using P3 TS_def by blast
  have P5: "C' \<noteq> T"
    using P3 P4 TS_def by blast
  have P6: "Col T B C \<longrightarrow> Coplanar A B C D"
    by (metis Col_def Coplanar_def P2 P4 P5 col_trivial_2 l6_16_1)
  {
    assume Q0: "\<not> Col T B C"
    {
      assume R0: "Bet T B A"
      have S1: "B C TS T A"
        by (metis P1 Q0 R0 bet__ts col_permutation_2 not_col_distincts)
      have "C' Out T D"
        using P4 P5 bet_out_1 by auto
      hence "B C OS T D"
        using P2 Q0 bet_col invert_one_side not_col_permutation_3 out_one_side_1 by blast
      hence R1: "B C TS D A"
        using S1 l9_8_2 by blast
      hence "Coplanar A B C D"
        using ncoplanar_perm_9 ts__coplanar by blast
    }
    hence Q1: "Bet T B A \<longrightarrow> Coplanar A B C D" by blast
    {
      assume R0: "\<not> Bet T B A"
      {
        have R2: "B C OS D T"
        proof -
          have S1: "\<not> Col B C D"
            by (metis Col_perm P2 P3 P4 Q0 bet_col colx ts_distincts)
          have S2: "Col B C C'"
            by (simp add: P2 bet_col col_permutation_4)
          have S3: "C' Out D T"
            using P4 P5 bet_out_1 l6_6 by auto
          thus ?thesis
            using S1 S2 out_one_side_1 by blast
        qed
        have R3: "B C OS T A"
          using P4 Q0 R0 col_permutation_2 col_permutation_5 not_bet_out out_one_side by blast
      }
      hence R1: "B C OS D A"
        by (metis P2 P4 Q0 bet_col bet_out_1 col_permutation_2 col_permutation_5 os_out_os)
      hence "Coplanar A B C D"
        by (simp add: R1 assms coplanar_perm_19 invert_one_side l9_31 one_side_symmetry 
            ts__coplanar)
    }
    hence "\<not> Bet T B A \<longrightarrow> Coplanar A B C D" by blast
    hence "Coplanar A B C D" using Q1 by blast
  }
  hence "\<not> Col T B C \<longrightarrow> Coplanar A B C D" by blast
  thus ?thesis using P6 by blast
qed

lemma coplanar_trans_1:
  assumes "\<not> Col P Q R" and
    "Coplanar P Q R A" and
    "Coplanar P Q R B"
  shows "Coplanar Q R A B"
proof -
  have P1: "Col Q R A \<longrightarrow> Coplanar Q R A B"
    by (simp add: col__coplanar)
  {
    assume T1: "\<not> Col Q R A"
    {
      assume T2: "\<not> Col Q R B"
      {
        have "Col Q A B \<longrightarrow> Coplanar Q R A B"
          using ncop__ncols by blast
        {
          assume S1: "\<not> Col Q A B"
          have U1: "Q R TS P A \<or> Q R OS P A"
            by (simp add: T1 assms(1) assms(2) cop__one_or_two_sides coplanar_perm_8 
                not_col_permutation_2)
          have U2: "Q R TS P B \<or> Q R OS P B"
            using T2 assms(1) assms(3) col_permutation_1 cop__one_or_two_sides 
              coplanar_perm_8 by blast
          have W1: "Q R TS P A \<and> Q R OS P A \<longrightarrow> Q R TS A B \<or> Q R OS A B"
            using l9_9 by blast
          have W2: "Q R TS P A \<and> Q R OS P B \<longrightarrow> Q R TS A B \<or> Q R OS A B"
            using l9_2 l9_8_2 by blast
          have W3: "Q R TS P B \<and> Q R OS P A \<longrightarrow> Q R TS A B \<or> Q R OS A B"
            using l9_8_2 by blast
          have "Q R TS P B \<and> Q R OS P B \<longrightarrow> Q R TS A B \<or> Q R OS A B"
            using l9_9 by blast
          hence S2: "Q R TS A B \<or> Q R OS A B" using U1 U2 W1 W2 W3
            using OS_def l9_2 one_side_transitivity by blast
          have "Coplanar Q R A B"
            using S2 os__coplanar ts__coplanar by blast
        }
        hence "\<not> Col Q A B \<longrightarrow> Coplanar Q R A B" by blast
      }
      hence "Coplanar Q R A B"
        using ncop__ncols by blast
    }
    hence "\<not> Col Q R B \<longrightarrow> Coplanar Q R A B"
      by blast
  }
  hence "\<not> Col Q R A \<longrightarrow> Coplanar Q R A B"
    using ncop__ncols by blast
  thus ?thesis using P1 by blast
qed

lemma col_cop__cop:
  assumes "Coplanar A B C D" and
    "C \<noteq> D" and
    "Col C D E"
  shows "Coplanar A B C E"
proof -
  have "Col D A C \<longrightarrow> Coplanar A B C E"
    by (meson assms(2) assms(3) col_permutation_1 l6_16_1 ncop__ncols)
  moreover
  {
    assume "\<not> Col D A C"
    hence "Coplanar A C B E"
      by (meson assms(1) assms(3) col__coplanar coplanar_trans_1 ncoplanar_perm_11 
          ncoplanar_perm_13)
    hence "Coplanar A B C E"
      using ncoplanar_perm_2 by blast
  }
  ultimately show ?thesis
    by blast
qed

lemma bet_cop__cop:
  assumes "Coplanar A B C E" and
    "Bet C D E"
  shows "Coplanar A B C D"
  by (metis NCol_perm col_cop__cop assms(1) assms(2) bet_col bet_neq12__neq)

lemma col2_cop__cop:
  assumes "Coplanar A B C D" and
    "C \<noteq> D" and
    "Col C D E" and
    "Col C D F"
  shows "Coplanar A B E F"
proof cases
  assume "C = E"
  thus ?thesis
    using assms(1) assms(2) assms(4) col_cop__cop by blast
next
  assume "C \<noteq> E"
  thus ?thesis
    by (metis assms(1) assms(2) assms(3) assms(4) col_cop__cop col_transitivity_1 
        ncoplanar_perm_1 not_col_permutation_4)
qed

lemma col_cop2__cop:
  assumes "U \<noteq> V" and
    "Coplanar A B C U" and
    "Coplanar A B C V" and
    "Col U V P"
  shows "Coplanar A B C P"
proof cases
  assume "Col A B C"
  thus ?thesis
    using ncop__ncol by blast
next
  assume "\<not> Col A B C"
  {
    fix A0 B0 C0
    assume "\<not> Col A0 B0 C0" and "\<not> Col U A0 B0" and "Coplanar A0 B0 C0 U" and 
      "Coplanar A0 B0 C0 V" and "\<not> Col A0 B0 C0"
    have "Coplanar U A0 B0 P" 
      by (meson col_trivial_3 \<open>Coplanar A0 B0 C0 U\<close> \<open>Coplanar A0 B0 C0 V\<close> \<open>\<not> Col A0 B0 C0\<close> 
          assms(1) assms(4) col2_cop__cop coplanar_trans_1 ncoplanar_perm_8 
          not_col_permutation_1)
    hence "Coplanar A0 B0 C0 P" 
      using \<open>Coplanar A0 B0 C0 U\<close>  \<open>\<not> Col U A0 B0\<close> coplanar_perm_18 coplanar_trans_1 by blast
  }
  moreover
  {
    assume "Col U A B" and "Col U A C"
    hence "Coplanar A B C P" 
      by (metis \<open>\<not> Col A B C\<close> assms(1) assms(3) assms(4) col_cop__cop 
          col_transitivity_2 coplanar_perm_14)
  }
  moreover
  {
    assume "Col U A B" and "\<not> Col U A C"
    hence "Coplanar A B C P" 
      using calculation(1) \<open>\<not> Col A B C\<close> assms(2) assms(3) coplanar_perm_2 
        not_col_permutation_5 by blast
  }
  moreover
  {
    assume "\<not> Col U A B" and "Col U A C"
    hence "Coplanar A B C P" 
      using \<open>\<not> Col A B C\<close> assms(2) assms(3) calculation(1) by blast
  }
  moreover
  {
    assume "\<not> Col U A B" and "\<not> Col U A C"
    hence "Coplanar A B C P" 
      using \<open>\<not> Col A B C\<close> assms(2) assms(3) calculation(1) by blast
  }
  ultimately show ?thesis 
    by blast
qed

lemma bet_cop2__cop:
  assumes "Coplanar A B C U" and
    "Coplanar A B C W" and
    "Bet U V W"
  shows "Coplanar A B C V"
proof -
  have "Col U V W"
    using assms(3) bet_col by blast
  hence "Col U W V"
    by (meson not_col_permutation_5)
  thus ?thesis
    using assms(1) assms(2) assms(3) bet_neq23__neq col_cop2__cop by blast
qed

lemma coplanar_pseudo_trans_lem1:
  assumes "\<not> Col P Q R" and
    "Coplanar P Q R A" and
    "Coplanar P Q R B" and
    "Coplanar P Q R C"
  shows "Coplanar A B C R" 
proof cases
  assume "Col R Q A"
  have "Coplanar B C R Q" 
    using assms(1) assms(3) assms(4) coplanar_perm_17 coplanar_trans_1 by blast
  thus ?thesis
    using \<open>Col R Q A\<close> assms(1) col_cop__cop coplanar_perm_2 coplanar_perm_20 
      not_col_distincts by blast
next
  assume "\<not> Col R Q A"
  thus ?thesis
    by (meson assms(1) assms(2) assms(3) assms(4) coplanar_trans_1 ncoplanar_perm_18 
        not_col_permutation_4)
qed

lemma coplanar_pseudo_trans:
  assumes "\<not> Col P Q R" and
    "Coplanar P Q R A" and
    "Coplanar P Q R B" and
    "Coplanar P Q R C" and
    "Coplanar P Q R D"
  shows "Coplanar A B C D"
proof cases
  assume "Col P Q D"
  moreover have "P \<noteq> Q"
    using assms(1) col_trivial_1 by blast
  moreover have "Coplanar A B C Q" 
    using coplanar_pseudo_trans_lem1 
    by (meson assms(1) assms(2) assms(3) assms(4) col_permutation_5 coplanar_perm_2)
  moreover have "Coplanar A B C P" 
  proof -
    have "\<not> Col Q R P"
      using Col_cases assms(1) by blast
    moreover have "Coplanar Q R P A"
      using assms(2) ncoplanar_perm_12 by blast
    moreover have "Coplanar Q R P B"
      using assms(3) ncoplanar_perm_12 by blast
    moreover have "Coplanar Q R P C"
      using assms(4) ncoplanar_perm_12 by blast
    ultimately show ?thesis
      using coplanar_pseudo_trans_lem1 by blast
  qed
  ultimately show ?thesis
    using coplanar_pseudo_trans_lem1 col_cop2__cop by blast
next
  assume "\<not> Col P Q D"
  moreover have "Coplanar P Q D A"
    using NCol_cases assms(1) assms(2) assms(5) coplanar_trans_1 ncoplanar_perm_8 by blast
  moreover have "Coplanar P Q D B"
    using assms(1) assms(3) assms(5) col_permutation_1 coplanar_perm_12 
      coplanar_trans_1 by blast
  moreover have "Coplanar P Q D C"
    by (meson assms(1) assms(4) assms(5) coplanar_perm_7 coplanar_trans_1 
        ncoplanar_perm_14 not_col_permutation_3)
  ultimately show ?thesis 
    using coplanar_pseudo_trans_lem1 by blast
qed

lemma l9_30:
  assumes "\<not> Coplanar A B C P" and
    "\<not> Col D E F" and
    "Coplanar D E F P" and
    "Coplanar A B C X" and
    "Coplanar A B C Y" and
    "Coplanar A B C Z" and
    "Coplanar D E F X" and
    "Coplanar D E F Y" and
    "Coplanar D E F Z"
  shows "Col X Y Z"
proof -
  {
    assume P1: "\<not> Col X Y Z"
    have P2: "\<not> Col A B C"
      using assms(1) col__coplanar by blast
    have "Coplanar A B C P"
    proof -
      have Q2: "Coplanar X Y Z A" 
        by (meson P2 col_trivial_3 assms(4) assms(5) assms(6) coplanar_pseudo_trans ncop__ncols)
      have Q3: "Coplanar X Y Z B"
        using P2 assms(4) assms(5) assms(6) col_trivial_3 coplanar_pseudo_trans 
          ncop__ncols by blast
      have Q4: "Coplanar X Y Z C"
        using P2 assms(4) assms(5) assms(6) col_trivial_2 coplanar_pseudo_trans 
          ncop__ncols by blast
      have "Coplanar X Y Z P"
        using assms(2) assms(3) assms(7) assms(8) assms(9) coplanar_pseudo_trans by blast
      thus ?thesis using P1 Q2 Q3 Q4
        using assms(2) assms(3) assms(7) assms(8) assms(9) coplanar_pseudo_trans by blast
    qed
    hence "False" using assms(1) by blast
  }
  thus ?thesis by blast
qed

lemma cop_per2__col:
  assumes "Coplanar A X Y Z" and
    "A \<noteq> Z" and
    "Per X Z A" and
    "Per Y Z A"
  shows "Col X Y Z"
proof cases
  assume "X = Y \<or> X = Z \<or> Y = Z"
  thus ?thesis
    using not_col_distincts by blast
next
  assume "\<not> (X = Y \<or> X = Z \<or> Y = Z)"
  obtain B where "Cong X A X B" and "Z Midpoint A B" and "Cong Y A Y B"
    using Per_def assms(3) assms(4) per_double_cong by blast
  have "X \<noteq> Y" 
    using \<open>\<not> (X = Y \<or> X = Z \<or> Y = Z)\<close> by blast
  have "X \<noteq> Z" 
    using \<open>\<not> (X = Y \<or> X = Z \<or> Y = Z)\<close> by blast
  have "Y \<noteq> Z" 
    using \<open>\<not> (X = Y \<or> X = Z \<or> Y = Z)\<close> by blast
  obtain I where "Col A X I \<and> Col Y Z I \<or> Col A Y I \<and> Col X Z I \<or> Col A Z I \<and> Col X Y I"
    using Coplanar_def assms(1) by auto
  moreover
  {
    assume "Col A X I" and "Col Y Z I"
    have "Col X Y Z" 
    proof (cases "X = I")
      assume "X = I"
      thus ?thesis
        using Col_cases \<open>Col Y Z I\<close> by blast
    next
      assume "X \<noteq> I"
      have "Col A X I \<and> Col Y Z I \<Longrightarrow> Col X Y Z"
        by (metis (full_types) \<open>\<not> (X = Y \<or> X = Z \<or> Y = Z)\<close> assms(2,3,4) col_per2__per l8_3 l8_8
            not_col_permutation_2 not_col_permutation_4) 
      moreover have "Col A Y I \<and> Col X Z I \<Longrightarrow> Col X Y Z"
        by (simp add: \<open>Col A X I\<close> \<open>Col Y Z I\<close> calculation) 
      moreover have "Col A Z I \<and> Col X Y I \<Longrightarrow> Col X Y Z"
        by (simp add: \<open>Col A X I\<close> \<open>Col Y Z I\<close> calculation(1))
      ultimately show "Col X Y Z"
        using \<open>Col A X I\<close> \<open>Col Y Z I\<close> by blast 
    qed
  }
  moreover have "Col A Y I \<and> Col X Z I \<longrightarrow> Col X Y Z"
  proof cases
    assume "X = I"
    thus ?thesis
      by (metis Col_cases \<open>Cong X A X B\<close> \<open>Cong Y A Y B\<close> \<open>Z Midpoint A B\<close> 
          l4_18 midpoint_distinct_3)
  next
    assume "X \<noteq> I"
    thus ?thesis 
      by (metis Col_cases midpoint_bet per_double_cong \<open>Cong Y A Y B\<close> \<open>Z Midpoint A B\<close> 
          assms(2) assms(3) between_equality between_trivial col_trivial_3 l4_18 l8_3)
  qed
  moreover 
  {
    assume "Col A Z I" and "Col X Y I"
    hence "Z = I \<longrightarrow> Col X Y Z" 
      by simp 
    moreover
    {
      assume "Z \<noteq> I" 
      have "Cong I A I B" 
        using l4_17 [where ?A = "X" and ?B ="Y"]
          \<open>X \<noteq> Y\<close> \<open>Col X Y I\<close> \<open>Cong X A X B\<close> \<open>Cong Y A Y B\<close> by auto
      have "(Col A I B \<and> Cong I A I B) \<longrightarrow> (A = B \<or> I Midpoint A B)" 
        using l7_20_bis by blast
      moreover have "A = B \<longrightarrow> Col X Y Z" 
        using \<open>Z Midpoint A B\<close> assms(2) l7_3 by auto
      moreover have "I Midpoint A B \<longrightarrow> ?thesis" 
        using \<open>Z Midpoint A B\<close> \<open>Z \<noteq> I\<close> l7_17 by blast
      ultimately have "Col X Y Z" 
        using Col_cases Per_def \<open>Col A Z I\<close> \<open>Cong I A I B\<close> \<open>Z Midpoint A B\<close> 
          assms(2) l8_9 by blast
    }
    ultimately have "Col X Y Z" 
      by blast
  }
  ultimately show ?thesis
    by blast
qed

lemma cop_perp2__col:
  assumes "Coplanar A B Y Z" and
    "X Y Perp A B" and
    "X Z Perp A B"
  shows "Col X Y Z"
proof cases
  assume P1: "Col A B X"
  {
    assume Q0: "X = A"
    hence Q1: "X \<noteq> B"
      using assms(3) perp_not_eq_2 by blast
    have Q2: "Coplanar B Y Z X"
      by (simp add: Q0 assms(1) coplanar_perm_9)
    have Q3: "Per Y X B"
      using Q0 assms(2) perp_per_2 by blast
    have "Per Z X B"
      using Q0 assms(3) perp_per_2 by blast
    hence "Col X Y Z"
      using Q1 Q2 Q3 cop_per2__col not_col_permutation_1 by blast
  }
  hence P2: "X = A \<longrightarrow> Col X Y Z" by blast
  {
    assume Q0: "X \<noteq> A"
    have Q1: "A X Perp X Y"
      by (metis P1 Perp_perm Q0 assms(2) perp_col1)
    have Q2: "A X Perp X Z"
      by (metis P1 Perp_perm Q0 assms(3) perp_col1)
    have Q3: "Coplanar A Y Z X" 
      by (metis P1 assms(1) assms(2) col_cop2__cop coplanar_perm_3 coplanar_trivial 
          perp_distinct)
    have Q4: "Per Y X A"
      using Perp_perm Q1 perp_per_2 by blast
    have "Per Z X A"
      using P1 Q0 assms(3) perp_col1 perp_per_1 by auto
    hence "Col X Y Z"
      using Q0 Q3 Q4 cop_per2__col not_col_permutation_1 by blast
  }
  hence "X \<noteq> A \<longrightarrow> Col X Y Z" by blast
  thus ?thesis
    using P2 by blast
next
  assume P1: "\<not> Col A B X"
  obtain Y0 where P2: "Y0 PerpAt X Y A B"
    using Perp_def assms(2) by blast
  obtain Z0 where P3: "Z0 PerpAt X Z A B"
    using Perp_def assms(3) by auto
  have P4: "X Y0 Perp A B"
    by (metis P1 P2 assms(2) perp_col perp_in_col)
  have P5: "X Z0 Perp A B"
    by (metis P1 P3 assms(3) perp_col perp_in_col)
  have P6: "Y0 = Z0"
    by (meson P1 P2 P3 P4 P5 Perp_perm l8_18_uniqueness perp_in_col)
  have P7: "X \<noteq> Y0"
    using P4 perp_not_eq_1 by blast
  have P8: "Col X Y0 Y"
    using P2 col_permutation_5 perp_in_col by blast
  have "Col X Y0 Z"
    using P3 P6 col_permutation_5 perp_in_col by blast
  thus ?thesis
    using P7 P8 col_transitivity_1 by blast
qed

lemma two_sides_dec:
  shows "A B TS C D \<or> \<not> A B TS C D"
  by simp

lemma cop_nts__os:
  assumes "Coplanar A B C D" and
    "\<not> Col C A B" and
    "\<not> Col D A B" and
    "\<not> A B TS C D"
  shows "A B OS C D"
  using assms(1) assms(2) assms(3) assms(4) cop__one_or_two_sides by blast

lemma cop_nos__ts:
  assumes "Coplanar A B C D" and
    "\<not> Col C A B" and
    "\<not> Col D A B" and
    "\<not> A B OS C D"
  shows "A B TS C D"
  using assms(1) assms(2) assms(3) assms(4) cop_nts__os by blast

lemma one_side_dec:
  "A B OS C D \<or> \<not> A B OS C D"
  by simp

lemma cop_dec:
  "Coplanar A B C D \<or> \<not> Coplanar A B C D"
  by simp

lemma ex_diff_cop:
  "\<exists> E. Coplanar A B C E \<and> D \<noteq> E"
  by (metis col_trivial_2 diff_col_ex ncop__ncols)

lemma ex_ncol_cop:
  assumes "D \<noteq> E"
  shows "\<exists> F. Coplanar A B C F \<and> \<not> Col D E F"
proof cases
  assume "Col A B C"
  thus ?thesis
    using assms ncop__ncols not_col_exists by blast
next
  assume P1: "\<not> Col A B C"
  thus ?thesis
  proof -
    have P2: "(Col D E A \<and> Col D E B) \<longrightarrow> (\<exists> F. Coplanar A B C F \<and> \<not> Col D E F)"
      by (meson P1 assms col3 col_trivial_2 ncop__ncols)
    have P3: "(\<not>Col D E A \<and> Col D E B) \<longrightarrow> (\<exists> F. Coplanar A B C F \<and> \<not> Col D E F)"
      using col_trivial_3 ncop__ncols by blast
    have P4: "(Col D E A \<and> \<not>Col D E B) \<longrightarrow> (\<exists> F. Coplanar A B C F \<and> \<not> Col D E F)"
      using col_trivial_2 ncop__ncols by blast
    have "(\<not>Col D E A \<and> \<not>Col D E B) \<longrightarrow> (\<exists> F. Coplanar A B C F \<and> \<not> Col D E F)"
      using col_trivial_3 ncop__ncols by blast
    thus ?thesis using P2 P3 P4 by blast
  qed
qed

lemma ex_ncol_cop2:
  "\<exists> E F. (Coplanar A B C E \<and> Coplanar A B C F \<and> \<not> Col D E F)"
proof -
  have "Coplanar A B C A" 
    by (simp add: coplanar_perm_3 coplanar_trivial)
  have "Coplanar A B C C" 
    using col_trivial_2 ncop__ncols by blast
  have "\<exists>p. A \<noteq> p"
    by (meson col_trivial_3 diff_col_ex3)
  moreover
  { 
    assume "B \<noteq> A"
    hence "D = B \<longrightarrow> (\<exists>p. \<not> Col D p A \<and> Coplanar A B C p)" 
      by (metis Col_cases \<open>Coplanar A B C C\<close> ncop__ncols not_col_exists)
    hence "D = B \<longrightarrow> (\<exists>p pa. Coplanar A B C p \<and> Coplanar A B C pa \<and> \<not> Col D p pa)" 
      using \<open>Coplanar A B C A\<close> by blast
  }
  moreover
  { 
    assume "D \<noteq> B"
    moreover
    { 
      assume "\<exists>p. D \<noteq> B \<and> \<not> Coplanar A B C p"
      hence "D \<noteq> B \<and> \<not> Col A B C"
        using ncop__ncols by blast
      hence "\<exists>p. \<not> Col D p B \<and> Coplanar A B C p" 
        by (metis Col_cases \<open>Coplanar A B C A\<close> \<open>Coplanar A B C C\<close> col2__eq)
    }
    ultimately have ?thesis 
      by (meson col_trivial_3 ncop__ncols not_col_exists)
  }
  ultimately show ?thesis 
    using coplanar_trivial ex_ncol_cop by blast
qed

lemma col2_cop2__eq:
  assumes "\<not> Coplanar A B C U" and
    "U \<noteq> V" and
    "Coplanar A B C P" and
    "Coplanar A B C Q" and
    "Col U V P" and
    "Col U V Q"
  shows "P = Q"
proof -
  have "Col U Q P"
    by (meson assms(2) assms(5) assms(6) col_transitivity_1)
  hence "Col P Q U"
    using not_col_permutation_3 by blast
  thus ?thesis
    using assms(1) assms(3) assms(4) col_cop2__cop by blast
qed

lemma cong3_cop2__col:
  assumes "Coplanar A B C P" and
    "Coplanar A B C Q" and
    "P \<noteq> Q" and
    "Cong A P A Q" and
    "Cong B P B Q" and
    "Cong C P C Q"
  shows "Col A B C"
proof cases
  assume "Col A B C"
  thus ?thesis by blast
next
  assume P1: "\<not> Col A B C"
  obtain M where P2: "M Midpoint P Q"
    using assms(6) l7_25 by blast
  have P3: "Per A M P"
    using P2 Per_def assms(4) by blast
  have P4: "Per B M P"
    using P2 Per_def assms(5) by blast
  have P5: "Per C M P"
    using P2 Per_def assms(6) by blast
  have "False"
  proof cases
    assume Q1: "A = M"
    have Q2: "Coplanar P B C A"
      using assms(1) ncoplanar_perm_21 by blast
    have Q3: "P \<noteq> A"
      by (metis assms(3) assms(4) cong_diff_4)
    have Q4: "Per B A P"
      by (simp add: P4 Q1)
    have Q5: "Per C A P"
      by (simp add: P5 Q1)
    thus ?thesis using Q1 Q2 Q3 Q4 cop_per2__col
      using P1 not_col_permutation_1 by blast
  next
    assume Q0: "A \<noteq> M"
    have Q1: "Col A B M"
    proof -
      have R1: "Coplanar A B P Q"
        using P1 assms(1) assms(2) coplanar_trans_1 ncoplanar_perm_8 not_col_permutation_2 
        by blast
      hence R2: "Coplanar P A B M"
        using P2 bet_cop__cop coplanar_perm_14 midpoint_bet ncoplanar_perm_6 by blast
      have R3: "P \<noteq> M"
        using P2 assms(3) l7_3_2 l7_9_bis by blast
      have R4: "Per A M P"
        by (simp add: P3)
      have R5: "Per B M P"
        by (simp add: P4)
      thus ?thesis
        using R2 R3 R4 cop_per2__col by blast
    qed
    have "Col A C M"
    proof -
      have R1: "Coplanar P A C M"
        using P1 Q1 assms(1) col2_cop__cop coplanar_perm_22 ncoplanar_perm_3 
          not_col_distincts by blast
      have R2: "P \<noteq> M"
        using P2 assms(3) l7_3_2 symmetric_point_uniqueness by blast
      have R3: "Per A M P"
        by (simp add: P3)
      have "Per C M P"
        by (simp add: P5)
      thus ?thesis
        using R1 R2 R3 cop_per2__col by blast
    qed
    thus ?thesis
      using NCol_perm P1 Q0 Q1 col_trivial_3 colx by blast
  qed
  thus ?thesis by blast
qed

lemma l9_38:
  assumes "A B C TSP P Q"
  shows "A B C TSP Q P"
  using Bet_perm TSP_def assms by blast

lemma l9_39:
  assumes "A B C TSP P R" and
    "Coplanar A B C D" and
    "D Out P Q"
  shows "A B C TSP Q R"
proof -
  have P1: "\<not> Col A B C"
    using TSP_def assms(1) ncop__ncol by blast
  have P2: "\<not> Coplanar A B C Q"
    by (metis TSP_def assms(1) assms(2) assms(3) col_cop2__cop l6_6 out_col out_diff2)
  have P3: "\<not> Coplanar A B C R"
    using TSP_def assms(1) by blast
  obtain T where P3A: "Coplanar A B C T \<and> Bet P T R"
    using TSP_def assms(1) by blast
  have W1: "D = T \<longrightarrow> A B C TSP Q R"
    using P2 P3 P3A TSP_def assms(3) bet_out__bet by blast
  {
    assume V1: "D \<noteq> T"
    have V1A: "\<not> Col P D T" using P3A col_cop2__cop
      by (metis TSP_def V1 assms(1) assms(2) col2_cop2__eq col_trivial_2)
    have V1B: "D T TS P R"
      by (metis P3 P3A V1A bet__ts invert_two_sides not_col_permutation_3)
    have "D T OS P Q"
      using V1A assms(3) not_col_permutation_1 out_one_side by blast
    hence V2: "D T TS Q R"
      using V1B l9_8_2 by blast
    then obtain T' where V3: "Col T' D T \<and> Bet Q T' R"
      using TS_def by blast
    have V4: "Coplanar A B C T'"
      using Col_cases P3A V1 V3 assms(2) col_cop2__cop by blast
    hence "A B C TSP Q R"
      using P2 P3 TSP_def V3 by blast
  }
  hence "D \<noteq> T \<longrightarrow> A B C TSP Q R" by blast
  thus ?thesis using W1 by blast
qed

lemma l9_41_1:
  assumes "A B C TSP P R" and
    "A B C TSP Q R"
  shows "A B C OSP P Q"
  using OSP_def assms(1) assms(2) by blast

lemma l9_41_2:
  assumes "A B C TSP P R" and
    "A B C OSP P Q"
  shows "A B C TSP Q R"
proof -
  have P1: "\<not> Coplanar A B C P"
    using TSP_def assms(1) by blast
  obtain S where P2: " A B C TSP P S \<and> A B C TSP Q S"
    using OSP_def assms(2) by blast
  obtain X where P3: "Coplanar A B C X \<and> Bet P X S"
    using P2 TSP_def by blast
  have P4: "\<not> Coplanar A B C P \<and> \<not> Coplanar A B C S"
    using P2 TSP_def by blast
  obtain Y where P5: "Coplanar A B C Y \<and> Bet Q Y S"
    using P2 TSP_def by blast
  have P6: "\<not> Coplanar A B C Q \<and> \<not> Coplanar A B C S"
    using P2 TSP_def by blast
  have P7: "X \<noteq> P \<and> S \<noteq> X \<and> Q \<noteq> Y \<and> S \<noteq> Y"
    using P3 P4 P5 P6 by blast
  {
    assume Q1: "Col P Q S"
    have Q2: "X = Y"
    proof -
      have R2: "Q \<noteq> S"
        using P5 P6 bet_neq12__neq by blast
      have R5: "Col Q S X" 
        by (metis Col_def P3 Q1 between_equality_2 l6_16_1 not_bet_distincts)
      have "Col Q S Y"
        by (simp add: P5 bet_col col_permutation_5)
      thus ?thesis
        using P2 P3 P5 R2 R5 TSP_def col2_cop2__eq by blast
    qed
    hence "X Out P Q"
      by (metis P3 P5 P7 l6_2)
    hence "A B C TSP Q R"
      using P3 assms(1) l9_39 by blast
  }
  hence P7: "Col P Q S \<longrightarrow> A B C TSP Q R" by blast
  {
    assume Q1: "\<not> Col P Q S"
    obtain Z where Q2: "Bet X Z Q \<and> Bet Y Z P"
      using P3 P5 inner_pasch by blast
    {
      assume "X = Z"
      hence "False"
        by (metis P2 P3 P5 Q1 Q2 TSP_def bet_col col_cop2__cop l6_16_1 not_col_permutation_5)
    }
    hence Q3: "X \<noteq> Z" by blast
    have "Y \<noteq> Z"
    proof -
      have "X \<noteq> Z"
        by (meson \<open>X = Z \<Longrightarrow> False\<close>)
      hence "Z \<noteq> Y"
        by (metis P2 P3 P5 Q2 TSP_def bet_col col_cop2__cop)
      thus ?thesis
        by meson
    qed
    hence "Y Out P Z"
      using Q2 bet_out l6_6 by auto
    hence Q4: "A B C TSP Z R"
      using assms(1) P5 l9_39 by blast
    have "X Out Z Q"
      using Q2 Q3 bet_out by auto
    hence "A B C TSP Q R"
      using Q4 P3 l9_39 by blast
  }
  hence "\<not> Col P Q S \<longrightarrow> A B C TSP Q R" by blast
  thus ?thesis using P7 by blast
qed

lemma tsp_exists:
  assumes "\<not> Coplanar A B C P"
  shows "\<exists> Q. A B C TSP P Q"
proof -
  obtain Q where "Bet P A Q" and "Cong A Q A P"
    using segment_construction by blast
  moreover have "Coplanar A B C A"
    using coplanar_trivial ncoplanar_perm_5 by blast
  moreover have "\<not> Coplanar A B C Q" 
    by (metis assms bet_col bet_col1 between_cong between_symmetry calculation(1) 
        calculation(2) calculation(3) col2_cop2__eq)
  ultimately show ?thesis
    using TSP_def assms by blast
qed

lemma osp_reflexivity:
  assumes "\<not> Coplanar A B C P"
  shows "A B C OSP P P"
  by (meson assms l9_41_1 tsp_exists)

lemma osp_symmetry:
  assumes "A B C OSP P Q"
  shows "A B C OSP Q P"
  using OSP_def assms by auto

lemma osp_transitivity:
  assumes "A B C OSP P Q" and
    "A B C OSP Q R"
  shows "A B C OSP P R"
  using OSP_def assms(1) assms(2) l9_41_2 by blast

lemma cop3_tsp__tsp:
  assumes "\<not> Col D E F" and
    "Coplanar A B C D" and
    "Coplanar A B C E" and
    "Coplanar A B C F" and
    "A B C TSP P Q"
  shows "D E F TSP P Q"
proof -
  obtain T where "Coplanar A B C T" and "Bet P T Q"
    using TSP_def assms(5) by blast
  have "\<not> Col A B C"
    using TSP_def assms(5) ncop__ncols by blast
  have "Coplanar D E F A \<and> Coplanar D E F B \<and> Coplanar D E F C \<and> Coplanar D E F T"
  proof -
    have "Coplanar D E F A"
      using assms(2) assms(3) assms(4) col_trivial_3 coplanar_pseudo_trans ncop__ncols 
      by (meson \<open>\<not> Col A B C\<close>)
    moreover have "Coplanar D E F B"
      using assms(2) assms(3) assms(4) col_trivial_2 coplanar_pseudo_trans ncop__ncols 
      by (meson \<open>\<not> Col A B C\<close>)
    moreover have "Coplanar D E F C"
      by (meson  \<open>\<not> Col A B C\<close> assms(2) assms(3) assms(4) coplanar_perm_16 coplanar_pseudo_trans 
          coplanar_trivial)
    moreover have "Coplanar D E F T"
      using \<open>\<not> Col A B C\<close> assms(2) assms(3) assms(4) coplanar_pseudo_trans \<open>Coplanar A B C T\<close> 
      by blast
    ultimately show ?thesis 
      by simp
  qed
  hence "\<not> Coplanar D E F P"
    using TSP_def assms(1) assms(5) coplanar_pseudo_trans by auto
  hence "\<not> Coplanar D E F Q"
    using TSP_def assms(5) bet_col bet_col1 col2_cop2__eq \<open>Bet P T Q\<close> \<open>Coplanar A B C T\<close> 
      \<open>Coplanar D E F A \<and> Coplanar D E F B \<and> Coplanar D E F C \<and> Coplanar D E F T\<close> by metis
  thus ?thesis 
    using TSP_def \<open>Bet P T Q\<close> \<open>\<not> Coplanar D E F P\<close>
      \<open>Coplanar D E F A \<and> Coplanar D E F B \<and> Coplanar D E F C \<and> Coplanar D E F T\<close> by blast
qed

lemma cop3_osp__osp:
  assumes "\<not> Col D E F" and
    "Coplanar A B C D" and
    "Coplanar A B C E" and
    "Coplanar A B C F" and
    "A B C OSP P Q"
  shows "D E F OSP P Q"
proof -
  obtain R where "A B C TSP P R" and "A B C TSP Q R"
    using OSP_def assms(5) by blast
  thus ?thesis
    using OSP_def assms(1) assms(2) assms(3) assms(4) cop3_tsp__tsp by blast
qed

lemma ncop_distincts:
  assumes "\<not> Coplanar A B C D"
  shows "A \<noteq> B \<and> A \<noteq> C \<and> A \<noteq> D \<and> B \<noteq> C \<and> B \<noteq> D \<and> C \<noteq> D"
  using Coplanar_def assms col_trivial_1 col_trivial_2 by blast

lemma tsp_distincts:
  assumes "A B C TSP P Q"
  shows "A \<noteq> B \<and> A \<noteq> C \<and> B \<noteq> C \<and> A \<noteq> P \<and> B \<noteq> P \<and> C \<noteq> P \<and> A \<noteq> Q \<and> B \<noteq> Q \<and> C \<noteq> Q \<and> P \<noteq> Q"
proof -
  obtain X where "\<not> Coplanar A B C P \<and> \<not> Coplanar A B C Q \<and> Coplanar A B C X \<and> Bet P X Q" 
    by (metis TSP_def assms)
  hence "Q \<noteq> X"
    by force
  thus ?thesis 
    using \<open>\<not> Coplanar A B C P \<and> \<not> Coplanar A B C Q \<and> Coplanar A B C X \<and> Bet P X Q\<close> 
      bet_neq32__neq ncop_distincts by blast
qed

lemma osp_distincts:
  assumes "A B C OSP P Q"
  shows "A \<noteq> B \<and> A \<noteq> C \<and> B \<noteq> C \<and> A \<noteq> P \<and> B \<noteq> P \<and> C \<noteq> P \<and> A \<noteq> Q \<and> B \<noteq> Q \<and> C \<noteq> Q"
  using OSP_def assms tsp_distincts by blast

lemma tsp__ncop1:
  assumes "A B C TSP P Q"
  shows "\<not> Coplanar A B C P"
  using TSP_def assms by blast

lemma tsp__ncop2:
  assumes "A B C TSP P Q"
  shows "\<not> Coplanar A B C Q"
  using TSP_def assms by auto

lemma osp__ncop1:
  assumes "A B C OSP P Q"
  shows "\<not> Coplanar A B C P"
  using OSP_def TSP_def assms by blast

lemma osp__ncop2:
  assumes "A B C OSP P Q"
  shows "\<not> Coplanar A B C Q"
  using assms osp__ncop1 osp_symmetry by blast

lemma tsp__nosp:
  assumes "A B C TSP P Q"
  shows "\<not> A B C OSP P Q"
  using assms l9_41_2 tsp_distincts by blast

lemma osp__ntsp:
  assumes "A B C OSP P Q"
  shows "\<not> A B C TSP P Q"
  using assms tsp__nosp by blast

lemma osp_bet__osp:
  assumes "A B C OSP P R" and
    "Bet P Q R"
  shows "A B C OSP P Q"
proof -
  obtain S where "A B C TSP P S"
    using OSP_def assms(1) by blast
  then obtain Y where "Coplanar A B C Y" and "Bet R Y S"
    using TSP_def assms(1) l9_41_2 by blast
  obtain X where "Coplanar A B C X" and "Bet P X S" 
    by (metis TSP_def \<open>A B C TSP P S\<close>)
  have "P \<noteq> X \<and> S \<noteq> X \<and> R \<noteq> Y" 
    using TSP_def \<open>A B C TSP P S\<close> \<open>Coplanar A B C X\<close> \<open>Coplanar A B C Y\<close> assms(1) 
      osp__ncop2 by blast
  {
    assume "Col P R S"
    have "A B C TSP Q S"
    proof -
      have "X = Y"
      proof -
        have "\<not> Coplanar A B C R"
          using assms(1) osp__ncop2 by blast
        moreover have "R \<noteq> S" 
          using \<open>Bet R Y S\<close> \<open>P \<noteq> X \<and> S \<noteq> X \<and> R \<noteq> Y\<close> between_identity by blast
        moreover have "Col R S X" 
          by (metis Bet_cases Col_cases Col_def \<open>Bet P X S\<close> \<open>Col P R S\<close> between_equality_2 
              col_transitivity_1 outer_transitivity_between point_construction_different)
        moreover have "Col R S Y" 
          using \<open>Bet R Y S\<close> bet_col1 between_trivial by blast
        ultimately show ?thesis 
          using  \<open>Coplanar A B C X\<close> \<open>Coplanar A B C Y\<close> col2_cop2__eq by blast
      qed
      have "X Out P R" 
        using \<open>Bet P X S\<close> \<open>Bet R Y S\<close> \<open>P \<noteq> X \<and> S \<noteq> X \<and> R \<noteq> Y\<close> \<open>X = Y\<close> l6_2 by blast
      hence "Y Out P Q" 
        using \<open>X = Y\<close> assms(2) out_bet_out_1 by blast 
      thus ?thesis 
        using \<open>A B C TSP P S\<close> \<open>Coplanar A B C Y\<close> l9_39 by blast
    qed
    hence "A B C OSP P Q" 
      using OSP_def \<open>A B C TSP P S\<close> by blast
  }
  moreover
  {
    assume "\<not> Col P R S"
    have "X Y OS P R"
    proof -
      have "P \<noteq> X \<and> S \<noteq> X \<and> R \<noteq> Y \<and> S \<noteq> Y" 
        using \<open>A B C TSP P S\<close> \<open>Coplanar A B C Y\<close> \<open>P \<noteq> X \<and> S \<noteq> X \<and> R \<noteq> Y\<close> tsp__ncop2 by force
      have "\<not> Col S X Y"
        using bet_out_1 col_out2_col col_permutation_5 not_col_permutation_4 
        by (metis \<open>Bet P X S\<close> \<open>Bet R Y S\<close> \<open>P \<noteq> X \<and> S \<noteq> X \<and> R \<noteq> Y \<and> S \<noteq> Y\<close> \<open>\<not> Col P R S\<close>)
      have "X Y TS P S"
        using Col_perm bet__ts bet_col col_transitivity_2 
        by (metis \<open>Bet P X S\<close> \<open>P \<noteq> X \<and> S \<noteq> X \<and> R \<noteq> Y\<close> \<open>\<not> Col S X Y\<close>)
      have "X Y TS R S"
        using assms(1) bet__ts col_cop2__cop invert_two_sides not_col_distincts osp__ncop2 
          \<open>Bet R Y S\<close> \<open>Coplanar A B C X\<close> \<open>Coplanar A B C Y\<close> \<open>\<not> Col S X Y\<close> by metis
      thus ?thesis 
        using \<open>X Y TS P S\<close> l9_8_1 by auto
    qed
    hence "X Y OS P Q" 
      using assms(2) l9_17 by blast
    then obtain S' where "X Y TS P S'" and "X Y TS Q S'"
      using OS_def by blast
    have "\<not> Col P X Y \<and> \<not> Col S' X Y \<and> (\<exists> T::'p. Col T X Y \<and> Bet P T S')" 
      using TS_def \<open>X Y TS P S'\<close> by blast
    have "\<not> Col Q X Y \<and> \<not> Col S' X Y \<and> (\<exists> T::'p. Col T X Y \<and> Bet Q T S')" 
      using TS_def \<open>X Y TS Q S'\<close> by force
    obtain X' where "Col X' X Y" and "Bet P X' S'" and "X Y TS Q S'" 
      using \<open>X Y TS Q S'\<close> \<open>\<not> Col P X Y \<and> \<not> Col S' X Y \<and> (\<exists>T. Col T X Y \<and> Bet P T S')\<close> by blast
    obtain Y' where "Col Y' X Y" and "Bet Q Y' S'" 
      using \<open>\<not> Col Q X Y \<and> \<not> Col S' X Y \<and> (\<exists>T. Col T X Y \<and> Bet Q T S')\<close> by blast
    have "Coplanar A B C X'" 
      using \<open>Coplanar A B C X\<close> Col_cases col_cop2__cop ts_distincts 
      by (metis not_col_distincts \<open>Col X' X Y\<close> \<open>Coplanar A B C Y\<close> 
          \<open>\<not> Col P X Y \<and> \<not> Col S' X Y \<and> (\<exists>T. Col T X Y \<and> Bet P T S')\<close>)
    have "Coplanar A B C Y'"
      using Col_cases col_cop2__cop ts_distincts 
      by (metis not_col_distincts \<open>Col Y' X Y\<close> \<open>Coplanar A B C X\<close> \<open>Coplanar A B C Y\<close> 
          \<open>\<not> Col P X Y \<and> \<not> Col S' X Y \<and> (\<exists>T. Col T X Y \<and> Bet P T S')\<close>)
    have "\<not> Coplanar A B C S'"
      using assms(1) bet_col bet_col1 col2_cop2__eq osp__ncop1 
      by (metis \<open>Bet P X' S'\<close> \<open>Col X' X Y\<close> \<open>Coplanar A B C X'\<close> 
          \<open>\<not> Col P X Y \<and> \<not> Col S' X Y \<and> (\<exists>T. Col T X Y \<and> Bet P T S')\<close>)
    hence "A B C OSP P Q"
    proof -
      have "A B C TSP P S'"
        using TSP_def 
        by (meson osp__ncop1 \<open>Bet P X' S'\<close> \<open>Coplanar A B C X'\<close> \<open>\<not> Coplanar A B C S'\<close> assms(1))
      moreover have "A B C TSP Q S'"
        using TSP_def bet_col col_cop2__cop 
        by (metis \<open>Bet Q Y' S'\<close> \<open>Col Y' X Y\<close> \<open>Coplanar A B C Y'\<close> 
            \<open>\<not> Col Q X Y \<and> \<not> Col S' X Y \<and> (\<exists>T. Col T X Y \<and> Bet Q T S')\<close> \<open>\<not> Coplanar A B C S'\<close>)
      ultimately show ?thesis 
        using l9_41_1 by blast
    qed
  }
  ultimately show ?thesis 
    by blast
qed

lemma l9_18_3:
  assumes "Coplanar A B C P" and
    "Col X Y P"
  shows "A B C TSP X Y \<longleftrightarrow> (Bet X P Y \<and> \<not> Coplanar A B C X \<and> \<not> Coplanar A B C Y)"
  by (meson TSP_def assms(1) assms(2) l9_39 not_bet_out not_col_permutation_5 tsp_distincts)

lemma bet_cop__tsp:
  assumes "\<not> Coplanar A B C X" and
    "P \<noteq> Y" and
    "Coplanar A B C P" and
    "Bet X P Y"
  shows "A B C TSP X Y"
  using TSP_def assms(1) assms(2) assms(3) assms(4) bet_col bet_col1 col2_cop2__eq by metis

lemma cop_out__osp:
  assumes "\<not> Coplanar A B C X" and
    "Coplanar A B C P" and
    "P Out X Y"
  shows "A B C OSP X Y"
  by (meson OSP_def assms(1) assms(2) assms(3) l9_39 tsp_exists)

lemma l9_19_3:
  assumes "Coplanar A B C P" and
    "Col X Y P"
  shows "A B C OSP X Y \<longleftrightarrow> (P Out X Y \<and> \<not> Coplanar A B C X)"
  by (meson assms(1) assms(2) cop_out__osp l6_4_2 l9_18_3 not_col_permutation_5 osp__ncop1 
      osp__ncop2 tsp__nosp)

lemma cop2_ts__tsp:
  assumes "\<not> Coplanar A B C X" and "Coplanar A B C D" and
    "Coplanar A B C E" and "D E TS X Y"
  shows "A B C TSP X Y"
proof -
  obtain T where "Col T D E" and "Bet X T Y"
    using TS_def assms(4) by blast
  moreover hence "Coplanar A B C T"
    using assms(2) assms(3) assms(4) col_cop2__cop not_col_permutation_2 ts_distincts by blast
  ultimately show ?thesis 
    by (metis TS_def assms(1) assms(4) bet_cop__tsp)
qed

lemma cop2_os__osp:
  assumes "\<not> Coplanar A B C X" and
    "Coplanar A B C D" and
    "Coplanar A B C E" and
    "D E OS X Y"
  shows "A B C OSP X Y"
proof -
  obtain Z where "D E TS X Z" and "D E TS Y Z"
    using OS_def assms(4) by blast
  hence "A B C TSP X Z"
    using assms(1) assms(2) assms(3) cop2_ts__tsp by blast
  moreover hence "A B C TSP Y Z"
    using assms(2) assms(3) cop2_ts__tsp l9_2 tsp__ncop2 \<open>D E TS Y Z\<close> by meson
  ultimately show ?thesis
    using l9_41_1 by blast
qed

lemma cop3_tsp__ts:
  assumes "D \<noteq> E" and
    "Coplanar A B C D" and
    "Coplanar A B C E" and
    "Coplanar D E X Y" and
    "A B C TSP X Y"
  shows "D E TS X Y"
  by (meson assms(1) assms(2) assms(3) assms(4) assms(5) col_cop2__cop cop2_os__osp
      cop_nts__os not_col_permutation_2 tsp__ncop1 tsp__ncop2 tsp__nosp)

lemma cop3_osp__os:
  assumes "D \<noteq> E" and
    "Coplanar A B C D" and
    "Coplanar A B C E" and
    "Coplanar D E X Y" and
    "A B C OSP X Y"
  shows "D E OS X Y"
  by (meson assms(1) assms(2) assms(3) assms(4) assms(5) col_cop2__cop cop2_ts__tsp 
      cop_nts__os not_col_permutation_2 osp__ncop1 osp__ncop2 tsp__nosp)

lemma cop_tsp__ex_cop2:
  assumes (*"Coplanar A B C P" and*)
    "A B C TSP D E"
  shows "\<exists> Q. (Coplanar A B C Q \<and> Coplanar D E P Q \<and> P \<noteq> Q)"
proof cases
  assume "Col D E P"
  thus ?thesis
    by (meson ex_diff_cop ncop__ncols)
next
  assume "\<not> Col D E P"
  then obtain Q where "Coplanar A B C Q \<and> Bet D Q E \<and> \<not> Col D E P"
    using TSP_def assms(1) by blast
  thus ?thesis
    using Col_perm bet_col ncop__ncols by blast
qed

lemma cop_osp__ex_cop2:
  assumes "Coplanar A B C P" and
    "A B C OSP D E"
  shows "\<exists> Q. Coplanar A B C Q \<and> Coplanar D E P Q \<and> P \<noteq> Q"
proof cases
  assume "Col D E P"
  thus ?thesis
    by (metis col_trivial_3 diff_col_ex ncop__ncols)
next
  assume P1: "\<not> Col D E P"
  obtain E' where P2: "Bet E P E' \<and> Cong P E' P E"
    using segment_construction by blast
  have P3: "\<not> Col D E' P"
    by (metis P1 P2 bet_col bet_cong_eq between_symmetry col_permutation_5 l5_2 l6_16_1)
  have P4: "A B C TSP D E'"
    by (metis P2 P3 assms(1) assms(2) bet_cop__tsp l9_41_2 not_col_distincts 
        osp__ncop2 osp_symmetry)
  hence "\<not> Coplanar A B C D \<and> \<not> Coplanar A B C E' \<and> (\<exists> T. Coplanar A B C T \<and> Bet D T E')"
    by (simp add: TSP_def)
  then obtain Q where P7: "Coplanar A B C Q \<and> Bet D Q E'"
    by blast
  hence "Coplanar D E' P Q"
    using bet_col ncop__ncols ncoplanar_perm_5 by blast
  hence "Coplanar D E P Q"
    using Col_perm P2 P3 bet_col col_cop__cop ncoplanar_perm_5 not_col_distincts by blast
  thus ?thesis
    using P3 P7 bet_col col_permutation_5 by blast
qed

lemma sac__coplanar:
  assumes "Saccheri A B C D"
  shows "Coplanar A B C D"
  using Saccheri_def assms ncoplanar_perm_4 os__coplanar by blast

lemma ex_sym:
  "\<exists> Y. (A B Perp X Y \<or> X = Y) \<and> (\<exists> M. Col A B M \<and> M Midpoint X Y)"
proof cases
  assume "Col A B X"
  thus ?thesis
    using l7_3_2 by blast
next
  assume "\<not> Col A B X"
  then obtain M0 where "Col A B M0" and "A B Perp X M0"
    using l8_18_existence by blast
  obtain Z where "M0 Midpoint X Z"
    using symmetric_point_construction by blast
  thus ?thesis 
    using Perp_cases bet_col midpoint_bet perp_col 
    by (metis \<open>A B Perp X M0\<close> \<open>Col A B M0\<close>)
qed

lemma is_image_is_image_spec:
  assumes "A \<noteq> B"
  shows "P' P Reflect A B \<longleftrightarrow> P' P ReflectL A B"
  by (simp add: Reflect_def assms)

lemma ex_sym1:
  assumes "A \<noteq> B"
  shows "\<exists> Y. (A B Perp X Y \<or> X = Y) \<and> (\<exists> M. Col A B M \<and> M Midpoint X Y \<and> X Y Reflect A B)"
proof cases
  assume "Col A B X"
  thus ?thesis
    by (meson ReflectL_def Reflect_def assms l7_3_2)
next
  assume P0: "\<not> Col A B X"
  then obtain M0 where P1: "Col A B M0 \<and> A B Perp X M0"
    using l8_18_existence by blast
  obtain Z where P2: "M0 Midpoint X Z"
    using symmetric_point_construction by blast
  have P3: "A B Perp X Z"
  proof cases
    assume "X = Z"
    thus ?thesis
      using P1 P2 P0 midpoint_distinct by blast
  next
    assume "X \<noteq> Z"
    hence P2: "X Z Perp A B"
      using P1 P2 Perp_cases bet_col midpoint_bet perp_col by blast
    show ?thesis
      by (simp add: Perp_perm Tarski_neutral_dimensionless_axioms P2)
  qed
  have P10: "(A B Perp X Z \<or> X = Z)"
    by (simp add: P3)
  have "\<exists> M. Col A B M \<and> M Midpoint X Z \<and> X Z Reflect A B"
    using P1 P2 P3 ReflectL_def assms is_image_is_image_spec l7_2 perp_right_comm by blast
  thus ?thesis
    using P3 by blast
qed

lemma l10_2_uniqueness:
  assumes "P1 P Reflect A B" and
    "P2 P Reflect A B"
  shows "P1 = P2"
proof cases
  assume "A = B"
  thus ?thesis
    using Reflect_def assms(1) assms(2) symmetric_point_uniqueness by auto
next
  assume "A \<noteq> B"
  hence "P1 P ReflectL A B"
    using assms(1) is_image_is_image_spec by auto
  hence "A B Perp P P1 \<or> P = P1"
    using ReflectL_def by blast
  have "P2 P ReflectL A B"
    using assms(2) is_image_is_image_spec \<open>A \<noteq> B\<close> by blast
  hence "A B Perp P P2 \<or> P = P2"
    using ReflectL_def by blast
  obtain X where "X Midpoint P P1" and "Col A B X"
    by (metis ReflectL_def assms(1) col_trivial_1 is_image_is_image_spec midpoint_existence)
  obtain Y where "Y Midpoint P P2" and "Col A B Y"
    by (metis ReflectL_def assms(2) col_trivial_1 is_image_is_image_spec midpoint_existence)
  {
    assume "A B Perp P P1" and "A B Perp P P2"
    have "P \<noteq> X" 
      using \<open>A B Perp P P1\<close> \<open>X Midpoint P P1\<close> is_midpoint_id perp_not_eq_2 by blast
    have "P \<noteq> Y" 
      using \<open>A B Perp P P2\<close> \<open>Y Midpoint P P2\<close> is_midpoint_id perp_not_eq_2 by blast
    have "P X Perp A B" 
      using Perp_perm \<open>A B Perp P P1\<close> \<open>P \<noteq> X\<close> \<open>X Midpoint P P1\<close> bet_col midpoint_bet 
        not_col_permutation_5 perp_col1 by blast
    have "P Y Perp A B" 
      using Perp_perm \<open>A B Perp P P2\<close> \<open>P \<noteq> Y\<close> \<open>Y Midpoint P P2\<close> bet_col midpoint_bet 
        not_col_permutation_5 perp_col1 by blast
    hence "P1 = P2" 
      by (metis Perp_perm l7_2 l7_9_bis \<open>Col A B Y\<close> \<open>P X Perp A B\<close> \<open>X Midpoint P P1\<close> 
          \<open>\<And>thesis. (\<And>X. \<lbrakk>X Midpoint P P1; Col A B X\<rbrakk> \<Longrightarrow> thesis) \<Longrightarrow> thesis\<close> 
          \<open>Y Midpoint P P2\<close> l7_17 l8_18_uniqueness)
  }
  hence "(A B Perp P P1 \<and> A B Perp P P2) \<longrightarrow> P1 = P2" by blast
  moreover have "(P = P1 \<and> A B Perp P P2) \<longrightarrow> P1 = P2" 
    by (metis \<open>Col A B X\<close> \<open>Col A B Y\<close> \<open>X Midpoint P P1\<close> \<open>Y Midpoint P P2\<close> colx is_midpoint_id 
        l8_16_1 midpoint_col midpoint_distinct_2)
  moreover have "(P = P2 \<and> A B Perp P P1) \<longrightarrow> P1 = P2" 
    by (metis \<open>Col A B X\<close> \<open>Col A B Y\<close> \<open>X Midpoint P P1\<close> \<open>Y Midpoint P P2\<close> l8_16_1 
        l8_20_2 midpoint_col not_col_distincts perp_col2)
  ultimately show ?thesis 
    using \<open>A B Perp P P1 \<or> P = P1\<close> \<open>A B Perp P P2 \<or> P = P2\<close> by fastforce
qed

lemma l10_2_uniqueness_spec:
  assumes "P1 P ReflectL A B" and
    "P2 P ReflectL A B"
  shows "P1 = P2"
proof -
  have "A B Perp P P1 \<or> P = P1"
    using ReflectL_def assms(1) by blast
  moreover obtain X1 where "X1 Midpoint P P1" and "Col A B X1"
    using ReflectL_def assms(1) by blast
  moreover have "A B Perp P P2 \<or> P = P2"
    using ReflectL_def assms(2) by blast
  moreover obtain X2 where "X2 Midpoint P P2" and "Col A B X2"
    using ReflectL_def assms(2) by blast
  {
    assume "A B Perp P P1" and "A B Perp P P2"
    have "P1 P Reflect A B" 
      using Reflect_def \<open>A B Perp P P1\<close> assms(1) perp_distinct by auto
    moreover have "P2 P Reflect A B" 
      using \<open>A B Perp P P2\<close> assms(2) is_image_is_image_spec perp_distinct by auto
    ultimately have "P1 = P2" 
      using l10_2_uniqueness by auto
  }
  moreover
  {
    assume "A B Perp P P1 \<and> P = P2"
    hence "P1 = P2" 
      by (metis colx perp_not_col2 \<open>Col A B X2\<close> \<open>X2 Midpoint P P2\<close> calculation(2) 
          calculation(3) l8_20_2 midpoint_col)
  }
  moreover
  {
    assume "P = P1" and "A B Perp P P2"
    hence "P1 = P2" 
      by (metis \<open>Col A B X2\<close> \<open>X2 Midpoint P P2\<close> calculation(2) calculation(3) 
          colx l8_20_2 midpoint_col perp_not_col2)
  }
  ultimately show ?thesis 
    by blast
qed

lemma l10_2_existence_spec:
  "\<exists> P'. P' P ReflectL A B"
proof cases
  assume "Col A B P"
  thus ?thesis
    using ReflectL_def l7_3_2 by blast
next
  assume "\<not> Col A B P"
  then obtain X where "Col A B X \<and> A B Perp P X"
    using l8_18_existence by blast
  moreover obtain P' where "X Midpoint P P'"
    using symmetric_point_construction by blast
  ultimately show ?thesis
    using ReflectL_def bet_col midpoint_bet perp_col1 by blast
qed

lemma l10_2_existence:
  "\<exists> P'. P' P Reflect A B"
  by (metis Reflect_def l10_2_existence_spec symmetric_point_construction)

lemma l10_4_spec:
  assumes "P P' ReflectL A B"
  shows "P' P ReflectL A B"
proof -
  obtain X where "X Midpoint P P' \<and> Col A B X"
    using ReflectL_def assms l7_2 by blast
  thus ?thesis
    using Perp_cases ReflectL_def assms by auto
qed

lemma l10_4:
  assumes "P P' Reflect A B"
  shows "P' P Reflect A B"
  using Reflect_def l7_2 Tarski_neutral_dimensionless_axioms assms l10_4_spec by fastforce

lemma l10_5:
  assumes "P' P Reflect A B" and
    "P'' P' Reflect A B"
  shows "P = P''"
  by (meson assms(1) assms(2) l10_2_uniqueness l10_4)

lemma l10_6_uniqueness:
  assumes "P P1 Reflect A B" and
    "P P2 Reflect A B"
  shows "P1 = P2"
  using assms(1) assms(2) l10_4 l10_5 by blast

lemma l10_6_uniqueness_spec:
  assumes "P P1 ReflectL A B" and
    "P P2 ReflectL A B"
  shows "P1 = P2"
  using assms(1) assms(2) l10_2_uniqueness_spec l10_4_spec by blast

lemma l10_6_existence_spec:
  assumes "A \<noteq> B"
  shows "\<exists> P. P' P ReflectL A B"
  using l10_2_existence_spec l10_4_spec by blast

lemma l10_6_existence:
  "\<exists> P. P' P Reflect A B"
  using l10_2_existence l10_4 by blast

lemma l10_7:
  assumes "P' P Reflect A B" and
    "Q' Q Reflect A B" and
    "P' = Q'"
  shows "P = Q"
  using assms(1) assms(2) assms(3) l10_6_uniqueness by blast

lemma l10_8:
  assumes "P P Reflect A B"
  shows "Col P A B"
  by (metis Col_perm assms col_trivial_2 ex_sym1 l10_6_uniqueness l7_3)

lemma col__refl:
  assumes "Col P A B"
  shows "P P ReflectL A B"
  using ReflectL_def assms col_permutation_1 l7_3_2 by blast

lemma is_image_col_cong:
  assumes "A \<noteq> B" and
    "P P' Reflect A B" and
    "Col A B X"
  shows "Cong P X P' X"
proof -
  have P1: "P P' ReflectL A B"
    using assms(1) assms(2) is_image_is_image_spec by blast
  obtain M0 where P2: "M0 Midpoint P' P \<and> Col A B M0"
    using P1 ReflectL_def by blast
  have "A B Perp P' P \<or> P' = P"
    using P1 ReflectL_def by auto
  moreover
  {
    assume S1: "A B Perp P' P"
    hence "A \<noteq> B \<and> P' \<noteq> P"
      using perp_distinct by blast
    have S2: "M0 = X \<longrightarrow> Cong P X P' X"
      using P2 cong_4312 midpoint_cong by blast
    {
      assume "M0 \<noteq> X"
      hence "M0 X Perp P' P"
        using P2 S1 assms(3) perp_col2 by blast
      hence "\<not> Col A B P \<and> Per P M0 X"
        by (metis Col_perm P2 S1 colx l8_2 midpoint_col midpoint_distinct_1 
            per_col perp_col1 perp_not_col2 perp_per_1)
      hence "Cong P X P' X"
        using P2 cong_commutativity l7_2 l8_2 per_double_cong by blast
    }
    hence "Cong P X P' X"
      using S2 by blast
  }
  hence "A B Perp P' P \<longrightarrow> Cong P X P' X" by blast
  moreover
  {
    assume "P = P'"
    hence "Cong P X P' X"
      by (simp add: cong_reflexivity)
  }
  ultimately show ?thesis by blast
qed

lemma is_image_spec_col_cong:
  assumes "P P' ReflectL A B" and
    "Col A B X"
  shows "Cong P X P' X"
  by (metis Col_def Reflect_def assms(1) assms(2) between_trivial col__refl 
      cong_reflexivity is_image_col_cong l10_6_uniqueness_spec)

lemma image_id:
  assumes "A \<noteq> B" and
    "Col A B T" and
    "T T' Reflect A B"
  shows "T = T'"
  using assms(1) assms(2) assms(3) cong_diff_4 is_image_col_cong by blast

lemma osym_not_col:
  assumes "P P' Reflect A B" and
    "\<not> Col A B P"
  shows "\<not> Col A B P'"
  using assms(1) assms(2) l10_4 local.image_id not_col_distincts by blast

lemma midpoint_preserves_image:
  assumes "A \<noteq> B" and
    "Col A B M" and
    "P P' Reflect A B" and
    "M Midpoint P Q" and
    "M Midpoint P' Q'"
  shows "Q Q' Reflect A B"
proof -
  obtain X where "X Midpoint P' P" and "Col A B X"
    using ReflectL_def assms(1) assms(3) is_image_is_image_spec by blast
  {
    assume "A B Perp P' P"
    obtain Y where "M Midpoint X Y"
      using symmetric_point_construction by blast
    have "Y Midpoint Q Q'"
    proof -
      have "X Midpoint P P'" 
        using \<open>X Midpoint P' P\<close> l7_2 by blast
      thus ?thesis 
        using assms(4) assms(5) symmetry_preserves_midpoint \<open>M Midpoint X Y\<close> by blast
    qed
    have "P \<noteq> P'" 
      using \<open>A B Perp P' P\<close> perp_distinct by blast
    hence "Q \<noteq> Q'"
      using l7_9 Tarski_neutral_dimensionless_axioms assms(4) assms(5) by fastforce
    have "Y Midpoint Q' Q \<and> Col A B Y"
      using assms(2) colx l7_2 midpoint_col midpoint_distinct_1 
      by (metis \<open>Col A B X\<close> \<open>M Midpoint X Y\<close> \<open>Y Midpoint Q Q'\<close>)
    have "A B Perp Q' Q \<or> Q = Q'"
    proof -
      have "Per M Y Q"
      proof -
        have "Y Midpoint Q Q'" 
          using \<open>Y Midpoint Q Q'\<close> by auto
        have "Cong M Q M Q'"
          using assms(1) assms(2) assms(3) assms(4) assms(5) cong_commutativity 
            is_image_col_cong l7_16 l7_3_2 by blast
        thus ?thesis 
          using Per_def \<open>Y Midpoint Q Q'\<close> by blast
      qed
      {
        have "X = Y \<longrightarrow> (A B Perp Q' Q \<or> Q = Q')"
          using Perp_cases assms(5) l7_3 l7_9_bis 
          by (metis \<open>A B Perp P' P\<close> \<open>M Midpoint X Y\<close> \<open>X Midpoint P' P\<close> assms(4))
        {
          assume "X \<noteq> Y"
          hence "Y PerpAt M Y Y Q"
            using midpoint_distinct_1 per_perp_in 
            by (metis \<open>M Midpoint X Y\<close> \<open>Per M Y Q\<close> \<open>Q \<noteq> Q'\<close> \<open>Y Midpoint Q' Q \<and> Col A B Y\<close>)
          hence "Y Y Perp Y Q \<or> M Y Perp Y Q" 
            by (simp add: perp_in_perp_bis)
          {
            have "Y Y Perp Y Q \<longrightarrow> A B Perp Q' Q \<or> Q = Q'"
              using perp_not_eq_1 by blast
            {
              assume "M Y Perp Y Q"
              have "Y Q Perp A B"
              proof cases
                assume "A = M"
                thus ?thesis 
                  using Perp_cases \<open>M Y Perp Y Q\<close> \<open>Y Midpoint Q' Q \<and> Col A B Y\<close> 
                    assms(1) not_col_permutation_5 perp_col1 by blast
              next
                assume "A \<noteq> M"
                thus ?thesis 
                  by (metis \<open>Y Midpoint Q' Q \<and> Col A B Y\<close> \<open>Y Y Perp Y Q \<or> M Y Perp Y Q\<close> 
                      assms(1) assms(2) col3 not_col_distincts perp_col0)
              qed
              have "A B Perp Q' Q \<or> Q = Q'"
                using midpoint_col not_col_distincts perp_col0 
                by (metis \<open>Y Midpoint Q Q'\<close> \<open>Y Q Perp A B\<close>)
            }
            hence "M Y Perp Y Q \<longrightarrow> A B Perp Q' Q \<or> Q = Q'" by blast
          }
          hence "A B Perp Q' Q \<or> Q = Q'"
            using perp_distinct \<open>Y Y Perp Y Q \<or> M Y Perp Y Q\<close> by blast
        }
        hence "X \<noteq> Y \<longrightarrow> (A B Perp Q' Q \<or> Q = Q')" by blast
      }
      thus ?thesis
        using Perp_cases assms(5) l7_3 l7_9_bis
        by (metis \<open>A B Perp P' P\<close> \<open>M Midpoint X Y\<close> \<open>Y Midpoint Q Q'\<close> assms(4))
    qed
    hence "Q Q' ReflectL A B" 
      using ReflectL_def \<open>Y Midpoint Q' Q \<and> Col A B Y\<close> by blast
  }
  moreover
  {
    assume "P = P'"
    hence "Q Q' ReflectL A B"
      using assms(2) assms(4) assms(5) col__refl col_permutation_2 colx midpoint_col 
        midpoint_distinct_3 symmetric_point_uniqueness by (metis \<open>Col A B X\<close> \<open>X Midpoint P' P\<close>)
  }
  ultimately show ?thesis
    using ReflectL_def assms(1) assms(3) is_image_is_image_spec by auto
qed

lemma image_in_is_image_spec:
  assumes "M ReflectLAt P P' A B"
  shows "P P' ReflectL A B"
proof -
  have P1: "M Midpoint P' P"
    using ReflectLAt_def assms by blast
  have P2: "Col A B M"
    using ReflectLAt_def assms by blast
  have "A B Perp P' P \<or> P' = P"
    using ReflectLAt_def assms by blast
  thus ?thesis using P1 P2
    using ReflectL_def by blast
qed

lemma image_in_gen_is_image:
  assumes "M ReflectAt P P' A B"
  shows "P P' Reflect A B"
  using ReflectAt_def Reflect_def assms image_in_is_image_spec by auto

lemma image_image_in:
  assumes "P \<noteq> P'" and
    "P P' ReflectL A B" and
    "Col A B M" and
    "Col P M P'"
  shows "M ReflectLAt P P' A B"
proof -
  obtain M' where P1: "M' Midpoint P' P \<and> Col A B M'"
    using ReflectL_def assms(2) by blast
  have Q1: "P M' Perp A B"
    by (metis Col_cases P1 Perp_perm ReflectL_def assms(1) assms(2) bet_col cong_diff_3 
        midpoint_bet midpoint_cong not_cong_4321 perp_col1)
  {
    assume R1: "A B Perp P' P"
    have R3: "P \<noteq> M'"
      using Q1 perp_not_eq_1 by auto
    have R4: "A B Perp P' P"
      by (simp add: R1)
    have R5: "Col P P' M'"
      using P1 midpoint_col not_col_permutation_3 by blast
    have R6: "M' Midpoint P' P"
      by (simp add: P1)
    have R7: "\<not> Col A B P"
      using assms(1) assms(2) col__refl col_permutation_2 l10_2_uniqueness_spec l10_4_spec 
      by blast
    have R8: "P \<noteq> P'"
      by (simp add: assms(1))
    have R9: "Col A B M'"
      by (simp add: P1)
    have R10: "Col A B M"
      by (simp add: assms(3))
    have R11: "Col P P' M'"
      by (simp add: R5)
    have R12: "Col P P' M"
      using Col_perm assms(4) by blast
    have "M = M'"
    proof cases
      assume S1: "A = M'"
      have "Per P M' A"
        by (simp add: S1 l8_5)
      thus ?thesis using l6_21 R8 R9 R10 R11 R12
        using R7 by blast
    next
      assume "A \<noteq> M'"
      thus ?thesis
        using R10 R12 R5 R7 R8 R9 l6_21 by blast
    qed
    hence "M Midpoint P' P"
      using R6 by blast
  }
  hence Q2: "A B Perp P' P \<longrightarrow> M Midpoint P' P" by blast
  have Q3: "P' = P \<longrightarrow> M Midpoint P' P"
    using assms(1) by auto
  have Q4: "A B Perp P' P \<or> P' = P"
    using ReflectL_def assms(2) by auto
  hence "M Midpoint P' P"
    using Q2 Q3 by blast
  thus ?thesis
    by (simp add: ReflectLAt_def Q4 assms(3))
qed

lemma image_in_col:
  assumes "Y ReflectLAt P P' A B"
  shows "Col P P' Y"
  using Col_perm ReflectLAt_def assms midpoint_col by blast

lemma is_image_spec_rev:
  assumes "P P' ReflectL A B"
  shows "P P' ReflectL B A"
proof -
  obtain M0 where P1: "M0 Midpoint P' P \<and> Col A B M0"
    using ReflectL_def assms by blast
  have P2: "Col B A M0"
    using Col_cases P1 by blast
  have "A B Perp P' P \<or> P' = P"
    using ReflectL_def assms by blast
  thus ?thesis
    using P1 P2 Perp_cases ReflectL_def by auto
qed

lemma is_image_rev:
  assumes "P P' Reflect A B"
  shows "P P' Reflect B A"
  using Reflect_def assms is_image_spec_rev by auto

lemma midpoint_preserves_per:
  assumes "Per A B C" and
    "M Midpoint A A1" and
    "M Midpoint B B1" and
    "M Midpoint C C1"
  shows "Per A1 B1 C1"
proof -
  obtain C' where P1: "B Midpoint C C' \<and> Cong A C A C'"
    using Per_def assms(1) by blast
  obtain C1' where P2: "M Midpoint C' C1'"
    using symmetric_point_construction by blast
  thus ?thesis
    by (meson P1 Per_def assms(2) assms(3) assms(4) l7_16 symmetry_preserves_midpoint)
qed

lemma col__image_spec:
  assumes "Col A B X"
  shows "X X ReflectL A B"
  by (simp add: assms col__refl col_permutation_2)

lemma image_triv:
  "A A Reflect A B"
  by (simp add: Reflect_def col__refl col_trivial_1 l7_3_2)

lemma cong_midpoint__image:
  assumes "Cong A X A Y" and
    "B Midpoint X Y"
  shows "Y X Reflect A B"
proof cases
  assume "A = B"
  thus ?thesis
    by (simp add: Reflect_def assms(2))
next
  assume S0: "A \<noteq> B"
  {
    assume S1: "X \<noteq> Y"
    hence "X Y Perp A B"
    proof -
      have T1: "B \<noteq> X"
        using S1 assms(2) midpoint_distinct_1 by blast
      have T2: "B \<noteq> Y"
        using S1 assms(2) midpoint_not_midpoint by blast
      have "Per A B X"
        using Per_def assms(1) assms(2) by auto
      thus ?thesis
        using S0 S1 T1 T2 assms(2) col_per_perp midpoint_col by auto
    qed
    hence "A B Perp X Y \<or> X = Y"
      using Perp_perm by blast
    hence "Y X Reflect A B"
      using ReflectL_def S0 assms(2) col_trivial_2 is_image_is_image_spec by blast
  }
  hence "X \<noteq> Y \<longrightarrow> Y X Reflect A B" by blast
  thus ?thesis
    using assms(2) image_triv is_image_rev l7_3 by blast
qed


lemma col_image_spec__eq:
  assumes "Col A B P" and
    "P P' ReflectL A B"
  shows "P = P'"
  using assms(1) assms(2) col__image_spec l10_2_uniqueness_spec l10_4_spec by blast

lemma image_spec_triv:
  "A A ReflectL B B"
  using col__image_spec not_col_distincts by blast

lemma image_spec__eq:
  assumes "P P' ReflectL A A"
  shows "P = P'"
  using assms col_image_spec__eq not_col_distincts by blast

lemma image__midpoint:
  assumes "P P' Reflect A A"
  shows "A Midpoint P' P"
  using Reflect_def assms by auto

lemma is_image_spec_dec:
  "A B ReflectL C D \<or> \<not> A B ReflectL C D"
  by simp

lemma l10_14:
  assumes "P \<noteq> P'" and
    "A \<noteq> B" and
    "P P' Reflect A B"
  shows "A B TS P P'"
proof -
  have P1: "P P' ReflectL A B"
    using assms(2) assms(3) is_image_is_image_spec by blast
  then obtain M0 where "M0 Midpoint P' P \<and> Col A B M0"
    using ReflectL_def by blast
  hence "A B Perp P' P \<longrightarrow> A B TS P P'"
    by (meson TS_def assms(1) assms(2) assms(3) between_symmetry col_permutation_2 
        image_id midpoint_bet osym_not_col)
  thus ?thesis
    using assms(1) P1 ReflectL_def by blast
qed

lemma l10_15:
  assumes "Col A B C" and
    "\<not> Col A B P"
  shows "\<exists> Q. A B Perp Q C \<and> A B OS P Q"
proof -
  have P1: "A \<noteq> B"
    using assms(2) col_trivial_1 by auto
  obtain X where P2: "A B TS P X"
    using assms(2) col_permutation_1 l9_10 by blast
  {
    assume Q1: "A = C"
    obtain Q where Q2: "\<exists> T. A B Perp Q A \<and> Col A B T \<and> Bet X T Q"
      using P1 l8_21 by blast
    then obtain T where "A B Perp Q A \<and> Col A B T \<and> Bet X T Q" by blast
    hence "A B TS Q X"
      by (meson P2 TS_def between_symmetry col_permutation_2 perp_not_col)
    hence Q5: "A B OS P Q"
      using P2 l9_8_1 by blast
    hence "\<exists> Q. A B Perp Q C \<and> A B OS P Q"
      using Q1 Q2 by blast
  }
  hence P3: "A = C \<longrightarrow> (\<exists> Q. A B Perp Q C \<and> A B OS P Q)" by blast
  {
    assume Q1: "A \<noteq> C"
    then obtain Q where Q2: "\<exists> T. C A Perp Q C \<and> Col C A T \<and> Bet X T Q"
      using l8_21 by presburger
    then obtain T where Q3: "C A Perp Q C \<and> Col C A T \<and> Bet X T Q" by blast
    have Q4: "A B Perp Q C"
      using NCol_perm P1 Q2 assms(1) col_trivial_2 perp_col2 by blast
    have "A B TS Q X"
    proof -
      have R1: "\<not> Col Q A B"
        using Col_perm P1 Q2 assms(1) col_trivial_2 colx perp_not_col by blast
      have R2: "\<not> Col X A B"
        using P2 TS_def by auto
      have R3: "Col T A B"
        by (metis Q1 Q3 assms(1) col_trivial_2 colx not_col_permutation_1)
      have "Bet Q T X"
        using Bet_cases Q3 by blast
      hence "\<exists> T. Col T A B \<and> Bet Q T X"
        using R3 by blast
      thus ?thesis using R1 R2
        by (simp add: TS_def)
    qed
    hence "A B OS P Q"
      using P2 l9_8_1 by blast
    hence "\<exists> Q. A B Perp Q C \<and> A B OS P Q"
      using Q4 by blast
  }
  thus ?thesis using P3 by blast
qed

lemma ex_per_cong:
  assumes "A \<noteq> B" and
    "X \<noteq> Y" and
    "Col A B C" and
    "\<not> Col A B D"
  shows "\<exists> P. Per P C A \<and> Cong P C X Y \<and> A B OS P D"
proof -
  obtain Q where P1: "A B Perp Q C \<and> A B OS D Q"
    using assms(3) assms(4) l10_15 by blast
  obtain P where P2: "C Out Q P \<and> Cong C P X Y"
    by (metis P1 assms(2) perp_not_eq_2 segment_construction_3)
  have P3: "Per P C A"
    using P1 P2 assms(3) col_trivial_3 l8_16_1 l8_3 out_col by blast
  have "A B OS P D"
    using P1 P2 assms(3) one_side_symmetry os_out_os by blast
  thus ?thesis
    using P2 P3 cong_left_commutativity by blast
qed

lemma exists_cong_per:
  "\<exists> C. Per A B C \<and> Cong B C X Y"
proof cases
  assume "A = B"
  thus ?thesis
    by (meson l8_5 Tarski_neutral_dimensionless_axioms l8_2 segment_construction)
next
  assume "A \<noteq> B"
  thus ?thesis
    by (metis Perp_perm bet_col between_trivial l8_16_1 l8_21 segment_construction)
qed

lemma upper_dim_implies_per2__col:
  assumes "upper_dim_axiom"
  shows "\<forall> A B C X. (Per A X C \<and> X \<noteq> C \<and> Per B X C) \<longrightarrow> Col A B X"
proof -
  {
    fix A B C X
    assume "Per A X C \<and> X \<noteq> C \<and> Per B X C"
    moreover then obtain C' where "X Midpoint C C'" and "Cong A C A C'"
      using Per_def by blast
    moreover obtain C'' where "X Midpoint C C''" and "Cong B C B C''" 
      using Per_def calculation(1) by blast
    moreover hence "C' = C''" 
      using \<open>X Midpoint C C''\<close> symmetric_point_uniqueness calculation(2) by auto
    have "C \<noteq> C'" 
      using calculation(1) calculation(2) l7_3 by blast
    moreover have "Cong B C B C'" 
      by (simp add: \<open>C' = C''\<close> calculation(5))
    moreover have "Cong X C X C'" 
      using calculation(2) cong_left_commutativity midpoint_cong by blast
    ultimately have "Col A B X" 
      using Col_def assms upper_dim_axiom_def by blast
  }
  thus ?thesis by blast
qed

lemma upper_dim_implies_col_perp2__col:
  assumes "upper_dim_axiom"
  shows "\<forall> A B X Y P. (Col A B P \<and> A B Perp X P \<and> P A Perp Y P) \<longrightarrow> Col Y X P"
proof -
  {
    fix A B X Y P
    assume H1: "Col A B P \<and> A B Perp X P \<and> P A Perp Y P"
    hence H2: "P \<noteq> A"
      using perp_not_eq_1 by blast
    have "Col Y X P"
    proof -
      have T1: "Per Y P A"
        using H1 l8_2 perp_per_1 by blast
      moreover have "Per X P A"
        using H1 col_trivial_3 l8_16_1 by blast
      thus ?thesis using T1 H2
        using assms upper_dim_implies_per2__col by blast
    qed
  }
  thus ?thesis by blast
qed

lemma upper_dim_implies_perp2__col:
  assumes "upper_dim_axiom"
  shows "\<forall> X Y Z A B. (X Y Perp A B \<and> X Z Perp A B) \<longrightarrow> Col X Y Z"
proof -
  {
    fix X Y Z A B
    assume H1: "X Y Perp A B \<and> X Z Perp A B"
    hence H1A: "X Y Perp A B" by blast
    have H1B: "X Z Perp A B" using H1 by blast
    obtain C where H2: "C PerpAt X Y A B"
      using H1 Perp_def by blast
    obtain C' where H3: "C' PerpAt X Z A B"
      using H1 Perp_def by blast
    have "Col X Y Z"
    proof cases
      assume H2: "Col A B X"
      {
        assume "X = A"
        hence "Col X Y Z" using upper_dim_implies_col_perp2__col
          by (metis H1 H2 Perp_cases assms col_permutation_1)
      }
      hence P1: "X = A \<longrightarrow> Col X Y Z" by blast
      {
        assume P2: "X \<noteq> A"
        hence P3: "A B Perp X Y" using perp_sym
          using H1 Perp_perm by blast
        have "Col A B X"
          by (simp add: H2)
        hence P4: "A X Perp X Y" using perp_col
          using P2 P3 by auto
        have P5: "A X Perp X Z"
          by (metis H1 H2 P2 Perp_perm col_trivial_3 perp_col0)
        have P6: "Col Y Z X"
        proof -
          have Q1: "upper_dim_axiom"
            by (simp add: assms)
          have Q2: "Per Y X A"
            using P4 Perp_cases perp_per_2 by blast
          have "Per Z X A"
            by (meson P5 Perp_cases Tarski_neutral_dimensionless_axioms perp_per_2)
          thus ?thesis using Q1 Q2 P2
            using upper_dim_implies_per2__col by blast
        qed
        hence "Col X Y Z"
          using Col_perm by blast
      }
      thus ?thesis
        using P1 by blast
    next
      assume T1: "\<not> Col A B X"
      obtain Y0 where Q3: "Y0 PerpAt X Y A B"
        using H1 Perp_def by blast
      obtain Z0 where Q4: "Z0 PerpAt X Z A B"
        using Perp_def H1 by blast
      have Q5: "X Y0 Perp A B"
      proof -
        have R1: "X \<noteq> Y0"
          using Q3 T1 perp_in_col by blast
        have R2: "X Y Perp A B"
          by (simp add: H1A)
        thus ?thesis using R1
          using Q3 perp_col perp_in_col by blast
      qed
      have "X Z0 Perp A B"
        by (metis H1B Q4 T1 perp_col perp_in_col)
      hence Q7: "Y0 = Z0"
        by (meson Q3 Q4 Q5 T1 Perp_perm Tarski_neutral_dimensionless_axioms 
            l8_18_uniqueness perp_in_col)
      have "Col X Y Z"
      proof -
        have "X \<noteq> Y0"
          using Q5 perp_distinct by auto
        moreover have "Col X Y0 Y"
          using Q3 not_col_permutation_5 perp_in_col by blast
        moreover have "Col X Y0 Z"
          using Q4 Q7 col_permutation_5 perp_in_col by blast
        ultimately show ?thesis
          using col_transitivity_1 by blast
      qed
      thus ?thesis 
        using l8_18_uniqueness Perp_cases T1 col_trivial_3 perp_col1 perp_in_col perp_not_col 
        by blast
    qed
  }
  thus ?thesis by blast
qed

lemma upper_dim_implies_not_two_sides_one_side_aux:
  assumes "upper_dim_axiom"
  shows "\<forall> A B X Y PX. (A \<noteq> B \<and> PX \<noteq> A \<and> A B Perp X PX \<and> Col A B PX \<and> 
                        \<not> Col X A B \<and> \<not> Col Y A B \<and> \<not> A B TS X Y) \<longrightarrow> A B OS X Y"
proof -
  {
    fix A B X Y PX
    assume "A \<noteq> B" and "PX \<noteq> A" and "A B Perp X PX" and 
      "Col A B PX" and "\<not> Col X A B" and 
      "\<not> Col Y A B" and "\<not> A B TS X Y"
    hence "\<exists> P T. PX A Perp P PX \<and> Col PX A T \<and> Bet Y T P"
      using l8_21 by blast
    then obtain P T where "PX A Perp P PX" and "Col PX A T" and "Bet Y T P"
      by blast
    have "Col P X PX" 
      using upper_dim_implies_col_perp2__col \<open>A B Perp X PX\<close> 
        \<open>Col A B PX\<close> \<open>PX A Perp P PX\<close> assms by blast
    have "\<not> Col P A B"
      using col_trivial_2 colx not_col_permutation_3 perp_not_col 
        \<open>A \<noteq> B\<close> \<open>Col A B PX\<close> \<open>PX A Perp P PX\<close> by blast
    have "PX A TS P Y"
    proof -
      have "\<not> Col P PX A" 
        using Col_cases \<open>PX A Perp P PX\<close> perp_not_col by blast
      moreover have "\<not> Col Y PX A" 
        by (meson \<open>Col A B PX\<close> \<open>PX \<noteq> A\<close> \<open>\<not> Col Y A B\<close> col_transitivity_2 
            not_col_permutation_1)
      moreover have "Col T PX A" 
        using Col_cases \<open>Col PX A T\<close> by blast
      moreover have "Bet P T Y" 
        using Bet_cases \<open>Bet Y T P\<close> by blast
      ultimately show ?thesis
        using TS_def by blast
    qed
    have "X \<noteq> PX" 
      using \<open>A B Perp X PX\<close> perp_not_eq_2 by auto
    have "P \<noteq> X" 
      using \<open>A \<noteq> B\<close> \<open>Col A B PX\<close> \<open>PX A TS P Y\<close> \<open>\<not> A B TS X Y\<close> 
        col_permutation_2 col_preserves_two_sides not_col_distincts by blast
    have "Bet X PX P \<or> PX Out X P \<or> \<not> Col X PX P"
      using l6_4_2 by blast
    hence "PX A TS P X" 
      using Out_cases TS_def bet__ts between_symmetry 
        col_permutation_1 col_preserves_two_sides col_trivial_2 l9_5 
      by (metis \<open>Col A B PX\<close> \<open>Col P X PX\<close> \<open>PX A TS P Y\<close> \<open>X \<noteq> PX\<close> 
          \<open>\<not> A B TS X Y\<close> \<open>\<not> Col P A B\<close>)
    hence "A B OS X Y" 
      using col2_os__os col_trivial_2 l9_2 l9_8_1 not_col_permutation_1 
      by (meson \<open>A \<noteq> B\<close> \<open>Col A B PX\<close> \<open>PX A TS P Y\<close>)
  }
  thus ?thesis by blast
qed

lemma upper_dim_implies_not_two_sides_one_side:
  assumes "upper_dim_axiom"
  shows "\<forall> A B X Y. (\<not> Col X A B \<and> \<not> Col Y A B \<and> \<not> A B TS X Y) \<longrightarrow> A B OS X Y"
proof -
  {
    fix A B X Y
    assume "\<not> Col X A B" and "\<not> Col Y A B" and "\<not> A B TS X Y"
    have "A \<noteq> B"
      using \<open>\<not> Col Y A B\<close> col_trivial_2 by blast
    obtain PX where "Col A B PX" and "A B Perp X PX"
      using Col_cases l8_18_existence \<open>\<not> Col X A B\<close> by blast
    have "A B OS X Y"
    proof cases
      assume "PX = A"
      have "B A OS X Y"
      proof -
        have "B A Perp X A" 
          using \<open>A B Perp X PX\<close> \<open>PX = A\<close> perp_left_comm by blast
        moreover have "Col B A A" 
          by (simp add: col_trivial_2)
        moreover have "\<not> Col X B A" 
          using Col_cases \<open>\<not> Col X A B\<close> by blast
        moreover have "\<not> Col Y B A"
          using Col_cases \<open>\<not> Col Y A B\<close> by blast
        moreover have "\<not> B A TS X Y" 
          using \<open>\<not> A B TS X Y\<close> invert_two_sides by blast
        ultimately show ?thesis 
          by (metis \<open>A \<noteq> B\<close> assms upper_dim_implies_not_two_sides_one_side_aux)
      qed
      thus ?thesis
        by (simp add: invert_one_side)
    next
      assume "PX \<noteq> A"
      thus ?thesis 
        using \<open>A B Perp X PX\<close> \<open>A \<noteq> B\<close> \<open>Col A B PX\<close> \<open>\<not> A B TS X Y\<close> 
          \<open>\<not> Col X A B\<close> \<open>\<not> Col Y A B\<close> assms 
          upper_dim_implies_not_two_sides_one_side_aux by blast
    qed
  }
  thus ?thesis by blast
qed

lemma upper_dim_implies_not_one_side_two_sides:
  assumes "upper_dim_axiom"
  shows "\<forall> A B X Y. (\<not> Col X A B \<and> \<not> Col Y A B \<and> \<not> A B OS X Y) \<longrightarrow> A B TS X Y"
  using assms upper_dim_implies_not_two_sides_one_side by blast

lemma upper_dim_implies_one_or_two_sides:
  assumes "upper_dim_axiom"
  shows "\<forall> A B X Y. (\<not> Col X A B \<and> \<not> Col Y A B) \<longrightarrow> (A B TS X Y \<or> A B OS X Y)"
  using assms upper_dim_implies_not_two_sides_one_side by blast

lemma upper_dim_implies_all_coplanar:
  assumes "upper_dim_axiom"
  shows "all_coplanar_axiom"
  using all_coplanar_axiom_def assms upper_dim_axiom_def by auto

lemma all_coplanar_implies_upper_dim:
  assumes "all_coplanar_axiom"
  shows "upper_dim_axiom"
  using all_coplanar_axiom_def assms upper_dim_axiom_def by auto

lemma all_coplanar_upper_dim:
  shows "all_coplanar_axiom \<longleftrightarrow> upper_dim_axiom"
  using all_coplanar_implies_upper_dim upper_dim_implies_all_coplanar by auto

lemma upper_dim_stab:
  shows "\<not> \<not> upper_dim_axiom \<longrightarrow> upper_dim_axiom" 
  by blast

lemma cop__cong_on_bissect:
  assumes "Coplanar A B X P" and
    "M Midpoint A B" and
    "M PerpAt A B P M" and
    "Cong X A X B"
  shows "Col M P X"
proof -
  have "X = M \<or> \<not> Col A B X \<and> M PerpAt X M A B"
    using assms(2) assms(3) assms(4) cong_commutativity cong_perp_or_mid 
      perp_in_distinct by blast
  moreover
  {
    assume "\<not> Col A B X" and "M PerpAt X M A B"
    hence "X M Perp A B"
      using perp_in_perp by blast
    have "A B Perp P M"
      using assms(3) perp_in_perp by blast
    have "Col M A B"
      by (simp add: assms(2) midpoint_col)
    hence "Col M P X" 
      using cop_perp2__col Perp_perm \<open>A B Perp P M\<close> \<open>X M Perp A B\<close> assms(1) 
        coplanar_perm_1 by blast
  }
  ultimately show ?thesis 
    using not_col_distincts by blast
qed

lemma cong_cop_mid_perp__col:
  assumes "Coplanar A B X P" and
    "Cong A X B X" and
    "M Midpoint A B" and
    "A B Perp P M"
  shows "Col M P X"
proof -
  have "M PerpAt A B P M"
    using Col_perm assms(3) assms(4) bet_col l8_15_1 midpoint_bet by blast
  thus ?thesis
    using assms(1) assms(2) assms(3) cop__cong_on_bissect not_cong_2143 by blast
qed

lemma cop_image_in2__col:
  assumes "Coplanar A B P Q" and
    "M ReflectLAt P P' A B" and
    "M ReflectLAt Q Q' A B"
  shows "Col M P Q"
proof -
  have P1: "P P' ReflectL A B"
    using assms(2) image_in_is_image_spec by auto
  hence P2: "A B Perp P' P \<or> P' = P"
    using ReflectL_def by auto
  have P3: "Q Q' ReflectL A B"
    using assms(3) image_in_is_image_spec by blast
  hence P4: "A B Perp Q' Q \<or> Q' = Q"
    using ReflectL_def by auto
  {
    assume S1: "A B Perp P' P \<and> A B Perp Q' Q"
    {
      assume T1: "A = M"
      have T2: "Per B A P"
        by (metis P1 Perp_perm S1 T1 assms(2) image_in_col is_image_is_image_spec l10_14 
            perp_col1 perp_distinct perp_per_1 ts_distincts)
      have T3: "Per B A Q"
        by (metis S1 T1 assms(3) image_in_col l8_5 perp_col1 perp_per_1 perp_right_comm)
      have T4: "Coplanar B P Q A"
        using assms(1) ncoplanar_perm_18 by blast
      have T5: "B \<noteq> A"
        using S1 perp_distinct by blast
      have T6: "Per P A B"
        by (simp add: T2 l8_2)
      have T7: "Per Q A B"
        using Per_cases T3 by blast
      hence "Col P Q A" using T4 T5 T6
        using cop_per2__col by blast
      hence "Col A P Q"
        using not_col_permutation_1 by blast
      hence "Col M P Q"
        using T1 by blast
    }
    hence S2: "A = M \<longrightarrow> Col M P Q" by blast
    {
      assume D0: "A \<noteq> M"
      have D1: "Per A M P"
      proof -
        have E1: "M Midpoint P P'"
          using ReflectLAt_def assms(2) l7_2 by blast
        have "Cong P A P' A"
          using P1 col_trivial_3 is_image_spec_col_cong by blast
        hence "Cong A P A P'"
          using Cong_perm by blast
        thus ?thesis
          using E1 Per_def by blast
      qed
      have D2: "Per A M Q"
      proof -
        have E2: "M Midpoint Q Q'"
          using ReflectLAt_def assms(3) l7_2 by blast
        have "Cong A Q A Q'"
          using P3 col_trivial_3 cong_commutativity is_image_spec_col_cong by blast
        thus ?thesis
          using E2 Per_def by blast
      qed
      have "Col P Q M"
      proof -
        have W1: "Coplanar P Q A B"
          using assms(1) ncoplanar_perm_16 by blast
        have W2: "A \<noteq> B"
          using S1 perp_distinct by blast
        have "Col A B M"
          using ReflectLAt_def assms(2) by blast
        hence "Coplanar P Q A M"
          using W1 W2 col2_cop__cop col_trivial_3 by blast
        hence V1: "Coplanar A P Q M"
          using ncoplanar_perm_8 by blast
        have V3: "Per P M A"
          by (simp add: D1 l8_2)
        have "Per Q M A"
          using D2 Per_perm by blast
        thus ?thesis
          using V1 D0 V3 cop_per2__col by blast
      qed
      hence "Col M P Q"
        using Col_perm by blast
    }
    hence "A \<noteq> M \<longrightarrow> Col M P Q" by blast
    hence "Col M P Q"
      using S2 by blast
  }
  hence P5: "(A B Perp P' P \<and> A B Perp Q' Q) \<longrightarrow> Col M P Q" by blast
  have P6: "(A B Perp P' P \<and> (Q' = Q)) \<longrightarrow> Col M P Q"
    using ReflectLAt_def assms(3) l7_3 not_col_distincts by blast
  have P7: "(P' = P \<and> A B Perp Q' Q) \<longrightarrow> Col M P Q"
    using ReflectLAt_def assms(2) l7_3 not_col_distincts by blast
  have "(P' = P \<and> Q' = Q) \<longrightarrow> Col M P Q"
    using ReflectLAt_def assms(3) col_trivial_3 l7_3 by blast
  thus ?thesis
    using P2 P4 P5 P6 P7 by blast
qed

lemma l10_10_spec:
  assumes "P' P ReflectL A B" and
    "Q' Q ReflectL A B"
  shows "Cong P Q P' Q'"
proof cases
  assume "A = B"
  thus ?thesis
    using assms(1) assms(2) cong_reflexivity image_spec__eq by blast
next
  assume H1: "A \<noteq> B"
  obtain X where P1: "X Midpoint P P' \<and> Col A B X"
    using ReflectL_def assms(1) by blast
  obtain Y where P2: "Y Midpoint Q Q' \<and> Col A B Y"
    using ReflectL_def assms(2) by blast
  obtain Z where P3: "Z Midpoint X Y"
    using midpoint_existence by blast
  have P4: "Col A B Z"
  proof cases
    assume "X = Y"
    thus ?thesis
      by (metis P2 P3 midpoint_distinct_3)
  next
    assume "X \<noteq> Y"
    thus ?thesis
      by (metis P1 P2 P3 l6_21 midpoint_col not_col_distincts)
  qed
  obtain R where P5: "Z Midpoint P R"
    using symmetric_point_construction by blast
  obtain R' where P6: "Z Midpoint P' R'"
    using symmetric_point_construction by blast
  have P7: "A B Perp P P' \<or> P = P'"
    using ReflectL_def assms(1) by auto
  have P8: "A B Perp Q Q' \<or> Q = Q'"
    using ReflectL_def assms(2) by blast
  {
    assume Q1: "A B Perp P P' \<and> A B Perp Q Q'"
    have Q2: "R R' ReflectL A B"
    proof -
      have "P P' Reflect A B"
        by (simp add: H1 assms(1) is_image_is_image_spec l10_4_spec)
      hence "R R' Reflect A B"
        using H1 P4 P5 P6 midpoint_preserves_image by blast
      thus ?thesis
        using H1 is_image_is_image_spec by blast
    qed
    have Q3: "R \<noteq> R'"
      using P5 P6 Q1 l7_9 perp_not_eq_2 by blast
    have Q4: "Y Midpoint R R'"
      using P1 P3 P5 P6 symmetry_preserves_midpoint by blast
    have Q5: "Cong Q' R' Q R"
      using P2 Q4 l7_13 by blast
    have Q6: "Cong P' Z P Z"
      using P4 assms(1) is_image_spec_col_cong by auto
    have Q7: "Cong Q' Z Q Z"
      using P4 assms(2) is_image_spec_col_cong by blast
    hence "Cong P Q P' Q'"
    proof -
      have S1: "Cong R Z R' Z"
        using P5 P6 Q6 cong_symmetry l7_16 l7_3_2 by blast
      have "R \<noteq> Z"
        using Q3 S1 cong_reverse_identity by blast
      thus ?thesis
        by (meson P5 P6 Q5 Q6 Q7 S1 between_symmetry five_segment midpoint_bet 
            not_cong_2143 not_cong_3412)
    qed
  }
  hence P9: "(A B Perp P P' \<and> A B Perp Q Q') \<longrightarrow> Cong P Q P' Q'" by blast
  have P10: "(A B Perp P P' \<and> Q = Q') \<longrightarrow> Cong P Q P' Q'"
    using P2 l7_3 Tarski_neutral_dimensionless_axioms assms(1) cong_symmetry 
      is_image_spec_col_cong by fastforce
  have P11: "(P = P' \<and> A B Perp Q Q') \<longrightarrow> Cong P Q P' Q'"
    using P1 l7_3 not_cong_4321 Tarski_neutral_dimensionless_axioms assms(2) 
      is_image_spec_col_cong by fastforce
  have "(P = P' \<and> Q = Q') \<longrightarrow> Cong P Q P' Q'"
    using cong_reflexivity by blast
  thus ?thesis
    using P10 P11 P7 P8 P9 by blast
qed

lemma l10_10:
  assumes "P' P Reflect A B" and
    "Q' Q Reflect A B"
  shows "Cong P Q P' Q'"
  using Reflect_def assms(1) assms(2) cong_4321 l10_10_spec l7_13 by auto

lemma image_preserves_bet:
  assumes "A A' ReflectL X Y" and
    "B B' ReflectL X Y" and
    "C C' ReflectL X Y" and
    "Bet A B C"
  shows "Bet A' B' C'"
proof -
  have "A B C Cong3 A' B' C'"
    using Cong3_def assms(1) assms(2) assms(3) l10_10_spec l10_4_spec by blast
  thus ?thesis
    using assms(4) l4_6 by blast
qed

lemma image_gen_preserves_bet:
  assumes "A A' Reflect X Y" and
    "B B' Reflect X Y" and
    "C C' Reflect X Y" and
    "Bet A B C"
  shows "Bet A' B' C'"
proof cases
  assume "X = Y"
  thus ?thesis 
    by (metis (full_types) assms(1) assms(2) assms(3) assms(4) image__midpoint l7_15 l7_2)
next
  assume "X \<noteq> Y"
  hence "A A' ReflectL X Y"
    using assms(1) is_image_is_image_spec by blast
  moreover have "B B' ReflectL X Y"
    using assms(2) is_image_is_image_spec \<open>X \<noteq> Y\<close> by blast
  moreover have "C C' ReflectL X Y"
    using assms(3) is_image_is_image_spec \<open>X \<noteq> Y\<close> by blast
  ultimately show ?thesis using image_preserves_bet
    using assms(4) by blast
qed

lemma image_preserves_col:
  assumes "A A' ReflectL X Y" and
    "B B' ReflectL X Y" and
    "C C' ReflectL X Y" and
    "Col A B C"
  shows "Col A' B' C'" using image_preserves_bet
  using Col_def assms(1) assms(2) assms(3) assms(4) by auto

lemma image_gen_preserves_col:
  assumes "A A' Reflect X Y" and
    "B B' Reflect X Y" and
    "C C' Reflect X Y" and
    "Col A B C"
  shows "Col A' B' C'"
  using Col_def assms(1) assms(2) assms(3) assms(4) image_gen_preserves_bet by auto

lemma image_gen_preserves_ncol:
  assumes "A A' Reflect X Y" and
    "B B' Reflect X Y" and
    "C C' Reflect X Y" and
    "\<not> Col A B C"
  shows "\<not> Col A' B' C'"
  using assms(1) assms(2) assms(3) assms(4)image_gen_preserves_col l10_4 by blast

lemma image_gen_preserves_inter:
  assumes "A A' Reflect X Y" and
    "B B' Reflect X Y" and
    "C C' Reflect X Y" and
    "D D' Reflect X Y" and
    "\<not> Col A B C" and
    "C \<noteq> D" and
    "Col A B I" and
    "Col C D I" and
    "Col A' B' I'" and
    "Col C' D' I'"
  shows "I I' Reflect X Y"
proof -
  obtain I0 where "I I0 Reflect X Y"
    using l10_6_existence by blast
  have "I0 = I'" 
  proof -
    have "\<not> Col A' B' C'" 
      by (meson assms(1) assms(2) assms(3) assms(5) image_gen_preserves_col l10_4)
    moreover have "C' \<noteq> D'" 
      using assms(3) assms(4) assms(6) l10_2_uniqueness by blast
    moreover have "Col A' B' I0" 
      using \<open>I I0 Reflect X Y\<close> assms(1) assms(2) assms(7) image_gen_preserves_col by auto
    moreover have "Col C' D' I0" 
      using \<open>I I0 Reflect X Y\<close> assms(3) assms(4) assms(8) image_gen_preserves_col by auto
    ultimately show ?thesis 
      using assms(10) assms(9) l6_21 by blast
  qed
  thus ?thesis 
    using \<open>I I0 Reflect X Y\<close> by blast
qed

lemma intersection_with_image_gen:
  assumes "A A' Reflect X Y" and
    "B B' Reflect X Y" and
    "\<not> Col A B A'" and
    "Col A B C" and
    "Col A' B' C"
  shows "Col C X Y"
proof -
  have "C C Reflect X Y" 
  proof -
    have "A' A Reflect X Y" 
      using assms(1) l10_4 by blast
    moreover have "B' B Reflect X Y" 
      by (simp add: assms(2) l10_4)
    moreover have "\<not> Col A' B' A" 
      by (meson image_gen_preserves_col assms(1) assms(2) assms(3) l10_4)
    moreover have "A' \<noteq> B'" 
      using calculation(3) col_trivial_1 by blast
    ultimately show ?thesis
      using image_gen_preserves_inter assms(1) assms(2) assms(3) assms(4) assms(5) by blast
  qed
  thus ?thesis 
    using l10_8 by blast
qed

lemma image_preserves_midpoint :
  assumes "A A' ReflectL X Y" and
    "B B' ReflectL X Y" and
    "C C' ReflectL X Y" and
    "A Midpoint B C"
  shows "A' Midpoint B' C'"
proof -
  have "Bet B' A' C'" using image_preserves_bet
    using assms(1) assms(2) assms(3) assms(4) midpoint_bet by auto
  moreover
  have "Cong B' A' A' C'" 
  proof -
    let ?C = "B"
    let ?D = "A"
    have "Cong B' A' ?C ?D" 
      using assms(1) assms(2) l10_10_spec by auto
    moreover have "Cong ?C ?D A' C'" 
    proof -
      let ?C' = "A"
      let ?D' = "C"
      have "Cong B A ?C' ?D'" 
        using Midpoint_def assms(4) by auto
      moreover have "Cong ?C' ?D' A' C'" 
        using assms(1) assms(3) cong_4321 l10_10_spec by blast
      ultimately show ?thesis 
        using cong_transitivity by blast
    qed
    ultimately show ?thesis 
      using cong_transitivity by blast
  qed
  ultimately show ?thesis
    by (simp add: Midpoint_def)
qed

lemma image_spec_preserves_per:
  assumes "A A' ReflectL X Y" and
    "B B' ReflectL X Y" and
    "C C' ReflectL X Y" and
    "Per A B C"
  shows "Per A' B' C'"
proof cases
  assume "X = Y"
  thus ?thesis
    using assms(1) assms(2) assms(3) assms(4) image_spec__eq by blast
next
  assume P1: "X \<noteq> Y"
  obtain C1 where P2: "B Midpoint C C1"
    using symmetric_point_construction by blast
  obtain C1' where P3: "C1 C1' ReflectL X Y"
    by (meson P1 l10_6_existence_spec)
  hence P4: "B' Midpoint C' C1'"
    using P2 assms(2) assms(3) image_preserves_midpoint by blast
  have "Cong A' C' A' C1'"
  proof -
    have Q1: "Cong A' C' A C"
      using assms(1) assms(3) l10_10_spec by auto
    have "Cong A C A' C1'"
      by (metis P2 P3 l10_10_spec assms(1) assms(4) cong_inner_transitivity cong_symmetry 
          per_double_cong)
    thus ?thesis
      using Q1 cong_transitivity by blast
  qed
  thus ?thesis
    using P4 Per_def by blast
qed

lemma image_preserves_per:
  assumes "A A' Reflect X Y" and
    "B B' Reflect X Y"and
    "C C' Reflect X Y" and
    "Per A B C"
  shows "Per A' B' C'"
proof cases
  assume "X = Y"
  thus ?thesis using midpoint_preserves_per
    using assms(1) assms(2) assms(3) assms(4) image__midpoint l7_2  by blast
next
  assume P1: "X \<noteq> Y"
  have P2: "X \<noteq> Y \<and> A A' ReflectL X Y"
    using P1 assms(1) is_image_is_image_spec by blast
  have P3: "X \<noteq> Y \<and> B B' ReflectL X Y"
    using P1 assms(2) is_image_is_image_spec by blast
  have P4: "X \<noteq> Y \<and> C C' ReflectL X Y"
    using P1 assms(3) is_image_is_image_spec by blast
  thus ?thesis using image_spec_preserves_per
    using P2 P3 assms(4) by blast
qed

lemma l10_12:
  assumes "Per A B C" and
    "Per A' B' C'" and
    "Cong A B A' B'" and
    "Cong B C B' C'"
  shows "Cong A C A' C'"
proof cases
  assume P1: "B = C"
  hence "B' = C'"
    using assms(4) cong_reverse_identity by blast
  thus ?thesis
    using P1 assms(3) by blast
next
  assume P2: "B \<noteq> C"
  have "Cong A C A' C'"
  proof cases
    assume "A = B"
    thus ?thesis
      using assms(3) assms(4) cong_diff_3 by force
  next
    assume P3: "A \<noteq> B"
    obtain X where P4: "X Midpoint B B'"
      using midpoint_existence by blast
    obtain A1 where P5: "X Midpoint A' A1"
      using Mid_perm symmetric_point_construction by blast
    obtain C1 where P6: "X Midpoint C' C1"
      using Mid_perm symmetric_point_construction by blast
    have Q1: "A' B' C' Cong3 A1 B C1"
      using Cong3_def P4 P5 P6 l7_13 l7_2 by blast
    have Q2: "Per A1 B C1"
      using assms(2)Q1 l8_10  by blast
    have Q3: "Cong A B A1 B"
      by (metis Cong3_def Q1 cong_symmetry assms(3) cong_inner_transitivity)
    have Q4: "Cong B C B C1"
      by (metis Cong3_def Q1 cong_symmetry assms(4) cong_inner_transitivity)
    obtain Y where P7: "Y Midpoint C C1"
      using midpoint_existence by auto
    hence R1: "C1 C Reflect B Y" using cong_midpoint__image
      using Q4 by blast
    obtain A2 where R2: "A1 A2 Reflect B Y"
      using l10_6_existence by blast
    have R3: "Cong C A2 C1 A1"
      using R1 R2 l10_10 by blast
    have R5: "B B Reflect B Y"
      using image_triv by blast
    have R6: "Per A2 B C" using image_preserves_per
      using Q2 R1 R2 image_triv by blast
    have R7: "Cong A B A2 B"
      using l10_10 Cong_perm Q3 R2 cong_transitivity image_triv by blast
    obtain Z where R7A: "Z Midpoint A A2"
      using midpoint_existence by blast
    have "Cong B A B A2"
      using Cong_perm R7 by blast
    hence T1: "A2 A Reflect B Z" using  R7A cong_midpoint__image
      by blast
    obtain C0 where T2: "B Midpoint C C0"
      using symmetric_point_construction by blast
    have T3: "Cong A C A C0"
      using T2 assms(1) per_double_cong by blast
    have T4: "Cong A2 C A2 C0"
      using R6 T2 per_double_cong by blast
    have T5: "C0 C Reflect B Z"
    proof -
      have "C0 C Reflect Z B"
      proof cases
        assume "A = A2"
        thus ?thesis
          by (metis R7A T2 T3 cong_midpoint__image midpoint_distinct_3)
      next
        assume "A \<noteq> A2"
        thus ?thesis using l4_17 cong_midpoint__image
          by (metis R7A T2 T3 T4 midpoint_col not_col_permutation_3)
      qed
      thus ?thesis
        using is_image_rev by blast
    qed
    have T6: "Cong A C A2 C0"
      using T1 T5 l10_10 by auto
    have R4: "Cong A C A2 C"
      by (metis T4 T6 cong_symmetry cong_inner_transitivity)
    hence Q5: "Cong A C A1 C1"
      by (meson R3 cong_inner_transitivity not_cong_3421)
    thus ?thesis
      using Cong3_def Q1 Q5 cong_symmetry cong_transitivity by blast
  qed
  thus ?thesis by blast
qed

lemma l10_16:
  assumes "\<not> Col A B C" and
    "\<not> Col A' B' P" and
    "Cong A B A' B'"
  shows "\<exists> C'. A B C Cong3 A' B' C' \<and> A' B' OS P C'"
proof cases
  assume "A = B"
  thus ?thesis
    using assms(1) not_col_distincts by auto
next
  assume P1: "A \<noteq> B"
  obtain X where P2: "Col A B X \<and> A B Perp C X"
    using assms(1) l8_18_existence by blast
  obtain X' where P3: "A B X Cong3 A' B' X'"
    using P2 assms(3) l4_14 by blast
  obtain Q where P4: "A' B' Perp Q X' \<and> A' B' OS P Q"
    using P2 P3 assms(2) l10_15 l4_13 by blast
  obtain C' where P5: "X' Out C' Q \<and> Cong X' C' X C"
    by (metis P2 P4 l6_11_existence perp_distinct)
  have P6: "Cong A C A' C'"
  proof cases
    assume "A = X"
    thus ?thesis
      by (metis Cong3_def P3 P5 cong_4321 cong_commutativity cong_diff_3)
  next
    assume "A \<noteq> X"
    have P7: "Per A X C"
      using P2 col_trivial_3 l8_16_1 l8_2 by blast
    have P8: "Per A' X' C'"
    proof -
      have "X' PerpAt A' X' X' C'"
      proof -
        have Z1: "A' X' Perp X' C'"
        proof -
          have W1: "X' \<noteq> C'"
            using P5 out_distinct by blast
          have W2: "X' Q Perp A' B'"
            using P4 Perp_perm by blast
          hence "X' C' Perp A' B'"
            by (metis P5 Perp_perm W1 col_trivial_3 not_col_permutation_5 out_col perp_col2_bis)
          thus ?thesis
            by (metis Cong3_def P2 P3 Perp_perm \<open>A \<noteq> X\<close> col_trivial_3 cong_identity 
                l4_13 perp_col2_bis)
        qed
        have Z2: "Col X' A' X'"
          using not_col_distincts by blast
        have "Col X' X' C'"
          by (simp add: col_trivial_1)
        thus ?thesis
          by (simp add: Z1 Z2 l8_14_2_1b_bis)
      qed
      thus ?thesis
        by (simp add: perp_in_per)
    qed
    have P9: "Cong A X A' X'"
      using Cong3_def P3 by auto
    have "Cong X C X' C'"
      using Cong_perm P5 by blast
    thus ?thesis using l10_12
      using P7 P8 P9 by blast
  qed
  have P10: "Cong B C B' C'"
  proof cases
    assume "B = X"
    thus ?thesis
      by (metis Cong3_def P3 P5 cong_4321 cong_commutativity cong_diff_3)
  next
    assume "B \<noteq> X"
    have Q1: "Per B X C"
      using P2 col_trivial_2 l8_16_1 l8_2 by blast
    have "X' PerpAt B' X' X' C'"
    proof -
      have Q2: "B' X' Perp X' C'"
      proof -
        have R1: "B' \<noteq> X'"
          using Cong3_def P3 \<open>B \<noteq> X\<close> cong_identity by blast
        have "X' C' Perp B' A'"
        proof -
          have S1: "X' \<noteq> C'"
            using Out_def P5 by blast
          have S2: "X' Q Perp B' A'"
            using P4 Perp_perm by blast
          have "Col X' Q C'"
            using Col_perm P5 out_col by blast
          thus ?thesis
            using S1 S2 perp_col by blast
        qed
        hence R2: "B' A' Perp X' C'"
          using Perp_perm by blast
        have R3: "Col B' A' X'"
          using Col_perm P2 P3 l4_13 by blast
        thus ?thesis
          using R1 R2 perp_col by blast
      qed
      have Q3: "Col X' B' X'"
        by (simp add: col_trivial_3)
      have "Col X' X' C'"
        by (simp add: col_trivial_1)
      thus ?thesis using l8_14_2_1b_bis
        using Q2 Q3 by blast
    qed
    hence Q2: "Per B' X' C'"
      by (simp add: perp_in_per)
    have Q3: "Cong B X B' X'"
      using Cong3_def P3 by blast
    have Q4: "Cong X C X' C'"
      using P5 not_cong_3412 by blast
    thus ?thesis
      using Q1 Q2 Q3 l10_12 by blast
  qed
  have P12: "A' B' OS C' Q \<longleftrightarrow> X' Out C' Q \<and> \<not> Col A' B' C'" using l9_19 l4_13
    by (meson P2 P3 P5 one_side_not_col123 out_one_side_1)
  hence P13: "A' B' OS C' Q" using l4_13
    by (meson P2 P3 P4 P5 l6_6 one_side_not_col124 out_one_side_1)
  thus ?thesis
    using Cong3_def P10 P4 P6 assms(3) one_side_symmetry one_side_transitivity by blast
qed

lemma cong_cop_image__col:
  assumes "P \<noteq> P'" and
    "P P' Reflect A B" and
    "Cong P X P' X" and
    "Coplanar A B P X"
  shows "Col A B X"
proof -
  have P1: "(A \<noteq> B \<and> P P' ReflectL A B) \<or> (A = B \<and> A Midpoint P' P)"
    by (metis assms(2) image__midpoint is_image_is_image_spec)
  {
    assume Q1: "A \<noteq> B \<and> P P' ReflectL A B"
    then obtain M where Q2: "M Midpoint P' P \<and> Col A B M"
      using ReflectL_def by blast
    have "Col A B X"
    proof cases
      assume R1: "A = M"
      have R2: "P A Perp A B"
      proof -
        have S1: "P \<noteq> A"
          using Q2 R1 assms(1) midpoint_distinct_2 by blast
        have S2: "P P' Perp A B"
          using Perp_perm Q1 ReflectL_def assms(1) by blast
        have "Col P P' A"
          using Q2 R1 midpoint_col not_col_permutation_3 by blast
        thus ?thesis using perp_col
          using S1 S2 by blast
      qed
      have R3: "Per P A B"
        by (simp add: R2 perp_comm perp_per_1)
      hence R3A: "Per B A P" using l8_2
        by blast
      have "A Midpoint P P' \<and> Cong X P X P'"
        using Cong_cases Q2 R1 assms(3) l7_2 by blast
      hence R4: "Per X A P"
        using Per_def by blast
      have R5: "Coplanar P B X A"
        using assms(4) ncoplanar_perm_20 by blast
      have "P \<noteq> A"
        using R2 perp_not_eq_1 by auto
      thus ?thesis using R4 R5 R3A
        using cop_per2__col not_col_permutation_1 by blast
    next
      assume T1: "A \<noteq> M"
      have T3: "P \<noteq> M"
        using Q2 assms(1) l7_3_2 sym_preserve_diff by blast
      have T2: "P M Perp M A"
      proof -
        have T4: "P P' Perp M A"
          using Perp_perm Q1 Q2 ReflectL_def T1 assms(1) col_trivial_3 perp_col0 by blast
        have "Col P P' M"
          by (simp add: Col_perm Q2 midpoint_col)
        thus ?thesis using T3 T4 perp_col by blast
      qed
      hence "M P Perp A M"
        using perp_comm by blast
      hence "M PerpAt M P A M"
        using perp_perp_in by blast
      hence "M PerpAt P M M A"
        by (simp add: perp_in_comm)
      hence U1: "Per P M A"
        by (simp add: perp_in_per)
      have U2: "Per X M P" using l7_2 cong_commutativity
        using Per_def Q2 assms(3) by blast
      have "Col A X M"
      proof -
        have W2: "Coplanar P A X M"
          by (meson Q1 Q2 assms(4) col_cop2__cop coplanar_perm_13 ncop_distincts)
        have "Per A M P"
          by (simp add: U1 l8_2)
        thus ?thesis using cop_per2__col
          using U2 T3 W2 by blast
      qed
      thus ?thesis
        using Q2 T1 col2__eq not_col_permutation_4 by blast
    qed
  }
  hence P2: "(A \<noteq> B \<and> P P' ReflectL A B) \<longrightarrow> Col A B X" by blast
  have "(A = B \<and> A Midpoint P' P) \<longrightarrow> Col A B X"
    using col_trivial_1 by blast
  thus ?thesis using P1 P2 by blast
qed

lemma cong_cop_per2_1:
  assumes "A \<noteq> B" and
    "Per A B X" and
    "Per A B Y" and
    "Cong B X B Y" and
    "Coplanar A B X Y"
  shows "X = Y \<or> B Midpoint X Y"
  by (meson Per_cases assms(1) assms(2) assms(3) assms(4) assms(5) cop_per2__col 
      coplanar_perm_3 l7_20_bis not_col_permutation_5)

lemma cong_cop_per2:
  assumes "A \<noteq> B" and
    "Per A B X" and
    "Per A B Y" and
    "Cong B X B Y" and
    "Coplanar A B X Y"
  shows "X = Y \<or> X Y ReflectL A B"
proof -
  have "X = Y \<or> B Midpoint X Y"
    using assms(1) assms(2) assms(3) assms(4) assms(5) cong_cop_per2_1 by blast
  thus ?thesis
    by (metis Mid_perm Per_def Reflect_def assms(1) assms(3) cong_midpoint__image 
        symmetric_point_uniqueness)
qed

lemma cong_cop_per2_gen:
  assumes "A \<noteq> B" and
    "Per A B X" and
    "Per A B Y" and
    "Cong B X B Y" and
    "Coplanar A B X Y"
  shows "X = Y \<or> X Y Reflect A B"
proof -
  have "X = Y \<or> B Midpoint X Y"
    using assms(1) assms(2) assms(3) assms(4) assms(5) cong_cop_per2_1 by blast
  thus ?thesis
    using assms(2) cong_midpoint__image l10_4 per_double_cong by blast
qed

lemma ex_perp_cop:
  assumes "A \<noteq> B"
  shows "\<exists> Q. A B Perp Q C \<and> Coplanar A B P Q"
proof -
  {
    assume "Col A B C \<and> Col A B P"
    hence "\<exists> Q. A B Perp Q C \<and> Coplanar A B P Q"
      using assms ex_ncol_cop l10_15 ncop__ncols by blast
  }
  hence T1: "(Col A B C \<and> Col A B P) \<longrightarrow>
(\<exists> Q. A B Perp Q C \<and> Coplanar A B P Q)" by blast
  {
    assume "\<not>Col A B C \<and> Col A B P"
    hence "\<exists> Q. A B Perp Q C \<and> Coplanar A B P Q"
      by (metis Perp_cases ncop__ncols not_col_distincts perp_exists)
  }
  hence T2: "(\<not>Col A B C \<and> Col A B P) \<longrightarrow>
(\<exists> Q. A B Perp Q C \<and> Coplanar A B P Q)" by blast
  {
    assume "Col A B C \<and> \<not>Col A B P"
    hence "\<exists> Q. A B Perp Q C \<and> Coplanar A B P Q"
      using l10_15 os__coplanar by blast
  }
  hence T3: "(Col A B C \<and> \<not>Col A B P) \<longrightarrow>
(\<exists> Q. A B Perp Q C \<and> Coplanar A B P Q)" by blast
  {
    assume "\<not>Col A B C \<and> \<not>Col A B P"
    hence "\<exists> Q. A B Perp Q C \<and> Coplanar A B P Q"
      using l8_18_existence ncop__ncols perp_right_comm by blast
  }
  hence "(\<not>Col A B C \<and> \<not>Col A B P) \<longrightarrow>
(\<exists> Q. A B Perp Q C \<and> Coplanar A B P Q)" by blast
  thus ?thesis using T1 T2 T3 by blast
qed

lemma hilbert_s_version_of_pasch_aux:
  assumes "Coplanar A B C P" and
    "\<not> Col A I P" and
    "\<not> Col B C P" and
    "Bet B I C" and
    "B \<noteq> I" and
    "I \<noteq> C" and
    "B \<noteq> C"
  shows "\<exists> X. Col I P X \<and> 
((Bet A X B \<and> A \<noteq> X \<and> X \<noteq> B \<and> A \<noteq> B) \<or> (Bet A X C \<and> A \<noteq> X \<and> X \<noteq> C \<and> A \<noteq> C))"
proof -
  have T1: "I P TS B C"
    using Col_perm assms(3) assms(4) assms(5) assms(6) bet__ts bet_col 
      col_transitivity_1 by blast
  have T2: "Coplanar A P B I"
    using assms(1) assms(4) bet_cop__cop coplanar_perm_6 ncoplanar_perm_9 by blast
  have T3: "I P TS A B \<or> I P TS A C"
    by (meson T1 T2 TS_def assms(2) cop_nos__ts coplanar_perm_21 l9_2 l9_8_2)
  have T4: "I P TS A B \<longrightarrow> (\<exists> X. Col I P X \<and>
((Bet A X B \<and> A \<noteq> X \<and> X \<noteq> B \<and> A \<noteq> B) \<or> (Bet A X C \<and> A \<noteq> X \<and> X \<noteq> C \<and> A \<noteq> C)))"
    by (metis TS_def not_col_permutation_2 ts_distincts)
  have "I P TS A C \<longrightarrow> (\<exists> X. Col I P X \<and>
((Bet A X B \<and> A \<noteq> X \<and> X \<noteq> B \<and> A \<noteq> B) \<or> (Bet A X C \<and> A \<noteq> X \<and> X \<noteq> C \<and> A \<noteq> C)))"
    by (metis TS_def not_col_permutation_2 ts_distincts)
  thus ?thesis using T3 T4 by blast
qed

lemma hilbert_s_version_of_pasch:
  assumes "Coplanar A B C P" and
    "\<not> Col C Q P" and
    "\<not> Col A B P" and
    "BetS A Q B"
  shows "\<exists> X. Col P Q X \<and> (BetS A X C \<or> BetS B X C)"
proof -
  obtain X where "Col Q P X \<and>
(Bet C X A \<and> C \<noteq> X \<and> X \<noteq> A \<and> C \<noteq> A \<or> Bet C X B \<and> C \<noteq> X \<and> X \<noteq> B \<and> C \<noteq> B)"
    using BetSEq assms(1) assms(2) assms(3) assms(4) coplanar_perm_12 
      hilbert_s_version_of_pasch_aux by fastforce
  thus ?thesis
    by (metis BetS_def Bet_cases Col_perm)
qed

lemma two_sides_cases:
  assumes "\<not> Col PO A B" and
    "PO P OS A B"
  shows  "PO A TS P B \<or> PO B TS P A"
  by (meson assms(1) assms(2) cop_nts__os l9_31 ncoplanar_perm_3 not_col_permutation_4 
      one_side_not_col124 one_side_symmetry os__coplanar)

lemma not_par_two_sides:
  assumes "C \<noteq> D" and
    "Col A B I" and
    "Col C D I" and
    "\<not> Col A B C"
  shows "\<exists> X Y. Col C D X \<and> Col C D Y \<and> A B TS X Y"
proof -
  obtain pp :: "'p \<Rightarrow> 'p \<Rightarrow> 'p" where
    f1: "\<forall>p pa. Bet p pa (pp p pa) \<and> pa \<noteq> (pp p pa)"
    by (meson point_construction_different)
  hence f2: "\<forall>p pa pb pc. (Col pc pb p \<or> \<not> Col pc pb (pp p pa)) \<or> \<not> Col pc pb pa"
    by (meson Col_def colx)
  have f3: "\<forall>p pa. Col pa p pa"
    by (meson Col_def between_trivial)
  have f4: "\<forall>p pa. Col pa p p"
    by (meson Col_def between_trivial)
  have f5: "Col I D C"
    by (meson Col_perm assms(3))
  have f6: "\<forall>p pa. Col (pp pa p) p pa"
    using f4 f3 f2 by blast
  hence f7: "\<forall>p pa. Col pa (pp pa p) p"
    by (meson Col_perm)
  hence f8: "\<forall>p pa pb pc. (pc pb TS p (pp p pa) \<or> Col pc pb p) \<or> \<not> Col pc pb pa"
    using f2 f1 by (meson l9_18)
  have "I = D \<or> Col D (pp D I) C"
    using f7 f5 f3 colx by blast
  hence "I = D \<or> Col C D (pp D I)"
    using Col_perm by blast
  thus ?thesis
    using f8 f6 f3 by (metis Col_perm assms(2) assms(4))
qed

lemma cop_not_par_other_side:
  assumes "C \<noteq> D" and
    "Col A B I" and
    "Col C D I" and
    "\<not> Col A B C" and
    "\<not> Col A B P" and
    "Coplanar A B C P"
  shows "\<exists> Q. Col C D Q \<and> A B TS P Q"
proof -
  obtain X Y where P1:"Col C D X \<and> Col C D Y \<and> A B TS X Y"
    using assms(1) assms(2) assms(3) assms(4) not_par_two_sides by blast
  hence "Coplanar C A B X"
    using Coplanar_def assms(1) assms(2) assms(3) col_transitivity_1 by blast
  hence "Coplanar A B P X"
    using assms(4) assms(6) col_permutation_3 coplanar_trans_1 ncoplanar_perm_2 
      ncoplanar_perm_6 by blast
  thus ?thesis
    by (meson P1 l9_8_2 TS_def assms(5) cop_nts__os not_col_permutation_2 one_side_symmetry)
qed

lemma cop_not_par_same_side:
  assumes "C \<noteq> D" and
    "Col A B I" and
    "Col C D I" and
    "\<not> Col A B C" and
    "\<not> Col A B P" and
    "Coplanar A B C P"
  shows "\<exists> Q. Col C D Q \<and> A B OS P Q"
proof -
  obtain X Y where P1: "Col C D X \<and> Col C D Y \<and> A B TS X Y"
    using assms(1) assms(2) assms(3) assms(4) not_par_two_sides by blast
  hence "Coplanar C A B X"
    using Coplanar_def assms(1) assms(2) assms(3) col_transitivity_1 by blast
  hence "Coplanar A B P X"
    using assms(4) assms(6) col_permutation_1 coplanar_perm_2 coplanar_trans_1 
      ncoplanar_perm_14 by blast
  thus ?thesis
    by (meson P1 TS_def assms(5) cop_nts__os l9_2 l9_8_1 not_col_permutation_2)
qed

lemma perp_bisect_equiv_defA:
  assumes "P Q PerpBisect A B"
  shows "P Q PerpBisectBis A B" 
proof -
  have P1: "A B ReflectL P Q \<and> A \<noteq> B"
    using Perp_bisect_def assms by auto 
  then obtain X where P2: "X Midpoint B A \<and> Col P Q X" 
    using ReflectL_def by auto 
  have "P Q Perp B A \<or> B = A" 
    using P1 ReflectL_def by auto
  have "X Midpoint A B"
    using P2 l7_2 by blast 
  moreover
  have "X PerpAt P Q A B"
    using P1 P2 \<open>P Q Perp B A \<or> B = A\<close> calculation col_permutation_2 l8_14_2_1b_bis 
      midpoint_col perp_right_comm by blast 
  ultimately
  show ?thesis
    using Perp_bisect_bis_def by blast 
qed

lemma perp_bisect_equiv_defB:
  assumes "P Q PerpBisectBis A B"
  shows "P Q PerpBisect A B" 
proof -
  obtain I where P1: "I PerpAt P Q A B \<and> I Midpoint A B"
    using Perp_bisect_bis_def assms by blast
  hence "A \<noteq> B"
    using perp_in_distinct by blast 
  have "P Q Perp B A \<or> B = A"
    using \<open>I PerpAt P Q A B \<and> I Midpoint A B\<close> perp_in_perp perp_right_comm by blast
  moreover
  have "I Midpoint B A"
    by (meson Mid_perm P1) 
  moreover
  have "Col P Q I" 
    using P1 by (meson perp_in_col Tarski_neutral_dimensionless_axioms) 
  ultimately
  have "A B ReflectL P Q \<and> A \<noteq> B" 
    using ReflectL_def \<open>A \<noteq> B\<close> by blast 
  thus ?thesis
    by (simp add: Perp_bisect_def) 
qed

lemma perp_bisect_equiv_def:
  shows "P Q PerpBisect A B \<longleftrightarrow> P Q PerpBisectBis A B"
  using perp_bisect_equiv_defA perp_bisect_equiv_defB by blast 

lemma perp_bisect_sym_1:
  assumes "P Q PerpBisect A B"
  shows "Q P PerpBisect A B" 
proof -
  have "P Q PerpBisectBis A B"
    by (simp add: assms perp_bisect_equiv_defA)
  then obtain x where "x PerpAt P Q A B \<and> x Midpoint A B"
    using Perp_bisect_bis_def by blast
  hence "Q P PerpBisectBis A B"
    using Perp_bisect_bis_def Perp_in_cases by blast 
  thus ?thesis
    by (simp add: perp_bisect_equiv_defB)
qed

lemma perp_bisect_sym_2:
  assumes "P Q PerpBisect A B"
  shows "P Q PerpBisect B A" 
proof -
  have "P Q PerpBisectBis A B"
    by (simp add: assms perp_bisect_equiv_defA)
  then obtain x where "x PerpAt P Q A B \<and> x Midpoint A B"
    using Perp_bisect_bis_def by blast
  hence "P Q PerpBisectBis B A"
    using Perp_bisect_bis_def l7_2 perp_in_right_comm by blast 
  thus ?thesis
    by (simp add: perp_bisect_equiv_defB)
qed

lemma perp_bisect_sym_3:
  assumes "A B PerpBisect C D"
  shows "B A PerpBisect D C"
  by (meson assms perp_bisect_sym_1 perp_bisect_sym_2)

lemma perp_bisect_perp:
  assumes "P Q PerpBisect A B"
  shows "P Q Perp A B" 
proof -
  have "P Q PerpBisectBis A B"
    by (simp add: assms perp_bisect_equiv_defA)
  then obtain x where "x PerpAt P Q A B \<and> x Midpoint A B"
    using Perp_bisect_bis_def by blast
  thus ?thesis
    using perp_in_perp by blast 
qed

lemma perp_bisect_cong_1:
  assumes "P Q PerpBisect A B"
  shows "Cong A P B P" 
proof -
  have "P Q PerpBisectBis A B"
    using assms perp_bisect_equiv_def by auto 
  then obtain I where "I PerpAt P Q A B \<and> I Midpoint A B"
    using Perp_bisect_bis_def by blast
  hence "Cong P A P B" 
    using per_double_cong perp_in_per_1 by blast 
  thus ?thesis
    using Cong_perm by blast 
qed

lemma perp_bisect_cong_2:
  assumes "P Q PerpBisect A B"
  shows "Cong A Q B Q"
  using assms perp_bisect_cong_1 perp_bisect_sym_1 by blast

lemma perp_bisect_cong2:
  assumes "P Q PerpBisect A B"
  shows "Cong A P B P \<and> Cong A Q B Q"
  using assms perp_bisect_cong_1 perp_bisect_cong_2 by blast 

lemma perp_bisect_cong:
  assumes (*"\<not> Col A B C" and
"pO A1 PerpBisect B C" and
"pO B1 PerpBisect A C" and
"pO C1 PerpBisect A B" and*)
    "Cong A pO B pO" and
    "Cong B pO C pO"
  shows "Cong A pO C pO"
  using assms(1) assms(2) cong_transitivity by blast 

lemma cong_cop_perp_bisect:
  assumes "P \<noteq> Q" and
    "A \<noteq> B" and
    "Coplanar P Q A B" and
    "Cong A P B P" and
    "Cong A Q B Q"
  shows "P Q PerpBisect A B" 
proof -
  obtain I where "I Midpoint A B"
    using midpoint_existence by blast
  moreover
  have "Bet A I B"
    by (simp add: calculation midpoint_bet) 
  have "Cong A I B I"
    using Cong_perm \<open>I Midpoint A B\<close> midpoint_cong by blast
  have "I PerpAt P Q A B" 
  proof -
    have "Per P I A"
      using Per_def assms(4) calculation not_cong_2143 by blast 
    have "A \<noteq> I"
      using \<open>Cong A I B I\<close> assms(2) cong_reverse_identity by blast
    hence "B \<noteq> I"
      using \<open>Cong A I B I\<close> cong_identity by blast 
    thus ?thesis 
    proof cases
      assume "P = I"
      thus ?thesis
        using assms(1) assms(2) assms(5) calculation cong_perp_or_mid perp_in_left_comm by blast 
    next
      assume "P \<noteq> I"
      have "Col I P Q" 
      proof -
        have "Coplanar A P Q I"
          using \<open>Bet A I B\<close> assms(3) bet_cop__cop ncoplanar_perm_8 by blast 
        moreover
        have "Per Q I A"
          using Per_def \<open>I Midpoint A B\<close> assms(5) not_cong_2143 by blast 
        ultimately
        show ?thesis
          using \<open>A \<noteq> I\<close> \<open>Per P I A\<close> col_permutation_2 cop_per2__col by blast 
      qed
      moreover
      have "Col I A B"
        using \<open>Bet A I B\<close> bet_col not_col_permutation_4 by blast 
      ultimately
      show ?thesis 
        by (metis assms(1) assms(2) perp_in_col_perp_in \<open>A \<noteq> I\<close> \<open>Bet A I B\<close> \<open>P \<noteq> I\<close> 
            \<open>Per P I A\<close> bet_col col_per_perp l8_15_1 l8_2 not_col_permutation_2) 
    qed
  qed
  ultimately 
  show ?thesis
    using Perp_bisect_bis_def perp_bisect_equiv_defB by blast 
qed

lemma cong_mid_perp_bisect:
  assumes "P \<noteq> Q" and
    "A \<noteq> B" and
    "Cong A P B P" and
    "Q Midpoint A B"
  shows "P Q PerpBisect A B"
  using Perp_bisect_bis_def assms(1) assms(2) assms(3) assms(4) cong_perp_or_mid 
    perp_bisect_equiv_defB by blast 

lemma perp_bisect_is_on_perp_bisect:
  assumes "P IsOnPerpBisect A B" and
    "P IsOnPerpBisect B C" 
  shows "P IsOnPerpBisect A C"
  by (metis Is_on_perp_bisect_def cong_3421 assms(1) assms(2) cong_transitivity) 

lemma perp_mid_perp_bisect:
  assumes "C Midpoint A B" and
    "C D Perp A B"
  shows "C D PerpBisect A B" 
proof -
  have "C PerpAt C D A B"
    using assms(1) assms(2) l8_14_2_1b_bis midpoint_col not_col_distincts by blast 
  hence "C D PerpBisectBis A B"
    using assms(1) Perp_bisect_bis_def by blast 
  thus ?thesis
    by (simp add: perp_bisect_equiv_defB) 
qed

lemma cong_cop2_perp_bisect_col:
  assumes "Coplanar A C D E" and
    "Coplanar B C D E" and
    "Cong C D C E" and
    "A B PerpBisect D E"
  shows "Col A B C" 
proof -
  have "Cong D A E A"
    using assms(4) perp_bisect_cong2 by blast
  have "A B PerpBisectBis D E"
    by (simp add: assms(4) perp_bisect_equiv_defA)
  then obtain F where P1: "F PerpAt A B D E \<and> F Midpoint D E"
    using Perp_bisect_bis_def by blast  
  hence "Bet D F E \<and> Cong D F F E"
    by (simp add: midpoint_bet midpoint_cong) 
  have "D \<noteq> E"
    using P1 perp_in_distinct by blast
  have "Col A B F \<and> Col D E F" using perp_in_col
    using P1 by blast
  hence "A B Perp D E"
    using P1 perp_in_perp by blast 
  {
    assume "A \<noteq> C"
    hence "A C Perp D E"
      using \<open>Cong D A E A\<close> \<open>D \<noteq> E\<close> assms(1) assms(3) cong_cop_perp_bisect not_cong_2143 
        perp_bisect_perp by blast 
    hence ?thesis
      using NCol_cases \<open>A B Perp D E\<close> assms(2) cop_perp2__col coplanar_perm_23 
        perp_right_comm by blast 
  }
  thus ?thesis
    using col_trivial_3 by blast 
qed

lemma perp_bisect_cop2_existence:
  assumes "A \<noteq> B"
  shows "\<exists> P Q. P Q PerpBisect A B \<and> Coplanar A B C P \<and> Coplanar A B C Q" 
proof -
  obtain M where P1: "M Midpoint A B"
    using midpoint_existence by blast
  have "Bet A M B"
    by (simp add: P1 midpoint_bet) 
  obtain Q where P2: "A B Perp Q M \<and> Coplanar A B C Q" 
    using ex_perp_cop assms(1) by blast
  have "A B ReflectL M Q" 
  proof -
    have "M Midpoint B A"
      using P1 l7_2 by blast 
    moreover
    have "Col M Q M"
      by (simp add: col_trivial_3) 
    moreover
    have "M Q Perp B A \<or> B = A"
      by (simp add: P2 Perp_perm Tarski_neutral_dimensionless_axioms) 
    ultimately
    show ?thesis 
      using ReflectL_def col_trivial_3 by blast  
  qed
  moreover
  have "Coplanar A B C M"
    using \<open>Bet A M B\<close> bet_col ncop__ncol ncoplanar_perm_4 by blast 
  ultimately
  show ?thesis 
    using assms(1) P2 Perp_bisect_def by blast 
qed

lemma perp_bisect_existence:
  assumes "A \<noteq> B"
  shows "\<exists> P Q. P Q PerpBisect A B"
proof -
  obtain P Q where "P Q PerpBisect A B \<and> Coplanar A B C P \<and> Coplanar A B C Q" 
    using perp_bisect_cop2_existence assms(1) by blast
  thus ?thesis by blast
qed

lemma perp_bisect_existence_cop:
  assumes "A \<noteq> B"
  shows "\<exists> P Q. P Q PerpBisect A B \<and> Coplanar A B C P \<and> Coplanar A B C Q" 
  using perp_bisect_cop2_existence assms(1) by blast

lemma l11_3:
  assumes "A B C CongA D E F"
  shows "\<exists> A' C' D' F'. B Out A' A \<and> B Out C C' \<and> 
E Out D' D \<and> E Out F F' \<and> 
A' B C' Cong3 D' E F'"
proof -
  obtain A' C' D' F' where "Bet B A A'" and "Cong A A' E D" and "Bet B C C'" and 
    "Cong C C' E F" and "Bet E D D'" and "Cong D D' B A" and 
    "Bet E F F'" and "Cong F F' B C" and "Cong A' C' D' F'" 
    using CongA_def assms by auto
  hence "A' B C' Cong3 D' E F'"
    by (meson Cong3_def between_symmetry l2_11_b not_cong_1243 not_cong_4312)
  thus ?thesis 
    by (metis CongA_def l6_6 \<open>Bet B A A'\<close> \<open>Bet B C C'\<close> \<open>Bet E D D'\<close> 
        \<open>Bet E F F'\<close> assms bet_out)
qed

lemma l11_aux:
  assumes "B Out A A'" and
    "E Out D D'" and
    "Cong B A' E D'" and
    "Bet B A A0" and
    "Bet E D D0" and
    "Cong A A0 E D" and
    "Cong D D0 B A"
  shows "Cong B A0 E D0 \<and> Cong A' A0 D' D0"
proof -
  have "Cong B A0 E D0"
    by (meson Bet_cases assms(4) assms(5) assms(6) assms(7) l2_11_b 
        not_cong_1243 not_cong_4312)
  moreover
  have "Bet E D D' \<or> Bet E D' D"
    using Out_def assms(2) by auto
  moreover
  have "Bet B A A' \<or> Bet B A' A"
    using Out_def assms(1) by auto
  {
    assume "Bet B A A' \<and> Bet E D' D"
    have "Cong A0 A' D0 D'" 
    proof -
      have "Bet B A' A0" 
      proof -
        have "Bet E D' D0" 
        proof -
          let ?C = "D"
          have "Bet E D' ?C" 
            using \<open>Bet B A A' \<and> Bet E D' D\<close> by auto
          moreover have "Bet E ?C D0" 
            by (simp add: assms(5))
          ultimately show ?thesis 
            using between_exchange4 by blast
        qed
        moreover have "B Out A' A0" 
          by (metis Out_def \<open>Bet B A A' \<and> Bet E D' D\<close> assms(1) assms(4) bet_out l5_1)
        ultimately show ?thesis 
          by (metis cong_preserves_bet  \<open>Cong B A0 E D0\<close> assms(3) cong_symmetry)
      qed
      hence "Bet A0 A' B" 
        using Bet_cases by blast
      moreover have "Bet D0 D' E" 
        using Bet_cases \<open>Bet B A A' \<and> Bet E D' D\<close> assms(5) between_exchange4 by blast
      ultimately show ?thesis 
        by (meson l4_3 \<open>Cong B A0 E D0\<close> assms(3) not_cong_2143)
    qed
    hence "Cong A' A0 D' D0" 
      using not_cong_2143 by blast
  }
  moreover
  {
    assume "Bet B A' A"
    have "E Out D D0"
      using assms(2) assms(5) bet_out out_diff1 by blast
    have "Cong A' A0 D' D0" 
      using assms(2) assms(3) assms(4) between_exchange4 cong_preserves_bet l4_3_1 l6_6 l6_7 
      by (meson \<open>Bet B A' A\<close> \<open>Cong B A0 E D0\<close> \<open>E Out D D0\<close>)
  }
  moreover
  have "Bet B A A' \<longrightarrow> Cong A' A0 D' D0" 
    by (metis (no_types, lifting) Out_cases \<open>Cong B A0 E D0\<close> assms(1) assms(2) assms(3) 
        assms(4) assms(5) bet_out l6_7 out_cong_cong out_diff1)
  have "Bet E D D' \<longrightarrow> Cong A' A0 D' D0" 
    using \<open>Bet B A A' \<longrightarrow> Cong A' A0 D' D0\<close> \<open>Bet B A A' \<or> Bet B A' A\<close> calculation(4) by blast
  moreover have "Bet E D' D \<longrightarrow> Cong A' A0 D' D0" 
    using \<open>Bet B A A' \<or> Bet B A' A\<close> \<open>Bet B A A' \<longrightarrow> Cong A' A0 D' D0\<close> calculation(4) by blast
  ultimately show ?thesis 
    using  \<open>Cong B A0 E D0\<close> by blast
qed

lemma l11_3_bis:
  assumes "\<exists> A' C' D' F'. 
(B Out A' A \<and> B Out C' C \<and> E Out D' D \<and> E Out F' F \<and> A' B C' Cong3 D' E F')"
  shows "A B C CongA D E F"
proof -
  obtain A' C' D' F' where "B Out A' A" and "B Out C' C" and "E Out D' D" and
    "E Out F' F" and "A' B C' Cong3 D' E F'"
    using assms by blast
  obtain A0 where "Bet B A A0" and "Cong A A0 E D"
    using segment_construction by blast
  obtain C0 where "Bet B C C0" and "Cong C C0 E F"
    using segment_construction by blast
  obtain D0 where "Bet E D D0" and "Cong D D0 B A"
    using segment_construction by blast
  obtain F0 where "Bet E F F0" and "Cong F F0 B C"
    using segment_construction by blast
  have "A \<noteq> B \<and> C \<noteq> B \<and> D \<noteq> E \<and> F \<noteq> E" 
    using assms out_diff2 by blast
  moreover
  have "Cong A0 C0 D0 F0"
  proof -
    have "Cong B A0 E D0 \<and> Cong A' A0 D' D0"
    proof -
      have "B Out A A'" 
        by (simp add: \<open>B Out A' A\<close> l6_6)
      moreover have "E Out D D'" 
        using Out_cases \<open>E Out D' D\<close> by auto
      moreover have "Cong B A' E D'" 
        using Cong3_def \<open>A' B C' Cong3 D' E F'\<close> not_cong_2143 by blast
      ultimately show ?thesis 
        using l11_aux  \<open>Bet B A A0\<close> \<open>Bet E D D0\<close> \<open>Cong A A0 E D\<close> \<open>Cong D D0 B A\<close> by blast
    qed
    have "Cong B C0 E F0 \<and> Cong C' C0 F' F0"
    proof -
      have "Cong B C0 E F0" 
        by (meson Bet_cases \<open>Bet B C C0\<close> \<open>Bet E F F0\<close> \<open>Cong C C0 E F\<close> \<open>Cong F F0 B C\<close> 
            l2_11_b not_cong_1243 not_cong_4321)
      moreover have "Cong C' C0 F' F0" 
        by (meson Cong3_def \<open>A' B C' Cong3 D' E F'\<close> \<open>B Out C' C\<close> \<open>Bet B C C0\<close> \<open>Bet E F F0\<close> 
            \<open>Cong C C0 E F\<close> \<open>Cong F F0 B C\<close> \<open>E Out F' F\<close> l11_aux l6_6)
      ultimately show ?thesis     
        by blast
    qed
    have "B A' A0 Cong3 E D' D0" 
      using Cong3_def \<open>A' B C' Cong3 D' E F'\<close> \<open>Cong B A0 E D0 \<and> Cong A' A0 D' D0\<close> 
        cong_3_swap by blast
    have "B C' C0 Cong3 E F' F0" 
      using Cong3_def \<open>A' B C' Cong3 D' E F'\<close> \<open>Cong B C0 E F0 \<and> Cong C' C0 F' F0\<close> by blast
    have "Cong C0 A' F0 D'"
    proof -
      have "B C' C0 A' FSC E F' F0 D'"
      proof -
        have "Col B C' C0" 
          by (metis Bet_cases Col_def Out_def \<open>B Out C' C\<close> \<open>Bet B C C0\<close> bet_out l6_7)
        moreover have "Cong B A' E D'" 
          using Cong3_def \<open>B A' A0 Cong3 E D' D0\<close> by presburger
        moreover have "Cong C' A' F' D'" 
          using Cong3_def \<open>A' B C' Cong3 D' E F'\<close> not_cong_2143 by blast
        ultimately show ?thesis 
          by (simp add: FSC_def \<open>B C' C0 Cong3 E F' F0\<close>)
      qed
      have "B \<noteq> C'" 
        using \<open>B Out C' C\<close> l6_3_1 by blast
      thus ?thesis 
        using \<open>B C' C0 A' FSC E F' F0 D'\<close> l4_16 by blast
    qed
    hence "B A' A0 C0 FSC E D' D0 F0" 
      using FSC_def  bet_out l6_7 not_cong_2143 out_col 
      by (meson \<open>A \<noteq> B \<and> C \<noteq> B \<and> D \<noteq> E \<and> F \<noteq> E\<close> \<open>B A' A0 Cong3 E D' D0\<close> \<open>B Out A' A\<close> 
          \<open>Bet B A A0\<close> \<open>Cong B C0 E F0 \<and> Cong C' C0 F' F0\<close>)
    moreover
    have "B \<noteq> A'" 
      using \<open>B Out A' A\<close> out_distinct by blast
    ultimately show ?thesis 
      using l4_16 by auto
  qed
  ultimately show ?thesis 
    using CongA_def \<open>Bet B A A0\<close> \<open>Bet B C C0\<close> \<open>Bet E D D0\<close> \<open>Bet E F F0\<close> \<open>Cong A A0 E D\<close> 
      \<open>Cong C C0 E F\<close> \<open>Cong D D0 B A\<close> \<open>Cong F F0 B C\<close> by auto
qed

lemma l11_4_1:
  assumes "A B C CongA D E F" and
    (*"A \<noteq> B" and "C \<noteq> B" and "D \<noteq> E" and "F \<noteq> E" and*)
    "B Out A' A" and
    "B Out C' C" and
    "E Out D' D" and
    "E Out F' F" and
    "Cong B A' E D'" and 
    "Cong B C' E F'"
  shows "Cong A' C' D' F'"
proof -
  obtain A0 C0 D0 F0 where "B Out A0 A" and "B Out C C0" and "E Out D0 D" and 
    "E Out F F0" and "A0 B C0 Cong3 D0 E F0"
    using assms(1) l11_3 by blast
  have "B Out A' A0" 
    using \<open>B Out A0 A\<close> assms(2) l6_6 l6_7 by blast
  have "E Out D' D0" 
    using Out_cases \<open>E Out D0 D\<close> assms(4) l6_7 by blast
  have "Cong A' A0 D' D0"
  proof -
    have "Cong B A0 E D0" 
      using Cong3_def Cong_cases \<open>A0 B C0 Cong3 D0 E F0\<close> by blast
    thus ?thesis 
      using \<open>B Out A' A0\<close> \<open>E Out D' D0\<close> assms(6) out_cong_cong by blast
  qed
  have "Cong A' C0 D' F0"
  proof -
    have "B A0 A' C0 FSC E D0 D' F0" 
      using Cong3_def Cong_perm FSC_def assms(6) not_col_permutation_5 out_col 
      by (meson \<open>A0 B C0 Cong3 D0 E F0\<close> \<open>B Out A' A0\<close> \<open>Cong A' A0 D' D0\<close>)
    thus ?thesis 
      using l4_16 Tarski_neutral_dimensionless_axioms out_diff2 \<open>B Out A' A0\<close> by blast
  qed
  have "B Out C' C0" 
    using \<open>B Out C C0\<close> assms(3) l6_7 by blast
  have "E Out F' F0" 
    by (meson l6_7 \<open>E Out F F0\<close> assms(5))
  hence "Cong C' C0 F' F0" 
    using Cong3_def out_cong_cong \<open>A0 B C0 Cong3 D0 E F0\<close> \<open>B Out C' C0\<close> assms(7) by fastforce
  hence "B C0 C' A' FSC E F0 F' D'" 
    by (meson Col_def Cong3_def FSC_def Out_def \<open>A0 B C0 Cong3 D0 E F0\<close> \<open>B Out C' C0\<close> 
        \<open>Cong A' C0 D' F0\<close> assms(6) assms(7) between_symmetry not_cong_2143)
  hence "Cong C' A' F' D'" 
    using \<open>B Out C' C0\<close> l4_16 out_diff2 by blast
  thus ?thesis 
    using not_cong_2143 by blast
qed

lemma l11_4_2:
  assumes "A \<noteq> B" and
    "C \<noteq> B" and
    "D \<noteq> E" and
    "F \<noteq> E" and
    "\<forall> A' C' D' F'. (B Out A' A \<and> B Out C' C \<and> E Out D' D \<and> E Out F' F \<and> Cong B A' E D' \<and> 
Cong B C' E F' \<longrightarrow> Cong A' C' D' F')"
  shows "A B C CongA D E F"
proof -
  obtain A' where "Bet B A A'" and "Cong A A' E D"
    using segment_construction by blast
  obtain C' where "Bet B C C'" and "Cong C C' E F"
    using segment_construction by blast
  obtain D' where "Bet E D D'" and "Cong D D' B A"
    using segment_construction by blast
  obtain F' where "Bet E F F'" and "Cong F F' B C"
    using segment_construction by blast
  have "Cong A' B D' E" 
    by (meson Bet_cases \<open>Bet B A A'\<close> \<open>Bet E D D'\<close> \<open>Cong A A' E D\<close> \<open>Cong D D' B A\<close> 
        l2_11_b not_cong_2134 not_cong_3421)
  have "Cong B C' E F'" 
    using between_symmetry cong_3421 cong_right_commutativity l2_11_b 
    by (meson \<open>Bet B C C'\<close> \<open>Bet E F F'\<close> \<open>Cong C C' E F\<close> \<open>Cong F F' B C\<close>)
  have "B Out A' A \<and> B Out C' C \<and> E Out D' D \<and> E Out F' F \<and> A' B C' Cong3 D' E F'"
  proof -
    have "B Out A' A" 
      using \<open>Bet B A A'\<close> assms(1) bet_out l6_6 by presburger
    moreover have "B Out C' C" 
      by (simp add: \<open>Bet B C C'\<close> assms(2) bet_out l6_6)
    moreover have "E Out D' D" 
      by (simp add: \<open>Bet E D D'\<close> assms(3) bet_out l6_6) 
    moreover have "E Out F' F" 
      by (simp add: \<open>Bet E F F'\<close> assms(4) bet_out l6_6)
    moreover have "A' B C' Cong3 D' E F'" 
      using Cong3_def Cong_cases \<open>Cong A' B D' E\<close> \<open>Cong B C' E F'\<close> assms(5) 
        calculation(1) calculation(2) calculation(3) calculation(4) by blast
    ultimately show ?thesis
      by blast
  qed
  thus ?thesis
    using l11_3_bis by blast
qed

lemma conga_refl:
  assumes "A \<noteq> B" and
    "C \<noteq> B"
  shows "A B C CongA A B C"
  by (meson CongA_def assms(1) assms(2) cong_reflexivity segment_construction)

lemma conga_sym:
  assumes "A B C CongA A' B' C'"
  shows "A' B' C' CongA A B C"
proof -
  obtain A0 C0 D0 F0 where "Bet B A A0" and "Cong A A0 B' A'" and "Bet B C C0" and
    "Cong C C0 B' C'" and "Bet B' A' D0" and "Cong A' D0 B A" and
    "Bet B' C' F0" and "Cong C' F0 B C" and "Cong A0 C0 D0 F0"
    using CongA_def assms by auto
  hence "\<exists>p pa pb pc. Bet B' A' p \<and> Cong A' p B A \<and> Bet B' C' pa \<and> Cong C' pa B C \<and>
Bet B A pb \<and> Cong A pb B' A' \<and>Bet B C pc \<and> Cong C pc B' C' \<and> 
Cong p pa pb pc"
    using cong_symmetry Tarski_neutral_dimensionless_axioms by blast
  thus ?thesis
    using CongA_def assms by auto
qed

lemma l11_10:
  assumes "A B C CongA D E F" and
    "B Out A' A" and
    "B Out C' C" and
    "E Out D' D" and
    "E Out F' F"
  shows "A' B C' CongA D' E F'"
proof -
  have "A' \<noteq> B"
    using assms(2) out_distinct by auto
  moreover have "C' \<noteq> B"
    using Out_def assms(3) by force
  moreover have "D' \<noteq> E"
    using Out_def assms(4) by blast
  moreover have "F' \<noteq> E"
    using assms(5) out_diff1 by auto
  moreover have "\<forall> A'0 C'0 D'0 F'0. (B Out A'0 A' \<and> B Out C'0 C' \<and> E Out D'0 D' \<and> 
E Out F'0 F' \<and> Cong B A'0 E D'0 \<and> Cong B C'0 E F'0) 
\<longrightarrow> Cong A'0 C'0 D'0 F'0"
    by (meson assms(1) assms(2) assms(3) assms(4) assms(5) l11_4_1 l6_7)
  ultimately show ?thesis 
    using l11_4_2 by blast
qed

lemma out2__conga:
  assumes "B Out A' A" and
    "B Out C' C"
  shows "A B C CongA A' B C'" 
proof -
  have "A B C CongA A B C" 
    using assms(1) assms(2) conga_refl out_diff2 by auto
  moreover have "B Out A A" 
    using assms(1) out_distinct out_trivial by blast
  moreover have "B Out C C" 
    using assms(2) out_distinct out_trivial by blast
  ultimately show ?thesis 
    using assms(1) assms(2) l11_10 by blast 
qed

lemma cong3_diff:
  assumes "A \<noteq> B" and
    "A B C Cong3 A' B' C'"
  shows "A' \<noteq> B'"
  using Cong3_def assms(1) assms(2) cong_diff by blast

lemma cong3_diff2:
  assumes "B \<noteq> C" and
    "A B C Cong3 A' B' C'"
  shows "B' \<noteq> C'"
  using Cong3_def assms(1) assms(2) cong_diff by blast

lemma cong3_conga:
  assumes "A \<noteq> B" and
    "C \<noteq> B" and
    "A B C Cong3 A' B' C'"
  shows "A B C CongA A' B' C'"
  by (metis assms(1) assms(2) assms(3) cong3_diff cong3_diff2 l11_3_bis out_trivial)

lemma cong3_conga2:
  assumes "A B C Cong3 A' B' C'" and
    "A B C CongA A'' B'' C''"
  shows "A' B' C' CongA A'' B'' C''"
proof -
  obtain A0 C0 A2 C2 where "Bet B A A0" and "Cong A A0 B'' A''" and "Bet B C C0" and
    "Cong C C0 B'' C''" and "Bet B'' A'' A2" and "Cong A'' A2 B A" and 
    "Bet B'' C'' C2" and "Cong C'' C2 B C" and "Cong A0 C0 A2 C2"
    using CongA_def assms(2) by auto
  obtain A1 where "Bet B' A' A1" and "Cong A' A1 B'' A''"
    using segment_construction by blast
  obtain C1 where "Bet B' C' C1" and "Cong C' C1 B'' C''"
    using segment_construction by blast
  have "Cong B'' A'' A' A1" 
    using Cong_cases \<open>Cong A' A1 B'' A''\<close> by blast
  hence "Cong A A0 A' A1"
    by (metis cong_transitivity \<open>Cong A A0 B'' A''\<close>)
  have "Cong B A0 B' A1" 
    using Cong3_def assms(1) cong_commutativity l2_11_b 
    by (meson \<open>Bet B A A0\<close> \<open>Bet B' A' A1\<close> \<open>Cong A A0 A' A1\<close>)
  have "Cong C C0 C' C1" 
    using cong_inner_transitivity cong_symmetry \<open>Cong C C0 B'' C''\<close> \<open>Cong C' C1 B'' C''\<close> by blast
  have "B A A0 C FSC B' A' A1 C'" 
    using FSC_def  Cong3_def assms(1) bet_col l4_3 \<open>Bet B A A0\<close> \<open>Bet B' A' A1\<close> 
      \<open>Cong A A0 A' A1\<close> \<open>Cong B A0 B' A1\<close> by presburger
  hence "Cong A0 C A1 C'" 
    using CongA_def assms(2) l4_16 by force
  hence "B C C0 A0 FSC B' C' C1 A1" 
    using Cong3_def FSC_def assms(1) bet_col cong_commutativity  \<open>Bet B C C0\<close> 
      \<open>Bet B' C' C1\<close> \<open>Cong B A0 B' A1\<close> \<open>Cong C C0 C' C1\<close> l2_11_b by presburger
  have "Cong B A B' A'" 
    using Cong3_def Cong_cases assms(1) by blast
  hence "Cong A'' A2 B' A'"
    using \<open>Cong A'' A2 B A\<close> cong_transitivity by blast
  have "Cong B C B' C'" 
    using Cong3_def assms(1) by blast
  hence "Cong C'' C2 B' C'"
    using \<open>Cong C'' C2 B C\<close> cong_transitivity by blast
  have "Cong C0 A0 C2 A2" 
    using \<open>Cong A0 C0 A2 C2\<close> not_cong_2143 by blast
  have "Cong C1 A1 A0 C0" 
    by (metis CongA_def l4_16 \<open>B C C0 A0 FSC B' C' C1 A1\<close> assms(2) cong_3421)
  hence "Cong A1 C1 A2 C2" 
    using Cong_cases \<open>Cong A0 C0 A2 C2\<close> cong_transitivity by blast
  thus ?thesis 
    by (metis CongA_def assms(1) assms(2) cong3_diff cong3_diff2 \<open>Bet B' A' A1\<close> 
        \<open>Bet B' C' C1\<close> \<open>Bet B'' A'' A2\<close> \<open>Bet B'' C'' C2\<close> \<open>Cong A' A1 B'' A''\<close> 
        \<open>Cong A'' A2 B' A'\<close> \<open>Cong C' C1 B'' C''\<close> \<open>Cong C'' C2 B' C'\<close>)
qed

lemma conga_diff1:
  assumes "A B C CongA A' B' C'"
  shows "A \<noteq> B"
  using CongA_def assms by blast

lemma conga_diff2:
  assumes "A B C CongA A' B' C'"
  shows "C \<noteq> B"
  using CongA_def assms by blast

lemma conga_diff45:
  assumes "A B C CongA A' B' C'"
  shows "A' \<noteq> B'"
  using CongA_def assms by blast

lemma conga_diff56:
  assumes "A B C CongA A' B' C'"
  shows "C' \<noteq> B'"
  using CongA_def assms by blast

lemma conga_trans:
  assumes "A B C CongA A' B' C'" and
    "A' B' C' CongA A'' B'' C''"
  shows "A B C CongA A'' B'' C''"
proof -
  obtain A0 C0 A1 C1 where "Bet B A A0" and "Cong A A0 B' A'" and "Bet B C C0" and 
    "Cong C C0 B' C'" and "Bet B' A' A1" and "Cong A' A1 B A" and
    "Bet B' C' C1" and "Cong C' C1 B C" and "Cong A0 C0 A1 C1"
    using CongA_def assms(1) by auto
  have "A''\<noteq> B''"  
    using CongA_def assms(2) by auto
  have "C'' \<noteq> B''"
    using CongA_def assms(2) by auto
  have "A1 B' C1 CongA A'' B'' C''"
  proof -
    have "B' Out A1 A'" 
      using Out_def assms(2) bet_neq12__neq conga_diff1 \<open>Bet B' A' A1\<close> by metis
    moreover have "B' Out C1 C'" 
      using Out_def assms(1) bet_neq12__neq conga_diff56 \<open>Bet B' C' C1\<close> by metis
    moreover have "B'' Out A'' A''" 
      by (simp add: \<open>A'' \<noteq> B''\<close> out_trivial)
    moreover have "B'' Out C'' C''" 
      by (simp add: \<open>C'' \<noteq> B''\<close> out_trivial)
    ultimately show ?thesis 
      using assms(2) l11_10 by blast
  qed
  have "A0 B C0 CongA A' B' C'" 
    using Out_cases conga_diff1 conga_diff2 conga_diff45 assms(1) bet_out 
      conga_diff56 l11_10 l5_3
    by (meson \<open>Bet B A A0\<close> \<open>Bet B C C0\<close> \<open>Bet B' A' A1\<close> \<open>Bet B' C' C1\<close>)
  have "Cong B A0 B' A1" 
    using between_symmetry cong_3421 l2_11_b not_cong_1243 
    by (meson \<open>Bet B A A0\<close> \<open>Bet B' A' A1\<close> \<open>Cong A A0 B' A'\<close> \<open>Cong A' A1 B A\<close>)
  have "Cong B C0 B' C1" 
    using between_symmetry cong_3421 l2_11_b not_cong_1243 
    by (meson \<open>Bet B C C0\<close> \<open>Bet B' C' C1\<close> \<open>Cong C C0 B' C'\<close> \<open>Cong C' C1 B C\<close>)
  have "A0 B C0 Cong3 A1 B' C1" 
    using Cong3_def \<open>Cong A0 C0 A1 C1\<close> \<open>Cong B A0 B' A1\<close> \<open>Cong B C0 B' C1\<close> 
      not_cong_2143 by blast
  hence "A0 B C0 CongA A'' B'' C''" 
    by (meson cong3_symmetry \<open>A1 B' C1 CongA A'' B'' C''\<close> cong3_conga2)
  thus ?thesis 
  proof -
    have "B Out A A0" 
      using CongA_def assms(1) bet_out \<open>Bet B A A0\<close> by presburger
    moreover have "B Out C C0" 
      using CongA_def assms(1) bet_out \<open>Bet B C C0\<close> by presburger
    moreover have "B'' Out A'' A''" 
      by (simp add: \<open>A'' \<noteq> B''\<close> out_trivial)
    moreover have "B'' Out C'' C''" 
      using \<open>C'' \<noteq> B''\<close> out_trivial by auto
    ultimately show ?thesis 
      using \<open>A0 B C0 CongA A'' B'' C''\<close>  l11_10 by blast
  qed
qed

lemma conga_pseudo_refl:
  assumes "A \<noteq> B" and
    "C \<noteq> B"
  shows "A B C CongA C B A"
  by (meson CongA_def assms(1) assms(2) between_trivial cong_pseudo_reflexivity 
      segment_construction)

lemma conga_trivial_1:
  assumes "A \<noteq> B" and
    "C \<noteq> D"
  shows "A B A CongA C D C"
  by (meson CongA_def assms(1) assms(2) cong_trivial_identity segment_construction)

lemma l11_13:
  assumes "A B C CongA D E F" and
    "Bet A B A'" and
    "A'\<noteq> B" and
    "Bet D E D'" and
    "D'\<noteq> E"
  shows "A' B C CongA D' E F"
proof -
  obtain A'' C'' D'' F'' where "Bet B A A''" and "Cong A A'' E D" and "Bet B C C''" and
    "Cong C C'' E F" and "Bet E D D''" and "Cong D D'' B A" and "Bet E F F''" and 
    "Cong F F'' B C" and "Cong A'' C'' D'' F''"
    using CongA_def assms(1) by auto
  obtain A0 where "Bet B A' A0" and "Cong A' A0 E D'"
    using segment_construction by blast
  obtain D0 where "Bet E D' D0" and "Cong D' D0 B A'"
    using segment_construction by blast
  have "Cong A0 C'' D0 F''"
  proof -
    have "A'' B A0 C'' OFSC D'' E D0 F''"
    proof -
      have "Bet A'' B A0"
      proof -
        have "Bet A'' A B" 
          by (simp add: \<open>Bet B A A''\<close> between_symmetry)
        have "Bet A B A0" 
          using \<open>Bet B A' A0\<close> assms(2) assms(3) outer_transitivity_between by blast
        have "A \<noteq> B"
          using CongA_def assms(1) by blast
        thus ?thesis 
          using \<open>Bet A B A0\<close> \<open>Bet A'' A B\<close> outer_transitivity_between2 by blast
      qed
      moreover have "Bet D'' E D0" 
        using Bet_perm outer_transitivity_between CongA_def 
        by (metis \<open>Bet E D D''\<close> \<open>Bet E D' D0\<close> assms(1) assms(4) assms(5))
      moreover have "Cong A'' B D'' E" 
        using between_symmetry cong_3421 cong_left_commutativity l2_11_b 
        by (meson \<open>Bet B A A''\<close> \<open>Bet E D D''\<close> \<open>Cong A A'' E D\<close> \<open>Cong D D'' B A\<close>)
      moreover have "Cong B A0 E D0" 
        using between_symmetry cong_3421 cong_right_commutativity l2_11_b 
        by (meson \<open>Bet B A' A0\<close> \<open>Bet E D' D0\<close> \<open>Cong A' A0 E D'\<close> \<open>Cong D' D0 B A'\<close>)
      moreover have "Cong B C'' E F''"
        using between_symmetry cong_3421 cong_right_commutativity l2_11_b 
        by (meson \<open>Bet B C C''\<close> \<open>Bet E F F''\<close> \<open>Cong C C'' E F\<close> \<open>Cong F F'' B C\<close>)
      ultimately show ?thesis 
        by (simp add: OFSC_def \<open>Cong A'' C'' D'' F''\<close> )
    qed
    have "A'' \<noteq> B" 
      using \<open>Bet B A A''\<close> assms(1) bet_neq23__neq conga_diff1 by blast
    thus ?thesis 
      using \<open>A'' B A0 C'' OFSC D'' E D0 F''\<close> five_segment_with_def by auto
  qed
  thus ?thesis 
    using CongA_def \<open>Bet B A' A0\<close> \<open>Bet B C C''\<close> \<open>Bet E D' D0\<close> \<open>Bet E F F''\<close> 
      \<open>Cong A' A0 E D'\<close> \<open>Cong C C'' E F\<close> \<open>Cong D' D0 B A'\<close> \<open>Cong F F'' B C\<close> 
      assms(1) assms(3) assms(5) by auto
qed

lemma conga_right_comm:
  assumes "A B C CongA D E F"
  shows "A B C CongA F E D"
  by (metis conga_diff45 conga_trans assms conga_diff56 conga_pseudo_refl)

lemma conga_left_comm:
  assumes "A B C CongA D E F"
  shows "C B A CongA D E F"
  by (meson assms conga_right_comm conga_sym)

lemma conga_comm:
  assumes "A B C CongA D E F"
  shows "C B A CongA F E D"
  by (meson conga_left_comm conga_right_comm Tarski_neutral_dimensionless_axioms assms)

lemma conga_line:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "A' \<noteq> B'" and
    "B' \<noteq> C'"
    and "Bet A B C" and
    "Bet A' B' C'"
  shows "A B C CongA A' B' C'"
  by (metis Bet_cases assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) 
      conga_trivial_1 l11_13)

lemma l11_14:
  assumes "Bet A B A'" and
    "A \<noteq> B" and
    "A' \<noteq> B" and
    "Bet C B C'" and
    "B \<noteq> C" and
    "B \<noteq> C'"
  shows "A B C CongA A' B C'"
  by (metis Bet_perm assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) 
      conga_pseudo_refl conga_right_comm l11_13)

lemma l11_16:
  assumes "Per A B C" and
    "A \<noteq> B" and
    "C \<noteq> B" and
    "Per A' B' C'" and
    "A'\<noteq> B'" and
    "C'\<noteq> B'"
  shows "A B C CongA A' B' C'"
proof -
  obtain C0 where "Bet B C C0" and "Cong C C0 B' C'"
    using segment_construction by blast
  obtain C1 where "Bet B' C' C1" and "Cong C' C1 B C"
    using segment_construction by blast
  obtain A0 where "Bet B A A0" and "Cong A A0 B' A'"
    using segment_construction by blast
  obtain A1 where "Bet B' A' A1" and "Cong A' A1 B A"
    using segment_construction by blast
  have "Cong A0 C0 A1 C1"
  proof -
    have "Per A0 B C0"
      using l8_3 assms(1) assms(2) assms(3) bet_col per_col 
      by (metis \<open>Bet B A A0\<close> \<open>Bet B C C0\<close>)
    moreover have "Per A1 B' C1"
      using l8_3 assms(4) assms(5) assms(6) bet_col per_col 
      by (metis \<open>Bet B' A' A1\<close> \<open>Bet B' C' C1\<close>)
    moreover have "Cong A0 B A1 B'"
      using between_symmetry cong_3421 cong_left_commutativity l2_11_b 
      by (meson \<open>Bet B A A0\<close> \<open>Bet B' A' A1\<close> \<open>Cong A A0 B' A'\<close> \<open>Cong A' A1 B A\<close>)
    moreover have "Cong B C0 B' C1"
      using between_symmetry cong_3421 l2_11_b not_cong_1243 
      by (meson \<open>Bet B C C0\<close> \<open>Bet B' C' C1\<close> \<open>Cong C C0 B' C'\<close> \<open>Cong C' C1 B C\<close>)
    ultimately show ?thesis 
      using l10_12 by auto
  qed
  thus ?thesis 
    using CongA_def \<open>Bet B A A0\<close> \<open>Bet B C C0\<close> \<open>Bet B' A' A1\<close> \<open>Bet B' C' C1\<close> 
      \<open>Cong A A0 B' A'\<close> \<open>Cong A' A1 B A\<close> \<open>Cong C C0 B' C'\<close> \<open>Cong C' C1 B C\<close> 
      assms(2) assms(3) assms(5) assms(6) by auto
qed

lemma l11_17:
  assumes "Per A B C" and
    "A B C CongA A' B' C'"
  shows "Per A' B' C'"
proof -
  obtain A0 C0 A1 C1 where "Bet B A A0" and "Cong A A0 B' A'" and "Bet B C C0" and
    "Cong C C0 B' C'" and "Bet B' A' A1" and "Cong A' A1 B A" and "Bet B' C' C1" and
    "Cong C' C1 B C" and "Cong A0 C0 A1 C1"
    using CongA_def assms(2) by auto
  have "Per A0 B C0"
  proof -
    have "B \<noteq> C"
      using assms(2) conga_diff2 by blast
    moreover have "Per A0 B C"
      using l8_2 assms(1) assms(2) bet_col conga_diff1 per_col 
      by (metis \<open>Bet B A A0\<close>)
    moreover have "Col B C C0" 
      by (simp add: \<open>Bet B C C0\<close> bet_col)
    ultimately show ?thesis 
      using per_col by blast
  qed
  have "Per A1 B' C1"
  proof -
    have "A0 B C0 Cong3 A1 B' C1"
    proof -
      have "Cong A0 B A1 B'" 
        using Bet_cases Cong3_def l2_11_b not_cong_2134 not_cong_3421 
        by (meson \<open>Bet B A A0\<close> \<open>Bet B' A' A1\<close> \<open>Cong A A0 B' A'\<close> \<open>Cong A' A1 B A\<close>)
      moreover have "Cong B C0 B' C1" 
        by (meson \<open>Bet B C C0\<close> \<open>Bet B' C' C1\<close> \<open>Cong C C0 B' C'\<close> \<open>Cong C' C1 B C\<close> 
            between_symmetry cong_3421 l2_11_b not_cong_1243)
      ultimately show ?thesis 
        by (simp add: Cong3_def \<open>Cong A0 C0 A1 C1\<close>)
    qed
    thus ?thesis 
      using \<open>Per A0 B C0\<close> l8_10 by blast
  qed
  have "B' \<noteq> C1"
    using assms(2) between_identity conga_diff56 \<open>Bet B' C' C1\<close> by blast
  moreover have "Per A' B' C1"
  proof -
    have "B' \<noteq> A1"
      using assms(2) between_identity conga_diff45 \<open>Bet B' A' A1\<close> by blast
    moreover have "Col B' A1 A'" 
      using Bet_cases Col_def \<open>Bet B' A' A1\<close> by blast
    ultimately show ?thesis 
      using l8_3 \<open>Per A1 B' C1\<close> by blast
  qed
  moreover have "Col B' C1 C'" 
    using \<open>Bet B' C' C1\<close> bet_col1 between_trivial by blast
  ultimately show ?thesis 
    using per_col by blast
qed

lemma l11_18_1:
  assumes "Bet C B D" and
    "B \<noteq> C" and
    "B \<noteq> D" and
    "A \<noteq> B" and
    "Per A B C"
  shows "A B C CongA A B D"
proof -
  have "Per A B D" 
    using Bet_cases Col_def assms(1) assms(2) assms(5) per_col by blast
  thus ?thesis 
    using assms(2) assms(3) assms(4) assms(5) l11_16 by auto
qed

lemma l11_18_2:
  assumes "Bet C B D" and
    "A B C CongA A B D"
  shows "Per A B C"
proof -
  obtain A0 C0 A1 D0 where P1: "Bet B A A0 \<and> Cong A A0 B A \<and> Bet B C C0 \<and>
Cong C C0 B D \<and> Bet B A A1 \<and> Cong A A1 B A \<and>
Bet B D D0 \<and> Cong D D0 B C \<and> Cong A0 C0 A1 D0"
    using CongA_def assms(2) by auto
  have P2: "A0 = A1"
    by (metis P1 assms(2) conga_diff45 construction_uniqueness)
  have P3: "Per A0 B C0"
  proof -
    have Q1: "Bet C0 B D0"
    proof -
      have R1: "Bet C0 C B"
        using P1 between_symmetry by blast
      have R2: "Bet C B D0"
      proof -
        have S1: "Bet C B D"
          by (simp add: assms(1))
        have S2: "Bet B D D0"
          using P1 by blast
        have "B \<noteq> D"
          using assms(2) conga_diff56 by blast
        thus ?thesis
          using S1 S2 outer_transitivity_between by blast
      qed
      have "C \<noteq> B"
        using assms(2) conga_diff2 by auto
      thus ?thesis
        using R1 R2 outer_transitivity_between2 by blast
    qed
    have Q2: "Cong C0 B B D0"
      by (meson P1 between_symmetry cong_3421 l2_11_b not_cong_1243)
    have "Cong A0 C0 A0 D0"
      using P1 P2 by blast
    thus ?thesis
      using Per_def Q1 Q2 midpoint_def by blast
  qed
  have P4: "B \<noteq> C0"
    using P1 assms(2) bet_neq12__neq conga_diff2 by blast
  have P5: "Per A B C0"
    by (metis P1 P3 l8_3 assms(2) bet_col bet_col1 bet_neq21__neq col_transitivity_1 
        conga_diff45)
  have "Col B C0 C" using P1
    using NCol_cases bet_col by blast
  thus ?thesis
    using P4 P5 per_col by blast
qed

lemma cong3_preserves_out:
  assumes "A Out B C" and
    "A B C Cong3 A' B' C'"
  shows "A' Out B' C'"
  by (meson assms(1) assms(2) col_permutation_4 cong3_symmetry cong_3_swap 
      l4_13 l4_6 not_bet_and_out or_bet_out out_col)

lemma l11_21_a:
  assumes "B Out A C" and
    "A B C CongA A' B' C'"
  shows "B' Out A' C'"
proof -
  obtain A0 C0 A1 C1 where P1: "Bet B A A0 \<and>
Cong A A0 B' A' \<and> Bet B C C0 \<and>
Cong C C0 B' C' \<and> Bet B' A' A1 \<and>
Cong A' A1 B A \<and> Bet B' C' C1 \<and>
Cong C' C1 B C \<and> Cong A0 C0 A1 C1"
    using CongA_def assms(2) by auto
  have P2: "B Out A0 C0"
    by (metis P1 assms(1) bet_out l6_6 l6_7 out_diff1)
  have P3: "B' Out A1 C1"
  proof -
    have "B A0 C0 Cong3 B' A1 C1"
      by (meson Cong3_def P1 between_symmetry cong_right_commutativity l2_11_b not_cong_4312)
    thus ?thesis
      using P2 cong3_preserves_out by blast
  qed
  thus ?thesis
    by (metis P1 assms(2) bet_out conga_diff45 conga_diff56 l6_6 l6_7)
qed

lemma l11_21_b:
  assumes "B Out A C" and
    "B' Out A' C'"
  shows "A B C CongA A' B' C'"
  using assms(1) assms(2) between_trivial2 conga_trivial_1 l11_10 out2_bet_out out_distinct
  by meson

lemma conga_cop__or_out_ts:
  assumes "Coplanar A B C C'" and
    "A B C CongA A B C'"
  shows "B Out C C' \<or> A B TS C C'"
proof -
  obtain A0 C0 A1 C1 where P1: "Bet B A A0 \<and>
Cong A A0 B A \<and>Bet B C C0 \<and>
Cong C C0 B C' \<and>Bet B A A1 \<and>
Cong A A1 B A \<and>Bet B C' C1 \<and>
Cong C' C1 B C \<and> Cong A0 C0 A1 C1"
    using CongA_def assms(2) by auto
  have P2: "A0 = A1" using P1
    by (metis assms(2) conga_diff1 construction_uniqueness)
  have "B Out C C' \<or> A B TS C C'"
  proof cases
    assume "C0 = C1"
    thus ?thesis
      by (metis P1 assms(2) bet2__out conga_diff2 conga_diff56)
  next
    assume R1: "C0 \<noteq> C1"
    obtain M where R2: "M Midpoint C0 C1"
      using midpoint_existence by blast
    have R3: "Cong B C0 B C1"
      by (meson Bet_cases P1 l2_11_b not_cong_2134 not_cong_3421)
    have R3A: "Cong A0 C0 A0 C1"
      using P1 P2 by blast
    hence R4: "Per A0 M C0" using R2
      using Per_def by blast
    have R5: "Per B M C0"
      using Per_def R2 R3 by auto
    hence R6: "Per B M C1"
      using R2 l8_4 by blast
    have R7: "B \<noteq> A0"
      using P1 assms(2) bet_neq12__neq conga_diff45 by blast
    hence "Cong A C0 A C1"
      by (meson Col_perm P1 R3 R3A bet_col l4_17)
    hence R9: "Per A M C0"
      using Per_def R2 by blast
    hence R10: "Per A M C1"
      by (meson R2 l8_4 Tarski_neutral_dimensionless_axioms)
    have R11: "Col B A M"
    proof -
      have S1: "Coplanar C0 B A M"
      proof -
        have "Coplanar B A C0 M"
        proof -
          have T1: "Coplanar B A C0 C1"
          proof -
            have "Coplanar A C0 B C'"
            proof -
              have "Coplanar A C' B C0"
              proof -
                have U1: "Coplanar A C' B C"
                  by (simp add: assms(1) coplanar_perm_4)
                have U2: "B \<noteq> C"
                  using assms(2) conga_diff2 by blast
                have "Col B C C0"
                  by (simp add: P1 bet_col)
                thus ?thesis
                  by (meson col_cop__cop Tarski_neutral_dimensionless_axioms U1 U2)
              qed
              thus ?thesis
                using ncoplanar_perm_5 by blast
            qed
            thus ?thesis
              by (metis P1 col_cop__cop assms(2) bet_col conga_diff56 coplanar_perm_12)
          qed
          have "Col C0 C1 M"
            using Col_perm R2 midpoint_col by blast
          thus ?thesis
            using T1 R1 col_cop__cop by blast
        qed
        thus ?thesis
          using ncoplanar_perm_8 by blast
      qed
      have "C0 \<noteq> M"
        using R1 R2 midpoint_distinct_1 by blast
      thus ?thesis
        using R5 R9 S1 cop_per2__col by blast
    qed
    have "B Out C C' \<or> A B TS C C'"
    proof cases
      assume Q1: "B = M"
      have Q2: "\<not> Col A B C"
        by (metis Col_def P1 Q1 R9 assms(2) conga_diff1 conga_diff2 l6_16_1 l8_9 
            not_bet_and_out out_trivial)
      have Q3: "\<not> Col A B C'"
        by (metis Col_def P1 Q1 R10 assms(2) conga_diff1 conga_diff56 l6_16_1 l8_9 
            not_bet_and_out out_trivial)
      have Q4: "Col B A B"
        by (simp add: col_trivial_3)
      have "Bet C B C'"
      proof -
        have S1: "Bet C1 C' B"
          using Bet_cases P1 by blast
        have "Bet C1 B C"
        proof -
          have T1: "Bet C0 C B"
            using Bet_cases P1 by blast
          have "Bet C0 B C1"
            by (simp add: Q1 R2 midpoint_bet)
          thus ?thesis
            using T1 between_exchange3 between_symmetry by blast
        qed
        thus ?thesis
          using S1 between_exchange3 between_symmetry by blast
      qed
      thus ?thesis
        by (metis Q2 Q3 Q4 bet__ts col_permutation_4 invert_two_sides)
    next
      assume L1: "B \<noteq> M"
      have L2: "B M TS C0 C1"
      proof -
        have M1: "\<not> Col C0 B M"
          by (metis (no_types) Col_perm L1 R1 R2 R5 is_midpoint_id l8_9)
        have M2: "\<not> Col C1 B M"
          using Col_perm L1 R1 R2 R6 l8_9 midpoint_not_midpoint by blast
        have M3: "Col M B M"
          using col_trivial_3 by auto
        have "Bet C0 M C1"
          by (simp add: R2 midpoint_bet)
        thus ?thesis
          using M1 M2 M3 TS_def by blast
      qed
      have "A B TS C C'"
      proof -
        have W2: "A B TS C C1"
        proof -
          have V1: "A B TS C0 C1"
            using L2 P1 R11 R7 col_two_sides cong_diff invert_two_sides 
              not_col_permutation_5 by blast
          have "B Out C0 C"
            using L2 Out_def P1 TS_def assms(2) col_trivial_1 conga_diff2 by auto
          thus ?thesis
            using V1 col_trivial_3 l9_5 by blast
        qed
        hence W1: "A B TS C' C"
        proof -
          have Z1: "A B TS C1 C"
            by (simp add: W2 l9_2)
          have Z2: "Col B A B"
            using not_col_distincts by blast
          have "B Out C1 C'"
            using L2 Out_def P1 TS_def assms(2) col_trivial_1 conga_diff56 by auto
          thus ?thesis
            using Z1 Z2 l9_5 by blast
        qed
        thus ?thesis
          by (simp add: l9_2)
      qed
      thus ?thesis by blast
    qed
    thus ?thesis by blast
  qed
  thus ?thesis by blast
qed

lemma conga_os__out:
  assumes "A B C CongA A B C'" and
    "A B OS C C'"
  shows "B Out C C'"
  using assms(1) assms(2) conga_cop__or_out_ts l9_9 os__coplanar by blast

lemma cong2_conga_cong:
  assumes "A B C CongA A' B' C'" and
    "Cong A B A' B'" and
    "Cong B C B' C'"
  shows "Cong A C A' C'"
  using assms(1) assms(2) assms(3) cong_4321 l11_3 l11_4_1 not_cong_3412 out_distinct out_trivial 
  by meson

lemma angle_construction_1:
  assumes "\<not> Col A B C" and
    "\<not> Col A' B' P"
  shows "\<exists> C'. (A B C CongA A' B' C' \<and> A' B' OS C' P)"
proof -
  obtain C0 where P1: "Col B A C0 \<and> B A Perp C C0"
    using assms(1) col_permutation_4 l8_18_existence by blast
  have "\<exists> C'. (A B C CongA A' B' C' \<and> A' B' OS C' P)"
  proof cases
    assume P1A: "B = C0"
    obtain C' where P2: "Per C' B' A' \<and> Cong C' B' C B \<and> A' B' OS C' P"
      by (metis assms(1) assms(2) col_trivial_1 col_trivial_2 ex_per_cong)
    have P3: "A B C CongA A' B' C'"
      by (metis P1 P2 l8_2 os_distincts P1A assms(1) l11_16 not_col_distincts perp_per_1)
    thus ?thesis using P2 by blast
  next
    assume P4: "B \<noteq> C0"
    have "\<exists> C'. (A B C CongA A' B' C' \<and> A' B' OS C' P)"
    proof cases
      assume R1: "B Out A C0"
      obtain C0' where R2: "B' Out A' C0' \<and> Cong B' C0' B C0"
        by (metis P4 assms(2) col_trivial_1 segment_construction_3)
      have "\<exists> C'. Per C' C0' B' \<and> Cong C' C0' C C0 \<and> B' C0' OS C' P"
      proof -
        have R4: "B' \<noteq> C0'"
          using Out_def R2 by auto
        have R5: "C \<noteq> C0"
          using P1 perp_distinct by blast
        have R6: "Col B' C0' C0'"
          by (simp add: col_trivial_2)
        have "\<not> Col B' C0' P"
          using NCol_cases R2 R4 assms(2) col_transitivity_1 out_col by blast
        hence "\<exists> C'. Per C' C0' B' \<and> Cong C' C0' C C0 \<and> B' C0' OS C' P" 
          using R4 R5 R6 ex_per_cong by blast
        thus ?thesis by auto
      qed
      then obtain C' where R7: "Per C' C0' B' \<and> Cong C' C0' C C0 \<and> B' C0' OS C' P" 
        by auto
      hence R8: "C0 B C Cong3 C0' B' C'"
        by (meson Cong3_def P1 R2 col_trivial_2 l10_12 l8_16_1 not_col_permutation_2 
            not_cong_2143 not_cong_4321)
      have R9: "A B C CongA A' B' C'"
      proof -
        have S1: "C0 B C CongA C0' B' C'"
          by (metis P4 R8 assms(1) cong3_conga not_col_distincts)
        have S3: "B Out C C"
          using assms(1) not_col_distincts out_trivial by force
        have "B' \<noteq> C'"
          using R8 assms(1) cong3_diff2 not_col_distincts by blast
        hence "B' Out C' C'"
          using out_trivial by auto
        thus ?thesis
          using S1 R1 S3 R2 l11_10 by blast
      qed
      have "B' A' OS C' P"
      proof -
        have T1: "Col B' C0' A'"
          by (meson NCol_cases R2 out_col Tarski_neutral_dimensionless_axioms)
        have "B' \<noteq> A'"
          using assms(2) col_trivial_1 by auto
        thus ?thesis
          using T1 R7 col_one_side by blast
      qed
      hence "A' B' OS C' P"
        by (simp add: invert_one_side)
      thus ?thesis
        using R9 by blast
    next
      assume U1: "\<not> B Out A C0"
      hence U2: "Bet A B C0"
        using NCol_perm P1 or_bet_out by blast
      obtain C0' where U3: "Bet A' B' C0' \<and> Cong B' C0' B C0"
        using segment_construction by blast
      have U4: "\<exists> C'. Per C' C0' B' \<and> Cong C' C0' C C0 \<and> B' C0' OS C' P"
      proof -
        have V2: "C \<noteq> C0"
          using Col_cases P1 assms(1) by blast
        have "B' \<noteq> C0'"
          using P4 U3 cong_diff_3 by blast
        hence "\<not> Col B' C0' P"
          using Col_def U3 assms(2) col_transitivity_1 by blast
        thus ?thesis using ex_per_cong
          using V2 not_col_distincts by blast
      qed
      then obtain C' where U5: "Per C' C0' B' \<and> Cong C' C0' C C0 \<and> B' C0' OS C' P"
        by blast
      have U98: "A B C CongA A' B' C'"
      proof -
        have X1: "C0 B C Cong3 C0' B' C'"
        proof -
          have X2: "Cong C0 B C0' B'"
            using Cong_cases U3 by blast
          have X3: "Cong C0 C C0' C'"
            using U5 not_cong_4321 by blast
          have "Cong B C B' C'"
          proof -
            have Y1: "Per C C0 B"
              using P1 col_trivial_3 l8_16_1 by blast
            have "Cong C C0 C' C0'"
              using U5 not_cong_3412 by blast
            thus ?thesis
              using Cong_perm Y1 U5 X2 l10_12 by blast
          qed
          thus ?thesis
            by (simp add: Cong3_def X2 X3)
        qed
        have X22: "Bet C0 B A"
          using U2 between_symmetry by blast
        have X24: "Bet C0' B' A'"
          using Bet_cases U3 by blast
        have "A' \<noteq> B'"
          using assms(2) not_col_distincts by blast
        thus ?thesis
          by (metis P4 X1 X22 X24 assms(1) cong3_conga l11_13 not_col_distincts)
      qed
      have "A' B' OS C' P"
      proof -
        have "B' A' OS C' P"
        proof -
          have W1: "Col B' C0' A'"
            by (simp add: Col_def U3)
          have "B' \<noteq> A'"
            using assms(2) not_col_distincts by blast
          thus ?thesis
            using W1 U5 col_one_side by blast
        qed
        thus ?thesis
          by (simp add: invert_one_side)
      qed
      thus ?thesis
        using U98 by blast
    qed
    thus ?thesis by auto
  qed
  thus ?thesis by auto
qed

lemma angle_construction_2:
  assumes "A \<noteq> B" (*and
"A \<noteq> C"*) and
    "B \<noteq> C" (*and
"A' \<noteq> B'"*) and
    "\<not> Col A' B' P"
  shows "\<exists> C'. (A B C CongA A' B' C' \<and> (A' B' OS C' P \<or> Col A' B' C'))"
  by (metis Col_def angle_construction_1 assms(1) assms(2) assms(3) col_trivial_3 
      conga_line l11_21_b or_bet_out out_trivial point_construction_different)

lemma ex_conga_ts:
  assumes "\<not> Col A B C" and
    "\<not> Col A' B' P"
  shows "\<exists> C'. A B C CongA A' B' C' \<and> A' B' TS C' P"
proof -
  obtain P' where P1: "A' Midpoint P P'"
    using symmetric_point_construction by blast
  have P2: "\<not> Col A' B' P'"
    by (metis P1 assms(2) col_transitivity_1 midpoint_col midpoint_distinct_2 not_col_distincts)
  obtain C' where P3:
    "A B C CongA A' B' C' \<and> A' B' OS C' P'"
    using P2 angle_construction_1 assms(1) by blast
  have "A' B' TS P' P"
    using P1 P2 assms(2) bet__ts l9_2 midpoint_bet not_col_distincts by auto
  thus ?thesis
    using P3 l9_8_2 one_side_symmetry by blast
qed

lemma l11_15:
  assumes "\<not> Col A B C" and
    "\<not> Col D E P"
  shows
    "\<exists> F. (A B C CongA D E F \<and> E D OS F P) \<and>
(\<forall> F1 F2. ((A B C CongA D E F1 \<and> E D OS F1 P) \<and>
(A B C CongA D E F2 \<and> E D OS F2 P))
\<longrightarrow> E Out F1 F2)"
proof -
  obtain F where P1: "A B C CongA D E F \<and>  D E OS F P"
    using angle_construction_1 assms(1) assms(2) by blast
  hence P2: "A B C CongA D E F \<and> E D OS F P"
    using invert_one_side by blast
  have "(\<forall> F1 F2. ((A B C CongA D E F1 \<and> E D OS F1 P) \<and>
(A B C CongA D E F2 \<and> E D OS F2 P)) \<longrightarrow> E Out F1 F2)"
  proof -
    {
      fix F1 F2
      assume P3: "((A B C CongA D E F1 \<and> E D OS F1 P) \<and>
  (A B C CongA D E F2 \<and> E D OS F2 P))"
      hence P4: "A B C CongA D E F1" by simp
      have P5: "E D OS F1 P" using P3 by simp
      have P6: "A B C CongA D E F2" using P3 by simp
      have P7: "E D OS F2 P" using P3 by simp
      have P8: "D E F1 CongA D E F2"
        using P4 conga_sym P6 conga_trans by blast
      have "D E OS F1 F2"
        using P5 P7 invert_one_side one_side_symmetry one_side_transitivity by blast
      hence "E Out F1 F2" using P8 conga_os__out
        by (meson P3 conga_sym conga_trans)
    }
    thus ?thesis by auto
  qed
  thus ?thesis
    using P2 by blast
qed

lemma l11_19:
  assumes "Per A B P1" and
    "Per A B P2" and
    "A B OS P1 P2"
  shows "B Out P1 P2"
proof cases
  assume "Col A B P1"
  thus ?thesis
    using assms(3) col123__nos by blast
next
  assume P1: "\<not> Col A B P1"
  have "B Out P1 P2"
  proof cases
    assume "Col A B P2"
    thus ?thesis
      using assms(3) one_side_not_col124 by blast
  next
    assume P2: "\<not> Col A B P2"
    obtain x where "A B P1 CongA A B x \<and> B A OS x P2 \<and>
(\<forall> F1 F2. ((A B P1 CongA A B F1 \<and> B A OS F1 P2) \<and>
(A B P1 CongA A B F2 \<and> B A OS F2 P2))\<longrightarrow> B Out F1 F2)"
      using P1 P2 l11_15 by blast
    thus ?thesis
      by (metis P1 P2 assms(1) assms(2) assms(3) conga_os__out l11_16 not_col_distincts)
  qed
  thus ?thesis
    by simp
qed

lemma l11_22_bet:
  assumes "Bet A B C" and
    "P' B' TS A' C'" and
    "A B P CongA A' B' P'" and
    "P B C CongA P' B' C'"
  shows "Bet A' B' C'"
proof -
  obtain C'' where P1: "Bet A' B' C'' \<and> Cong B' C'' B C"
    using segment_construction by blast
  have P2: "C B P CongA C'' B' P'"
    by (metis P1 assms(1) assms(3) assms(4) cong_diff_3 conga_diff2 l11_13)
  have P3: "C'' B' P' CongA C' B' P'"
    by (meson P2 conga_sym Tarski_neutral_dimensionless_axioms assms(4) conga_comm conga_trans)
  have P4: "B' Out C' C'' \<or> P' B' TS C' C''"
  proof -
    have P5: "Coplanar P' B' C' C''"
      by (meson P1 TS_def assms(2) bet__coplanar coplanar_trans_1 ncoplanar_perm_1 
          ncoplanar_perm_8 ts__coplanar)
    have "P' B' C' CongA P' B' C''"
      using P3 conga_comm conga_sym by blast
    thus ?thesis
      by (simp add: P5 conga_cop__or_out_ts)
  qed
  have P6: "B' Out C' C'' \<longrightarrow> Bet A' B' C'"
  proof -
    {
      assume "B' Out C' C''"
      hence "Bet A' B' C'"
        using P1 bet_out_out_bet between_exchange4 between_trivial2 col_trivial_3 
          l6_6 not_bet_out by blast
    }
    thus ?thesis by simp
  qed
  have "P' B' TS C' C'' \<longrightarrow> Bet A' B' C'"
  proof -
    {
      assume P7: "P' B' TS C' C''"
      hence "Bet A' B' C'"
      proof cases
        assume "Col C' B' P'"
        thus ?thesis
          using Col_perm TS_def assms(2) by blast
      next
        assume Q1: "\<not> Col C' B' P'"
        hence Q2: "B' \<noteq> P'"
          using not_col_distincts by blast
        have Q3: "B' P' TS A' C''"
          by (metis Col_perm P1 TS_def P7 assms(2) col_trivial_3)
        have Q4: "B' P' OS C' C''"
          using P7 Q3 assms(2) invert_two_sides l9_8_1 l9_9 by blast
        thus ?thesis
          using P7 invert_one_side l9_9 by blast
      qed
    }
    thus ?thesis by simp
  qed
  thus ?thesis using P6 P4 by blast
qed

lemma l11_22a:
  assumes "B P TS A C" and
    "B' P' TS A' C'" and
    "A B P CongA A' B' P'" and
    "P B C CongA P' B' C'"
  shows "A B C CongA A' B' C'"
proof -
  have P1: "A \<noteq> B \<and> A' \<noteq> B' \<and> P \<noteq> B \<and> P' \<noteq> B' \<and> C \<noteq> B \<and> C' \<noteq> B'"
    using assms(3) assms(4) conga_diff1 conga_diff2 conga_diff45 conga_diff56 by auto
  have P2: "A \<noteq> C \<and> A' \<noteq> C'"
    using assms(1) assms(2) not_two_sides_id by blast
  obtain A'' where P3: "B' Out A' A'' \<and> Cong B' A'' B A"
    using P1 segment_construction_3 by force
  have P4: "\<not> Col A B P"
    using TS_def assms(1) by blast
  obtain T where P5: "Col T B P \<and> Bet A T C"
    using TS_def assms(1) by blast
  have "A B C CongA A' B' C'"
  proof cases
    assume "B = T"
    thus ?thesis
      by (metis P1 P5 assms(2) assms(3) assms(4) conga_line invert_two_sides l11_22_bet)
  next
    assume P6: "B \<noteq> T"
    have "A B C CongA A' B' C'"
    proof cases
      assume P7A: "Bet P B T"
      obtain T'' where T1: "Bet P' B' T'' \<and> Cong B' T'' B T"
        using segment_construction by blast
      have "\<exists> T''. Col B' P' T'' \<and> (B' Out P' T'' \<longleftrightarrow> B Out P T) \<and> Cong B' T'' B T"
      proof -
        have T2: "Col B' P' T''" using T1
          by (simp add: bet_col col_permutation_4)
        have "(B' Out P' T'' \<longleftrightarrow> B Out P T) \<and> Cong B' T'' B T"
          using P7A T1 not_bet_and_out by blast
        thus ?thesis using T2 by blast
      qed
      then obtain T'' where T3:
        "Col B' P' T'' \<and> (B' Out P' T'' \<longleftrightarrow> B Out P T) \<and> Cong B' T'' B T" by blast
      hence T4: "B' \<noteq> T''"
        using P6 cong_diff_3 by blast
      obtain C'' where T5: "Bet A'' T'' C'' \<and> Cong T'' C'' T C"
        using segment_construction by blast
      have "T B A CongA T'' B' A'" 
      proof -
        have "P B A CongA P' B' A'" 
          by (simp add: assms(3) conga_comm)
        moreover
        have "Col P' B' T''" 
          using T3 Col_cases by blast
        hence "Bet P' B' T''"
          using P7A T3 not_bet_and_out not_out_bet by blast
        ultimately show ?thesis 
          using T4 P6 P7A l11_13 by blast
      qed
      hence T6: "A B T CongA A' B' T''"
        using Out_cases P5 P6 T3 T4 P7A assms(3) between_symmetry col_permutation_4 
          conga_comm l11_13 l6_4_1 or_bet_out by blast
      have T7: "A B T CongA A'' B'  T''"
      proof -
        have "B Out A A" 
          using P4 not_col_distincts out_trivial by presburger
        moreover have "B Out T T" 
          using P6 not_col_distincts out_trivial by presburger
        moreover have "B' Out A'' A'" 
          using P3 l6_6 by blast
        ultimately show ?thesis
          using l11_10 T4 T6 out_trivial by presburger
      qed
      hence T8: "Cong A T A'' T''"
        using P3 T3 cong2_conga_cong cong_4321 not_cong_3412 by blast
      have T9: "Cong A C A'' C''"
        using P5 T5 T8 cong_symmetry l2_11_b by blast
      have "A \<noteq> T" 
        using P4 P5 by auto
      hence T10: "Cong C B C'' B'"
        using P3 T3 T5 T8 cong_commutativity cong_symmetry five_segment P5
        by meson 
      have "A B C Cong3 A'' B' C''"
        using Cong3_def P3 T10 T9 not_cong_2143 not_cong_4321 by blast
      hence T11: "A B C CongA A'' B' C''"
        by (simp add: cong3_conga Tarski_neutral_dimensionless_axioms P1)
      have "C B T Cong3 C'' B' T''"
        by (simp add: Cong3_def T10 T3 T5 cong_4321 cong_symmetry)
      hence T12: "C B T CongA C'' B' T''"
        using P1 P6 cong3_conga by auto
      have T13: "P B C CongA P' B' C''"
      proof -
        have "Bet T B P"
          using Bet_perm P7A by blast
        moreover have "Bet T'' B' P'"
          using Col_perm P7A T3 l6_6 not_bet_and_out or_bet_out by blast
        ultimately show ?thesis
          using P1 T12 conga_comm l11_13 by blast
      qed
      have T14: "P' B' C' CongA P' B' C''"
      proof -
        have "P' B' C' CongA P B C"
          by (simp add: assms(4) conga_sym)
        thus ?thesis
          using T13 conga_trans by blast
      qed
      have T15: "B' Out C' C'' \<or> P' B' TS C' C''"
      proof -
        have K7: "Coplanar P' B' C' C''"
        proof -
          have K8: "Coplanar A' P' B' C'"
            using assms(2) coplanar_perm_14 ts__coplanar by blast
          have K8A: "Coplanar P' C'' B' A''"
          proof -
            have "Col P' B' T'' \<and> Col C'' A'' T''"
              using Col_def Col_perm T3 T5 by blast
            hence "Col P' C'' T'' \<and> Col B' A'' T'' \<or> 
Col P' B' T'' \<and> Col C'' A'' T'' \<or> 
Col P' A'' T'' \<and> Col C'' B' T''" by simp
            thus ?thesis
              using Coplanar_def by auto
          qed
          hence "Coplanar A' P' B' C''"
          proof -
            have K9: "B' \<noteq> A''"
              using P3 out_distinct by blast
            have "Col B' A'' A'"
              using Col_perm P3 out_col by blast
            thus ?thesis
              using K8A K9 col_cop__cop coplanar_perm_19 by blast
          qed
          thus ?thesis
            by (meson K8 TS_def assms(2) coplanar_perm_7 coplanar_trans_1 ncoplanar_perm_2)
        qed
        thus ?thesis
          by (simp add: T14 K7 conga_cop__or_out_ts)
      qed
      have "A B C CongA A' B' C'"
      proof cases
        assume "B' Out C' C''"
        thus ?thesis
          using P1 P3 T11 l11_10 out_trivial by blast
      next
        assume W1: "\<not> B' Out C' C''"
        hence W1A: " P' B' TS C' C''" using T15 by simp
        have W2: "B' P' TS A'' C'"
          using P3 assms(2) col_trivial_1 l9_5 by blast
        hence W3: "B' P' OS A'' C''"
          using T15 W1 invert_two_sides l9_2 l9_8_1 by blast
        have W4: "B' P' TS A''  C''"
        proof -
          have "\<not> Col A' B' P'"
            using TS_def assms(2) by auto
          thus ?thesis
            using Col_perm T3 T5 W3 one_side_chara by blast
        qed
        thus ?thesis
          using W1A W2 invert_two_sides l9_8_1 l9_9 by blast
      qed
      thus ?thesis by simp
    next
      assume R1: "\<not> Bet P B T"
      hence R2: "B Out P T"
        using Col_cases P5 l6_4_2 by blast
      have R2A: "\<exists> T''. Col B' P' T'' \<and> (B' Out P' T'' \<longleftrightarrow> B Out P T) \<and> Cong B' T'' B T"
      proof -
        obtain T'' where R3: "B' Out P' T'' \<and> Cong B' T'' B T"
          using P1 P6 segment_construction_3 by fastforce
        thus ?thesis
          using R2 out_col by blast
      qed
      then obtain T'' where R4: "Col B' P' T'' \<and> 
(B' Out P' T'' \<longleftrightarrow> B Out P T) \<and> 
Cong B' T'' B T" by auto
      have R5: "B' \<noteq> T''"
        using P6 R4 cong_diff_3 by blast
      obtain C'' where R6: "Bet A'' T'' C'' \<and> Cong T'' C'' T C"
        using segment_construction by blast
      have R7: "A B T CongA A' B' T''"
        using P1 R2 R4 assms(3) l11_10 l6_6 out_trivial by auto
      have R8: "A B T CongA A'' B'  T''"
        using P3 P4 R2 R4 assms(3) l11_10 l6_6 not_col_distincts out_trivial by blast
      have R9: "Cong A T A'' T''"
        using Cong_cases P3 R4 R8 cong2_conga_cong by blast
      have R10: "Cong A C A'' C''"
        using P5 R6 R9 l2_11_b not_cong_3412 by blast
      have R11: "Cong C B C'' B'"
      proof -
        have "Cong T C T'' C''" 
          using Cong_cases R6 by blast
        moreover have "Cong A B A'' B'"
          by (metis P3 Cong_cases)
        moreover have "Cong T B T'' B'" 
          using R4 not_cong_4321 by blast
        moreover have "A \<noteq> T" 
          using P4 P5 by auto
        ultimately show ?thesis 
          using R9 P5 R6 five_segment by blast
      qed
      have "A B C Cong3 A'' B' C''"
        by (simp add: Cong3_def P3 R10 R11 cong_4321 cong_commutativity)
      hence R12: "A B C CongA A'' B' C''"
        using P1 by (simp add: cong3_conga)
      have "C B T Cong3 C'' B' T''"
        using Cong3_def R11 R4 R6 not_cong_3412 not_cong_4321 by blast
      hence R13: "C B T CongA C'' B' T''"
        using P1 P6 cong3_conga Tarski_neutral_dimensionless_axioms by fastforce
      have R13A: "\<not> Col A' B' P'"
        using TS_def assms(2) by blast
      have R14: "B' Out C' C'' \<or> P' B' TS C' C''"
      proof -
        have S1: "Coplanar P' B' C' C''"
        proof -
          have T1: "Coplanar A' P' B' C'"
            using assms(2) ncoplanar_perm_14 ts__coplanar by blast
          have "Coplanar A' P' B' C''"
          proof -
            have U6: "B' \<noteq> A''"
              using P3 out_diff2 by blast
            have "Coplanar P' C'' B' A''"
            proof -
              have "Col P' B' T'' \<and> Col C'' A'' T''"
                using Col_def Col_perm R4 R6 by blast
              thus ?thesis using Coplanar_def by auto
            qed
            thus ?thesis
              by (meson Col_cases P3 U6 col_cop__cop ncoplanar_perm_21 ncoplanar_perm_6 out_col)
          qed
          thus ?thesis
            using NCol_cases R13A T1 coplanar_trans_1 by blast
        qed
        have "P' B' C' CongA P' B' C''"
        proof -
          have "C B P CongA C'' B' P'"
            using P1 R12 R13 R2 R4 conga_diff56 l11_10 out_trivial by presburger
          hence "C' B' P' CongA C'' B' P'"
            by (meson conga_trans Tarski_neutral_dimensionless_axioms assms(4) 
                conga_comm conga_sym)
          thus ?thesis
            by (simp add: conga_comm)
        qed
        thus ?thesis
          by (simp add: S1 conga_cop__or_out_ts)
      qed
      have S1: "B Out A A"
        using P4 not_col_distincts out_trivial by blast
      have S2: "B Out C C"
        using TS_def assms(1) not_col_distincts out_trivial by auto
      have S3: "B' Out A' A''" using P3 by simp
      have "A B C CongA A' B' C'"
      proof cases
        assume "B' Out C' C''"
        thus ?thesis using S1 S2 S3
          using R12 l11_10 by blast
      next
        assume "\<not> B' Out C' C''"
        hence Z3: "P' B' TS C' C''" using R14 by simp
        have Q1: "B' P' TS A'' C'"
          using S3 assms(2) l9_5 not_col_distincts by blast
        have Q2: "B' P' OS A'' C''"
        proof -
          have "B' P' TS C'' C'"
          proof -
            have "B' P' TS C' C''" using Z3
              using invert_two_sides by blast
            thus ?thesis
              by (simp add: l9_2)
          qed
          thus ?thesis
            using Q1 l9_8_1 by blast
        qed
        have "B' P' TS A''  C''"
          using Col_perm Q2 R4 R6 one_side_chara by blast
        thus ?thesis
          using Q2 l9_9 by blast
      qed
      thus ?thesis using S1 S2 S3
        using R12 l11_10 by blast
    qed
    thus ?thesis by simp
  qed
  thus ?thesis by simp
qed

lemma l11_22b:
  assumes "B P OS A C" and
    "B' P' OS A' C'" and
    "A B P CongA A' B' P'" and
    "P B C CongA P' B' C'"
  shows "A B C CongA A' B' C'"
proof -
  obtain D where P1: "Bet A B D \<and> Cong B D A B"
    using segment_construction by blast
  obtain D' where P2: "Bet A' B' D' \<and> Cong B' D' A' B'"
    using segment_construction by blast
  have P3: "D B P CongA D' B' P'"
  proof -
    have Q3: "D \<noteq> B"
      by (metis P1 assms(1) col_trivial_3 cong_diff_3 one_side_not_col124 one_side_symmetry)
    have Q5: "D' \<noteq> B'"
      by (metis P2 assms(2) col_trivial_3 cong_diff_3 one_side_not_col124 one_side_symmetry)
    thus ?thesis
      using assms(3) P1 Q3 P2 l11_13 by blast
  qed
  have P5: "D B C CongA D' B' C'"
  proof -
    have Q1: "B P TS D C"
      by (metis P1 assms(1) bet__ts col_trivial_3 cong_diff_3 l9_2 l9_8_2 one_side_not_col124 
          one_side_symmetry)
    have "B' P' TS D' C'" 
      by (metis Cong_perm P2 assms(2) bet__ts between_cong between_trivial2 l9_2 l9_8_2 
          one_side_not_col123 point_construction_different ts_distincts)
    thus ?thesis
      using assms(4) Q1 P3 l11_22a by blast
  qed
  have P6: "Bet D B A"
    using Bet_perm P1 by blast
  have P7: "A \<noteq> B"
    using assms(3) conga_diff1 by auto
  have P8: "Bet D' B' A'"
    using Bet_cases P2 by blast
  have "A' \<noteq> B'"
    using assms(3) conga_diff45 by blast
  thus ?thesis
    using P5 P6 P7 P8 l11_13 by blast
qed

lemma l11_22:
  assumes "((B P TS A C \<and> B' P' TS A' C')\<or>(B P OS A C \<and> B' P' OS A' C'))" and
    "A B P CongA A' B' P'" and
    "P B C CongA P' B' C'"
  shows "A B C CongA A' B' C'"
  by (meson assms(1) assms(2) assms(3) l11_22a l11_22b)

lemma l11_24:
  assumes "P InAngle A B C"
  shows "P InAngle C B A"
proof -
  obtain X where "Bet A X C" and "X = B \<or> B Out X P" 
    using InAngle_def assms by auto
  thus ?thesis 
    using Bet_cases InAngle_def assms by auto
qed

lemma col_in_angle:
  assumes "A \<noteq> B" and
    "C \<noteq> B" and
    "P \<noteq> B" and
    "B Out A P \<or> B Out C P"
  shows "P InAngle A B C"
  by (meson InAngle_def assms(1) assms(2) assms(3) assms(4) between_trivial between_trivial2)

lemma out321__inangle:
  assumes "C \<noteq> B" and
    "B Out A P"
  shows "P InAngle A B C"
  using assms(1) assms(2) col_in_angle out_distinct by auto

lemma inangle1123:
  assumes "A \<noteq> B" and
    "C \<noteq> B"
  shows "A InAngle A B C"
  by (simp add: assms(1) assms(2) out321__inangle out_trivial)

lemma out341__inangle:
  assumes "A \<noteq> B" and
    "B Out C P"
  shows "P InAngle A B C"
  using assms(1) assms(2) col_in_angle out_distinct by auto

lemma inangle3123:
  assumes "A \<noteq> B" and
    "C \<noteq> B"
  shows "C InAngle A B C"
  by (simp add: assms(1) assms(2) inangle1123 l11_24)

lemma in_angle_two_sides:
  assumes "\<not> Col B A P" and
    "\<not> Col B C P" and
    "P InAngle A B C"
  shows "P B TS A C"           
  by (metis InAngle_def TS_def assms(1) assms(2) assms(3) not_col_distincts 
      not_col_permutation_1 out_col)

lemma in_angle_out:
  assumes "B Out A C" and
    "P InAngle A B C"
  shows "B Out A P"
  by (metis InAngle_def assms(1) assms(2) not_bet_and_out out2_bet_out)

lemma col_in_angle_out:
  assumes "Col B A P" and
    "\<not> Bet A B C" and
    "P InAngle A B C"
  shows "B Out A P"
proof -
  obtain X where P1: "Bet A X C \<and> (X = B \<or> B Out X P)"
    using InAngle_def assms(3) by auto
  have "B Out A P"
  proof cases
    assume "X = B"
    thus ?thesis
      using P1 assms(2) by blast
  next
    assume P2: "X \<noteq> B"
    thus ?thesis
    proof -
      have f1: "Bet B A P \<or> A Out B P"
        by (meson assms(1) l6_4_2)
      have f2: "B Out X P"
        using P1 P2 by blast
      have f3: "(Bet B P A \<or>Bet B A P) \<or>Bet P B A"
        using f1 by (meson Bet_perm Out_def)
      have f4: "Bet B P X \<or>Bet P X B"
        using f2 by (meson Bet_perm Out_def)
      hence f5: "((Bet B P X \<or>Bet X B A) \<or>Bet B P A) \<or>Bet B A P"
        using f3 by (meson between_exchange3)
      have "\<forall>p. Bet p X C \<or> \<not>Bet A p X"
        using P1 between_exchange3 by blast
      hence f6: "(P = B \<or>Bet B A P) \<or>Bet B P A"
        using f5 f3 by (meson Bet_perm P2 assms(2) outer_transitivity_between2)
      have f7: "Bet C X A"
        using Bet_perm P1 by blast
      have "P \<noteq> B"
        using f2 by (simp add: Out_def)
      moreover
      { assume "Bet B B C"
        hence "A \<noteq> B"
          using assms(2) by blast }
      ultimately have "A \<noteq> B"
        using f7 f4 f1 by (meson Bet_perm Out_def P2 between_exchange3 
            outer_transitivity_between2)
      thus ?thesis
        using f6 f2 by (simp add: Out_def)
    qed
  qed
  thus ?thesis by blast
qed

lemma l11_25_aux:
  assumes "P InAngle A B C" and
    "\<not> Bet A B C" and
    "B Out A' A"
  shows "P InAngle A' B C"
proof -
  have P1: "Bet B A' A \<or> Bet B A A'"
    using Out_def assms(3) by auto
  {
    assume P2: "Bet B A' A"
    obtain X where P3: "Bet A X C \<and> (X = B \<or> B Out X P)"
      using InAngle_def assms(1) by auto
    obtain T where P4: "Bet A' T C \<and> Bet X T B"
      using Bet_perm P2 P3 inner_pasch by blast
    {
      assume "X = B"
      hence "P InAngle A' B C"
        using P3 assms(2) by blast
    }
    {
      assume "B Out X P"
      hence "P InAngle A' B C"
        by (metis InAngle_def P4 assms(1) assms(3) bet_out_1 l6_7 out_diff1)
    }
    hence "P InAngle A' B C"
      using P3 \<open>X = B \<Longrightarrow> P InAngle A' B C\<close> by blast
  }
  {
    assume Q0: "Bet B A A'"
    obtain X where Q1: "Bet A X C \<and> (X = B \<or> B Out X P)"
      using InAngle_def assms(1) by auto
    {
      assume "X = B"
      hence "P InAngle A' B C"
        using Q1 assms(2) by blast
    }
    {
      assume Q2: "B Out X P"
      obtain T where Q3: "Bet A' T C \<and> Bet B X T"
        using Bet_perm Q1 Q0 outer_pasch by blast
      hence "P InAngle A' B C"
        by (metis InAngle_def Q0 Q2 assms(1) bet_out l6_6 l6_7 out_diff1)
    }
    hence "P InAngle A' B C"
      using \<open>X = B \<Longrightarrow> P InAngle A' B C\<close> Q1 by blast
  }
  thus ?thesis
    using P1 \<open>Bet B A' A \<Longrightarrow> P InAngle A' B C\<close> by blast
qed

lemma l11_25:
  assumes "P InAngle A B C" and
    "B Out A' A" and
    "B Out C' C" and
    "B Out P' P"
  shows "P' InAngle A' B C'"
proof cases
  assume "Bet A B C"
  thus ?thesis
    by (metis Bet_perm InAngle_def assms(2) assms(3) assms(4) bet_out__bet l6_6 out_distinct)
next
  assume P1: "\<not> Bet A B C"
  have P2: "P InAngle A' B C"
    using P1 assms(1) assms(2) l11_25_aux by blast
  have P3: "P InAngle A' B C'"
  proof -
    have "P InAngle C' B A'" using l11_25_aux
      using Bet_perm P1 P2 assms(2) assms(3) bet_out__bet l11_24 by blast
    thus ?thesis using l11_24 by blast
  qed
  obtain X where P4: "Bet A' X C' \<and> (X = B \<or> B Out X P)"
    using InAngle_def P3 by auto
  {
    assume "X = B"
    hence "P' InAngle A' B C'"
      using InAngle_def P3 P4 assms(4) out_diff1 by auto
  }
  {
    assume "B Out X P"
    hence "P' InAngle A' B C'"
    proof -
      have "\<forall>p. B Out p P' \<or> \<not> B Out p P"
        by (meson Out_cases assms(4) l6_7)
      thus ?thesis
        by (metis (no_types) InAngle_def P3 assms(4) out_diff1)
    qed
  }
  thus ?thesis
    using InAngle_def P4 assms(2) assms(3) assms(4) out_distinct by auto
qed

lemma inangle_distincts:
  assumes "P InAngle A B C"
  shows "A \<noteq> B \<and> C \<noteq> B \<and> P \<noteq> B"
  using InAngle_def assms by auto

lemma segment_construction_0:
  shows "\<exists> B'. Cong A' B' A B"
  using segment_construction by blast

lemma angle_construction_3:
  assumes "A \<noteq> B" and
    "C \<noteq> B" and
    "A' \<noteq> B'"
  shows "\<exists> C'. A B C CongA A' B' C'"
  by (metis angle_construction_2 assms(1) assms(2) assms(3) not_col_exists)

lemma l11_28:
  assumes "A B C Cong3 A' B' C'" and
    "Col A C D"
  shows "\<exists> D'. (Cong A D A' D' \<and> Cong B D B' D' \<and> Cong C D C' D')"
proof cases
  assume P1: "A = C"
  have "\<exists> D'. (Cong A D A' D' \<and> Cong B D B' D' \<and> Cong C D C' D')"
  proof cases
    assume "A = B"
    thus ?thesis
      by (metis P1 assms(1) cong3_diff cong3_symmetry cong_3_swap_2 not_cong_3421 
          segment_construction_0)
  next
    assume "A \<noteq> B"
    have "\<exists> D'. (Cong A D A' D' \<and> Cong B D B' D' \<and> Cong C D C' D')"
    proof cases
      assume "A = D"
      thus ?thesis
        using Cong3_def P1 assms(1) cong_trivial_identity by blast
    next
      assume "A \<noteq> D"
      have "\<exists> D'. (Cong A D A' D' \<and> Cong B D B' D' \<and> Cong C D C' D')"
      proof cases
        assume "B = D"
        thus ?thesis
          using Cong3_def assms(1) cong_3_swap_2 cong_trivial_identity by blast
      next
        assume Q1: "B \<noteq> D"
        obtain D'' where Q2: "B A D CongA B' A' D''"
          by (metis \<open>A \<noteq> B\<close> \<open>A \<noteq> D\<close> angle_construction_3 assms(1) cong3_diff)
        obtain D' where Q3: "A' Out D'' D' \<and> Cong A' D' A D"
          by (metis Q2 \<open>A \<noteq> D\<close> conga_diff56 segment_construction_3)
        have Q5: "Cong A D A' D'"
          using Q3 not_cong_3412 by blast
        have "B A D CongA B' A' D'"
          using Q2 Q3 \<open>A \<noteq> B\<close> \<open>A \<noteq> D\<close> conga_diff45 l11_10 l6_6 out_trivial by auto
        hence "Cong B D B' D'"
          using Cong3_def Cong_perm Q5 assms(1) cong2_conga_cong by blast
        thus ?thesis
          using Cong3_def P1 Q5 assms(1) cong_reverse_identity by blast
      qed
      thus ?thesis by simp
    qed
    thus ?thesis by simp
  qed
  thus ?thesis by simp
next
  assume Z1: "A \<noteq> C"
  have  "\<exists> D'. (Cong A D A' D' \<and> Cong B D B' D' \<and> Cong C D C' D')"
  proof cases
    assume "A = D"
    thus ?thesis
      using Cong3_def Cong_perm assms(1) cong_trivial_identity by blast
  next
    assume "A \<noteq> D"
    {
      assume "Bet A C D"
      obtain D' where W1: "Bet A' C' D' \<and> Cong C' D' C D"
        using segment_construction by blast
      have W2: "Cong A D A' D'"
        by (meson Cong3_def W1 \<open>Bet A C D\<close> assms(1) cong_symmetry l2_11_b)
      have W3: "Cong B D B' D'"
      proof -
        have X1: "Cong C D C' D'"
          using W1 not_cong_3412 by blast
        have "Cong C B C' B'"
          using Cong3_def assms(1) cong_commutativity by presburger
        hence W4: "A C D B OFSC A' C' D' B'"
          using Cong3_def OFSC_def W1 X1 \<open>Bet A C D\<close> assms(1) by blast
        have "Cong D B D' B'"
          using W4 \<open>A \<noteq> C\<close> five_segment_with_def by blast
        thus ?thesis
          using Z1 not_cong_2143 by blast
      qed
      have "Cong C D C' D'"
        by (simp add: W1 cong_symmetry)
      hence "\<exists> D'. (Cong A D A' D' \<and> Cong B D B' D' \<and> Cong C D C' D')"
        using W2 W3 by blast
    }
    {
      assume W3B: "Bet C D A"
      then obtain D' where W4A: "Bet A' D' C' \<and> A D C Cong3 A' D' C'"
        using Bet_perm Cong3_def assms(1) l4_5 by blast
      have W5: "Cong A D A' D'"
        using Cong3_def W4A by blast
      have "A D C B IFSC A' D' C' B'"
        by (meson Bet_perm Cong3_def Cong_perm IFSC_def W4A W3B assms(1))
      hence "Cong D B D' B'"
        using l4_2 by blast
      hence W6: "Cong B D B' D'"
        using Cong_perm by blast
      hence "Cong C D C' D'"
        using Cong3_def W4A not_cong_2143 by blast
      hence "\<exists> D'. (Cong A D A' D' \<and> Cong B D B' D' \<and> Cong C D C' D')"
        using W5 W6 by blast
    }
    {
      assume W7: "Bet D A C"
      obtain D' where W7A: "Bet C' A' D' \<and> Cong A' D' A D"
        using segment_construction by blast
      hence W8: "Cong A D A' D'"
        using Cong_cases by blast
      have "C A D B OFSC C' A' D' B'"
        by (meson Bet_perm Cong3_def Cong_perm OFSC_def W7 W7A assms(1))
      hence "Cong D B D' B'"
        using Z1 five_segment_with_def by auto
      hence w9: "Cong B D B' D'"
        using Cong_perm by blast
      have "Cong C D C' D'"
      proof -
        have L1: "Bet C A D"
          using Bet_perm W7 by blast
        have L2: "Bet C' A' D'"
          using Bet_perm W7
          using W7A by blast
        have "Cong C A C' A'" using assms(1)
          using Cong3_def assms(1) not_cong_2143 by blast
        thus ?thesis using l2_11
          using L1 L2 W8 l2_11 by blast
      qed
      hence "\<exists> D'. (Cong A D A' D' \<and> Cong B D B' D' \<and> Cong C D C' D')"
        using W8 w9 by blast
    }
    thus ?thesis
      using Bet_cases \<open>Bet A C D \<Longrightarrow> \<exists>D'. Cong A D A' D' \<and> Cong B D B' D' \<and> Cong C D C' D'\<close> 
        \<open>Bet C D A \<Longrightarrow> \<exists>D'. Cong A D A' D' \<and> Cong B D B' D' \<and> Cong C D C' D'\<close> 
        assms(2) third_point by blast
  qed
  thus ?thesis
    by blast
qed

lemma bet_conga__bet:
  assumes "Bet A B C" and
    "A B C CongA A' B' C'"
  shows "Bet A' B' C'"
proof -
  obtain A0 C0 A1 C1 where P1:"
Bet B A A0 \<and>Cong A A0 B' A' \<and>
Bet B C C0 \<and>Cong C C0 B' C' \<and>
Bet B' A' A1 \<and>Cong A' A1 B A \<and>
Bet B' C' C1 \<and>Cong C' C1 B C \<and>
Cong A0 C0 A1 C1" using CongA_def assms(2)
    by auto
  have "Bet C B A0" using P1 outer_transitivity_between
    by (metis assms(1) assms(2) between_symmetry conga_diff1)
  hence "Bet A0 B C"
    using Bet_cases by blast
  hence P2: "Bet A0 B C0"
    using P1 assms(2) conga_diff2 outer_transitivity_between by blast
  have P3: "A0 B C0 Cong3 A1 B' C1"
  proof -
    have Q1: "Cong A0 B A1 B'"
      by (meson Bet_cases P1 l2_11_b not_cong_1243 not_cong_4312)
    have Q3: "Cong B C0 B' C1"
      using P1 between_symmetry cong_3421 l2_11_b not_cong_1243 by blast
    thus ?thesis
      by (simp add: Cong3_def Q1 P1)
  qed
  hence P4: "Bet A1 B' C1" using P2 l4_6 by blast
  hence "Bet A' B' C1"
    using P1 Bet_cases between_exchange3 by blast
  thus ?thesis using between_inner_transitivity P1 by blast
qed

lemma in_angle_one_side:
  assumes "\<not> Col A B C" and
    "\<not> Col B A P" and
    "P InAngle A B C"
  shows "A B OS P C"
proof -
  obtain X where P1: "Bet A X C \<and> (X = B \<or> B Out X P)"
    using InAngle_def assms(3) by auto
  {
    assume "X = B"
    hence  "A B OS P C"
      using P1 assms(1) bet_col by blast
  }
  {
    assume P2: "B Out X P"
    obtain C' where P2A: "Bet C A C' \<and> Cong A C' C A"
      using segment_construction by blast
    have "A B TS X C'"
    proof -
      have Q1: "\<not> Col X A B"
        by (metis Col_def P1 assms(1) assms(2) col_transitivity_2 out_col)
      have Q2 :"\<not> Col C' A B"
        by (metis Col_def Cong_perm P2A assms(1) cong_diff l6_16_1)
      have "\<exists> T. Col T A B \<and> Bet X T C'"
        using Bet_cases P1 P2A between_exchange3 col_trivial_1 by blast
      thus ?thesis
        by (simp add: Q1 Q2 TS_def)
    qed
    hence P3: "A B TS P C'"
      using P2 col_trivial_3 l9_5 by blast
    hence "A B TS C C'"
      using P1 P2 bet_out bet_ts__os between_trivial col123__nos col_trivial_3 
        invert_two_sides l6_6 l9_2 l9_5 
      by (metis P2A assms(1) bet__ts)
    hence "A B OS P C"
      using OS_def P3 by blast
  }
  thus ?thesis
    using P1 \<open>X = B \<Longrightarrow> A B OS P C\<close> by blast
qed

lemma inangle_one_side:
  assumes "\<not> Col A B C" and
    "\<not> Col A B P" and
    "\<not> Col A B Q" and
    "P InAngle A B C" and
    "Q InAngle A B C"
  shows "A B OS P Q"
  by (meson assms(1) assms(2) assms(3) assms(4) assms(5) in_angle_one_side 
      not_col_permutation_4 one_side_symmetry one_side_transitivity)

lemma inangle_one_side2:
  assumes "\<not> Col A B C" and
    "\<not> Col A B P" and
    "\<not> Col A B Q" and
    "\<not> Col C B P" and
    "\<not> Col C B Q" and
    "P InAngle A B C" and
    "Q InAngle A B C"
  shows "A B OS P Q \<and> C B OS P Q"
  by (meson assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) assms(7) 
      inangle_one_side l11_24 not_col_permutation_3)

lemma col_conga_col:
  assumes "Col A B C" and
    "A B C CongA D E F"
  shows  "Col D E F"
proof -
  {
    assume "Bet A B C"
    hence "Col D E F"
      using Col_def assms(2) bet_conga__bet by blast
  }
  {
    assume "Bet B C A"
    hence "Col D E F"
      by (meson Col_perm l11_21_a \<open>Bet A B C \<Longrightarrow> Col D E F\<close> assms(1) assms(2) 
          or_bet_out out_col)
  }
  {
    assume "Bet C A B"
    hence "Col D E F"
      by (meson Col_perm l11_21_a \<open>Bet A B C \<Longrightarrow> Col D E F\<close> assms(1) assms(2) 
          or_bet_out out_col)
  }
  thus ?thesis
    using Col_def \<open>Bet A B C \<Longrightarrow> Col D E F\<close> \<open>Bet B C A \<Longrightarrow> Col D E F\<close> assms(1) by blast
qed

lemma ncol_conga_ncol:
  assumes "\<not> Col A B C" and
    "A B C CongA D E F"
  shows "\<not> Col D E F"
  using assms(1) assms(2) col_conga_col conga_sym by blast

lemma angle_construction_4:
  assumes "A \<noteq> B" and
    "C \<noteq> B" and
    "A' \<noteq> B'"
  shows "\<exists>C'. (A B C CongA A' B' C' \<and> Coplanar A' B' C' P)"
proof cases
  assume "Col A' B' P"
  thus ?thesis
    using angle_construction_3 assms(1) assms(2) assms(3) ncop__ncols by blast
next
  assume "\<not> Col A' B' P"
  {
    assume "Col A B C"
    hence "\<exists>C'. (A B C CongA A' B' C' \<and> Coplanar A' B' C' P)"
      by (meson angle_construction_3 assms(1) assms(2) assms(3) col__coplanar col_conga_col)
  }
  {
    assume "\<not> Col A B C"
    then obtain C' where "A B C CongA A' B' C' \<and> A' B' OS C' P"
      using \<open>\<not> Col A' B' P\<close> angle_construction_1 by blast
    hence  "\<exists>C'. (A B C CongA A' B' C' \<and> Coplanar A' B' C' P)"
      using os__coplanar by blast
  }
  thus ?thesis
    using \<open>Col A B C \<Longrightarrow> \<exists>C'. A B C CongA A' B' C' \<and> Coplanar A' B' C' P\<close> by blast
qed

lemma lea_distincts:
  assumes "A B C LeA D E F"
  shows "A\<noteq>B \<and> C\<noteq>B \<and> D\<noteq>E \<and> F\<noteq>E"
  by (metis LeA_def conga_diff1 conga_diff2 assms inangle_distincts)

lemma l11_29_a:
  assumes "A B C LeA D E F"
  shows "\<exists> Q. (C InAngle A B Q \<and> A B Q CongA D E F)"
proof -
  obtain P where P1: "P InAngle D E F \<and> A B C CongA D E P"
    using LeA_def assms by blast
  hence P2: "E \<noteq> D \<and> B \<noteq> A \<and> E \<noteq> F \<and> E \<noteq> P \<and> B \<noteq> C"
    using conga_diff1 conga_diff2 inangle_distincts by blast
  hence P3: "A \<noteq> B \<and> C \<noteq> B" by blast
  {
    assume Q1: "Bet A B C"
    hence Q2: "Bet D E P"
      by (meson P1 bet_conga__bet Tarski_neutral_dimensionless_axioms)
    have Q3: "C InAngle A B C"
      by (simp add: P3 inangle3123)
    obtain X where Q4: "Bet D X F \<and> (X = E \<or> E Out X P)"
      using InAngle_def P1 by auto
    have "A B C CongA D E F"
    proof -
      {
        assume R1: "X = E"
        have R2: "Bet E F P \<or> Bet E P F"
        proof -
          have R3: "D \<noteq> E" using P2 by blast
          have "Bet D E F"
            using Col_def Col_perm P1 Q2 col_in_angle_out not_bet_and_out by blast
          have "Bet D E P" using Q2 by blast
          thus ?thesis using l5_2
            using R3 \<open>Bet D E F\<close> by blast
        qed
        have "A B C CongA D E F"
        proof -
          have "B Out A A" 
            using P3 out_trivial by auto
          moreover have "B Out C C" 
            using P3 out_trivial by auto
          moreover have "E Out D D" 
            using P2 out_trivial by auto
          moreover have "E Out F P" 
            using Out_def P2 R2 by blast
          ultimately show ?thesis 
            using P1 l11_10 by blast
        qed
      }
      {
        assume S1: "E Out X P"
        have S2: "E Out P F"
        proof -
          {
            assume "Bet E X P"
            hence "E Out P F"
            proof -
              have "Bet E X F"
                by (meson Bet_perm Q2 Q4 \<open>Bet E X P\<close> between_exchange3)
              thus ?thesis
                by (metis Bet_perm S1 bet2__out between_equality_2 between_trivial2 
                    out2_bet_out out_diff1)
            qed
          }
          {
            assume "Bet E P X"
            hence "Bet D P X" 
              using Q2 S1 out_diff2 outer_transitivity_between2 by blast
            hence "Bet D P F" 
              using Q4 between_exchange4 by blast
            hence "E Out P F" 
              by (metis P2 Q2 bet_out between_exchange3)
          }
          thus ?thesis
            using Out_def S1 \<open>Bet E X P \<Longrightarrow> E Out P F\<close> by blast
        qed
        hence "A B C CongA D E F"
          by (metis Bet_perm P2 Q1 Q2 bet_out__bet conga_line)
      }
      thus ?thesis
        using Q4 \<open>X = E \<Longrightarrow> A B C CongA D E F\<close> by blast
    qed
    hence "\<exists> Q. (C InAngle A B Q \<and> A B Q CongA D E F)"
      using conga_diff1 conga_diff2 inangle3123 by blast
  }
  {
    assume "B Out A C"
    obtain Q where "D E F CongA A B Q"
      by (metis P2 angle_construction_3)
    hence "\<exists> Q. (C InAngle A B Q \<and> A B Q CongA D E F)"
      by (metis conga_comm \<open>B Out A C\<close> conga_diff1 conga_sym out321__inangle)
  }
  {
    assume ZZ: "\<not> Col A B C"
    have Z1: "D \<noteq> E"
      using P2 by blast
    have Z2: "F \<noteq> E"
      using P2 by blast
    have Z3: "Bet D E F \<or> E Out D F \<or> \<not> Col D E F"
      using not_bet_out by blast
    {
      assume "Bet D E F"
      obtain Q where Z4: "Bet A B Q \<and> Cong B Q E F"
        using segment_construction by blast
      hence "\<exists> Q. (C InAngle A B Q \<and> A B Q CongA D E F)"
        by (metis InAngle_def P3 Z1 Z2 \<open>Bet D E F\<close> conga_line point_construction_different)
    }
    {
      assume "E Out D F"
      hence Z5: "E Out D P"
        using P1 in_angle_out by blast
      have "D E P CongA A B C"
        by (simp add: P1 conga_sym)
      hence Z6: "B Out A C" using l11_21_a Z5
        by blast
      hence "\<exists> Q. (C InAngle A B Q \<and> A B Q CongA D E F)"
        using \<open>B Out A C \<Longrightarrow> \<exists>Q. C InAngle A B Q \<and> A B Q CongA D E F\<close> by blast
    }
    {
      assume W1: "\<not> Col D E F"
      obtain Q where W2: "D E F CongA A B Q \<and> A B OS Q C"
        using W1 ZZ angle_construction_1 by blast
      obtain DD where W3: "E Out D DD \<and> Cong E DD B A"
        using P3 Z1 segment_construction_3 by force
      obtain FF where W4: "E Out F FF \<and> Cong E FF B Q"
        by (metis P2 W2 conga_diff56 segment_construction_3)
      hence W5: "P InAngle DD E FF"
        using Out_cases P1 P2 W3 l11_25 out_trivial by fastforce
      obtain X where W6: "Bet DD X FF \<and> (X = E \<or> E Out X P)"
        using InAngle_def W5 by presburger
      {
        assume W7: "X = E"
        have W8: "Bet D E F"
        proof -
          have W10: "E Out DD D"
            by (simp add: W3 l6_6)
          have "E Out FF F"
            by (simp add: W4 l6_6)
          thus ?thesis using W6 W7 W10 bet_out_out_bet by blast
        qed
        hence "\<exists> Q. (C InAngle A B Q \<and> A B Q CongA D E F)"
          using \<open>Bet D E F \<Longrightarrow> \<exists>Q. C InAngle A B Q \<and> A B Q CongA D E F\<close> by blast
      }
      {
        assume V1: "E Out X P"
        have "B \<noteq> C \<and> E \<noteq> X"
          using P3 V1 out_diff1 by blast
        then obtain CC where V2: "B Out C CC \<and> Cong B CC E X"
          using segment_construction_3 by blast
        hence V3: "A B CC CongA DD E X"
          using P1 P2 V1 W3 l11_10 l6_6 out_trivial by force
        have V4: "Cong A CC DD X"
        proof -
          have "Cong A B DD E"
            using W3 not_cong_4321 by blast
          thus ?thesis
            using V2 V3 cong2_conga_cong by blast
        qed
        have V5: "A B Q CongA DD E FF"
        proof -
          have U1: "D E F CongA A B Q"
            by (simp add: W2)
          hence U1A: "A B Q CongA D E F"
            by (simp add: conga_sym)
          have U2: "B Out A A"
            by (simp add: P3 out_trivial)
          have U3: "B Out Q Q"
            using W2 conga_diff56 out_trivial by blast
          have U4: "E Out DD D"
            using W3 l6_6 by blast
          have "E Out FF F"
            by (simp add: W4 l6_6)
          thus ?thesis using l11_10
            using U1A U2 U3 U4 by blast
        qed
        hence V6: "Cong A Q DD FF"
          using Cong_perm W3 W4 cong2_conga_cong by blast
        have "CC B Q CongA X E FF"
        proof -
          have U1: "B A OS CC Q"
            by (metis V2 W2 col124__nos invert_one_side one_side_symmetry 
                one_side_transitivity out_one_side)
          have U2: "E DD OS X FF"
          proof -
            have "\<not> Col DD E FF"
              by (metis Col_perm OS_def TS_def U1 V5 ncol_conga_ncol)
            hence "\<not> Col E DD X"
              by (metis Col_def V2 V4 W6 ZZ cong_identity l6_16_1 os_distincts out_one_side)
            hence "DD E OS X FF"
              by (metis Col_perm W6 bet_out not_col_distincts one_side_reflexivity 
                  out_out_one_side)
            thus ?thesis
              by (simp add: invert_one_side)
          qed
          have "CC B A CongA X E DD"
            by (simp add: V3 conga_comm)
          thus ?thesis
            using U1 U2 V5 l11_22b by blast
        qed
        hence V8: "Cong CC Q X FF"
          using V2 W4 cong2_conga_cong cong_commutativity not_cong_3412 by blast
        have V9: "CC InAngle A B Q"
        proof -
          have T2: "Q \<noteq> B"
            using W2 conga_diff56 by blast
          have T3: "CC \<noteq> B"
            using V2 out_distinct by blast
          have "Bet A CC Q"
          proof -
            have T4: "DD X FF Cong3 A CC Q"
              using Cong3_def V4 V6 V8 not_cong_3412 by blast
            thus ?thesis
              using W6 l4_6 by blast
          qed
          hence "\<exists> X0. Bet A X0 Q \<and> (X0 = B \<or> B Out X0 CC)"
            using out_trivial by blast
          thus ?thesis
            by (simp add: InAngle_def P3 T2 T3)
        qed
        hence "C InAngle A B Q"
          using V2 inangle_distincts l11_25 out_trivial by blast
        hence "\<exists> Q. (C InAngle A B Q \<and> A B Q CongA D E F)"
          using W2 conga_sym by blast
      }
      hence "\<exists> Q. (C InAngle A B Q \<and> A B Q CongA D E F)"
        using W6 \<open>X = E \<Longrightarrow> \<exists>Q. C InAngle A B Q \<and> A B Q CongA D E F\<close> by blast
    }
    hence "\<exists> Q. (C InAngle A B Q \<and> A B Q CongA D E F)"
      using Z3 \<open>E Out D F \<Longrightarrow> \<exists>Q. C InAngle A B Q \<and> A B Q CongA D E F\<close> 
        \<open>Bet D E F \<Longrightarrow> \<exists>Q. C InAngle A B Q \<and> A B Q CongA D E F\<close> by blast
  }
  thus ?thesis
    using \<open>B Out A C \<Longrightarrow> \<exists>Q. C InAngle A B Q \<and> A B Q CongA D E F\<close> 
      \<open>Bet A B C \<Longrightarrow> \<exists>Q. C InAngle A B Q \<and> A B Q CongA D E F\<close> not_bet_out by blast
qed

lemma in_angle_line:
  assumes "P \<noteq> B" and
    "A \<noteq> B" and
    "C \<noteq> B" and
    "Bet A B C"
  shows "P InAngle A B C"
  using InAngle_def assms(1) assms(2) assms(3) assms(4) by auto

lemma l11_29_b:
  assumes "\<exists> Q. (C InAngle A B Q \<and> A B Q CongA D E F)"
  shows "A B C LeA D E F"
proof -
  obtain Q where P1: "C InAngle A B Q \<and> A B Q CongA D E F"
    using assms by blast
  obtain X where P2: "Bet A X Q \<and> (X = B \<or> B Out X C)"
    using InAngle_def P1 by auto
  {
    assume P2A: "X = B"
    obtain P where P3: "A B C CongA D E P"
      using angle_construction_3 assms conga_diff45 inangle_distincts by fastforce
    have "P InAngle D E F"
    proof -
      have O1: "Bet D E F"
        by (metis P1 P2 bet_conga__bet P2A)
      have O2: "P \<noteq> E"
        using P3 conga_diff56 by auto
      have O3: "D \<noteq> E"
        using P3 conga_diff45 by auto
      have "F \<noteq> E"
        using P1 conga_diff56 by blast
      thus ?thesis using in_angle_line
        by (simp add: O1 O2 O3)
    qed
    hence "A B C LeA D E F"
      using LeA_def P3 by blast
  }
  {
    assume G1: "B Out X C"
    obtain DD where G2: "E Out D DD \<and> Cong E DD B A"
      by (metis assms conga_diff1 conga_diff45 segment_construction_3)
    have G3: "D \<noteq> E \<and> DD \<noteq> E"
      using G2 out_diff1 out_diff2 by blast
    obtain FF where G3G: "E Out F FF \<and> Cong E FF B Q"
      by (metis P1 conga_diff56 inangle_distincts segment_construction_3)
    hence G3A: "F \<noteq> E"
      using out_diff1 by blast
    have G3B: "FF \<noteq> E"
      using G3G out_distinct by blast
    have G4: "Bet A B C \<or> B Out A C \<or> \<not> Col A B C"
      using not_bet_out by blast
    {
      assume G5: "Bet A B C"
      have G6: "F InAngle D E F"
        by (simp add: G3 G3A inangle3123)
      have "A B C CongA A B Q" 
      proof -
        have "A \<noteq> B" 
          using P1 conga_diff1 by blast
        hence "B Out A A" 
          using out_trivial by blast
        moreover 
        have "B Out Q C"
        proof -
          {
            assume "Bet B X C"
            hence "Bet A B X" 
              using G5 between_inner_transitivity by blast
            hence "Bet B X Q" 
              using P2 between_exchange3 by blast
            hence "B Out Q C" 
              by (metis G1 bet_out l6_6 l6_7 out_diff1)
          }
          moreover
          {
            assume "Bet B C X"
            hence "Bet B X Q" 
              by (metis G5 P2 between_exchange3 l6_3_1 outer_transitivity_between)
            hence "Bet B C Q" 
              using \<open>Bet B C X\<close> between_exchange4 by blast
            hence "B Out Q C" 
              by (metis G1 Out_def between_identity)
          }
          ultimately show ?thesis 
            using G1 Out_def by blast
        qed
        ultimately show ?thesis 
          by (simp add: out2__conga)
      qed
      hence "A B C CongA D E F" 
        using P1 conga_trans by blast
      hence  "A B C LeA D E F"
        using G6 LeA_def by blast
    }
    {
      assume G8: "B Out A C"
      have G9: "D InAngle D E F"
        by (simp add: G3 G3A inangle1123)
      have "A B C CongA D E D"
        by (simp add: G3 G8 l11_21_b out_trivial)
      hence  "A B C LeA D E F" using G9 LeA_def by blast
    }
    {
      assume R1: "\<not> Col A B C"
      have R2: "Bet A B Q \<or> B Out A Q \<or> \<not> Col A B Q"
        using not_bet_out by blast
      {
        assume R3: "Bet A B Q"
        obtain P where R4: "A B C CongA D E P"
          by (metis G3 LeA_def \<open>Bet A B C \<Longrightarrow> A B C LeA D E F\<close> angle_construction_3 
              not_bet_distincts)
        have R5: "P InAngle D E F"
        proof -
          have R6: "P \<noteq> E"
            using R4 conga_diff56 by auto
          have "Bet D E F"
            by (metis P1 R3 bet_conga__bet)
          thus ?thesis
            by (simp add: R6 G3 G3A in_angle_line)
        qed
        hence  "A B C LeA D E F" using R4 R5 LeA_def by blast
      }
      {
        assume S1: "B Out A Q"
        have S2: "B Out A C"
          using G1 P2 S1 l6_7 out_bet_out_1 by blast
        have S3: "Col A B C"
          by (simp add: Col_perm S2 out_col)
        hence  "A B C LeA D E F"
          using R1 by blast
      }
      {
        assume S3B: "\<not> Col A B Q"
        obtain P where S4: "A B C CongA D E P \<and> D E OS P F"
          by (meson P1 R1 ncol_conga_ncol Tarski_neutral_dimensionless_axioms 
              S3B angle_construction_1)
        obtain PP where S4A: "E Out P PP \<and> Cong E PP B X"
          by (metis G1 S4 os_distincts out_diff1 segment_construction_3)
        have S5: "P InAngle D E F"
        proof -
          have "PP InAngle DD E FF"
          proof -
            have Z3: "PP \<noteq> E"
              using S4A l6_3_1 by blast
            have Z4: "Bet DD PP FF"
            proof -
              have L1: "C B Q CongA P E F"
              proof -
                have K1: "B A OS C Q"
                  using Col_perm P1 R1 S3B in_angle_one_side invert_one_side by blast
                have K2: "E D OS P F"
                  by (simp add: S4 invert_one_side)
                have "C B A CongA P E D"
                  by (simp add: S4 conga_comm)
                thus ?thesis
                  using K1 K2 P1 l11_22b by auto
              qed
              have L2: "Cong DD FF A Q"
              proof -
                have "DD E FF CongA A B Q"
                proof -
                  have L3: "A B Q CongA D E F"
                    by (simp add: P1)
                  hence L3A: "D E F CongA A B Q"
                    using conga_sym by blast
                  have L4: "E Out DD D"
                    using G2 Out_cases by auto
                  have L5: "E Out FF F"
                    using G3G Out_cases by blast
                  have L6: "B Out A A"
                    using S3B not_col_distincts out_trivial by auto
                  have "B Out Q Q"
                    by (metis S3B not_col_distincts out_trivial)
                  thus ?thesis using L3A L4 L5 L6 l11_10
                    by blast
                qed
                have L2B: "Cong DD E A B"
                  using Cong_perm G2 by blast
                have "Cong E FF B Q"
                  by (simp add: G3G)
                thus ?thesis
                  using L2B \<open>DD E FF CongA A B Q\<close> cong2_conga_cong by auto
              qed
              have L8: "Cong A X DD PP"
              proof -
                have L9: "A B X CongA DD E PP"
                proof -
                  have L9B: "B Out A A"
                    using S3B not_col_distincts out_trivial by blast
                  have L9D: "E Out D D "
                    using G3 out_trivial by auto
                  have "E Out PP P"
                    using Out_cases S4A by blast
                  thus ?thesis using l11_10 S4 L9B G1 L9D
                    using G2 Out_cases by blast
                qed
                have L10: "Cong A B DD E"
                  using G2 not_cong_4321 by blast
                have "Cong B X E PP"
                  using Cong_perm S4A by blast
                thus ?thesis
                  using L10 L9 cong2_conga_cong by blast
              qed
              have "A X Q Cong3 DD PP FF"
              proof -
                have L12B: "Cong A Q DD FF"
                  using L2 not_cong_3412 by blast
                have "Cong X Q PP FF"
                proof -
                  have L13A: "X B Q CongA PP E FF"
                  proof -
                    have L13AC: "B Out Q Q"
                      by (metis S3B col_trivial_2 out_trivial)
                    have L13AD: "E Out PP P"
                      by (simp add: S4A l6_6)
                    have "E Out FF F"
                      by (simp add: G3G l6_6)
                    thus ?thesis
                      using L1 G1 L13AC L13AD l11_10 by blast
                  qed
                  have L13B: "Cong X B PP E"
                    using S4A not_cong_4321 by blast
                  have "Cong B Q E FF"
                    using G3G not_cong_3412 by blast
                  thus ?thesis
                    using L13A L13B cong2_conga_cong by auto
                qed
                thus ?thesis
                  by (simp add: Cong3_def L12B L8)
              qed
              thus ?thesis using P2 l4_6 by blast
            qed
            have "PP = E \<or> E Out PP PP"
              using out_trivial by auto
            thus ?thesis
              using InAngle_def G3 G3B Z3 Z4 by auto
          qed
          thus ?thesis
            using G2 G3G S4A l11_25 by blast
        qed
        hence  "A B C LeA D E F"
          using S4 LeA_def by blast
      }
      hence  "A B C LeA D E F"
        using R2 \<open>B Out A Q \<Longrightarrow> A B C LeA D E F\<close> \<open>Bet A B Q \<Longrightarrow> A B C LeA D E F\<close> by blast
    }
    hence "A B C LeA D E F"
      using G4 \<open>B Out A C \<Longrightarrow> A B C LeA D E F\<close> \<open>Bet A B C \<Longrightarrow> A B C LeA D E F\<close> by blast
  }
  thus ?thesis
    using P2 \<open>X = B \<Longrightarrow> A B C LeA D E F\<close> by blast
qed

lemma bet_in_angle_bet:
  assumes "Bet A B P" and
    "P InAngle A B C"
  shows "Bet A B C"
  by (metis (no_types) Col_def Col_perm assms(1) assms(2) col_in_angle_out not_bet_and_out)

lemma lea_line:
  assumes "Bet A B P" and
    "A B P LeA A B C"
  shows "Bet A B C"
  by (metis bet_conga__bet l11_29_a assms(1) assms(2) bet_in_angle_bet)

lemma eq_conga_out:
  assumes "A B A CongA D E F"
  shows "E Out D F"
  by (metis CongA_def assms l11_21_a out_trivial)

lemma out_conga_out:
  assumes "B Out A C" and
    "A B C CongA D E F"
  shows "E Out D F"
  using assms(1) assms(2) l11_21_a by blast

lemma conga_ex_cong3:
  assumes "A B C CongA A' B' C'"
  shows "\<exists> AA CC. ((B Out A AA \<and> B Out C CC) \<longrightarrow> AA B CC Cong3 A' B' C')"
  using out_diff2 by blast

lemma conga_preserves_in_angle:
  assumes "A B C CongA A' B' C'" and
    "A B I CongA A' B' I'" and
    "I InAngle A B C" and "A' B' OS I' C'"
  shows "I' InAngle A' B' C'"
proof -
  have P1: "A \<noteq> B"
    using assms(1) conga_diff1 by auto
  have P2: "B \<noteq> C"
    using assms(1) conga_diff2 by blast
  have P3: "A' \<noteq> B'"
    using assms(1) conga_diff45 by auto
  have P4: "B' \<noteq> C'"
    using assms(1) conga_diff56 by blast
  have P5: "I \<noteq> B"
    using assms(2) conga_diff2 by auto
  have P6: "I' \<noteq> B'"
    using assms(2) conga_diff56 by blast
  have P7: "Bet A B C \<or> B Out A C \<or> \<not> Col A B C"
    using l6_4_2 by blast
  {
    assume "Bet A B C"
    have Q1: "Bet A' B' C'"
      using \<open>Bet A B C\<close> assms(1) assms(4) bet_col col124__nos col_conga_col by blast
    hence "I' InAngle A' B' C'"
      using assms(4) bet_col col124__nos by auto
  }
  {
    assume "B Out A C"
    hence "I' InAngle A' B' C'"
      by (metis P4 assms(2) assms(3) in_angle_out l11_21_a out321__inangle)
  }
  {
    assume Z1: "\<not> Col A B C"
    have Q2: "Bet A B I \<or> B Out A I \<or> \<not> Col A B I"
      by (simp add: or_bet_out)
    {
      assume "Bet A B I"
      hence "I' InAngle A' B' C'"
        using \<open>Bet A B C \<Longrightarrow> I' InAngle A' B' C'\<close> assms(3) bet_in_angle_bet by blast
    }
    {
      assume "B Out A I"
      hence "I' InAngle A' B' C'"
        using P4 assms(2) l11_21_a out321__inangle by auto
    }
    {
      assume "\<not> Col A B I"
      obtain AA' where Q3: "B' Out A' AA' \<and> Cong B' AA' B A"
        using P1 P3 segment_construction_3 by presburger
      obtain CC' where Q4: "B' Out C' CC' \<and> Cong B' CC' B C"
        using P2 P4 segment_construction_3 by presburger
      obtain J where Q5: "Bet A J C \<and> (J = B \<or> B Out J I)"
        using InAngle_def assms(3) by auto
      have Q6: "B \<noteq> J"
        using Q5 Z1 bet_col by auto
      have Q7: "\<not> Col A B J"
        using Q5 Q6 \<open>\<not> Col A B I\<close> col_permutation_2 col_transitivity_1 out_col by blast
      have "\<not> Col A' B' I'"
        by (metis assms(4) col123__nos)
      hence "\<exists> C'. (A B J CongA A' B' C' \<and> A' B' OS C' I')"
        using Q7 angle_construction_1 by blast
      then obtain J' where Q8: "A B J CongA A' B' J' \<and> A' B' OS J' I'" by blast
      have Q9: "B' \<noteq> J'"
        using Q8 conga_diff56 by blast
      obtain JJ' where Q10: "B' Out J' JJ' \<and> Cong B' JJ' B J"
        using segment_construction_3 Q6 Q9 by blast
      have Q11: "\<not> Col A' B' J'"
        using Q8 col123__nos by blast
      have Q12: "A' \<noteq> JJ'"
        by (metis Col_perm Q10 Q11 out_col)
      have Q13: "B' \<noteq> JJ'"
        using Q10 out_distinct by blast
      have Q14: "\<not> Col A' B' JJ'"
        using Col_perm Q10 Q11 Q13 l6_16_1 out_col by blast
      have Q15: "A B C CongA AA' B' CC'"
      proof -
        have T2: "C \<noteq> B" using P2 by auto
        have T3: "AA' \<noteq> B'"
          using Out_def Q3 by blast
        have T4: "CC' \<noteq> B'"
          using Q4 out_distinct by blast
        have T5: "\<forall> A' C' D' F'. (B Out A' A \<and> B Out C' C \<and> B' Out D' AA' \<and>
    B' Out F' CC' \<and>Cong B A' B' D' \<and> Cong B C' B' F' 
    \<longrightarrow> Cong A' C' D' F')"
          using Q3 Q4 l11_4_1 Tarski_neutral_dimensionless_axioms assms(1) l6_6 l6_7 
          by blast
        thus ?thesis using P1 T2 T3 T4 l11_4_2 by blast
      qed
      have Q16: "A' B' J' CongA A' B' JJ'"
      proof -
        have P9: "B' Out A' A'"
          by (simp add: P3 out_trivial)
        have "B' Out JJ' J'"
          using Out_cases Q10 by auto
        thus ?thesis
          using l11_10
          by (simp add: P9 out2__conga)
      qed
      have Q17: "B' Out I' JJ' \<or> A' B' TS I' JJ'"
      proof -
        have "Coplanar A' I' B' J'"
          by (metis Q8 ncoplanar_perm_3 os__coplanar)
        hence "Coplanar A' I' B' JJ'"
          using Q10 Q9 col_cop__cop out_col by blast
        hence R1: "Coplanar A' B' I' JJ'" using coplanar_perm_2
          by blast
        have "A' B' I' CongA A' B' JJ'"
        proof -
          have R2: "A' B' I' CongA A B I"
            by (simp add: assms(2) conga_sym)
          have "A B I CongA A' B' JJ'"
          proof -
            have f1: "\<forall>p pa pb. \<not> p Out pa pb \<and> \<not> p Out pb pa \<or> p Out pa pb"
              using Out_cases by blast
            hence f2: "B' Out JJ' J'"
              using Q10 by blast
            have "B Out J I"
              by (metis Q5 Q6)
            thus ?thesis
              using f2 f1 by (meson P3 Q8 l11_10 Tarski_neutral_dimensionless_axioms 
                  \<open>\<not> Col A B I\<close> col_one_side_out col_trivial_2 one_side_reflexivity out_trivial)
          qed
          thus ?thesis
            using R2 conga_trans by blast
        qed
        thus ?thesis using R1 conga_cop__or_out_ts by blast
      qed
      {
        assume Z2: "B' Out I' JJ'"
        have Z3: "J B C CongA J' B' C'"
        proof -
          have R1: "B A OS J C"
            by (metis Q5 Q7 Z1 bet_out invert_one_side not_col_distincts out_one_side)
          have R2: "B' A' OS J' C'"
            by (meson Q10 Z2 assms(4) invert_one_side l6_6 one_side_symmetry out_out_one_side)
          have  "J B A CongA J' B' A'"
            using Q8 conga_comm by blast
          thus ?thesis using assms(1) R1 R2 l11_22b by blast
        qed
        hence "I' InAngle A' B' C'"
        proof -
          have "A J C Cong3 AA' JJ' CC'"
          proof -
            have R8: "Cong A J AA' JJ'"
            proof -
              have R8A: "A B J CongA AA' B' JJ'"
              proof -
                have R8AB: "B Out A A"
                  by (simp add: P1 out_trivial)
                have R8AC: "B Out J I"
                  using Q5 Q6 by auto
                have R8AD: "B' Out AA' A'"
                  using Out_cases Q3 by auto
                have "B' Out JJ' I'"
                  using Out_cases Z2 by blast
                thus ?thesis
                  using assms(2) R8AB R8AC R8AD l11_10 by blast
              qed
              have R8B: "Cong A B AA' B'"
                using Q3 not_cong_4321 by blast
              have R8C: "Cong B J B' JJ'"
                using Q10 not_cong_3412 by blast
              thus ?thesis
                using R8A R8B cong2_conga_cong by blast
            qed
            have LR8A: "Cong A C AA' CC'"
              using Q15 Q3 Q4 cong2_conga_cong cong_4321 cong_symmetry by blast
            have "Cong J C JJ' CC'"
            proof -
              have K1:"B' Out JJ' J'"
                using Out_cases Q10 by auto
              have "B' Out CC' C'"
                using Out_cases Q4 by auto
              hence "J' B' C' CongA JJ' B' CC'" using K1
                by (simp add: out2__conga)
              hence LR9A: "J B C CongA JJ' B' CC'"
                using Z3 conga_trans by blast                                     
              have LR9B: "Cong J B JJ' B'"
                using Q10 not_cong_4321 by blast
              have "Cong B C B' CC'"
                using Q4 not_cong_3412 by blast
              thus ?thesis
                using LR9A LR9B cong2_conga_cong by blast
            qed
            thus ?thesis using R8 LR8A
              by (simp add: Cong3_def)
          qed
          hence R10: "Bet AA' JJ' CC'" using Q5 l4_6 by blast
          have "JJ' InAngle AA' B' CC'"
          proof -
            have R11: "AA' \<noteq> B'"
              using Out_def Q3 by auto
            have R12: "CC' \<noteq> B'"
              using Out_def Q4 by blast
            have "Bet AA' JJ' CC' \<and> (JJ' = B' \<or> B' Out JJ' JJ')"
              using R10 out_trivial by auto
            thus ?thesis
              using InAngle_def Q13 R11 R12 by auto
          qed
          thus ?thesis
            using Z2 Q3 Q4 l11_25 by blast
        qed
      }
      {
        assume X1: "A' B' TS I' JJ'"
        have "A' B' OS I' J'"
          by (simp add: Q8 one_side_symmetry)
        hence X2: "B' A' OS I' JJ'"
          using Q10 invert_one_side out_out_one_side by blast
        hence "I' InAngle A' B' C'"
          using X1 invert_one_side l9_9 by blast
      }
      hence "I' InAngle A' B' C'"
        using Q17 \<open>B' Out I' JJ' \<Longrightarrow> I' InAngle A' B' C'\<close> by blast
    }
    hence "I' InAngle A' B' C'"
      using Q2 \<open>B Out A I \<Longrightarrow> I' InAngle A' B' C'\<close> \<open>Bet A B I \<Longrightarrow> I' InAngle A' B' C'\<close> by blast
  }
  thus ?thesis
    using P7 \<open>B Out A C \<Longrightarrow> I' InAngle A' B' C'\<close> \<open>Bet A B C \<Longrightarrow> I' InAngle A' B' C'\<close> by blast
qed

lemma l11_30:
  assumes "A B C LeA D E F" and
    "A B C CongA A' B' C'" and
    "D E F CongA D' E' F'"
  shows "A' B' C' LeA D' E' F'"
proof -
  obtain Q where P1: "C InAngle A B Q \<and> A B Q CongA D E F"
    using assms(1) l11_29_a by blast
  have P1A: "C InAngle A B Q" 
    using P1 by simp
  have P1B: "A B Q CongA D E F" 
    using P1 by simp
  have P2: "A \<noteq> B"
    using P1A inangle_distincts by auto
  have P3: "C \<noteq> B"
    using P1A inangle_distincts by blast
  have P4: "A' \<noteq> B'"
    using CongA_def assms(2) by blast
  have P5: "C' \<noteq> B'"
    using CongA_def assms(2) by auto
  have P6: "D \<noteq> E"
    using CongA_def P1B by blast
  have P7: "F \<noteq> E"
    using CongA_def P1B by blast
  have P8: "D' \<noteq> E'"
    using CongA_def assms(3) by blast
  have P9: "F' \<noteq> E'"
    using CongA_def assms(3) by blast
  have P10: "Bet A' B' C' \<or> B' Out A' C' \<or> \<not> Col A' B' C'"
    using or_bet_out by blast
  {
    assume "Bet A' B' C'"
    hence "\<exists> Q'. (C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F')"
      by (metis P1 P4 P5 P8 P9 assms(2) assms(3) bet_conga__bet bet_in_angle_bet 
          conga_line conga_sym inangle3123)
  }
  {
    assume R1: "B' Out A' C'"
    obtain Q' where R2: "D' E' F' CongA A' B' Q'"
      using P4 P8 P9 angle_construction_3 by blast
    hence "C' InAngle A' B' Q'"
      using col_in_angle P1 R1 conga_diff56 out321__inangle by auto
    hence "\<exists> Q'. (C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F')"
      using R2 conga_sym by blast
  }
  {
    assume R3: "\<not> Col A' B' C'"
    have R3A: "Bet D' E' F' \<or> E' Out D' F' \<or> \<not> Col D' E' F'"
      using or_bet_out by blast
    {
      assume "Bet D' E' F'"
      have "\<exists> Q'. (C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F')"
        by (metis P4 P5 P8 P9 \<open>Bet D' E' F'\<close> conga_line in_angle_line 
            point_construction_different)
    }
    {
      assume R4A: "E' Out D' F'"
      obtain Q' where R4: "D' E' F' CongA A' B' Q'"
        using P4 P8 P9 angle_construction_3 by blast
      hence R5: "B' Out A' Q'" using out_conga_out R4A by blast
      have R6: "A B Q CongA D' E' F'"
        using P1 assms(3) conga_trans by blast
      hence R7: "B Out A Q" using out_conga_out R4A R4
        using conga_sym by blast
      have R8: "B Out A C"
        using P1A R7 in_angle_out by blast
      hence R9: "B' Out A' C'" using out_conga_out assms(2)
        by blast
      have "\<exists> Q'. (C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F')"
        by (simp add: R9 \<open>B' Out A' C' \<Longrightarrow> \<exists>Q'. C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F'\<close>)
    }
    {
      assume "\<not> Col D' E' F'"
      obtain QQ where S1: "D' E' F' CongA A' B' QQ \<and> A' B' OS QQ C'"
        using R3 \<open>\<not> Col D' E' F'\<close> angle_construction_1 by blast
      have S1A: "A B Q CongA A' B' QQ" using S1
        using P1 assms(3) conga_trans by blast
      have "A' B' OS C' QQ" using S1
        by (simp add: S1 one_side_symmetry)
      hence S2: "C' InAngle A' B' QQ" using conga_preserves_in_angle S1A
        using P1A assms(2) by blast
      have S3: "A' B' QQ CongA D' E' F'"
        by (simp add: S1 conga_sym)
      hence "\<exists> Q'. (C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F')"
        using S2 by auto
    }
    hence "\<exists> Q'. (C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F')"
      using R3A \<open>E' Out D' F' \<Longrightarrow> \<exists>Q'. C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F'\<close> 
        \<open>Bet D' E' F' \<Longrightarrow> \<exists>Q'. C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F'\<close> by blast
  }
  thus ?thesis using l11_29_b
    using P10 \<open>B' Out A' C' \<Longrightarrow> \<exists>Q'. C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F'\<close> 
      \<open>Bet A' B' C' \<Longrightarrow> \<exists>Q'. C' InAngle A' B' Q' \<and> A' B' Q' CongA D' E' F'\<close> by blast
qed

lemma l11_31_1:
  assumes "B Out A C" and
    "D \<noteq> E" and
    "F \<noteq> E"
  shows "A B C LeA D E F"
  by (metis (full_types) LeA_def assms(1) assms(2) assms(3) l11_21_b out321__inangle 
      segment_construction_3)

lemma l11_31_2:
  assumes "A \<noteq> B" and
    "C \<noteq> B" and
    "D \<noteq> E" and
    "F \<noteq> E" and
    "Bet D E F"
  shows "A B C LeA D E F"
  by (metis LeA_def angle_construction_3 assms(1) assms(2) assms(3) assms(4) assms(5) 
      conga_diff56 in_angle_line)

lemma lea_refl:
  assumes "A \<noteq> B" and
    "C \<noteq> B"
  shows "A B C LeA A B C"
  by (meson assms(1) assms(2) conga_refl l11_29_b out341__inangle out_trivial)

lemma conga__lea:
  assumes "A B C CongA D E F"
  shows "A B C LeA D E F"
  by (metis conga_diff1 conga_diff2 l11_30 assms conga_refl lea_refl)

lemma conga__lea456123:
  assumes "A B C CongA D E F"
  shows "D E F LeA A B C"
  by (simp add: conga__lea Tarski_neutral_dimensionless_axioms assms conga_sym)

lemma lea_left_comm:
  assumes "A B C LeA D E F"
  shows "C B A LeA D E F"
  by (metis assms conga_pseudo_refl conga_refl l11_30 lea_distincts)

lemma lea_right_comm:
  assumes "A B C LeA D E F"
  shows "A B C LeA F E D"
  by (meson assms conga_right_comm l11_29_a l11_29_b)

lemma lea_comm:
  assumes"A B C LeA D E F"
  shows "C B A LeA F E D"
  using assms lea_left_comm lea_right_comm by blast

lemma lta_left_comm:
  assumes "A B C LtA D E F"
  shows "C B A LtA D E F"
  by (meson LtA_def conga_left_comm lea_left_comm assms)

lemma lta_right_comm:
  assumes "A B C LtA D E F"
  shows "A B C LtA F E D"
  by (meson LtA_def conga_comm lea_comm lta_left_comm assms)

lemma lta_comm:
  assumes "A B C LtA D E F"
  shows "C B A LtA F E D"
  using assms lta_left_comm lta_right_comm by blast

lemma lea_out4__lea:
  assumes "A B C LeA D E F" and
    "B Out A A'" and
    "B Out C C'" and
    "E Out D D'" and
    "E Out F F'"
  shows "A' B C' LeA D' E F'"
  using assms(1) assms(2) assms(3) assms(4) assms(5) l11_30 l6_6 out2__conga by auto


lemma lea121345:
  assumes "A \<noteq> B" and
    "C \<noteq> D" and
    "D \<noteq> E"
  shows "A B A LeA C D E"
  using assms(1) assms(2) assms(3) l11_31_1 out_trivial by auto

lemma inangle__lea:
  assumes "P InAngle A B C"
  shows "A B P LeA A B C"
  by (metis l11_29_b assms conga_refl inangle_distincts)

lemma inangle__lea_1:
  assumes "P InAngle A B C"
  shows "P B C LeA A B C"
  by (simp add: inangle__lea lea_comm assms l11_24)

lemma inangle__lta:
  assumes "\<not> Col P B C" and
    "P InAngle A B C"
  shows "A B P LtA A B C"
  by (metis LtA_def TS_def conga_cop__or_out_ts conga_os__out inangle__lea ncol_conga_ncol 
      assms(1) assms(2) col_one_side_out col_trivial_3 in_angle_one_side inangle__coplanar 
      invert_two_sides l11_21_b ncoplanar_perm_12 not_col_permutation_3 one_side_reflexivity)

lemma in_angle_trans:
  assumes "C InAngle A B D" and
    "D InAngle A B E"
  shows "C InAngle A B E"
proof -
  obtain CC where P1: "Bet A CC D \<and> (CC = B \<or> B Out CC C)"
    using InAngle_def assms(1) by auto
  obtain DD where P2: "Bet A DD E \<and> (DD = B \<or> B Out DD D)"
    using InAngle_def assms(2) by auto
  hence P3: "Bet A DD E" by simp
  have P4: "DD = B \<or> B Out DD D" using P2 by simp
  {
    assume "CC = B \<and> DD = B"
    hence "C InAngle A B E"
      using InAngle_def P2 assms(1) assms(2) by auto
  }
  {
    assume "CC = B \<and> B Out DD D"
    hence "C InAngle A B E"
      by (metis InAngle_def P1 assms(1) assms(2) bet_in_angle_bet)
  }
  {
    assume "B Out CC C \<and> DD = B"
    hence "C InAngle A B E"
      by (metis Out_def P2 assms(2) in_angle_line inangle_distincts)
  }
  {
    assume P3: "B Out CC C \<and> B Out DD D"
    hence P3A: "B Out CC C" by simp
    have P3B: "B Out DD D" using P3 by simp
    have "C InAngle A B DD"
      using P3 assms(1) inangle_distincts l11_25 out_trivial by blast
    then obtain CC' where T1: "Bet A CC' DD \<and> (CC' = B \<or> B Out CC' C)"
      using InAngle_def by auto
    {
      assume "CC' = B"
      hence "C InAngle A B E"
        by (metis P2 P3 T1 assms(2) between_exchange4 in_angle_line inangle_distincts 
            out_diff2)
    }
    {
      assume "B Out CC' C"
      hence "C InAngle A B E"
        by (metis InAngle_def P2 T1 assms(1) assms(2) between_exchange4)
    }
    hence "C InAngle A B E"
      using T1 \<open>CC' = B \<Longrightarrow> C InAngle A B E\<close> by blast
  }
  thus ?thesis
    using P1 P2 \<open>B Out CC C \<and> DD = B \<Longrightarrow> C InAngle A B E\<close> 
      \<open>CC = B \<and> B Out DD D \<Longrightarrow> C InAngle A B E\<close> 
      \<open>CC = B \<and> DD = B \<Longrightarrow> C InAngle A B E\<close> by blast
qed

lemma lea_trans:
  assumes "A B C LeA A1 B1 C1" and
    "A1 B1 C1 LeA A2 B2 C2"
  shows "A B C LeA A2 B2 C2"
proof -
  obtain P1 where T1: "P1 InAngle A1 B1 C1 \<and> A B C CongA A1 B1 P1"
    using LeA_def assms(1) by auto
  obtain P2 where T2: "P2 InAngle A2 B2 C2 \<and> A1 B1 C1 CongA A2 B2 P2"
    using LeA_def assms(2) by blast
  have T3: "A \<noteq> B"
    using CongA_def T1 by auto
  have T4: "C \<noteq> B"
    using CongA_def T1 by blast
  have T5: "A1 \<noteq> B1"
    using T1 inangle_distincts by blast
  have T6: "C1 \<noteq> B1"
    using T1 inangle_distincts by blast
  have T7: "A2 \<noteq> B2"
    using T2 inangle_distincts by blast
  have T8: "C2 \<noteq> B2"
    using T2 inangle_distincts by blast
  have T9: "Bet A B C  \<or> B Out A C \<or> \<not> Col A B C"
    using not_out_bet by auto
  {
    assume "Bet A B C"
    hence "A B C LeA A2 B2 C2"
      by (metis T1 T2 T3 T4 T7 T8 bet_conga__bet bet_in_angle_bet l11_31_2)
  }
  {
    assume "B Out A C"
    hence "A B C LeA A2 B2 C2"
      by (simp add: T7 T8 l11_31_1)
  }
  {
    assume H1: "\<not> Col A B C"
    have T10: "Bet A2 B2 C2 \<or> B2 Out A2 C2 \<or> \<not> Col A2 B2 C2"
      using not_out_bet by auto
    {
      assume "Bet A2 B2 C2"
      hence "A B C LeA A2 B2 C2"
        by (simp add: T3 T4 T7 T8 l11_31_2)
    }
    {
      assume T10A: "B2 Out A2 C2"
      have "B Out A C"
      proof -
        have "B1 Out A1 P1"
        proof -
          have "B1 Out A1 C1" using T2 conga_sym T2 T10A in_angle_out out_conga_out by blast
          thus ?thesis using T1 in_angle_out by blast
        qed
        thus ?thesis using T1 conga_sym l11_21_a by blast
      qed
      hence "A B C LeA A2 B2 C2"
        using \<open>B Out A C \<Longrightarrow> A B C LeA A2 B2 C2\<close> by blast
    }
    {
      assume T12: "\<not> Col A2 B2 C2"
      obtain P where T13: "A B C CongA A2 B2 P \<and> A2 B2 OS P C2"
        using T12 H1 angle_construction_1 by blast
      have T14: "A2 B2 OS P2 C2"
      proof -
        have "\<not> Col B2 A2 P2"
        proof -
          have "B2 \<noteq> A2"
            using T7 by auto
          {
            assume H2: "P2 = A2"
            have "A2 B2 A2 CongA A1 B1 C1"
              using T2 H2 conga_sym by blast
            hence "B1 Out A1 C1"
              using eq_conga_out by blast
            hence "B1 Out A1 P1"
              using T1 in_angle_out by blast
            hence "B Out A C"
              using T1 conga_sym out_conga_out by blast
            hence False
              using Col_cases H1 out_col by blast
          }
          hence "P2 \<noteq> A2" by blast
          have "Bet A2 B2 P2 \<or> B2 Out A2 P2 \<or> \<not> Col A2 B2 P2"
            using not_out_bet by auto
          {
            assume H4: "Bet A2 B2 P2"
            hence "Bet A2 B2 C2"
              using T2 bet_in_angle_bet by blast
            hence "Col B2 A2 P2 \<longrightarrow> False"
              using Col_def T12 by blast
            hence "\<not> Col B2 A2 P2"
              using H4 bet_col not_col_permutation_4 by blast
          }
          {
            assume H5: "B2 Out A2 P2"
            hence "B1 Out A1 C1"
              using T2 conga_sym out_conga_out by blast
            hence "B1 Out A1 P1"
              using T1 in_angle_out by blast
            hence "B Out A C"
              using H1 T1 ncol_conga_ncol not_col_permutation_4 out_col by blast
            hence "\<not> Col B2 A2 P2"
              using Col_perm H1 out_col by blast
          }
          {
            assume "\<not> Col A2 B2 P2"
            hence "\<not> Col B2 A2 P2"
              using Col_perm by blast
          }
          thus ?thesis
            using \<open>B2 Out A2 P2 \<Longrightarrow> \<not> Col B2 A2 P2\<close> \<open>Bet A2 B2 P2 \<Longrightarrow> \<not> Col B2 A2 P2\<close> 
              \<open>Bet A2 B2 P2 \<or> B2 Out A2 P2 \<or> \<not> Col A2 B2 P2\<close> by blast
        qed
        thus ?thesis
          by (simp add: T2 T12 in_angle_one_side)
      qed
      have S1: "A2 B2 OS P P2"
        using T13 T14 one_side_symmetry one_side_transitivity by blast
      have "A1 B1 P1 CongA A2 B2 P"
        using conga_trans conga_sym T1 T13 by blast
      hence "P InAngle A2 B2 P2"
        using conga_preserves_in_angle T2 T1 S1 by blast
      hence "P InAngle A2 B2 C2"
        using T2 in_angle_trans by blast
      hence "A B C LeA A2 B2 C2"
        using T13 LeA_def by blast
    }
    hence "A B C LeA A2 B2 C2"
      using T10 \<open>B2 Out A2 C2 \<Longrightarrow> A B C LeA A2 B2 C2\<close> 
        \<open>Bet A2 B2 C2 \<Longrightarrow> A B C LeA A2 B2 C2\<close> by blast
  }
  thus ?thesis
    using T9 \<open>B Out A C \<Longrightarrow> A B C LeA A2 B2 C2\<close> 
      \<open>Bet A B C \<Longrightarrow> A B C LeA A2 B2 C2\<close> by blast
qed

lemma in_angle_asym:
  assumes "D InAngle A B C" and
    "C InAngle A B D"
  shows "A B C CongA A B D"
proof -
  obtain CC where P1: "Bet A CC D \<and> (CC = B \<or> B Out CC C)"
    using InAngle_def assms(2) by auto
  obtain DD where P2: "Bet A DD C \<and> (DD = B \<or> B Out DD D)"
    using InAngle_def assms(1) by auto
  {
    assume "(CC = B) \<and> (DD = B)"
    hence "A B C CongA A B D"
      by (metis P1 P2 assms(2) conga_line inangle_distincts)
  }
  {
    assume "(CC = B) \<and> (B Out DD D)"
    hence "A B C CongA A B D"
      by (metis P1 assms(1) bet_in_angle_bet conga_line inangle_distincts)
  }
  {
    assume "(B Out CC C) \<and> (DD = B)"
    hence "A B C CongA A B D"
      by (metis P2 assms(2) bet_in_angle_bet conga_line inangle_distincts)
  }
  {
    assume V1: "(B Out CC C) \<and> (B Out DD D)"
    obtain X where P3: "Bet CC X C \<and> Bet DD X D"
      using P1 P2 between_symmetry inner_pasch by blast
    hence "B Out X D"
      using V1 out_bet_out_2 by blast
    hence "B Out C D"
      using P3 V1 out2_bet_out by blast
    hence "A B C CongA A B D"
      using assms(2) inangle_distincts l6_6 out2__conga out_trivial by blast
  }
  thus ?thesis using P1 P2
    using \<open>B Out CC C \<and> DD = B \<Longrightarrow> A B C CongA A B D\<close> 
      \<open>CC = B \<and> B Out DD D \<Longrightarrow> A B C CongA A B D\<close> 
      \<open>CC = B \<and> DD = B \<Longrightarrow> A B C CongA A B D\<close> by blast
qed

lemma lea_asym:
  assumes "A B C LeA D E F" and
    "D E F LeA A B C"
  shows "A B C CongA D E F"
proof cases
  assume P1: "Col A B C"
  {
    assume P1A: "Bet A B C"
    have P2: "D \<noteq> E"
      using assms(1) lea_distincts by blast
    have P3: "F \<noteq> E"
      using assms(2) lea_distincts by auto
    have P4: "A \<noteq> B"
      using assms(1) lea_distincts by auto
    have P5: "C \<noteq> B"
      using assms(2) lea_distincts by blast
    obtain P where P6: "P InAngle D E F \<and> A B C CongA D E P"
      using LeA_def assms(1) by blast
    hence "A B C CongA D E P" by simp
    hence "Bet D E P" using P1 P1A bet_conga__bet
      by blast
    hence "Bet D E F"
      using P6 bet_in_angle_bet by blast
    hence "A B C CongA D E F"
      by (metis bet_conga__bet conga_line l11_29_a P2 P3 P4 P5 assms(2) bet_in_angle_bet)
  }
  {
    assume T1: "\<not> Bet A B C"
    hence T2: "B Out A C"
      using P1 not_out_bet by auto
    obtain P where T3: "P InAngle A B C \<and> D E F CongA A B P"
      using LeA_def assms(2) by blast
    hence T3A: "P InAngle A B C" by simp
    have T3B: "D E F CongA A B P" using T3 by simp
    have T4: "E Out D F"
    proof -
      have T4A: "B Out A P"
        using T2 T3 in_angle_out by blast
      have "A B P CongA D E F"
        by (simp add: T3 conga_sym)
      thus ?thesis
        using T4A l11_21_a by blast
    qed
    hence "A B C CongA D E F"
      by (simp add: T2 l11_21_b)
  }
  thus ?thesis
    using \<open>Bet A B C \<Longrightarrow> A B C CongA D E F\<close> by blast
next
  assume T5: "\<not> Col A B C"
  obtain Q where T6: "C InAngle A B Q \<and> A B Q CongA D E F"
    using assms(1) l11_29_a by blast
  hence T6A: "C InAngle A B Q" by simp
  have T6B: "A B Q CongA D E F" by (simp add: T6)
  obtain P where T7: "P InAngle A B C \<and> D E F CongA A B P"
    using LeA_def assms(2) by blast
  hence T7A: "P InAngle A B C" by simp
  have T7B: "D E F CongA A B P" by (simp add: T7)
  have T13: "A B Q CongA  A B P"
    using T6 T7 conga_trans by blast
  have T14: "Bet A B Q \<or> B Out A Q \<or> \<not> Col A B Q"
    using not_out_bet by auto
  {
    assume R1: "Bet A B Q"
    hence "A B C CongA D E F"
      using T13 T5 T7 bet_col bet_conga__bet bet_in_angle_bet by blast
  }
  {
    assume R2: "B Out A Q"
    hence "A B C CongA D E F"
      using T6 in_angle_out l11_21_a l11_21_b by blast
  }
  {
    assume R3: "\<not> Col A B Q"
    have R3A: "Bet A B P \<or> B Out A P \<or> \<not> Col A B P"
      using not_out_bet by blast
    {
      assume R3AA: "Bet A B P"
      hence "A B C CongA D E F"
        using T5 T7 bet_col bet_in_angle_bet by blast
    }
    {
      assume R3AB: "B Out A P"
      hence "A B C CongA D E F"
        by (meson Col_cases R3 T13 ncol_conga_ncol out_col)
    }
    {
      assume R3AC: "\<not> Col A B P"
      have R3AD: "B Out P Q \<or> A B TS P Q"
      proof -
        have "Coplanar A B P Q"
          using T6A T7A coplanar_perm_8 in_angle_trans inangle__coplanar by blast
        thus ?thesis
          by (simp add: T13 conga_sym conga_cop__or_out_ts)
      qed
      {
        assume "B Out P Q"
        hence "C InAngle A B P"
          by (meson R3 T6A bet_col between_symmetry l11_24 l11_25_aux)
        hence "A B C CongA A B P"
          by (simp add: T7A in_angle_asym)
        hence "A B C CongA D E F"
          by (meson T7B conga_sym conga_trans Tarski_neutral_dimensionless_axioms)
      }
      {
        assume W1: "A B TS P Q"
        have "A B OS P Q"
          using Col_perm R3 R3AC T6A T7A in_angle_one_side in_angle_trans by blast
        hence "A B C CongA D E F"
          using W1 l9_9 by blast
      }
      hence "A B C CongA D E F"
        using R3AD \<open>B Out P Q \<Longrightarrow> A B C CongA D E F\<close> by blast
    }
    hence "A B C CongA D E F"
      using R3A \<open>B Out A P \<Longrightarrow> A B C CongA D E F\<close> 
        \<open>Bet A B P \<Longrightarrow> A B C CongA D E F\<close> by blast
  }
  thus ?thesis
    using T14 \<open>B Out A Q \<Longrightarrow> A B C CongA D E F\<close> 
      \<open>Bet A B Q \<Longrightarrow> A B C CongA D E F\<close> by blast
qed

lemma col_lta__bet:
  assumes "Col X Y Z" and
    "A B C LtA X Y Z"
  shows "Bet X Y Z"
proof -
  have "A B C LeA X Y Z \<and> \<not> A B C CongA X Y Z"
    using LtA_def assms(2) by auto
  hence "Y Out X Z \<longrightarrow> False"
    using lea_asym lea_distincts Tarski_neutral_dimensionless_axioms l11_31_1
    by fastforce
  thus ?thesis using not_out_bet assms(1)
    by blast
qed

lemma col_lta__out:
  assumes "Col A B C" and
    "A B C LtA X Y Z"
  shows "B Out A C"
proof -
  have "A B C LeA X Y Z \<and> \<not> A B C CongA X Y Z"
    using LtA_def assms(2) by auto
  thus ?thesis
    by (metis assms(1) l11_31_2 lea_asym lea_distincts or_bet_out)
qed

lemma lta_distincts:
  assumes "A B C LtA D E F"
  shows "A\<noteq>B \<and> C\<noteq>B \<and> D\<noteq>E \<and> F\<noteq>E \<and> D \<noteq> F"
  by (metis LtA_def assms bet_neq12__neq col_lta__bet lea_distincts not_col_distincts)

lemma gta_distincts:
  assumes "A B C GtA D E F"
  shows "A\<noteq>B \<and> C\<noteq>B \<and> D\<noteq>E \<and> F\<noteq>E \<and> A \<noteq> C"
  using GtA_def assms lta_distincts by presburger

lemma acute_distincts:
  assumes "Acute A B C"
  shows "A\<noteq>B \<and> C\<noteq>B"
  using Acute_def assms lta_distincts by blast

lemma obtuse_distincts:
  assumes "Obtuse A B C"
  shows "A\<noteq>B \<and> C\<noteq>B \<and> A \<noteq> C"
  using Obtuse_def assms lta_distincts by blast

lemma two_sides_in_angle:
  assumes "B \<noteq> P'" and
    "B P TS A C" and
    "Bet P B P'"
  shows "P InAngle A B C \<or> P' InAngle A B C"
proof -
  obtain T where P1: "Col T B P \<and> Bet A T C"
    using TS_def assms(2) by auto
  have P2: "A \<noteq> B"
    using assms(2) ts_distincts by blast
  have P3: "C \<noteq> B"
    using assms(2) ts_distincts by blast
  show ?thesis
  proof cases
    assume "B = T"
    thus ?thesis
      using P1 P2 P3 assms(1) in_angle_line by auto
  next
    assume "B \<noteq> T"
    thus ?thesis
      by (metis InAngle_def P1 assms(1) assms(2) assms(3) between_symmetry l6_3_2 
          or_bet_out ts_distincts)
  qed
qed

lemma in_angle_reverse:
  assumes "A' \<noteq> B" and
    "Bet A B A'" and
    "C InAngle A B D"
  shows "D InAngle A' B C"
proof -
  have P1: "A \<noteq> B"
    using assms(3) inangle_distincts by auto
  have P2: "D \<noteq> B"
    using assms(3) inangle_distincts by blast
  have P3: "C \<noteq> B"
    using assms(3) inangle_distincts by auto
  show ?thesis
  proof cases
    assume "Col B A C"
    {
      assume "Bet C B A"
      hence "Bet A B D" 
        using Bet_cases assms(3) bet_in_angle_bet by blast
      hence "D InAngle A' B C" 
        by (metis P1 P2 P3 \<open>Bet C B A\<close> assms(1) between_symmetry l6_2 out341__inangle)
    }
    moreover
    {
      assume "B Out C A"
      hence "D InAngle A' B C" 
        by (meson InAngle_def Out_cases P2 P3 assms(1) assms(2) bet_out__bet l11_24)
    }
    moreover
    {
      assume "\<not> Col C B A"
      hence "D InAngle A' B C" 
        using \<open>Col B A C\<close> not_col_permutation_1 by blast
    }
    ultimately show ?thesis
      using l6_4_2 by blast
  next
    assume P4: "\<not> Col B A C"
    thus ?thesis
    proof cases
      assume "Col B D C"
      {
        assume "Bet C B D"
        hence "D InAngle A' B C" 
          by (meson Col_def P2 P4 assms(3) bet_in_angle_bet between_symmetry 
              col_transitivity_2 l11_24)
      }
      moreover
      {
        assume "B Out C D"
        hence "D InAngle A' B C" 
          using assms(1) out341__inangle by blast
      }
      moreover
      {
        assume "\<not> Col C B D"
        hence "D InAngle A' B C" 
          using \<open>Col B D C\<close> col_permutation_2 by blast
      }
      ultimately show ?thesis 
        using not_out_bet by blast
    next
      assume P5: "\<not> Col B D C"
      have P6: "C B TS A D"
        using P4 P5 assms(3) in_angle_two_sides by auto
      obtain X where P7: "Bet A X D \<and> (X = B \<or> B Out X C)"
        using InAngle_def assms(3) by auto
      have P8: "X = B \<Longrightarrow> D InAngle A' B C"
        using Out_def P1 P2 P3 P7 assms(1) assms(2) l5_2 out321__inangle by auto
      {
        assume P9: "B Out X C"
        have P10: "C \<noteq> B"
          by (simp add: P3)
        have P10A: "\<not> Col B A C"
          by (simp add: P4)
        have P10B: "\<not> Col B D C"
          by (simp add: P5)
        have P10C: "C InAngle D B A"
          by (simp add: assms(3) l11_24)
        {
          assume "Col D B A"
          have "Col B A C"
          proof -
            have "B \<noteq> X"
              using P9 out_distinct by blast
            have "Col B X A"
              by (meson Bet_perm P10C P5 P7 \<open>Col D B A\<close> bet_col1 col_permutation_3 
                  in_angle_out or_bet_out out_col)
            have "Col B X C"
              by (simp add: P9 out_col)
            thus ?thesis
              using \<open>B \<noteq> X\<close> \<open>Col B X A\<close> col_transitivity_1 by blast
          qed
          hence False
            by (simp add: P4)
        }
        hence P10E: "\<not> Col D B A" by auto
        have P11: "D B OS C A"
          by (simp add: P10C P10E P5 in_angle_one_side)
        have P12: "\<not> Col A D B"
          using Col_cases P10E by auto
        have P13: "\<not> Col A' D B"
          by (metis Col_def \<open>Col D B A \<Longrightarrow> False\<close> assms(1) assms(2) col_transitivity_1)
        have P14: "D B TS A A'"
          using P12 P13 TS_def assms(2) col_trivial_3 by blast
        have P15: "D B TS C A'"
          using P11 P14 l9_8_2 one_side_symmetry by blast
        have P16: "\<not> Col C D B"
          by (simp add: P5 not_col_permutation_3)
        obtain Y where P17: "Col Y D B \<and> Bet C Y A'"
          using P15 TS_def by auto
        have P18: "Bet A' Y C"
          using Bet_perm P17 by blast
        {
          assume S1: "Y \<noteq> B"
          have S2: "Col D B Y"
            using P17 not_col_permutation_2 by blast
          hence S3: "Bet D B Y \<or> Bet B Y D \<or> Bet Y D B"
            using Col_def S2 by auto
          {
            assume S4: "Bet D B Y"
            have S5: "C B OS A' Y"
              by (metis P17 P18 P5 S1 bet_out_1 col_transitivity_2 l6_6 
                  not_col_permutation_3 not_col_permutation_5 out_one_side)
            have S6: "C B TS Y D"
              by (metis Bet_perm P16 P17 S1 S4 bet__ts col3 col_trivial_3 
                  invert_two_sides not_col_permutation_1)
            have "C B TS A A'"
              by (metis P4 assms(1) assms(2) bet__ts invert_two_sides 
                  not_col_permutation_5)
            hence "C B TS Y A"
              using S5 l9_2 l9_8_2 by blast
            hence S9: "C B OS A D"
              using P6 S6 l9_8_1 l9_9 by blast
            hence "B Out Y D"
              using P6 S9 l9_9 by auto
          }
          {
            assume "Bet B Y D"
            hence "B Out Y D"
              by (simp add: S1 bet_out)
          }
          {
            assume "Bet Y D B"
            hence "B Out Y D"
              by (simp add: P2 bet_out_1 l6_6)
          }
          have "B Out Y D"
            using S3 \<open>Bet B Y D \<Longrightarrow> B Out Y D\<close> \<open>Bet D B Y \<Longrightarrow> B Out Y D\<close> 
              \<open>Bet Y D B \<Longrightarrow> B Out Y D\<close> by blast
        }
        hence P19: "(Y = B \<or> B Out Y D)" by auto
        have "D InAngle A' B C"
          using InAngle_def P18 P19 P2 P3 assms(1) by auto
      }
      thus ?thesis using P7 P8 by blast
    qed
  qed
qed

lemma in_angle_trans2:
  assumes "C InAngle A B D" and
    "D InAngle A B E"
  shows "D InAngle C B E"
  by (smt (verit, ccfv_threshold) Bet_cases assms(1,2) in_angle_reverse in_angle_trans
      inangle_distincts l11_24 point_construction_different)

lemma l11_36_aux1:
  assumes "A \<noteq> B" and
    "A' \<noteq> B" and
    "D \<noteq> E" and
    "D' \<noteq> E" and
    "Bet A B A'" and
    "Bet D E D'" and
    "A B C LeA D E F"
  shows "D' E F LeA A' B C"
proof -
  obtain P where P1: "C InAngle A B P \<and>
A B P CongA D E F"
    using assms(7) l11_29_a by blast
  thus ?thesis
    by (metis LeA_def l11_13 assms(2) assms(4) assms(5) assms(6) conga_sym in_angle_reverse)
qed

lemma l11_36_aux2:
  assumes "A \<noteq> B" and
    "A' \<noteq> B" and
    "D \<noteq> E" and
    "D' \<noteq> E" and
    "Bet A B A'" and
    "Bet D E D'" and
    "D' E F LeA A' B C"
  shows "A B C LeA D E F"
  by (metis Bet_cases assms(1) assms(3) assms(5) assms(6) assms(7) l11_36_aux1 lea_distincts)

lemma l11_36:
  assumes "A \<noteq> B" and
    "A' \<noteq> B" and
    "D \<noteq> E" and
    "D' \<noteq> E" and
    "Bet A B A'" and
    "Bet D E D'"
  shows "A B C LeA D E F \<longleftrightarrow> D' E F LeA A' B C"
  using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) l11_36_aux1 l11_36_aux2 by auto

lemma l11_41_aux:
  assumes "\<not> Col A B C" and
    "Bet B A D" and
    "A \<noteq> D"
  shows "A C B LtA C A D"
proof -
  obtain M where P1: "M Midpoint A C"
    using midpoint_existence by auto
  obtain P where P2: "M Midpoint B P"
    using symmetric_point_construction by auto
  have P3: "A C B Cong3 C A P"
    using Cong3_def P1 P2 assms(1) l7_13_R1 l7_2 midpoint_distinct_1 not_col_distincts 
    by meson
  have P4: "A \<noteq> C"
    using assms(1) col_trivial_3 by blast
  have P5: "B \<noteq> C"
    using assms(1) col_trivial_2 by blast
  have P7: "A \<noteq> M"
    using P1 P4 is_midpoint_id by blast
  have P8: "A C B CongA C A P"
    by (simp add: P3 P4 P5 cong3_conga)
  have P8A: "Bet D A B"
    using Bet_perm assms(2) by blast
  have P8B: "Bet P M B"
    by (simp add: P2 between_symmetry midpoint_bet)
  then obtain X where P9: "Bet A X P \<and> Bet M X D" using P8A inner_pasch by blast
  have P9A: "Bet A X P" by (simp add: P9)
  have P9B: "Bet M X D" by (simp add: P9)
  have P10A: "P InAngle C A D"
  proof -
    have K1: "P InAngle M A D"
      by (metis InAngle_def P3 P5 P7 P9 assms(3) bet_out cong3_diff2)
    have K2: "A Out C M"
      using Out_def P1 P4 P7 midpoint_bet by auto
    have K3: "A Out D D"
      using assms(3) out_trivial by auto
    have "A Out P P"
      using K1 inangle_distincts out_trivial by auto
    thus ?thesis
      using K1 K2 K3 l11_25 by blast
  qed
  hence P10: "A C B LeA C A D"
    using LeA_def P8 by auto
  {
    assume K5: "A C B CongA C A D"
    hence K6: "C A D CongA C A P"
      using P8 conga_sym conga_trans by blast
    have K7: "Coplanar C A D P"
      using P10A inangle__coplanar ncoplanar_perm_18 by blast
    hence K8: "A Out D P \<or> C A TS D P"
      by (simp add: K6 conga_cop__or_out_ts)
    {
      assume "A Out D P"
      hence "Col M B A"
        by (meson P8A P8B bet_col1 bet_out__bet between_symmetry not_col_permutation_4)
      hence K8F: "Col A M B"
        using not_col_permutation_1 by blast
      have "Col A M C"
        by (simp add: P1 bet_col midpoint_bet)
      hence "False"
        using K8F P7 assms(1) col_transitivity_1 by blast
    }
    hence K9: "\<not> A Out D P" by auto
    {
      assume V1: "C A TS D P"
      hence V3: "A C TS B P"
        by (metis P10A P8A assms(1) col_trivial_1 col_trivial_2 in_angle_reverse 
            in_angle_two_sides invert_two_sides l11_24 l9_18 not_col_permutation_5)
      have "A C TS B D"
        by (simp add: assms(1) assms(2) assms(3) bet__ts not_col_permutation_5)
      hence "A C OS D P"
        using V1 V3 invert_two_sides l9_8_1 l9_9 by blast
      hence "False"
        using V1 invert_one_side l9_9 by blast
    }
    hence "\<not> C A TS D P" by auto
    hence "False" using K8 K9 by auto
  }
  hence "\<not> A C B CongA C A D" by auto
  thus ?thesis
    by (simp add: LtA_def P10)
qed

lemma l11_41:
  assumes "\<not> Col A B C" and
    "Bet B A D" and
    "A \<noteq> D"
  shows "A C B LtA C A D \<and> A B C LtA C A D"
proof -
  have P1: "A C B LtA C A D"
    using assms(1) assms(2) assms(3) l11_41_aux by auto
  have "A B C LtA C A D"
  proof -
    obtain E where T1: "Bet C A E \<and> Cong A E C A"
      using segment_construction by blast
    have T1A: "Bet C A E" using T1 by simp
    have T1B: "Cong A E C A" using T1 by simp
    have T2: "A B C LtA B A E"
      using T1 assms(1) cong_reverse_identity l11_41_aux not_col_distincts 
        not_col_permutation_5 by blast
    have T3: "B A C CongA C A B"
      by (metis assms(1) conga_pseudo_refl not_col_distincts)
    have T3A: "D A C CongA E A B"
      by (metis CongA_def T1 T3 assms(2) assms(3) cong_reverse_identity l11_13)
    hence T4: "B A E CongA C A D"
      using conga_comm conga_sym by blast
    have "A B C CongA A B C"
      using T2 conga_refl lta_distincts by blast
    hence T5: "A B C LeA C A D"
      by (meson T2 T4 LtA_def l11_30)
    have "\<not> A B C CongA C A D"
      by (meson T2 LtA_def conga_right_comm conga_trans T3A)
    thus ?thesis
      by (simp add: LtA_def T5)
  qed
  thus ?thesis by (simp add: P1)
qed

lemma not_conga:
  assumes "A B C CongA A' B' C'" and
    "\<not> A B C CongA D E F"
  shows "\<not> A' B' C' CongA D E F"
  by (meson assms(1) assms(2) conga_trans)

lemma not_conga_sym:
  assumes "\<not> A B C CongA D E F"
  shows "\<not> D E F CongA A B C"
  using assms conga_sym by blast

lemma not_and_lta:
  shows "\<not> (A B C LtA D E F \<and> D E F LtA A B C)"
proof -
  {
    assume "A B C LtA D E F" and "D E F LtA A B C"
    hence "A B C CongA D E F"
      using LtA_def lea_asym by blast
    hence "False" 
      using LtA_def \<open>D E F LtA A B C\<close> not_conga_sym by blast
  }
  thus ?thesis by auto
qed

lemma conga_preserves_lta:
  assumes "A B C CongA A' B' C'" and
    "D E F CongA D' E' F'" and
    "A B C LtA D E F"
  shows "A' B' C' LtA D' E' F'"
  by (meson LtA_def conga_trans l11_30 not_conga_sym assms(1) assms(2) assms(3))

lemma lta_trans:
  assumes "A B C LtA A1 B1 C1" and
    "A1 B1 C1 LtA A2 B2 C2"
  shows "A B C LtA A2 B2 C2"
proof -
  have P1: "A B C LeA A2 B2 C2"
    by (meson LtA_def assms(1) assms(2) lea_trans)
  {
    assume "A B C CongA A2 B2 C2"
    hence "False"
      by (meson LtA_def lea_asym lea_trans assms(1) assms(2) conga__lea456123)
  }
  thus ?thesis
    using LtA_def P1 by blast
qed

lemma obtuse_sym:
  assumes "Obtuse A B C"
  shows "Obtuse C B A"
  by (meson Obtuse_def lta_right_comm assms)

lemma acute_sym:
  assumes "Acute A B C"
  shows "Acute C B A"
  by (meson Acute_def lta_left_comm assms)

lemma acute_col__out:
  assumes "Col A B C" and
    "Acute A B C"
  shows "B Out A C"
  by (meson Acute_def assms(1) assms(2) col_lta__out)

lemma col_obtuse__bet:
  assumes "Col A B C" and
    "Obtuse A B C"
  shows "Bet A B C"
  using Obtuse_def assms(1) assms(2) col_lta__bet by blast

lemma out__acute:
  assumes "B Out A C"
  shows "Acute A B C"
proof -
  have P1: "A \<noteq> B"
    using assms out_diff1 by auto
  then obtain D where P3: "B D Perp A B"
    using perp_exists by blast
  hence P4: "B \<noteq> D"
    using perp_distinct by auto
  have P5: "Per A B D"
    by (simp add: P3 l8_2 perp_per_1)
  have P6: "A B C LeA A B D"
    using P1 P4 assms l11_31_1 by auto
  {
    assume "A B C CongA A B D"
    hence "False"
      by (metis Col_cases P1 P4 P5 assms col_conga_col l8_9 out_col)
  }
  hence "A B C LtA A B D"
    using LtA_def P6 by auto
  thus ?thesis
    using P5 Acute_def by auto
qed

lemma bet__obtuse:
  assumes "Bet A B C" and
    "A \<noteq> B" and "B \<noteq> C"
  shows "Obtuse A B C"
proof -
  obtain D where P1: "B D Perp A B"
    using assms(2) perp_exists by blast
  have P5: "B \<noteq> D"
    using P1 perp_not_eq_1 by auto
  have P6: "Per A B D"
    using P1 Perp_cases perp_per_1 by blast
  have P7: "A B D LeA A B C"
    using assms(2) assms(3) P5 assms(1) l11_31_2 by auto
  {
    assume "A B D CongA A B C"
    hence "False"
      using assms(2) P5 P6 assms(1) bet_col ncol_conga_ncol per_not_col by blast
  }
  hence "A B D LtA A B C"
    using LtA_def P7 by blast
  thus ?thesis
    using Obtuse_def P6 by blast
qed

lemma l11_43_aux:
  assumes "A \<noteq> B" and
    "A \<noteq> C" and
    "Per B A C \<or> Obtuse B A C"
  shows "Acute A B C"
proof cases
  assume P1: "Col A B C"
  {
    assume "Per B A C"
    hence "Acute A B C"
      using Col_cases P1 assms(1) assms(2) per_col_eq by blast
  }
  {
    assume "Obtuse B A C"
    hence "Bet B A C"
      using P1 col_obtuse__bet col_permutation_4 by blast
    hence "Acute A B C"
      by (simp add: assms(1) bet_out out__acute)
  }
  thus ?thesis
    using \<open>Per B A C \<Longrightarrow> Acute A B C\<close> assms(3) by blast
next
  assume P2: "\<not> Col A B C"
  hence P3: "B \<noteq> C"
    using col_trivial_2 by auto
  obtain B' where P4: "Bet B A B' \<and> Cong A B' B A"
    using segment_construction by blast
  have P5: "\<not> Col B' A C"
    by (metis Col_def P2 P4 col_transitivity_2 cong_reverse_identity)
  hence P6: "B' \<noteq> A \<and> B' \<noteq> C"
    using not_col_distincts by blast
  hence P7: "A C B LtA C A B' \<and> A B C LtA C A B'"
    using P2 P4 l11_41 by auto
  hence P7A: "A C B LtA C A B'" by simp
  have P7B: "A B C LtA C A B'" by (simp add: P7)
  {
    assume "Per B A C"
    have "Acute A B C"
      by (metis Acute_def P4 P7B \<open>Per B A C\<close> assms(1) bet_col col_per2__per 
          col_trivial_3 l8_3 lta_right_comm)
  }
  {
    assume T1: "Obtuse B A C"
    then obtain a b c where T2: "Per a b c \<and> a b c LtA B A C"
      using Obtuse_def by blast
    hence T2A: "Per a b c" by simp
    have T2B: "a b c LtA B A C" by (simp add: T2)
    hence T3: "a b c LeA B A C \<and> \<not> a b c CongA B A C"
      by (simp add: LtA_def)
    hence T3A: "a b c LeA B A C" by simp
    have T3B: "\<not> a b c CongA B A C" by (simp add: T3)
    obtain P where T4: "P InAngle B A C \<and> a b c CongA B A P"
      using LeA_def T3 by blast
    hence T5: "Per B A P" using T4 T2 l11_17 by blast
    hence T6: "Per P A B"
      using l8_2 by blast
    have "Col A B B'"
      by (simp add: P4 bet_col col_permutation_4)
    hence "Per P A B'"
      using T6 assms(1) per_col by blast
    hence S3: "B A P CongA B' A P"
      using l8_2 P6 T5 T4 CongA_def assms(1) l11_16 by auto
    have "C A B' LtA P A B"
    proof -
      have S4: "B A P LeA B A C \<longleftrightarrow> B' A C LeA B' A P"
        using P4 P6 assms(1) l11_36 by auto
      have S5: "C A B' LeA P A B"
      proof -
        have S6: "B A P LeA B A C"
          using T4 inangle__lea by auto
        have "B' A P CongA P A B"
          using S3 conga_left_comm not_conga_sym by blast
        thus ?thesis
          using P6 S4 S6 assms(2) conga_pseudo_refl l11_30 by auto
      qed
      {
        assume T10: "C A B' CongA P A B"
        have "Per B' A C"
        proof -
          have "B A P CongA B' A C"
            using T10 conga_comm conga_sym by blast
          thus ?thesis
            using T5 l11_17 by blast
        qed
        hence "Per C A B"
          using Col_cases P6 \<open>Col A B B'\<close> l8_2 l8_3 by blast
        have "a b c CongA B A C"
        proof -
          have "a \<noteq> b"
            using T3A lea_distincts by auto
          have "c \<noteq> b"
            using T2B lta_distincts by blast
          have "Per B A C"
            using Per_cases \<open>Per C A B\<close> by blast
          thus ?thesis
            using T2 \<open>a \<noteq> b\<close> \<open>c \<noteq> b\<close> assms(1) assms(2) l11_16 by auto
        qed
        hence "False"
          using T3B by blast
      }
      hence "\<not> C A B' CongA P A B" by blast
      thus ?thesis
        by (simp add: LtA_def S5)
    qed
    hence "A B C LtA B A P"
      by (meson P7 lta_right_comm lta_trans)
    hence "Acute A B C"  using T5
      using Acute_def by blast
  }
  thus ?thesis
    using \<open>Per B A C \<Longrightarrow> Acute A B C\<close> assms(3) by blast
qed

lemma l11_43:
  assumes "A \<noteq> B" and
    "A \<noteq> C" and
    "Per B A C \<or> Obtuse B A C"
  shows "Acute A B C \<and> Acute A C B"
  using Per_perm assms(1) assms(2) assms(3) l11_43_aux obtuse_sym by blast

lemma acute_lea_acute:
  assumes "Acute D E F" and
    "A B C LeA D E F"
  shows "Acute A B C"
proof -
  obtain A' B' C' where P1: "Per A' B' C' \<and> D E F LtA A' B' C'"
    using Acute_def assms(1) by auto
  have P2: "A B C LeA A' B' C'"
    using LtA_def P1 assms(2) lea_trans by blast
  have "\<not> A B C CongA A' B' C'"
    by (meson LtA_def P1 assms(2) conga__lea456123 lea_asym lea_trans)
  hence "A B C LtA A' B' C'"
    by (simp add: LtA_def P2)
  thus ?thesis
    using Acute_def P1 by auto
qed

lemma lea_obtuse_obtuse:
  assumes "Obtuse D E F" and
    "D E F LeA A B C"
  shows "Obtuse A B C"
proof -
  obtain A' B' C' where P1: "Per A' B' C' \<and> A' B' C' LtA D E F"
    using Obtuse_def assms(1) by auto
  hence P2: "A' B' C' LeA A B C"
    using LtA_def assms(2) lea_trans by blast
  have "\<not> A' B' C' CongA A B C"
    by (meson LtA_def P1 assms(2) conga__lea456123 lea_asym lea_trans)
  hence "A' B' C' LtA A B C"
    by (simp add: LtA_def P2)
  thus ?thesis
    using Obtuse_def P1 by auto
qed

lemma l11_44_1_a:
  assumes "A \<noteq> B" and
    "A \<noteq> C" and
    "Cong B A B C"
  shows "B A C CongA B C A"
  using Cong3_def assms(1) assms(2) assms(3) cong3_conga cong_inner_transitivity 
    cong_pseudo_reflexivity by metis

lemma l11_44_2_a:
  assumes "\<not> Col A B C" and
    "B A Lt B C"
  shows "B C A LtA B A C"
proof -
  have T1: "A \<noteq> B"
    using assms(1) col_trivial_1 by auto
  have T3: "A \<noteq> C"
    using assms(1) col_trivial_3 by auto
  have "B A Le B C"
    by (simp add: assms(2) lt__le)
  then obtain C' where P1: "Bet B C' C \<and> Cong B A B C'"
    using assms(2) Le_def by blast
  have T5: "C \<noteq> C'"
    using P1 assms(2) cong__nlt by blast
  have T5A: "C' \<noteq> A"
    using Col_def Col_perm P1 assms(1) by blast
  hence T6: "C' InAngle B A C"
    using InAngle_def P1 T1 T3 out_trivial by auto
  have T7: "C' A C LtA A C' B \<and> C' C A LtA A C' B"
  proof -
    have W1: "\<not> Col C' C A"
      by (metis Col_def P1 T5 assms(1) col_transitivity_2)
    have W2: "Bet C C' B"
      using Bet_perm P1 by blast
    have "C' \<noteq> B"
      using P1 T1 cong_identity by blast
    thus ?thesis
      using l11_41 W1 W2 by simp
  qed
  have T90: "B A C' LtA B A C"
  proof -
    have T90A: "B A C' LeA B A C"
      by (simp add: T6 inangle__lea)
    have "B A C' CongA B A C'"
      using T1 T5A conga_refl by auto
    {
      assume "B A C' CongA B A C"
      hence R1: "A Out C' C"
        by (metis P1 T7 assms(1) bet_out conga_os__out lta_distincts not_col_permutation_4 
            out_one_side)
      have "B A OS C' C"
        by (metis Col_perm P1 T1 assms(1) bet_out cong_diff_2 out_one_side)
      hence "False"
        using Col_perm P1 T5 R1 bet_col col2__eq one_side_not_col123 out_col by blast
    }
    hence "\<not> B A C' CongA B A C" by blast
    thus ?thesis
      by (simp add: LtA_def T90A)
  qed
  have "B A C' CongA B C' A"
    using P1 T1 T5A l11_44_1_a by auto
  hence K2: "A C' B CongA B A C'"
    using conga_left_comm not_conga_sym by blast
  have "B C A LtA B A C'"
  proof -
    have K1: "B C A CongA B C A"
      using assms(1) conga_refl not_col_distincts by blast
    have "B C A LtA A C' B"
    proof -
      have "C' C A CongA B C A"
      proof -
        have K2: "C Out B C'"
          using P1 T5 bet_out_1 l6_6 by auto
        have "C Out A A"
          by (simp add: T3 out_trivial)
        thus ?thesis
          by (simp add: K2 out2__conga)
      qed
      have "A C' B CongA A C' B"
        using CongA_def K2 conga_refl by auto
      thus ?thesis
        using T7 \<open>C' C A CongA B C A\<close> conga_preserves_lta by auto
    qed
    thus ?thesis
      using K1 K2 conga_preserves_lta by auto
  qed
  thus ?thesis
    using T90 lta_trans by blast
qed

lemma not_lta_and_conga:
  "\<not> ( A B C LtA D E F \<and> A B C CongA D E F)"
  by (simp add: LtA_def)

lemma conga_sym_equiv:
  "A B C CongA A' B' C' \<longleftrightarrow> A' B' C' CongA A B C"
  using not_conga_sym by blast

lemma conga_dec:
  "A B C CongA D E F \<or> \<not> A B C CongA D E F"
  by auto

lemma lta_not_conga:
  assumes "A B C LtA D E F"
  shows "\<not> A B C CongA D E F"
  using assms not_lta_and_conga by auto

lemma lta__lea:
  assumes "A B C LtA D E F"
  shows "A B C LeA D E F"
  using LtA_def assms by auto

lemma nlta:
  "\<not> A B C LtA A B C"
  using not_and_lta by blast

lemma lea__nlta:
  assumes "A B C LeA D E F"
  shows "\<not> D E F LtA A B C"
  by (meson lea_asym not_lta_and_conga assms lta__lea)

lemma lta__nlea:
  assumes "A B C LtA D E F"
  shows "\<not> D E F LeA A B C"
  using assms lea__nlta by blast

lemma l11_44_1_b:
  assumes "\<not> Col A B C" and
    "B A C CongA B C A"
  shows "Cong B A B C"
proof -
  have "B A Lt B C \<or> B A Gt B C \<or> Cong B A B C"
    by (simp add: or_lt_cong_gt)
  thus ?thesis
    by (meson Gt_def assms(1) assms(2) conga_sym l11_44_2_a not_col_permutation_3 
        not_lta_and_conga)
qed

lemma l11_44_2_b:
  assumes "B A C LtA B C A"
  shows "B C Lt B A"
proof cases
  assume "Col A B C"
  thus ?thesis
    using Col_perm assms bet__lt1213 col_lta__bet lta_distincts by blast
next
  assume P1: "\<not> Col A B C"
  hence P2: "A \<noteq> B"
    using col_trivial_1 by blast
  have P3: "A \<noteq> C"
    using P1 col_trivial_3 by auto
  have "B A Lt B C \<or> B A Gt B C \<or> Cong B A B C"
    by (simp add: or_lt_cong_gt)
  {
    assume "B A Lt B C"
    hence "B C Lt B A"
      using P1 assms l11_44_2_a not_and_lta by blast
  }
  {
    assume "B A Gt B C"
    hence "B C Lt B A"
      using Gt_def P1 assms l11_44_2_a not_and_lta by blast
  }
  {
    assume "Cong B A B C"
    hence "B A C CongA B C A"
      by (simp add: P2 P3 l11_44_1_a)
    hence "B C Lt B A"
      using assms not_lta_and_conga by blast
  }
  thus ?thesis
    by (meson P1 not_and_lta 
        \<open>B A Gt B C \<Longrightarrow> B C Lt B A\<close> \<open>B A Lt B C \<or> B A Gt B C \<or> Cong B A B C\<close> assms l11_44_2_a)
qed

lemma l11_44_1:
  assumes "\<not> Col A B C"
  shows "B A C CongA B C A \<longleftrightarrow> Cong B A B C"
  using assms l11_44_1_a l11_44_1_b not_col_distincts by blast

lemma l11_44_2:
  assumes "\<not> Col A B C"
  shows "B A C LtA B C A \<longleftrightarrow> B C Lt B A"
  using assms l11_44_2_a l11_44_2_b not_col_permutation_3 by blast

lemma l11_44_2bis:
  assumes "\<not> Col A B C"
  shows "B A C LeA B C A \<longleftrightarrow> B C Le B A"
proof -
  {
    assume P1: "B A C LeA B C A"
    {
      assume "B A Lt B C"
      hence "B C A LtA B A C"
        by (simp add: assms l11_44_2_a)
      hence "False"
        using P1 lta__nlea by auto
    }
    hence "\<not> B A Lt B C" by blast
    have "B C Le B A"
      using \<open>\<not> B A Lt B C\<close> nle__lt by blast
  }
  {
    assume P2: "B C Le B A"
    have "B A C LeA B C A"
    proof cases
      assume "Cong B C B A"
      hence "B A C CongA B C A"
        by (metis assms conga_sym l11_44_1_a not_col_distincts)
      thus ?thesis
        by (simp add: conga__lea)
    next
      assume "\<not> Cong B C B A"
      hence "B A C LtA B C A"
        by (simp add: l11_44_2 assms Lt_def P2)
      thus ?thesis
        by (simp add: lta__lea)
    qed
  }
  thus ?thesis
    using \<open>B A C LeA B C A \<Longrightarrow> B C Le B A\<close> by blast
qed

lemma l11_46:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Per A B C \<or> Obtuse A B C"
  shows "B A Lt A C \<and> B C Lt A C"
proof cases
  assume "Col A B C"
  thus ?thesis
    by (meson assms(1) assms(2) assms(3) bet__lt1213 bet__lt2313 col_obtuse__bet 
        lt_left_comm per_not_col)
next
  assume P1: "\<not> Col A B C"
  have P2: "A \<noteq> C"
    using P1 col_trivial_3 by auto
  have P3: "Acute B A C \<and> Acute B C A"
    using assms(1) assms(2) assms(3) l11_43 by auto
  then obtain A' B' C' where P4: "Per A' B' C' \<and> B C A LtA A' B' C'"
    using Acute_def P3 by auto
  {
    assume P5: "Per A B C"
    have P5A: "A C B CongA A C B"
      by (simp add: P2 assms(2) conga_refl)
    have S1: "A \<noteq> B"
      by (simp add: assms(1))
    have S2: "B \<noteq> C"
      by (simp add: assms(2))
    have S3: "A' \<noteq> B'"
      using P4 lta_distincts by blast
    have S4: "B' \<noteq> C'"
      using P4 lta_distincts by blast
    hence "A' B' C' CongA A B C" using l11_16
      using S1 S2 S3 S4 P4 P5 by blast
    hence "A C B LtA A B C"
      using P5A P4 conga_preserves_lta lta_left_comm by blast
  }
  {
    assume "Obtuse A B C"
    obtain A'' B'' C'' where P6: "Per A'' B'' C'' \<and> A'' B'' C'' LtA A B C"
      using Obtuse_def \<open>Obtuse A B C\<close> by auto
    have "B C A LtA A' B' C'"
      by (simp add: P4)
    hence P7: "A C B LtA A' B' C'"
      by (simp add: lta_left_comm)
    have "A' B' C' LtA A B C"
    proof -
      have U1: "A'' B'' C'' CongA A' B' C'"
      proof -
        have V2: "A'' \<noteq> B''"
          using P6 lta_distincts by blast
        have V3: "C'' \<noteq> B''"
          using P6 lta_distincts by blast
        have V5: "A' \<noteq> B'"
          using P7 lta_distincts by blast
        have "C' \<noteq> B'"
          using P4 lta_distincts by blast
        thus ?thesis using P6 V2 V3 P4 V5
          by (simp add: l11_16)
      qed
      have U2: "A B C CongA A B C"
        using assms(1) assms(2) conga_refl by auto
      have U3: "A'' B'' C'' LtA A B C"
        by (simp add: P6)
      thus ?thesis
        using U1 U2 conga_preserves_lta by auto
    qed
    hence "A C B LtA A B C"
      using P7 lta_trans by blast
  }
  hence "A C B LtA A B C"
    using \<open>Per A B C \<Longrightarrow> A C B LtA A B C\<close> assms(3) by blast
  hence "A B Lt A C"
    by (simp add: l11_44_2_b)
  hence "B A Lt A C"
    using Lt_cases by blast
  have "C A B LtA C B A"
  proof -
    obtain A' B' C' where U4: "Per A' B' C' \<and> B A C LtA A' B' C'"
      using Acute_def P3 by blast
    {
      assume "Per A B C"
      hence W3: "A' B' C' CongA C B A"
        using U4 assms(2) l11_16 l8_2 lta_distincts by blast
      have W2: "C A B CongA C A B"
        using P2 assms(1) conga_refl by auto
      have "C A B LtA A' B' C'"
        by (simp add: U4 lta_left_comm)
      hence "C A B LtA C B A"
        using W2 W3 conga_preserves_lta by blast
    }
    {
      assume "Obtuse A B C"
      then obtain A'' B'' C'' where W4: "Per A'' B'' C'' \<and> A'' B'' C'' LtA A B C"
        using Obtuse_def by auto
      have W5: "C A B LtA A' B' C'"
        by (simp add: U4 lta_left_comm)
      have "A' B' C' LtA C B A"
      proof -
        have W6: "A'' B'' C'' CongA A' B' C'" using l11_16 W4 U4
          using lta_distincts by blast
        have "C B A CongA C B A"
          using assms(1) assms(2) conga_refl by auto
        thus ?thesis
          using W4 W6 conga_left_comm conga_preserves_lta by blast
      qed
      hence "C A B LtA C B A"
        using W5 lta_trans by blast
    }
    thus ?thesis
      using \<open>Per A B C \<Longrightarrow> C A B LtA C B A\<close> assms(3) by blast
  qed
  hence "C B Lt C A"
    by (simp add: l11_44_2_b)
  hence "C B Lt A C"
    using Lt_cases by auto
  hence "B C Lt A C"
    using Lt_cases by blast
  thus ?thesis
    by (simp add: \<open>B A Lt A C\<close>)
qed

lemma l11_47:
  assumes "Per A C B" and
    "H PerpAt C H A B"       
  shows "Bet A H B \<and> A \<noteq> H \<and> B \<noteq> H"
proof -
  have P1: "Per C H A"
    using assms(2) perp_in_per_1 by auto
  have P2: "C H Perp A B"
    using assms(2) perp_in_perp by auto
  thus ?thesis
  proof cases
    assume "Col A C B"
    thus ?thesis
      by (metis P1 assms(1) assms(2) per_distinct_1 per_not_col perp_in_distinct perp_in_id)
  next
    assume P3: "\<not> Col A C B"
    have P4: "A \<noteq> H"
      by (metis P2 Per_perm l8_7 assms(1) assms(2) col_trivial_1 perp_in_per_2 perp_not_col2)
    have P5: "Per C H B"
      using assms(2) perp_in_per_2 by auto
    have P6: "B \<noteq> H"
      using P1 P2 assms(1) l8_2 l8_7 perp_not_eq_1 by blast
    have P7: "H A Lt A C \<and> H C Lt A C"
      by (metis P1 P2 P4 l11_46 l8_2 perp_distinct)
    have P8: "C A Lt A B \<and> C B Lt A B"
      using P3 assms(1) l11_46 not_col_distincts by blast
    have P9: "H B Lt B C \<and> H C Lt B C"
      by (metis P2 P5 P6 Per_cases l11_46 perp_not_eq_1)
    have P10: "Bet A H B"
    proof -
      have T1: "Col A H B"
        using assms(2) col_permutation_5 perp_in_col by blast
      have T2: "A H Le A B" using P7 P8
        by (meson lt_comm lt_transitivity nlt__le not_and_lt)
      have "H B Le A B"
        by (meson Lt_cases P8 P9 le_transitivity le_cases lt__nle)
      thus ?thesis
        using T1 T2 l5_12_b by blast
    qed
    thus ?thesis
      by (simp add: P4 P6)
  qed
qed

lemma l11_49:
  assumes "A B C CongA A' B' C'" and
    "Cong B A B' A'" and
    "Cong B C B' C'"
  shows "Cong A C A' C' \<and> (A \<noteq> C \<longrightarrow> (B A C CongA B' A' C' \<and> B C A CongA B' C' A'))"
proof -
  have " Cong A C A' C'"
    using assms(1) assms(2) assms(3) cong2_conga_cong not_cong_2143 by blast
  {
    assume "A \<noteq> C"
    have "A \<noteq> B"
      using CongA_def assms(1) by blast
    have "C \<noteq> B"
      using CongA_def assms(1) by blast
    have "B A C Cong3 B' A' C'" 
      using Cong3_def \<open>Cong A C A' C'\<close> assms(2) assms(3) by blast
    hence "B A C CongA B' A' C'" 
      using \<open>A \<noteq> B\<close> \<open>A \<noteq> C\<close> cong3_conga by auto
    moreover have "B C A Cong3 B' C' A'" 
      using \<open>B A C Cong3 B' A' C'\<close> cong_3_swap_2 by blast
    hence "B C A CongA B' C' A'" 
      using \<open>A \<noteq> C\<close> \<open>C \<noteq> B\<close> cong3_conga by auto
    ultimately have "B A C CongA B' A' C' \<and> B C A CongA B' C' A'" 
      by blast
  }
  thus ?thesis 
    using \<open>Cong A C A' C'\<close> by blast
qed

lemma l11_50_1:
  assumes "\<not> Col A B C" and
    "B A C CongA B' A' C'" and
    "A B C CongA A' B' C'" and
    "Cong A B A' B'"
  shows "Cong A C A' C' \<and> Cong B C B' C' \<and> A C B CongA A' C' B'"
proof -
  obtain C'' where P1: "B' Out C'' C' \<and> Cong B' C'' B C"
    by (metis Col_perm assms(1) assms(3) col_trivial_3 conga_diff56 l6_11_existence)
  have P2: "B' \<noteq> C''"
    using P1 out_diff1 by auto
  have P3: "\<not> Col A' B' C'"
    using assms(1) assms(3) ncol_conga_ncol by blast
  have P4: "\<not> Col A' B' C''"
    by (meson P1 P2 P3 col_transitivity_1 not_col_permutation_2 out_col)
  have P5: "Cong A C A' C''"
  proof -
    have Q1: "B Out A A"
      using assms(1) not_col_distincts out_trivial by auto
    have Q2: "B Out C C"
      using assms(1) col_trivial_2 out_trivial by force
    have Q3: "B' Out A' A'"
      using P3 not_col_distincts out_trivial by auto
    have Q5: "Cong B A B' A'"
      using assms(4) not_cong_2143 by blast
    have "Cong B C B' C''"
      using P1 not_cong_3412 by blast
    thus ?thesis
      using l11_4_1 P1 Q1 Q2 Q3 Q5 assms(3) by blast
  qed
  have P6: "B A C Cong3 B' A' C''"
    using Cong3_def Cong_perm P1 P5 assms(4) by blast
  have P7: "B A C CongA B' A' C''"
    by (metis P6 assms(1) cong3_conga not_col_distincts)
  have P8: "B' A' C' CongA B' A' C''"
    by (meson P7 assms(2) conga_sym conga_trans)
  have "B' A' OS C' C''"
    using Col_perm Out_cases P1 P3 out_one_side by blast
  hence "A' Out C' C''"
    using P8 conga_os__out by auto
  hence "Col A' C' C''"
    using out_col by auto
  hence P9: "C' = C''"
    using Col_perm P1 out_col P3 col_transitivity_1 by blast
  have T1: "Cong A C A' C'"
    by (simp add: P5 P9)
  have T2: "Cong B C B' C'"
    using Cong_perm P1 P9 by blast
  hence "A C B CongA A' C' B'"
    using T1 assms(1) assms(2) assms(4) col_trivial_2 l11_49 by blast
  thus ?thesis using T1 T2 by blast
qed

lemma l11_50_2:
  assumes "\<not> Col A B C" and
    "B C A CongA B' C' A'" and
    "A B C CongA A' B' C'" and
    "Cong A B A' B'"
  shows "Cong A C A' C' \<and> Cong B C B' C' \<and> C A B CongA C' A' B'"
proof -
  have P1: "A \<noteq> B"
    using assms(1) col_trivial_1 by auto
  have P2: "B \<noteq> C"
    using assms(1) col_trivial_2 by auto
  have P3: "A' \<noteq> B'"
    using P1 assms(4) cong_diff by blast
  have P4: "B' \<noteq> C'"
    using assms(2) conga_diff45 by auto
  then obtain C'' where P5: "B' Out C'' C' \<and> Cong B' C'' B C"
    using P2 l6_11_existence by presburger
  have P5BIS: "B' \<noteq> C''"
    using P5 out_diff1 by auto
  have P5A: "Col B' C'' C'"
    using P5 out_col by auto
  have P6: "\<not> Col A' B' C'"
    using assms(1) assms(3) ncol_conga_ncol by blast
  {
    assume "Col A' B' C''"
    hence "Col B' C'' A'"
      using not_col_permutation_2 by blast
    hence "Col B' C' A'" using col_transitivity_1 P5BIS P5A by blast
    hence "Col A' B' C'"
      using Col_perm by blast
    hence False
      using P6 by auto
  }
  hence P7: "\<not> Col A' B' C''" by blast
  have P8: "Cong A C A' C''"
  proof -
    have "B Out A A"
      by (simp add: P1 out_trivial)
    have K1: "B Out C C"
      using P2 out_trivial by auto
    have K2: "B' Out A' A'"
      using P3 out_trivial by auto
    have "Cong B A B' A'"
      by (simp add: Cong_perm assms(4))
    have "Cong B C B' C''"
      using Cong_perm P5 by blast
    thus ?thesis
      using P5 \<open>Cong B A B' A'\<close> P1 out_trivial K1 K2 assms(3) l11_4_1 by blast
  qed
  have P9: "B C A Cong3 B' C'' A'"
    using Cong3_def Cong_perm P5 P8 assms(4) by blast
  hence P10: "B C A CongA B' C'' A'"
    using assms(1) cong3_conga not_col_distincts by auto
  have P11: "B' C' A' CongA B' C'' A'"
    using P9 assms(2) cong3_conga2 conga_sym by blast
  show ?thesis
  proof cases
    assume L1: "C' = C''"
    hence L2: "Cong A C A' C'"
      by (simp add: P8)
    have L3: "Cong B C B' C'"
      using Cong_perm L1 P5 by blast
    have "C A B Cong3 C' A' B'"
      by (simp add: L1 P9 cong_3_swap cong_3_swap_2)
    hence "C A B CongA C' A' B'"
      by (metis CongA_def P1 assms(2) cong3_conga)
    thus ?thesis using L2 L3 by auto
  next
    assume R1: "C' \<noteq> C''"
    have R1A: "\<not> Col C'' C' A'"
      by (metis P5A P7 R1 col_permutation_2 col_trivial_2 colx)
    have R1B: "Bet B' C'' C' \<or> Bet B' C' C''"
      using Out_def P5 by auto
    {
      assume S1: "Bet B' C'' C'"
      hence S2: "C'' A' C' LtA A' C'' B' \<and> C'' C' A' LtA A' C'' B'"
        using P5BIS R1A between_symmetry l11_41 by blast
      have "B' C' A' CongA C'' C' A'"
        by (metis P11 R1 conga_comm S1 bet_out_1 conga_diff45 not_conga_sym 
            out2__conga out_trivial)
      hence "B' C' A' LtA A' C'' B'"
        by (meson P11 conga_right_comm not_conga not_conga_sym S2 not_lta_and_conga)
      hence "Cong A C A' C' \<and> Cong B C B' C'"
        by (meson P11 conga_right_comm not_lta_and_conga)
    }
    {
      assume Z1: "Bet B' C' C''"
      have Z2: "\<not> Col C' C'' A'"
        by (simp add: R1A not_col_permutation_4)
      have Z3: "C'' Out C' B'"
        by (simp add: R1 Z1 bet_out_1)
      have Z4: "C'' Out A' A'"
        using P7 not_col_distincts out_trivial by blast
      hence Z4A: "B' C'' A' CongA C' C'' A'"
        by (simp add: Z3 out2__conga)
      have Z4B: "B' C'' A' LtA A' C' B'"
      proof -
        have Z5: "C' C'' A' CongA B' C'' A'"
          using Z4A not_conga_sym by blast
        have Z6: "A' C' B' CongA A' C' B'"
          using P11 P4 conga_diff2 conga_refl by blast
        have "C' C'' A' LtA A' C' B'"
          using P4 Z1 Z2 between_symmetry l11_41 by blast
        thus ?thesis
          using Z5 Z6 conga_preserves_lta by auto
      qed
      have "B' C'' A' CongA B' C' A'"
        using P11 not_conga_sym by blast
      hence "Cong A C A' C' \<and> Cong B C B' C'"
        by (meson conga_right_comm Z4B not_lta_and_conga)
    }
    hence R2: "Cong A C A' C' \<and> Cong B C B' C'"
      using R1B \<open>Bet B' C'' C' \<Longrightarrow> Cong A C A' C' \<and> Cong B C B' C'\<close> by blast
    hence "C A B CongA C' A' B'"
      using P1 assms(2) l11_49 not_cong_2143 by blast
    thus ?thesis using R2 by auto
  qed
qed

lemma l11_51:
  assumes "A \<noteq> B" and
    "A \<noteq> C" and
    "B \<noteq> C" and
    "Cong A B A' B'" and
    "Cong A C A' C'" and
    "Cong B C B' C'"
  shows
    "B A C CongA B' A' C' \<and> A B C CongA A' B' C' \<and> B C A CongA B' C' A'"
proof -
  have "B A C Cong3 B' A' C' \<and> A B C Cong3 A' B' C' \<and> B C A Cong3 B' C' A'"
    using Cong3_def Cong_perm assms(4) assms(5) assms(6) by blast
  thus ?thesis
    using assms(1) assms(2) assms(3) cong3_conga by auto
qed

lemma conga_distinct:
  assumes "A B C CongA D E F"
  shows "A \<noteq> B \<and> C \<noteq> B \<and> D \<noteq> E \<and> F \<noteq> E"
  using CongA_def assms by auto

lemma l11_52:
  assumes "A B C CongA A' B' C'" and
    "Cong A C A' C'" and
    "Cong B C B' C'" and
    "B C Le A C"
  shows "Cong B A B' A' \<and> B A C CongA B' A' C' \<and> B C A CongA B' C' A'"
proof -
  have P1: "A \<noteq> B"
    using CongA_def assms(1) by blast
  have P2: "C \<noteq> B"
    using CongA_def assms(1) by blast
  have P3: "A' \<noteq> B'"
    using CongA_def assms(1) by blast
  have P4: "C' \<noteq> B'"
    using assms(1) conga_diff56 by auto
  have P5: "Cong B A B' A'"
  proof cases
    assume P6: "Col A B C"
    hence P7: "Bet A B C \<or> Bet B C A \<or> Bet C A B"
      using Col_def by blast
    {
      assume P8: "Bet A B C"
      hence "Bet A' B' C'"
        using assms(1) bet_conga__bet by blast
      hence "Cong B A B' A'"
        using P8 assms(2) assms(3) l4_3 not_cong_2143 by blast
    }
    {
      assume P9: "Bet B C A"
      hence P10: "B' Out A' C'"
        using Out_cases P2 assms(1) bet_out l11_21_a by blast
      hence P11: "Bet B' A' C' \<or> Bet B' C' A'"
        by (simp add: Out_def)
      {
        assume "Bet B' A' C'"
        hence "Cong B A B' A'"
          using P3 assms(2) assms(3) assms(4) bet_le_eq l5_6 by blast
      }
      {
        assume "Bet B' C' A'"
        hence "Cong B A B' A'"
          using Cong_perm P9 assms(2) assms(3) l2_11_b by blast
      }
      hence "Cong B A B' A'"
        using P11 \<open>Bet B' A' C' \<Longrightarrow> Cong B A B' A'\<close> by blast
    }
    {
      assume "Bet C A B"
      hence "Cong B A B' A'"
        using P1 assms(4) bet_le_eq between_symmetry by blast
    }
    thus ?thesis
      using P7 \<open>Bet A B C \<Longrightarrow> Cong B A B' A'\<close> \<open>Bet B C A \<Longrightarrow> Cong B A B' A'\<close> by blast
  next
    assume Z1: "\<not> Col A B C"
    obtain A'' where Z2: "B' Out A'' A' \<and> Cong B' A'' B A"
      using P1 P3 l6_11_existence by force
    hence Z3: "A' B' C' CongA A'' B' C'"
      by (simp add: P4 out2__conga out_trivial)
    have Z4: "A B C CongA A'' B' C'"
      using Z3 assms(1) not_conga by blast
    have Z5: "Cong A'' C' A C"
      using Z2 Z4 assms(3) cong2_conga_cong cong_4321 cong_symmetry by blast
    have Z6: "A'' B' C' Cong3 A B C"
      using Cong3_def Cong_perm Z2 Z5 assms(3) by blast
    have Z7: "Cong A'' C' A' C'"
      using Z5 assms(2) cong_transitivity by blast
    have Z8: "\<not> Col A' B' C'"
      by (metis Z1 assms(1) ncol_conga_ncol)
    hence Z9: "\<not> Col A'' B' C'"
      by (metis Z2 col_transitivity_1 not_col_permutation_4 out_col out_diff1)
    {
      assume Z9A: "A'' \<noteq> A'"
      have Z10: "Bet B' A'' A' \<or> Bet B' A' A''"
        using Out_def Z2 by auto
      {
        assume Z11: "Bet B' A'' A'"
        have Z12: "A'' C' B' LtA C' A'' A' \<and> A'' B' C' LtA C' A'' A'"
          by (simp add: Z11 Z9 Z9A l11_41)
        have Z13: "Cong A' C' A'' C'"
          using Cong_perm Z7 by blast
        have Z14: "\<not> Col A'' C' A'"
          by (metis Col_def Z11 Z9 Z9A col_transitivity_1)
        have Z15: "C' A'' A' CongA C' A' A'' \<longleftrightarrow> Cong C' A'' C' A'"
          by (simp add: Z14 l11_44_1)
        have Z16: "Cong C' A' C' A''"
          using Cong_perm Z7 by blast
        hence Z17: "Cong C' A'' C' A'"
          using Cong_perm by blast
        hence Z18: "C' A'' A' CongA C' A' A''"
          by (simp add: Z15)
        have Z19: "\<not> Col B' C' A''"
          using Col_perm Z9 by blast
        have Z20: "B' A' C' CongA A'' A' C'"
          by (metis col_conga_col Z11 Z3 Z9 Z9A bet_out_1 col_trivial_3 out2__conga out_trivial)
        have Z21: "\<not> Col B' C' A'"
          using Col_perm Z8 by blast
        hence Z22: "C' B' A' LtA C' A' B' \<longleftrightarrow> C' A' Lt C' B'"
          by (simp add: l11_44_2)
        have "A'' B' C' CongA C' B' A'"
          using Z3 conga_right_comm not_conga_sym by blast
        hence U1: "C' B' A' LtA C' A' B'" 
        proof -
          have "C' A'' A' CongA C' A' A''"
            by (metis Z15 Z17)
          hence "\<not> C' B' A' LtA C' A'' A' \<or> A'' B' C' LtA C' A' A''" 
            by (meson Z3 conga_preserves_lta lta_left_comm)
          hence "C' B' A' LtA C' A' B' \<or> A'' B' C' LtA A'' A' C' \<or> A'' = B'" 
            using Z12 \<open>A'' B' C' CongA C' B' A'\<close> conga_preserves_lta conga_refl 
            by (metis Z15 Z17 lta_right_comm)
          thus ?thesis 
            by (metis conga_diff1 conga_sym Z20 \<open>A'' B' C' CongA C' B' A'\<close> 
                conga_preserves_lta conga_right_comm)
        qed
        hence Z23: "C' A' Lt C' B'"
          using Z22 by auto
        have Z24: "C' A'' Lt C' B'"
          using Z16 Z23 cong2_lt__lt cong_reflexivity by blast
        have Z25: "C A Le C B"
        proof -
          have Z26: "Cong C' A'' C A"
            using Z5 not_cong_2143 by blast
          have "Cong C' B' C B"
            using assms(3) not_cong_4321 by blast
          thus ?thesis
            using l5_6 Z24 Z26 lt__le by blast
        qed
        hence Z27: "Cong C A C B"
          by (simp add: assms(4) le_anti_symmetry le_comm)
        have "Cong C' A'' C' B'"
          by (metis Cong_perm Z13 Z27 assms(2) assms(3) cong_transitivity)
        hence "False"
          using Z24 cong__nlt by blast
        hence "Cong B A B' A'" by simp
      }
      {
        assume W1: "Bet B' A' A''"
        have W2: "A' \<noteq> A''"
          using Z9A by auto
        have W3: "A' C' B' LtA C' A' A'' \<and> A' B' C' LtA C' A' A''"
          using W1 Z8 Z9A l11_41 by blast
        have W4: "Cong A' C' A'' C'"
          using Z7 not_cong_3412 by blast
        have "\<not> Col A'' C' A'"
          by (metis Col_def W1 Z8 Z9A col_transitivity_1)
        hence W6: "C' A'' A' CongA C' A' A'' \<longleftrightarrow> Cong C' A'' C' A'"
          using l11_44_1 by auto
        have W7: "Cong C' A' C' A''"
          using Z7 not_cong_4321 by blast
        hence W8: "Cong C' A'' C' A'"
          using W4 not_cong_4321 by blast
        have W9: "\<not> Col B' C' A''"
          by (simp add: Z9 not_col_permutation_1)
        have W10: "B' A'' C' CongA A' A'' C'"
          by (metis Out_def W1 Z9 Z9A bet_out_1 between_trivial not_col_distincts out2__conga)
        have W12: "C' B' A'' LtA C' A'' B' \<longleftrightarrow> C' A'' Lt C' B'"
          by (simp add: W9 l11_44_2)
        have W12A: "C' B' A'' LtA C' A'' B'"
        proof -
          have V1: "A' B' C' CongA C' B' A''"
            by (simp add: Z3 conga_right_comm)
          have "A' A'' C' CongA B' A'' C'"
            by (metis Out_def W1 \<open>\<not> Col A'' C' A'\<close> between_equality_2 not_col_distincts 
                or_bet_out out2__conga out_col)
          hence "C' A' A'' CongA C' A'' B'"
            by (meson W6 W8 conga_left_comm not_conga not_conga_sym)
          thus ?thesis
            using W3 V1 conga_preserves_lta by auto
        qed
        hence "C' A'' Lt C' B'" using W12 by auto
        hence W14: "C' A' Lt C' B'"
          using W8 cong2_lt__lt cong_reflexivity by blast
        have W15: "C A Le C B"
        proof -
          have Q1: "C' A'' Le C' B'"
            using W12 W12A lt__le by blast
          have Q2: "Cong C' A'' C A"
            using Z5 not_cong_2143 by blast
          have "Cong C' B' C B"
            using assms(3) not_cong_4321 by blast
          thus ?thesis using Q1 Q2 l5_6 by blast
        qed
        have "C B Le C A"
          by (simp add: assms(4) le_comm)
        hence "Cong C A C B"
          by (simp add: W15 le_anti_symmetry)
        hence "Cong C' A' C' B'"
          by (metis Cong_perm assms(2) assms(3) cong_inner_transitivity)
        hence "False"
          using W14 cong__nlt by blast
        hence "Cong B A B' A'" by simp
      }
      then  have "Cong B A B' A'"
        using Z10 \<open>Bet B' A'' A' \<Longrightarrow> Cong B A B' A'\<close> by blast
    }
    {
      assume "A'' = A'"
      hence "Cong B A B' A'"
        using Z2 not_cong_3412 by blast
    }
    thus ?thesis
      using \<open>A'' \<noteq> A' \<Longrightarrow> Cong B A B' A'\<close> by blast
  qed
  have P6: "A B C Cong3 A' B' C'"
    using Cong3_def Cong_perm P5 assms(2) assms(3) by blast
  thus ?thesis
    using P2 P5 assms(1) assms(3) assms(4) l11_49 le_zero by blast
qed

lemma l11_53:
  assumes "Per D C B" and
    "C \<noteq> D" and
    "A \<noteq> B" and
    "B \<noteq> C" and
    "Bet A B C"
  shows "C A D LtA C B D \<and> B D Lt A D"
proof -
  have P1: "C \<noteq> A"
    using assms(3) assms(5) between_identity by blast
  have P2: "\<not> Col B A D"
    using assms(1) assms(2) assms(3) assms(4) assms(5) bet_col bet_col1 col3 
      col_permutation_4 l8_9 by metis
  have P3: "A \<noteq> D"
    using P2 col_trivial_2 by blast
  have P4: "C A D LtA C B D"
  proof -
    have P4A: "B D A LtA D B C \<and> B A D LtA D B C"
      by (simp add: P2 assms(4) assms(5) l11_41)
    have P4AA:"A Out B C"
      using assms(3) assms(5) bet_out by auto
    have "A Out D D"
      using P3 out_trivial by auto
    hence P4B: "C A D CongA B A D" using P4AA
      by (simp add: out2__conga)
    hence P4C: "B A D CongA C A D"
      by (simp add: P4B conga_sym)
    have "D B C CongA C B D"
      using assms(1) assms(4) conga_pseudo_refl per_distinct_1 by auto
    thus ?thesis
      using P4A P4C conga_preserves_lta by blast
  qed
  obtain B' where P5: "C Midpoint B B' \<and> Cong D B D B'"
    using Per_def assms(1) by auto
  have K2: "A \<noteq> B'"
    using Bet_cases P5 assms(4) assms(5) between_equality_2 midpoint_bet by blast
  {
    assume "Col B D B'"
    hence "Col B A D"
      by (metis Col_cases P5 assms(1) assms(2) assms(4) col2__eq midpoint_col 
          midpoint_distinct_2 per_not_col)
    hence "False"
      by (simp add: P2)
  }
  hence P6: "\<not> Col B D B'" by blast
  hence "D B B' CongA D B' B \<longleftrightarrow> Cong D B D B'"
    by (simp add: l11_44_1)
  hence "D B B' CongA D B' B" using P5 by simp
  {
    assume K1: "Col A D B'"
    have "Col B' A B"
      using Col_def P5 assms(4) assms(5) midpoint_bet outer_transitivity_between by blast
    hence "Col B' B D"
      using K1 K2 Col_perm col_transitivity_2 by blast
    hence "Col B D B'"
      using Col_perm by blast
    hence "False"
      by (simp add: P6)
  }
  hence K3B: "\<not> Col A D B'" by blast
  hence K4: "D A B' LtA D B' A \<longleftrightarrow> D B' Lt D A"
    by (simp add: l11_44_2)
  have K4A: "C A D LtA C B' D"
    by (metis Midpoint_def P1 P3 P4 P5 P5 P6 assms(2) assms(4) col_trivial_1 
        cong_reflexivity conga_preserves_lta conga_refl l11_51 not_cong_2134)
  have "D B' Lt D A"
  proof -
    have "D A B' LtA D B' A"
    proof -
      have K5A: "A Out D D"
        using P3 out_trivial by auto
      have K5AA: "A Out B' C"
        using K2 Out_def P1 P5 assms(4) assms(5) midpoint_bet outer_transitivity_between2 
        by fast
      hence K5: "D A C CongA D A B'"
        by (simp add: K5A out2__conga)
      have K6A: "B' Out D D"
        using K3B not_col_distincts out_trivial by blast
      have "B' Out A C"
        using P5 K5AA assms(4) assms(5) between_equality_2 l6_4_2 midpoint_bet 
          midpoint_distinct_2 out_col outer_transitivity_between2 by metis
      hence K6: "D B' C CongA D B' A"
        by (simp add: K6A out2__conga)
      have "D A C LtA D B' C"
        by (simp add: K4A lta_comm)
      thus ?thesis
        using K5 K6 conga_preserves_lta by auto
    qed
    thus ?thesis
      by (simp add: K4)
  qed
  thus ?thesis
    using P4 P5 cong2_lt__lt cong_pseudo_reflexivity not_cong_4312 by blast
qed

lemma cong2_conga_obtuse__cong_conga2:
  assumes "Obtuse A B C" and
    "A B C CongA A' B' C'" and
    "Cong A C A' C'" and
    "Cong B C B' C'"
  shows "Cong B A B' A' \<and> B A C CongA B' A' C' \<and> B C A CongA B' C' A'"
proof -
  have "B C Le A C"
  proof cases
    assume "Col A B C"
    thus ?thesis
      by (simp add: assms(1) col_obtuse__bet l5_12_a)
  next
    assume "\<not> Col A B C"
    thus ?thesis
      using l11_46 assms(1) lt__le not_col_distincts by auto
  qed
  thus ?thesis
    using l11_52 assms(2) assms(3) assms(4) by blast
qed

lemma cong2_per2__cong_conga2:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Per A B C" and
    "Per A' B' C'" and
    "Cong A C A' C'" and
    "Cong B C B' C'"
  shows "Cong B A B' A' \<and> B A C CongA B' A' C' \<and> B C A CongA B' C' A'"
proof -
  have "B C Le A C \<and> \<not> Cong B C A C"
    using assms(1) assms(2) assms(3) cong__nlt l11_46 lt__le by blast
  hence "A B C CongA A' B' C'"
    using assms(2) assms(3) assms(4) assms(5) assms(6) cong_diff 
      cong_inner_transitivity cong_symmetry l11_16 by metis
  thus ?thesis 
    using \<open>B C Le A C \<and> \<not> Cong B C A C\<close> assms(5) assms(6) l11_52 by blast
qed

lemma cong2_per2__cong:
  assumes "Per A B C" and
    "Per A' B' C'" and
    "Cong A C A' C'" and
    "Cong B C B' C'"
  shows "Cong B A B' A'"
proof cases
  assume "B = C"
  thus ?thesis
    using assms(3) assms(4) cong_reverse_identity not_cong_2143 by blast
next
  assume "B \<noteq> C"
  show ?thesis
  proof cases
    assume "A = B"
    thus ?thesis
    proof -
      have "Cong A C B' C'"
        using \<open>A = B\<close> assms(4) by blast
      hence "B' = A'"
        by (meson Cong3_def Per_perm assms(2) assms(3) cong_inner_transitivity 
            cong_pseudo_reflexivity l8_10 l8_7)
      thus ?thesis
        using \<open>A = B\<close> cong_trivial_identity by blast
    qed
  next
    assume "A \<noteq> B"
    show ?thesis
    proof cases
      assume "A' = B'"
      thus ?thesis
        by (metis Cong3_def Per_perm \<open>A \<noteq> B\<close> assms(1) assms(3) assms(4) 
            cong_inner_transitivity cong_pseudo_reflexivity l8_10 l8_7)
    next
      assume "A' \<noteq> B'"
      thus ?thesis
        using cong2_per2__cong_conga2 \<open>A \<noteq> B\<close> \<open>B \<noteq> C\<close> assms(1) assms(2) assms(3) assms(4) 
        by blast
    qed
  qed
qed

lemma cong2_per2__cong_3:
  assumes "Per A B C"
    "Per A' B' C'" and
    "Cong A C A' C'" and
    "Cong B C B' C'"
  shows "A B C Cong3 A' B' C'"
  by (metis Cong3_def assms(1) assms(2) assms(3) assms(4) cong2_per2__cong cong_3_swap)

lemma cong_lt_per2__lt:
  assumes "Per A B C" and
    "Per A' B' C'" and
    "Cong A B A' B'" and
    "B C Lt B' C'"
  shows "A C Lt A' C'"
proof cases
  assume "A = B"
  thus ?thesis
    using assms(3) assms(4) cong_reverse_identity by blast
next
  assume "A \<noteq> B"
  show ?thesis
  proof cases
    assume "B = C"
    have "B' A' Lt C' A'" 
      by (metis cong_diff_2 l11_46 \<open>A \<noteq> B\<close> assms(2) assms(3) assms(4) lt_diff lt_right_comm)
    thus ?thesis 
      by (metis cong2_lt__lt cong_reflexivity cong_symmetry lt_comm \<open>B = C\<close> assms(3))
  next
    assume P0: "B \<noteq> C"
    have "B C Lt B' C'"
      by (simp add: assms(4))
    hence R1: "B C Le B' C' \<and> \<not> Cong B C B' C'"
      by (simp add: Lt_def)
    then obtain C0 where P1: "Bet B' C0 C' \<and> Cong B C B' C0"
      using Le_def by auto
    hence P2: "Per A' B' C0"
      by (metis Col_def Per_cases assms(2) bet_out_1 col_col_per_per col_trivial_1 
          l8_5 out_diff2)
    have "C0 A' Lt C' A'" using l11_53
      by (metis P1 P2 R1 P0 bet__lt2313 between_symmetry cong_diff)
    hence P3: "A' C0 Lt A' C'"
      using Lt_cases by blast
    have P4: "Cong A' C0 A C"
      using P1 P2 assms(1) assms(3) l10_12 not_cong_3412 by blast
    have "Cong A' C' A' C'"
      by (simp add: cong_reflexivity)
    thus ?thesis
      using cong2_lt__lt P3 P4 by blast
  qed
qed

lemma cong_le_per2__le:
  assumes "Per A B C" and
    "Per A' B' C'" and
    "Cong A B A' B'" and
    "B C Le B' C'"
  shows "A C Le A' C'"
proof cases
  assume "Cong B C B' C'"
  thus ?thesis
    using assms(1) assms(2) assms(3) cong__le l10_12 by blast
next
  assume "\<not> Cong B C B' C'"
  hence "B C Lt B' C'"
    using Lt_def assms(4) by blast
  thus ?thesis
    using assms(1) assms(2) assms(3) cong_lt_per2__lt lt__le by auto
qed

lemma lt2_per2__lt:
  assumes "Per A B C" and
    "Per A' B' C'" and
    "A B Lt A' B'" and
    "B C Lt B' C'"
  shows "A C Lt A' C'"
proof -
  have P2: "B A Lt B' A'"
    by (simp add: assms(3) lt_comm)
  have P3: "B C Le B' C' \<and> \<not> Cong B C B' C'"
    using assms(4) cong__nlt lt__le by auto
  then obtain C0 where P4: "Bet B' C0 C' \<and> Cong B C B' C0"
    using Le_def by auto
  have P4A: "B' \<noteq> C'"
    using assms(4) lt_diff by auto
  have "Col B' C' C0"
    using P4 bet_col not_col_permutation_5 by blast
  hence P5: "Per A' B' C0"
    using assms(2) P4A per_col by blast
  have P6: "A C Lt A' C0"
    by (meson P2 P4 P5 assms(1) cong_lt_per2__lt l8_2 lt_comm not_cong_2143)
  have "B' C0 Lt B' C'"
    by (metis P4 assms(4) bet__lt1213 cong__nlt)
  hence "A' C0 Lt A' C'"
    using P5 assms(2) cong_lt_per2__lt cong_reflexivity by blast
  thus ?thesis
    using P6 lt_transitivity by blast
qed

lemma le_lt_per2__lt:
  assumes "Per A B C" and
    "Per A' B' C'" and
    "A B Le A' B'" and
    "B C Lt B' C'"
  shows "A C Lt A' C'"
  using Lt_def assms(1) assms(2) assms(3) assms(4) cong_lt_per2__lt lt2_per2__lt by blast

lemma le2_per2__le:
  assumes "Per A B C" and
    "Per A' B' C'" and
    "A B Le A' B'" and
    "B C Le B' C'"
  shows "A C Le A' C'"
proof cases
  assume "Cong B C B' C'"
  thus ?thesis
    by (meson Per_cases cong_le_per2__le assms(1) assms(2) assms(3) le_comm not_cong_2143)
next
  assume "\<not> Cong B C B' C'"
  hence "B C Lt B' C'"
    by (simp add: Lt_def assms(4))
  thus ?thesis
    using assms(1) assms(2) assms(3) le_lt_per2__lt lt__le by blast
qed

lemma cong_lt_per2__lt_1:
  assumes "Per A B C" and
    "Per A' B' C'" and
    "A B Lt A' B'" and
    "Cong A C A' C'"
  shows "B' C' Lt B C"
  by (meson Gt_def assms(1) assms(2) assms(3) assms(4) cong2_per2__cong cong_4321 
      cong__nlt cong_symmetry lt2_per2__lt or_lt_cong_gt)

lemma symmetry_preserves_conga:
  assumes "A \<noteq> B" and "C \<noteq> B" and
    "M Midpoint A A'" and
    "M Midpoint B B'" and
    "M Midpoint C C'"
  shows "A B C CongA A' B' C'"
  by (metis Mid_perm assms(1) assms(2) assms(3) assms(4) assms(5) conga_trivial_1 
      l11_51 l7_13 symmetric_point_uniqueness)

lemma l11_57:
  assumes "A A' OS B B'" and
    "Per B A A'" and
    "Per B' A' A" and
    "A A' OS C C'" and
    "Per C A A'" and
    "Per C' A' A"
  shows "B A C CongA B' A' C'"
proof -
  obtain M where P1: "M Midpoint A A'"
    using midpoint_existence by auto
  obtain B'' where P2: "M Midpoint B B''"
    using symmetric_point_construction by auto
  obtain C'' where P3: "M Midpoint C C''"
    using symmetric_point_construction by auto
  have P4: "\<not> Col A A' B"
    using assms(1) col123__nos by auto
  have P5: "\<not> Col A A' C"
    using assms(4) col123__nos by auto
  have P6: "B A C CongA B'' A' C''"
    by (metis P1 P2 P3 assms(1) assms(4) os_distincts symmetry_preserves_conga)
  have "B'' A' C'' CongA B' A' C'"
  proof -
    have "B \<noteq> M"
      using P1 P4 midpoint_col not_col_permutation_2 by blast
    hence P7: "\<not> Col B'' A A'"
      using Mid_cases P1 P2 P4 mid_preserves_col not_col_permutation_3 by blast
    have K3: "Bet B'' A' B'"
    proof -
      have "Per B'' A' A"
        using P1 P2 assms(2) per_mid_per by blast
      have "Col B B'' M \<and> Col A A' M"
        using P1 P2 midpoint_col not_col_permutation_2 by blast
      hence "Coplanar B A A' B''"
        using Coplanar_def by auto
      hence "Coplanar A B' B'' A'"
        by (meson assms(1) between_trivial2 coplanar_trans_1 ncoplanar_perm_4 
            ncoplanar_perm_8 one_side_chara os__coplanar)
      hence P8: "Col B' B'' A'"
        using cop_per2__col P1 P2 P7 assms(2) assms(3) not_col_distincts per_mid_per 
        by blast
      have "A A' TS B B''"
        using P1 P2 P4 mid_two_sides by auto
      hence "A' A TS B'' B'"
        using assms(1) invert_two_sides l9_2 l9_8_2 by blast
      thus ?thesis
        using Col_cases P8 col_two_sides_bet by blast
    qed
    have "\<not> Col C'' A A'"
      using Col_def P1 P3 P5 l7_15 l7_2 not_col_permutation_5 by meson
    have "Bet C'' A' C'"
    proof -
      have Z2: "Col C' C'' A'"
      proof -
        have "Col C C'' M \<and> Col A A' M"
          using P1 P3 col_permutation_1 midpoint_col by blast
        hence "Coplanar C A A' C''"
          using Coplanar_def by blast
        hence Z1: "Coplanar A C' C'' A'"
          by (meson assms(4) between_trivial2 coplanar_trans_1 ncoplanar_perm_4 
              ncoplanar_perm_8 one_side_chara os__coplanar)
        have "Per C'' A' A"
          using P1 P3 assms(5) per_mid_per by blast
        thus ?thesis
          using Z1 P5 assms(6) col_trivial_1 cop_per2__col by blast
      qed
      have "A A' TS C C''"
        using P1 P3 P5 mid_two_sides by auto
      hence "A' A TS C'' C'"
        using assms(4) invert_two_sides l9_2 l9_8_2 by blast
      thus ?thesis
        using Col_cases Z2 col_two_sides_bet by blast
    qed
    thus ?thesis
      by (metis P6 K3 assms(1) assms(4) conga_diff45 conga_diff56 l11_14 os_distincts)
  qed
  thus ?thesis
    using P6 conga_trans by blast
qed

lemma cop3_orth_at__orth_at:
  assumes "\<not> Col D E F" and
    "Coplanar A B C D" and
    "Coplanar A B C E" and
    "Coplanar A B C F" and
    "X OrthAt A B C U V"
  shows "X OrthAt D E F U V"
proof -
  have P1: "\<not> Col A B C \<and> Coplanar A B C X"
    using OrthAt_def assms(5) by blast
  hence P2: "Coplanar D E F X"
    using assms(2) assms(3) assms(4) coplanar_pseudo_trans by blast
  {
    fix M
    assume "Coplanar A B C M"
    hence "Coplanar D E F M"
      using P1 assms(2) assms(3) assms(4) coplanar_pseudo_trans by blast
  }
  have T1: "U \<noteq> V"
    using OrthAt_def assms(5) by blast
  have T2: "Col U V X"
    using OrthAt_def assms(5) by auto
  {
    fix P Q
    assume P7: "Coplanar D E F P \<and> Col U V Q"
    hence "Coplanar A B C P"
      by (meson \<open>\<And>M. Coplanar A B C M \<Longrightarrow> Coplanar D E F M\<close> assms(1) assms(2) 
          assms(3) assms(4) l9_30)
    hence "Per P X Q" using P7 OrthAt_def assms(5) by blast
  }
  thus ?thesis using assms(1)
    by (simp add: OrthAt_def P2 T1 T2)
qed

lemma col2_orth_at__orth_at:
  assumes "U \<noteq> V" and
    "Col P Q U" and
    "Col P Q V" and
    "X OrthAt A B C P Q"
  shows "X OrthAt A B C U V"
proof -
  have "Col P Q X"
    using OrthAt_def assms(4) by auto
  hence "Col U V X"
    by (metis OrthAt_def assms(2) assms(3) assms(4) col3)
  thus ?thesis
    using OrthAt_def assms(1) assms(2) assms(3) assms(4) colx by presburger
qed

lemma col_orth_at__orth_at:
  assumes "U \<noteq> W" and
    "Col U V W" and
    "X OrthAt A B C U V"
  shows "X OrthAt A B C U W"
  using assms(1) assms(2) assms(3) col2_orth_at__orth_at col_trivial_3 by blast

lemma orth_at_symmetry:
  assumes "X OrthAt A B C U V"
  shows "X OrthAt A B C V U"
  by (metis assms col2_orth_at__orth_at col_trivial_2 col_trivial_3)

lemma orth_at_distincts:
  assumes "X OrthAt A B C U V"
  shows "A \<noteq> B \<and> B \<noteq> C \<and> A \<noteq> C \<and> U \<noteq> V"
  using OrthAt_def assms not_col_distincts by fastforce

lemma orth_at_chara:
  "X OrthAt A B C X P \<longleftrightarrow>
(\<not> Col A B C \<and> X \<noteq> P \<and> Coplanar A B C X \<and> (\<forall> D.(Coplanar A B C D \<longrightarrow> Per D X P)))"
proof -
  {
    assume "X OrthAt A B C X P"
    hence "\<not> Col A B C \<and> X \<noteq> P \<and> Coplanar A B C X \<and> (\<forall> D.(Coplanar A B C D \<longrightarrow> Per D X P))"
      using OrthAt_def col_trivial_2 by auto
  }
  {
    assume T1: "\<not> Col A B C \<and> X \<noteq> P \<and> 
  Coplanar A B C X \<and> (\<forall> D.(Coplanar A B C D \<longrightarrow> Per D X P))"
    {
      fix P0 Q
      assume "Coplanar A B C P0 \<and> Col X P Q"
      hence "Per P0 X Q" using T1 OrthAt_def per_col by auto
    }
    hence "X OrthAt A B C X P"
      by (simp add: T1 \<open>\<And>Q P0. Coplanar A B C P0 \<and> Col X P Q \<Longrightarrow> Per P0 X Q\<close> 
          OrthAt_def col_trivial_3)
  }
  thus ?thesis
    using \<open>X OrthAt A B C X P \<Longrightarrow> \<not> Col A B C \<and> X \<noteq> P \<and> 
  Coplanar A B C X \<and> (\<forall>D. Coplanar A B C D \<longrightarrow> Per D X P)\<close> 
    by blast
qed

lemma cop3_orth__orth:
  assumes "\<not> Col D E F" and
    "Coplanar A B C D" and
    "Coplanar A B C E" and
    "Coplanar A B C F" and
    "A B C Orth U V"
  shows "D E F Orth U V"
  using Orth_def assms(1) assms(2) assms(3) assms(4) assms(5) cop3_orth_at__orth_at by blast

lemma col2_orth__orth:
  assumes "U \<noteq> V" and
    "Col P Q U" and
    "Col P Q V" and
    "A B C Orth P Q"
  shows "A B C Orth U V"
  by (meson Orth_def col2_orth_at__orth_at Tarski_neutral_dimensionless_axioms 
      assms(1) assms(2) assms(3) assms(4))

lemma col_orth__orth:
  assumes "U \<noteq> W" and
    "Col U V W" and
    "A B C Orth U V"
  shows "A B C Orth U W"
  by (meson assms(1) assms(2) assms(3) col2_orth__orth col_trivial_3)

lemma orth_symmetry:
  assumes "A B C Orth U V"
  shows "A B C Orth V U"
  by (meson Orth_def assms orth_at_symmetry)

lemma orth_distincts:
  assumes "A B C Orth U V"
  shows "A \<noteq> B \<and> B \<noteq> C \<and> A \<noteq> C \<and> U \<noteq> V"
  using Orth_def assms orth_at_distincts by blast

lemma col_cop_orth__orth_at:
  assumes "A B C Orth U V" and
    "Coplanar A B C X" and
    "Col U V X"
  shows "X OrthAt A B C U V"
proof -
  obtain Y where P1:
    "\<not> Col A B C \<and> U \<noteq> V \<and> Coplanar A B C Y \<and> Col U V Y \<and>
(\<forall> P Q. (Coplanar A B C P \<and> Col U V Q) \<longrightarrow> Per P Y Q)"
    by (metis OrthAt_def Orth_def assms(1))
  hence P2: "X = Y"
    using assms(2) assms(3) per_distinct_1 by blast
  {
    fix P Q
    assume "Coplanar A B C P \<and> Col U V Q"
    hence "Per P X Q" using P1 P2 by auto
  }
  thus ?thesis
    using OrthAt_def Orth_def assms(1) assms(2) assms(3) by auto
qed

lemma l11_60_aux:
  assumes "\<not> Col A B C" and
    "Cong A P A Q" and
    "Cong B P B Q" and
    "Cong C P C Q" and
    "Coplanar A B C D"
  shows "Cong D P D Q"
proof -
  obtain M where P1: "Bet P M Q \<and> Cong P M M Q"
    by (meson Midpoint_def midpoint_existence Tarski_neutral_dimensionless_axioms)
  obtain X where P2: " (Col A B X \<and> Col C D X) \<or>
(Col A C X \<and> Col B D X) \<or>
(Col A D X \<and> Col B C X)"
    using assms(5) Coplanar_def by auto
  {
    assume "Col A B X \<and> Col C D X"
    have "C \<noteq> X" 
      using \<open>Col A B X \<and> Col C D X\<close> assms(1) by auto
    have "Cong X P X Q" 
      by (metis l4_17 not_col_distincts \<open>Col A B X \<and> Col C D X\<close> assms(1) assms(2) assms(3))
    hence "Cong D P D Q"
      by (metis l4_17 \<open>C \<noteq> X\<close> \<open>Col A B X \<and> Col C D X\<close> assms(4) not_col_permutation_5)
  }
  {
    assume "Col A C X \<and> Col B D X"
    have "B \<noteq> X" 
      using Col_cases \<open>Col A C X \<and> Col B D X\<close> assms(1) by blast
    have "Cong X P X Q" 
      by (metis l4_17 \<open>Col A C X \<and> Col B D X\<close> assms(1) assms(2) assms(4) not_col_distincts)
    hence "Cong D P D Q"
      by (metis l4_17 \<open>B \<noteq> X\<close> \<open>Col A C X \<and> Col B D X\<close> assms(3) not_col_permutation_5)
  }
  {
    assume "Col A D X \<and> Col B C X"
    have "A \<noteq> X" 
      by (metis not_col_permutation_1 \<open>Col A D X \<and> Col B C X\<close> assms(1))
    have "Cong X P X Q" 
      by (metis l4_17 not_col_distincts \<open>Col A D X \<and> Col B C X\<close> assms(1) assms(3) assms(4))
    hence "Cong D P D Q"
      by (metis col_permutation_5 l4_17 \<open>A \<noteq> X\<close> \<open>Col A D X \<and> Col B C X\<close> assms(2))
  }
  thus ?thesis
    using P2 \<open>Col A B X \<and> Col C D X \<Longrightarrow> Cong D P D Q\<close> 
      \<open>Col A C X \<and> Col B D X \<Longrightarrow> Cong D P D Q\<close> by blast
qed

lemma l11_60:
  assumes "\<not> Col A B C" and
    "Per A D P" and
    "Per B D P" and
    "Per C D P" and
    "Coplanar A B C E"
  shows "Per E D P"
  by (meson Per_def assms(1) assms(2) assms(3) assms(4) assms(5) l11_60_aux per_double_cong)

lemma l11_60_bis:
  assumes "\<not> Col A B C" and
    "D \<noteq> P" and
    "Coplanar A B C D" and
    "Per A D P" and
    "Per B D P" and
    "Per C D P"
  shows "D OrthAt A B C D P"
  using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) l11_60 orth_at_chara by auto

lemma l11_61:
  assumes "A \<noteq> A'" and
    "A \<noteq> B" and
    "A \<noteq> C" and
    "Coplanar A A' B B'" and
    "Per B A A'" and
    "Per B' A' A" and
    "Coplanar A A' C C'" and
    "Per C A A'" and
    "Per B A C"
  shows "Per B' A' C'"
proof -
  have P1: "\<not> Col C A A'"
    using assms(1) assms(3) assms(8) per_col_eq by blast
  obtain C'' where P2: "A A' Perp C'' A' \<and> A A' OS C C''" using l10_15
    using Col_perm P1 col_trivial_2 by blast
  have P6: "B' \<noteq> A"
    using assms(1) assms(6) per_distinct by blast
  have P8: "\<not> Col A' A C''"
    using P2 not_col_permutation_4 one_side_not_col124 by blast
  have P9: "Per A' A' B'"
    by (simp add: l8_2 l8_5)
  have P10: "Per A A' B'"
    by (simp add: assms(6) l8_2)
  {
    fix B'
    assume "A A' OS B B' \<and> Per B' A' A"
    hence "B A C CongA B' A' C''" using l11_17
      by (meson P2 Perp_cases l11_57 Tarski_neutral_dimensionless_axioms assms(5) assms(8) perp_per_1)
    hence "Per B' A' C''"
      using assms(9) l11_17 by blast
  }
  hence Q1: "\<forall> B'. (A A' OS B B' \<and> Per B' A' A) \<longrightarrow> Per B' A' C''" by simp
  {
    fix B'
    assume P12: "Coplanar A A' B B' \<and> Per B' A' A \<and> B' \<noteq> A"
    have "Per B' A' C''"
    proof cases
      assume "B' = A'"
      thus ?thesis
        by (simp add: Per_perm l8_5)
    next
      assume P13: "B' \<noteq> A'"
      have P14: "\<not> Col B' A' A"
        using P12 P13 assms(1) l8_9 by auto
      have P15: "\<not> Col B A A'"
        using assms(1) assms(2) assms(5) per_not_col by auto
      hence Z1: "A A' TS B B' \<or> A A' OS B B'"
        using P12 P14 cop__one_or_two_sides not_col_permutation_5 by blast
      {
        assume "A A' OS B B'"
        hence "Per B' A' C''"
          by (simp add: P12 \<open>\<And>B'a. A A' OS B B'a \<and> Per B'a A' A \<Longrightarrow> Per B'a A' C''\<close>)
      }
      {
        assume Q2: "A A' TS B B'"
        obtain B'' where Z2: "Bet B' A' B'' \<and> Cong A' B'' A' B'"
          using segment_construction by blast
        have "B' \<noteq> B''"
          using P13 Z2 bet_neq12__neq by blast
        hence Z4: "A' \<noteq> B''"
          using Z2 cong_diff_4 by blast
        hence "A A' TS B'' B'"
          by (meson TS_def Z2 Q2 bet__ts invert_two_sides l9_2 not_col_permutation_1)
        hence Z5: "A A' OS B B''"
          using Q2 l9_8_1 by auto
        have "Per B'' A' A"
          using P12 P13 Z2 bet_col col_per2__per l8_2 l8_5 by blast
        hence "Per C'' A' B''"
          using l8_2 Q1 Z5 by blast
        hence "Per B' A' C''"
          by (metis Col_def Per_perm l8_3 Z2 Z4)
      }
      thus ?thesis using Z1
        using \<open>A A' OS B B' \<Longrightarrow> Per B' A' C''\<close> by blast
    qed
  }
  hence "\<forall> B'. (Coplanar A A' B B' \<and> Per B' A' A \<and> B' \<noteq> A) \<longrightarrow> Per B' A' C''"
    by simp
  hence "Per B' A' C''"
    using P6 assms(4) assms(6) by blast
  hence P11: "Per C'' A' B'"
    using Per_cases by auto
  have "Coplanar A' A C'' C'"
    by (meson P1 P2 assms(7) coplanar_trans_1 ncoplanar_perm_6 ncoplanar_perm_8 os__coplanar)
  thus ?thesis
    using P8 P9 P10 P11 l8_2 l11_60 by blast
qed

lemma l11_61_bis:
  assumes "D OrthAt A B C D P" and
    "D E Perp E Q" and
    "Coplanar A B C E" and
    "Coplanar D E P Q"
  shows "E OrthAt A B C E Q"
proof -
  have P4: "D \<noteq> E"
    using assms(2) perp_not_eq_1 by auto
  have P5: "E \<noteq> Q"
    using assms(2) perp_not_eq_2 by auto
  have "\<exists> D'. (D E Perp D' D \<and> Coplanar A B C D')"
  proof -
    obtain F where T1: "Coplanar A B C F \<and> \<not> Col D E F"
      using P4 ex_ncol_cop by blast
    obtain D' where T2: "D E Perp D' D \<and> Coplanar D E F D'"
      using P4 ex_perp_cop by blast
    have "Coplanar A B C D'"
    proof -
      have T3A: "\<not> Col A B C"
        using OrthAt_def assms(1) by auto
      have T3B: "Coplanar A B C D"
        using OrthAt_def assms(1) by blast
      hence T4: "Coplanar D E F A"
        by (meson T1 T3A assms(3) coplanar_pseudo_trans ncop_distincts)
      have T5: "Coplanar D E F B"
        using T1 T3A T3B assms(3) coplanar_pseudo_trans ncop_distincts by blast
      have "Coplanar D E F C"
        using T1 T3A T3B assms(3) coplanar_pseudo_trans ncop_distincts by blast
      thus ?thesis
        using T1 T2 T4 T5 coplanar_pseudo_trans by blast
    qed
    thus ?thesis
      using T2 by auto
  qed
  then obtain D' where R1: "D E Perp D' D \<and> Coplanar A B C D'" by auto
  hence R2: "D \<noteq> D'"
    using perp_not_eq_2 by blast
  {
    fix M
    assume R3: "Coplanar A B C M"
    have "Col D P P"
      by (simp add: col_trivial_2)
    hence "Per E D P"
      using assms(1) assms(3) orth_at_chara by auto
    hence R4: "Per P D E" using l8_2 by auto
    have R5: "Per Q E D"
      using Perp_cases assms(2) perp_per_2 by blast
    have R6: "Coplanar D E D' M"
    proof -
      have S1: "\<not> Col A B C"
        using OrthAt_def assms(1) by auto
      have "Coplanar A B C D"
        using OrthAt_def assms(1) by auto
      thus ?thesis
        using S1 assms(3) R1 R3 coplanar_pseudo_trans by blast
    qed
    have R7: "Per D' D E"
      using Perp_cases R1 perp_per_1 by blast
    have "Per D' D P"
      using R1 assms(1) orth_at_chara by blast
    hence "Per P D D'"
      using Per_cases by blast
    hence "Per Q E M"
      using l11_61 R4 R5 R6 R7 OrthAt_def P4 R2 assms(1) assms(4) by blast
    hence "Per M E Q" using l8_2 by auto
  }
  {
    fix P0 Q0
    assume "Coplanar A B C P0 \<and> Col E Q Q0"
    hence "Per P0 E Q0"
      using P5 \<open>\<And>M. Coplanar A B C M \<Longrightarrow> Per M E Q\<close> per_col by blast
  }
  thus ?thesis
    using OrthAt_def P5 assms(1) assms(3) col_trivial_3 by auto
qed

lemma l11_62_unicity:
  assumes "Coplanar A B C D" and
    "Coplanar A B C D'" and
    "\<forall> E. Coplanar A B C E \<longrightarrow> Per E D P" and
    "\<forall> E. Coplanar A B C E \<longrightarrow> Per E D' P"
  shows "D = D'"
  by (metis assms(1) assms(2) assms(3) assms(4) l8_8 not_col_distincts per_not_colp)

lemma l11_62_unicity_bis:
  assumes "X OrthAt A B C X U" and
    "Y OrthAt A B C Y U"
  shows "X = Y"
proof -
  have P1: "Coplanar A B C X"
    using assms(1) orth_at_chara by blast
  have P2: "Coplanar A B C Y"
    using assms(2) orth_at_chara by blast
  {
    fix E
    assume "Coplanar A B C E"
    hence "Per E X U"
      using OrthAt_def assms(1) col_trivial_2 by auto
  }
  {
    fix E
    assume "Coplanar A B C E"
    hence "Per E Y U"
      using assms(2) orth_at_chara by auto
  }
  thus ?thesis
    by (meson P1 P2 \<open>\<And>E. Coplanar A B C E \<Longrightarrow> Per E X U\<close> l8_2 l8_7)
qed

lemma orth_at2__eq:
  assumes "X OrthAt A B C U V" and
    "Y OrthAt A B C U V"
  shows "X = Y"
proof -
  have P1: "Coplanar A B C X"
    using assms(1)
    by (simp add: OrthAt_def)
  have P2: "Coplanar A B C Y"
    using OrthAt_def assms(2) by auto
  {
    fix E
    assume "Coplanar A B C E"
    hence "Per E X U"
      using OrthAt_def assms(1) col_trivial_3 by auto
  }
  {
    fix E
    assume "Coplanar A B C E"
    hence "Per E Y U"
      using OrthAt_def assms(2) col_trivial_3 by auto
  }
  thus ?thesis
    by (meson P1 P2 Per_perm \<open>\<And>E. Coplanar A B C E \<Longrightarrow> Per E X U\<close> l8_7)
qed

lemma col_cop_orth_at__eq:
  assumes "X OrthAt A B C U V" and
    "Coplanar A B C Y" and
    "Col U V Y"
  shows "X = Y"
proof -
  have "Y OrthAt A B C U V"
    using Orth_def assms(1) assms(2) assms(3) col_cop_orth__orth_at by blast
  thus ?thesis
    using assms(1) orth_at2__eq by auto
qed

lemma orth_at__ncop1:
  assumes "U \<noteq> X" and
    "X OrthAt A B C U V"
  shows "\<not> Coplanar A B C U"
  using assms(1) assms(2) col_cop_orth_at__eq not_col_distincts by blast

lemma orth_at__ncop2:
  assumes "V \<noteq> X" and
    "X OrthAt A B C U V"
  shows "\<not> Coplanar A B C V"
  using assms(1) assms(2) col_cop_orth_at__eq not_col_distincts by blast

lemma orth_at__ncop:
  assumes "X OrthAt A B C X P"
  shows "\<not> Coplanar A B C P"
  by (metis assms orth_at__ncop2 orth_at_distincts)

lemma l11_62_existence:
  "\<exists> D. (Coplanar A B C D \<and> (\<forall> E. (Coplanar A B C E \<longrightarrow> Per E D P)))"
proof cases
  assume "Coplanar A B C P"
  thus ?thesis
    using l8_5 by auto
next
  assume P1: "\<not> Coplanar A B C P"
  hence P2: "\<not> Col A B C"
    using ncop__ncol by auto
  have "\<not> Col A B P"
    using P1 ncop__ncols by auto
  then obtain D0 where P4: "Col A B D0 \<and> A B Perp P D0" using l8_18_existence by blast
  have P5: "Coplanar A B C D0"
    using P4 ncop__ncols by auto
  have "A \<noteq> B"
    using P2 not_col_distincts by auto
  then obtain D1 where P10: "A B Perp D1 D0 \<and> Coplanar A B C D1"
    using ex_perp_cop by blast
  have P11: "\<not> Col A B D1"
    using P10 P4 perp_not_col2 by blast
  {
    fix D
    assume "Col D0 D1 D"
    hence "Coplanar A B C D"
      by (metis P10 P5 col_cop2__cop perp_not_eq_2)
  }
  obtain A0 where P11: "A \<noteq> A0 \<and> B \<noteq> A0 \<and> D0 \<noteq> A0 \<and> Col A B A0"
    using P4 diff_col_ex3 by blast
  have P12: "Coplanar A B C A0"
    using P11 ncop__ncols by blast
  have P13: "Per P D0 A0"
    using l8_16_1 P11 P4 by blast
  show ?thesis
  proof cases
    assume Z1: "Per P D0 D1"
    {
      fix E
      assume R1: "Coplanar A B C E"
      have R2: "\<not> Col A0 D1 D0"
        by (metis P10 P11 P4 col_permutation_5 colx perp_not_col2)
      have R3: "Per A0 D0 P"
        by (simp add: P13 l8_2)
      have R4: "Per D1 D0 P"
        by (simp add: Z1 l8_2)
      have R5: "Per D0 D0 P"
        by (simp add: l8_2 l8_5)
      have "Coplanar A0 D1 D0 E"
        using R1 P2 P12 P10 P5 coplanar_pseudo_trans by blast
      hence "Per E D0 P"
        using l11_60 R2 R3 R4 R5 by blast
    }
    thus ?thesis using P5 by auto
  next
    assume S1: "\<not> Per P D0 D1"
    {
      assume S2: "Col D0 D1 P"
      have "\<forall> D. Col D0 D1 D \<longrightarrow> Coplanar A B C D"
        by (simp add: \<open>\<And>Da. Col D0 D1 Da \<Longrightarrow> Coplanar A B C Da\<close>)
      hence "False"
        using P1 S2 by blast
    }
    hence S2A: "\<not> Col D0 D1 P" by blast
    then obtain D where S3: "Col D0 D1 D \<and> D0 D1 Perp P D"
      using l8_18_existence by blast
    have S4: "Coplanar A B C D"
      by (simp add: S3 \<open>\<And>Da. Col D0 D1 Da \<Longrightarrow> Coplanar A B C Da\<close>)
    {
      fix E
      assume S5: "Coplanar A B C E"
      have S6: "D \<noteq> D0"
        using S1 S3 l8_2 perp_per_1 by blast
      have S7: "Per D0 D P"
        by (metis Perp_cases S3 S6 perp_col perp_per_1)
      have S8: "Per D D0 A0"
      proof -
        have V4: "D0 \<noteq> D1"
          using P10 perp_not_eq_2 by blast
        have V6: "Per A0 D0 D1"
          using P10 P11 P4 l8_16_1 l8_2 by blast
        thus ?thesis
          using S3 V4 V6 l8_2 per_col by blast
      qed
      have S9: "Per A0 D P"
      proof -
        obtain A0' where W1: "D Midpoint A0 A0'"
          using symmetric_point_construction by auto
        obtain D0' where W2: "D Midpoint D0 D0'"
          using symmetric_point_construction by auto
        have "Cong P A0 P A0'"
        proof -
          have V3: "Cong P D0 P D0'"
            using S7 W2 l8_2 per_double_cong by blast
          have V4: "Cong D0 A0 D0' A0'"
            using W1 W2 cong_4321 l7_13 by blast
          have "Per P D0' A0'"
          proof -
            obtain P' where V5: "D Midpoint P P'"
              using symmetric_point_construction by blast
            have "Per P' D0 A0"
            proof -
              have "\<not> Col P D D0"
                by (metis S2A S3 S6 col2__eq col_permutation_1)
              thus ?thesis
                by (metis (full_types) P13 S3 S8 V5 S2A col_per2__per midpoint_col)
            qed
            thus ?thesis
              using midpoint_preserves_per V5 Mid_cases W1 W2 by blast
          qed
          thus ?thesis using l10_12 P13 V3 V4 by blast
        qed
        thus ?thesis
          using Per_def Per_perm W1 by blast
      qed
      have S13: "Per D D P"
        using Per_perm l8_5 by blast
      have S14: "\<not> Col D0 A0 D" 
        using P11 S7 S9 per_not_col Col_perm S6 S8 by blast
      have "Coplanar A B C D" 
        using S4 by auto
      hence "Coplanar D0 A0 D E"
        using P12 P2 P5 S5 coplanar_pseudo_trans by blast
      hence "Per E D P"
        using S13 S14 S7 S9 l11_60 by blast
    }
    thus ?thesis using S4 by blast
  qed
qed

lemma l11_62_existence_bis:
  assumes "\<not> Coplanar A B C P"
  shows "\<exists> X. X OrthAt A B C X P"
proof -
  obtain X where P1: "Coplanar A B C X \<and> (\<forall> E. Coplanar A B C E \<longrightarrow> Per E X P)"
    using l11_62_existence by blast
  hence P2: "X \<noteq> P"
    using assms by auto
  have P3: "\<not> Col A B C"
    using assms ncop__ncol by auto
  thus ?thesis
    using P1 P2 P3 orth_at_chara by auto
qed

lemma l11_63_aux:
  assumes "Coplanar A B C D" and
    "D \<noteq> E" and
    "E OrthAt A B C E P"
  shows "\<exists> Q. (D E OS P Q \<and> A B C Orth D Q)"
proof -
  have P1: "\<not> Col A B C"
    using OrthAt_def assms(3) by blast
  have P2: "E \<noteq> P"
    using OrthAt_def assms(3) by blast
  have P3: "Coplanar A B C E"
    using OrthAt_def assms(3) by blast
  have P4: "\<forall> P0 Q. (Coplanar A B C P0 \<and> Col E P Q) \<longrightarrow> Per P0 E Q"
    using OrthAt_def assms(3) by blast
  have P5: "\<not> Coplanar A B C P"
    using assms(3) orth_at__ncop by auto
  have P6: "Col D E D"
    by (simp add: col_trivial_3)
  have "\<not> Col D E P"
    using P3 P5 assms(1) assms(2) col_cop2__cop by blast
  then obtain Q where P6: "D E Perp Q D \<and> D E OS P Q"
    using P6 l10_15 by blast
  have "A B C Orth D Q"
  proof -
    obtain F where P7: "Coplanar A B C F \<and> \<not> Col D E F"
      using assms(2) ex_ncol_cop by blast
    obtain D' where P8: "D E Perp D' D \<and> Coplanar D E F D'"
      using assms(2) ex_perp_cop by presburger
    have P9: "\<not> Col D' D E"
      using P8 col_permutation_1 perp_not_col by blast
    have P10: "Coplanar D E F A"
      by (meson P1 P3 P7 assms(1) coplanar_pseudo_trans ncop_distincts)
    have P11: "Coplanar D E F B"
      by (meson P1 P3 P7 assms(1) coplanar_pseudo_trans ncop_distincts)
    have P12: "Coplanar D E F C"
      by (meson P1 P3 P7 assms(1) coplanar_pseudo_trans ncop_distincts)
    hence "D OrthAt A B C D Q"
    proof -
      have "Per D' D Q"
      proof -
        obtain E' where Y1: "D E Perp E' E \<and> Coplanar D E F E'"
          using assms(2) ex_perp_cop by blast
        have Y2: "E \<noteq> E'"
          using Y1 perp_distinct by auto
        have Y5: "Coplanar E D E' D'"
          by (meson P7 P8 Y1 coplanar_perm_12 coplanar_perm_7 coplanar_trans_1 
              not_col_permutation_2)
        have Y6: "Per E' E D"
          by (simp add: Perp_perm perp_per_2 Y1)
        have Y7: "Per D' D E"
          using P8 col_trivial_2 col_trivial_3 l8_16_1 by blast
        have Y8: "Coplanar E D P Q"
          using P6 ncoplanar_perm_6 os__coplanar by blast
        have Y9: "Per P E D" using P4
          using assms(1) assms(3) l8_2 orth_at_chara by blast
        have Y10: "Coplanar A B C E'"
          using P10 P11 P12 P7 Y1 coplanar_pseudo_trans by blast
        hence Y11: "Per E' E P"
          using P4 Y10 col_trivial_2 by auto
        have "E \<noteq> D" using assms(2) by blast
        thus ?thesis
          using l11_61 Y2 assms(2) P2 Y5 Y6 Y7 Y8 Y9 Y10 Y11 by blast
      qed
      hence X1: "D OrthAt D' D E D Q" using l11_60_bis
        by (metis OS_def P6 P9 Per_perm TS_def l8_5 col_trivial_3 invert_one_side 
            ncop_distincts perp_per_1)
      have X3: "Coplanar D' D E A"
        using P10 P7 P8 coplanar_perm_14 coplanar_trans_1 not_col_permutation_3 by blast
      have X4: "Coplanar D' D E B"
        using P11 P7 P8 coplanar_perm_14 coplanar_trans_1 not_col_permutation_3 by blast
      have "Coplanar D' D E C"
        using P12 P7 P8 coplanar_perm_14 coplanar_trans_1 not_col_permutation_3 by blast
      thus ?thesis
        using X1 P1 X3 X4 cop3_orth_at__orth_at by blast
    qed
    thus ?thesis
      using Orth_def by blast
  qed
  thus ?thesis using P6 by blast
qed

lemma l11_63_existence:
  assumes "Coplanar A B C D" and
    "\<not> Coplanar A B C P"
  shows "\<exists> Q. A B C Orth D Q"
  using Orth_def assms(1) assms(2) l11_62_existence_bis l11_63_aux by fastforce

lemma l8_21_3:
  assumes "Coplanar A B C D" and
    "\<not> Coplanar A B C X"
  shows
    "\<exists> P T. (A B C Orth D P \<and> Coplanar A B C T \<and> Bet X T P)"
proof -
  obtain E where P1: "E OrthAt A B C E X"
    using assms(2) l11_62_existence_bis by blast
  thus ?thesis
  proof cases
    assume P2: "D = E"
    obtain Y where P3: "Bet X D Y \<and> Cong D Y D X"
      using segment_construction by blast
    have P4: "D \<noteq> X"
      using assms(1) assms(2) by auto
    have P5: "A B C Orth D X"
      using Orth_def P1 P2 by auto
    have P6: "D \<noteq> Y"
      using P3 P4 cong_reverse_identity by blast
    have "Col D X Y"
      using Col_def Col_perm P3 by blast
    hence "A B C Orth D Y"
      using P5 P6 col_orth__orth by auto
    thus ?thesis
      using P3 assms(1) by blast
  next
    assume K1: "D \<noteq> E"
    then obtain P' where P7: "D E OS X P' \<and> A B C Orth D P'"
      using P1 assms(1) l11_63_aux by blast
    have P8: "\<not> Col A B C"
      using assms(2) ncop__ncol by auto
    have P9: "E \<noteq> X"
      using P7 os_distincts by auto
    have P10: "\<forall> P Q. (Coplanar A B C P \<and> Col E X Q) \<longrightarrow> Per P E Q"
      using OrthAt_def P1 by auto
    have P11: "D OrthAt A B C D P'"
      by (simp add: P7 assms(1) col_cop_orth__orth_at col_trivial_3)
    have P12: "D \<noteq> P'"
      using P7 os_distincts by auto
    have P13: "\<not> Coplanar A B C P'"
      using P11 orth_at__ncop by auto
    have P14: "\<forall> P Q. (Coplanar A B C P \<and> Col D P' Q) \<longrightarrow> Per P D Q"
      using OrthAt_def P11 by auto
    obtain P where P15: "Bet P' D P \<and> Cong D P D P'"
      using segment_construction by blast
    have P16: "D E TS X P"
    proof -
      have P16A: "D E OS P' X" using P7 one_side_symmetry by blast
      hence "D E TS P' P"
        by (metis P12 P15 bet__ts cong_diff_3 one_side_not_col123)
      thus ?thesis using l9_8_2 P16A by blast
    qed
    obtain T where P17: "Col T D E \<and> Bet X T P"
      using P16 TS_def by blast
    have P18: "D \<noteq> P"
      using P16 ts_distincts by blast
    have "Col D P' P"
      using Col_def Col_perm P15 by blast
    hence "A B C Orth D P"
      using P18 P7 col_orth__orth by blast
    thus ?thesis using col_cop2__cop
      by (meson P1 P17 orth_at_chara K1 assms(1) col_permutation_1)
  qed
qed

lemma mid2_orth_at2__cong:
  assumes "X OrthAt A B C X P" and
    "Y OrthAt A B C Y Q" and
    "X Midpoint P P'" and
    "Y Midpoint Q Q'"
  shows "Cong P Q P' Q'"
proof -
  have Q1: "\<not> Col A B C"
    using assms(1) col__coplanar orth_at__ncop by blast
  have Q2: "X \<noteq> P"
    using assms(1) orth_at_distincts by auto
  have Q3: "Coplanar A B C X"
    using OrthAt_def assms(1) by auto
  have Q4: "\<forall> P0 Q. (Coplanar A B C P0 \<and> Col X P Q) \<longrightarrow> Per P0 X Q"
    using OrthAt_def assms(1) by blast
  have Q5: "Y \<noteq> P"
    by (metis assms(1) assms(2) orth_at__ncop2 orth_at_chara)
  have Q6: "Coplanar A B C Y"
    using OrthAt_def assms(2) by blast
  have Q7: "\<forall> P Q0. (Coplanar A B C P \<and> Col Y Q Q0) \<longrightarrow> Per P Y Q0"
    using OrthAt_def assms(2) by blast
  obtain Z where P1: "Z Midpoint X Y"
    using midpoint_existence by auto
  obtain R where P2: "Z Midpoint P R"
    using symmetric_point_construction by auto
  obtain R' where P3: "Z Midpoint P' R'"
    using symmetric_point_construction by auto
  have T1: "Coplanar A B C Z"
    using P1 Q3 Q6 bet_cop2__cop midpoint_bet by blast
  hence "Per Z X P"
    using Q4 assms(1) orth_at_chara by auto
  hence T2: "Cong Z P Z P'"
    using assms(3) per_double_cong by blast
  have T3: "Cong R Z R' Z"
    by (metis Cong_perm Midpoint_def P2 P3 T2 cong_transitivity)
  have T4: "Cong R Q R' Q'"
    by (meson P1 P2 P3 assms(3) assms(4) l7_13 not_cong_4321 symmetry_preserves_midpoint)
  have "Per Z Y Q"
    using Q7 T1 assms(2) orth_at_chara by auto
  hence T5: "Cong Z Q Z Q'"
    using assms(4) per_double_cong by auto
  have "R \<noteq> Z"
    by (metis P2 P3 Q2 T3 assms(3) cong_diff_3 l7_17_bis midpoint_not_midpoint)
  thus ?thesis
    using P2 P3 T2 T3 T4 T5 five_segment l7_2 midpoint_bet by blast
qed

lemma orth_at2_tsp__ts:
  assumes "P \<noteq> Q" and
    "P OrthAt A B C P X" and
    "Q OrthAt A B C Q Y" and
    "A B C TSP X Y"
  shows "P Q TS X Y"
proof -
  obtain T where P0: "Coplanar A B C T \<and> Bet X T Y"
    using TSP_def assms(4) by auto
  have P1: "\<not> Col A B C"
    using assms(4) ncop__ncol tsp__ncop1 by blast
  have P2: "P \<noteq> X "
    using assms(2) orth_at_distincts by auto
  have P3: "Coplanar A B C P"
    using OrthAt_def assms(2) by blast
  have P4: "\<forall> D. Coplanar A B C D \<longrightarrow> Per D P X"
    using assms(2) orth_at_chara by blast
  have P5: "Q \<noteq> Y"
    using assms(3) orth_at_distincts by auto
  have P6: "Coplanar A B C Q"
    using OrthAt_def assms(3) by blast
  have P7: "\<forall> D. Coplanar A B C D \<longrightarrow> Per D Q Y"
    using assms(3) orth_at_chara by blast
  have P8: "\<not> Col X P Q"
    using P3 P6 assms(1) assms(4) col_cop2__cop not_col_permutation_2 tsp__ncop1 by blast
  have P9: "\<not> Col Y P Q"
    using P3 P6 assms(1) assms(4) col_cop2__cop not_col_permutation_2 tsp__ncop2 by blast
  have "Col T P Q"
  proof -
    obtain X' where Q1: "P Midpoint X X'"
      using symmetric_point_construction by auto
    obtain Y' where Q2: "Q Midpoint Y Y'"
      using symmetric_point_construction by auto
    have "Per T P X"
      using P0 P4 by auto
    hence Q3: "Cong T X T X'"
      using Q1 per_double_cong by auto
    have "Per T Q Y"
      using P0 P7 by auto
    hence Q4: "Cong T Y T Y'" using Q2 per_double_cong by auto
    have "Cong X Y X' Y'"
      using P1 Q1 Q2 assms(2) assms(3) mid2_orth_at2__cong by blast
    hence "X T Y Cong3 X' T Y'"
      using Cong3_def Q3 Q4 not_cong_2143 by blast
    hence "Bet X' T Y'"
      using l4_6 P0 by blast
    thus ?thesis
      using Q1 Q2 Q3 Q4 Col_def P0 between_symmetry l7_22 by blast
  qed
  thus ?thesis
    using P0 P8 P9 TS_def by blast
qed

lemma orth_dec:
  shows "A B C Orth U V \<or> \<not> A B C Orth U V" 
  by auto

lemma orth_at_dec:
  shows "X OrthAt A B C U V \<or> \<not> X OrthAt A B C U V" 
  by auto

lemma tsp_dec:
  shows "A B C TSP X Y \<or> \<not> A B C TSP X Y" 
  by auto

lemma osp_dec:
  shows "A B C OSP X Y \<or> \<not> A B C OSP X Y" 
  by auto

lemma ts2__inangle:
  assumes "A C TS B P" and
    "B P TS A C"
  shows "P InAngle A B C"
  by (metis InAngle_def assms(1) assms(2) bet_out ts2__ex_bet2 ts_distincts)

lemma os_ts__inangle:
  assumes "B P TS A C" and
    "B A OS C P"
  shows "P InAngle A B C"
proof -
  have P1: "\<not> Col A B P"
    using TS_def assms(1) by auto
  have P2: "\<not> Col B A C"
    using assms(2) col123__nos by blast
  obtain P' where P3: "B Midpoint P P'"
    using symmetric_point_construction by blast
  hence P4: "\<not> Col B A P'"
    by (metis assms(2) col_one_side col_permutation_5 midpoint_col 
        midpoint_distinct_2 one_side_not_col124)
  have P5: "(B \<noteq> P' \<and> B P TS A C \<and> Bet P B P') \<longrightarrow> (P InAngle A B C \<or> P' InAngle A B C)"
    using two_sides_in_angle by auto
  hence P6: "P InAngle A B C \<or> P' InAngle A B C"
    using P3 P4 assms(1) midpoint_bet not_col_distincts by blast
  {
    assume "P' InAngle A B C"
    hence P7: "A B OS P' C"
      using Col_cases P2 P4 in_angle_one_side by blast
    hence P8: "\<not> A B TS P' C"
      using l9_9 by auto
    have "B A TS P P'"
      using P1 P3 P4 bet__ts midpoint_bet not_col_distincts not_col_permutation_4 by auto
    hence "B A TS C P'"
      using P7 assms(2) invert_one_side l9_2 l9_8_2 l9_9 by blast
    hence "B A TS P' C"
      using l9_2 by blast
    hence "A B TS P' C"
      by (simp add: invert_two_sides)
    hence "P InAngle A B C"
      using P8 by auto
  }
  thus ?thesis
    using P6 by blast
qed

lemma os2__inangle:
  assumes "B A OS C P" and
    "B C OS A P"
  shows "P InAngle A B C"
  using assms(1) assms(2) col124__nos l9_9_bis os_ts__inangle two_sides_cases by blast

lemma acute_conga__acute:
  assumes "Acute A B C" and
    "A B C CongA D E F"
  shows "Acute D E F"
proof -
  have "D E F LeA A B C"
    by (simp add: assms(2) conga__lea456123)
  thus ?thesis
    using acute_lea_acute assms(1) by blast
qed

lemma acute_out2__acute:
  assumes "B Out A' A" and
    "B Out C' C" and
    "Acute A B C"
  shows "Acute A' B C'"
  by (meson out2__conga acute_conga__acute assms(1) assms(2) assms(3))

lemma conga_obtuse__obtuse:
  assumes "Obtuse A B C" and
    "A B C CongA D E F"
  shows "Obtuse D E F"
  using assms(1) assms(2) conga__lea lea_obtuse_obtuse by blast

lemma obtuse_out2__obtuse:
  assumes "B Out A' A" and
    "B Out C' C" and
    "Obtuse A B C"
  shows "Obtuse A' B C'"
  by (meson out2__conga assms(1) assms(2) assms(3) conga_obtuse__obtuse)

lemma bet_lea__bet:
  assumes "Bet A B C" and
    "A B C LeA D E F"
  shows "Bet D E F"
proof -
  have "A B C CongA D E F"
    by (metis assms(1) assms(2) l11_31_2 lea_asym lea_distincts)
  thus ?thesis
    using assms(1) bet_conga__bet by blast
qed

lemma out_lea__out:
  assumes "E Out D F" and
    "A B C LeA D E F"
  shows "B Out A C"
proof -
  have "D E F CongA A B C"
    using l11_31_1 lea_asym lea_distincts assms(1) assms(2) by fastforce
  thus ?thesis
    using assms(1) out_conga_out by blast
qed

lemma bet2_lta__lta:
  assumes "A B C LtA D E F" and
    "Bet A B A'" and
    "A' \<noteq> B" and
    "Bet D E D'" and
    "D' \<noteq> E"
  shows "D' E F LtA A' B C"
proof -
  have P1: "D' E F LeA A' B C"
    by (metis Bet_cases assms(1) assms(2) assms(3) assms(4) assms(5) l11_36_aux2 
        lea_distincts lta__lea)
  have "\<not> D' E F CongA A' B C"
    by (metis assms(1) assms(2) assms(4) between_symmetry conga_sym l11_13 
        lta_distincts not_lta_and_conga)
  thus ?thesis
    by (simp add: LtA_def P1)
qed

lemma lea123456_lta__lta:
  assumes "A B C LeA D E F" and
    "D E F LtA G H I"
  shows "A B C LtA G H I"
proof -
  have "\<not> G H I LeA F E D"
    by (metis (no_types) lea__nlta lta_left_comm assms(2))
  hence "\<not> A B C CongA G H I"
    by (metis lta_distincts assms(1) assms(2) conga_pseudo_refl l11_30)
  thus ?thesis
    by (meson LtA_def lea_trans assms(1) assms(2))
qed

lemma lea456789_lta__lta:
  assumes "A B C LtA D E F" and
    "D E F LeA G H I"
  shows "A B C LtA G H I"
  by (meson LtA_def assms(1) assms(2) conga__lea456123 lea_trans lta__nlea)

lemma acute_per__lta:
  assumes "Acute A B C" and
    "D \<noteq> E" and
    "E \<noteq> F" and
    "Per D E F"
  shows "A B C LtA D E F"
proof -
  obtain G H I where P1: "Per G H I \<and> A B C LtA G H I"
    using Acute_def assms(1) by auto
  hence "G H I CongA D E F"
    using assms(2) assms(3) assms(4) l11_16 lta_distincts by blast
  thus ?thesis
    by (metis P1 conga_preserves_lta conga_refl lta_distincts)
qed

lemma obtuse_per__lta:
  assumes "Obtuse A B C" and
    "D \<noteq> E" and
    "E \<noteq> F" and
    "Per D E F"
  shows "D E F LtA A B C"
proof -
  obtain G H I where P1: "Per G H I \<and> G H I LtA A B C"
    using Obtuse_def assms(1) by auto
  hence "G H I CongA D E F"
    using assms(2) assms(3) assms(4) l11_16 lta_distincts by blast
  thus ?thesis
    by (metis P1 l11_51 assms(1) cong_reflexivity conga_preserves_lta obtuse_distincts)
qed

lemma acute_obtuse__lta:
  assumes "Acute A B C" and
    "Obtuse D E F"
  shows "A B C LtA D E F"
  by (metis Acute_def assms(1) assms(2) lta_distincts lta_trans obtuse_per__lta)

lemma lea_in_angle:
  assumes "A B P LeA A B C" and
    "A B OS C P"
  shows "P InAngle A B C"
proof -
  obtain T where P3: "T InAngle A B C \<and> A B P CongA A B T"
    using LeA_def assms(1) by blast
  thus ?thesis
    by (metis assms(2) conga_preserves_in_angle conga_refl not_conga_sym 
        one_side_symmetry os_distincts)
qed

lemma acute_bet__obtuse:
  assumes "Bet A B A'" and
    "A' \<noteq> B" and
    "Acute A B C"
  shows "Obtuse A' B C"
proof cases
  assume P1: "Col A B C"
  show ?thesis
  proof cases
    assume "Bet A B C"
    thus ?thesis
      using P1 acute_col__out assms(3) not_bet_and_out by blast
  next
    assume "\<not> Bet A B C"
    hence "B Out A C" 
      using P1 or_bet_out by blast
    hence "Bet C B A'" 
      using assms(1) bet_out__bet by blast
    hence "Bet A' B C" 
      using between_symmetry by blast
    thus ?thesis 
      using \<open>\<not> Bet A B C\<close> assms(2) bet__obtuse between_trivial by blast
  qed
next
  assume P2: "\<not> Col A B C"
  then obtain D where P3: "A B Perp D B \<and> A B OS C D"
    using col_trivial_2 l10_15 by blast
  {
    assume P4: "Col C B D"
    hence "Per A B C"
    proof -
      have P5: "B \<noteq> D"
        using P3 perp_not_eq_2 by auto
      have "Per A B D"
        using P3 Perp_perm perp_per_2 by blast
      thus ?thesis
        using P4 P5 not_col_permutation_2 per_col by blast
    qed
    hence "A B C LtA A B C"
      by (metis Acute_def acute_per__lta assms(3) lta_distincts)
    hence "False"
      by (simp add: nlta)
  }
  hence P6: "\<not> Col C B D" by auto
  have P7: "B A' OS C D"
    by (metis P3 assms(1) assms(2) bet_col col2_os__os l5_3)
  have T1: "Per A B D"
    by (simp add: P3 perp_left_comm perp_per_1)
  have Q1: "B C TS A' A"
    using P2 assms(1) assms(2) bet__ts l9_2 not_col_permutation_1 by auto
  have "A B C LtA A B D"
    using P2 P6 T1 acute_per__lta assms(3) not_col_distincts by auto
  hence "A B C LeA A B D"
    by (simp add: lta__lea)
  hence "C InAngle A B D"
    by (simp add: P3 lea_in_angle one_side_symmetry)
  hence "C InAngle D B A"
    using l11_24 by blast
  hence "C B TS D A"
    by (simp add: P2 P6 in_angle_two_sides not_col_permutation_1 not_col_permutation_4)
  hence "B C TS D A"
    using invert_two_sides by blast
  hence "B C OS A' D"
    using Q1 l9_8_1 by auto
  hence T1A: "D InAngle A' B C"
    by (simp add: P7 os2__inangle)
  hence "A B D CongA A' B D"
    by (metis Per_cases T1 conga_comm l11_18_1 acute_distincts assms(1) assms(3) inangle_distincts)
  hence T2: "A B D LeA A' B C"
    using LeA_def T1A by auto
  {
    assume "A B D CongA A' B C"
    hence "False"
      by (metis OS_def P7 T1 TS_def out2__conga \<open>A B C LtA A B D\<close> \<open>A B D CongA A' B D\<close> 
          \<open>\<And>thesis. (\<And>D. A B Perp D B \<and> A B OS C D \<Longrightarrow> thesis) \<Longrightarrow> thesis\<close> 
          col_trivial_2 invert_one_side l11_17 l11_19 not_lta_and_conga out_trivial)
  }
  hence "\<not> A B D CongA A' B C" by auto
  hence "A B D LtA A' B C"
    using T2 LtA_def by auto
  thus ?thesis
    using T1 Obtuse_def by blast
qed

lemma bet_obtuse__acute:
  assumes "Bet A B A'" and
    "A' \<noteq> B" and
    "Obtuse A B C"
  shows "Acute A' B C"
proof cases
  assume P1: "Col A B C"
  thus ?thesis
  proof cases
    assume "Bet A B C"
    hence "B Out A' C"
      using Out_def assms(1) assms(2) assms(3) l5_2 obtuse_distincts by meson
    thus ?thesis
      by (simp add: out__acute)
  next
    assume "\<not> Bet A B C"
    thus ?thesis
      using P1 assms(3) col_obtuse__bet by blast
  qed
next
  assume P2: "\<not> Col A B C"
  then obtain D where P3: "A B Perp D B \<and> A B OS C D"
    using col_trivial_2 l10_15 by blast
  {
    assume P3A: "Col C B D"
    have P3B: "B \<noteq> D"
      using P3 perp_not_eq_2 by blast
    have P3C: "Per A B D"
      using P3 Perp_perm perp_per_2 by blast
    hence "Per A B C"
      using P3A P3B not_col_permutation_2 per_col by blast
    hence "A B C LtA A B C"
      using P2 assms(3) not_col_distincts obtuse_per__lta by auto
    hence "False"
      by (simp add: nlta)
  }
  hence P4: "\<not> Col C B D" by auto
  have "Col B A A'"
    using Col_def Col_perm assms(1) by blast
  hence P5: "B A' OS C D"
    using P3 assms(2) invert_one_side col2_os__os col_trivial_3 by blast
  have P7: "Per A' B D"
    by (meson Col_def P3 Per_perm Tarski_neutral_dimensionless_axioms assms(1) 
        col_trivial_2 l8_16_1)
  have "A' B C LtA A' B D"
  proof -
    have P8: "A' B C LeA A' B D"
    proof -
      have P9: "C InAngle A' B D"
      proof -
        have P10: "B A' OS D C"
          by (simp add: P5 one_side_symmetry)
        have "B D OS A' C"
        proof -
          have P6: "\<not> Col A B D"
            using P3 col124__nos by auto
          hence P11: "B D TS A' A"
            using Col_perm P5 assms(1) bet__ts l9_2 os_distincts by blast
          have "A B D LtA A B C"
          proof -
            have P11A: "A \<noteq> B"
              using P2 col_trivial_1 by auto
            have P11B: "B \<noteq> D"
              using P4 col_trivial_2 by blast
            have "Per A B D"
              using P3 Perp_cases perp_per_2 by blast
            thus ?thesis
              by (simp add: P11A P11B obtuse_per__lta assms(3))
          qed
          hence "A B D LeA A B C"
            by (simp add: lta__lea)
          hence "D InAngle A B C"
            by (simp add: P3 lea_in_angle)
          hence "D InAngle C B A"
            using l11_24 by blast
          hence "D B TS C A"
            by (simp add: P4 P6 in_angle_two_sides not_col_permutation_4)
          hence "B D TS C A"
            by (simp add: invert_two_sides)
          thus ?thesis
            using OS_def P11 by blast
        qed
        thus ?thesis
          by (simp add: P10 os2__inangle)
      qed
      have "A' B C CongA A' B C"
        using assms(2) assms(3) conga_refl obtuse_distincts by blast
      thus ?thesis
        by (simp add: P9 inangle__lea)
    qed
    {
      assume "A' B C CongA A' B D"
      hence "B Out C D"
        using P5 conga_os__out invert_one_side by blast
      hence "False"
        using P4 not_col_permutation_4 out_col by blast
    }
    hence "\<not> A' B C CongA A' B D" by auto
    thus ?thesis
      by (simp add: LtA_def P8)
  qed
  thus ?thesis
    using Acute_def P7 by blast
qed

lemma inangle_dec:
  "P InAngle A B C \<or> \<not> P InAngle A B C" 
  by blast

lemma lea_dec:
  "A B C LeA D E F \<or> \<not> A B C LeA D E F" 
  by blast

lemma lta_dec:
  "A B C LtA D E F \<or> \<not> A B C LtA D E F" 
  by blast

lemma lea_total:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "D \<noteq> E" and
    "E \<noteq> F"
  shows "A B C LeA D E F \<or> D E F LeA A B C"
proof cases
  assume P1: "Col A B C"
  show ?thesis
  proof cases
    assume "B Out A C"
    thus ?thesis
      using assms(3) assms(4) l11_31_1 by auto
  next
    assume "\<not> B Out A C"
    thus ?thesis
      by (metis P1 assms(1) assms(2) assms(3) assms(4) l11_31_2 or_bet_out)
  qed
next
  assume P2: "\<not> Col A B C"
  show ?thesis
  proof cases
    assume P3: "Col D E F"
    show ?thesis
    proof cases
      assume "E Out D F"
      thus ?thesis
        using assms(1) assms(2) l11_31_1 by auto
    next
      assume "\<not> E Out D F"
      thus ?thesis
        by (metis P3 assms(1) assms(2) assms(3) assms(4) l11_31_2 l6_4_2)
    qed
  next
    assume P4: "\<not> Col D E F"
    show ?thesis
    proof cases
      assume "A B C LeA D E F"
      thus ?thesis
        by simp
    next
      assume P5: "\<not> A B C LeA D E F"
      obtain P where P6: "D E F CongA A B P \<and> A B OS P C"
        using P2 P4 angle_construction_1 by blast
      hence P7: "B A OS C P"
        using invert_one_side one_side_symmetry by blast
      have "B C OS A P"
      proof -
        {
          assume "Col P B C"
          hence P7B: "B Out C P"
            using Col_cases P7 col_one_side_out by blast
          have "A B C CongA D E F"
          proof -
            have P7C: "A B P CongA D E F"
              by (simp add: P6 conga_sym)
            have P7D: "B Out A A"
              by (simp add: assms(1) out_trivial)
            have P7E: "E Out D D"
              by (simp add: assms(3) out_trivial)
            have "E Out F F"
              using assms(4) out_trivial by auto
            thus ?thesis
              using P7B P7C P7D P7E l11_10 by blast
          qed
          hence "A B C LeA D E F"
            by (simp add: conga__lea)
          hence "False"
            by (simp add: P5)
        }
        hence P8: "\<not> Col P B C" by auto
        {
          assume T0: "B C TS A P"
          have "A B C CongA D E F"
          proof -
            have T1: "A B C LeA A B P"
            proof -
              have T1A: "C InAngle A B P"
                by (simp add: P7 T0 one_side_symmetry os_ts__inangle)
              have "A B C CongA A B C"
                using assms(1) assms(2) conga_refl by auto
              thus ?thesis
                by (simp add: T1A inangle__lea)
            qed
            have T2: "A B C CongA A B C"
              using assms(1) assms(2) conga_refl by auto
            have "A B P CongA D E F"
              by (simp add: P6 conga_sym)
            thus ?thesis
              using P5 T1 T2 l11_30 by blast
          qed
          hence "A B C LeA D E F"
            by (simp add: conga__lea)
          hence "False"
            by (simp add: P5)
        }
        hence "\<not> B C TS A P" by auto
        thus ?thesis
          using Col_perm P7 P8 one_side_symmetry os_ts1324__os two_sides_cases by blast
      qed
      hence "P InAngle A B C"
        using P7 os2__inangle by blast
      hence "D E F LeA A B C"
        using P6 LeA_def by blast
      thus ?thesis
        by simp
    qed
  qed
qed

lemma or_lta2_conga:
  assumes "A \<noteq> B" and
    "C \<noteq> B" and
    "D \<noteq> E" and
    "F \<noteq> E"
  shows "A B C LtA D E F \<or> D E F LtA A B C \<or> A B C CongA D E F"
proof -
  have P1: "A B C LeA D E F \<or> D E F LeA A B C"
    using assms(1) assms(2) assms(3) assms(4) lea_total by auto
  {
    assume "A B C LeA D E F"
    hence "A B C LtA D E F \<or> D E F LtA A B C \<or> A B C CongA D E F"
      using LtA_def by blast
  }
  {
    assume "D E F LeA A B C"
    hence "A B C LtA D E F \<or> D E F LtA A B C \<or> A B C CongA D E F"
      using LtA_def conga_sym by blast
  }
  thus ?thesis
    using P1 \<open>A B C LeA D E F \<Longrightarrow> A B C LtA D E F \<or> D E F LtA A B C \<or> A B C CongA D E F\<close> 
    by blast
qed

lemma angle_partition:
  assumes "A \<noteq> B" and
    "B \<noteq> C"
  shows "Acute A B C \<or> Per A B C \<or> Obtuse A B C"
proof -
  obtain A' B' D' where P1: "\<not> (Bet A' B' D' \<or> Bet B' D' A' \<or> Bet D' A' B')"
    using lower_dim by auto
  hence "\<not> Col A' B' D'"
    by (simp add: Col_def)
  obtain C' where P3: "A' B' Perp C' B'"
    by (metis P1 Perp_perm between_trivial2 perp_exists)
  hence P4: "A B C LtA A' B' C' \<or> A' B' C' LtA A B C \<or> A B C CongA A' B' C'"
    by (metis P1 assms(1) assms(2) between_trivial2 or_lta2_conga perp_not_eq_2)
  {
    assume "A B C LtA A' B' C'"
    hence "Acute A B C \<or> Per A B C \<or> Obtuse A B C"
      using Acute_def P3 Perp_cases perp_per_2 by blast
  }
  {
    assume "A' B' C' LtA A B C"
    hence "Acute A B C \<or> Per A B C \<or> Obtuse A B C"
      using Obtuse_def P3 Perp_cases perp_per_2 by blast
  }
  {
    assume "A B C CongA A' B' C'"
    hence "Acute A B C \<or> Per A B C \<or> Obtuse A B C"
      by (metis P3 Perp_cases conga_sym l11_17 perp_per_2)
  }
  thus ?thesis
    using P4 \<open>A B C LtA A' B' C' \<Longrightarrow> Acute A B C \<or> Per A B C \<or> Obtuse A B C\<close> 
      \<open>A' B' C' LtA A B C \<Longrightarrow> Acute A B C \<or> Per A B C \<or> Obtuse A B C\<close> by auto
qed

lemma acute_chara_1:
  assumes "Bet A B A'" and
    "B \<noteq> A'" and
    "Acute A B C"
  shows "A B C LtA A' B C"
proof -
  have "Obtuse A' B C"
    using acute_bet__obtuse assms(1) assms(2) assms(3) by blast
  thus ?thesis
    by (simp add: acute_obtuse__lta assms(3))
qed

lemma acute_chara_2:
  assumes "Bet A B A'" and
    "A B C LtA A' B C"
  shows "Acute A B C"
  by (metis Acute_def acute_chara_1 angle_partition assms(1) assms(2) bet_obtuse__acute 
      between_symmetry lta_distincts not_and_lta)

lemma acute_chara:
  assumes "Bet A B A'" and
    "B \<noteq> A'"
  shows "Acute A B C \<longleftrightarrow> A B C LtA A' B C"
  using acute_chara_1 acute_chara_2 assms(1) assms(2) by blast

lemma obtuse_chara:
  assumes "Bet A B A'" and
    "B \<noteq> A'"
  shows "Obtuse A B C \<longleftrightarrow> A' B C LtA A B C"
  by (metis Obtuse_def acute_bet__obtuse acute_chara assms(1) assms(2) 
      bet_obtuse__acute between_symmetry lta_distincts)

lemma conga__acute:
  assumes "A B C CongA A C B"
  shows "Acute A B C"
  by (metis acute_sym angle_partition assms conga_distinct conga_obtuse__obtuse l11_17 l11_43)

lemma cong__acute:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Cong A B A C"
  shows "Acute A B C"
  using angle_partition assms(1) assms(2) assms(3) cong__nlt l11_46 lt_left_comm by blast

lemma nlta__lea:
  assumes "\<not> A B C LtA D E F" and
    "A \<noteq> B" and
    "B \<noteq> C" and
    "D \<noteq> E" and
    "E \<noteq> F"
  shows "D E F LeA A B C"
  by (metis LtA_def assms(1) assms(2) assms(3) assms(4) assms(5) conga__lea456123 or_lta2_conga)

lemma nlea__lta:
  assumes "\<not> A B C LeA D E F" and
    "A \<noteq> B" and
    "B \<noteq> C" and
    "D \<noteq> E" and
    "E \<noteq> F"
  shows "D E F LtA A B C"
  using assms(1) assms(2) assms(3) assms(4) assms(5) nlta__lea by blast

lemma triangle_strict_inequality:
  assumes "Bet A B D" and
    "Cong B C B D" and
    "\<not> Bet A B C"
  shows "A C Lt A D"
proof cases
  assume P1: "Col A B C"
  hence P2: "B Out A C"
    using assms(3) not_out_bet by auto
  {
    assume "Bet B A C"
    hence P3: "A C Le A D"
      by (meson assms(1) assms(2) cong__le l5_12_a le_transitivity)
    have "\<not> Cong A C A D"
      by (metis Out_def P1 P2 assms(1) assms(2) assms(3) l4_18)
    hence "A C Lt A D"
      by (simp add: Lt_def P3)
  }
  {
    assume "Bet A C B"
    hence P5: "Bet A C D"
      using assms(1) between_exchange4 by blast
    hence P6: "A C Le A D"
      by (simp add: bet__le1213)
    have "\<not> Cong A C A D"
      using P5 assms(1) assms(3) between_cong by blast
    hence "A C Lt A D"
      by (simp add: Lt_def P6)
  }
  thus ?thesis
    using P1 \<open>Bet B A C \<Longrightarrow> A C Lt A D\<close> assms(3) between_symmetry third_point by blast
next
  assume T1: "\<not> Col A B C"
  have T2: "A \<noteq> D"
    using T1 assms(1) between_identity col_trivial_1 by auto
  have T3: "\<not> Col A C D"
    by (metis Col_def T1 T2 assms(1) col_transitivity_2)
  have T4: "Bet D B A"
    using Bet_perm assms(1) by blast
  have T5: "C D A CongA D C B"
  proof -
    have T6: "C D B CongA D C B"
      by (metis assms(1) assms(2) assms(3) between_trivial conga_comm l11_44_1_a not_conga_sym)
    have T7: "D Out C C"
      using T3 not_col_distincts out_trivial by blast
    have T8: "D Out A B"
      by (metis assms(1) assms(2) assms(3) bet_out_1 cong_diff l6_6)
    have T9: "C Out D D"
      using T7 out_trivial by force
    have "C Out B B"
      using T1 not_col_distincts out_trivial by auto
    thus ?thesis
      using T6 T7 T8 T9 l11_10 by blast
  qed
  have "A D C LtA A C D"
  proof -
    have "B InAngle D C A"
      by (metis InAngle_def T1 T3 T4 not_col_distincts out_trivial)
    hence "C D A LeA D C A"
      using T5 LeA_def by auto
    hence T10: "A D C LeA A C D"
      by (simp add: lea_comm)
    have "\<not> A D C CongA A C D"
      by (metis Col_perm T3 assms(1) assms(2) assms(3) bet_col l11_44_1_b l4_18 
          not_bet_distincts not_cong_3412)
    thus ?thesis
      using LtA_def T10 by blast
  qed
  thus ?thesis
    by (simp add: l11_44_2_b)
qed

lemma triangle_inequality:
  assumes "Bet A B D" and
    "Cong B C B D"
  shows "A C Le A D"
proof cases
  assume "Bet A B C"
  thus ?thesis
    by (metis assms(1) assms(2) between_cong_3 cong__le le_reflexivity)
next
  assume "\<not> Bet A B C"
  hence "A C Lt A D"
    using assms(1) assms(2) triangle_strict_inequality by auto
  thus ?thesis
    by (simp add: Lt_def)
qed

lemma triangle_strict_inequality_2:
  assumes "Bet A' B' C'" and
    "Cong A B A' B'" and
    "Cong B C B' C'"  and
    "\<not> Bet A B C"
  shows "A C Lt A' C'"
proof -
  obtain D where P1: "Bet A B D \<and> Cong B D B C"
    using segment_construction by blast
  hence P2: "A C Lt A D"
    using P1 assms(4) cong_symmetry triangle_strict_inequality by blast
  have "Cong A D A' C'"
    using P1 assms(1) assms(2) assms(3) cong_transitivity l2_11_b by blast
  thus ?thesis
    using P2 cong2_lt__lt cong_reflexivity by blast
qed

lemma triangle_inequality_2:
  assumes "Bet A' B' C'" and
    "Cong A B A' B'" and
    "Cong B C B' C'"
  shows "A C Le A' C'"
proof -
  obtain D where P1: "Bet A B D \<and> Cong B D B C"
    using segment_construction by blast
  have P2: "A C Le A D"
    by (meson P1 triangle_inequality not_cong_3412)
  have "Cong A D A' C'"
    using P1 assms(1) assms(2) assms(3) cong_transitivity l2_11_b by blast
  thus ?thesis
    using P2 cong__le le_transitivity by blast
qed

lemma triangle_strict_reverse_inequality:
  assumes "A Out B D" and
    "Cong A C A D" and
    "\<not> A Out B C"
  shows "B D Lt B C"
proof cases
  assume "Col A B C"
  thus ?thesis
    using assms(1) assms(2) assms(3) col_permutation_4 cong_symmetry not_bet_and_out 
      or_bet_out triangle_strict_inequality by blast
next
  assume P1: "\<not> Col A B C"
  show ?thesis
  proof cases
    assume "B = D"
    thus ?thesis
      using P1 lt1123 not_col_distincts by auto
  next
    assume P2: "B \<noteq> D"
    hence P3: "\<not> Col B C D"
      using P1 assms(1) col_trivial_2 colx not_col_permutation_5 out_col by blast
    have P4: "\<not> Col A C D"
      using P1 assms(1) col2__eq col_permutation_4 out_col out_distinct by blast
    have P5: "C \<noteq> D"
      using assms(1) assms(3) by auto
    hence P6: "A C D CongA A D C"
      by (metis P1 assms(2) col_trivial_3 l11_44_1_a)
    {
      assume T1: "Bet A B D"
      hence T2: "Bet D B A"
        using Bet_perm by blast
      have "B C D LtA B D C"
      proof -
        have T3: "D C B CongA B C D"
          by (metis P3 conga_pseudo_refl not_col_distincts)
        have T3A: "D Out B A"
          by (simp add: P2 T1 bet_out_1)
        have T3B: "C Out D D"
          using P5 out_trivial by auto
        have T3C: "C Out A A"
          using P1 not_col_distincts out_trivial by blast
        have "D Out C C"
          by (simp add: P5 out_trivial)
        hence T4: "D C A CongA B D C" using T3A T3B T3C
          by (meson conga_comm conga_right_comm l11_10 P6)
        have "D C B LtA D C A"
        proof -
          have T4A: "D C B LeA D C A"
          proof -
            have T4AA: "B InAngle D C A"
              using InAngle_def P1 P5 T2 not_col_distincts out_trivial by auto
            have "D C B CongA D C B"
              using T3 conga_right_comm by blast
            thus ?thesis
              by (simp add: T4AA inangle__lea)
          qed
          {
            assume T5: "D C B CongA D C A"
            have "D C OS B A"
              using Col_perm P3 T3A out_one_side by blast
            hence "C Out B A"
              using T5 conga_os__out by blast
            hence "False"
              using Col_cases P1 out_col by blast
          }
          hence "\<not> D C B CongA D C A" by auto
          thus ?thesis
            using LtA_def T4A by blast
        qed
        thus ?thesis
          using T3 T4 conga_preserves_lta by auto
      qed
    }
    {
      assume Q1: "Bet B D A"
      obtain E where Q2: "Bet B C E \<and> Cong B C C E"
        using Cong_perm segment_construction by blast
      have "A D C LtA E C D"
      proof -
        have Q3: "D C OS A E"
        proof -
          have Q4: "D C TS A B"
            by (metis Col_perm P2 Q1 P4 bet__ts between_symmetry)
          hence "D C TS E B"
            by (metis Col_def Q1 Q2 TS_def bet__ts cong_identity invert_two_sides l9_2)
          thus ?thesis
            using OS_def Q4 by blast
        qed
        have Q5: "A C D LtA E C D"
        proof -
          have "D C A LeA D C E"
          proof -
            have "B Out D A"
              using P2 Q1 bet_out by auto
            hence "B C OS D A"
              by (simp add: P3 out_one_side)
            hence "C B OS D A"
              using invert_one_side by blast
            hence "C E OS D A"
              by (metis Col_def Q2 Q3 col124__nos col_one_side diff_col_ex 
                  not_col_permutation_5)
            hence Q5A: "A InAngle D C E"
              by (simp add: \<open>C E OS D A\<close> Q3 invert_one_side one_side_symmetry os2__inangle)
            have "D C A CongA D C A"
              using CongA_def P6 conga_refl by auto
            thus ?thesis
              by (simp add: Q5A inangle__lea)
          qed
          hence Q6: "A C D LeA E C D"
            using lea_comm by blast
          {
            assume "A C D CongA E C D"
            hence "D C A CongA D C E"
              by (simp add: conga_comm)
            hence "C Out A E"
              using Q3 conga_os__out by auto
            hence "False"
              by (meson Col_def Out_cases P1 Q2 not_col_permutation_3 
                  one_side_not_col123 out_one_side)
          }
          hence "\<not> A C D CongA E C D" by auto
          thus ?thesis
            by (simp add: LtA_def Q6)
        qed
        have "E C D CongA E C D"
          by (metis P1 P5 Q2 cong_diff conga_refl not_col_distincts)
        thus ?thesis
          using Q5 P6 conga_preserves_lta by auto
      qed
      hence "B C D LtA B D C"
        using Bet_cases P1 P2 Q1 Q2 bet2_lta__lta not_col_distincts by blast
    }
    hence "B C D LtA B D C"
      by (meson Out_def \<open>Bet A B D \<Longrightarrow> B C D LtA B D C\<close> assms(1) between_symmetry)
    thus ?thesis
      by (simp add: l11_44_2_b)
  qed
qed

lemma triangle_reverse_inequality:
  assumes "A Out B D" and
    "Cong A C A D"
  shows "B D Le B C"
proof cases
  assume "A Out B C"
  thus ?thesis
    by (metis assms(1) assms(2) bet__le1213 cong_pseudo_reflexivity l6_11_uniqueness 
        l6_6 not_bet_distincts not_cong_4312)
next
  assume "\<not> A Out B C"
  thus ?thesis
    using triangle_strict_reverse_inequality assms(1) assms(2) lt__le by auto
qed

lemma os3__lta:
  assumes "A B OS C D" and
    "B C OS A D" and
    "A C OS B D"
  shows "B A C LtA B D C"
proof -
  have P1: "D InAngle A B C"
    by (simp add: assms(1) assms(2) invert_one_side os2__inangle)
  then obtain E where P2: "Bet A E C \<and> (E = B \<or> B Out E D)"
    using InAngle_def by auto
  have P3: "\<not> Col A B C"
    using assms(1) one_side_not_col123 by auto
  have P4: "\<not> Col A C D"
    using assms(3) col124__nos by auto
  have P5: "\<not> Col B C D"
    using assms(2) one_side_not_col124 by auto
  have P6: "\<not> Col A B D"
    using assms(1) one_side_not_col124 by auto
  {
    assume "E = B"
    hence "B A C LtA B D C"
      using P2 P3 bet_col by blast
  }
  {
    assume P7: "B Out E D"
    have P8: "A \<noteq> E"
      using P6 P7 not_col_permutation_4 out_col by blast
    have P9: "C \<noteq> E"
      using P5 P7 out_col by blast
    have P10: "B A C LtA B E C"
    proof -
      have P10A: "\<not> Col E A B"
        by (metis Col_def P2 P3 P8 col_transitivity_1)
      hence P10B: "E B A LtA B E C"
        using P2 P9 l11_41_aux Tarski_neutral_dimensionless_axioms by fastforce
      have P10C: "E A B LtA B E C"
        using P2 P9 P10A l11_41 by auto
      have P11: "E A B CongA B A C"
      proof -
        have P11A: "A Out B B"
          using assms(2) os_distincts out_trivial by auto
        have "A Out C E"
          using P2 P8 bet_out l6_6 by auto
        thus ?thesis
          using P11A conga_right_comm out2__conga by blast
      qed
      have P12: "B E C CongA B E C"
        by (metis Col_def P2 P3 P9 conga_refl)
      thus ?thesis
        using P11 P10C conga_preserves_lta by auto
    qed
    have "B E C LtA B D C"
    proof -
      have U1: "E Out D B"
        by (metis Bet_cases P2 P8 assms(3) bet_out_1 col123__nos not_bet_out one_side_chara
            one_side_reflexivity out_col)
      have U2: "D \<noteq> E"
        using P2 P4 bet_col not_col_permutation_5 by blast
      have U3: "\<not> Col D E C"
        by (metis Col_def P2 P4 P9 col_transitivity_1)
      have U4: "Bet E D B"
        by (simp add: P7 U1 out2__bet)
      have "D C E LtA C D B"
        using P5 U3 U4 l11_41_aux not_col_distincts by blast
      have U5: "D E C LtA C D B"
        using P7 U4 U3 l11_41 out_diff2 by auto
      have "D E C CongA B E C"
        by (simp add: P9 U1 l6_6 out2__conga out_trivial)
      thus ?thesis
        by (metis U5 conga_preserves_lta conga_pseudo_refl lta_distincts)
    qed
    hence "B A C LtA B D C"
      using P10 lta_trans by blast
  }
  thus ?thesis
    using P2 \<open>E = B \<Longrightarrow> B A C LtA B D C\<close> by blast
qed

(** GEOCOQ **)
(** This is Theorem 18.17 from Martin *)

lemma bet_le__lt:
  assumes "Bet A D B" and
    "A \<noteq> D" and
    "D \<noteq> B" and
    "A C Le B C"
  shows "D C Lt B C"
proof -
  have P1: "A \<noteq> B"
    using assms(1) assms(2) between_identity by blast
  have "C D Lt C B"
  proof cases
    assume P3: "Col A B C"
    thus ?thesis
    proof cases
      assume "Bet C D B"
      thus ?thesis
        by (simp add: assms(3) bet__lt1213)
    next
      assume "\<not> Bet C D B"
      hence "D Out C B"
        by (metis Out_def P1 P3 assms(1) col_transitivity_2 not_col_permutation_3 
            not_out_bet out_col)
      have "\<not> Bet A B C" 
        by (meson P1 bet_le_eq assms(4))
      have "B \<noteq> C" 
        using \<open>\<not> Bet A B C\<close> assms(1) assms(2) l5_1 by blast
      have "Bet B C A" 
        using Col_def P3 \<open>\<not> Bet A B C\<close> \<open>\<not> Bet C D B\<close> assms(1) 
          between_inner_transitivity outer_transitivity_between2 by blast
      obtain C' where "C Midpoint A C'" 
        using symmetric_point_construction by blast
      have "Bet C C' B" 
      proof -
        have "C Out C' B"        
        proof -
          have "C' \<noteq> C" 
            by (metis \<open>C Midpoint A C'\<close> \<open>\<not> Bet C D B\<close> assms(1) midpoint_not_midpoint)
          moreover have "A \<noteq> C" 
            using \<open>\<not> Bet C D B\<close> assms(1) by auto
          moreover have "Bet C' C A" 
            by (metis midpoint_bet \<open>C Midpoint A C'\<close> between_symmetry)
          ultimately show ?thesis
            using \<open>B \<noteq> C\<close> \<open>Bet B C A\<close> l6_2 by blast
        qed
        moreover have "C C' Le C B" 
          by (metis cong_reflexivity l5_6 le_right_comm midpoint_cong 
              \<open>C Midpoint A C'\<close> assms(4))
        ultimately show ?thesis 
          using l6_13_1 by blast
      qed
      thus ?thesis 
        by (metis bet__lt1213 l6_2 l6_6 le3456_lt__lt le_comm not_bet_distincts  
            \<open>D Out C B\<close> \<open>\<not> Bet C D B\<close> assms(1) assms(2) assms(4) between_symmetry)
    qed
  next
    assume Q0A: "\<not> Col A B C"
    hence Q0B: "\<not> Col B C D"
      by (meson Col_def assms(1) assms(3) col_transitivity_2)
    have "C B D LtA C D B"
    proof -
      have Q1: "B Out C C"
        using Q0A not_col_distincts out_trivial by force
      have Q2: "B Out A D"
        using Out_cases assms(1) assms(3) bet_out_1 by blast
      have Q3: "A Out C C"
        by (metis Q0A col_trivial_3 out_trivial)
      have Q4: "A Out B B"
        using P1 out_trivial by auto
      have "C B A LeA C A B"
        using Col_perm Le_cases Q0A assms(4) l11_44_2bis by blast
      hence T9: "C B D LeA C A B"
        using Q1 Q2 Q3 Q4 lea_out4__lea by blast
      have "C A B LtA C D B"
      proof -
        have U2: "\<not> Col D A C"
          using Q0B assms(1) assms(2) bet_col col_transitivity_2 not_col_permutation_3 
            not_col_permutation_4 by blast
        have U3: "D \<noteq> C"
          using Q0B col_trivial_2 by blast
        have U4: "D C A LtA C D B"
          using U2 assms(1) assms(3) l11_41_aux by auto
        have U5: "D A C LtA C D B"
          by (simp add: U2 assms(1) assms(3) l11_41)
        have "A Out B D"
          using Out_def P1 assms(1) assms(2) by auto
        hence "D A C CongA C A B"
          using Q3 conga_right_comm out2__conga by blast
        thus ?thesis
          by (metis U5 U3 assms(3) conga_preserves_lta conga_refl)
      qed
      thus ?thesis
        using T9 lea123456_lta__lta by blast
    qed
    thus ?thesis
      by (simp add: l11_44_2_b)
  qed
  thus ?thesis
    using Lt_cases by auto
qed

lemma cong2__ncol:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "A \<noteq> C" and
    "Cong A P B P" and
    "Cong A P C P"
  shows "\<not> Col A B C"
proof -
  have "Cong B P C P"
    using assms(4) assms(5) cong_inner_transitivity by blast
  thus ?thesis using bet_le__lt
    by (metis assms(1) assms(2) assms(3) assms(4) assms(5) cong__le cong__nlt 
        lt__nle not_col_permutation_5 third_point)
qed

lemma cong4_cop2__eq:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "A \<noteq> C" and
    "Cong A P B P" and
    "Cong A P C P" and
    "Coplanar A B C P" and
    "Cong A Q B Q" and
    "Cong A Q C Q" and
    "Coplanar A B C Q"
  shows "P = Q"
proof -
  have P1: "\<not> Col A B C"
    using assms(1) assms(2) assms(3) assms(4) assms(5) cong2__ncol by auto
  {
    assume P2: "P \<noteq> Q"
    have P3: "\<forall> R. Col P Q R \<longrightarrow> (Cong A R B R \<and> Cong A R C R)"
      using P2 assms(4) assms(5) assms(7) assms(8) l4_17 not_cong_4321 by blast
    obtain D where P4: "D Midpoint A B"
      using midpoint_existence by auto
    have P5: "Coplanar A B C D"
      using P4 coplanar_perm_9 midpoint__coplanar by blast
    have P6: "Col P Q D"
    proof -
      have P6A: "Coplanar P Q D A"
        using P1 P5 assms(6) assms(9) coplanar_pseudo_trans ncop_distincts by blast
      hence P6B: "Coplanar P Q D B"
        by (metis P4 col_cop__cop midpoint_col midpoint_distinct_1)
      have P6D: "Cong P A P B"
        using assms(4) not_cong_2143 by blast
      have P6E: "Cong Q A Q B"
        using assms(7) not_cong_2143 by blast
      have "Cong D A D B"
        using Midpoint_def P4 not_cong_2134 by blast
      thus ?thesis using cong3_cop2__col P6A P6B assms(1) P6D P6E by blast
    qed
    obtain R1 where P7: "P \<noteq> R1 \<and> Q \<noteq> R1 \<and> D \<noteq> R1 \<and> Col P Q R1"
      using P6 diff_col_ex3 by blast
    obtain R2 where P8: "Bet R1 D R2 \<and> Cong D R2 R1 D"
      using segment_construction by blast
    have P9: "Col P Q R2"
      by (metis P6 P7 P8 bet_col colx)
    have P9A: "Cong R1 A R1 B"
      using P3 P7 not_cong_2143 by blast
    hence "Per R1 D A"
      using P4 Per_def by auto
    hence "Per A D R1" using l8_2 by blast
    have P10: "Cong A R1 A R2"
    proof -
      have f1: "Bet R2 D R1 \<or>Bet R1 R2 D"
        by (metis Col_def P7 P8 between_equality not_col_permutation_5)
      have f2: "Cong B R1 A R1"
        using Cong_perm \<open>Cong R1 A R1 B\<close> by blast
      have "Cong B R1 A R2 \<or> Bet R1 R2 D"
        using f1 Cong_perm Midpoint_def P4 P8 l7_13 by blast
      thus ?thesis
        using f2 P8 bet_cong_eq cong_inner_transitivity by blast
    qed
    have "Col A B C"
    proof -
      have P11: "Cong B R1 B R2"
        by (metis Cong_perm P10 P3 P9 P9A cong_inner_transitivity)
      have P12: "Cong C R1 C R2"
        using P10 P3 P7 P9 cong_inner_transitivity by blast
      have P12A: "Coplanar A B C R1"
        using P2 P7 assms(6) assms(9) col_cop2__cop by blast
      have P12B: "Coplanar A B C R2"
        using P2 P9 assms(6) assms(9) col_cop2__cop by blast
      have "R1 \<noteq> R2"
        using P7 P8 between_identity by blast
      thus ?thesis
        using P10 P11 P12A P12B P12 cong3_cop2__col by blast
    qed
    hence False
      by (simp add: P1)
  }
  thus ?thesis by auto
qed

lemma t18_18_aux:
  assumes "Cong A B D E" and
    "Cong A C D F" and
    "F D E LtA C A B" and
    "\<not> Col A B C" and
    "\<not> Col D E F" and
    "D F Le D E"
  shows "E F Lt B C"
proof -
  obtain G0 where P1: "C A B CongA F D G0 \<and> F D OS G0 E"
    using angle_construction_1 assms(4) assms(5) not_col_permutation_2 by blast
  hence P2: "\<not> Col F D G0"
    using col123__nos by auto
  then obtain G where P3: "D Out G0 G \<and> Cong D G A B"
    by (metis assms(4) bet_col between_trivial2 col_trivial_2 segment_construction_3)
  have P4: "C A B CongA F D G"
  proof -
    have P4B: "A Out C C"
      by (metis assms(4) col_trivial_3 out_trivial)
    have P4C: "A Out B B"
      by (metis assms(4) col_trivial_1 out_trivial)
    have P4D: "D Out F F"
      using P2 not_col_distincts out_trivial by blast
    have "D Out G G0"
      by (simp add: P3 l6_6)
    thus ?thesis using P1 P4B P4C P4D
      using l11_10 by blast
  qed
  have "D Out G G0"
    by (simp add: P3 l6_6)
  hence "D F OS G G0"
    using P2 not_col_permutation_4 out_one_side by blast
  hence "F D OS G G0"
    by (simp add: invert_one_side)
  hence P5: "F D OS G E"
    using P1 one_side_transitivity by blast
  have P6: "\<not> Col F D G"
    by (meson P5 one_side_not_col123)
  have P7: "Cong C B F G"
    using P3 P4 assms(2) cong2_conga_cong cong_commutativity cong_symmetry by blast
  have P8: "F E Lt F G"
  proof -
    have P9: "F D E LtA F D G"
      by (metis P4 assms(3) assms(5) col_trivial_1 col_trivial_3 
          conga_preserves_lta conga_refl)
    have P10: "Cong D G D E"
      using P3 assms(1) cong_transitivity by blast
    {
      assume P11: "Col E F G"
      have P12: "F D E LeA F D G"
        by (simp add: P9 lta__lea)
      have P13: "\<not> F D E CongA F D G"
        using P9 not_lta_and_conga by blast
      have "F D E CongA F D G"
      proof -
        have "F Out E G"
          using Col_cases P11 P5 col_one_side_out l6_6 by blast
        hence "E F D CongA G F D"
          by (metis assms(5) conga_os__out conga_refl l6_6 not_col_distincts 
              one_side_reflexivity out2__conga)
        thus ?thesis
          by (meson P10 assms(2) assms(6) cong_4321 cong_inner_transitivity l11_52 le_comm)
      qed
      hence "False"
        using P13 by blast
    }
    hence P15: "\<not> Col E F G" by auto
    {
      assume P20: "Col D E G"
      have P21: "F D E LeA F D G"
        by (simp add: P9 lta__lea)
      have P22: "\<not> F D E CongA F D G"
        using P9 not_lta_and_conga by blast
      have "F D E CongA F D G"
      proof -
        have "D Out E G"
          by (meson Out_cases P5 P20 col_one_side_out invert_one_side not_col_permutation_5)
        thus ?thesis
          using P10 P15 \<open>D Out G G0\<close> cong_inner_transitivity l6_11_uniqueness 
            l6_7 not_col_distincts by blast
      qed
      hence "False"
        by (simp add: P22)
    }
    hence P16: "\<not> Col D E G" by auto
    have P17: "E InAngle F D G"
      using lea_in_angle by (simp add: P5 P9 lta__lea)
    then obtain H where P18: "Bet F H G \<and> (H = D \<or> D Out H E)"
      using InAngle_def by auto
    {
      assume "H = D"
      hence "F G E LtA F E G"
        using P18 P6 bet_col by blast
    }
    {
      assume P19: "D Out H E"
      have P20: "H \<noteq> F"
        using Out_cases P19 assms(5) out_col by blast
      have P21: "H \<noteq> G"
        using P19 P16 l6_6 out_col by blast
      have "F D Le G D"
        using P10 assms(6) cong_pseudo_reflexivity l5_6 not_cong_4312 by blast
      hence "H D Lt G D"
        using P18 P20 P21 bet_le__lt by blast
      hence P22: "D H Lt D E"
        using Lt_cases P10 cong2_lt__lt cong_reflexivity by blast
      hence P23: "D H Le D E \<and> \<not> Cong D H D E"
        using Lt_def by blast
      have P24: "H \<noteq> E"
        using P23 cong_reflexivity by blast
      have P25: "Bet D H E"
        by (simp add: P19 P23 l6_13_1)
      have P26: "E G OS F D"
        by (metis InAngle_def P15 P16 P18 P25 bet_out_1 between_symmetry 
            in_angle_one_side not_col_distincts not_col_permutation_1)
      have "F G E LtA F E G"
      proof -
        have P27: "F G E LtA D E G"
        proof -
          have P28: "D G E CongA D E G"
            by (metis P10 P16 l11_44_1_a not_col_distincts)
          have "F G E LtA D G E"
          proof -
            have P29: "F G E LeA D G E"
              by (metis OS_def P17 P26 P5 TS_def in_angle_one_side inangle__lea_1 
                  invert_one_side l11_24 os2__inangle)
            {
              assume "F G E CongA D G E"
              hence "E G F CongA E G D"
                by (simp add: conga_comm)
              hence "G Out F D"
                using P26 conga_os__out by auto
              hence "False"
                using P6 not_col_permutation_2 out_col by blast
            }
            hence "\<not> F G E CongA D G E" by auto
            thus ?thesis
              by (simp add: LtA_def P29)
          qed
          thus ?thesis
            by (metis P28 P6 col_trivial_3 conga_preserves_lta conga_refl)
        qed
        have "G E D LtA G E F"
        proof -
          have P30: "G E D LeA G E F"
          proof -
            have P31: "D InAngle G E F"
              by (simp add: P16 P17 P26 assms(5) in_angle_two_sides l11_24 
                  not_col_permutation_5 os_ts__inangle)
            have "G E D CongA G E D"
              by (metis P16 col_trivial_1 col_trivial_2 conga_refl)
            thus ?thesis
              using P31 inangle__lea by auto
          qed
          have "\<not> G E D CongA G E F"
            by (metis OS_def P26 P5 TS_def conga_os__out invert_one_side out_col)
          thus ?thesis
            by (simp add: LtA_def P30)
        qed
        hence "D E G LtA F E G"
          using lta_comm by blast
        thus ?thesis
          using P27 lta_trans by blast
      qed
    }
    hence "F G E LtA F E G"
      using P18 \<open>H = D \<Longrightarrow> F G E LtA F E G\<close> by blast
    thus ?thesis
      by (simp add: l11_44_2_b)
  qed
  hence "E F Lt F G"
    using lt_left_comm by blast
  thus ?thesis
    using P7 cong2_lt__lt cong_pseudo_reflexivity not_cong_4312 by blast
qed

lemma t18_18:
  assumes "Cong A B D E" and
    "Cong A C D F" and
    "F D E LtA C A B"
  shows "E F Lt B C"
proof -
  have P1: "F \<noteq> D"
    using assms(3) lta_distincts by blast
  have P2: "E \<noteq> D"
    using assms(3) lta_distincts by blast
  have P3: "C \<noteq> A"
    using assms(3) lta_distincts by auto
  have P4: "B \<noteq> A"
    using assms(3) lta_distincts by blast
  {
    assume P6: "Col A B C"
    {
      assume P7: "Bet B A C"
      obtain C' where P8:"Bet E D C' \<and> Cong D C' A C"
        using segment_construction by blast
      have P9: "Cong E F E F"
        by (simp add: cong_reflexivity)
      have P10: "Cong E C' B C"
        using P7 P8 assms(1) l2_11_b not_cong_4321 by blast
      have "E F Lt E C'"
      proof -
        have P11: "Cong D F D C'"
          using P8 assms(2) cong_transitivity not_cong_3412 by blast
        have "\<not> Bet E D F"
          using Bet_perm Col_def assms(3) col_lta__out not_bet_and_out by blast
        thus ?thesis
          using P11 P8 triangle_strict_inequality by blast
      qed
      hence "E F Lt B C"
        using P9 P10 cong2_lt__lt by blast
    }
    {
      assume "\<not> Bet B A C"
      hence "E F Lt B C"
        using P6 assms(3) between_symmetry col_lta__bet col_permutation_2 by blast
    }
    hence "E F Lt B C"
      using \<open>Bet B A C \<Longrightarrow> E F Lt B C\<close> by auto
  }
  {
    assume P12: "\<not> Col A B C"
    {
      assume P13: "Col D E F"
      {
        assume P14: "Bet F D E"
        hence "C A B LeA F D E"
          by (simp add: P1 P2 P3 P4 l11_31_2)
        hence "F D E LtA F D E"
          using assms(3) lea__nlta by auto
        hence "False"
          by (simp add: nlta)
        hence "E F Lt B C" by auto
      }
      {
        assume "\<not> Bet F D E"
        hence P16: "D Out F E"
          using P13 not_col_permutation_1 not_out_bet by blast
        obtain F' where P17: "A Out B F' \<and> Cong A F' A C"
          using P3 P4 segment_construction_3 by fastforce
        hence P18: "B F' Lt B C"
          by (meson P12 triangle_strict_reverse_inequality not_cong_3412 out_col)
        have "Cong B F' E F"
          by (meson Out_cases P16 P17 assms(1) assms(2) cong_transitivity out_cong_cong)
        hence "E F Lt B C"
          using P18 cong2_lt__lt cong_reflexivity by blast
      }
      hence "E F Lt B C"
        using \<open>Bet F D E \<Longrightarrow> E F Lt B C\<close> by blast
    }
    {
      assume P20: "\<not> Col D E F"
      {
        assume "D F Le D E"
        hence "E F Lt B C"
          by (meson P12 t18_18_aux P20 assms(1) assms(2) assms(3))
      }
      {
        assume "D E Le D F"
        hence "E F Lt B C"
          by (meson P12 P20 lta_comm t18_18_aux assms(1) assms(2) assms(3) lt_comm 
              not_col_permutation_5)
      }
      hence "E F Lt B C"
        using \<open>D F Le D E \<Longrightarrow> E F Lt B C\<close> local.le_cases by blast
    }
    hence "E F Lt B C"
      using \<open>Col D E F \<Longrightarrow> E F Lt B C\<close> by blast
  }
  thus ?thesis
    using \<open>Col A B C \<Longrightarrow> E F Lt B C\<close> by auto
qed

lemma t18_19:
  assumes "A \<noteq> B" and
    "A \<noteq> C" and
    "Cong A B D E" and
    "Cong A C D F" and
    "E F Lt B C"
  shows "F D E LtA C A B"
proof -
  {
    assume P1: "C A B LeA F D E"
    {
      assume "C A B CongA F D E"
      hence "False"
        using Cong_perm assms(3) assms(4) assms(5) cong__nlt l11_49 by blast
    }
    {
      assume P2: "\<not> C A B CongA F D E"
      hence "C A B LtA F E D"
        by (metis P1 assms(3) assms(4) assms(5) cong_symmetry lea_distincts 
            lta__nlea not_and_lt or_lta2_conga t18_18)
      hence "B C Lt E F"
        by (metis P1 P2 assms(3) assms(4) cong_symmetry lta__nlea lta_distincts 
            or_lta2_conga t18_18)
      hence "False"
        using assms(5) not_and_lt by auto
    }
    hence "False"
      using \<open>C A B CongA F D E \<Longrightarrow> False\<close> by auto
  }
  hence "\<not> C A B LeA F D E" by auto
  thus ?thesis
    using assms(1) assms(2) assms(3) assms(4) cong_identity nlea__lta by blast
qed

lemma acute_trivial:
  assumes "A \<noteq> B"
  shows "Acute A B A"
  by (metis acute_distincts angle_partition assms l11_43)

lemma acute_not_per:
  assumes "Acute A B C"
  shows "\<not> Per A B C"
proof -
  obtain A' B' C' where "Per A' B' C' \<and> A B C LtA A' B' C'"
    using Acute_def assms by auto
  thus ?thesis
    using acute_distincts acute_per__lta assms nlta by fastforce
qed

lemma angle_bisector:
  assumes "A \<noteq> B" and
    "C \<noteq> B"
  shows "\<exists> P. (P InAngle A B C \<and> P B A CongA P B C)"
proof cases
  assume P1: "Col A B C"
  thus ?thesis
  proof cases
    assume P2: "Bet A B C"
    then obtain Q where P3: "\<not> Col A B Q"
      using assms(1) not_col_exists by auto
    then obtain P where P4: "A B Perp P B \<and> A B OS Q P"
      using P1 l10_15 os_distincts by blast
    hence P5: "P InAngle A B C"
      by (metis P2 assms(2) in_angle_line os_distincts)
    have "P B A CongA P B C"
    proof -
      have P9: "P \<noteq> B"
        using P4 os_distincts by blast
      have "Per P B A"
        by (simp add: P4 Perp_perm perp_per_2)
      thus ?thesis
        using P2 assms(1) assms(2) P9 l11_18_1 by auto
    qed
    thus ?thesis
      using P5 by auto
  next
    assume T1: "\<not> Bet A B C"
    hence T2: "B Out A C"
      by (simp add: P1 l6_4_2)
    have T3: "C InAngle A B C"
      by (simp add: assms(1) assms(2) inangle3123)
    have "C B A CongA C B C"
      using T2 between_trivial2 l6_6 out2__conga out2_bet_out by blast
    thus ?thesis
      using T3 by auto
  qed
next
  assume T4: "\<not> Col A B C"
  obtain C0 where T5: "B Out C0 C \<and> Cong B C0 B A"
    using assms(1) assms(2) l6_11_existence by fastforce
  obtain P where T6: "P Midpoint A C0"
    using midpoint_existence by auto
  have T6A: "\<not> Col A B C0"
    by (metis T4 T5 col3 l6_3_1 not_col_distincts out_col)
  have T6B: "P \<noteq> B"
    using Col_def Midpoint_def T6 T6A by auto
  have T6D: "P \<noteq> A"
    using T6 T6A is_midpoint_id not_col_distincts by blast
  have "P InAngle A B C0"
    using InAngle_def T5 T6 T6B assms(1) l6_3_1 midpoint_bet out_trivial by fastforce
  hence T7: "P InAngle A B C"
    using T5 T6B in_angle_trans2 l11_24 out341__inangle by blast
  have T8: "(P = B) \<or> B Out P P"
    using out_trivial by auto
  have T9: "B Out A A"
    by (simp add: assms(1) out_trivial)
  {
    assume T9A: "B Out P P"
    have "P B A CongA P B C0 \<and> B P A CongA B P C0 \<and> P A B CongA P C0 B"
    proof -
      have T9B: "Cong B P B P"
        by (simp add: cong_reflexivity)
      have T9C: "Cong B A B C0"
        using Cong_perm T5 by blast
      have "Cong P A P C0"
        using Midpoint_def T6 not_cong_2134 by blast
      thus ?thesis using l11_51 T6B assms(1) T9B T9C T6D by presburger
    qed
    hence "P B A CongA P B C0" by auto
    hence "P B A CongA P B C" using l11_10 T9A T9
      by (meson T5 l6_6)
    hence  "\<exists> P. (P InAngle A B C \<and> P B A CongA P B C)"
      using T7 by auto
  }
  thus ?thesis
    using T6B T8 by blast
qed

lemma reflectl__conga:
  assumes "A \<noteq> B" and
    "B \<noteq> P" and
    "P P' ReflectL A B"
  shows "A B P CongA A B P'"
proof -
  obtain A' where P1: "A' Midpoint P' P \<and> Col A B A' \<and> (A B Perp P' P \<or> P = P')"
    using ReflectL_def assms(3) by auto
  {
    assume P2: "A B Perp P' P"
    hence P3: "P \<noteq> P'"
      using perp_not_eq_2 by blast
    hence P4: "A' \<noteq> P'"
      using P1 is_midpoint_id by blast
    have P5: "A' \<noteq> P"
      using P1 P3 is_midpoint_id_2 by auto
    have  "A B P CongA A B P'"
    proof cases
      assume P6: "A' = B"
      hence P8: "B \<noteq> P'"
        using P4 by auto
      have P9: "Per A B P"
        using P1 P3 P6 Perp_cases col_transitivity_2 midpoint_col midpoint_distinct_1
          not_col_permutation_2 perp_col2_bis perp_per_2 by metis
      have "Per A B P'" 
        by (metis P1 P6 P9 midpoint_col assms(2) not_col_permutation_5 per_col)
      thus ?thesis
        using l11_16 P4 P5 P6 P9 assms(1) by auto
    next
      assume T1: "A' \<noteq> B"
      have T2: "B A' P CongA B A' P'"
      proof -
        have T2A: "Cong B P B P'"
          using assms(3) col_trivial_2 is_image_spec_col_cong l10_4_spec not_cong_4321 
          by blast
        hence T2B: "A' B P CongA A' B P'"
          by (metis Cong_perm Midpoint_def P1 P5 T1 l11_51 assms(2) cong_reflexivity)
        have "A' P B CongA A' P' B"
          by (simp add: P5 T2A T2B cong_reflexivity conga_comm l11_49)
        thus ?thesis
          using P5 T2A T2B cong_reflexivity l11_49 by blast
      qed
      have T3: "Cong A' B A' B"
        by (simp add: cong_reflexivity)
      have "Cong A' P A' P'"
        using Midpoint_def P1 not_cong_4312 by blast
      hence T4: "A' B P CongA A' B P' \<and> A' P B CongA A' P' B" using l11_49
        using assms(2) T2 T3 by blast
      show ?thesis
      proof cases
        assume "Bet A' B A"
        thus ?thesis
          using T4 assms(1) l11_13 by blast
      next
        assume "\<not> Bet A' B A"
        hence T5: "B Out A' A"
          using P1 not_col_permutation_3 or_bet_out by blast
        have T6: "B \<noteq> P'"
          using T4 conga_distinct by blast
        have T8: "B Out A A'"
          by (simp add: T5 l6_6)
        have T9: "B Out P P"
          using assms(2) out_trivial by auto
        have "B Out P' P'"
          using T6 out_trivial by auto
        thus ?thesis
          using l11_10 T4 T8 T9 by blast
      qed
    qed
  }
  {
    assume "P = P'"
    hence "A B P CongA A B P'"
      using assms(1) assms(2) conga_refl by auto
  }
  thus ?thesis
    using P1 \<open>A B Perp P' P \<Longrightarrow> A B P CongA A B P'\<close> by blast
qed

lemma conga_cop_out_reflectl__out:
  assumes "\<not> B Out A C" and
    "Coplanar A B C P" and
    "P B A CongA P B C" and
    "B Out A T" and
    "T T' ReflectL B P"
  shows "B Out C T'"
proof -
  have P1: "P B T CongA P B T'"
    by (metis assms(3) assms(4) assms(5) conga_distinct is_image_spec_rev 
        out_distinct reflectl__conga)
  have P2: "T T' Reflect B P"
    by (metis P1 assms(5) conga_distinct is_image_is_image_spec)
  have P3: "B \<noteq> T'"
    using CongA_def P1 by blast
  have P4: "P B C CongA P B T'"
  proof -
    have P5: "P B C CongA P B A"
      by (simp add: assms(3) conga_sym)
    have "P B A CongA P B T'"
    proof -
      have P7: "B Out P P"
        using assms(3) conga_diff45 out_trivial by blast
      have P8: "B Out A T"
        by (simp add: assms(4))
      have "B Out T' T'"
        using P3 out_trivial by auto
      thus ?thesis
        using P1 P7 P8 l11_10 by blast
    qed
    thus ?thesis
      using P5 not_conga by blast
  qed
  have "P B OS C T'"
  proof -
    have P9: "P B TS A C"
      using assms(1) assms(2) assms(3) conga_cop__or_out_ts coplanar_perm_20 by blast
    hence "T \<noteq> T'"
      by (metis Col_perm P2 P3 TS_def assms(4) col_transitivity_2 l10_8 out_col)
    hence "P B TS T T'"
      by (metis P2 P4 conga_diff45 invert_two_sides l10_14)
    hence "P B TS A T'"
      using assms(4) col_trivial_2 out_two_sides_two_sides by blast
    thus ?thesis
      using OS_def P9 l9_2 by blast
  qed
  thus ?thesis
    using P4 conga_os__out by auto
qed

lemma col_conga_cop_reflectl__col:
  assumes "\<not> B Out A C" and
    "Coplanar A B C P" and
    "P B A CongA P B C" and
    "Col B A T" and
    "T T' ReflectL B P"
  shows "Col B C T'"
proof cases
  assume "B = T"
  thus ?thesis
    using assms(5) col_image_spec__eq not_col_distincts by blast
next
  assume P1: "B \<noteq> T"
  thus ?thesis
  proof cases
    assume "B Out A T"
    thus ?thesis
      using out_col conga_cop_out_reflectl__out assms(1) assms(2) assms(3) assms(5) by blast
  next
    assume P2: "\<not> B Out A T"
    obtain A' where P3: "Bet A B A' \<and> Cong B A' A B"
      using segment_construction by blast
    obtain C' where P4: "Bet C B C' \<and> Cong B C' C B"
      using segment_construction by blast
    have P5: "B Out C' T'"
    proof -
      have P6: "\<not> B Out A' C'"
        by (metis P3 P4 assms(1) between_symmetry cong_diff_2 l6_2 out_diff1 out_diff2)
      have P7: "Coplanar A' B C' P"
      proof cases
        assume "Col A B C"
        thus ?thesis
          using P3 P4 assms(1) assms(2) assms(3) bet_col bet_neq32__neq 
            col2_cop__cop col_transitivity_1 colx conga_diff2 conga_diff56 l6_4_2 
            ncoplanar_perm_15 not_col_permutation_5 by meson
      next
        assume P7B: "\<not> Col A B C"
        have P7C: "Coplanar A B C A'"
          using P3 bet_col ncop__ncols by blast
        have P7D: "Coplanar A B C B"
          using ncop_distincts by blast
        have "Coplanar A B C C'"
          using P4 bet__coplanar coplanar_perm_20 by blast
        thus ?thesis
          using P7B P7C P7D assms(2) coplanar_pseudo_trans by blast
      qed
      have P8: "P B A' CongA P B C'"
        by (metis CongA_def P3 P4 assms(3) cong_reverse_identity conga_left_comm 
            l11_13 not_conga_sym)
      have "Bet A B T" 
        using Col_cases P2 assms(4) not_out_bet by blast
      hence P9: "B Out A' T" 
        by (metis P1 P3 assms(3) between_symmetry cong_reverse_identity conga_distinct l6_3_2)
      thus ?thesis
        using P6 P7 P8 P9 assms(5) conga_cop_out_reflectl__out by blast
    qed
    thus ?thesis
      by (metis Col_def P4 col_transitivity_1 out_col out_diff1)
  qed
qed

lemma conga2_cop2__col:
  assumes "\<not> B Out A C" and
    "P B A CongA P B C" and
    "P' B A CongA P' B C" and
    "Coplanar A B P P'" and
    "Coplanar B C P P'"
  shows "Col B P P'"
proof -
  obtain C' where P1: "B Out C' C \<and> Cong B C' B A"
    by (metis assms(2) conga_distinct l6_11_existence)
  have P1A: "Cong P A P C' \<and> (P \<noteq> A \<longrightarrow> (B P A CongA B P C' \<and> B A P CongA B C' P))"
  proof -
    have P2: "P B A CongA P B C'"
    proof -
      have P2A: "B Out P P"
        using assms(2) conga_diff45 out_trivial by auto
      have "B Out A A"
        using assms(2) conga_distinct out_trivial by auto
      thus ?thesis
        using P1 P2A assms(2) l11_10 by blast
    qed
    have P3: "Cong B P B P"
      by (simp add: cong_reflexivity)
    have "Cong B A B C'"
      using Cong_perm P1 by blast
    thus ?thesis using l11_49 P2 cong_reflexivity by blast
  qed
  have P4: "P' B A CongA P' B C'"
  proof -
    have P4A: "B Out P' P'"
      using assms(3) conga_diff1 out_trivial by auto
    have "B Out A A"
      using assms(2) conga_distinct out_trivial by auto
    thus ?thesis
      using P1 P4A assms(3) l11_10 by blast
  qed
  have P5: "Cong B P' B P'"
    by (simp add: cong_reflexivity)
  have P5A: "Cong B A B C'"
    using Cong_perm P1 by blast
  hence P6: "P' \<noteq> A \<longrightarrow> (B P' A CongA B P' C' \<and> B A P' CongA B C' P')"
    using P4 P5 l11_49 by blast
  have P7: "Coplanar B P P' A"
    using assms(4) ncoplanar_perm_18 by blast
  have P8: "Coplanar B P P' C'"
  proof -
    have "Coplanar P P' B C" 
      using assms(5) ncoplanar_perm_16 by blast
    moreover have "B \<noteq> C" 
      using P1 out_diff2 by blast
    moreover have "Col B C C'" 
      using P1 not_col_permutation_5 out_col by blast
    ultimately show ?thesis 
      using col_cop__cop coplanar_perm_12 by blast
  qed
  have "A \<noteq> C'"
    using P1 assms(1) by auto
  thus ?thesis
    using P4 P5 P7 P8 P5A P1A cong3_cop2__col l11_49 by blast
qed

lemma conga2_cop2__col_1:
  assumes "\<not> Col A B C" and
    "P B A CongA P B C" and
    "P' B A CongA P' B C" and
    "Coplanar A B C P" and
    "Coplanar A B C P'"
  shows "Col B P P'"
proof -
  have P1: "\<not> B Out A C"
    using Col_cases assms(1) out_col by blast
  have P2: "Coplanar A B P P'"
    by (meson assms(1) assms(4) assms(5) coplanar_perm_12 coplanar_trans_1 
        not_col_permutation_2)
  have "Coplanar B C P P'"
    using assms(1) assms(4) assms(5) coplanar_trans_1 by auto
  thus ?thesis using P1 P2 conga2_cop2__col assms(2) assms(3) conga2_cop2__col by auto
qed

lemma col_conga__conga:
  assumes "P B A CongA P B C" and
    "Col B P P'" and
    "B \<noteq> P'"
  shows "P' B A CongA P' B C"
proof cases
  assume "Bet P B P'"
  thus ?thesis
    using assms(1) assms(3) l11_13 by blast
next
  assume "\<not> Bet P B P'"
  hence P1: "B Out P P'"
    using Col_cases assms(2) or_bet_out by blast
  hence P2: "B Out P' P"
    by (simp add: l6_6)
  have P3: "B Out A A"
    using CongA_def assms(1) out_trivial by auto
  have "B Out C C"
    using assms(1) conga_diff56 out_trivial by blast
  thus ?thesis
    using P2 P3 assms(1) l11_10 by blast
qed

lemma cop_inangle__ex_col_inangle:
  assumes "\<not> B Out A C" and
    "P InAngle A B C" and
    "Coplanar A B C Q"
  shows "\<exists> R. (R InAngle A B C \<and> P \<noteq> R \<and> Col P Q R)"
proof -
  have P1: "A \<noteq> B"
    using assms(2) inangle_distincts by blast
  hence P4: "A \<noteq> C"
    using assms(1) out_trivial by blast
  have P2: "C \<noteq> B"
    using assms(2) inangle_distincts by auto
  have P3: "P \<noteq> B"
    using InAngle_def assms(2) by auto
  thus ?thesis
  proof cases
    assume "P = Q"
    thus ?thesis
      using P1 P2 P4 col_trivial_1 inangle1123 inangle3123 by blast
  next
    assume P5: "P \<noteq> Q"
    thus ?thesis
    proof cases
      assume P6: "Col B P Q"
      obtain R where P7: "Bet B P R \<and> Cong P R B P"
        using segment_construction by blast
      have P8: "R InAngle A B C"
        using Out_cases P1 P2 P3 P7 assms(2) bet_out l11_25 out_trivial by blast
      have "P \<noteq> R"
        using P3 P7 cong_reverse_identity by blast
      thus ?thesis
        by (metis P3 P6 P7 P8 bet_col col_transitivity_2)
    next
      assume T1: "\<not> Col B P Q"
      thus ?thesis
      proof cases
        assume T2: "Col A B C"
        have T3: "Q InAngle A B C"
          by (metis P1 P2 T1 T2 assms(1) in_angle_line l6_4_2 not_col_distincts)
        thus ?thesis
          using P5 col_trivial_2 by blast
      next
        assume Q1: "\<not> Col A B C"
        thus ?thesis
        proof cases
          assume Q2: "Col B C P"
          have Q3: "\<not> Col B A P"
            using Col_perm P3 Q1 Q2 col_transitivity_2 by blast
          have Q4: "Coplanar B P Q A"
            using P2 Q2 assms(3) col2_cop__cop col_trivial_3 ncoplanar_perm_22 
              ncoplanar_perm_3 by blast
          have Q5: "Q \<noteq> P"
            using P5 by auto
          have Q6: "Col B P P"
            using not_col_distincts by blast
          have Q7: "Col Q P P"
            using not_col_distincts by auto
          have "\<not> Col B P A"
            using Col_cases Q3 by auto
          then obtain Q0 where P10: "Col Q P Q0 \<and> B P OS A Q0"
            using cop_not_par_same_side Q4 Q5 Q6 Q7 T1 by blast
          have P13: "P \<noteq> Q0"
            using P10 os_distincts by auto
          {
            assume "B A OS P Q0"
            hence ?thesis
              using P10 P13 assms(2) in_angle_trans not_col_permutation_4 os2__inangle by blast
          }
          {
            assume V1: "\<not> B A OS P Q0"
            have "\<exists> R. Bet P R Q0 \<and> Col P Q R \<and> Col B A R"
            proof cases
              assume V3: "Col B A Q0"
              have "Col P Q Q0"
                using Col_cases P10 by auto
              thus ?thesis
                using V3 between_trivial by auto
            next
              assume V4: "\<not> Col B A Q0"
              hence V5: "\<not> Col Q0 B A"
                using Col_perm by blast
              have "\<not> Col P B A"
                using Col_cases Q3 by blast
              then obtain R where V8: "Col R B A \<and> Bet P R Q0"
                using cop_nos__ts V1 V5
                by (meson P10 TS_def ncoplanar_perm_2 os__coplanar)
              thus ?thesis
                by (metis Col_def P10 P13 col_transitivity_2)
            qed
            then obtain R where V9: "Bet P R Q0 \<and> Col P Q R \<and> Col B A R" by auto
            have V10: "P \<noteq> R"
              using Q3 V9 by blast
            have "R InAngle A B C"
            proof -
              have W1: "\<not> Col B P Q0"
                using P10 P13 T1 col2__eq by blast
              have "P Out Q0 R"
                using V10 V9 bet_out l6_6 by auto
              hence "B P OS Q0 R"
                using Q6 W1 out_one_side_1 by blast
              hence "B P OS A R"
                using P10 one_side_transitivity by blast
              hence "B Out A R"
                using V9 col_one_side_out by auto
              thus ?thesis
                by (simp add: P2 out321__inangle)
            qed
            hence ?thesis
              using V10 V9 by blast
          }
          thus ?thesis
            using \<open>B A OS P Q0 \<Longrightarrow> \<exists>R. R InAngle A B C \<and> P \<noteq> R \<and> Col P Q R\<close> by blast
        next
          assume Z1: "\<not> Col B C P"
          hence Z6: "\<not> Col B P C"
            by (simp add: not_col_permutation_5)
          have Z3: "Col B P P"
            by (simp add: col_trivial_2)
          have Z4: "Col Q P P"
            by (simp add: col_trivial_2)
          have "Coplanar A B C P"
            using Q1 assms(2) inangle__coplanar ncoplanar_perm_18 by blast
          hence "Coplanar B P Q C"
            using Q1 assms(3) coplanar_trans_1 ncoplanar_perm_5 by blast
          then obtain Q0 where Z5: "Col Q P Q0 \<and> B P OS C Q0"
            using cop_not_par_same_side by (metis Z3 Z4 T1 Z6)
          thus ?thesis
          proof cases
            assume "B C OS P Q0"
            thus ?thesis
            proof -
              have "\<forall>p. p InAngle C B A \<or> \<not> p InAngle C B P"
                using assms(2) in_angle_trans l11_24 by blast
              thus ?thesis
                by (metis Col_perm Z5 \<open>B C OS P Q0\<close> l11_24 os2__inangle os_distincts)
            qed
          next
            assume Z6: "\<not> B C OS P Q0"
            have Z7: "\<exists> R. Bet P R Q0 \<and> Col P Q R \<and> Col B C R"
            proof cases
              assume "Col B C Q0"
              thus ?thesis
                using Col_def Col_perm Z5 between_trivial by blast
            next
              assume Z8: "\<not> Col B C Q0"
              have "\<exists> R. Col R B C \<and> Bet P R Q0"
              proof -
                have Z10: "Coplanar B C P Q0"
                  using Z5 ncoplanar_perm_2 os__coplanar by blast
                have Z11: "\<not> Col P B C"
                  using Col_cases Z1 by blast
                have "\<not> Col Q0 B C"
                  using Col_perm Z8 by blast
                thus ?thesis
                  using cop_nos__ts Z6 Z10 Z11 by (simp add: TS_def)
              qed
              then obtain R where "Col R B C \<and> Bet P R Q0" by blast
              thus ?thesis
                using Z5 bet_col col2__eq col_permutation_1 os_distincts by metis
            qed
            then obtain R where Z12: "Bet P R Q0 \<and> Col P Q R \<and> Col B C R" by blast
            have Z13: "P \<noteq> R"
              using Z1 Z12 by auto
            have Z14: "\<not> Col B P Q0"
              using Z5 one_side_not_col124 by blast
            have "P Out Q0 R"
              using Z12 Z13 bet_out l6_6 by auto
            hence "B P OS Q0 R"
              using Z14 Z3 out_one_side_1 by blast
            hence "B P OS C R"
              using Z5 one_side_transitivity by blast
            hence "B Out C R"
              using Z12 col_one_side_out by blast
            hence "R InAngle A B C"
              using P1 out341__inangle by auto
            thus ?thesis
              using Z12 Z13 by auto
          qed
        qed
      qed
    qed
  qed
qed

lemma col_inangle2__out:
  assumes "\<not> Bet A B C" and
    "P InAngle A B C" and
    "Q InAngle A B C" and
    "Col B P Q"
  shows "B Out P Q"
proof cases
  assume "Col A B C"
  thus ?thesis
    by (meson assms(1) assms(2) assms(3) assms(4) bet_in_angle_bet bet_out__bet 
        in_angle_out l6_6 not_col_permutation_4 or_bet_out)
next
  assume P1: "\<not> Col A B C"
  thus ?thesis
  proof cases
    assume "Col B A P"
    thus ?thesis
      by (meson assms(1) assms(2) assms(3) assms(4) bet_in_angle_bet bet_out__bet 
          l6_6 not_col_permutation_4 or_bet_out)
  next
    assume P2: "\<not> Col B A P"
    have "\<not> Col B A Q"
      using P2 assms(3) assms(4) col2__eq col_permutation_4 inangle_distincts by blast
    hence "B A OS P Q"
      using P1 P2 assms(2) assms(3) inangle_one_side invert_one_side 
        not_col_permutation_4 by auto
    thus ?thesis
      using assms(4) col_one_side_out by auto
  qed
qed

lemma inangle2__lea:
  assumes "P InAngle A B C" and
    "Q InAngle A B C"
  shows "P B Q LeA A B C"
proof -
  have P1: "P InAngle C B A"
    by (simp add: assms(1) l11_24)
  have P2: "Q InAngle C B A"
    by (simp add: assms(2) l11_24)
  have P3: "A \<noteq> B"
    using assms(1) inangle_distincts by auto
  have P4: "C \<noteq> B"
    using assms(1) inangle_distincts by blast
  have P5: "P \<noteq> B"
    using assms(1) inangle_distincts by auto
  have P6: "Q \<noteq> B"
    using assms(2) inangle_distincts by auto
  thus ?thesis
  proof cases
    assume P7: "Col A B C"
    thus ?thesis
    proof cases
      assume "Bet A B C"
      thus ?thesis
        by (simp add: P3 P4 P5 P6 l11_31_2)
    next
      assume "\<not> Bet A B C"
      hence "B Out A C"
        using P7 not_out_bet by blast
      hence "B Out P Q"
        using Out_cases assms(1) assms(2) in_angle_out l6_7 by blast
      thus ?thesis
        by (simp add: P3 P4 l11_31_1)
    qed
  next
    assume T1: "\<not> Col A B C"
    thus ?thesis
    proof cases
      assume T2: "Col B P Q"
      have "\<not> Bet A B C"
        using T1 bet_col by auto
      hence "B Out P Q"
        using T2 assms(1) assms(2) col_inangle2__out by auto
      thus ?thesis
        by (simp add: P3 P4 l11_31_1)
    next
      assume T3: "\<not> Col B P Q"
      thus ?thesis
      proof cases
        assume "Col B A P"
        hence "B Out A P"
          using Col_def T1 assms(1) col_in_angle_out by blast
        hence "P B Q CongA A B Q"
          using P6 out2__conga out_trivial by auto
        thus ?thesis
          using LeA_def assms(2) by blast
      next
        assume W0: "\<not> Col B A P"
        show ?thesis
        proof cases
          assume "Col B C P"
          hence "B Out C P"
            by (metis P1 P3 T1 bet_out_1 col_in_angle_out out_col)
          thus ?thesis
            using P3 P4 P6 lea_left_comm lea_out4__lea assms(2) 
              inangle__lea_1 out_trivial by meson
        next
          assume W0A: "\<not> Col B C P"
          show ?thesis
          proof cases
            assume "Col B A Q"
            hence "B Out A Q"
              using Col_def T1 assms(2) col_in_angle_out by blast
            thus ?thesis
              using P3 P4 P5 lea_left_comm lea_out4__lea assms(1) 
                inangle__lea out_trivial by meson
          next
            assume W0AA: "\<not> Col B A Q"
            thus ?thesis
            proof cases
              assume "Col B C Q"
              hence "B Out C Q"
                using Bet_cases P2 T1 bet_col col_in_angle_out by blast
              thus ?thesis
                using P1 P3 P4 P5 lea_comm lea_out4__lea inangle__lea out_trivial by meson
            next
              assume W0B: "\<not> Col B C Q"
              have W1: "Coplanar B P A Q"
                by (metis Col_perm T1 assms(1) assms(2) col__coplanar 
                    inangle_one_side ncoplanar_perm_13 os__coplanar)
              have W2: "\<not> Col A B P"
                by (simp add: W0 not_col_permutation_4)
              have W3: "\<not> Col Q B P"
                using Col_perm T3 by blast
              hence W4: "B P TS A Q \<or> B P OS A Q"
                using cop__one_or_two_sides
                by (simp add: W1 W2)
              {
                assume W4A: "B P TS A Q"
                have "Q InAngle P B C"
                proof -
                  have W5: "P B OS C Q"
                    using OS_def P1 W0 W0A W4A in_angle_two_sides invert_two_sides l9_2 by blast
                  have "C B OS P Q"
                    by (meson P1 P2 T1 W0A W0B inangle_one_side not_col_permutation_3 
                        not_col_permutation_4)
                  thus ?thesis
                    by (simp add: W5 invert_one_side os2__inangle)
                qed
                hence "P B Q LeA A B C"
                  by (meson assms(1) inangle__lea inangle__lea_1 lea_trans)
              }
              {
                assume W6: "B P OS A Q"
                have "B A OS P Q"
                  using Col_perm T1 W2 W0AA assms(1) assms(2) inangle_one_side 
                    invert_one_side by blast
                hence "Q InAngle P B A"
                  by (simp add: W6 os2__inangle)
                hence "P B Q LeA A B C"
                  by (meson P1 inangle__lea inangle__lea_1 lea_right_comm lea_trans)
              }
              thus ?thesis
                using W4 \<open>B P TS A Q \<Longrightarrow> P B Q LeA A B C\<close> by blast
            qed
          qed
        qed
      qed
    qed
  qed
qed

lemma conga_inangle_per__acute:
  assumes "Per A B C" and
    "P InAngle A B C" and
    "P B A CongA P B C"
  shows "Acute A B P"
proof -
  have P1: "\<not> Col A B C"
    using assms(1) assms(3) conga_diff2 conga_diff56 l8_9 by blast
  have P2: "A B P LeA A B C"
    by (simp add: assms(2) inangle__lea)
  {
    assume "A B P CongA A B C"
    hence P3: "Per A B P"
      by (meson l11_17 not_conga_sym assms(1))
    have P4: "Coplanar P C A B"
      using assms(2) inangle__coplanar ncoplanar_perm_3 by blast
    have P5: "P \<noteq> B"
      using assms(2) inangle_distincts by blast
    have "Per C B P"
      using P3 Per_cases assms(3) l11_17 by blast
    hence "False"
      using P1 P3 P4 P5 col_permutation_1 cop_per2__col by blast
  }
  hence "\<not> A B P CongA A B C" by auto
  hence "A B P LtA A B C"
    by (simp add: LtA_def P2)
  thus ?thesis
    using Acute_def assms(1) by blast
qed

lemma conga_inangle2_per__acute:
  assumes "Per A B C" and
    "P InAngle A B C" and
    "P B A CongA P B C" and
    "Q InAngle A B C"
  shows "Acute P B Q"
proof -
  have P1: "P InAngle C B A"
    using assms(2) l11_24 by auto
  have P2: "Q InAngle C B A"
    using assms(4) l11_24 by blast
  have P3: "A \<noteq> B"
    using assms(3) conga_diff2 by auto
  have P5: "P \<noteq> B"
    using assms(2) inangle_distincts by blast
  have P7: "\<not> Col A B C"
    using assms(1) assms(3) conga_distinct l8_9 by blast
  have P8: "Acute A B P"
    using assms(1) assms(2) assms(3) conga_inangle_per__acute by auto
  {
    assume "Col P B A"
    hence "Col P B C"
      using assms(3) col_conga_col by blast
    hence "False"
      using Col_perm P5 P7 \<open>Col P B A\<close> col_transitivity_2 by blast
  }
  hence P9: "\<not> Col P B A" by auto
  have P10: "\<not> Col P B C"
    using \<open>Col P B A \<Longrightarrow> False\<close> assms(3) ncol_conga_ncol by blast
  have P11: "\<not> Bet A B C"
    using P7 bet_col by blast
  show ?thesis
  proof cases
    assume "Col B A Q"
    hence "B Out A Q"
      using P11 assms(4) col_in_angle_out by auto
    thus ?thesis
      using Out_cases P5 P8 acute_out2__acute acute_sym out_trivial by blast
  next
    assume S0: "\<not> Col B A Q"
    show ?thesis
    proof cases
      assume S1: "Col B C Q"
      hence "B Out C Q"
        using P11 P2 between_symmetry col_in_angle_out by blast
      hence S2: "B Out Q C"
        using l6_6 by blast
      have S3: "B Out P P"
        by (simp add: P5 out_trivial)
      have "B Out A A"
        by (simp add: P3 out_trivial)
      hence "A B P CongA P B Q"
        using S2 conga_left_comm l11_10 S3 assms(3) by blast
      thus ?thesis
        using P8 acute_conga__acute by blast
    next
      assume S4: "\<not> Col B C Q"
      show ?thesis
      proof cases
        assume "Col B P Q"
        thus ?thesis
          using out__acute col_inangle2__out P11 assms(2) assms(4) by blast
      next
        assume S5: "\<not> Col B P Q"
        have S6: "Coplanar B P A Q"
          by (metis Col_perm P7 assms(2) assms(4) coplanar_trans_1 
              inangle__coplanar ncoplanar_perm_12 ncoplanar_perm_21)
        have S7: "\<not> Col A B P"
          using Col_cases P9 by auto
        have "\<not> Col Q B P"
          using Col_perm S5 by blast
        hence S8: "B P TS A Q \<or> B P OS A Q"
          using cop__one_or_two_sides S6 S7 by blast
        {
          assume S9: "B P TS A Q"
          have S10: "Acute P B C"
            using P8 acute_conga__acute acute_sym assms(3) by blast
          have "Q InAngle P B C"
          proof -
            have S11: "P B OS C Q"
              by (metis Col_perm OS_def P1 P10 P9 S9 in_angle_two_sides 
                  invert_two_sides l9_2)
            have "C B OS P Q"
              by (meson P1 P10 P2 P7 S4 inangle_one_side not_col_permutation_3 
                  not_col_permutation_4)
            thus ?thesis
              by (simp add: S11 invert_one_side os2__inangle)
          qed
          hence "P B Q LeA P B C"
            by (simp add: inangle__lea)
          hence "Acute P B Q"
            using S10 acute_lea_acute by blast
        }
        {
          assume S12: "B P OS A Q"
          have "B A OS P Q"
            using Col_perm P7 S7 S0 assms(2) assms(4) inangle_one_side 
              invert_one_side by blast
          hence "Q InAngle P B A"
            by (simp add: S12 os2__inangle)
          hence "Q B P LeA P B A"
            by (simp add: P3 P5 inangle1123 inangle2__lea)
          hence "P B Q LeA A B P"
            by (simp add: lea_comm)
          hence "Acute P B Q"
            using P8 acute_lea_acute by blast
        }
        thus ?thesis
          using \<open>B P TS A Q \<Longrightarrow> Acute P B Q\<close> S8 by blast
      qed
    qed
  qed
qed

lemma lta_os__ts:
  assumes (*"\<not> Col A O1 P" and*)
    "A O1 P LtA A O1 B" and
    "O1 A OS B P"
  shows "O1 P TS A B"
proof -
  have "A O1 P LeA A O1 B"
    by (simp add: assms(1) lta__lea)
  hence "\<exists> P0. P0 InAngle A O1 B \<and> A O1 P CongA A O1 P0"
    by (simp add: LeA_def)
  then obtain P' where P1: "P' InAngle A O1 B \<and> A O1 P CongA A O1 P'" by blast
  have P2: "\<not> Col A O1 B"
    using assms(2) col123__nos not_col_permutation_4 by blast
  obtain R where P3: "O1 A TS B R \<and> O1 A TS P R"
    using OS_def assms(2) by blast
  {
    assume "Col B O1 P"
    hence "Bet B O1 P"
      by (metis out2__conga assms(1) assms(2) between_trivial col_trivial_2 
          lta_not_conga one_side_chara or_bet_out out_trivial)
    hence "O1 A TS B P"
      using assms(2) col_trivial_1 one_side_chara by blast
    hence P6: "\<not> O1 A OS B P"
      using l9_9_bis by auto
    hence "False"
      using P6 assms(2) by auto
  }
  hence P4: "\<not> Col B O1 P" by auto
  thus ?thesis
    by (meson P3 assms(1) inangle__lta l9_8_1 not_and_lta not_col_permutation_4 
        os_ts__inangle two_sides_cases)
qed

lemma bet__suppa:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "B \<noteq> A'" and
    "Bet A B A'"
  shows "A B C SuppA C B A'"
proof -
  have "C B A' CongA C B A'"
    using assms(2) assms(3) conga_refl by auto
  thus ?thesis using assms(4) assms(1) SuppA_def by auto
qed

lemma ex_suppa:
  assumes "A \<noteq> B" and
    "B \<noteq> C"
  shows "\<exists> D E F. A B C SuppA D E F"
proof -
  obtain A' where "Bet A B A' \<and> Cong B A' A B"
    using segment_construction by blast
  thus ?thesis
    by (meson assms(1) assms(2) bet__suppa point_construction_different)
qed

lemma suppa_distincts:
  assumes "A B C SuppA D E F"
  shows "A \<noteq> B \<and> B \<noteq> C \<and> D \<noteq> E \<and> E \<noteq> F"
  using CongA_def SuppA_def assms by auto

lemma suppa_right_comm:
  assumes "A B C SuppA D E F"
  shows "A B C SuppA F E D"
  using SuppA_def assms conga_left_comm by auto

lemma suppa_left_comm:
  assumes "A B C SuppA D E F"
  shows "C B A SuppA D E F"
proof -
  obtain A' where P1: "Bet A B A' \<and> D E F CongA C B A'"
    using SuppA_def assms by auto
  obtain C' where P2: "Bet C B C' \<and> Cong B C' C B"
    using segment_construction by blast
  hence "C B A' CongA A B C'"
    by (metis Bet_cases P1 SuppA_def assms cong_diff_3 conga_diff45 conga_diff56 
        conga_left_comm l11_14)
  hence "D E F CongA A B C'"
    using P1 conga_trans by blast
  thus ?thesis
    by (metis CongA_def P1 P2 SuppA_def)
qed

lemma suppa_comm:
  assumes "A B C SuppA D E F"
  shows "C B A SuppA F E D"
  using assms suppa_left_comm suppa_right_comm by blast

lemma suppa_sym:
  assumes "A B C SuppA D E F"
  shows "D E F SuppA A B C"
proof -
  obtain A' where P1: "Bet A B A' \<and> D E F CongA C B A'"
    using SuppA_def assms by auto
  obtain D' where P2: "Bet D E D' \<and> Cong E D' D E"
    using segment_construction by blast
  have "A' B C CongA D E F"
    using P1 conga_right_comm not_conga_sym by blast
  hence "A B C CongA F E D'"
    by (metis P1 P2 conga_right_comm l11_13 suppa_distincts assms between_symmetry 
        cong_diff_3)
  thus ?thesis
    by (metis CongA_def P1 P2 SuppA_def)
qed

lemma conga2_suppa__suppa:
  assumes "A B C CongA A' B' C'" and
    "D E F CongA D' E' F'" and
    "A B C SuppA D E F"
  shows "A' B' C' SuppA D' E' F'"
proof -
  obtain A0 where P1: "Bet A B A0 \<and> D E F CongA C B A0"
    using SuppA_def assms(3) by auto
  hence "A B C SuppA D' E' F'"
    by (metis SuppA_def assms(2) assms(3) conga_sym conga_trans)
  hence P2: "D' E' F' SuppA A B C"
    by (simp add: suppa_sym)
  then obtain D0 where P3: "Bet D' E' D0 \<and> A B C CongA F' E' D0"
    using P2 SuppA_def by auto
  have P5: "A' B' C' CongA F' E' D0"
    using P3 assms(1) not_conga not_conga_sym by blast
  hence "D' E' F' SuppA A' B' C'"
    using P2 P3 SuppA_def by auto
  thus ?thesis
    by (simp add: suppa_sym)
qed

lemma suppa2__conga456:
  assumes "A B C SuppA D E F" and
    "A B C SuppA D' E' F'"
  shows "D E F CongA D' E' F'"
proof -
  obtain A' where P1: "Bet A B A' \<and> D E F CongA C B A'"
    using SuppA_def assms(1) by auto
  obtain A'' where P2: "Bet A B A'' \<and> D' E' F' CongA C B A''"
    using SuppA_def assms(2) by auto
  have "C B A' CongA C B A''"
  proof -
    have P3: "B Out C C" using P1
      by (simp add: CongA_def out_trivial)
    have "B Out A'' A'" using P1 P2 l6_2
      by (metis assms(1) between_symmetry conga_distinct suppa_distincts)
    thus ?thesis
      by (simp add: P3 out2__conga)
  qed
  hence "C B A' CongA D' E' F'"
    using P2 not_conga not_conga_sym by blast
  thus ?thesis
    using P1 not_conga by blast
qed

lemma suppa2__conga123:
  assumes "A B C SuppA D E F" and
    "A' B' C' SuppA D E F"
  shows "A B C CongA A' B' C'"
  using assms(1) assms(2) suppa2__conga456 suppa_sym by blast

lemma bet_out__suppa:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Bet A B C" and
    "E Out D F"
  shows "A B C SuppA D E F"
proof -
  have "D E F CongA C B C"
    using assms(2) assms(4) l11_21_b out_trivial by auto
  thus ?thesis
    using SuppA_def assms(1) assms(3) by blast
qed

lemma bet_suppa__out:
  assumes "Bet A B C" and
    "A B C SuppA D E F"
  shows "E Out D F"
proof -
  have "A B C SuppA C B C"
    using assms(1) assms(2) bet__suppa suppa_distincts by auto
  hence "C B C CongA D E F"
    using assms(2) suppa2__conga456 by auto
  thus ?thesis
    using eq_conga_out by auto
qed

lemma out_suppa__bet:
  assumes "B Out A C" and
    "A B C SuppA D E F"
  shows "Bet D E F"
proof -
  obtain B' where P1: "Bet A B B' \<and> Cong B B' A B"
    using segment_construction by blast
  have "A B C SuppA A B B'"
    by (metis P1 assms(1) assms(2) bet__suppa bet_cong_eq bet_out__bet 
        suppa_distincts suppa_left_comm)
  hence "A B B' CongA D E F"
    using assms(2) suppa2__conga456 by auto
  thus ?thesis
    using P1 bet_conga__bet by blast
qed

lemma per_suppa__per:
  assumes  "Per A B C" and
    "A B C SuppA D E F"
  shows "Per D E F"
proof -
  obtain A' where P1: "Bet A B A' \<and> D E F CongA C B A'"
    using SuppA_def assms(2) by auto
  have "Per C B A'"
  proof -
    have P2: "A \<noteq> B"
      using assms(2) suppa_distincts by auto
    have P3: "Per C B A"
      by (simp add: assms(1) l8_2)
    have "Col B A A'"
      using P1 Col_cases Col_def by blast
    thus ?thesis
      by (metis P2 P3 per_col)
  qed
  thus ?thesis
    using P1 l11_17 not_conga_sym by blast
qed

lemma per2__suppa:
  assumes  "A \<noteq> B" and
    "B \<noteq> C" and
    "D \<noteq> E" and
    "E \<noteq> F" and
    "Per A B C" and
    "Per D E F"
  shows "A B C SuppA D E F"
proof -
  obtain D' E' F' where P1: "A B C SuppA D' E' F'"
    using assms(1) assms(2) ex_suppa by blast
  have "D' E' F' CongA D E F"
    using P1 assms(3) assms(4) assms(5) assms(6) l11_16 per_suppa__per 
      suppa_distincts by blast
  thus ?thesis
    by (meson P1 conga2_suppa__suppa suppa2__conga123)
qed

lemma suppa__per:
  assumes "A B C SuppA A B C"
  shows "Per A B C"
proof -
  obtain A' where P1: "Bet A B A' \<and> A B C CongA C B A'"
    using SuppA_def assms by auto
  hence "C B A CongA C B A'"
    by (simp add: conga_left_comm)
  thus ?thesis
    using P1 Per_perm l11_18_2 by blast
qed

lemma acute_suppa__obtuse:
  assumes "Acute A B C" and
    "A B C SuppA D E F"
  shows "Obtuse D E F"
proof -
  obtain A' where P1: "Bet A B A' \<and> D E F CongA C B A'"
    using SuppA_def assms(2) by auto
  hence "Obtuse C B A'"
    by (metis obtuse_sym acute_bet__obtuse assms(1) conga_distinct)
  thus ?thesis
    by (meson P1 conga_obtuse__obtuse not_conga_sym)
qed

lemma obtuse_suppa__acute:
  assumes "Obtuse A B C" and
    "A B C SuppA D E F"
  shows "Acute D E F"
proof -
  obtain A' where P1: "Bet A B A' \<and> D E F CongA C B A'"
    using SuppA_def assms(2) by auto
  hence "Acute C B A'"
    using acute_sym assms(1) bet_obtuse__acute conga_distinct by blast
  thus ?thesis
    using P1 acute_conga__acute not_conga_sym by blast
qed

lemma lea_suppa2__lea:
  assumes "A B C SuppA A' B' C'" and
    "D E F SuppA D' E' F'"
    "A B C LeA D E F"
  shows "D' E' F' LeA A' B' C'"
proof -
  obtain A0 where P1: "Bet A B A0 \<and> A' B' C' CongA C B A0"
    using SuppA_def assms(1) by auto
  obtain D0 where P2: "Bet D E D0 \<and> D' E' F' CongA F E D0"
    using SuppA_def assms(2) by auto
  have "F E D0 LeA C B A0"
  proof -
    have P3: "D0 \<noteq> E"
      using CongA_def P2 by auto
    have P4: "A0 \<noteq> B"
      using CongA_def P1 by blast
    have P6: "Bet D0 E D"
      by (simp add: P2 between_symmetry)
    have "Bet A0 B A"
      by (simp add: P1 between_symmetry)
    thus ?thesis
      by (metis P3 P4 P6 assms(3) l11_36_aux2 lea_comm lea_distincts)
  qed
  thus ?thesis
    by (meson P1 P2 l11_30 not_conga_sym)
qed

lemma lta_suppa2__lta:
  assumes "A B C SuppA A' B' C'"
    and "D E F SuppA D' E' F'"
    and "A B C LtA D E F"
  shows "D' E' F' LtA A' B' C'"
proof -
  obtain A0 where P1: "Bet A B A0 \<and> A' B' C' CongA C B A0"
    using SuppA_def assms(1) by auto
  obtain D0 where P2: "Bet D E D0 \<and> D' E' F' CongA F E D0"
    using SuppA_def assms(2) by auto
  have "F E D0 LtA C B A0"
  proof -
    have P5: "A0 \<noteq> B"
      using CongA_def P1 by blast
    have "D0 \<noteq> E"
      using CongA_def P2 by auto
    thus ?thesis
      using assms(3) P1 P5 P2 bet2_lta__lta lta_comm by blast
  qed
  thus ?thesis
    using P1 P2 conga_preserves_lta not_conga_sym by blast
qed

lemma suppa_dec:
  "A B C SuppA D E F \<or> \<not> A B C SuppA D E F"
  by simp

lemma acute_one_side_aux:
  assumes "C A OS P B" and
    "Acute A C P" and
    "C A Perp B C"
  shows "C B OS A P"
proof -
  obtain R where T1: "C A TS P R \<and> C A TS B R"
    using OS_def assms(1) by blast
  obtain A' B' C' where P1: "Per A' B' C' \<and> A C P LtA A' B' C'"
    using Acute_def assms(2) by auto
  have P2: "Per A C B"
    by (simp add: assms(3) perp_per_1)
  hence P3: "A' B' C' CongA A C B"
    using P1 assms(1) l11_16 lta_distincts os_distincts by blast
  have P4: "A C P LtA A C B"
    by (metis P2 acute_per__lta assms(1) assms(2) os_distincts)
  {
    assume P4A: "Col P C B"
    have "Per A C P"
    proof -
      have P4B: "C \<noteq> B"
        using assms(1) os_distincts by blast
      have P4C: "Per A C B"
        by (simp add: P2)
      have "Col C B P"
        using P4A Col_cases by auto
      thus ?thesis using per_col P4B P4C by blast
    qed
    hence "False"
      using acute_not_per assms(2) by auto
  }
  hence P5: "\<not> Col P C B" by auto
  have P6: "\<not> Col A C P"
    using assms(1) col123__nos not_col_permutation_4 by blast
  have P7: "C B TS A P \<or> C B OS A P"
    using P5 assms(1) not_col_permutation_4 os_ts1324__os two_sides_cases by blast
  {
    assume P8: "C B TS A P"
    then obtain T where P9: "Col T C B \<and> Bet A T P"
      using TS_def by blast
    hence P10: "C \<noteq> T"
      using Col_def P6 P9 by auto
    have "T InAngle A C P"
      by (meson P4 P5 P8 inangle__lta assms(1) not_and_lta 
          not_col_permutation_3 os_ts__inangle)
    hence "C A OS T P"
      by (metis P10 P9 T1 TS_def col123__nos in_angle_one_side invert_one_side 
          l6_16_1 one_side_reflexivity)
    hence P13: "C A OS T B"
      using assms(1) one_side_transitivity by blast
    have "C B OS A P"
      by (meson P4 lta_os__ts assms(1) one_side_symmetry os_ts1324__os)
  }
  thus ?thesis
    using P7 by blast
qed

lemma acute_one_side_aux0:
  assumes "Col A C P" and
    "Acute A C P" and
    "C A Perp B C"
  shows "C B OS A P"
proof -
  have "Per A C B"
    by (simp add: assms(3) perp_per_1)
  hence P1: "A C P LtA A C B"
    using acute_per__lta acute_distincts assms(2) assms(3) perp_not_eq_2 by fastforce
  have P2: "C Out A P"
    using acute_col__out assms(1) assms(2) by auto
  thus ?thesis
    using Perp_cases assms(3) out_one_side perp_not_col by blast
qed

lemma acute_cop_perp__one_side:
  assumes "Acute A C P" and
    "C A Perp B C" and
    "Coplanar A B C P"
  shows "C B OS A P"
proof cases
  assume "Col A C P"
  thus ?thesis
    by (simp add: acute_one_side_aux0 assms(1) assms(2))
next
  assume P1: "\<not> Col A C P"
  have P2: "C A TS P B \<or> C A OS P B"
    using Col_cases P1 assms(2) assms(3) cop_nos__ts coplanar_perm_13 
      perp_not_col by blast
  {
    assume P3: "C A TS P B"
    obtain Bs where P4: "C Midpoint B Bs"
      using symmetric_point_construction by auto
    have "C A TS Bs B"
      by (metis P3 P4 assms(2) bet__ts l9_2 midpoint_bet midpoint_distinct_2 
          perp_not_col ts_distincts)
    hence P6: "C A OS P Bs"
      using P3 l9_8_1 by auto
    have "C Bs Perp A C"
    proof -
      have P6A: "C \<noteq> Bs"
        using P6 os_distincts by blast
      have "Col C B Bs"
        using Bet_cases Col_def P4 midpoint_bet by blast
      thus ?thesis
        using Perp_cases P6A assms(2) perp_col by blast
    qed
    hence "Bs C Perp C A"
      using Perp_perm by blast
    hence "C A Perp Bs C"
      using Perp_perm by blast
    hence "C B OS A P" using acute_one_side_aux
      by (metis P4 P6 assms(1) assms(2) col_one_side midpoint_col 
          not_col_permutation_5 perp_distinct)
  }
  {
    assume "C A OS P B"
    hence "C B OS A P" using acute_one_side_aux
      using assms(1) assms(2) by blast
  }
  thus ?thesis
    using P2 \<open>C A TS P B \<Longrightarrow> C B OS A P\<close> by auto
qed

lemma acute__not_obtuse:
  assumes "Acute A B C"
  shows "\<not> Obtuse A B C"
  using acute_obtuse__lta assms nlta by blast

lemma suma_distincts:
  assumes "A B C D E F SumA G H I"
  shows "A \<noteq> B \<and> B \<noteq> C \<and> D \<noteq> E \<and> E \<noteq> F \<and> G \<noteq> H \<and> H \<noteq> I"
proof -
  obtain J where "C B J CongA D E F \<and> \<not> B C OS A J \<and> Coplanar A B C J \<and> A B J CongA G H I"
    using SumA_def assms by auto
  thus ?thesis
    using CongA_def by blast
qed

lemma trisuma_distincts:
  assumes "A B C TriSumA D E F"
  shows "A \<noteq> B \<and> B \<noteq> C \<and> A \<noteq> C \<and> D \<noteq> E \<and> E \<noteq> F"
proof -
  obtain G H I where "A B C B C A SumA G H I \<and> G H I C A B SumA D E F"
    using TriSumA_def assms by auto
  thus ?thesis
    using suma_distincts by blast
qed

lemma ex_suma:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "D \<noteq> E" and
    "E \<noteq> F"
  shows "\<exists> G H I. A B C D E F SumA G H I"
proof -
  have "\<exists> I. A B C D E F SumA A B I"
  proof cases
    assume P1: "Col A B C"
    obtain J where P2: "D E F CongA C B J \<and> Coplanar C B J A" 
      using angle_construction_4 assms(2) assms(3) assms(4) by presburger
    have P3: "J \<noteq> B"
      using CongA_def P2 by blast
    have "\<not> B C OS A J"
      by (metis P1 between_trivial2 one_side_chara)
    hence "A B C D E F SumA A B J"
      by (meson P2 P3 SumA_def assms(1) conga_refl ncoplanar_perm_15 not_conga_sym)
    thus ?thesis by blast
  next
    assume T1: "\<not> Col A B C"
    show ?thesis
    proof cases
      assume T2: "Col D E F"
      show ?thesis
      proof cases
        assume T3: "Bet D E F"
        obtain J where T4: "B Midpoint C J"
          using symmetric_point_construction by blast
        have "A B C D E F SumA A B J"
        proof -
          have "C B J CongA D E F"
            by (metis T3 T4 assms(2) assms(3) assms(4) conga_line 
                midpoint_bet midpoint_distinct_2)
          moreover have "\<not> B C OS A J"
            by (simp add: T4 col124__nos midpoint_col)
          moreover have "Coplanar A B C J"
            using T3 bet__coplanar bet_conga__bet calculation(1) conga_sym 
              ncoplanar_perm_15 by blast
          moreover have "A B J CongA A B J"
            using CongA_def assms(1) calculation(1) conga_refl by auto
          ultimately show ?thesis
            using SumA_def by blast
        qed
        thus ?thesis
          by auto
      next
        assume T5: "\<not> Bet D E F"
        have "A B C D E F SumA A B C"
        proof -
          have "E Out D F"
            using T2 T5 l6_4_2 by auto
          hence "C B C CongA D E F"
            using assms(2) l11_21_b out_trivial by auto
          moreover have "\<not> B C OS A C"
            using os_distincts by blast
          moreover have "Coplanar A B C C"
            using ncop_distincts by auto
          moreover have "A B C CongA A B C"
            using assms(1) assms(2) conga_refl by auto
          ultimately show ?thesis
            using SumA_def by blast
        qed
        thus ?thesis
          by auto
      qed
    next
      assume T6: "\<not> Col D E F"
      then obtain J where T7: "D E F CongA C B J \<and> C B TS J A"
        using T1 ex_conga_ts not_col_permutation_4 not_col_permutation_5 by presburger
      thus ?thesis
      proof -
        have "C B J CongA D E F"
          using T7 not_conga_sym by blast
        moreover have "\<not> B C OS A J"
          by (simp add: T7 invert_two_sides l9_2 l9_9)
        moreover have "Coplanar A B C J"
          using T7 ncoplanar_perm_15 ts__coplanar by blast
        moreover have "A B J CongA A B J"
          using T7 assms(1) conga_diff56 conga_refl by blast
        ultimately show ?thesis
          using SumA_def by blast
      qed
    qed
  qed
  thus ?thesis
    by auto
qed

lemma suma2__conga:
  assumes "A B C D E F SumA G H I" and
    "A B C D E F SumA G' H' I'"
  shows "G H I CongA G' H' I'"
proof -
  obtain J where P1: "C B J CongA D E F \<and> \<not> B C OS A J \<and> Coplanar A B C J \<and> A B J CongA G H I"
    using SumA_def assms(1) by blast
  obtain J' where P2: "C B J' CongA D E F \<and> \<not> B C OS A J' \<and> Coplanar A B C J' \<and> 
A B J' CongA G' H' I'"
    using SumA_def assms(2) by blast
  have P3: "C B J CongA C B J'"
  proof -
    have "C B J CongA D E F"
      by (simp add: P1)
    moreover have "D E F CongA C B J'"
      by (simp add: P2 conga_sym)
    ultimately show ?thesis
      using not_conga by blast
  qed
  have P4: "A B J CongA A B J'"
  proof cases
    assume P5: "Col A B C"
    thus ?thesis
    proof cases
      assume P6: "Bet A B C"
      show ?thesis
      proof -
        have "C B J CongA C B J'"
          by (simp add: P3)
        moreover have "Bet C B A"
          by (simp add: P6 between_symmetry)
        moreover have "A \<noteq> B"
          using assms(1) suma_distincts by blast
        ultimately show ?thesis
          using l11_13 by blast
      qed
    next
      assume P7: "\<not> Bet A B C"
      moreover have "B Out A C"
        by (simp add: P5 calculation l6_4_2)
      moreover have "B \<noteq> J"
        using CongA_def P3 by blast
      then moreover have "B Out J J"
        using out_trivial by auto
      moreover have "B \<noteq> J'"
        using CongA_def P3 by blast
      then moreover have "B Out J' J'"
        using out_trivial by auto
      ultimately show ?thesis
        using P3 l11_10 by blast
    qed
  next
    assume P8: "\<not> Col A B C"
    show ?thesis
    proof cases
      assume P9: "Col D E F"
      have "B Out J' J"
      proof cases
        assume P10: "Bet D E F"
        show ?thesis
        proof -
          have "D E F CongA J' B C"
            using P2 conga_right_comm not_conga_sym by blast
          hence "Bet J' B C"
            using P10 bet_conga__bet by blast
          moreover have "D E F CongA J B C"
            by (simp add: P1 conga_right_comm conga_sym)
          then moreover have "Bet J B C"
            using P10 bet_conga__bet by blast
          ultimately show ?thesis
            by (metis CongA_def P3 l6_2)
        qed
      next
        assume P11: "\<not> Bet D E F"
        have P12: "E Out D F"
          by (simp add: P11 P9 l6_4_2)
        show ?thesis
        proof -
          have "B Out J' C"
          proof -
            have "D E F CongA J' B C"
              using P2 conga_right_comm conga_sym by blast
            thus ?thesis
              using l11_21_a P12 by blast
          qed
          moreover have "B Out C J"
            by (metis P3 P8 bet_conga__bet calculation col_conga_col col_out2_col 
                l6_4_2 l6_6 not_col_distincts not_conga_sym out_bet_out_1 out_trivial)
          ultimately show ?thesis
            using l6_7 by blast
        qed
      qed
      thus ?thesis
        using P8 not_col_distincts out2__conga out_trivial by blast
    next
      assume P13: "\<not> Col D E F"
      show ?thesis
      proof -
        have "B C TS A J"
        proof -
          have "Coplanar B C A J"
            using P1 coplanar_perm_8 by blast
          moreover have "\<not> Col A B C"
            by (simp add: P8)
          moreover have "\<not> B C OS A J"
            using P1 by simp
          moreover have "\<not> Col J B C"
          proof -
            have "D E F CongA J B C"
              using P1 conga_right_comm not_conga_sym by blast
            thus ?thesis
              using P13 ncol_conga_ncol by blast
          qed
          ultimately show ?thesis
            using cop__one_or_two_sides by blast
        qed
        moreover have "B C TS A J'"
        proof -
          have "Coplanar B C A J'"
            using P2 coplanar_perm_8 by blast
          moreover have "\<not> Col A B C"
            by (simp add: P8)
          moreover have "\<not> B C OS A J'"
            using P2 by simp
          moreover have "\<not> Col J' B C"
          proof -
            have "D E F CongA J' B C"
              using P2 conga_right_comm not_conga_sym by blast
            thus ?thesis
              using P13 ncol_conga_ncol by blast
          qed
          ultimately show ?thesis
            using cop_nos__ts by blast
        qed
        moreover have "A B C CongA A B C"
          by (metis P8 conga_pseudo_refl conga_right_comm not_col_distincts)
        moreover have "C B J CongA C B J'"
          by (simp add: P3)
        ultimately show ?thesis
          using l11_22a by blast
      qed
    qed
  qed
  thus ?thesis
    by (meson P1 P2 not_conga not_conga_sym)
qed

lemma suma_sym:
  assumes "A B C D E F SumA G H I"
  shows "D E F A B C SumA G H I"
proof -
  obtain J where P1: "C B J CongA D E F \<and> \<not> B C OS A J \<and> Coplanar A B C J \<and> A B J CongA G H I"
    using SumA_def assms(1) by blast
  show ?thesis
  proof cases
    assume P2: "Col A B C"
    thus ?thesis
    proof cases
      assume P3: "Bet A B C"
      obtain K where P4: "Bet F E K \<and> Cong F E E K"
        using Cong_perm segment_construction by blast
      show ?thesis
      proof -
        have P5: "F E K CongA A B C"
          by (metis CongA_def P1 P3 P4 cong_diff conga_line)
        moreover have "\<not> E F OS D K"
          using P4 bet_col col124__nos invert_one_side by blast
        moreover have "Coplanar D E F K"
          using P4 bet__coplanar ncoplanar_perm_15 by blast
        moreover have "D E K CongA G H I"
        proof -
          have "D E K CongA A B J"
          proof -
            have "F E D CongA C B J"
              by (simp add: P1 conga_left_comm conga_sym)
            moreover have "Bet F E K"
              by (simp add: P4)
            moreover have "K \<noteq> E"
              using P4 calculation(1) cong_identity conga_diff1 by blast
            moreover have "Bet C B A"
              by (simp add: Bet_perm P3)
            moreover have "A \<noteq> B"
              using CongA_def P5 by blast
            ultimately show ?thesis
              using conga_right_comm l11_13 not_conga_sym by blast
          qed
          thus ?thesis
            using P1 not_conga by blast
        qed
        ultimately show ?thesis
          using SumA_def by blast
      qed
    next
      assume T1: "\<not> Bet A B C"
      hence T2: "B Out A C"
        by (simp add: P2 l6_4_2)
      show ?thesis
      proof -
        have "F E F CongA A B C"
          by (metis T2 assms l11_21_b out_trivial suma_distincts)
        moreover have "\<not> E F OS D F"
          using os_distincts by auto
        moreover have "Coplanar D E F F"
          using ncop_distincts by auto
        moreover have "D E F CongA G H I"
        proof -
          have "A B J CongA D E F"
          proof -
            have "C B J CongA D E F"
              by (simp add: P1)
            moreover have "B Out A C"
              by (simp add: T2)
            moreover have "J \<noteq> B"
              using calculation(1) conga_distinct by auto
            moreover have "D \<noteq> E"
              using calculation(1) conga_distinct by blast
            moreover have "F \<noteq> E"
              using calculation(1) conga_distinct by blast
            ultimately show ?thesis
              by (meson Out_cases not_conga out2__conga out_trivial)
          qed
          hence "D E F CongA A B J"
            using not_conga_sym by blast
          thus ?thesis
            using P1 not_conga by blast
        qed
        ultimately show ?thesis
          using SumA_def by blast
      qed
    qed
  next
    assume Q1: "\<not> Col A B C"
    show ?thesis
    proof cases
      assume Q2: "Col D E F"
      obtain K where Q3: "A B C CongA F E K"
        using P1 angle_construction_3 conga_diff1 conga_diff56 by fastforce
      show ?thesis
      proof -
        have "F E K CongA A B C"
          by (simp add: Q3 conga_sym)
        moreover have "\<not> E F OS D K"
          using Col_cases Q2 one_side_not_col123 by blast
        moreover have "Coplanar D E F K"
          by (simp add: Q2 col__coplanar)
        moreover have "D E K CongA G H I"
        proof -
          have "D E K CongA A B J"
          proof cases
            assume "Bet D E F"
            hence "J B A CongA D E K"
              by (metis P1 bet_conga__bet calculation(1) conga_diff45 
                  conga_right_comm l11_13 not_conga_sym)
            thus ?thesis
              using conga_right_comm not_conga_sym by blast
          next
            assume "\<not> Bet D E F"
            hence W2: "E Out D F"
              using Q2 or_bet_out by blast
            have "A B J CongA D E K"
            proof -
              have "A B C CongA F E K"
                by (simp add: Q3)
              moreover have "A \<noteq> B"
                using Q1 col_trivial_1 by auto
              moreover have "E Out D F"
                by (simp add: W2)
              moreover have "B Out J C"
              proof -
                have "D E F CongA J B C"
                  by (simp add: P1 conga_left_comm conga_sym)
                thus ?thesis
                  using W2 out_conga_out by blast
              qed
              moreover have "K \<noteq> E"
                using CongA_def Q3 by blast
              ultimately show ?thesis
                using l11_10 out_trivial by blast
            qed
            thus ?thesis
              using not_conga_sym by blast
          qed
          thus ?thesis
            using P1 not_conga by blast
        qed
        ultimately show ?thesis
          using SumA_def by blast
      qed
    next
      assume W3: "\<not> Col D E F"
      then obtain K where W4: "A B C CongA F E K \<and> F E TS K D"
        using Q1 ex_conga_ts not_col_permutation_3 by blast
      show ?thesis
      proof -
        have "F E K CongA A B C"
          using W4 not_conga_sym by blast
        moreover have "\<not> E F OS D K"
        proof -
          have "E F TS D K"
            using W4 invert_two_sides l9_2 by blast
          thus ?thesis
            using l9_9 by auto
        qed
        moreover have "Coplanar D E F K"
        proof -
          have "E F TS D K"
            using W4 invert_two_sides l9_2 by blast
          thus ?thesis
            using ncoplanar_perm_8 ts__coplanar by blast
        qed
        moreover have "D E K CongA G H I"
        proof -
          have "A B J CongA K E D"
          proof -
            have "B C TS A J"
            proof -
              have "Coplanar B C A J"
                using P1 ncoplanar_perm_12 by blast
              moreover have "\<not> Col A B C"
                by (simp add: Q1)
              moreover have "\<not> B C OS A J"
                using P1 by simp
              moreover have "\<not> Col J B C"
              proof -
                {
                  assume "Col J B C"
                  have "Col D E F"
                  proof -
                    have "Col C B J"
                      using Col_perm \<open>Col J B C\<close> by blast
                    moreover have "C B J CongA D E F"
                      by (simp add: P1)
                    ultimately show ?thesis
                      using col_conga_col by blast
                  qed
                  hence "False"
                    by (simp add: W3)
                }
                thus ?thesis by blast
              qed
              ultimately show ?thesis
                using cop_nos__ts by blast
            qed
            moreover have "E F TS K D"
              using W4 invert_two_sides by blast
            moreover have "A B C CongA K E F"
              by (simp add: W4 conga_right_comm)
            moreover have "C B J CongA F E D"
              by (simp add: P1 conga_right_comm)
            ultimately show ?thesis
              using l11_22a by auto
          qed
          hence "D E K CongA A B J"
            using conga_left_comm not_conga_sym by blast
          thus ?thesis
            using P1 not_conga by blast
        qed
        ultimately show ?thesis
          using SumA_def by blast
      qed
    qed
  qed
qed

lemma conga3_suma__suma:
  assumes "A B C D E F SumA G H I" and
    "A B C CongA A' B' C'" and
    "D E F CongA D' E' F'" and
    "G H I CongA G' H' I'"
  shows "A' B' C' D' E' F' SumA G' H' I'"
proof -
  have "D' E' F' A B C SumA G' H' I'"
  proof -
    obtain J where P1: "C B J CongA D E F \<and> \<not> B C OS A J \<and> Coplanar A B C J \<and> A B J CongA G H I"
      using SumA_def assms(1) by blast
    have "A B C D' E' F' SumA G' H' I'"
    proof -
      have "C B J CongA D' E' F'"
        using P1 assms(3) not_conga by blast
      moreover have "\<not> B C OS A J"
        using P1 by simp
      moreover have "Coplanar A B C J"
        using P1 by simp
      moreover have "A B J CongA G' H' I'"
        using P1 assms(4) not_conga by blast
      ultimately show ?thesis
        using SumA_def by blast
    qed
    thus ?thesis
      by (simp add: suma_sym)
  qed
  then obtain J where P2: "F' E' J CongA A B C  \<and> \<not> E' F' OS D' J \<and> 
Coplanar D' E' F' J \<and> D' E' J CongA G' H' I'"
    using SumA_def by blast
  have "D' E' F' A' B' C' SumA G' H' I'"
  proof -
    have "F' E' J CongA A' B' C'"
    proof -
      have "F' E' J CongA A B C"
        by (simp add: P2)
      moreover have "A B C CongA A' B' C'"
        by (simp add: assms(2))
      ultimately show ?thesis
        using not_conga by blast
    qed
    moreover have "\<not> E' F' OS D' J"
      using P2 by simp
    moreover have "Coplanar D' E' F' J"
      using P2 by simp
    moreover have "D' E' J CongA G' H' I'"
      by (simp add: P2)
    ultimately show ?thesis
      using SumA_def by blast
  qed
  thus ?thesis
    by (simp add: suma_sym)
qed

lemma out6_suma__suma:
  assumes "A B C D E F SumA G H I" and
    "B Out A A'" and
    "B Out C C'" and
    "E Out D D'" and
    "E Out F F'" and
    "H Out G G'" and
    "H Out I I'"
  shows "A' B C' D' E F' SumA G' H I'"
proof -
  have "A B C CongA A' B C'"
    using Out_cases assms(2) assms(3) out2__conga by blast
  moreover have "D E F CongA D' E F'"
    using Out_cases assms(4) assms(5) out2__conga by blast
  moreover have "G H I CongA G' H I'"
    by (simp add: assms(6) assms(7) l6_6 out2__conga)
  ultimately show ?thesis
    using assms(1) conga3_suma__suma by blast
qed

lemma out546_suma__conga:
  assumes "A B C D E F SumA G H I" and
    "E Out D F"
  shows "A B C CongA G H I"
proof -
  have "A B C D E F SumA A B C"
  proof -
    have "C B C CongA D E F"
      by (metis assms(1) assms(2) l11_21_b out_trivial suma_distincts)
    moreover have "\<not> B C OS A C"
      using os_distincts by auto
    moreover have "Coplanar A B C C"
      using ncop_distincts by auto
    moreover have "A B C CongA A B C"
      by (metis suma_distincts assms(1) conga_pseudo_refl conga_right_comm)
    ultimately show ?thesis
      using SumA_def by blast
  qed
  thus ?thesis using suma2__conga assms(1) by blast
qed

lemma out546__suma:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "E Out D F"
  shows "A B C D E F SumA A B C"
proof -
  have P1: "D \<noteq> E"
    using assms(3) out_diff1 by auto
  have P2: "F \<noteq> E"
    using Out_def assms(3) by auto
  then obtain G H I where P3: "A B C D E F SumA G H I"
    using P1 assms(1) assms(2) ex_suma by presburger
  hence "G H I CongA A B C"
    by (meson conga_sym out546_suma__conga assms(3))
  thus ?thesis
    using P1 P2 P3 assms(1) assms(2) assms(3) conga3_suma__suma
      conga_refl out_diff1 by auto
qed

lemma out213_suma__conga:
  assumes "A B C D E F SumA G H I" and
    "B Out A C"
  shows "D E F CongA G H I"
  using assms(1) assms(2) out546_suma__conga suma_sym by blast

lemma out213__suma:
  assumes "D \<noteq> E" and
    "E \<noteq> F" and
    "B Out A C"
  shows "A B C D E F SumA D E F"
  by (simp add: assms(1) assms(2) assms(3) out546__suma suma_sym)

lemma suma_left_comm:
  assumes "A B C D E F SumA G H I"
  shows "C B A D E F SumA G H I"
proof -
  have "A B C CongA C B A"
    using assms conga_pseudo_refl suma_distincts by fastforce
  moreover have "D E F CongA D E F"
    by (metis assms conga_refl suma_distincts)
  moreover have "G H I CongA G H I"
    by (metis assms conga_refl suma_distincts)
  ultimately show ?thesis
    using assms conga3_suma__suma by blast
qed

lemma suma_middle_comm:
  assumes "A B C D E F SumA G H I"
  shows "A B C F E D SumA G H I"
  using assms suma_left_comm suma_sym by blast

lemma suma_right_comm:
  assumes "A B C D E F SumA G H I"
  shows "A B C D E F SumA I H G"
proof -
  have "A B C CongA A B C"
    using assms conga_refl suma_distincts by fastforce
  moreover have "D E F CongA D E F"
    by (metis assms conga_refl suma_distincts)
  moreover have "G H I CongA I H G"
    by (meson conga_right_comm suma2__conga assms)
  ultimately show ?thesis
    using assms conga3_suma__suma by blast
qed

lemma suma_comm:
  assumes "A B C D E F SumA G H I"
  shows "C B A F E D SumA I H G"
  by (simp add: assms suma_left_comm suma_middle_comm suma_right_comm)

lemma ts__suma:
  assumes "A B TS C D"
  shows "C B A A B D SumA C B D"
proof -
  have "A B D CongA A B D"
    by (metis conga_right_comm assms conga_pseudo_refl ts_distincts)
  moreover have "\<not> B A OS C D"
    using assms invert_one_side l9_9 by blast
  moreover have "Coplanar C B A D"
    using assms ncoplanar_perm_14 ts__coplanar by blast
  moreover have "C B D CongA C B D"
    by (metis assms conga_refl ts_distincts)
  ultimately show ?thesis
    using SumA_def by blast
qed

lemma ts__suma_1:
  assumes "A B TS C D"
  shows "C A B B A D SumA C A D"
  by (simp add: assms invert_two_sides ts__suma)

lemma inangle__suma:
  assumes "P InAngle A B C"
  shows "A B P P B C SumA A B C"
proof -
  have "Coplanar A B P C"
    by (simp add: assms coplanar_perm_8 inangle__coplanar)
  moreover have "\<not> B P OS A C"
    by (meson assms col123__nos col124__nos in_angle_two_sides invert_two_sides 
        l9_9_bis not_col_permutation_5)
  ultimately show ?thesis
    using SumA_def assms conga_refl inangle_distincts by blast
qed

lemma bet__suma:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "P \<noteq> B" and "Bet A B C"
  shows "A B P P B C SumA A B C"
proof -
  have "P InAngle A B C"
    using assms(1) assms(2) assms(3) assms(4) in_angle_line by auto
  thus ?thesis
    by (simp add: inangle__suma)
qed

lemma sams_chara:
  assumes "A \<noteq> B" and
    "A' \<noteq> B" and
    "Bet A B A'"
  shows "SAMS A B C D E F \<longleftrightarrow> D E F LeA C B A'"
proof -
  {
    assume "SAMS A B C D E F"
    then obtain J where "C B J CongA D E F \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J"
      using SAMS_def by auto
    have "C B J CongA D E F" 
      by (simp add: \<open>C B J CongA D E F \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J\<close>)
    have "\<not> B C OS A J"      
      by (simp add: \<open>C B J CongA D E F \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J\<close>)
    have "\<not> A B TS C J" 
      by (simp add: \<open>C B J CongA D E F \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J\<close>)
    have "Coplanar A B C J" 
      by (simp add: \<open>C B J CongA D E F \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J\<close>)
    have "A \<noteq> A'"
      using assms(2) assms(3) between_identity by blast
    have "C \<noteq> B" 
      using \<open>C B J CongA D E F\<close> conga_diff1 by auto
    have "J \<noteq> B" 
      using CongA_def \<open>C B J CongA D E F\<close> by presburger
    have "D \<noteq> E" 
      using \<open>C B J CongA D E F\<close> conga_diff45 by blast
    have "F \<noteq> E" 
      using CongA_def \<open>C B J CongA D E F\<close> by blast
    {
      assume "E Out D F"
      hence "D E F LeA C B A'"
        using assms(2) l11_31_1 \<open>C \<noteq> B\<close> by force
    }
    {
      assume "\<not> Bet A B C"
      have "D E F LeA C B A'"
      proof cases
        assume "Col A B C"
        hence "Bet C B A'"
          using assms(1) assms(3) between_exchange3 outer_transitivity_between2 
            third_point \<open>\<not> Bet A B C\<close> by blast
        thus ?thesis
          using assms(2) l11_31_2 by (simp add: \<open>C \<noteq> B\<close> \<open>D \<noteq> E\<close> \<open>F \<noteq> E\<close>)
      next
        assume T9: "\<not> Col A B C"
        show ?thesis
        proof cases
          assume T10: "Col D E F"
          show ?thesis
          proof cases
            assume T11: "Bet D E F"
            have "D E F CongA C B J" 
              using \<open>C B J CongA D E F\<close> conga_sym_equiv by auto
            hence T12: "Bet C B J"
              using T11 bet_conga__bet by blast
            have "A B TS C J"
            proof -
              have "\<not> Col J A B"
                using T9 T12 bet_col col2__eq col_permutation_1 \<open>J \<noteq> B\<close> by blast
              moreover have "\<exists> T. Col T A B \<and> Bet C T J"
                using T12 col_trivial_3 by blast
              ultimately show ?thesis
                using T9 TS_def col_permutation_1 by blast
            qed
            hence "False" 
              using \<open>\<not> A B TS C J\<close> by blast
            thus ?thesis by simp
          next
            assume "\<not> Bet D E F"
            thus ?thesis
              using T10 \<open>E Out D F \<Longrightarrow> D E F LeA C B A'\<close> or_bet_out by auto
          qed
        next
          assume T13: "\<not> Col D E F"
          show ?thesis
          proof -
            have "C B J LeA C B A'"
            proof -
              have "J InAngle C B A'"
              proof -
                have "A' \<noteq> B"
                  by (simp add: assms(2))
                moreover have "Bet A B A'"
                  by (simp add: assms(3))
                moreover have "C InAngle A B J"
                proof -
                  have "\<not> Col J B C"
                  proof -
                    have "\<not> Col D E F"
                      by (simp add: T13)
                    moreover have "D E F CongA J B C" 
                      using conga_left_comm not_conga_sym 
                        \<open>C B J CongA D E F\<close> by blast
                    ultimately show ?thesis
                      using ncol_conga_ncol by blast
                  qed
                  hence "B C TS A J" 
                    using T9 cop_nos__ts coplanar_perm_8 
                      \<open>Coplanar A B C J\<close> \<open>\<not> B C OS A J\<close> by blast
                  then obtain X where T14: "Col X B C \<and> Bet A X J"
                    using TS_def by blast
                  {
                    assume T15: "X \<noteq> B"
                    have "B Out X C"
                    proof -
                      have "Col B X C"
                        by (simp add: Col_perm T14)
                      moreover have "B A OS X C"
                      proof -
                        have "A B OS X C"
                        proof -
                          have "A B OS X J"
                            using T14 T9 T15 bet_out calculation col_transitivity_2 
                              col_trivial_2 l6_21 out_one_side by metis
                          moreover have "A B OS J C"
                            using T14 T9 calculation cop_nts__os l5_2 
                              not_col_permutation_2 one_side_chara one_side_symmetry 
                              \<open> \<not> A B TS C J\<close> \<open>Coplanar A B C J\<close> by (metis)
                          ultimately show ?thesis
                            using one_side_transitivity by blast
                        qed
                        thus ?thesis
                          by (simp add: invert_one_side)
                      qed
                      ultimately show ?thesis
                        using col_one_side_out by auto
                    qed
                  }
                  hence "Bet A X J \<and> (X = B \<or> B Out X C)"
                    using T14 by blast
                  thus ?thesis
                    using InAngle_def \<open>C \<noteq> B\<close> \<open>J \<noteq> B\<close> assms(1) by auto
                qed
                ultimately show ?thesis
                  using in_angle_reverse l11_24 by blast
              qed
              moreover have "C B J CongA C B J" 
                using \<open>C \<noteq> B\<close> \<open>J \<noteq> B\<close> conga_refl by presburger
              ultimately show ?thesis
                by (simp add: inangle__lea)
            qed
            moreover have "D E F LeA C B J" 
              using \<open>C B J CongA D E F\<close> conga__lea456123 by auto
            ultimately show ?thesis
              using lea_trans by blast
          qed
        qed
      qed
    }
    hence "D E F LeA C B A'" 
      using SAMS_def \<open>E Out D F \<Longrightarrow> D E F LeA C B A'\<close> 
        \<open>SAMS A B C D E F\<close> by fastforce
  }
  {
    assume P1: "D E F LeA C B A'"
    have P2: "A \<noteq> A'"
      using assms(2) assms(3) between_identity by blast
    have P3: "C \<noteq> B"
      using P1 lea_distincts by auto
    have P4: "D \<noteq> E"
      using P1 lea_distincts by auto
    have P5: "F \<noteq> E"
      using P1 lea_distincts by auto
    have "SAMS A B C D E F"
    proof cases
      assume P6: "Col A B C"
      show ?thesis
      proof cases
        assume P7: "Bet A B C"
        have "E Out D F"
        proof -
          have "B Out C A'"
            by (meson Bet_perm P3 P7 assms(1) assms(2) assms(3) l6_2)
          moreover have "C B A' CongA D E F"
            using P1 calculation l11_21_b out_lea__out by blast
          ultimately show ?thesis
            using out_conga_out by blast
        qed
        moreover have "C B C CongA D E F"
          using P3 calculation l11_21_b out_trivial by auto
        moreover have "\<not> B C OS A C"
          using os_distincts by auto
        moreover have "\<not> A B TS C C"
          by (simp add: not_two_sides_id)
        moreover have "Coplanar A B C C"
          using ncop_distincts by auto
        ultimately show ?thesis
          using SAMS_def assms(1) by blast
      next
        assume P8: "\<not> Bet A B C"
        have P9: "B Out A C"
          by (simp add: P6 P8 l6_4_2)
        obtain J where P10: "D E F CongA C B J"
          using P3 P4 P5 angle_construction_3 by blast
        show ?thesis
        proof -
          have "C B J CongA D E F"
            using P10 not_conga_sym by blast
          moreover have "\<not> B C OS A J"
            using Col_cases P6 one_side_not_col123 by blast
          moreover have "\<not> A B TS C J"
            using Col_cases P6 TS_def by blast
          moreover have "Coplanar A B C J"
            using P6 col__coplanar by auto
          ultimately show ?thesis
            using P8 SAMS_def assms(1) by blast
        qed
      qed
    next
      assume P11: "\<not> Col A B C"
      have P12: "\<not> Col A' B C"
        using P11 assms(2) assms(3) bet_col bet_col1 colx by blast
      show ?thesis
      proof cases
        assume P13: "Col D E F"
        have P14: "E Out D F"
        proof -
          {
            assume P14: "Bet D E F"
            have "D E F LeA C B A'"
              by (simp add: P1)
            hence "Bet C B A'"
              using P14 bet_lea__bet by blast
            hence "Col A' B C"
              using Col_def Col_perm by blast
            hence "False"
              by (simp add: P12)
          }
          hence "\<not> Bet D E F" by auto
          thus ?thesis
            by (simp add: P13 l6_4_2)
        qed
        show ?thesis
        proof -
          have "C B C CongA D E F"
            by (simp add: P3 P14 l11_21_b out_trivial)
          moreover have "\<not> B C OS A C"
            using os_distincts by auto
          moreover have "\<not> A B TS C C"
            by (simp add: not_two_sides_id)
          moreover have "Coplanar A B C C"
            using ncop_distincts by auto
          ultimately show ?thesis
            using P14 SAMS_def assms(1) by blast
        qed
      next
        assume P15: "\<not> Col D E F"
        obtain J where P16: "D E F CongA C B J \<and> C B TS J A"
          using P11 P15 ex_conga_ts not_col_permutation_3 by presburger
        show ?thesis
        proof -
          have "C B J CongA D E F"
            by (simp add: P16 conga_sym)
          moreover have "\<not> B C OS A J"
          proof -
            have "C B TS A J"
              using P16 by (simp add: l9_2)
            thus ?thesis
              using invert_one_side l9_9 by blast
          qed
          moreover have "\<not> A B TS C J \<and> Coplanar A B C J"
          proof cases
            assume "Col A B J"
            thus ?thesis
              using TS_def ncop__ncols not_col_permutation_1 by blast
          next
            assume P17: "\<not> Col A B J"
            have "\<not> A B TS C J"
            proof -
              have "A' B OS J C"
              proof -
                have "\<not> Col A' B C"
                  by (simp add: P12)
                moreover have "\<not> Col B A' J"
                proof -
                  {
                    assume "Col B A' J"
                    hence "False"
                      by (metis P17 assms(2) assms(3) bet_col col_trivial_2 colx)
                  }
                  thus ?thesis by auto
                qed
                moreover have "J InAngle A' B C"
                proof -
                  obtain K where P20: "K InAngle C B A' \<and> D E F CongA C B K"
                    using LeA_def P1 by blast
                  have "J InAngle C B A'"
                  proof -
                    have "C B A' CongA C B A'"
                      by (simp add: P3 assms(2) conga_pseudo_refl conga_right_comm)
                    moreover have "C B K CongA C B J"
                    proof -
                      have "C B K CongA D E F"
                        using P20 not_conga_sym by blast
                      moreover have "D E F CongA C B J"
                        by (simp add: P16)
                      ultimately show ?thesis
                        using not_conga by blast
                    qed
                    moreover have "K InAngle C B A'"
                      using P20 by simp
                    moreover have "C B OS J A'"
                    proof -
                      have "C B TS J A" using P16
                        by simp
                      moreover have "C B TS A' A"
                        using Col_perm P12 assms(3) bet__ts between_symmetry 
                          calculation invert_two_sides ts_distincts by blast
                      ultimately show ?thesis
                        using OS_def by auto
                    qed
                    ultimately show ?thesis
                      using conga_preserves_in_angle by blast
                  qed
                  thus ?thesis
                    by (simp add: l11_24)
                qed
                ultimately show ?thesis
                  by (simp add: in_angle_one_side)
              qed
              hence "A' B OS C J"
                by (simp add: one_side_symmetry)
              hence "\<not> A' B TS C J"
                by (simp add: l9_9_bis)
              thus ?thesis
                using assms(2) assms(3) bet_col bet_col1 col_preserves_two_sides by blast
            qed
            moreover have "Coplanar A B C J"
            proof -
              have "C B TS J A"
                using P16 by simp
              thus ?thesis
                by (simp add: coplanar_perm_20 ts__coplanar)
            qed
            ultimately  show ?thesis by auto
          qed
          ultimately show ?thesis
            using P11 SAMS_def assms(1) bet_col by auto
        qed
      qed
    qed
  }
  thus ?thesis
    using \<open>SAMS A B C D E F \<Longrightarrow> D E F LeA C B A'\<close> by blast
qed

lemma sams_distincts:
  assumes "SAMS A B C D E F"
  shows "A \<noteq> B \<and> B \<noteq> C \<and> D \<noteq> E \<and> E \<noteq> F"
proof -
  obtain J where "C B J CongA D E F \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J"
    using SAMS_def assms by auto
  thus ?thesis
    by (metis SAMS_def assms conga_distinct)
qed

lemma sams_sym:
  assumes "SAMS A B C D E F"
  shows "SAMS D E F A B C"
proof -
  have P1: "A \<noteq> B"
    using assms sams_distincts by blast
  have P3: "D \<noteq> E"
    using assms sams_distincts by blast
  obtain D' where P5: "E Midpoint D D'"
    using symmetric_point_construction by blast
  obtain A' where P6: "B Midpoint A A'"
    using symmetric_point_construction by blast
  have P8: "E \<noteq> D'"
    using P3 P5 is_midpoint_id_2 by blast
  have P9: "A \<noteq> A'"
    using P1 P6 l7_3 by auto
  hence P10: "B \<noteq> A'"
    using P6 P9 midpoint_not_midpoint by auto
  hence "D E F LeA C B A'"
    using P1 P6 assms midpoint_bet sams_chara by fastforce
  hence "D E F LeA A' B C"
    by (simp add: lea_right_comm)
  hence "A B C LeA D' E F"
    by (metis Mid_cases P1 P10 P3 P5 P6 P8 l11_36 midpoint_bet)
  hence "A B C LeA F E D'"
    by (simp add: lea_right_comm)
  moreover have "D \<noteq> E"
    by (simp add: P3)
  moreover have "D' \<noteq> E"
    using P8 by auto
  moreover have "Bet D E D'"
    by (simp add: P5 midpoint_bet)
  thus ?thesis
    using P3 P8 calculation(1) sams_chara by fastforce
qed

lemma sams_right_comm:
  assumes "SAMS A B C D E F"
  shows "SAMS A B C F E D"
proof -
  have P1: "E Out D F \<or> \<not> Bet A B C"
    using SAMS_def assms by blast
  obtain J where P2: "C B J CongA D E F \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J"
    using SAMS_def assms by auto
  {
    assume "E Out D F"
    hence "E Out F D \<or> \<not> Bet A B C"
      by (simp add: l6_6)
  }
  {
    assume "\<not> Bet A B C"
    hence "E Out F D \<or> \<not> Bet A B C" by auto
  }
  hence "E Out F D \<or> \<not> Bet A B C"
    using \<open>E Out D F \<Longrightarrow> E Out F D \<or> \<not> Bet A B C\<close> P1 by auto
  moreover have "C B J CongA F E D"
  proof -
    have "C B J CongA D E F"
      by (simp add: P2)
    thus ?thesis
      by (simp add: conga_right_comm)
  qed
  ultimately show ?thesis
    using P2 SAMS_def assms by auto
qed

lemma sams_left_comm:
  assumes "SAMS A B C D E F"
  shows "SAMS C B A D E F"
proof -
  have "SAMS D E F A B C"
    by (simp add: assms sams_sym)
  hence "SAMS D E F C B A"
    using sams_right_comm by blast
  thus ?thesis
    using sams_sym by blast
qed

lemma sams_comm:
  assumes "SAMS A B C D E F"
  shows "SAMS C B A F E D"
  using assms sams_left_comm sams_right_comm by blast

lemma conga2_sams__sams:
  assumes "A B C CongA A' B' C'" and
    "D E F CongA D' E' F'" and
    "SAMS A B C D E F"
  shows "SAMS A' B' C' D' E' F'"
proof -
  obtain A0 where "B Midpoint A A0"
    using symmetric_point_construction by auto
  obtain A'0 where "B' Midpoint A' A'0"
    using symmetric_point_construction by blast
  have "D' E' F' LeA C' B' A'0"
  proof -
    have "D E F LeA C B A0"
      by (metis \<open>B Midpoint A A0\<close> assms(1) assms(3) conga_distinct 
          midpoint_bet midpoint_distinct_2 sams_chara)
    moreover have "D E F CongA D' E' F'"
      by (simp add: assms(2))
    moreover have "C B A0 CongA C' B' A'0"
    proof -
      have "A0 B C CongA A'0 B' C'"
        by (metis \<open>B Midpoint A A0\<close> \<open>B' Midpoint A' A'0\<close> assms(1) calculation(1) 
            conga_diff45 l11_13 lea_distincts midpoint_bet midpoint_not_midpoint)
      thus ?thesis
        using conga_comm by blast
    qed
    ultimately show ?thesis
      using l11_30 by blast
  qed
  thus ?thesis
    by (metis \<open>B' Midpoint A' A'0\<close> assms(1) conga_distinct lea_distincts 
        midpoint_bet sams_chara)
qed

lemma out546__sams:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "E Out D F"
  shows "SAMS A B C D E F"
proof -
  obtain A' where "Bet A B A' \<and> Cong B A' A B"
    using segment_construction by blast
  hence "D E F LeA C B A'"
    using assms(1) assms(2) assms(3) cong_diff_3 l11_31_1 by fastforce
  thus ?thesis
    using \<open>Bet A B A' \<and> Cong B A' A B\<close> assms(1) lea_distincts sams_chara by blast
qed

lemma out213__sams:
  assumes "D \<noteq> E" and
    "E \<noteq> F" and
    "B Out A C"
  shows "SAMS A B C D E F"
  by (simp add: sams_sym assms(1) assms(2) assms(3) out546__sams)

lemma bet_suma__sams:
  assumes "A B C D E F SumA G H I" and
    "Bet G H I"
  shows "SAMS A B C D E F"
proof -
  obtain A' where P1: "C B A' CongA D E F \<and> \<not> B C OS A A' \<and> Coplanar A B C A' \<and> A B A' CongA G H I"
    using SumA_def assms(1) by auto
  hence "G H I CongA A B A'"
    using not_conga_sym by blast
  hence "Bet A B A'"
    using assms(2) bet_conga__bet by blast
  show ?thesis
  proof -
    have "E Out D F \<or> \<not> Bet A B C"
    proof -
      {
        assume "Bet A B C"
        hence "E Out D F"
        proof -
          have "B Out C A'"
          proof -
            have "C \<noteq> B"
              using assms(1) suma_distincts by blast
            moreover have "A' \<noteq> B"
              using CongA_def \<open>G H I CongA A B A'\<close> by blast
            moreover have "A \<noteq> B"
              using CongA_def \<open>G H I CongA A B A'\<close> by blast
            moreover have "Bet C B A"
              by (simp add: Bet_perm \<open>Bet A B C\<close>)
            ultimately show ?thesis
              using Out_def \<open>Bet A B A'\<close> \<open>Bet A B C\<close> l5_2 by auto
          qed
          moreover have "C B A' CongA D E F"
            using P1 by simp
          ultimately show ?thesis
            using l11_21_a by blast
        qed
      }
      thus ?thesis
        by blast
    qed
    moreover have "\<exists> J. (C B J CongA D E F \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J)"
    proof -
      have "C B A' CongA D E F"
        by (simp add: P1)
      moreover have "\<not> B C OS A A'"
        by (simp add: P1)
      moreover have "\<not> A B TS C A'"
        using Col_def TS_def \<open>Bet A B A'\<close> by blast
      moreover have "Coplanar A B C A'"
        by (simp add: P1)
      ultimately show ?thesis
        by blast
    qed
    ultimately show ?thesis
      using CongA_def SAMS_def 
        \<open>C B A' CongA D E F \<and> \<not> B C OS A A' \<and> Coplanar A B C A' \<and> A B A' CongA G H I\<close> 
      by auto
  qed
qed

lemma bet__sams:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "P \<noteq> B" and
    "Bet A B C"
  shows "SAMS A B P P B C"
  by (meson assms(1) assms(2) assms(3) assms(4) bet__suma bet_suma__sams)

lemma suppa__sams:
  assumes "A B C SuppA D E F"
  shows "SAMS A B C D E F"
proof -
  obtain A' where P1: "Bet A B A' \<and> D E F CongA C B A'"
    using SuppA_def assms by auto
  hence "SAMS A B C C B A'"
    by (metis assms bet__sams conga_diff45 conga_diff56 suppa2__conga123)
  thus ?thesis
    by (meson P1 assms conga2_sams__sams not_conga_sym suppa2__conga123)
qed

lemma os_ts__sams:
  assumes "B P TS A C" and
    "A B OS P C"
  shows "SAMS A B P P B C"
proof -
  have "B Out P C \<or> \<not> Bet A B P"
    using assms(2) bet_col col123__nos by blast
  moreover have "\<exists> J. (P B J CongA P B C \<and> \<not> B P OS A J \<and> \<not> A B TS P J \<and> Coplanar A B P J)"
    by (metis assms(1) assms(2) conga_refl l9_9 os__coplanar os_distincts)
  ultimately show ?thesis
    using SAMS_def assms(2) os_distincts by auto
qed

lemma os2__sams:
  assumes "A B OS P C" and
    "C B OS P A"
  shows "SAMS A B P P B C"
  by (simp add: os_ts__sams assms(1) assms(2) invert_one_side l9_31)

lemma inangle__sams:
  assumes "P InAngle A B C"
  shows "SAMS A B P P B C"
proof -
  have "Bet A B C \<or> B Out A C \<or> \<not> Col A B C"
    using l6_4_2 by blast
  {
    assume "Bet A B C"
    hence "SAMS A B P P B C"
      using assms bet__sams inangle_distincts by fastforce
  }
  {
    assume "B Out A C"
    hence "SAMS A B P P B C"
      by (metis assms in_angle_out inangle_distincts out213__sams)
  }
  {
    assume "\<not> Col A B C"
    hence "\<not> Bet A B C"
      using Col_def by auto
    {
      assume "Col B A P"
      have "SAMS A B P P B C"
        by (metis \<open>Col B A P\<close> \<open>\<not> Bet A B C\<close> assms col_in_angle_out inangle_distincts out213__sams)
    }
    {
      assume "\<not> Col B A P"
      {
        assume "Col B C P"
        have "SAMS A B P P B C"
          by (metis sams_comm \<open>Col B C P\<close> \<open>\<not> Bet A B C\<close> assms between_symmetry 
              col_in_angle_out inangle_distincts l11_24 out546__sams)
      }
      {
        assume "\<not> Col B C P"
        have "SAMS A B P P B C"
        proof -
          have "B P TS A C"
            by (simp add: \<open>\<not> Col B A P\<close> \<open>\<not> Col B C P\<close> assms in_angle_two_sides 
                invert_two_sides)
          moreover have "A B OS P C"
            by (simp add: \<open>\<not> Col A B C\<close> \<open>\<not> Col B A P\<close> assms in_angle_one_side)
          ultimately show ?thesis
            by (simp add: os_ts__sams)
        qed
      }
      hence "SAMS A B P P B C"
        using \<open>Col B C P \<Longrightarrow> SAMS A B P P B C\<close> by blast
    }
    hence "SAMS A B P P B C"
      using \<open>Col B A P \<Longrightarrow> SAMS A B P P B C\<close> by blast
  }
  thus ?thesis
    using \<open>B Out A C \<Longrightarrow> SAMS A B P P B C\<close> 
      \<open>Bet A B C \<Longrightarrow> SAMS A B P P B C\<close> \<open>Bet A B C \<or> B Out A C \<or> \<not> Col A B C\<close> 
    by blast
qed

lemma sams_suma__lea123789:
  assumes "A B C D E F SumA G H I" and
    "SAMS A B C D E F"
  shows "A B C LeA G H I"
proof cases
  assume "Col A B C"
  show ?thesis
  proof cases
    assume "Bet A B C"
    have P1: "(A \<noteq> B \<and> (E Out D F \<or> \<not> Bet A B C)) \<and> 
(\<exists> J. (C B J CongA D E F \<and> \<not> (B C OS A J) \<and> \<not> (A B TS C J) \<and> Coplanar A B C J))"
      using SAMS_def assms(2) by auto
    {
      assume "E Out D F"
      hence "A B C CongA G H I"
        using assms(1) out546_suma__conga by auto
      hence "A B C LeA G H I"
        by (simp add: conga__lea)
    }
    thus ?thesis
      using P1 \<open>Bet A B C\<close> by blast
  next
    assume "\<not> Bet A B C"
    hence "B Out A C"
      using \<open>Col A B C\<close> or_bet_out by auto
    thus ?thesis
      by (metis assms(1) l11_31_1 suma_distincts)
  qed
next
  assume "\<not> Col A B C"
  show ?thesis
  proof cases
    assume "Col D E F"
    show ?thesis
    proof cases
      assume "Bet D E F"
      have "SAMS D E F A B C"
        using assms(2) sams_sym by auto
      hence "B Out A C"
        using SAMS_def \<open>Bet D E F\<close> by blast
      thus ?thesis using l11_31_1
        by (metis assms(1) suma_distincts)
    next
      assume "\<not> Bet D E F"
      have "A B C CongA G H I"
      proof -
        have "A B C D E F SumA G H I"
          by (simp add: assms(1))
        moreover have "E Out D F"
          using \<open>Col D E F\<close> \<open>\<not> Bet D E F\<close> l6_4_2 by auto
        ultimately show ?thesis
          using out546_suma__conga by auto
      qed
      show ?thesis
        by (simp add: \<open>A B C CongA G H I\<close> conga__lea)
    qed
  next
    assume "\<not> Col D E F"
    show ?thesis
    proof cases
      assume "Col G H I"
      show ?thesis
      proof cases
        assume "Bet G H I"
        thus ?thesis
          by (metis assms(1) l11_31_2 suma_distincts)
      next
        assume "\<not> Bet G H I"
        hence "H Out G I"
          by (simp add: \<open>Col G H I\<close> l6_4_2)
        have "E Out D F \<or> \<not> Bet A B C"
          using \<open>\<not> Col A B C\<close> bet_col by auto
        {
          assume "\<not> Bet A B C"
          then obtain J where P2: "C B J CongA D E F \<and> \<not> B C OS A J \<and> 
  Coplanar A B C J \<and> A B J CongA G H I"
            using SumA_def assms(1) by blast
          have "G H I CongA A B J"
            using P2 not_conga_sym by blast
          hence "B Out A J"
            using \<open>H Out G I\<close> out_conga_out by blast
          hence "B C OS A J"
            using Col_perm \<open>\<not> Col A B C\<close> out_one_side by blast
          hence "False"
            using \<open>C B J CongA D E F \<and> \<not> B C OS A J \<and> 
    Coplanar A B C J \<and> A B J CongA G H I\<close> by linarith
        }
        hence "False"
          using Col_def \<open>\<not> Col A B C\<close> by blast
        thus ?thesis by blast
      qed
    next
      assume "\<not> Col G H I"
      obtain J where P4: "C B J CongA D E F \<and> \<not> B C OS A J \<and> 
\<not> A B TS C J \<and> Coplanar A B C J"
        using SAMS_def assms(2) by auto
      {
        assume "Col J B C"
        have "J B C CongA D E F"
          by (simp add: P4 conga_left_comm)
        hence "Col D E F"
          using col_conga_col \<open>Col J B C\<close> by blast
        hence "False"
          using \<open>\<not> Col D E F\<close> by blast
      }
      hence "\<not> Col J B C" by blast
      have "A B J CongA G H I"
      proof -
        have "A B C D E F SumA A B J"
        proof -
          have "C B J CongA D E F"
            using P4 by simp
          moreover have "\<not> B C OS A J"
            by (simp add: P4)
          moreover have "Coplanar A B C J"
            by (simp add: P4)
          moreover have "A B J CongA A B J"
            by (metis \<open>\<not> Col A B C\<close> \<open>\<not> Col J B C\<close> col_trivial_1 conga_refl)
          ultimately show ?thesis
            using SumA_def by blast
        qed
        thus ?thesis
          using assms(1) suma2__conga by auto
      qed
      have "\<not> Col J B A"
        using \<open>A B J CongA G H I\<close> \<open>\<not> Col G H I\<close> col_conga_col not_col_permutation_3 by blast
      have "A B C LeA A B J"
      proof -
        have "C InAngle A B J"
          by (metis Col_perm P4 \<open>\<not> Col A B C\<close> \<open>\<not> Col J B A\<close> \<open>\<not> Col J B C\<close> 
              cop_nos__ts coplanar_perm_7 coplanar_perm_8 invert_two_sides 
              l9_2 os_ts__inangle)
        moreover have "A B C CongA A B C"
          using calculation in_angle_asym inangle3123 inangle_distincts by auto
        ultimately show ?thesis
          using inangle__lea by blast
      qed
      thus ?thesis
        using \<open>A B J CongA G H I\<close> conga__lea lea_trans by blast
    qed
  qed
qed

lemma sams_suma__lea456789:
  assumes "A B C D E F SumA G H I" and
    "SAMS A B C D E F"
  shows "D E F LeA G H I"
proof -
  have "D E F A B C SumA G H I"
    by (simp add: assms(1) suma_sym)
  moreover have "SAMS D E F A B C"
    using assms(2) sams_sym by blast
  ultimately show ?thesis
    using sams_suma__lea123789 by auto
qed

lemma sams_lea2__sams:
  assumes "SAMS A' B' C' D' E' F'" and
    "A B C LeA A' B' C'" and
    "D E F LeA D' E' F'"
  shows "SAMS A B C D E F"
proof -
  obtain A0 where "B Midpoint A A0"
    using symmetric_point_construction by auto
  obtain A'0 where "B' Midpoint A' A'0"
    using symmetric_point_construction by auto
  have "D E F LeA C B A0"
  proof -
    have "D' E' F' LeA C B A0"
    proof -
      have "D' E' F' LeA C' B' A'0"
        by (metis \<open>B' Midpoint A' A'0\<close> assms(1) assms(2) lea_distincts 
            midpoint_bet midpoint_distinct_2 sams_chara)
      moreover have "C' B' A'0 LeA C B A0"
        by (metis Mid_cases \<open>B Midpoint A A0\<close> \<open>B' Midpoint A' A'0\<close> assms(2) 
            l11_36_aux2 l7_3_2 lea_comm lea_distincts midpoint_bet sym_preserve_diff)
      ultimately show ?thesis
        using lea_trans by blast
    qed
    moreover have "D E F LeA D' E' F'"
      using assms(3) by auto
    ultimately show ?thesis
      using \<open>D' E' F' LeA C B A0\<close> assms(3) lea_trans by blast
  qed
  thus ?thesis
    by (metis \<open>B Midpoint A A0\<close> assms(2) lea_distincts midpoint_bet sams_chara)
qed

lemma sams_lea456_suma2__lea:
  assumes "D E F LeA D' E' F'" and
    "SAMS A B C D' E' F'" and
    "A B C D E F SumA G H I" and
    "A B C D' E' F' SumA G' H' I'"
  shows "G H I LeA G' H' I'"
proof cases
  assume "E' Out D' F'"
  have "G H I CongA G' H' I'"
  proof -
    have "G H I CongA A B C"
    proof -
      have "A B C D E F SumA G H I"
        by (simp add: assms(3))
      moreover have "E Out D F"
        using \<open>E' Out D' F'\<close> assms(1) out_lea__out by blast
      ultimately show ?thesis
        using conga_sym out546_suma__conga by blast
    qed
    moreover have "A B C CongA G' H' I'"
      using \<open>E' Out D' F'\<close> assms(4) out546_suma__conga by blast
    ultimately show ?thesis
      using conga_trans by blast
  qed
  thus ?thesis
    by (simp add: conga__lea)
next
  assume T1: "\<not> E' Out D' F'"
  show ?thesis
  proof cases
    assume T2: "Col A B C"
    have "E' Out D' F' \<or> \<not> Bet A B C"
      using assms(2) SAMS_def by simp
    {
      assume "\<not> Bet A B C"
      hence "B Out A C"
        by (simp add: T2 l6_4_2)
      have "G H I LeA G' H' I'"
      proof -
        have "D E F LeA D' E' F'"
          by (simp add: assms(1))
        moreover have "D E F CongA G H I"
          using \<open>B Out A C\<close> assms(3) out213_suma__conga by auto
        moreover have "D' E' F' CongA G' H' I'"
          using \<open>B Out A C\<close> assms(4) out213_suma__conga by auto
        ultimately show ?thesis
          using l11_30 by blast
      qed
    }
    thus ?thesis
      using T1 \<open>E' Out D' F' \<or> \<not> Bet A B C\<close> by auto
  next
    assume "\<not> Col A B C"
    show ?thesis
    proof cases
      assume "Col D' E' F'"
      have "SAMS D' E' F' A B C"
        by (simp add: assms(2) sams_sym)
      {
        assume "\<not> Bet D' E' F'"
        hence "G H I LeA G' H' I'"
          using not_bet_out T1 \<open>Col D' E' F'\<close> by auto
      }
      thus ?thesis
        by (metis assms(2) assms(3) assms(4) bet_lea__bet l11_31_2 
            sams_suma__lea456789 suma_distincts)
    next
      assume "\<not> Col D' E' F'"
      show ?thesis
      proof cases
        assume "Col D E F"
        have "\<not> Bet D E F"
          using bet_lea__bet Col_def \<open>\<not> Col D' E' F'\<close> assms(1) by blast
        thus ?thesis
        proof -
          have "A B C LeA G' H' I'"
            using assms(2) assms(4) sams_suma__lea123789 by auto
          moreover have "A B C CongA G H I"
            by (meson \<open>Col D E F\<close> \<open>\<not> Bet D E F\<close> assms(3) or_bet_out 
                out213_suma__conga suma_sym)
          moreover have "G' H' I' CongA G' H' I'"
          proof -
            have "G' \<noteq> H'"
              using calculation(1) lea_distincts by blast
            moreover have "H' \<noteq> I'"
              using \<open>A B C LeA G' H' I'\<close> lea_distincts by blast
            ultimately show ?thesis
              using conga_refl by auto
          qed
          ultimately show ?thesis
            using l11_30 by blast
        qed
      next
        assume "\<not> Col D E F"
        show ?thesis
        proof cases
          assume "Col G' H' I'"
          show ?thesis
          proof cases
            assume "Bet G' H' I'"
            show ?thesis
            proof -
              have "G \<noteq> H"
                using assms(3) suma_distincts by auto
              moreover have "I \<noteq> H"
                using assms(3) suma_distincts by blast
              moreover have "G' \<noteq> H'"
                using assms(4) suma_distincts by auto
              moreover have "I' \<noteq> H'"
                using assms(4) suma_distincts by blast
              ultimately show ?thesis
                by (simp add: \<open>Bet G' H' I'\<close> l11_31_2)
            qed
          next
            assume "\<not> Bet G' H' I'"
            have "B Out A C"
            proof -
              have "H' Out G' I'"
                using \<open>Col G' H' I'\<close> l6_4_2 by (simp add: \<open>\<not> Bet G' H' I'\<close>)
              moreover have "A B C LeA G' H' I'" using sams_suma__lea123789
                using assms(2) assms(4) by auto
              ultimately show ?thesis
                using out_lea__out by blast
            qed
            hence "Col A B C"
              using Col_perm out_col by blast
            hence "False"
              using \<open>\<not> Col A B C\<close> by auto
            thus ?thesis by blast
          qed
        next
          assume "\<not> Col G' H' I'"
          obtain F'1 where P5: "C B F'1 CongA D' E' F' \<and> \<not> B C OS A F'1 \<and> 
\<not> A B TS C F'1 \<and> Coplanar A B C F'1"
            using SAMS_def assms(2) by auto
          have P6: "D E F LeA C B F'1"
          proof -
            have "D E F CongA D E F"
              using \<open>\<not> Col D E F\<close> conga_refl not_col_distincts by fastforce
            moreover have "D' E' F' CongA C B F'1"
              by (simp add: P5 conga_sym)
            ultimately show ?thesis
              using assms(1) l11_30 by blast
          qed
          then obtain F1 where P6: "F1 InAngle C B F'1 \<and> D E F CongA C B F1"
            using LeA_def by auto
          have "A B F'1 CongA G' H' I'"
          proof -
            have "A B C D' E' F' SumA A B F'1"
            proof -
              have "C B F'1 CongA D' E' F'"
                using P5 by blast
              moreover have "\<not> B C OS A F'1"
                using P5 by auto
              moreover have "Coplanar A B C F'1"
                by (simp add: P5)
              moreover have "A B F'1 CongA A B F'1"
              proof -
                have "A \<noteq> B"
                  using \<open>\<not> Col A B C\<close> col_trivial_1 by blast
                moreover have "B \<noteq> F'1"
                  using P6 inangle_distincts by auto
                ultimately show ?thesis
                  using conga_refl by auto
              qed
              ultimately show ?thesis
                using SumA_def by blast
            qed
            moreover have "A B C D' E' F' SumA G' H' I'"
              by (simp add: assms(4))
            ultimately show ?thesis
              using suma2__conga by auto
          qed
          have "\<not> Col A B F'1"
            using \<open>A B F'1 CongA G' H' I'\<close> \<open>\<not> Col G' H' I'\<close> col_conga_col by blast
          have "\<not> Col C B F'1"
          proof -
            have "\<not> Col D' E' F'"
              by (simp add: \<open>\<not> Col D' E' F'\<close>)
            moreover have "D' E' F' CongA C B F'1"
              using P5 not_conga_sym by blast
            ultimately show ?thesis
              using ncol_conga_ncol by blast
          qed
          show ?thesis
          proof -
            have "A B F1 LeA A B F'1"
            proof -
              have "F1 InAngle A B F'1"
              proof -
                have "F1 InAngle F'1 B A"
                proof -
                  have "F1 InAngle F'1 B C"
                    by (simp add: P6 l11_24)
                  moreover have "C InAngle F'1 B A"
                  proof -
                    have "B C TS A F'1"
                      using Col_perm P5 \<open>\<not> Col A B C\<close> \<open>\<not> Col C B F'1\<close> 
                        cop_nos__ts ncoplanar_perm_12 by blast
                    obtain X where "Col X B C \<and> Bet A X F'1"
                      using TS_def \<open>B C TS A F'1\<close> by auto
                    have "Bet F'1 X A"
                      using Bet_perm \<open>Col X B C \<and> Bet A X F'1\<close> by blast
                    moreover have "(X = B) \<or> (B Out X C)"
                    proof -
                      have "B A OS X C"
                      proof -
                        have "A B OS X F'1"
                          by (metis \<open>Col X B C \<and> Bet A X F'1\<close> \<open>\<not> Col A B C\<close> 
                              \<open>\<not> Col A B F'1\<close> bet_out_1 calculation out_one_side)
                        moreover have "A B OS F'1 C"
                          using Col_perm P5 \<open>\<not> Col A B C\<close> \<open>\<not> Col A B F'1\<close> 
                            cop_nos__ts one_side_symmetry by blast
                        ultimately show ?thesis
                          using invert_one_side one_side_transitivity by blast
                      qed
                      thus ?thesis
                        using Col_cases \<open>Col X B C \<and> Bet A X F'1\<close> col_one_side_out by blast
                    qed
                    ultimately show ?thesis
                      by (metis InAngle_def \<open>\<not> Col A B C\<close> \<open>\<not> Col A B F'1\<close> not_col_distincts)
                  qed
                  ultimately show ?thesis
                    using in_angle_trans by blast
                qed
                thus ?thesis
                  using l11_24 by blast
              qed
              moreover have "A B F1 CongA A B F1"
              proof -
                have "A \<noteq> B"
                  using \<open>\<not> Col A B C\<close> col_trivial_1 by blast
                moreover have "B \<noteq> F1"
                  using P6 conga_diff56 by blast
                ultimately show ?thesis
                  using conga_refl by auto
              qed
              ultimately show ?thesis
                by (simp add: inangle__lea)
            qed
            moreover have "A B F1 CongA G H I"
            proof -
              have "A B C D E F SumA A B F1"
              proof -
                have "B C TS F1 A"
                proof -
                  have "B C TS F'1 A"
                  proof -
                    have "B C TS A F'1"
                      using Col_perm P5 \<open>\<not> Col A B C\<close> \<open>\<not> Col C B F'1\<close> 
                        cop_nos__ts ncoplanar_perm_12 by blast
                    thus ?thesis
                      using l9_2 by blast
                  qed
                  moreover have "B C OS F'1 F1"
                  proof -
                    have "\<not> Col C B F'1"
                      by (simp add: \<open>\<not> Col C B F'1\<close>)
                    moreover have "\<not> Col B C F1"
                    proof -
                      have "\<not> Col D E F"
                        using \<open>\<not> Col D E F\<close> by auto
                      moreover have "D E F CongA C B F1"
                        by (simp add: P6)
                      ultimately show ?thesis
                        using ncol_conga_ncol not_col_permutation_4 by blast
                    qed
                    moreover have "F1 InAngle C B F'1" using P6 by blast
                    ultimately show ?thesis
                      using in_angle_one_side invert_one_side one_side_symmetry by blast
                  qed
                  ultimately show ?thesis
                    using l9_8_2 by blast
                qed
                thus ?thesis
                proof -
                  have "C B F1 CongA D E F"
                    using P6 not_conga_sym by blast
                  moreover have "\<not> B C OS A F1"
                    using \<open>B C TS F1 A\<close> l9_9 one_side_symmetry by blast
                  moreover have "Coplanar A B C F1"
                    using \<open>B C TS F1 A\<close> ncoplanar_perm_9 ts__coplanar by blast
                  moreover have "A B F1 CongA A B F1"
                  proof -
                    have "A \<noteq> B"
                      using \<open>\<not> Col A B C\<close> col_trivial_1 by blast
                    moreover have "B \<noteq> F1"
                      using P6 conga_diff56 by blast
                    ultimately show ?thesis
                      using conga_refl by auto
                  qed
                  ultimately show ?thesis
                    using SumA_def by blast
                qed
              qed
              moreover have "A B C D E F SumA G H I"
                by (simp add: assms(3))
              ultimately show ?thesis
                using suma2__conga by auto
            qed
            ultimately show ?thesis
              using \<open>A B F'1 CongA G' H' I'\<close> l11_30 by blast
          qed
        qed
      qed
    qed
  qed
qed

lemma sams_lea123_suma2__lea:
  assumes "A B C LeA A' B' C'" and
    "SAMS A' B' C' D E F" and
    "A B C D E F SumA G H I" and
    "A' B' C' D E F SumA G' H' I'"
  shows "G H I LeA G' H' I'"
  by (meson assms(1) assms(2) assms(3) assms(4) sams_lea456_suma2__lea sams_sym suma_sym)

lemma sams_lea2_suma2__lea:
  assumes "A B C LeA A' B' C'" and
    "D E F LeA D' E' F'" and
    "SAMS A' B' C' D' E' F'" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "G H I LeA G' H' I'"
proof -
  obtain G'' H'' I'' where "A B C D' E' F' SumA G'' H'' I''"
    using assms(4) assms(5) ex_suma suma_distincts by presburger
  have "G H I LeA G'' H'' I''"
  proof -
    have "D E F LeA D' E' F'"
      by (simp add: assms(2))
    moreover have "SAMS A B C D' E' F'"
    proof -
      have "SAMS A' B' C' D' E' F'"
        by (simp add: assms(3))
      moreover have "A B C LeA A' B' C'"
        using assms(1) by auto
      moreover have "D' E' F' LeA D' E' F'"
        using assms(2) lea_distincts lea_refl by blast
      ultimately show ?thesis
        using sams_lea2__sams by blast
    qed
    moreover have "A B C D E F SumA G H I"
      by (simp add: assms(4))
    moreover have "A B C D' E' F' SumA G'' H'' I''"
      by (simp add: \<open>A B C D' E' F' SumA G'' H'' I''\<close>)
    ultimately show ?thesis
      using sams_lea456_suma2__lea by blast
  qed
  moreover have "G'' H'' I'' LeA G' H' I'"
    using sams_lea123_suma2__lea assms(3) assms(5) \<open>A B C D' E' F' SumA G'' H'' I''\<close> 
      assms(1) by blast
  ultimately show ?thesis
    using lea_trans by blast
qed

lemma sams2_suma2__conga456:
  assumes "SAMS A B C D E F" and
    "SAMS A B C D' E' F'" and
    "A B C D E F SumA G H I" and
    "A B C D' E' F' SumA G H I"
  shows "D E F CongA D' E' F'"
proof cases
  assume "Col A B C"
  show ?thesis
  proof cases
    assume P2: "Bet A B C"
    hence "E Out D F"
      using assms(1) SAMS_def by blast
    moreover have "E' Out D' F'"
      using P2 assms(2) SAMS_def by blast
    ultimately show ?thesis
      by (simp add: l11_21_b)
  next
    assume "\<not> Bet A B C"
    hence "B Out A C"
      using \<open>Col A B C\<close> or_bet_out by blast
    show ?thesis
    proof -
      have "D E F CongA G H I"
      proof -
        have "A B C D E F SumA G H I"
          by (simp add: assms(3))
        thus ?thesis
          using \<open>B Out A C\<close> out213_suma__conga by auto
      qed
      moreover have "G H I CongA D' E' F'"
      proof -
        have "A B C D' E' F' SumA G H I"
          by (simp add: assms(4))
        hence "D' E' F' CongA G H I"
          using \<open>B Out A C\<close> out213_suma__conga by auto
        thus ?thesis
          using not_conga_sym by blast
      qed
      ultimately show ?thesis
        using not_conga by blast
    qed
  qed
next
  assume "\<not> Col A B C"
  obtain J where T1: "C B J CongA D E F \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J"
    using assms(1) SAMS_def by blast
  have T1A: "C B J CongA D E F"
    using T1 by simp
  have T1B: "\<not> B C OS A J"
    using T1 by simp
  have T1C: "\<not> A B TS C J"
    using T1 by simp
  have T1D: "Coplanar A B C J"
    using T1 by simp
  obtain J' where T2: "C B J' CongA D' E' F' \<and> \<not> B C OS A J' \<and> 
\<not> A B TS C J' \<and> Coplanar A B C J'"
    using assms(2) SAMS_def by blast
  have T2A: "C B J' CongA D' E' F'"
    using T2 by simp
  have T2B: "\<not> B C OS A J'"
    using T2 by simp
  have T2C: "\<not> A B TS C J'"
    using T2 by simp
  have T2D: "Coplanar A B C J'"
    using T2 by simp
  have T3: "C B J CongA C B J'"
  proof -
    have "A B J CongA A B J'"
    proof -
      have "A B J CongA G H I"
      proof -
        have "A B C D E F SumA A B J"
          using SumA_def T1A T1B T1D \<open>\<not> Col A B C\<close> conga_distinct 
            conga_refl not_col_distincts by auto
        thus ?thesis
          using assms(3) suma2__conga by blast
      qed
      moreover have "G H I CongA A B J'"
      proof -
        have "A B C D' E' F' SumA G H I"
          by (simp add: assms(4))
        moreover have "A B C D' E' F' SumA A B J'"
          using SumA_def T2A T2B T2D \<open>\<not> Col A B C\<close> conga_distinct 
            conga_refl not_col_distincts by auto
        ultimately show ?thesis
          using suma2__conga by auto
      qed
      ultimately show ?thesis
        using conga_trans by blast
    qed
    have "B Out J J' \<or> A B TS J J'"
    proof -
      have "Coplanar A B J J'"
        using T1D T2D \<open>\<not> Col A B C\<close> coplanar_trans_1 ncoplanar_perm_8 
          not_col_permutation_2 by blast
      moreover have "A B J CongA A B J'"
        by (simp add: \<open>A B J CongA A B J'\<close>)
      ultimately show ?thesis
        by (simp add: conga_cop__or_out_ts)
    qed
    {
      assume "B Out J J'"
      hence "C B J CongA C B J'"
        by (metis Out_cases \<open>\<not> Col A B C\<close> bet_out between_trivial 
            not_col_distincts out2__conga)
    }
    {
      assume "A B TS J J'"
      hence "A B OS J C"
        by (meson T1C T1D TS_def \<open>\<not> Col A B C\<close> cop_nts__os not_col_permutation_2 
            one_side_symmetry)
      hence "A B TS C J'"
        using \<open>A B TS J J'\<close> l9_8_2 by blast
      hence "False"
        by (simp add: T2C)
      hence "C B J CongA C B J'"
        by blast
    }
    thus ?thesis
      using \<open>B Out J J' \<Longrightarrow> C B J CongA C B J'\<close> \<open>B Out J J' \<or> A B TS J J'\<close> by blast
  qed
  hence "C B J CongA D' E' F'"
    using T2A not_conga by blast
  thus ?thesis
    using T1A not_conga not_conga_sym by blast
qed

lemma sams2_suma2__conga123:
  assumes "SAMS A B C D E F" and
    "SAMS A' B' C' D E F" and
    "A B C D E F SumA G H I" and
    "A' B' C' D E F SumA G H I"
  shows "A B C CongA A' B' C'"
proof -
  have "SAMS D E F A B C"
    by (simp add: assms(1) sams_sym)
  moreover have "SAMS D E F A' B' C'"
    by (simp add: assms(2) sams_sym)
  moreover have "D E F  A B C SumA G H I"
    by (simp add: assms(3) suma_sym)
  moreover have "D E F A' B' C' SumA G H I"
    using assms(4) suma_sym by blast
  ultimately show ?thesis
    using sams2_suma2__conga456 by auto
qed

lemma suma_assoc_1:
  assumes "SAMS A B C D E F" and
    "SAMS D E F G H I" and
    "A B C D E F SumA A' B' C'" and
    "D E F G H I SumA D' E' F'" and
    "A' B' C' G H I SumA K L M"
  shows "A B C D' E' F' SumA K L M"
proof -
  obtain A0 where P1: "Bet A B A0 \<and> Cong A B B A0"
    using Cong_perm segment_construction by blast
  obtain D0 where P2: "Bet D E D0 \<and> Cong D E E D0"
    using Cong_perm segment_construction by blast
  show ?thesis
  proof cases
    assume "Col A B C"
    show ?thesis
    proof cases
      assume "Bet A B C"
      hence "E Out D F"
        using SAMS_def assms(1) by simp
      show ?thesis
      proof -
        have "A' B' C' CongA A B C"
          using assms(3) \<open>E Out D F\<close> conga_sym out546_suma__conga by blast
        moreover have "G H I CongA D' E' F'"
          using assms(4) \<open>E Out D F\<close> out213_suma__conga by auto
        ultimately show ?thesis
          by (meson conga3_suma__suma suma2__conga assms(5))
      qed
    next
      assume "\<not> Bet A B C"
      hence "B Out A C"
        using \<open>Col A B C\<close> l6_4_2 by auto
      have "A \<noteq> B"
        using \<open>B Out A C\<close> out_distinct by auto
      have "B \<noteq> C"
        using \<open>\<not> Bet A B C\<close> between_trivial by auto
      have "D' \<noteq> E'"
        using assms(4) suma_distincts by blast
      have "E' \<noteq> F'"
        using assms(4) suma_distincts by auto
      show ?thesis
      proof -
        obtain K0 L0 M0 where P3:"A B C D' E' F' SumA K0 L0 M0"
          using ex_suma \<open>A \<noteq> B\<close> \<open>B \<noteq> C\<close> \<open>D' \<noteq> E'\<close> \<open>E' \<noteq> F'\<close> by presburger
        moreover have "A B C CongA A B C"
          using \<open>A \<noteq> B\<close> \<open>B \<noteq> C\<close> conga_refl by auto
        moreover have "D' E' F' CongA D' E' F'"
          using \<open>D' \<noteq> E'\<close> \<open>E' \<noteq> F'\<close> conga_refl by auto
        moreover have "K0 L0 M0 CongA K L M"
        proof -
          have "K0 L0 M0 CongA D' E' F'"
            using P3 \<open>B Out A C\<close> conga_sym out213_suma__conga by blast
          moreover have "D' E' F' CongA K L M"
          proof -
            have "D E F G H I SumA D' E' F'"
              by (simp add: assms(4))
            moreover have "D E F G H I SumA K L M"
              by (meson conga3_suma__suma out213_suma__conga sams2_suma2__conga456 
                  suma2__conga \<open>B Out A C\<close> assms(2) assms(3) assms(5) 
                  calculation not_conga_sym)
            ultimately show ?thesis
              using suma2__conga by auto
          qed
          ultimately show ?thesis
            using not_conga by blast
        qed
        ultimately show ?thesis
          using conga3_suma__suma by blast
      qed
    qed
  next
    assume T1: "\<not> Col A B C"
    have "\<not> Col C B A0"
      by (metis Col_def P1 \<open>\<not> Col A B C\<close> cong_diff l6_16_1)
    show ?thesis
    proof cases
      assume "Col D E F"
      show ?thesis
      proof cases
        assume "Bet D E F"
        have "H Out G I" using SAMS_def assms(2) \<open>Bet D E F\<close> by blast
        have "A B C D E F SumA A' B' C'"
          by (simp add: assms(3))
        moreover have "A B C CongA A B C"
          by (metis \<open>\<not> Col A B C\<close> conga_pseudo_refl conga_right_comm not_col_distincts)
        moreover have "D E F CongA D' E' F'"
          using \<open>H Out G I\<close> assms(4) out546_suma__conga by auto
        moreover have "A' B' C' CongA K L M"
          using \<open>H Out G I\<close> assms(5) out546_suma__conga by auto
        ultimately show ?thesis
          using conga3_suma__suma by blast
      next
        assume "\<not> Bet D E F"
        hence "E Out D F"
          using not_bet_out by (simp add: \<open>Col D E F\<close>)
        show ?thesis
        proof -
          have "A' B' C' CongA A B C"
            using assms(3) \<open>E Out D F\<close> conga_sym out546_suma__conga by blast
          moreover have "G H I CongA D' E' F'"
            using out546_suma__conga \<open>E Out D F\<close> assms(4) out213_suma__conga by auto
          moreover have "K L M CongA K L M"
          proof -
            have "K \<noteq> L \<and> L \<noteq> M"
              using assms(5) suma_distincts by blast
            thus ?thesis
              using conga_refl by auto
          qed
          ultimately show ?thesis
            using assms(5) conga3_suma__suma by blast
        qed
      qed
    next
      assume "\<not> Col D E F"
      hence "\<not> Col F E D0"
        by (metis Col_def P2 cong_diff l6_16_1 not_col_distincts)
      show ?thesis
      proof cases
        assume "Col G H I"
        show ?thesis
        proof cases
          assume "Bet G H I"
          have "SAMS G H I D E F"
            by (simp add: assms(2) sams_sym)
          hence "E Out D F"
            using SAMS_def \<open>Bet G H I\<close> by blast
          hence "Col D E F"
            using Col_perm out_col by blast
          hence "False"
            using \<open>\<not> Col D E F\<close> by auto
          thus ?thesis by simp
        next
          assume "\<not> Bet G H I"
          hence "H Out G I"
            using SAMS_def by (simp add: \<open>Col G H I\<close> l6_4_2)
          show ?thesis
          proof -
            have "A B C CongA A B C"
              by (metis \<open>\<not> Col A B C\<close> conga_refl not_col_distincts)
            moreover have "D E F CongA D' E' F'"
              using assms(4) out546_suma__conga \<open>H Out G I\<close> by auto
            moreover have "A' B' C' CongA K L M"
              using \<open>H Out G I\<close> assms(5) out546_suma__conga by auto
            ultimately show ?thesis
              using assms(3) conga3_suma__suma by blast
          qed
        qed
      next
        assume "\<not> Col G H I"
        have "\<not> B C OS A A0"
          using P1 col_trivial_1 one_side_chara by blast
        have "E F TS D D0"
          by (metis P2 \<open>\<not> Col D E F\<close> \<open>\<not> Col F E D0\<close> bet__ts bet_col 
              between_trivial not_col_permutation_1)
        show ?thesis
        proof cases
          assume "Col A' B' C'"
          show ?thesis
          proof cases
            assume "Bet A' B' C'"
            show ?thesis
            proof cases
              assume "Col D' E' F'"
              show ?thesis
              proof cases
                assume "Bet D' E' F'"
                have "A B C CongA G H I"
                proof -
                  have "A B C CongA D0 E F"
                  proof -
                    have "SAMS A B C D E F"
                      by (simp add: assms(1))
                    moreover have "SAMS D0 E F D E F"
                      by (metis P2 \<open>\<not> Col D E F\<close> \<open>\<not> Col F E D0\<close> bet__sams 
                          between_symmetry not_col_distincts sams_right_comm)
                    moreover have "A B C D E F SumA A' B' C'"
                      by (simp add: assms(3))
                    moreover have "D0 E F D E F SumA A' B' C'"
                    proof -
                      have "D E F D0 E F SumA A' B' C'"
                      proof -
                        have "F E D0 CongA D0 E F"
                          by (metis \<open>\<not> Col F E D0\<close> col_trivial_1 col_trivial_2 
                              conga_pseudo_refl)
                        moreover have "\<not> E F OS D D0"
                          using P2 col_trivial_1 one_side_chara by blast
                        moreover have "Coplanar D E F D0"
                          by (meson P2 bet__coplanar ncoplanar_perm_1)
                        moreover have "D E D0 CongA A' B' C'"
                          using assms(3) P2 \<open>Bet A' B' C'\<close> \<open>\<not> Col F E D0\<close> 
                            conga_line not_col_distincts suma_distincts by auto
                        ultimately show ?thesis
                          using SumA_def by blast
                      qed
                      thus ?thesis
                        by (simp add: \<open>D E F D0 E F SumA A' B' C'\<close> suma_sym)
                    qed
                    ultimately show ?thesis
                      using sams2_suma2__conga123 by blast
                  qed
                  moreover have "D0 E F CongA G H I"
                  proof -
                    have "SAMS D E F D0 E F"
                      using P2 \<open>\<not> Col D E F\<close> \<open>\<not> Col F E D0\<close> bet__sams 
                        not_col_distincts sams_right_comm by auto
                    moreover have "D E F D0 E F SumA D' E' F'"
                    proof -
                      have "F E D0 CongA D0 E F"
                        by (metis (no_types) \<open>\<not> Col F E D0\<close> col_trivial_1
                            col_trivial_2 conga_pseudo_refl)
                      moreover have "\<not> E F OS D D0"
                        using P2 col_trivial_1 one_side_chara by blast
                      moreover have "Coplanar D E F D0"
                        using P2 bet__coplanar ncoplanar_perm_1 by blast
                      moreover have "D E D0 CongA D' E' F'"
                        using assms(3) P2 \<open>Bet D' E' F'\<close> \<open>\<not> Col F E D0\<close> assms(4) 
                          conga_line not_col_distincts suma_distincts by auto
                      ultimately show ?thesis
                        using SumA_def by blast
                    qed
                    ultimately show ?thesis
                      using assms(2) assms(4) sams2_suma2__conga456 by auto
                  qed
                  ultimately show ?thesis
                    using conga_trans by blast
                qed
                hence "G H I CongA A B C"
                  using not_conga_sym by blast
                have "D' E' F' A B C SumA K L M"
                proof -
                  have "A' B' C' CongA D' E' F'"
                    by (metis suma_distincts \<open>Bet A' B' C'\<close> \<open>Bet D' E' F'\<close> 
                        assms(4) assms(5) conga_line)
                  thus ?thesis
                    by (meson conga3_suma__suma suma2__conga 
                        \<open>G H I CongA A B C\<close> assms(5))
                qed
                thus ?thesis
                  by (simp add: suma_sym)
              next
                assume "\<not> Bet D' E' F'"
                hence "E' Out D' F'"
                  by (simp add: \<open>Col D' E' F'\<close> l6_4_2)
                have "D E F LeA D' E' F'"
                  using assms(2) assms(4) sams_suma__lea123789 by blast
                hence "E Out D F"
                  using \<open>E' Out D' F'\<close> out_lea__out by blast
                hence "Col D E F"
                  using Col_perm out_col by blast
                hence "False"
                  using \<open>\<not> Col D E F\<close> by auto
                thus ?thesis by simp
              qed
            next
              assume "\<not> Col D' E' F'"
              have "D E F CongA C B A0"
              proof -
                have "SAMS A B C D E F"
                  by (simp add: assms(1))
                moreover have "SAMS A B C C B A0"
                  using P1 \<open>\<not> Col A B C\<close> \<open>\<not> Col C B A0\<close> bet__sams not_col_distincts by auto
                moreover have "A B C D E F SumA A' B' C'"
                  by (simp add: assms(3))
                moreover have "A B C C B A0 SumA A' B' C'"
                proof -
                  have "A B C C B A0 SumA A B A0"
                    by (metis P1 \<open>\<not> Col A B C\<close> \<open>\<not> Col C B A0\<close> bet__suma 
                        col_trivial_1 col_trivial_2)
                  moreover have "A B C CongA A B C"
                    using \<open>SAMS A B C C B A0\<close> calculation 
                      sams2_suma2__conga123 by auto
                  moreover have "C B A0 CongA C B A0"
                    using \<open>SAMS A B C C B A0\<close> calculation(1) 
                      sams2_suma2__conga456 by auto
                  moreover have "A B A0 CongA A' B' C'"
                    using P1 \<open>Bet A' B' C'\<close> \<open>\<not> Col C B A0\<close> assms(3) 
                      conga_line not_col_distincts suma_distincts by auto
                  ultimately show ?thesis
                    using conga3_suma__suma by blast
                qed
                ultimately show ?thesis
                  using sams2_suma2__conga456 by blast
              qed
              have "SAMS C B A0 G H I"
              proof -
                have "D E F CongA C B A0"
                  by (simp add: \<open>D E F CongA C B A0\<close>)
                moreover have "G H I CongA G H I"
                  using \<open>\<not> Col G H I\<close> conga_refl not_col_distincts by fastforce
                moreover have "SAMS D E F G H I"
                  by (simp add: assms(2))
                ultimately show ?thesis
                  using conga2_sams__sams by blast
              qed
              then obtain J where P3: "A0 B J CongA G H I \<and> \<not> B A0 OS C J \<and> 
\<not> C B TS A0 J \<and> Coplanar C B A0 J"
                using SAMS_def by blast
              obtain F1 where P4: "F E F1 CongA G H I \<and> \<not> E F OS D F1 \<and> 
\<not> D E TS F F1 \<and> Coplanar D E F F1"
                using SAMS_def assms(2) by auto
              have "C B J CongA D' E' F'"
              proof -
                have "C B J CongA D E F1"
                proof -
                  have "(B A0 TS C J \<and> E F TS D F1) \<or> (B A0 OS C J \<and> E F OS D F1)"
                  proof -
                    have "B A0 TS C J"
                    proof -
                      have "Coplanar B A0 C J"
                        using P3 ncoplanar_perm_12 by blast
                      moreover have "\<not> Col C B A0"
                        by (simp add: \<open>\<not> Col C B A0\<close>)
                      moreover have "\<not> Col J B A0"
                        using P3 \<open>\<not> Col G H I\<close> col_conga_col not_col_permutation_3 by blast
                      moreover have "\<not> B A0 OS C J"
                        using P3 by simp
                      ultimately show ?thesis
                        by (simp add: cop_nos__ts)
                    qed
                    moreover have "E F TS D F1"
                    proof -
                      have "Coplanar E F D F1"
                        using P4 ncoplanar_perm_12 by blast
                      moreover have "\<not> Col D E F"
                        by (simp add: \<open>\<not> Col D E F\<close>)
                      moreover have "\<not> Col F1 E F"
                        using P4 \<open>\<not> Col G H I\<close> col_conga_col col_permutation_3 by blast
                      moreover have "\<not> E F OS D F1"
                        using P4 by auto
                      ultimately show ?thesis
                        by (simp add: cop_nos__ts)
                    qed
                    ultimately show ?thesis
                      by simp
                  qed
                  moreover have "C B A0 CongA D E F"
                    using \<open>D E F CongA C B A0\<close> not_conga_sym by blast
                  moreover have "A0 B J CongA F E F1"
                  proof -
                    have "A0 B J CongA G H I"
                      by (simp add: P3)
                    moreover have "G H I CongA F E F1"
                      using P4 not_conga_sym by blast
                    ultimately show ?thesis
                      using conga_trans by blast
                  qed
                  ultimately show ?thesis
                    using l11_22 by auto
                qed
                moreover have "D E F1 CongA D' E' F'"
                proof -
                  have "D E F G H I SumA D E F1"
                    using P4 SumA_def \<open>\<not> Col D E F\<close> conga_distinct 
                      conga_refl not_col_distincts by auto
                  moreover have "D E F G H I SumA D' E' F'"
                    by (simp add: assms(4))
                  ultimately show ?thesis
                    using suma2__conga by auto
                qed
                ultimately show ?thesis
                  using conga_trans by blast
              qed
              show ?thesis
              proof -
                have "A B C D' E' F' SumA A B J"
                proof -
                  have "C B TS J A"
                  proof -
                    have "C B TS A0 A"
                    proof -
                      have "B \<noteq> A0"
                        using \<open>\<not> Col C B A0\<close> not_col_distincts by blast
                      moreover have "\<not> Col B C A"
                        using Col_cases \<open>\<not> Col A B C\<close> by auto
                      moreover have "Bet A B A0"
                        by (simp add: P1)
                      ultimately show ?thesis
                        by (metis Bet_cases Col_cases \<open>\<not> Col C B A0\<close> 
                            bet__ts invert_two_sides not_col_distincts)
                    qed
                    moreover have "C B OS A0 J"
                    proof -
                      have "\<not> Col J C B"
                        using \<open>C B J CongA D' E' F'\<close> \<open>\<not> Col D' E' F'\<close> 
                          col_conga_col not_col_permutation_2 by blast
                      moreover have "\<not> Col A0 C B"
                        using Col_cases \<open>\<not> Col C B A0\<close> by blast
                      ultimately show ?thesis
                        using P3 cop_nos__ts by blast
                    qed
                    ultimately show ?thesis
                      using l9_8_2 by blast
                  qed
                  moreover have "C B J CongA D' E' F'"
                    by (simp add: \<open>C B J CongA D' E' F'\<close>)
                  moreover have "\<not> B C OS A J"
                    using calculation(1) invert_one_side l9_9_bis 
                      one_side_symmetry by blast
                  moreover have "Coplanar A B C J"
                    using calculation(1) ncoplanar_perm_15 ts__coplanar by blast
                  moreover have "A B J CongA A B J"
                  proof -
                    have "A \<noteq> B"
                      using \<open>\<not> Col A B C\<close> col_trivial_1 by auto
                    moreover have "B \<noteq> J"
                      using \<open>C B TS J A\<close> ts_distincts by blast
                    ultimately show ?thesis
                      using conga_refl by auto
                  qed
                  ultimately show ?thesis
                    using SumA_def by blast
                qed
                moreover have "A B J CongA K L M"
                proof -
                  have "A' B' C' G H I SumA A B J"
                  proof -
                    have "A B A0 CongA A' B' C'"
                      using P1 \<open>Bet A' B' C'\<close> \<open>\<not> Col A B C\<close> \<open>\<not> Col C B A0\<close> 
                        assms(5) conga_line not_col_distincts suma_distincts by auto
                    moreover have "A0 B J CongA G H I"
                      by (simp add: P3)
                    moreover have "A B A0 A0 B J SumA A B J"
                    proof -
                      have "A0 B J CongA A0 B J"
                      proof -
                        have "A0 \<noteq> B"
                          using \<open>\<not> Col C B A0\<close> col_trivial_2 by auto
                        moreover have "B \<noteq> J"
                          using CongA_def \<open>A0 B J CongA G H I\<close> by blast
                        ultimately show ?thesis
                          using conga_refl by auto
                      qed
                      moreover have "\<not> B A0 OS A J"
                        by (simp add: Col_def P1 col123__nos)
                      moreover have "Coplanar A B A0 J"
                        using P1 bet__coplanar by auto
                      moreover have "A B J CongA A B J"
                        using P1 \<open>\<not> Col A B C\<close> between_symmetry calculation(1) 
                          l11_13 not_col_distincts by blast
                      ultimately show ?thesis
                        using SumA_def by blast
                    qed
                    ultimately show ?thesis
                      by (meson conga3_suma__suma suma2__conga)
                  qed
                  moreover have "A' B' C' G H I SumA K L M"
                    by (simp add: assms(5))
                  ultimately show ?thesis
                    using suma2__conga by auto
                qed
                ultimately show ?thesis
                proof -
                  have "A B C CongA A B C \<and> D' E' F' CongA D' E' F'"
                    using CongA_def \<open>A B J CongA K L M\<close> \<open>C B J CongA D' E' F'\<close> 
                      conga_refl by presburger
                  thus ?thesis
                    using \<open>A B C D' E' F' SumA A B J\<close> \<open>A B J CongA K L M\<close> 
                      conga3_suma__suma by blast
                qed
              qed
            qed
          next
            assume "\<not> Bet A' B' C'"
            have "B Out A C"
            proof -
              have "A B C LeA A' B' C'" 
                using assms(1) assms(3) sams_suma__lea123789 by auto
              moreover have "B' Out A' C'"
                using \<open>Col A' B' C'\<close> \<open>\<not> Bet A' B' C'\<close> or_bet_out by blast
              ultimately show ?thesis
                using out_lea__out by blast
            qed
            hence "Col A B C"
              using Col_perm out_col by blast
            hence "False"
              using \<open>\<not> Col A B C\<close> by auto
            thus ?thesis by simp
          qed
        next
          assume "\<not> Col A' B' C'"
          obtain C1 where P6: "C B C1 CongA D E F \<and> \<not> B C OS A C1 \<and> 
\<not> A B TS C C1 \<and> Coplanar A B C C1"
            using SAMS_def assms(1) by auto
          have P6A: "C B C1 CongA D E F"
            using P6 by simp
          have P6B: "\<not> B C OS A C1"
            using P6 by simp
          have P6C: "\<not> A B TS C C1"
            using P6 by simp
          have P6D: "Coplanar A B C C1"
            using P6 by simp
          have "A' B' C' CongA A B C1"
          proof -
            have "A B C D E F SumA A B C1"
              using P6A P6B P6D SumA_def \<open>\<not> Col A B C\<close> conga_distinct conga_refl 
                not_col_distincts by auto
            moreover have "A B C D E F SumA A' B' C'"
              by (simp add: assms(3))
            ultimately show ?thesis
              using suma2__conga by auto
          qed
          have "B C1 OS C A"
          proof -
            have "B A OS C C1"
            proof -
              have "A B OS C C1"
              proof -
                have "\<not> Col C A B"
                  using Col_perm \<open>\<not> Col A B C\<close> by blast
                moreover have "\<not> Col C1 A B"
                  using \<open>\<not> Col A' B' C'\<close> \<open>A' B' C' CongA A B C1\<close> col_permutation_1 
                    ncol_conga_ncol by blast
                ultimately show ?thesis
                  using P6C P6D cop_nos__ts by blast
              qed
              thus ?thesis
                by (simp add: invert_one_side)
            qed
            moreover have "B C TS A C1"
            proof -
              have "Coplanar B C A C1"
                using P6D ncoplanar_perm_12 by blast
              moreover have "\<not> Col C1 B C"
              proof -
                have "D E F CongA C1 B C"
                  using P6A conga_left_comm not_conga_sym by blast
                thus ?thesis
                  using  \<open>\<not> Col D E F\<close> ncol_conga_ncol by blast
              qed
              ultimately show ?thesis
                using T1 P6B cop_nos__ts by blast
            qed
            ultimately show ?thesis
              using os_ts1324__os one_side_symmetry by blast
          qed
          show ?thesis
          proof cases
            assume "Col D' E' F'"
            show ?thesis
            proof cases
              assume "Bet D' E' F'"
              obtain C0 where P7: "Bet C B C0 \<and> Cong C B B C0"
                using Cong_perm segment_construction by blast
              have "B C1 TS C C0"
                by (metis P7 \<open>B C1 OS C A\<close> bet__ts cong_diff_2 not_col_distincts 
                    one_side_not_col123)
              show ?thesis
              proof -
                have "A B C C B C0 SumA A B C0"
                proof -
                  have "C B C0 CongA C B C0"
                    by (metis P7 T1 cong_diff conga_line not_col_distincts)
                  moreover have "\<not> B C OS A C0"
                    using P7 bet_col col124__nos invert_one_side by blast
                  moreover have "Coplanar A B C C0"
                    using P7 bet__coplanar ncoplanar_perm_15 by blast
                  moreover have "A B C0 CongA A B C0"
                    by (metis P7 T1 cong_diff conga_refl not_col_distincts)
                  ultimately show ?thesis
                    using SumA_def by blast
                qed
                moreover have "A B C0 CongA K L M"
                proof -
                  have "A' B' C' G H I SumA A B C0"
                  proof -
                    have "A B C1 C1 B C0 SumA A B C0"
                      using \<open>B C1 TS C C0\<close> \<open>B C1 OS C A\<close> l9_8_2 ts__suma_1 by blast
                    moreover have "A B C1 CongA A' B' C'"
                      by (simp add: P6 \<open>A' B' C' CongA A B C1\<close> conga_sym)
                    moreover have "C1 B C0 CongA G H I"
                    proof -
                      have "SAMS C B C1 C1 B C0"
                        by (metis P7 \<open>B C1 TS C C0\<close> bet__sams ts_distincts)
                      moreover have "SAMS C B C1 G H I"
                      proof -
                        have "D E F CongA C B C1"
                          using P6A not_conga_sym by blast
                        moreover have "G H I CongA G H I"
                          using \<open>\<not> Col G H I\<close> conga_refl not_col_distincts by fastforce
                        moreover have "SAMS D E F G H I"
                          by (simp add: assms(2))
                        ultimately show ?thesis
                          using conga2_sams__sams by blast
                      qed
                      moreover have "C B C1 C1 B C0 SumA C B C0"
                        by (simp add: \<open>B C1 TS C C0\<close> ts__suma_1)
                      moreover have "C B C1 G H I SumA C B C0"
                      proof -
                        have "D E F G H I SumA D' E' F'"
                          by (simp add: assms(4))
                        moreover have "D E F CongA C B C1"
                          using P6A not_conga_sym by blast
                        moreover have "G H I CongA G H I"
                          using \<open>\<not> Col G H I\<close> conga_refl not_col_distincts by fastforce
                        moreover have "D' E' F' CongA C B C0" using P7 assms(4)
                          by (metis P6A suma_distincts \<open>Bet D' E' F'\<close> 
                              cong_diff conga_diff1 conga_line)
                        ultimately show ?thesis
                          using conga3_suma__suma by blast
                      qed
                      ultimately show ?thesis
                        using sams2_suma2__conga456 by auto
                    qed
                    moreover have "A B C0 CongA A B C0"
                      by (metis P7 T1 cong_diff conga_refl not_col_distincts)
                    ultimately show ?thesis
                      using conga3_suma__suma by blast
                  qed
                  thus ?thesis
                    using assms(5) suma2__conga by auto
                qed
                moreover have "A B C CongA A B C"
                proof -
                  have "A \<noteq> B \<and> B \<noteq> C"
                    using T1 col_trivial_1 col_trivial_2 by auto
                  thus ?thesis
                    using conga_refl by auto
                qed
                moreover have "C B C0 CongA D' E' F'"
                proof -
                  have "C \<noteq> B"
                    using T1 col_trivial_2 by blast
                  moreover have "B \<noteq> C0"
                    using \<open>B C1 TS C C0\<close> ts_distincts by blast
                  moreover have "D' \<noteq> E'"
                    using assms(4) suma_distincts by blast
                  moreover have "E' \<noteq> F'"
                    using assms(4) suma_distincts by auto
                  ultimately show ?thesis
                    by (simp add: P7 \<open>Bet D' E' F'\<close> conga_line)
                qed
                ultimately show ?thesis
                  using conga3_suma__suma by blast
              qed
            next
              assume "\<not> Bet D' E' F'"
              hence "E' Out D' F'"
                by (simp add: \<open>Col D' E' F'\<close> l6_4_2)
              have "D E F LeA D' E' F'"
                using sams_suma__lea123789 assms(2) assms(4) by auto
              hence "E Out D F"
                using \<open>E' Out D' F'\<close> out_lea__out by blast
              hence "False"
                using Col_cases \<open>\<not> Col D E F\<close> out_col by blast
              thus ?thesis by simp
            qed
          next
            assume "\<not> Col D' E' F'"
            have "SAMS C B C1 G H I"
            proof -
              have "D E F CongA C B C1"
                using P6A not_conga_sym by blast
              moreover have "G H I CongA G H I"
                using \<open>\<not> Col G H I\<close> conga_refl not_col_distincts by fastforce
              ultimately show ?thesis
                using assms(2) conga2_sams__sams by blast
            qed
            then obtain J where P7: "C1 B J CongA G H I \<and> \<not> B C1 OS C J \<and> 
\<not> C B TS C1 J \<and> Coplanar C B C1 J"
              using SAMS_def by blast
            have P7A: "C1 B J CongA G H I"
              using P7 by simp
            have P7B: "\<not> B C1 OS C J"
              using P7 by simp
            have P7C: "\<not> C B TS C1 J"
              using P7 by simp
            have P7D: "Coplanar C B C1 J"
              using P7 by simp
            obtain F1 where P8: "F E F1 CongA G H I \<and> \<not> E F OS D F1 \<and> 
\<not> D E TS F F1 \<and> Coplanar D E F F1"
              using SAMS_def assms(2) by auto
            have P8A: "F E F1 CongA G H I"
              using P8 by simp
            have P8B: "\<not> E F OS D F1"
              using P8 by simp
            have P8C: "\<not> D E TS F F1"
              using P8 by simp
            have P8D: "Coplanar D E F F1"
              using P8 by simp
            have "C B J CongA D' E' F'"
            proof -
              have "C B J CongA D E F1"
              proof -
                have "B C1 TS C J"
                proof -
                  have "Coplanar B C1 C J"
                    using P7D ncoplanar_perm_12 by blast
                  moreover have "\<not> Col C B C1"
                    using \<open>B C1 OS C A\<close> not_col_permutation_2 
                      one_side_not_col123 by blast
                  moreover have "\<not> Col J B C1"
                    using P7 \<open>\<not> Col G H I\<close> col_conga_col not_col_permutation_3 by blast
                  moreover have "\<not> B C1 OS C J"
                    by (simp add: P7B)
                  ultimately show ?thesis
                    by (simp add: cop_nos__ts)
                qed
                moreover have "E F TS D F1"
                proof -
                  have "Coplanar E F D F1"
                    using P8D ncoplanar_perm_12 by blast
                  moreover have "\<not> Col F1 E F"
                    using P8 \<open>\<not> Col G H I\<close> col_conga_col 
                      not_col_permutation_3 by blast
                  ultimately show ?thesis
                    using P8B \<open>\<not> Col D E F\<close> cop_nos__ts by blast
                qed
                moreover have "C B C1 CongA D E F"
                  using P6A by blast
                moreover have "C1 B J CongA F E F1"
                  using P8 by (meson P7A not_conga not_conga_sym)
                ultimately show ?thesis
                  using l11_22a by blast
              qed
              moreover have "D E F1 CongA D' E' F'"
              proof -
                have "D E F G H I SumA D E F1"
                  using P8A P8B P8D SumA_def \<open>\<not> Col D E F\<close> conga_distinct 
                    conga_refl not_col_distincts by auto
                moreover have "D E F G H I SumA D' E' F'"
                  by (simp add: assms(4))
                ultimately show ?thesis
                  using suma2__conga by auto
              qed
              ultimately show ?thesis
                using conga_trans by blast
            qed
            have "\<not> Col C B C1"
              using \<open>B C1 OS C A\<close> col123__nos col_permutation_1 by blast
            show ?thesis
            proof -
              have "A B C C B J SumA A B J"
              proof -
                have "B C TS J A"
                proof -
                  have "B C TS C1 A" using cop_nos__ts
                    using P6B P6D T1 \<open>\<not> Col C B C1\<close> l9_2 ncoplanar_perm_12 
                      not_col_permutation_3 by blast
                  moreover have "B C OS C1 J"
                  proof -
                    have "\<not> Col C1 C B"
                      using Col_perm \<open>\<not> Col C B C1\<close> by blast
                    moreover have "\<not> Col J C B"
                      using \<open>C B J CongA D' E' F'\<close> \<open>\<not> Col D' E' F'\<close> col_conga_col 
                        col_permutation_1 by blast
                    ultimately show ?thesis
                      using P7C P7D cop_nos__ts invert_one_side by blast
                  qed
                  ultimately show ?thesis
                    using l9_8_2 by blast
                qed
                thus ?thesis
                  by (simp add: l9_2 ts__suma_1)
              qed
              moreover have "A B C CongA A B C"
                using T1 conga_refl not_col_distincts by fastforce
              moreover have "A B J CongA K L M"
              proof -
                have "A' B' C' G H I SumA A B J"
                proof -
                  have "A B C1 C1 B J SumA A B J"
                  proof -
                    have "B C1 TS A J"
                    proof -
                      have "B C1 TS C J"
                      proof -
                        have "Coplanar B C1 C J"
                          using P7D ncoplanar_perm_12 by blast
                        moreover have "\<not> Col J B C1"
                          using P7 \<open>\<not> Col G H I\<close> col_conga_col 
                            not_col_permutation_3 by blast
                        ultimately show ?thesis
                          by (simp add: \<open>\<not> Col C B C1\<close> P7B cop_nos__ts)
                      qed
                      moreover have "B C1 OS C A"
                        by (simp add: \<open>B C1 OS C A\<close>)
                      ultimately show ?thesis
                        using l9_8_2 by blast
                    qed
                    thus ?thesis
                      by (simp add: ts__suma_1)
                  qed
                  moreover have "A B C1 CongA A' B' C'"
                    using \<open>A' B' C' CongA A B C1\<close> not_conga_sym by blast
                  moreover have "C1 B J CongA G H I"
                    by (simp add: P7A)
                  moreover have "A B J CongA A B J"
                    using \<open>A B C C B J SumA A B J\<close> suma2__conga by auto
                  ultimately show ?thesis
                    using conga3_suma__suma by blast
                qed
                moreover have "A' B' C' G H I SumA K L M"
                  using assms(5) by simp
                ultimately show ?thesis
                  using suma2__conga by auto
              qed
              ultimately show ?thesis
                using  \<open>C B J CongA D' E' F'\<close> conga3_suma__suma by blast
            qed
          qed
        qed
      qed
    qed
  qed
qed

lemma suma_assoc_2:
  assumes "SAMS A B C D E F" and
    "SAMS D E F G H I" and
    "A B C D E F SumA A' B' C'" and
    "D E F G H I SumA D' E' F'" and
    "A B C D' E' F' SumA K L M"
  shows "A' B' C' G H I SumA K L M"
  by (meson assms(1) assms(2) assms(3) assms(4) assms(5) sams_sym suma_assoc_1 suma_sym)

lemma suma_assoc:
  assumes "SAMS A B C D E F" and
    "SAMS D E F G H I" and
    "A B C D E F SumA A' B' C'" and
    "D E F G H I SumA D' E' F'"
  shows
    "A' B' C' G H I SumA K L M \<longleftrightarrow> A B C D' E' F' SumA K L M"
  by (meson assms(1) assms(2) assms(3) assms(4) suma_assoc_1 suma_assoc_2)

lemma sams_assoc_1:
  assumes "SAMS A B C D E F" and
    "SAMS D E F G H I" and
    "A B C D E F SumA A' B' C'" and
    "D E F G H I SumA D' E' F'" and
    "SAMS A' B' C' G H I"
  shows "SAMS A B C D' E' F'"
proof cases
  assume "Col A B C"
  {
    assume "E Out D F"
    have "SAMS A B C D' E' F'"
    proof -
      have "A' B' C' CongA A B C"
        using assms(3) \<open>E Out D F\<close> conga_sym out546_suma__conga by blast
      moreover have "G H I CongA D' E' F'"
        using \<open>E Out D F\<close> assms(4) out213_suma__conga by blast
      ultimately show ?thesis
        using assms(5) conga2_sams__sams by blast
    qed
  }
  {
    assume "\<not> Bet A B C"
    hence P1: "B Out A C"
      using \<open>Col A B C\<close> or_bet_out by blast
    have "SAMS D' E' F' A B C"
    proof -
      have "D' \<noteq> E'"
        using assms(4) suma_distincts by auto
      moreover have "F' E' F' CongA A B C"
      proof -
        have "E' \<noteq> F'"
          using assms(4) suma_distincts by blast
        hence "E' Out F' F'"
          using out_trivial by auto
        thus ?thesis
          using P1 l11_21_b by blast
      qed
      moreover have "\<not> E' F' OS D' F'"
        using os_distincts by blast
      moreover have "\<not> D' E' TS F' F'"
        by (simp add: not_two_sides_id)
      moreover have "Coplanar D' E' F' F'"
        using ncop_distincts by blast
      ultimately show ?thesis using SAMS_def P1 by blast
    qed
    hence "SAMS A B C D' E' F'"
      using sams_sym by blast
  }
  thus ?thesis
    using SAMS_def assms(1) \<open>E Out D F \<Longrightarrow> SAMS A B C D' E' F'\<close> by blast
next
  assume "\<not> Col A B C"
  show ?thesis
  proof cases
    assume "Col D E F"
    have "H Out G I \<or> \<not> Bet D E F"
      using SAMS_def assms(2) by blast
    {
      assume "H Out G I"
      have "SAMS A B C D' E' F'"
      proof -
        have "A B C CongA A B C"
          using \<open>\<not> Col A B C\<close> conga_refl not_col_distincts by fastforce
        moreover have "D E F CongA D' E' F'"
          using \<open>H Out G I\<close> assms(4) out546_suma__conga by blast
        ultimately show ?thesis
          using assms(1) conga2_sams__sams by blast
      qed
    }
    {
      assume "\<not> Bet D E F"
      hence "E Out D F"
        using \<open>Col D E F\<close> l6_4_2 by blast
      have "SAMS A B C D' E' F'"
      proof -
        have "A' B' C' CongA A B C"
          using out546_suma__conga \<open>E Out D F\<close> assms(3) not_conga_sym by blast
        moreover have "G H I CongA D' E' F'"
          using out213_suma__conga \<open>E Out D F\<close> assms(4) by auto
        ultimately show ?thesis
          using assms(5) conga2_sams__sams by auto
      qed
    }
    thus ?thesis
      using \<open>H Out G I \<Longrightarrow> SAMS A B C D' E' F'\<close> \<open>H Out G I \<or> \<not> Bet D E F\<close> by blast
  next
    assume "\<not> Col D E F"
    show ?thesis
    proof cases
      assume "Col G H I"
      have "SAMS G H I D E F"
        by (simp add: assms(2) sams_sym)
      hence "E Out D F \<or> \<not> Bet G H I"
        using SAMS_def by blast
      {
        assume "E Out D F"
        hence "False"
          using Col_cases \<open>\<not> Col D E F\<close> out_col by blast
        hence "SAMS A B C D' E' F'"
          by simp
      }
      {
        assume "\<not> Bet G H I"
        hence "H Out G I"
          by (simp add: \<open>Col G H I\<close> l6_4_2)
        have "SAMS A B C D' E' F'"
        proof -
          have "A B C CongA A B C"
            by (metis \<open>\<not> Col A B C\<close> col_trivial_1 col_trivial_2 conga_refl)
          moreover have "D E F CongA D' E' F'"
            using out546_suma__conga \<open>H Out G I\<close> assms(4) by blast
          moreover have "SAMS A B C D E F"
            using assms(1) by auto
          ultimately show ?thesis
            using conga2_sams__sams by auto
        qed
      }
      thus ?thesis
        using \<open>E Out D F \<or> \<not> Bet G H I\<close> \<open>E Out D F \<Longrightarrow> SAMS A B C D' E' F'\<close> by blast
    next
      assume "\<not> Col G H I"
      show ?thesis
      proof -
        have "\<not> Bet A B C"
          using Col_def \<open>\<not> Col A B C\<close> by auto
        moreover have "\<exists> J. (C B J CongA D' E' F' \<and> \<not> B C OS A J \<and> 
\<not> A B TS C J \<and> Coplanar A B C J)"
        proof -
          have "\<not> Col A' B' C'"
          proof -
            {
              assume "Col A' B' C'"
              have "H Out G I \<or> \<not> Bet A' B' C'"
                using SAMS_def assms(5) by blast
              {
                assume "H Out G I"
                hence "False"
                  using Col_cases \<open>\<not> Col G H I\<close> out_col by blast
              }
              {
                assume "\<not> Bet A' B' C'"
                hence "B' Out A' C'"
                  using not_bet_out \<open>Col A' B' C'\<close> by blast
                have "A B C LeA A' B' C'"
                  using assms(1) assms(3) sams_suma__lea123789 by auto
                hence "B Out A C"
                  using \<open>B' Out A' C'\<close> out_lea__out by blast
                hence "Col A B C"
                  using Col_perm out_col by blast
                hence "False"
                  using \<open>\<not> Col A B C\<close> by auto
              }
              hence "False"
                using \<open>H Out G I \<Longrightarrow> False\<close> \<open>H Out G I \<or> \<not> Bet A' B' C'\<close> by blast
            }
            thus ?thesis by blast
          qed
          obtain C1 where P1: "C B C1 CongA D E F \<and> \<not> B C OS A C1 \<and> 
\<not> A B TS C C1 \<and> Coplanar A B C C1"
            using SAMS_def assms(1) by auto
          have P1A: "C B C1 CongA D E F"
            using P1 by simp
          have P1B: "\<not> B C OS A C1"
            using P1 by simp
          have P1C: "\<not> A B TS C C1"
            using P1 by simp
          have P1D: "Coplanar A B C C1"
            using P1 by simp
          have "A B C1 CongA A' B' C'"
          proof -
            have "A B C D E F SumA A B C1"
              using P1A P1B P1D SumA_def \<open>\<not> Col A B C\<close> conga_distinct 
                conga_refl not_col_distincts by auto
            thus ?thesis
              using assms(3) suma2__conga by auto
          qed
          have "SAMS C B C1 G H I"
          proof -
            have "D E F CongA C B C1"
              using P1A not_conga_sym by blast
            moreover have "G H I CongA G H I"
              using \<open>\<not> Col G H I\<close> conga_refl not_col_distincts by fastforce
            ultimately show ?thesis using conga2_sams__sams
              using assms(2) by blast
          qed
          then obtain J where T1: "C1 B J CongA G H I \<and> \<not> B C1 OS C J \<and>
\<not> C B TS C1 J \<and> Coplanar C B C1 J"
            using SAMS_def by auto
          have T1A: "C1 B J CongA G H I"
            using T1 by simp
          have T1B: "\<not> B C1 OS C J"
            using T1 by simp
          have T1C: "\<not> C B TS C1 J"
            using T1 by simp
          have T1D: "Coplanar C B C1 J"
            using T1 by simp
          have "SAMS A B C1 C1 B J"
          proof -
            have "A' B' C' CongA A B C1"
              using \<open>A B C1 CongA A' B' C'\<close> not_conga_sym by blast
            moreover have "G H I CongA C1 B J"
              using T1A not_conga_sym by blast
            ultimately show ?thesis
              using assms(5) conga2_sams__sams by auto
          qed
          then obtain J' where T2: "C1 B J' CongA C1 B J \<and> \<not> B C1 OS A J' \<and> 
\<not> A B TS C1 J' \<and> Coplanar A B C1 J'"
            using SAMS_def by auto
          have T2A: "C1 B J' CongA C1 B J"
            using T2 by simp
          have T2B: "\<not> B C1 OS A J'"
            using T2 by simp
          have T2C: "\<not> A B TS C1 J'"
            using T2 by simp
          have T2D: "Coplanar A B C1 J'"
            using T2 by simp
          have "A' B' C' CongA A B C1"
            using \<open>A B C1 CongA A' B' C'\<close> not_conga_sym by blast
          hence "\<not> Col A B C1"
            using ncol_conga_ncol \<open>\<not> Col A' B' C'\<close> by blast
          have "D E F CongA C B C1"
            using P1A not_conga_sym by blast
          hence "\<not> Col C B C1"
            using ncol_conga_ncol \<open>\<not> Col D E F\<close> by blast
          hence "Coplanar C1 B A J"
            using coplanar_trans_1 P1D T1D coplanar_perm_15 
              coplanar_perm_6 by blast
          have "Coplanar C1 B J' J"
            using T2D \<open>Coplanar C1 B A J\<close> \<open>\<not> Col A B C1\<close> coplanar_perm_14 
              coplanar_perm_6 coplanar_trans_1 by blast
          have "B Out J' J \<or> C1 B TS J' J"
            by (meson T2 \<open>Coplanar C1 B A J\<close> \<open>\<not> Col A B C1\<close> conga_cop__or_out_ts 
                coplanar_trans_1 ncoplanar_perm_14 ncoplanar_perm_6)
          {
            assume "B Out J' J"
            have "\<exists> J. (C B J CongA D' E' F' \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J)"
            proof -
              have "C B C1 C1 B J SumA C B J"
              proof -
                have "C1 B J CongA C1 B J"
                  using CongA_def T2A conga_refl by auto
                moreover have "C B J CongA C B J"
                  using \<open>\<not> Col C B C1\<close> calculation conga_diff56 
                    conga_pseudo_refl conga_right_comm not_col_distincts by blast
                ultimately show ?thesis
                  using T1D T1B SumA_def by blast
              qed
              hence "D E F G H I SumA C B J"
                using conga3_suma__suma by (meson P1A T1A suma2__conga)
              hence "C B J CongA D' E' F'"
                using assms(4) suma2__conga by auto
              moreover have "\<not> B C OS A J"
                by (metis Col_perm P1B P1D T1C \<open>\<not> Col A B C\<close> 
                    \<open>\<not> Col C B C1\<close> cop_nos__ts coplanar_perm_8 invert_two_sides 
                    l9_2 l9_8_2)
              moreover have "\<not> A B TS C J"
              proof cases
                assume "Col A B J"
                thus ?thesis
                  using TS_def invert_two_sides not_col_permutation_3 by blast
              next
                assume "\<not> Col A B J"
                have "A B OS C J"
                proof -
                  have "A B OS C C1"
                    by (simp add: P1C P1D \<open>\<not> Col A B C1\<close> \<open>\<not> Col A B C\<close> cop_nts__os 
                        not_col_permutation_2)
                  moreover have "A B OS C1 J"
                  proof -
                    have "A B OS C1 J'"
                      by (metis T2C T2D \<open>B Out J' J\<close> \<open>\<not> Col A B C1\<close> \<open>\<not> Col A B J\<close> 
                          col_trivial_2 colx cop_nts__os not_col_permutation_2 
                          out_col out_distinct)
                    thus ?thesis
                      using \<open>B Out J' J\<close> invert_one_side out_out_one_side by blast
                  qed
                  ultimately show ?thesis
                    using one_side_transitivity by blast
                qed
                thus ?thesis
                  using l9_9 by blast
              qed
              moreover have "Coplanar A B C J"
                by (meson P1D \<open>Coplanar C1 B A J\<close> \<open>\<not> Col A B C1\<close> coplanar_perm_18 
                    coplanar_perm_2 coplanar_trans_1 not_col_permutation_2)
              ultimately show ?thesis
                by blast
            qed
          }
          {
            assume "C1 B TS J' J"
            have "B C1 OS C J"
            proof -
              have "B C1 TS C J'"
              proof -
                have "B C1 TS A J'"
                  by (meson T2B T2D TS_def \<open>C1 B TS J' J\<close> \<open>\<not> Col A B C1\<close> 
                      cop_nts__os invert_two_sides ncoplanar_perm_12)
                moreover have "B C1 OS A C"
                  by (meson P1B P1C P1D \<open>\<not> Col A B C1\<close> \<open>\<not> Col A B C\<close> \<open>\<not> Col C B C1\<close> 
                      cop_nts__os invert_one_side ncoplanar_perm_12 
                      not_col_permutation_2 not_col_permutation_3 os_ts1324__os)
                ultimately show ?thesis
                  using l9_8_2 by blast
              qed
              moreover have "B C1 TS J J'"
                using \<open>C1 B TS J' J\<close> invert_two_sides l9_2 by blast
              ultimately show ?thesis
                using OS_def by blast
            qed
            hence "False"
              by (simp add: T1B)
            hence "\<exists> J. (C B J CongA D' E' F' \<and> \<not> B C OS A J \<and> \<not> A B TS C J \<and> Coplanar A B C J)"
              by auto
          }
          thus ?thesis
            using \<open>B Out J' J \<Longrightarrow> \<exists>J. C B J CongA D' E' F' \<and> \<not> B C OS A J \<and> 
  \<not> A B TS C J \<and> Coplanar A B C J\<close> 
              \<open>B Out J' J \<or> C1 B TS J' J\<close> by blast
        qed
        ultimately show ?thesis
          using SAMS_def not_bet_distincts by auto
      qed
    qed
  qed
qed

lemma sams_assoc_2:
  assumes "SAMS A B C D E F" and
    "SAMS D E F G H I" and
    "A B C D E F SumA A' B' C'" and
    "D E F G H I SumA D' E' F'" and
    "SAMS A B C D' E' F'"
  shows "SAMS A' B' C' G H I"
proof -
  have "SAMS G H I A' B' C'"
  proof -
    have "SAMS G H I D E F"
      by (simp add: assms(2) sams_sym)
    moreover have "SAMS D E F A B C"
      by (simp add: assms(1) sams_sym)
    moreover have "G H I D E F SumA D' E' F'"
      by (simp add: assms(4) suma_sym)
    moreover have "D E F A B C SumA A' B' C'"
      by (simp add: assms(3) suma_sym)
    moreover have "SAMS D' E' F' A B C"
      by (simp add: assms(5) sams_sym)
    ultimately show ?thesis
      using sams_assoc_1 by blast
  qed
  thus ?thesis
    using sams_sym by blast
qed

lemma sams_assoc:
  assumes "SAMS A B C D E F" and
    "SAMS D E F G H I" and
    "A B C D E F SumA A' B' C'" and
    "D E F G H I SumA D' E' F'"
  shows "(SAMS A' B' C' G H I) \<longleftrightarrow> (SAMS A B C D' E' F')"
  using sams_assoc_1 sams_assoc_2 by (meson assms(1) assms(2) assms(3) assms(4))

lemma sams_nos__nts:
  assumes "SAMS A B C C B J" and
    "\<not> B C OS A J"
  shows "\<not>  A B TS C J"
proof -
  obtain J' where P1: "C B J' CongA C B J \<and> \<not> B C OS A J' \<and> \<not> A B TS C J' \<and> Coplanar A B C J'"
    using SAMS_def assms(1) by blast
  have P1A: "C B J' CongA C B J"
    using P1 by simp
  have P1B: "\<not> B C OS A J'"
    using P1 by simp
  have P1C: "\<not> A B TS C J'"
    using P1 by simp
  have P1D: "Coplanar A B C J'"
    using P1 by simp
  have P2: "B Out C J \<or> \<not> Bet A B C"
    using SAMS_def assms(1) by blast
  {
    assume "A B TS C J"
    have "Coplanar C B J J'"
    proof -
      have "\<not> Col A C B"
        using TS_def \<open>A B TS C J\<close> not_col_permutation_4 by blast
      moreover have "Coplanar A C B J"
        using \<open>A B TS C J\<close> ncoplanar_perm_2 ts__coplanar by blast
      moreover have "Coplanar A C B J'"
        using P1D ncoplanar_perm_2 by blast
      ultimately show ?thesis
        using coplanar_trans_1 by blast
    qed
    have "B Out J J' \<or> C B TS J J'"
      by (metis P1 \<open>A B TS C J\<close> \<open>Coplanar C B J J'\<close> bet_conga__bet bet_out 
          col_conga_col col_two_sides_bet conga_distinct conga_os__out conga_sym cop_nts__os invert_two_sides l5_2 l6_6 not_col_permutation_3 not_col_permutation_4)
    {
      assume "B Out J J'"
      have "\<not> Col B A J \<or> \<not> Col B A J'"
        using TS_def \<open>A B TS C J\<close> not_col_permutation_3 by blast
      hence "A B OS C J'"
        by (metis \<open>B Out J J'\<close> Col_cases P1C P1D TS_def 
            \<open>A B TS C J\<close> col2__eq cop_nts__os l6_3_1 out_col)
      hence "A B TS C J'"
        by (meson \<open>A B TS C J\<close> \<open>B Out J J'\<close> l6_6 l9_2 not_col_distincts 
            out_two_sides_two_sides)
      hence "False"
        by (simp add: P1C)
    }
    {
      assume "C B TS J J'"
      hence "\<not> Col C A B \<and> \<not> Col J A B"
        using TS_def \<open>A B TS C J\<close> by blast
      hence "False"
        by (metis P1B P1D TS_def \<open>C B TS J J'\<close> assms(2) cop_nts__os 
            invert_two_sides l9_8_1 ncoplanar_perm_12 not_col_permutation_1)
    }
    hence "False"
      using \<open>B Out J J' \<Longrightarrow> False\<close> \<open>B Out J J' \<or> C B TS J J'\<close> by blast
  }
  thus ?thesis by auto
qed

lemma conga_sams_nos__nts:
  assumes "SAMS A B C D E F" and
    "C B J CongA D E F" and
    "\<not> B C OS A J"
  shows "\<not> A B TS C J"
proof -
  have "SAMS A B C C B J"
  proof -
    have "A B C CongA A B C"
      using assms(1) conga_refl sams_distincts by fastforce
    moreover have "D E F CongA C B J"
      using assms(2) not_conga_sym by blast
    ultimately show ?thesis
      using assms(1) conga2_sams__sams by auto
  qed
  thus ?thesis
    by (simp add: assms(3) sams_nos__nts)
qed

lemma sams_lea2_suma2__conga123:
  assumes "A B C LeA A' B' C'" and
    "D E F LeA D' E' F'" and
    "SAMS A' B' C' D' E' F'" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G H I"
  shows "A B C CongA A' B' C'"
proof -
  have "SAMS A B C D E F"
    using assms(1) assms(2) assms(3) sams_lea2__sams by blast
  moreover have "SAMS A' B' C' D E F"
    by (metis assms(2) assms(3) lea_refl sams_distincts sams_lea2__sams)
  moreover have "A' B' C' D E F SumA G H I"
  proof -
    obtain G' H' I' where P1: "A' B' C' D E F SumA G' H' I'"
      using calculation(2) ex_suma sams_distincts by blast
    show ?thesis
    proof -
      have "A' \<noteq> B' \<and> B' \<noteq> C'"
        using assms(1) lea_distincts by blast
      hence "A' B' C' CongA A' B' C'"
        using conga_refl by auto
      moreover
      have "D \<noteq> E \<and> E \<noteq> F"
        using \<open>SAMS A B C D E F\<close> sams_distincts by blast
      hence "D E F CongA D E F"
        using conga_refl by auto
      moreover have "G' H' I' CongA G H I"
      proof -
        have "G' H' I' LeA G H I"
          using P1 assms(2) assms(3) assms(5) sams_lea456_suma2__lea by blast
        moreover have "G H I LeA G' H' I'"
        proof -
          have "SAMS A' B' C' D E F"
            using \<open>SAMS A' B' C' D E F\<close> by auto
          thus ?thesis
            using P1 assms(1) assms(4) sams_lea123_suma2__lea by blast
        qed
        ultimately show ?thesis
          by (simp add: lea_asym)
      qed
      ultimately show ?thesis
        using P1 conga3_suma__suma by blast
    qed
  qed
  ultimately show ?thesis
    using assms(4) sams2_suma2__conga123 by blast
qed

lemma sams_lea2_suma2__conga456:
  assumes "A B C LeA A' B' C'" and
    "D E F LeA D' E' F'" and
    "SAMS A' B' C' D' E' F'" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G H I"
  shows "D E F CongA D' E' F'"
proof -
  have "SAMS D' E' F' A' B' C'"
    by (simp add: assms(3) sams_sym)
  moreover have "D E F A B C SumA G H I"
    by (simp add: assms(4) suma_sym)
  moreover have "D' E' F' A' B' C' SumA G H I"
    by (simp add: assms(5) suma_sym)
  ultimately show ?thesis
    using assms(1) assms(2) sams_lea2_suma2__conga123 by auto
qed

lemma sams_suma__out213:
  assumes "A B C D E F SumA D E F" and
    "SAMS A B C D E F"
  shows "B Out A C"
proof -
  have "E \<noteq> D"
    using assms(2) sams_distincts by blast
  hence "E Out D D"
    using out_trivial by auto
  moreover have "D E D CongA A B C"
  proof -
    have "D E D LeA A B C"
      using assms(1) lea121345 suma_distincts by auto
    moreover
    have "E \<noteq> D \<and> E \<noteq> F"
      using assms(2) sams_distincts by blast
    hence "D E F LeA D E F"
      using lea_refl by auto
    moreover have "D E D D E F SumA D E F"
    proof -
      have "\<not> E D OS D F"
        using os_distincts by auto
      moreover have "Coplanar D E D F"
        using ncop_distincts by auto
      ultimately show ?thesis
        using SumA_def \<open>D E F LeA D E F\<close> lea_asym by blast
    qed
    ultimately show ?thesis
      using assms(1) assms(2) sams_lea2_suma2__conga123 by auto
  qed
  ultimately show ?thesis
    using eq_conga_out by blast
qed

lemma sams_suma__out546:
  assumes "A B C D E F SumA A B C" and
    "SAMS A B C D E F"
  shows "E Out D F"
proof -
  have "D E F A B C SumA A B C"
    using assms(1) suma_sym by blast
  moreover have "SAMS D E F A B C"
    using assms(2) sams_sym by blast
  ultimately show ?thesis
    using sams_suma__out213 by blast
qed

lemma sams_lea_lta123_suma2__lta:
  assumes "A B C LtA A' B' C'" and
    "D E F LeA D' E' F'" and
    "SAMS A' B' C' D' E' F'" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "G H I LtA G' H' I'"
proof -
  have "G H I LeA G' H' I'"
  proof -
    have "A B C LeA A' B' C'"
      by (simp add: assms(1) lta__lea)
    thus ?thesis
      using assms(2) assms(3) assms(4) assms(5) sams_lea2_suma2__lea by blast
  qed
  moreover have "\<not> G H I CongA G' H' I'"
  proof -
    {
      assume "G H I CongA G' H' I'"
      have "A B C CongA A' B' C'"
      proof -
        have "A B C LeA A' B' C'"
          by (simp add: assms(1) lta__lea)
        moreover have "A' B' C' D' E' F' SumA G H I"
          by (meson \<open>G H I CongA G' H' I'\<close> assms(3) assms(5) conga3_suma__suma 
              conga_sym sams2_suma2__conga123 sams2_suma2__conga456)
        ultimately show ?thesis
          using assms(2) assms(3) assms(4) sams_lea2_suma2__conga123 by blast
      qed
      hence "False"
        using assms(1) lta_not_conga by auto
    }
    thus ?thesis
      by auto
  qed
  ultimately show ?thesis
    using LtA_def by blast
qed

lemma sams_lea_lta456_suma2__lta:
  assumes "A B C LeA A' B' C'" and
    "D E F LtA D' E' F'" and
    "SAMS A' B' C' D' E' F'" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "G H I LtA G' H' I'"
  using sams_lea_lta123_suma2__lta 
  by (meson assms(1) assms(2) assms(3) assms(4) assms(5) sams_sym suma_sym)

lemma sams_lta2_suma2__lta:
  assumes "A B C LtA A' B' C'" and
    "D E F LtA D' E' F'" and
    "SAMS A' B' C' D' E' F'" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "G H I LtA G' H' I'"
  using sams_lea_lta123_suma2__lta 
  by (meson LtA_def assms(1) assms(2) assms(3) assms(4) assms(5))

lemma sams_lea2_suma2__lea123:
  assumes "D' E' F' LeA D E F" and
    "G H I LeA G' H' I'" and
    "SAMS A B C D E F" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "A B C LeA A' B' C'"
proof cases
  assume "A' B' C' LtA A B C"
  hence "G' H' I' LtA G H I" using sams_lea_lta123_suma2__lta
    using assms(1) assms(3) assms(4) assms(5) by blast
  hence "\<not> G H I LeA G' H' I'"
    using lea__nlta by blast
  hence "False"
    using assms(2) by auto
  thus ?thesis by auto
next
  assume "\<not> A' B' C' LtA A B C"
  have "A' \<noteq> B' \<and> B' \<noteq> C' \<and> A \<noteq> B \<and> B \<noteq> C"
    using assms(4) assms(5) suma_distincts by auto
  thus ?thesis
    by (simp add: \<open>\<not> A' B' C' LtA A B C\<close> nlta__lea)
qed

lemma sams_lea2_suma2__lea456:
  assumes "A' B' C' LeA A B C" and
    "G H I LeA G' H' I'" and
    "SAMS A B C D E F" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "D E F LeA D' E' F'"
proof -
  have "SAMS D E F A B C"
    by (simp add: assms(3) sams_sym)
  moreover have "D E F A B C SumA G H I"
    by (simp add: assms(4) suma_sym)
  moreover have "D' E' F' A' B' C' SumA G' H' I'"
    by (simp add: assms(5) suma_sym)
  ultimately show ?thesis
    using assms(1) assms(2) sams_lea2_suma2__lea123 by blast
qed

lemma sams_lea_lta456_suma2__lta123:
  assumes "D' E' F' LtA D E F" and
    "G H I LeA G' H' I'" and
    "SAMS A B C D E F" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "A B C LtA A' B' C'"
proof cases
  assume "A' B' C' LeA A B C"
  hence "G' H' I' LtA G H I"
    using sams_lea_lta456_suma2__lta assms(1) assms(3) assms(4) assms(5) by blast
  hence "\<not> G H I LeA G' H' I'"
    using lea__nlta by blast
  hence "False"
    using assms(2) by blast
  thus ?thesis by blast
next
  assume "\<not> A' B' C' LeA A B C"
  have "A' \<noteq> B' \<and> B' \<noteq> C' \<and> A \<noteq> B \<and> B \<noteq> C"
    using assms(4) assms(5) suma_distincts by auto
  thus ?thesis using nlea__lta
    by (simp add: \<open>\<not> A' B' C' LeA A B C\<close>)
qed

lemma sams_lea_lta123_suma2__lta456:
  assumes "A' B' C' LtA A B C" and
    "G H I LeA G' H' I'" and
    "SAMS A B C D E F" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "D E F LtA D' E' F'"
proof -
  have "SAMS D E F A B C"
    by (simp add: assms(3) sams_sym)
  moreover have "D E F A B C SumA G H I"
    by (simp add: assms(4) suma_sym)
  moreover have "D' E' F' A' B' C' SumA G' H' I'"
    by (simp add: assms(5) suma_sym)
  ultimately show ?thesis
    using assms(1) assms(2) sams_lea_lta456_suma2__lta123 by blast
qed

lemma sams_lea_lta789_suma2__lta123:
  assumes "D' E' F' LeA D E F" and
    "G H I LtA G' H' I'" and
    "SAMS A B C D E F" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "A B C LtA A' B' C'"
proof cases
  assume "A' B' C' LeA A B C"
  hence "G' H' I' LeA G H I"
    using assms(1) assms(3) assms(4) assms(5) sams_lea2_suma2__lea by blast
  hence "\<not> G H I LtA G' H' I'"
    by (simp add: lea__nlta)
  hence "False"
    using assms(2) by blast
  thus ?thesis by auto
next
  assume "\<not> A' B' C' LeA A B C"
  have "A' \<noteq> B' \<and> B' \<noteq> C' \<and> A \<noteq> B \<and> B \<noteq> C"
    using assms(4) assms(5) suma_distincts by auto
  thus ?thesis
    using nlea__lta by (simp add: \<open>\<not> A' B' C' LeA A B C\<close>)
qed

lemma sams_lea_lta789_suma2__lta456:
  assumes "A' B' C' LeA A B C" and
    "G H I LtA G' H' I'" and
    "SAMS A B C D E F" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "D E F LtA D' E' F'"
proof -
  have "SAMS D E F A B C"
    by (simp add: assms(3) sams_sym)
  moreover have "D E F A B C SumA G H I"
    by (simp add: assms(4) suma_sym)
  moreover have "D' E' F' A' B' C' SumA G' H' I'"
    using assms(5) suma_sym by blast
  ultimately show ?thesis
    using assms(1) assms(2) sams_lea_lta789_suma2__lta123 by blast
qed

lemma sams_lta2_suma2__lta123:
  assumes "D' E' F' LtA D E F" and
    "G H I LtA G' H' I'" and
    "SAMS A B C D E F" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "A B C LtA A' B' C'"
proof -
  have "D' E' F' LeA D E F"
    by (simp add: assms(1) lta__lea)
  thus ?thesis
    using assms(2) assms(3) assms(4) assms(5) sams_lea_lta789_suma2__lta123 by blast
qed

lemma sams_lta2_suma2__lta456:
  assumes "A' B' C' LtA A B C" and
    "G H I LtA G' H' I'" and
    "SAMS A B C D E F" and
    "A B C D E F SumA G H I" and
    "A' B' C' D' E' F' SumA G' H' I'"
  shows "D E F LtA D' E' F'"
proof -
  have "A' B' C' LeA A B C"
    by (simp add: assms(1) lta__lea)
  thus ?thesis
    using assms(2) assms(3) assms(4) assms(5) sams_lea_lta789_suma2__lta456 by blast
qed

lemma sams123231:
  assumes "A \<noteq> B" and
    "A \<noteq> C" and
    "B \<noteq> C"
  shows "SAMS A B C B C A"
proof -
  obtain A' where "B Midpoint A A'"
    using symmetric_point_construction by auto
  hence "A' \<noteq> B"
    using assms(1) midpoint_not_midpoint by blast
  moreover have "Bet A B A'"
    by (simp add: \<open>B Midpoint A A'\<close> midpoint_bet)
  moreover have "B C A LeA C B A'"
  proof cases
    assume "Col A B C"
    show ?thesis
    proof cases
      assume "Bet A C B"
      thus ?thesis
        by (metis assms(2) assms(3) between_exchange3 calculation(1) 
            calculation(2) l11_31_2)
    next
      assume "\<not> Bet A C B"
      hence "C Out B A"
        using Col_cases \<open>Col A B C\<close> l6_6 or_bet_out by blast
      thus ?thesis
        using assms(3) calculation(1) l11_31_1 by auto
    qed
  next
    assume "\<not> Col A B C"
    thus ?thesis
      using l11_41_aux \<open>B Midpoint A A'\<close> calculation(1) lta__lea 
        midpoint_bet not_col_permutation_4 by blast
  qed
  ultimately show ?thesis
    using assms(1) sams_chara by blast
qed

lemma col_suma__col:
  assumes "Col D E F" and
    "A B C B C A SumA D E F"
  shows "Col A B C"
proof -
  {
    assume "\<not> Col A B C"
    have "False"
    proof cases
      assume "Bet D E F"
      obtain P where P1: "Bet A B P \<and> Cong A B B P"
        using Cong_perm segment_construction by blast
      have "D E F LtA A B P"
      proof -
        have "A B C LeA A B C"
          using \<open>\<not> Col A B C\<close> lea_total not_col_distincts by blast
        moreover
        have "B C TS A P"
          by (metis Cong_perm P1 \<open>\<not> Col A B C\<close> bet__ts bet_col between_trivial2 
              cong_reverse_identity not_col_permutation_1)
        hence "B C A LtA C B P"
          using Col_perm P1 \<open>\<not> Col A B C\<close> calculation l11_41_aux ts_distincts by blast
        moreover have "A B C C B P SumA A B P"
          by (simp add: \<open>B C TS A P\<close> ts__suma_1)
        ultimately show ?thesis
          by (meson P1 sams_lea_lta456_suma2__lta assms(2) bet_suma__sams)
      qed
      thus ?thesis
        using P1 \<open>Bet D E F\<close> bet2_lta__lta lta_distincts by blast
    next
      assume "\<not> Bet D E F"
      have "C Out B A"
      proof -
        have "E Out D F"
          by (simp add: \<open>\<not> Bet D E F\<close> assms(1) l6_4_2)
        moreover have "B C A LeA D E F"
          using sams_suma__lea456789
          by (metis assms(2) sams123231 suma_distincts)
        ultimately show ?thesis
          using out_lea__out by blast
      qed
      thus ?thesis
        using Col_cases \<open>\<not> Col A B C\<close> out_col by blast
    qed
  }
  thus ?thesis by auto
qed

lemma ncol_suma__ncol:
  assumes "\<not> Col A B C" and
    "A B C B C A SumA D E F"
  shows "\<not> Col D E F"
  using col_suma__col assms(1) assms(2) by blast

lemma per2_suma__bet:
  assumes "Per A B C" and
    "Per D E F" and
    "A B C D E F SumA G H I"
  shows "Bet G H I"
proof -
  obtain A1 where P1: "C B A1 CongA D E F \<and> \<not> B C OS A A1 \<and> 
Coplanar A B C A1 \<and> A B A1 CongA G H I"
    using SumA_def assms(3) by blast
  hence "D E F CongA A1 B C"
    using conga_right_comm conga_sym by blast
  hence "Per A1 B C"
    using assms(2) l11_17 by blast
  have "Bet A B A1"
  proof -
    have "Col B A A1"
    proof -
      have "Coplanar C A A1 B"
        using P1 ncoplanar_perm_10 by blast
      moreover have "C \<noteq> B"
        using \<open>D E F CongA A1 B C\<close> conga_distinct by auto
      ultimately show ?thesis
        using assms(1) \<open>Per A1 B C\<close> col_permutation_2 cop_per2__col by blast
    qed
    moreover have "B C TS A A1"
    proof -
      have "Coplanar B C A A1"
        using calculation ncop__ncols by blast
      moreover
      have "A \<noteq> B \<and> B \<noteq> C"
        using CongA_def P1 by blast
      hence "\<not> Col A B C"
        by (simp add: assms(1) per_not_col)
      moreover
      have "A1 \<noteq> B \<and> B \<noteq> C"
        using \<open>D E F CongA A1 B C\<close> conga_distinct by blast
      hence "\<not> Col A1 B C"
        using  \<open>Per A1 B C\<close> by (simp add: per_not_col)
      ultimately show ?thesis
        by (simp add: P1 cop_nos__ts)
    qed
    ultimately show ?thesis
      using col_two_sides_bet by blast
  qed
  thus ?thesis
    using P1 bet_conga__bet by blast
qed

lemma bet_per2__suma:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "D \<noteq> E" and
    "E \<noteq> F" and
    "G \<noteq> H" and
    "H \<noteq> I" and
    "Per A B C" and
    "Per D E F" and
    "Bet G H I"
  shows "A B C D E F SumA G H I"
proof -
  obtain G' H' I' where "A B C D E F SumA G' H' I'"
    using assms(1) assms(2) assms(3) assms(4) ex_suma by blast
  moreover have "A B C CongA A B C"
    using assms(1) assms(2) conga_refl by auto
  moreover have "D E F CongA D E F"
    using assms(3) assms(4) conga_refl by auto
  moreover have "G' H' I' CongA G H I"
  proof -
    have "G' \<noteq> H'"
      using calculation(1) suma_distincts by auto
    moreover have "H' \<noteq> I'"
      using \<open>A B C D E F SumA G' H' I'\<close> suma_distincts by blast
    moreover have "Bet G' H' I'"
      using  \<open>A B C D E F SumA G' H' I'\<close> assms(7) assms(8) per2_suma__bet by auto
    ultimately show ?thesis
      using conga_line by (simp add: assms(5) assms(6) assms(9))
  qed
  ultimately show ?thesis
    using conga3_suma__suma by blast
qed

lemma per2__sams:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "D \<noteq> E" and
    "E \<noteq> F" and
    "Per A B C" and
    "Per D E F"
  shows "SAMS A B C D E F"
proof -
  obtain G H I where "A B C D E F SumA G H I"
    using assms(1) assms(2) assms(3) assms(4) ex_suma by blast
  moreover hence "Bet G H I"
    using assms(5) assms(6) per2_suma__bet by auto
  ultimately show ?thesis
    using bet_suma__sams by blast
qed

lemma bet_per_suma__per456:
  assumes "Per A B C" and
    "Bet G H I" and
    "A B C D E F SumA G H I"
  shows "Per D E F"
proof -
  obtain A1 where "B Midpoint A A1"
    using symmetric_point_construction by auto
  have "\<not> Col A B C"
    using assms(1) assms(3) per_col_eq suma_distincts by blast
  have "A B C CongA D E F"
  proof -
    have "SAMS A B C A B C"
      using \<open>\<not> Col A B C\<close> assms(1) not_col_distincts per2__sams by auto
    moreover have "SAMS A B C D E F"
      using assms(2) assms(3) bet_suma__sams by blast
    moreover have "A B C A B C SumA G H I"
      using assms(1) assms(2) assms(3) bet_per2__suma suma_distincts by blast
    ultimately show ?thesis
      using assms(3) sams2_suma2__conga456 by auto
  qed
  thus ?thesis
    using assms(1) l11_17 by blast
qed

lemma bet_per_suma__per123:
  assumes "Per D E F" and
    "Bet G H I" and
    "A B C D E F SumA G H I"
  shows "Per A B C"
  using bet_per_suma__per456 by (meson assms(1) assms(2) assms(3) suma_sym)

lemma bet_suma__per:
  assumes "Bet D E F" and
    "A B C A B C SumA D E F"
  shows "Per A B C"
proof -
  obtain A' where "C B A' CongA A B C \<and> A B A' CongA D E F"
    using SumA_def assms(2) by blast
  have "Per C B A"
  proof -
    have "Bet A B A'"
    proof -
      have "D E F CongA A B A'"
        using \<open>C B A' CongA A B C \<and> A B A' CongA D E F\<close> not_conga_sym by blast
      thus ?thesis
        using assms(1) bet_conga__bet by blast
    qed
    moreover have "C B A CongA C B A'"
      using conga_left_comm not_conga_sym 
        \<open>C B A' CongA A B C \<and> A B A' CongA D E F\<close> by blast
    ultimately show ?thesis
      using l11_18_2 by auto
  qed
  thus ?thesis
    using Per_cases by auto
qed

lemma acute__sams:
  assumes "Acute A B C"
  shows "SAMS A B C A B C"
proof -
  obtain A' where "B Midpoint A A'"
    using symmetric_point_construction by auto
  show ?thesis
  proof -
    have "A \<noteq> B \<and> A' \<noteq> B"
      using \<open>B Midpoint A A'\<close> acute_distincts assms is_midpoint_id_2 by blast
    moreover have "Bet A B A'"
      by (simp add: \<open>B Midpoint A A'\<close> midpoint_bet)
    moreover
    have "Obtuse C B A'"
      using acute_bet__obtuse assms calculation(1) calculation(2) 
        obtuse_sym by blast
    hence "A B C LeA C B A'"
      by (metis acute__not_obtuse assms calculation(1) lea_obtuse_obtuse 
          lea_total obtuse_distincts)
    ultimately show ?thesis
      using sams_chara by blast
  qed
qed

lemma acute_suma__nbet:
  assumes "Acute A B C" and
    "A B C A B C SumA D E F"
  shows  "\<not> Bet D E F"
proof -
  {
    assume "Bet D E F"
    hence "Per A B C"
      using assms(2) bet_suma__per by auto
    hence "A B C LtA A B C"
      using acute_not_per assms(1) by auto
    hence "False"
      by (simp add: nlta)
  }
  thus ?thesis by blast
qed

lemma acute2__sams:
  assumes "Acute A B C" and
    "Acute D E F"
  shows "SAMS A B C D E F"
  by (metis acute__sams acute_distincts assms(1) assms(2) lea_total sams_lea2__sams)

lemma acute2_suma__nbet_a:
  assumes "Acute A B C" and
    "D E F LeA A B C" and
    "A B C D E F SumA G H I"
  shows "\<not> Bet G H I"
proof -
  {
    assume "Bet G H I"
    obtain A' B' C' where "A B C A B C SumA A' B' C'"
      using acute_distincts assms(1) ex_suma by metis 
    have "G H I LeA A' B' C'"
    proof -
      have "A B C LeA A B C"
        using acute_distincts assms(1) lea_refl by blast
      moreover have "SAMS A B C A B C"
        by (simp add: acute__sams assms(1))
      ultimately show ?thesis
        using \<open>A B C A B C SumA A' B' C'\<close> assms(1) assms(2) assms(3) 
          sams_lea456_suma2__lea by blast
    qed
    hence "Bet A' B' C'"
      using \<open>Bet G H I\<close> bet_lea__bet by blast
    hence "False"
      using acute_suma__nbet assms(1) assms(3) \<open>A B C A B C SumA A' B' C'\<close> by blast
  }
  thus ?thesis by blast
qed

lemma acute2_suma__nbet:
  assumes "Acute A B C" and
    "Acute D E F" and
    "A B C D E F SumA G H I"
  shows "\<not> Bet G H I"
proof -
  have "A \<noteq> B \<and> B \<noteq> C \<and> D \<noteq> E \<and> E \<noteq> F"
    using assms(3) suma_distincts by auto
  hence "A B C LeA D E F \<or> D E F LeA A B C"
    by (simp add: lea_total)
  moreover
  {
    assume P3: "A B C LeA D E F"
    have "D E F A B C SumA G H I"
      by (simp add: assms(3) suma_sym)
    hence "\<not> Bet G H I"
      using P3 assms(2) acute2_suma__nbet_a by auto
  }
  {
    assume "D E F LeA A B C"
    hence  "\<not> Bet G H I"
      using acute2_suma__nbet_a assms(1) assms(3) by auto
  }
  thus ?thesis
    using \<open>A B C LeA D E F \<Longrightarrow> \<not> Bet G H I\<close> calculation by blast
qed

lemma acute_per__sams:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Per A B C" and
    "Acute D E F"
  shows "SAMS A B C D E F"
proof -
  have "SAMS A B C A B C"
    by (simp add: assms(1) assms(2) assms(3) per2__sams)
  moreover have "A B C LeA A B C"
    using assms(1) assms(2) lea_refl by auto
  moreover have "D E F LeA A B C"
    by (metis acute_distincts acute_lea_acute acute_not_per assms(1) 
        assms(2) assms(3) assms(4) lea_total)
  ultimately show ?thesis
    using sams_lea2__sams by blast
qed

lemma acute_per_suma__nbet:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Per A B C" and
    "Acute D E F" and
    "A B C D E F SumA G H I"
  shows "\<not> Bet G H I"
proof -
  {
    assume "Bet G H I"
    have "G H I LtA G H I"
    proof -
      have "A B C LeA A B C"
        using assms(1) assms(2) lea_refl by auto
      moreover have "D E F LtA A B C"
        by (simp add: acute_per__lta assms(1) assms(2) assms(3) assms(4))
      moreover have "SAMS A B C A B C"
        by (simp add: assms(1) assms(2) assms(3) per2__sams)
      moreover have "A B C D E F SumA G H I"
        by (simp add: assms(5))
      moreover have "A B C A B C SumA G H I"
        by (meson bet_per_suma__per456 Tarski_neutral_dimensionless_axioms 
            \<open>Bet G H I\<close> acute_not_per assms(3) assms(4) calculation(4))
      ultimately show ?thesis
        using sams_lea_lta456_suma2__lta by blast
    qed
    hence "False"
      by (simp add: nlta)
  }
  thus ?thesis by blast
qed

lemma obtuse__nsams:
  assumes "Obtuse A B C"
  shows "\<not> SAMS A B C A B C"
proof -
  {
    assume "SAMS A B C A B C"
    obtain A' where "B Midpoint A A'"
      using symmetric_point_construction by auto
    have "A B C LtA A B C"
    proof -
      have "A B C LeA A' B C"
        by (metis \<open>B Midpoint A A'\<close> \<open>SAMS A B C A B C\<close> lea_right_comm 
            midpoint_bet midpoint_distinct_2 sams_chara sams_distincts)
      moreover have "A' B C LtA A B C"
        using \<open>B Midpoint A A'\<close> assms calculation lea_distincts 
          midpoint_bet obtuse_chara by blast
      ultimately show ?thesis
        using lea__nlta by blast
    qed
    hence "False"
      by (simp add: nlta)
  }
  thus ?thesis by blast
qed

lemma nbet_sams_suma__acute:
  assumes "\<not> Bet D E F" and
    "SAMS A B C A B C" and
    "A B C A B C SumA D E F"
  shows "Acute A B C"
proof -
  have "Acute A B C \<or> Per A B C \<or> Obtuse A B C"
    by (metis angle_partition l8_20_1_R1 l8_5)
  {
    assume "Per A B C"
    hence "Bet D E F"
      using assms(3) per2_suma__bet by auto
    hence "False"
      using assms(1) by auto
  }
  {
    assume "Obtuse A B C"
    hence "\<not> SAMS A B C A B C"
      by (simp add: obtuse__nsams)
    hence "False"
      using assms(2) by auto
  }
  thus ?thesis
    using \<open>Acute A B C \<or> Per A B C \<or> Obtuse A B C\<close> \<open>Per A B C \<Longrightarrow> False\<close> by auto
qed

lemma nsams__obtuse:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "\<not> SAMS A B C A B C"
  shows "Obtuse A B C"
proof -
  {
    assume "Per A B C"
    obtain A' where "B Midpoint A A'"
      using symmetric_point_construction by blast
    have "\<not> Col A B C"
      using \<open>Per A B C\<close> assms(1) assms(2) per_col_eq by blast
    have "SAMS A B C A B C"
    proof -
      have "C B A' CongA A B C"
        using \<open>Per A B C\<close> assms(1) assms(2) assms(3) per2__sams by blast
      moreover have "\<not> B C OS A A'"
        by (meson Col_cases \<open>B Midpoint A A'\<close> col_one_side_out l6_4_1
            midpoint_bet midpoint_col)
      moreover have "\<not> A B TS C A'"
        using Col_def Midpoint_def TS_def \<open>B Midpoint A A'\<close> by blast
      moreover have "Coplanar A B C A'"
        using \<open>Per A B C\<close> assms(1) assms(2) assms(3) per2__sams by blast
      ultimately show ?thesis
        using SAMS_def \<open>\<not> Col A B C\<close> assms(1) bet_col by auto
    qed
    hence "False"
      using assms(3) by auto
  }
  {
    assume "Acute A B C"
    hence "SAMS A B C A B C"
      by (simp add: acute__sams)
    hence "False"
      using assms(3) by auto
  }
  thus ?thesis
    using \<open>Per A B C \<Longrightarrow> False\<close> angle_partition assms(1) assms(2) by auto
qed

lemma sams2_suma2__conga:
  assumes "SAMS A B C A B C" and
    "A B C A B C SumA D E F" and
    "SAMS A' B' C' A' B' C'" and
    "A' B' C' A' B' C' SumA D E F"
  shows "A B C CongA A' B' C'"
proof -
  have "A B C LeA A' B' C' \<or> A' B' C' LeA A B C"
    using assms(1) assms(3) lea_total sams_distincts by auto
  moreover
  have "A B C LeA A' B' C' \<longrightarrow> A B C CongA A' B' C'"
    using assms(2) assms(3) assms(4) sams_lea2_suma2__conga123 by auto
  ultimately show ?thesis
    by (meson conga_sym sams_lea2_suma2__conga123 assms(1) assms(2) assms(4))
qed

lemma acute2_suma2__conga:
  assumes "Acute A B C" and
    "A B C A B C SumA D E F" and
    "Acute A' B' C'" and
    "A' B' C' A' B' C' SumA D E F"
  shows "A B C CongA A' B' C'"
proof -
  have "SAMS A B C A B C"
    by (simp add: acute__sams assms(1))
  moreover have "SAMS A' B' C' A' B' C'"
    by (simp add: acute__sams assms(3))
  ultimately show ?thesis
    using assms(2) assms(4) sams2_suma2__conga by auto
qed

lemma bet2_suma__out:
  assumes "Bet A B C" and
    "Bet D E F" and
    "A B C D E F SumA G H I"
  shows "H Out G I"
proof -
  have "A B C D E F SumA A B A"
  proof -
    have "C B A CongA D E F"
      by (metis Bet_cases assms(1) assms(2) assms(3) conga_line suma_distincts)
    moreover have "\<not> B C OS A A"
      by (simp add: Col_def assms(1) col124__nos)
    moreover have "Coplanar A B C A"
      using ncop_distincts by blast
    moreover have "A B A CongA A B A"
      using calculation(1) conga_diff2 conga_trivial_1 by auto
    ultimately show ?thesis
      using SumA_def by blast
  qed
  hence "A B A CongA G H I"
    using assms(3) suma2__conga by auto
  thus ?thesis
    using eq_conga_out by auto
qed

lemma col2_suma__col:
  assumes "Col A B C" and
    "Col D E F" and
    "A B C D E F SumA G H I"
  shows "Col G H I"
proof cases
  assume "Bet A B C"
  show ?thesis
  proof cases
    assume "Bet D E F"
    thus ?thesis using bet2_suma__out
      by (meson \<open>Bet A B C\<close> assms(3) not_col_permutation_4 out_col)
  next
    assume "\<not> Bet D E F"
    show ?thesis
    proof -
      have "E Out D F"
        using \<open>\<not> Bet D E F\<close> assms(2) or_bet_out by auto
      hence "A B C CongA G H I"
        using assms(3) out546_suma__conga by auto
      thus ?thesis
        using assms(1) col_conga_col by blast
    qed
  qed
next
  assume "\<not> Bet A B C"
  have "D E F CongA G H I"
  proof -
    have "B Out A C"
      by (simp add: \<open>\<not> Bet A B C\<close> assms(1) l6_4_2)
    thus ?thesis
      using assms(3) out213_suma__conga by auto
  qed
  thus ?thesis
    using assms(2) col_conga_col by blast
qed

lemma suma_suppa__bet:
  assumes "A B C SuppA D E F" and
    "A B C D E F SumA G H I"
  shows "Bet G H I"
proof -
  obtain A' where P1: "Bet A B A' \<and> D E F CongA C B A'"
    using SuppA_def assms(1) by auto
  obtain A'' where P2: "C B A'' CongA D E F \<and> \<not> B C OS A A'' \<and> 
Coplanar A B C A'' \<and> A B A'' CongA G H I"
    using SumA_def assms(2) by auto
  have "B Out A' A'' \<or> C B TS A' A''"
  proof -
    have "Coplanar C B A' A''"
    proof -
      have "Coplanar C A'' B A"
        using P2 coplanar_perm_17 by blast
      moreover have "B \<noteq> A"
        using SuppA_def assms(1) by auto
      moreover have "Col B A A'" using P1
        by (simp add: bet_col col_permutation_4)
      ultimately show ?thesis
        using col_cop__cop coplanar_perm_3 by blast
    qed
    moreover have "C B A' CongA C B A''"
    proof -
      have "C B A' CongA D E F"
        using P1 not_conga_sym by blast
      moreover have "D E F CongA C B A''"
        using P2 not_conga_sym by blast
      ultimately show ?thesis
        using not_conga by blast
    qed
    ultimately show ?thesis
      using conga_cop__or_out_ts by simp
  qed
  have "Bet A B A''"
  proof -
    have "\<not> C B TS A' A''"
    proof -
      {
        assume "C B TS A' A''"
        have "B C TS A A'"
        proof -
          {
            assume "Col A B C"
            hence "Col A' C B"
              using P1 assms(1) bet_col bet_col1 col3 suppa_distincts by blast
            hence "False"
              using TS_def \<open>C B TS A' A''\<close> by auto
          }
          hence "\<not> Col A B C" by auto
          moreover have "\<not> Col A' B C"
            using TS_def \<open>C B TS A' A''\<close> not_col_permutation_5 by blast
          moreover
          have "\<exists> T. (Col T B C \<and> Bet A T A')"
            using P1 not_col_distincts by blast
          ultimately show ?thesis
            by (simp add: TS_def)
        qed
        hence "B C OS A A''"
          using OS_def \<open>C B TS A' A''\<close> invert_two_sides l9_2 by blast
        hence "False"
          using P2 by simp
      }
      thus ?thesis by blast
    qed
    hence "B Out A' A''"
      using \<open>B Out A' A'' \<or> C B TS A' A''\<close> by auto
    moreover have "A' \<noteq> B \<and> A'' \<noteq> B \<and> A \<noteq> B"
      using P2 calculation conga_diff1 out_diff1 out_diff2 by blast
    moreover have "Bet A' B A"
      using P1 Bet_perm by blast
    ultimately show ?thesis
      using bet_out__bet between_symmetry by blast
  qed
  moreover have "A B A'' CongA G H I"
    using P2 by blast
  ultimately show ?thesis
    using bet_conga__bet by blast
qed

lemma bet_suppa__suma:
  assumes "G \<noteq> H" and
    "H \<noteq> I" and
    "A B C SuppA D E F" and
    "Bet G H I"
  shows "A B C D E F SumA G H I"
proof -
  obtain G' H' I' where "A B C D E F SumA G' H' I'"
    using assms(3) ex_suma suppa_distincts by blast
  moreover have "A B C CongA A B C"
    using calculation conga_refl suma_distincts by fastforce
  moreover
  have "D \<noteq> E \<and> E \<noteq> F"
    using assms(3) suppa_distincts by auto
  hence "D E F CongA D E F"
    using conga_refl by auto
  moreover
  have "G' H' I' CongA G H I"
  proof -
    have "G' \<noteq> H' \<and> H' \<noteq> I'"
      using calculation(1) suma_distincts by auto
    moreover have "Bet G' H' I'"
      using  \<open>A B C D E F SumA G' H' I'\<close> assms(3) suma_suppa__bet by blast
    ultimately show ?thesis
      using assms(1) assms(2) assms(4) conga_line by auto
  qed
  ultimately show ?thesis
    using conga3_suma__suma by blast
qed

lemma bet_suma__suppa:
  assumes "A B C D E F SumA G H I" and
    "Bet G H I"
  shows "A B C SuppA D E F"
proof -
  obtain A' where "C B A' CongA D E F \<and> A B A' CongA G H I"
    using SumA_def assms(1) by blast
  moreover
  have "G H I CongA A B A'"
    using calculation not_conga_sym by blast
  hence "Bet A B A'"
    using assms(2) bet_conga__bet by blast
  moreover have "D E F CongA C B A'"
    using calculation(1) not_conga_sym by blast
  ultimately show ?thesis
    by (metis SuppA_def conga_diff1)
qed

lemma bet2_suma__suma:
  assumes "A' \<noteq> B" and
    "D' \<noteq>  E" and
    "Bet A B A'" and
    "Bet D E D'" and
    "A B C D E F SumA G H I"
  shows "A' B C D' E F SumA G H I"
proof -
  obtain J where P1: "C B J CongA D E F \<and> \<not> B C OS A J \<and> 
Coplanar A B C J \<and> A B J CongA G H I"
    using SumA_def assms(5) by auto
  moreover
  obtain C' where P2: "Bet C B C' \<and> Cong B C' B C"
    using segment_construction by blast
  moreover
  have "A B C' D' E F SumA G H I"
  proof -
    have "C' B J CongA D' E F"
      by (metis assms(2) assms(4) calculation(1) calculation(2) 
          cong_diff_3 conga_diff1 l11_13)
    moreover have "\<not> B C' OS A J"
      by (metis Col_cases P1 P2 bet_col col_one_side cong_diff)
    moreover have "Coplanar A B C' J"
    proof-
      have "Coplanar A B C J" 
        using P1 by blast
      hence "Coplanar A B J C" 
        using coplanar_perm_1 by blast
      hence "Coplanar A B J C'" 
        by (metis P2 bet_col col_cop2__cop cong_diff ncop_distincts)
      thus ?thesis 
        using coplanar_perm_1 by blast
    qed
    ultimately show ?thesis
      using P1 SumA_def by blast
  qed
  moreover have "A B C' CongA A' B C"
    using assms(1) assms(3) assms(5) between_symmetry calculation(2) 
      calculation(3) l11_14 suma_distincts by auto
  moreover
  have "D' \<noteq> E \<and> E \<noteq> F"
    using assms(2) calculation(1) conga_distinct by blast
  hence "D' E F CongA D' E F"
    using conga_refl by auto
  moreover
  have "G \<noteq> H \<and> H \<noteq> I"
    using assms(5) suma_distincts by blast
  hence "G H I CongA G H I"
    using conga_refl by auto
  ultimately show ?thesis
    using conga3_suma__suma by blast
qed

lemma suma_suppa2__suma:
  assumes "A B C SuppA A' B' C'" and
    "D E F SuppA D' E' F'" and
    "A B C D E F SumA G H I"
  shows "A' B' C' D' E' F' SumA G H I"
proof -
  obtain A0 where P1: "Bet A B A0 \<and> A' B' C' CongA C B A0"
    using SuppA_def assms(1) by auto
  obtain D0 where P2: "Bet D E D0 \<and> D' E' F' CongA F E D0"
    using SuppA_def assms(2) by auto
  show ?thesis
  proof -
    have "A0 B C D0 E F SumA G H I"
    proof -
      have "A0 \<noteq> B"
        using CongA_def P1 by auto
      moreover have "D0 \<noteq> E"
        using CongA_def P2 by blast
      ultimately show ?thesis
        using P1 P2 assms(3) bet2_suma__suma by auto
    qed
    moreover have "A0 B C CongA A' B' C'"
      using P1 conga_left_comm not_conga_sym by blast
    moreover have "D0 E F CongA D' E' F'"
      using P2 conga_left_comm not_conga_sym by blast
    moreover
    have "G \<noteq> H \<and> H \<noteq> I"
      using assms(3) suma_distincts by blast
    hence "G H I CongA G H I"
      using conga_refl by auto
    ultimately show ?thesis
      using conga3_suma__suma by blast
  qed
qed

lemma suma2_obtuse2__conga:
  assumes "Obtuse A B C" and
    "A B C A B C SumA D E F" and
    "Obtuse A' B' C'" and
    "A' B' C' A' B' C' SumA D E F"
  shows "A B C CongA A' B' C'"
proof -
  obtain A0 where P1: "Bet A B A0 \<and> Cong B A0 A B"
    using segment_construction by blast
  moreover
  obtain A0' where P2: "Bet A' B' A0' \<and> Cong B' A0' A' B'"
    using segment_construction by blast
  moreover
  have "A0 B C CongA A0' B' C'"
  proof -
    have "Acute A0 B C"
      using assms(1) bet_obtuse__acute P1 cong_diff_3 obtuse_distincts by blast
    moreover have "A0 B C A0 B C SumA D E F"
      using P1 acute_distincts assms(2) bet2_suma__suma calculation by blast
    moreover have "Acute A0' B' C'"
      using P2 assms(3) bet_obtuse__acute cong_diff_3 obtuse_distincts by blast
    moreover have "A0' B' C' A0' B' C' SumA D E F"
      by (metis P2 assms(4) bet2_suma__suma cong_diff_3)
    ultimately show ?thesis
      using acute2_suma2__conga by blast
  qed
  moreover have "Bet A0 B A"
    using Bet_perm calculation(1) by blast
  moreover have "Bet A0' B' A'"
    using Bet_perm calculation(2) by blast
  moreover have "A \<noteq> B"
    using assms(1) obtuse_distincts by blast
  moreover have "A' \<noteq> B'"
    using assms(3) obtuse_distincts by blast
  ultimately show ?thesis
    using l11_13 by blast
qed

lemma bet_suma2__or_conga:
  assumes "A0 \<noteq> B" and
    "Bet A B A0" and
    "A B C A B C SumA D E F" and
    "A' B' C' A' B' C' SumA D E F"
  shows "A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'"
proof -
  {
    fix A' B' C'
    assume"Acute A' B' C' \<and> A' B' C' A' B' C' SumA D E F"
    have "Per A B C \<or> Acute A B C \<or> Obtuse A B C"
      by (metis angle_partition l8_20_1_R1 l8_5)
    {
      assume "Per A B C"
      hence "Bet D E F"
        using per2_suma__bet assms(3) by auto
      hence "False"
        using  \<open>Acute A' B' C' \<and> A' B' C' A' B' C' SumA D E F\<close> acute_suma__nbet by blast
      hence "A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'" by blast
    }
    {
      assume "Acute A B C"
      have "Acute A' B' C'"
        by (simp add: \<open>Acute A' B' C' \<and> A' B' C' A' B' C' SumA D E F\<close>)
      moreover have "A' B' C' A' B' C' SumA D E F"
        by (simp add: \<open>Acute A' B' C' \<and> A' B' C' A' B' C' SumA D E F\<close>)
      ultimately
      have "A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'"
        using assms(3) \<open>Acute A B C\<close> acute2_suma2__conga by auto
    }
    {
      assume "Obtuse A B C"
      have "Acute A0 B C"
        using \<open>Obtuse A B C\<close> assms(1) assms(2) bet_obtuse__acute by auto
      moreover have "A0 B C A0 B C SumA D E F"
        using assms(1) assms(2) assms(3) bet2_suma__suma by auto
      ultimately have "A0 B C CongA A' B' C'"
        using \<open>Acute A' B' C' \<and> A' B' C' A' B' C' SumA D E F\<close> acute2_suma2__conga by auto
      hence "A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'" by blast
    }
    hence "A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'"
      using \<open>Acute A B C \<Longrightarrow> A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'\<close> 
        \<open>Per A B C \<Longrightarrow> A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'\<close> 
        \<open>Per A B C \<or> Acute A B C \<or> Obtuse A B C\<close> by blast
  }
  hence P1: "\<forall> A' B' C'. (Acute A' B' C' \<and> A' B' C' A' B' C' SumA D E F) 
\<longrightarrow> (A B C CongA A' B' C' \<or> A0 B C CongA A' B' C')" by blast
  have "Per A' B' C' \<or> Acute A' B' C' \<or> Obtuse A' B' C'"
    by (metis angle_partition l8_20_1_R1 l8_5)
  {
    assume P2: "Per A' B' C'"
    have "A B C CongA A' B' C'"
    proof -
      have "A \<noteq> B \<and> B \<noteq> C"
        using assms(3) suma_distincts by blast
      moreover have "A' \<noteq> B' \<and> B' \<noteq> C'"
        using assms(4) suma_distincts by auto
      moreover have "Per A B C"
      proof -
        have "Bet D E F"
          using P2 assms(4) per2_suma__bet by blast
        moreover have "A B C A B C SumA D E F"
          by (simp add: assms(3))
        ultimately show ?thesis
          using assms(3) bet_suma__per by blast
      qed
      ultimately show ?thesis
        using P2 l11_16 by blast
    qed
    hence "A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'" by blast
  }
  {
    assume "Acute A' B' C'"
    hence "A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'"
      using P1 assms(4) by blast
  }
  {
    assume "Obtuse A' B' C'"
    obtain A0' where "Bet A' B' A0' \<and> Cong B' A0' A' B'"
      using segment_construction by blast
    moreover
    have "Acute A0' B' C'"
      using \<open>Obtuse A' B' C'\<close> bet_obtuse__acute calculation cong_diff_3 
        obtuse_distincts by blast
    moreover have "A0' B' C' A0' B' C' SumA D E F"
      using acute_distincts assms(4) bet2_suma__suma calculation(1) 
        calculation(2) by blast
    ultimately
    have "A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'"
      using P1 by (metis assms(1) assms(2) assms(3) assms(4) between_symmetry
          l11_13 suma_distincts)
  }
  thus ?thesis
    using \<open>Acute A' B' C' \<Longrightarrow> A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'\<close> 
      \<open>Per A' B' C' \<Longrightarrow> A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'\<close> 
      \<open>Per A' B' C' \<or> Acute A' B' C' \<or> Obtuse A' B' C'\<close> by blast
qed

lemma suma2__or_conga_suppa:
  assumes "A B C A B C SumA D E F" and
    "A' B' C' A' B' C' SumA D E F"
  shows "A B C CongA A' B' C' \<or> A B C SuppA A' B' C'"
proof -
  obtain A0 where P1: "Bet A B A0 \<and> Cong B A0 A B"
    using segment_construction by blast
  hence "A0 \<noteq> B"
    using  assms(1) bet_cong_eq suma_distincts by blast
  hence "A B C CongA A' B' C' \<or> A0 B C CongA A' B' C'"
    using assms(1) assms(2) P1 bet_suma2__or_conga by blast
  thus ?thesis
    by (metis P1 SuppA_def cong_diff conga_right_comm conga_sym)
qed

lemma ex_trisuma:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "A \<noteq> C"
  shows "\<exists> D E F. A B C TriSumA D E F"
proof -
  obtain G H I where "A B C B C A SumA G H I"
    using assms(1) assms(2) assms(3) ex_suma by presburger
  moreover
  then obtain D E F where "G H I C A B SumA D E F"
    using ex_suma suma_distincts by presburger
  ultimately show ?thesis
    using TriSumA_def by blast
qed

lemma trisuma_perm_231:
  assumes "A B C TriSumA D E F"
  shows "B C A TriSumA D E F"
proof -
  obtain A1 B1 C1 where P1: "A B C B C A SumA A1 B1 C1 \<and> A1 B1 C1 C A B SumA D E F"
    using TriSumA_def assms(1) by auto
  obtain A2 B2 C2 where P2: "B C A C A B SumA B2 C2 A2"
    using P1 ex_suma suma_distincts by fastforce
  have "A B C B2 C2 A2 SumA D E F"
  proof -
    have "SAMS A B C B C A"
      using assms sams123231 trisuma_distincts by auto
    moreover have "SAMS B C A C A B"
      using P2 sams123231 suma_distincts by fastforce
    ultimately show ?thesis
      using P1 P2 suma_assoc by blast
  qed
  thus ?thesis
    using P2 TriSumA_def suma_sym by blast
qed

lemma trisuma_perm_312:
  assumes "A B C TriSumA D E F"
  shows "C A B TriSumA D E F"
  by (simp add: assms trisuma_perm_231)

lemma trisuma_perm_321:
  assumes "A B C TriSumA D E F"
  shows "C B A TriSumA D E F"
proof -
  obtain G H I where "A B C B C A SumA G H I \<and> G H I C A B SumA D E F"
    using TriSumA_def assms(1) by auto
  thus ?thesis
    by (meson TriSumA_def suma_comm suma_right_comm suma_sym trisuma_perm_231)
qed

lemma trisuma_perm_213:
  assumes "A B C TriSumA D E F"
  shows "B A C TriSumA D E F"
  using assms trisuma_perm_231 trisuma_perm_321 by blast

lemma trisuma_perm_132:
  assumes "A B C TriSumA D E F"
  shows "A C B TriSumA D E F"
  using assms trisuma_perm_312 trisuma_perm_321 by blast

lemma conga_trisuma__trisuma:
  assumes "A B C TriSumA D E F" and
    "D E F CongA D' E' F'"
  shows "A B C TriSumA D' E' F'"
proof -
  obtain G H I where P1: "A B C B C A SumA G H I \<and> G H I C A B SumA D E F"
    using TriSumA_def assms(1) by auto
  have "G H I C A B SumA D' E' F'"
  proof -
    have f1: "B \<noteq> A"
      by (metis P1 suma_distincts)
    have f2: "C \<noteq> A"
      using P1 suma_distincts by blast
    have "G H I CongA G H I"
      by (metis (full_types) P1 conga_refl suma_distincts)
    thus ?thesis
      using f2 f1 by (meson P1 assms(2) conga3_suma__suma conga_refl)
  qed
  thus ?thesis using P1 TriSumA_def by blast
qed

lemma trisuma2__conga:
  assumes "A B C TriSumA D E F" and
    "A B C TriSumA D' E' F'"
  shows "D E F CongA D' E' F'"
proof -
  obtain G H I where P1: "A B C B C A SumA G H I \<and> G H I C A B SumA D E F"
    using TriSumA_def assms(1) by auto
  hence P1A: "G H I C A B SumA D E F" by simp
  obtain G' H' I' where P2: "A B C B C A SumA G' H' I' \<and> G' H' I' C A B SumA D' E' F'"
    using TriSumA_def assms(2) by auto
  have "G' H' I' C A B SumA D E F"
  proof -
    have "G H I CongA G' H' I'" using P1 P2 suma2__conga by blast
    moreover have "D E F CongA D E F \<and> C A B CongA C A B"
      by (metis assms(1) conga_refl trisuma_distincts)
    ultimately show ?thesis
      by (meson P1 conga3_suma__suma)
  qed
  thus ?thesis
    using P2 suma2__conga by auto
qed

lemma conga3_trisuma__trisuma:
  assumes "A B C TriSumA D E F" and
    "A B C CongA A' B' C'" and
    "B C A CongA B' C' A'" and
    "C A B CongA C' A' B'"
  shows "A' B' C' TriSumA D E F"
proof -
  obtain G H I where P1: "A B C B C A SumA G H I \<and> G H I C A B SumA D E F"
    using TriSumA_def assms(1) by auto
  thus ?thesis
  proof -
    have "A' B' C' B' C' A' SumA G H I"
      using conga3_suma__suma P1 by (meson assms(2) assms(3) suma2__conga)
    moreover have "G H I C' A' B' SumA D E F"
      using conga3_suma__suma P1 by (meson P1 assms(4) suma2__conga)
    ultimately show ?thesis
      using TriSumA_def by blast
  qed
qed

lemma col_trisuma__bet:
  assumes "Col A B C" and
    "A B C TriSumA P Q R"
  shows "Bet P Q R"
proof -
  obtain D E F where P1: "A B C B C A SumA D E F \<and> D E F C A B SumA P Q R"
    using TriSumA_def assms(2) by auto
  {
    assume "Bet A B C"
    have "A B C CongA P Q R"
    proof -
      have "A B C CongA D E F"
      proof -
        have "C \<noteq> A \<and> C \<noteq> B"
          using assms(2) trisuma_distincts by blast
        hence "C Out B A"
          using \<open> Bet A B C\<close> bet_out_1 by fastforce
        thus ?thesis
          using P1 out546_suma__conga by auto
      qed
      moreover have "D E F CongA P Q R"
      proof -
        have "A \<noteq> C \<and> A \<noteq> B"
          using assms(2) trisuma_distincts by blast
        hence "A Out C B"
          using Out_def \<open>Bet A B C\<close> by auto
        thus ?thesis
          using P1 out546_suma__conga by auto
      qed
      ultimately show ?thesis
        using conga_trans by blast
    qed
    hence "Bet P Q R"
      using \<open>Bet A B C\<close> bet_conga__bet by blast
  }
  {
    assume "Bet B C A"
    have "B C A CongA P Q R"
    proof -
      have "B C A CongA D E F"
      proof -
        have "B \<noteq> A \<and> B \<noteq> C"
          using assms(2) trisuma_distincts by blast
        hence "B Out A C"
          using Out_def \<open>Bet B C A\<close> by auto
        thus ?thesis
          using P1 out213_suma__conga by blast
      qed
      moreover have "D E F CongA P Q R"
      proof -
        have "A \<noteq> C \<and> A \<noteq> B"
          using assms(2) trisuma_distincts by auto
        hence "A Out C B"
          using \<open>Bet B C A\<close> bet_out_1 by auto
        thus ?thesis
          using P1 out546_suma__conga by blast
      qed
      ultimately show ?thesis
        using not_conga by blast
    qed
    hence "Bet P Q R"
      using \<open>Bet B C A\<close> bet_conga__bet by blast
  }
  {
    assume "Bet C A B"
    have "E Out D F"
    proof -
      have "C Out B A"
        using \<open>Bet C A B\<close> assms(2) bet_out l6_6 trisuma_distincts by blast
      moreover have "B C A CongA D E F"
      proof -
        have "B \<noteq> A \<and> B \<noteq> C"
          using P1 suma_distincts by blast
        hence "B Out A C"
          using \<open>Bet C A B\<close> bet_out_1 by auto
        thus ?thesis using out213_suma__conga P1 by blast
      qed
      ultimately show ?thesis
        using l11_21_a by blast
    qed
    hence "C A B CongA P Q R"
      using P1 out213_suma__conga by blast
    hence "Bet P Q R"
      using \<open>Bet C A B\<close> bet_conga__bet by blast
  }
  thus ?thesis
    using Col_def \<open>Bet A B C \<Longrightarrow> Bet P Q R\<close> \<open>Bet B C A \<Longrightarrow> Bet P Q R\<close> assms(1) by blast
qed

lemma suma_dec:
  "A B C D E F SumA G H I \<or> \<not> A B C D E F SumA G H I" 
  by simp

lemma sams_dec:
  "SAMS A B C D E F \<or> \<not> SAMS A B C D E F" 
  by simp

lemma trisuma_dec:
  "A B C TriSumA P Q R \<or> \<not> A B C TriSumA P Q R"
  by simp

lemma acute_not_bet:
  assumes "Acute A B C"
  shows "\<not> Bet A B C"
  using acute_col__out assms bet_col not_bet_and_out by blast

lemma upper_dim_3_stab:
  assumes "\<not> \<not> upper_dim_3_axiom"
  shows "upper_dim_3_axiom" 
  using assms by blast

lemma median_planes_implies_upper_dim: 
  assumes "median_planes_axiom"
  shows "upper_dim_3_axiom"
proof -
  {
    fix A B C P Q R
    assume "P \<noteq> Q" and "Q \<noteq> R" and "P \<noteq> R" and
      "Cong A P A Q" and "Cong B P B Q" and "Cong C P C Q" and
      "Cong A P A R" and "Cong B P B R" and "Cong C P C R" 
    have "Bet A B C \<or> Bet B C A \<or> Bet C A B"
    proof (cases "Col A B C")
      case True
      thus ?thesis
        by (simp add: Col_def upper_dim_3_axiom_def)
    next
      case False
      hence "\<not> Col A B C" 
        by simp
      {
        obtain X where "X Midpoint P Q" 
          using midpoint_existence by blast
        obtain Y where "Y Midpoint P R" 
          using midpoint_existence by blast
        have "X = Y" 
        proof -
          have "Per X Y P"
          proof -
            have "\<not> Col A B C"
              by (simp add: False)
            moreover have "Per A Y P" 
              using Per_def \<open>Cong A P A R\<close> \<open>Y Midpoint P R\<close> by blast
            moreover have "Per B Y P" 
              using Per_def \<open>Cong B P B R\<close> \<open>Y Midpoint P R\<close> by blast
            moreover have "Per C Y P"   
              using Per_def \<open>Cong C P C R\<close> \<open>Y Midpoint P R\<close> by blast
            moreover 
            have "Cong X P X Q" 
              using Cong_cases \<open>X Midpoint P Q\<close> midpoint_cong by blast
            hence "Coplanar A B C X"
              using \<open>P \<noteq> Q\<close> \<open>Cong A P A Q\<close> \<open>Cong B P B Q\<close> \<open>Cong C P C Q\<close> assms 
                median_planes_axiom_def by blast
            ultimately show ?thesis
              using l11_60 by blast
          qed
          moreover have "Per Y X P" 
          proof -
            have "\<not> Col A B C" 
              by (simp add: False)
            moreover have "Per A X P"     
              using Per_def \<open>Cong A P A Q\<close> \<open>X Midpoint P Q\<close> by blast
            moreover have "Per B X P" 
              using Per_def \<open>Cong B P B Q\<close> \<open>X Midpoint P Q\<close> by blast
            moreover have "Per C X P" 
              using Per_def \<open>Cong C P C Q\<close> \<open>X Midpoint P Q\<close> by blast
            moreover
            have "Cong Y P Y R" 
              using Cong_cases \<open>Y Midpoint P R\<close> midpoint_cong by blast
            hence "Coplanar A B C Y"
              using \<open>P \<noteq> R\<close> \<open>Cong A P A R\<close> \<open>Cong B P B R\<close> \<open>Cong C P C R\<close> 
                assms median_planes_axiom_def by blast
            ultimately show ?thesis
              using l11_60 by blast
          qed
          ultimately show ?thesis 
            using Per_cases l8_7 by blast
        qed
        hence "X Midpoint P R" 
          by (simp add: \<open>Y Midpoint P R\<close>)
        hence "Q = R" 
          using \<open>X Midpoint P Q\<close> symmetric_point_uniqueness by blast
        hence False 
          using \<open>Q \<noteq> R\<close> by blast
        hence "Bet A B C \<or> Bet B C A \<or> Bet C A B" 
          by simp
      }
      thus ?thesis 
        by (simp add: upper_dim_3_axiom_def)
    qed
  }
  thus ?thesis 
    using upper_dim_3_axiom_def by blast
qed

lemma median_planes_aux:
  assumes "\<forall> A B C P Q M. P \<noteq> Q \<and> Cong A P A Q \<and> Cong B P B Q \<and> 
                          Cong C P C Q \<and> M Midpoint P Q \<longrightarrow> Coplanar M A B C" 
  shows "median_planes_axiom"
proof -
  {
    fix A B C D P Q
    assume "P \<noteq> Q" and
      "Cong A P A Q" and
      "Cong B P B Q" and
      "Cong C P C Q" and
      "Cong D P D Q"
    have "Coplanar A B C D" 
    proof (cases "Col A B C")
      case True
      thus ?thesis 
        using ncop__ncols by blast
    next
      case False
      obtain M where "M Midpoint P Q" 
        using midpoint_existence by blast
      obtain A1 A2 where "Coplanar A B C A1" and "Coplanar A B C A2" and "\<not> Col M A1 A2" 
        using ex_ncol_cop2 by blast
      have "Cong A1 P A1 Q" 
        using False l11_60_aux \<open>Cong A P A Q\<close> \<open>Cong B P B Q\<close> 
          \<open>Cong C P C Q\<close> \<open>Coplanar A B C A1\<close> by blast
      have "Cong A2 P A2 Q" 
        using False \<open>Cong A P A Q\<close> \<open>Cong B P B Q\<close> \<open>Cong C P C Q\<close> \<open>Coplanar A B C A2\<close> 
          l11_60_aux by blast
      have "Coplanar M A B C" 
        using assms \<open>M Midpoint P Q\<close> \<open>Cong A P A Q\<close> \<open>Cong B P B Q\<close> \<open>Cong C P C Q\<close> 
          \<open>P \<noteq> Q\<close> by presburger
      have "Coplanar M A B D" 
        using assms \<open>M Midpoint P Q\<close> \<open>Cong A P A Q\<close> \<open>Cong B P B Q\<close> \<open>Cong D P D Q\<close> 
          \<open>P \<noteq> Q\<close> by blast
      have "Coplanar A B C D" 
      proof -
        have "Coplanar M A1 A2 A"
          by (meson False coplanar_pseudo_trans \<open>Coplanar A B C A1\<close> \<open>Coplanar A B C A2\<close> 
              \<open>Coplanar M A B C\<close> coplanar_perm_9 ncop_distincts) 
        moreover have "Coplanar M A1 A2 B" 
          by (meson False coplanar_pseudo_trans \<open>Coplanar A B C A1\<close> \<open>Coplanar A B C A2\<close> 
              \<open>Coplanar M A B C\<close> coplanar_perm_9 ncop_distincts) 
        moreover have "Coplanar M A1 A2 C" 
          by (meson False \<open>Coplanar A B C A1\<close> \<open>Coplanar A B C A2\<close> \<open>Coplanar M A B C\<close> 
              coplanar_perm_9 coplanar_pseudo_trans ncop_distincts)
        moreover 
        have "Coplanar M A1 A2 D" 
        proof (cases "Col A B M")
          case True
          hence "Col A B M" 
            by blast
          have "Coplanar P Q A B" 
            using Col_def Coplanar_def Midpoint_def True \<open>M Midpoint P Q\<close> 
              ncoplanar_perm_15 by blast
          show ?thesis
          proof (cases "Col P Q A")
            case True
            hence "Col P Q A"
              by blast
            thus ?thesis
              using \<open>Cong A1 P A1 Q\<close> \<open>Cong A2 P A2 Q\<close> \<open>Cong D P D Q\<close> \<open>M Midpoint P Q\<close> 
                \<open>P \<noteq> Q\<close> assms by blast
          next
            case False
            hence "\<not> Col P Q A"
              by simp
            thus ?thesis 
              using \<open>Cong A1 P A1 Q\<close> \<open>Cong A2 P A2 Q\<close> \<open>Cong D P D Q\<close> \<open>M Midpoint P Q\<close> 
                \<open>P \<noteq> Q\<close> assms by blast
          qed
        next
          case False
          hence "\<not> Col A B M" 
            by blast
          moreover have "Coplanar M A B M" 
            using ncop_distincts by blast
          moreover have "Coplanar M A B A1" 
            by (meson \<open>Coplanar M A1 A2 A\<close> \<open>Coplanar M A1 A2 B\<close> 
                \<open>\<not> Col M A1 A2\<close> coplanar_pseudo_trans ncop_distincts)
          moreover have "Coplanar M A B A2" 
            by (meson \<open>Coplanar M A1 A2 A\<close> \<open>Coplanar M A1 A2 B\<close> 
                \<open>\<not> Col M A1 A2\<close> coplanar_pseudo_trans ncop_distincts)
          ultimately
          show ?thesis 
            by (meson \<open>Coplanar M A B D\<close> coplanar_pseudo_trans not_col_permutation_2)
        qed
        ultimately show ?thesis 
          using \<open>\<not> Col M A1 A2\<close> coplanar_pseudo_trans by blast
      qed
      thus ?thesis 
        by blast
    qed
  }
  thus ?thesis 
    using median_planes_axiom_def by blast
qed

lemma orthonormal_family_aux_1:
  assumes "orthonormal_family_axiom"
  shows "\<forall> A B X P Q. \<not> Col P Q X \<and> Per A X P \<and> Per A X Q \<and> Per B X P \<and> Per B X Q \<longrightarrow> Col A B X" 
proof -
  {
    fix A B X P Q
    assume "\<not> Col P Q X" and
      "Per A X P" and
      "Per A X Q" and
      "Per B X P" and "Per B X Q"
    {
      assume "\<not> Col A B X"
      obtain P' where "Bet P X P'" and "Cong X P' P X" 
        using segment_construction by blast
      have "A \<noteq> X" 
        using \<open>\<not> Col A B X\<close> not_col_distincts by blast
      have "X \<noteq> P" 
        using \<open>\<not> Col P Q X\<close> not_col_distincts by blast
      have "X \<noteq> P'" 
        using \<open>Cong X P' P X\<close> \<open>X \<noteq> P\<close> cong_reverse_identity by blast
      obtain Q' where "Per Q' X P" and "Cong Q' X P' X" and "P X OS Q' Q" 
        using ex_per_cong \<open>X \<noteq> P'\<close> \<open>X \<noteq> P\<close> \<open>\<not> Col P Q X\<close> col_trivial_2 
          not_col_permutation_5 by metis
      have "Per Q' X A" 
      proof (rule l11_60 [where ?A = "P" and ?B = "Q" and ?C = "X"], insert \<open>\<not> Col P Q X\<close>)
        show "Per P X A" 
          using \<open>Per A X P\<close> l8_2 by blast
        show "Per Q X A" 
          by (simp add: \<open>Per A X Q\<close> l8_2)
        show "Per X X A" 
          by (simp add: l8_20_1_R1)
        show "Coplanar P Q X Q'" 
          using \<open>P X OS Q' Q\<close> coplanar_perm_4 os__coplanar by blast
      qed
      have "Per Q' X B" 
      proof (rule l11_60 [where ?A = "P" and ?B = "Q" and ?C = "X"], insert \<open>\<not> Col P Q X\<close>)
        show "Per P X B" 
          by (simp add: \<open>Per B X P\<close> l8_2)
        show "Per Q X B" 
          using \<open>Per B X Q\<close> l8_2 by blast
        show "Per X X B" 
          by (simp add: l8_20_1_R1)
        show "Coplanar P Q X Q'" 
          using \<open>P X OS Q' Q\<close> coplanar_perm_4 os__coplanar by blast
      qed
      have "\<not> Col P X Q'" 
        using \<open>P X OS Q' Q\<close> col123__nos by blast
      let ?Q = "Q'"
      obtain A' where "Bet A X A'" and "Cong X A' P' X" 
        using segment_construction by blast
      have "Per P X A'" 
        using Col_def Per_cases \<open>A \<noteq> X\<close> \<open>Bet A X A'\<close> \<open>Per A X P\<close> between_symmetry l8_3 by blast
      have "Per ?Q X A'" 
        by (metis Bet_cases Col_def \<open>A \<noteq> X\<close> \<open>Bet A X A'\<close> \<open>Per Q' X A\<close> per_col)
      have "\<not> Col A' B X" 
        by (metis \<open>Bet A X A'\<close> \<open>Cong X A' P' X\<close> \<open>X \<noteq> P'\<close> \<open>\<not> Col A B X\<close> 
            bet_col cong_diff_4 l6_16_1 not_col_permutation_2)
      let ?A = "A'"
      have "\<exists> B'. Per B' X ?A \<and> Cong B' X P' X \<and> ?A X OS B' B"
      proof (rule ex_per_cong)
        show "?A \<noteq> X" 
          using \<open>\<not> Col A' B X\<close> col_trivial_3 by force
        show "P' \<noteq> X" 
          using \<open>X \<noteq> P'\<close> by blast
        show "Col ?A X X" 
          by (simp add: col_trivial_2)
        show "\<not> Col ?A X B" 
          using \<open>\<not> Col A' B X\<close> not_col_permutation_5 by blast
      qed
      then obtain B' where "Per B' X ?A" and "Cong B' X P' X" and "?A X OS B' B"
        by blast
      have "Per B' X P"
      proof (rule l11_60 [where ?A = "?A" and ?B = "B" and ?C = "X"], insert \<open>\<not> Col A' B X\<close>)
        show "Per ?A X P" 
          by (simp add: \<open>Per P X A'\<close> l8_2)
        show "Per B X P" 
          by (simp add: \<open>Per B X P\<close>)
        show "Per X X P" 
          by (simp add: l8_20_1_R1)
        show "Coplanar ?A B X B'" 
          using \<open>A' X OS B' B\<close> coplanar_perm_4 os__coplanar by blast
      qed
      have "Per B' X ?Q" 
      proof (rule l11_60 [where ?A = "?A" and ?B = "B" and ?C = "X"], insert \<open>\<not> Col A' B X\<close>)
        show "\<not> Col A' B X \<Longrightarrow> Per A' X ?Q" 
          by (simp add: \<open>Per Q' X A'\<close> l8_2) 
        show "\<not> Col A' B X \<Longrightarrow> Per B X ?Q" 
          using Per_cases \<open>Per Q' X B\<close> by auto
        show "\<not> Col A' B X \<Longrightarrow> Per X X ?Q" 
          by (simp add: l8_20_1_R1)
        show "\<not> Col A' B X \<Longrightarrow> Coplanar A' B X B'" 
          using \<open>A' X OS B' B\<close> coplanar_perm_4 os__coplanar by blast
      qed
      let ?B = "B'"
      have "Cong ?Q P ?Q P'" 
      proof (rule per_double_cong [where ?B = "X"])
        show "Per ?Q X P" 
          using \<open>Per Q' X P\<close> by blast
        show "X Midpoint P P'" 
          using Cong_perm Midpoint_def \<open>Bet P X P'\<close> \<open>Cong X P' P X\<close> by blast
      qed
      have False 
      proof -
        have "Cong X P X P'"
          using Cong_cases \<open>Cong X P' P X\<close> by blast
        moreover have "Cong X ?Q X P'" 
          using Cong_cases \<open>Cong Q' X P' X\<close> by blast
        moreover have "Cong X ?A X P'" 
          using Cong_cases \<open>Cong X A' P' X\<close> by blast
        moreover have "Cong X ?B X P'" 
          using Cong_cases \<open>Cong B' X P' X\<close> by blast
        moreover have "Cong P ?Q P' ?Q" 
          using Cong_cases \<open>Cong Q' P Q' P'\<close> by blast
        moreover 
        have "Cong P ?A P ?Q" 
          by (meson \<open>Cong Q' X P' X\<close> \<open>Cong X P' P X\<close> \<open>Per P X A'\<close> \<open>Per Q' X P\<close> 
              calculation(3) cong_4312 cong_inner_transitivity l10_12)
        hence "Cong P ?A P' ?Q"
          using cong_transitivity calculation(5) by blast
        moreover
        have "Cong P ?B P ?Q" 
        proof (rule l10_12 [where ?B = "X" and ?B' = "X"])
          show "Per P X B'" 
            by (simp add: \<open>Per B' X P\<close> l8_2)
          show "Per P X Q'" 
            by (simp add: \<open>Per Q' X P\<close> l8_2)
          show "Cong P X P X" 
            by (simp add: cong_reflexivity)
          show "Cong X B' X Q'" 
            by (meson calculation(2) calculation(4) cong_3421 cong_inner_transitivity)
        qed
        hence "Cong P ?B P' ?Q"
          using \<open>Cong P Q' P' Q'\<close> cong_transitivity by blast
        moreover have "Cong ?Q ?A P' ?Q" 
          using l10_12 
          by (metis Col_def not_cong_1243 \<open>Bet P X P'\<close> \<open>Per Q' X A'\<close> \<open>Per Q' X P\<close> 
              \<open>X \<noteq> P\<close> between_symmetry calculation(3) cong_reflexivity per_col)
        moreover have "Cong ?A ?B P' ?Q" 
          by (meson \<open>Cong X P' P X\<close> \<open>Per B' X A'\<close> \<open>Per P X A'\<close> calculation(4) calculation(6) 
              cong_pseudo_reflexivity cong_transitivity l10_12)
        moreover 
        have "Cong P ?Q ?Q ?B" 
        proof (rule l10_12 [where ?B = "X" and ?B' = "X"])
          show "Per P X Q'" 
            by (simp add: \<open>Per Q' X P\<close> l8_2)
          show "Per Q' X B'" 
            by (simp add: \<open>Per B' X Q'\<close> l8_2)
          show "Cong P X Q' X" 
            using \<open>Cong X P' P X\<close> calculation(2) cong_3421 cong_transitivity by blast
          show "Cong X Q' X B'" 
            by (meson calculation(2) calculation(4) cong_3421 cong_inner_transitivity)
        qed
        have "Cong P' ?Q P ?Q" 
          using \<open>Cong P Q' P' Q'\<close> Cong_cases by auto
        hence "Cong ?Q ?B P' ?Q" 
          using \<open>Cong P Q' Q' B'\<close> calculation(5) cong_inner_transitivity by blast
        ultimately show ?thesis 
          using assms \<open>X \<noteq> P'\<close> \<open>Bet P X P'\<close> orthonormal_family_axiom_def by blast
      qed
    }
    hence "Col A B X" 
      by blast
  }
  thus ?thesis 
    by blast
qed

lemma orthonormal_family_aux_2:
  assumes "\<forall> A B X P Q. \<not> Col P Q X \<and> Per A X P \<and> Per A X Q \<and> 
                        Per B X P \<and> Per B X Q \<longrightarrow> Col A B X" 
  shows "orthonormal_family_axiom" 
proof -
  {
    fix S U1' U1 U2 U3 U4
    assume "S \<noteq> U1'" and 
      "Bet U1 S U1'" and
      "Cong S U1 S U1'" and
      "Cong S U2 S U1'" and
      "Cong S U3 S U1'" and
      "Cong S U4 S U1'" and
      "Cong U1 U2 U1' U2" and
      "Cong U1 U3 U1' U2" and
      "Cong U1 U4 U1' U2" and
      "Cong U2 U3 U1' U2"  and 
      "Cong U2 U4 U1' U2" and
      "Cong U3 U4 U1' U2"
    have "S Midpoint U1 U1'" 
      using Cong_cases \<open>Bet U1 S U1'\<close> \<open>Cong S U1 S U1'\<close> midpoint_def by blast
    hence "Per U2 S U1" 
      using Per_def \<open>Cong U1 U2 U1' U2\<close> \<open>S Midpoint U1 U1'\<close> not_cong_2143 by blast
    have "S \<noteq> U4" 
      using \<open>Cong S U4 S U1'\<close> \<open>S \<noteq> U1'\<close> cong_diff_3 by blast
    have "S \<noteq> U3" 
      using \<open>Cong S U3 S U1'\<close> \<open>S \<noteq> U1'\<close> cong_reverse_identity by auto
    have "S \<noteq> U2" 
      using \<open>Cong S U2 S U1'\<close> \<open>S \<noteq> U1'\<close> cong_diff_3 by blast
    have "S \<noteq> U1" 
      using \<open>Cong S U1 S U1'\<close> \<open>S \<noteq> U1'\<close> cong_diff_3 by blast
    have "U1 \<noteq> U2" 
      using \<open>Per U2 S U1\<close> \<open>S \<noteq> U2\<close> per_distinct_1 by blast
    have "Col U2 U1 S"
    proof -
      have "U2 S U1 CongA U4 S U3" 
        using l11_51 
        by (metis Tarski_neutral_dimensionless.cong_transitivity 
            Tarski_neutral_dimensionless_axioms \<open>Cong S U1 S U1'\<close> 
            \<open>Cong S U2 S U1'\<close> \<open>Cong S U3 S U1'\<close> \<open>Cong S U4 S U1'\<close> 
            \<open>Cong U1 U2 U1' U2\<close> \<open>Cong U3 U4 U1' U2\<close> \<open>S \<noteq> U1\<close> \<open>S \<noteq> U2\<close> \<open>U1 \<noteq> U2\<close> cong_3421)
      hence "Per U4 S U3" 
        using l8_10 l11_17 \<open>Per U2 S U1\<close> by blast
      hence "\<not> Col U4 S U3" 
        using \<open>S \<noteq> U3\<close> \<open>S \<noteq> U4\<close> per_col_eq by blast
      hence "\<not> Col U3 U4 S" 
        using Col_cases by blast
      moreover 
      have "U2 S U1 CongA U2 S U3" 
        by (metis cong_transitivity l11_51 midpoint_cong 
            \<open>Cong S U3 S U1'\<close> \<open>Cong U1 U2 U1' U2\<close> \<open>Cong U2 U3 U1' U2\<close> \<open>S Midpoint U1 U1'\<close> 
            \<open>S \<noteq> U1\<close> \<open>S \<noteq> U2\<close> \<open>U1 \<noteq> U2\<close> cong_3421 cong_pseudo_reflexivity)
      hence "Per U2 S U3" 
        using \<open>Per U2 S U1\<close> l11_17 by blast
      moreover 
      have "U2 S U1 CongA U2 S U4" 
        by (metis cong_reflexivity cong_transitivity l11_51 
            \<open>Cong S U1 S U1'\<close> \<open>Cong S U4 S U1'\<close> \<open>Cong U1 U2 U1' U2\<close> \<open>Cong U2 U4 U1' U2\<close> 
            \<open>S \<noteq> U1\<close> \<open>S \<noteq> U2\<close> \<open>U1 \<noteq> U2\<close> cong_left_commutativity cong_symmetry)
      hence "Per U2 S U4" 
        using \<open>Per U2 S U1\<close> l11_17 by blast
      moreover have "U2 S U1 CongA U1 S U3" 
        by (metis cong_transitivity \<open>Cong S U1 S U1'\<close> \<open>Cong S U2 S U1'\<close> 
            \<open>Cong S U3 S U1'\<close> \<open>Cong U1 U2 U1' U2\<close> \<open>Cong U1 U3 U1' U2\<close> \<open>S \<noteq> U1\<close> \<open>S \<noteq> U2\<close> 
            \<open>U1 \<noteq> U2\<close> cong_left_commutativity cong_symmetry l11_51)
      hence "Per U1 S U3" 
        using \<open>Per U2 S U1\<close> l11_17 by blast
      moreover have "U2 S U1 CongA U1 S U4" 
        by (metis cong_transitivity \<open>Cong S U1 S U1'\<close> \<open>Cong S U2 S U1'\<close> 
            \<open>Cong S U4 S U1'\<close> \<open>Cong U1 U2 U1' U2\<close> \<open>Cong U1 U4 U1' U2\<close> \<open>S \<noteq> U1\<close> 
            \<open>S \<noteq> U2\<close> \<open>U1 \<noteq> U2\<close> cong_left_commutativity cong_symmetry l11_51)
      hence "Per U1 S U4" 
        using \<open>Per U2 S U1\<close> l11_17 by blast
      ultimately show ?thesis 
        using assms by blast
    qed
    hence False 
      by (metis NCol_perm \<open>Bet U1 S U1'\<close> \<open>Cong S U1 S U1'\<close> \<open>Cong S U2 S U1'\<close> 
          \<open>Cong U1 U2 U1' U2\<close> \<open>S \<noteq> U1'\<close> bet_neq23__neq cong_3421 cong_diff_2 l4_18)
  }
  thus ?thesis 
    using orthonormal_family_axiom_def by blast
qed

lemma orthonormal_family_aux:
  shows "orthonormal_family_axiom \<longleftrightarrow>
   (\<forall> A B X P Q. \<not> Col P Q X \<and> Per A X P \<and> Per A X Q \<and> Per B X P \<and> Per B X Q \<longrightarrow> Col A B X)" 
  using orthonormal_family_aux_1 orthonormal_family_aux_2 by blast

lemma upper_dim_implies_orthonormal_family_axiom:
  assumes "upper_dim_3_axiom"
  shows "orthonormal_family_axiom" 
proof -
  {
    fix A B X P Q
    assume "\<not> Col P Q X" and 
      "Per A X P" and
      "Per A X Q" and
      "Per B X P" and
      "Per B X Q"
    obtain Q' where "Bet Q X Q'" and "Cong X Q' X P" 
      using segment_construction by blast
    have "\<not> Col P Q' X" 
      by (metis \<open>Bet Q X Q'\<close> \<open>Cong X Q' X P\<close> \<open>\<not> Col P Q X\<close> bet_col bet_cong_eq 
          between_trivial2 col_permutation_2 l6_16_1)
    have "Per A X Q'"  
      by (metis Bet_cases Col_def \<open>Bet Q X Q'\<close> \<open>Per A X Q\<close> \<open>\<not> Col P Q X\<close> col_trivial_2 per_col)
    have "Per B X Q'" 
      by (metis Bet_cases Col_def \<open>Bet Q X Q'\<close> \<open>Per B X Q\<close> \<open>\<not> Col P Q X\<close> col_trivial_2 per_col)
    obtain R where "X Midpoint P R" 
      using symmetric_point_construction by blast
    have "P \<noteq> Q'" 
      using \<open>\<not> Col P Q' X\<close> col_trivial_1 by blast
    have "P \<noteq> X" 
      using \<open>\<not> Col P Q X\<close> not_col_distincts by blast
    hence "P \<noteq> R" 
      using \<open>X Midpoint P R\<close> l7_3 by auto
    have "Col A B X"  
    proof -
      have "Q' \<noteq> R" 
        using Bet_cases Col_def Midpoint_def \<open>X Midpoint P R\<close> \<open>\<not> Col P Q' X\<close> by blast
      moreover have "Cong A P A Q'" 
        using \<open>Cong X Q' X P\<close> \<open>Per A X P\<close> \<open>Per A X Q'\<close> cong_reflexivity 
          l10_12 not_cong_3412 by blast
      moreover have "Cong B P B Q'" 
        using cong_reflexivity l10_12 \<open>Cong X Q' X P\<close> \<open>Per B X P\<close> 
          \<open>Per B X Q'\<close> cong_symmetry by blast
      moreover have "Cong X P X Q'" 
        using \<open>Cong X Q' X P\<close> not_cong_3412 by blast
      moreover have "Cong A P A R" 
        using \<open>Per A X P\<close> \<open>X Midpoint P R\<close> per_double_cong by auto
      moreover have "Cong B P B R" 
        using \<open>Per B X P\<close> \<open>X Midpoint P R\<close> per_double_cong by blast
      moreover have "Cong X P X R" 
        using Mid_cases \<open>X Midpoint P R\<close> l7_13 l7_3_2 by blast
      ultimately show ?thesis 
        using Col_def \<open>P \<noteq> Q'\<close> \<open>P \<noteq> R\<close> assms upper_dim_3_axiom_def by blast
    qed
  }
  thus ?thesis
    using orthonormal_family_aux by blast
qed

lemma orthonormal_family_axiom_implies_orth_at2__col:
  assumes "orthonormal_family_axiom"
  shows "\<forall> A B C P Q X. X OrthAt A B C X P \<and> X OrthAt A B C X Q \<longrightarrow> Col P Q X" 
proof -
  {
    assume "\<forall> A B X P Q. \<not> Col P Q X \<and> Per A X P \<and> Per A X Q \<and> 
                         Per B X P \<and> Per B X Q \<longrightarrow> Col A B X" 
    {
      fix A B C P Q X
      assume "X OrthAt A B C X P" and 
        "X OrthAt A B C X Q"
      have 1: "\<not> Col A B C \<and> X \<noteq> P \<and> Coplanar A B C X \<and> (\<forall> D.(Coplanar A B C D \<longrightarrow> Per D X P))" 
        using \<open>X OrthAt A B C X P\<close> orth_at_chara by blast
      have 2: "\<not> Col A B C \<and> X \<noteq> Q \<and> Coplanar A B C X \<and> (\<forall> D.(Coplanar A B C D \<longrightarrow> Per D X Q))" 
        using \<open>X OrthAt A B C X Q\<close> orth_at_chara by blast
      obtain D E where "Coplanar A B C D" and "Coplanar A B C E" and "\<not> Col X D E" 
        using ex_ncol_cop2 by blast
      have "\<not> Col D E X" 
        using \<open>\<not> Col X D E\<close> not_col_permutation_1 by blast
      moreover have "Per P X D" 
        using \<open>Coplanar A B C D\<close> 1 l8_2 by blast
      moreover have "Per P X E" 
        using \<open>Coplanar A B C E\<close> 1 l8_2 by blast
      moreover have "Per Q X D" 
        using \<open>Coplanar A B C D\<close> 2 l8_2 by blast
      moreover have "Per Q X E" 
        using \<open>Coplanar A B C E\<close> 2 l8_2 by blast
      ultimately have "Col P Q X" 
        using \<open>\<forall>A B X P Q. \<not> Col P Q X \<and> Per A X P \<and> Per A X Q \<and> 
                           Per B X P \<and> Per B X Q \<longrightarrow> Col A B X\<close> by blast
    }
    hence "\<forall> A B C P Q X. X OrthAt A B C X P \<and> X OrthAt A B C X Q \<longrightarrow> Col P Q X" 
      by blast
  }
  thus ?thesis 
    using assms orthonormal_family_aux by blast
qed

lemma orthonormal_family_axiom_implies_not_two_sides_one_side:
  assumes "orthonormal_family_axiom"
  shows "\<forall> A B C X Y. \<not> Coplanar A B C X \<and> \<not> Coplanar A B C Y \<and> 
                      \<not> A B C TSP X Y \<longrightarrow> A B C OSP X Y" 
proof -
  {
    fix A B C X Y
    assume "\<not> Coplanar A B C X" and
      "\<not> Coplanar A B C Y" and 
      "\<not> A B C TSP X Y"
    obtain P where "P OrthAt A B C P X" 
      using \<open>\<not> Coplanar A B C X\<close> l11_62_existence_bis by blast
    have "\<forall> A B X P Q. \<not> Col P Q X \<and> Per A X P \<and> Per A X Q \<and> Per B X P \<and> Per B X Q \<longrightarrow> Col A B X"
      using assms orthonormal_family_aux by blast 
    have "\<not> Col A B C" 
      using \<open>\<not> Coplanar A B C Y\<close> col__coplanar by auto
    have "P \<noteq> X" 
      using \<open>P OrthAt A B C P X\<close> orth_at_chara by auto
    have "Coplanar A B C P"  using \<open>P OrthAt A B C P X\<close> 
      using orth_at_chara by blast
    have "\<forall> D. Coplanar A B C D \<longrightarrow> Per D P X" 
      using orth_at_chara \<open>P OrthAt A B C P X\<close> by presburger
    obtain X' T where "A B C Orth P X'" and "Coplanar A B C T" and "Bet Y T X'" 
      using l8_21_3 \<open>Coplanar A B C P\<close> \<open>\<not> Coplanar A B C Y\<close> by blast
    hence "P OrthAt A B C P X'" 
      using Col_def \<open>Coplanar A B C P\<close> col_cop_orth__orth_at not_bet_distincts by blast
    have "\<not> Coplanar A B C X'"
      using orth_at__ncop [where ?X = "P"] \<open>P OrthAt A B C P X'\<close> by blast
    have "A B C TSP Y X'" 
      using TSP_def \<open>Bet Y T X'\<close> \<open>Coplanar A B C T\<close> \<open>\<not> Coplanar A B C X'\<close> 
        \<open>\<not> Coplanar A B C Y\<close> by auto
    moreover
    have "Bet X P X'" 
    proof -
      have "Col X' X P" 
        using  \<open>P OrthAt A B C P X\<close> \<open>P OrthAt A B C P X'\<close> 
          orthonormal_family_axiom_implies_orth_at2__col assms by blast
      hence "Col X P X'" 
        using col_permutation_1 by blast
      moreover
      {
        assume "P Out X X'"
        have "A B C TSP X' Y" 
          by (simp add: \<open>A B C TSP Y X'\<close> l9_38)
        moreover have "A B C OSP X' X" 
          using \<open>Coplanar A B C P\<close> \<open>P Out X X'\<close> \<open>\<not> Coplanar A B C X\<close> 
            cop_out__osp osp_symmetry by blast
        ultimately have "A B C TSP X Y" 
          using l9_41_2 by blast
      }
      hence "\<not> P Out X X'" 
        using  \<open>\<not> A B C TSP X Y\<close>  by blast
      ultimately show ?thesis 
        using or_bet_out by blast
    qed
    hence "A B C TSP X X'"
      using TSP_def \<open>Coplanar A B C P\<close> \<open>\<not> Coplanar A B C X'\<close> \<open>\<not> Coplanar A B C X\<close> by blast
    ultimately
    have "A B C OSP X Y" 
      using OSP_def by blast
  }
  thus ?thesis 
    by blast
qed

lemma orthonormal_family_axiom_implies_space_separation:
  assumes "orthonormal_family_axiom"
  shows "space_separation_axiom" 
proof -
  {
    fix A B C P Q
    assume "\<not> Coplanar A B C P" and 
      "\<not> Coplanar A B C Q"
    {
      assume "\<not> A B C TSP P Q"
      hence "A B C OSP P Q" 
        using orthonormal_family_axiom_implies_not_two_sides_one_side 
          \<open>\<not> Coplanar A B C P\<close> \<open>\<not> Coplanar A B C Q\<close> assms by blast
    }
    hence "A B C TSP P Q \<or> A B C OSP P Q" 
      by blast
  }
  thus ?thesis 
    using space_separation_axiom_def by blast
qed

lemma space_separation_implies_plane_intersection:
  assumes "space_separation_axiom"
  shows "plane_intersection_axiom" 
proof -
  {
    fix A B C D E P
    assume "Coplanar A B C P" and 
      "\<not> Col D E P" 
    have "\<exists> Q. Coplanar A B C Q \<and> Coplanar D E P Q \<and> P \<noteq> Q" 
    proof (cases "Coplanar A B C D")
      case True
      thus ?thesis 
        using \<open>\<not> Col D E P\<close> col_trivial_3 ncop_distincts by blast
    next
      case False
      hence "\<not> Coplanar A B C D" 
        by blast
      show ?thesis 
      proof (cases "Coplanar A B C E")
        case True
        thus ?thesis 
          using \<open>\<not> Col D E P\<close> col_trivial_2 ncop_distincts by blast
      next
        case False
        thus ?thesis 
          using \<open>Coplanar A B C P\<close> \<open>\<not> Coplanar A B C D\<close> assms 
            cop_osp__ex_cop2 cop_tsp__ex_cop2 space_separation_axiom_def by fastforce
      qed
    qed
  }
  hence 1: "\<forall> A B C D E P. Coplanar A B C P \<and> \<not> Col D E P \<longrightarrow>
    (\<exists> Q. Coplanar A B C Q \<and> Coplanar D E P Q \<and> P \<noteq> Q)" 
    by blast
  {
    fix A B C D E F P
    assume "Coplanar A B C P" 
      and "Coplanar D E F P"
    obtain D' E' where "Coplanar D E F D'" and "Coplanar D E F E'" and "\<not> Col P D' E'" 
      using ex_ncol_cop2 by blast
    have "\<not> Col D' E' P" 
      by (simp add: \<open>\<not> Col P D' E'\<close> not_col_permutation_1)
    then obtain Q where "Coplanar A B C Q" and "Coplanar D' E' P Q" and "P \<noteq> Q" 
      using 1 \<open>Coplanar A B C P\<close> by blast
    hence "\<exists> Q. Coplanar A B C Q \<and> Coplanar D E F Q \<and> P \<noteq> Q" 
      by (meson \<open>Coplanar D E F D'\<close> \<open>Coplanar D E F E'\<close> \<open>Coplanar D E F P\<close> 
          \<open>\<not> Col D' E' P\<close> l9_30 ncop_distincts)
  }
  thus ?thesis 
    using plane_intersection_axiom_def by blast
qed

lemma plane_intersection_implies_space_separation:
  assumes "plane_intersection_axiom"
  shows "space_separation_axiom" 
proof -
  {
    fix A B C P Q
    assume "\<not> Coplanar A B C P" and "\<not> Coplanar A B C Q"
    have "Coplanar A P Q A" 
      using ncop_distincts by blast
    then obtain D where "Coplanar A B C D" and "Coplanar A P Q D" and "A \<noteq> D"
      using assms plane_intersection_axiom_def coplanar_perm_5 coplanar_trivial by blast
    have "A D TS P Q \<or> A D OS P Q"
    proof (rule cop__one_or_two_sides)
      show "Coplanar A D P Q" 
        by (simp add: \<open>Coplanar A P Q D\<close> coplanar_perm_4)
      show "\<not> Col P A D" 
        by (metis \<open>A \<noteq> D\<close> \<open>Coplanar A B C D\<close> \<open>\<not> Coplanar A B C P\<close> 
            col_cop2__cop col_permutation_3 ncop_distincts)
      show "\<not> Col Q A D" 
        by (meson \<open>A \<noteq> D\<close> \<open>Coplanar A B C D\<close> \<open>\<not> Coplanar A B C Q\<close> 
            col_cop__cop col_permutation_1 ncoplanar_perm_8)
    qed
    moreover
    have "A D TS P Q \<longrightarrow> A B C TSP P Q"
      using \<open>Coplanar A B C D\<close> \<open>\<not> Coplanar A B C P\<close> cop2_ts__tsp ncop_distincts by blast
    moreover have "A D OS P Q \<longrightarrow> A B C OSP P Q"
      using \<open>Coplanar A B C D\<close> \<open>\<not> Coplanar A B C P\<close> cop2_os__osp ncop_distincts by blast
    ultimately have "A B C TSP P Q \<or> A B C OSP P Q" 
      by blast
  }
  thus ?thesis 
    using space_separation_axiom_def by blast
qed

lemma space_separation_implies_median_planes:
  assumes "space_separation_axiom"
  shows "median_planes_axiom" 
proof -
  {
    fix A B C P Q M
    assume "P \<noteq> Q" and 
      "Cong A P A Q" and
      "Cong B P B Q" and
      "Cong C P C Q" and
      "M Midpoint P Q"
    {
      fix X P Q M
      assume "P \<noteq> Q" and 
        "Cong A P A Q" and
        "Cong B P B Q" and
        "M Midpoint P Q" and
        "M A B TSP Q X" and
        "Cong X P X Q"
      have "\<not> Coplanar M A B Q" 
        using \<open>M A B TSP Q X\<close> tsp__ncop1 by blast
      have "\<not> Coplanar M A B X" 
        using \<open>M A B TSP Q X\<close> tsp__ncop2 by auto
      obtain T where "Coplanar M A B T" and "Bet Q T X" 
        using TSP_def \<open>M A B TSP Q X\<close> by blast
      {
        fix C
        assume "Coplanar M A B C"
        have "\<not> Col M A B" 
          using \<open>M A B TSP Q X\<close> ncop__ncol tsp__ncop2 by blast
        moreover have "Cong M P M Q" 
          using midpoint_cong \<open>M Midpoint P Q\<close> cong_left_commutativity by blast
        ultimately have "Cong C P C Q" 
          using \<open>Coplanar M A B C\<close> \<open>Cong A P A Q\<close> \<open>Cong B P B Q\<close> l11_60_aux by blast
      }
      hence "\<forall> C. Coplanar M A B C \<longrightarrow> Cong C P C Q" 
        by blast
      {
        assume "Bet X T P" 
        have "\<not> Coplanar M A B P" 
          by (metis \<open>P \<noteq> Q\<close> \<open>\<forall>C. Coplanar M A B C \<longrightarrow> Cong C P C Q\<close> cong_diff_4)
        have "\<forall> Z. \<not> Coplanar M A B Z \<longrightarrow> Z \<noteq> T" 
          using \<open>Coplanar M A B T\<close> by blast
        have "X \<noteq> T" 
          using \<open>Coplanar M A B T\<close> \<open>\<not> Coplanar M A B X\<close> by blast
        hence "M Out P Q" 
          by (metis \<open>Bet Q T X\<close> \<open>Bet X T P\<close> \<open>Cong X P X Q\<close> \<open>P \<noteq> Q\<close> 
              between_cong between_symmetry l5_1 not_cong_3412)
        moreover
        have "Bet P M Q" 
          using Midpoint_def \<open>M Midpoint P Q\<close> by auto
        ultimately have False 
          using not_bet_and_out by auto
      }
      moreover
      have "Cong T P T Q" 
        using \<open>Coplanar M A B T\<close> \<open>\<forall>C. Coplanar M A B C \<longrightarrow> Cong C P C Q\<close> by blast
      ultimately have False         
        using triangle_strict_inequality \<open>Bet Q T X\<close> \<open>Cong X P X Q\<close>
          between_symmetry cong__nlt by blast
    }
    hence 1: "\<forall> X P Q M. P \<noteq> Q \<and> Cong A P A Q \<and> Cong B P B Q \<and> M Midpoint P Q \<and> 
M A B TSP Q X \<and> Cong X P X Q \<longrightarrow> False" 
      by blast
    {
      assume "\<not> Coplanar M A B C" 
      have "\<not> Col M A B" 
        using \<open>\<not> Coplanar M A B C\<close> ncop__ncols by blast
      {
        assume "Coplanar M A B Q"
        have "Cong M P M Q" 
          using Cong_cases \<open>M Midpoint P Q\<close> midpoint_cong by blast
        hence False 
          using l11_60_aux \<open>Cong A P A Q\<close> \<open>Cong B P B Q\<close> \<open>Coplanar M A B Q\<close> 
            \<open>P \<noteq> Q\<close> \<open>\<not> Col M A B\<close> cong_diff_2 by blast
      }
      hence "\<not> Coplanar M A B Q"
        by blast
      {
        assume "M A B TSP Q C"
        hence False 
          using 1 \<open>Cong A P A Q\<close> \<open>Cong B P B Q\<close> \<open>Cong C P C Q\<close> \<open>M Midpoint P Q\<close> \<open>P \<noteq> Q\<close> by blast
      }
      moreover
      {
        assume "M A B OSP Q C"
        have "M Midpoint Q P" 
          using Mid_cases \<open>M Midpoint P Q\<close> by auto
        have "M A B TSP P C" 
        proof -
          have "M A B TSP Q P" 
            by (metis Bet_cases Midpoint_def \<open>Coplanar M A B Q \<Longrightarrow> False\<close> \<open>M Midpoint P Q\<close> 
                bet_cop__tsp midpoint_distinct_3 ncop_distincts)
          thus ?thesis
            using \<open>M A B OSP Q C\<close> l9_38 l9_41_2 by blast
        qed
        hence False 
          using 1 by (metis \<open>Cong A P A Q\<close> \<open>Cong B P B Q\<close> \<open>Cong C P C Q\<close> 
              \<open>M Midpoint Q P\<close> \<open>P \<noteq> Q\<close> not_cong_3412)
      }
      ultimately have False 
        using \<open>\<not> Coplanar M A B C\<close> \<open>\<not> Coplanar M A B Q\<close> assms 
          space_separation_axiom_def by blast
    }
    hence "Coplanar M A B C" 
      by blast
  }
  thus ?thesis 
    using median_planes_aux by blast
qed

theorem upper_dim_3_equivalent_axioms: 
  shows "(upper_dim_3_axiom        \<longleftrightarrow> orthonormal_family_axiom) \<and>
         (orthonormal_family_axiom \<longleftrightarrow> space_separation_axiom)   \<and>
         (space_separation_axiom   \<longleftrightarrow> plane_intersection_axiom) \<and>
         (plane_intersection_axiom \<longleftrightarrow> median_planes_axiom)" 
  using median_planes_implies_upper_dim orthonormal_family_axiom_implies_space_separation 
    plane_intersection_implies_space_separation space_separation_implies_median_planes 
    space_separation_implies_plane_intersection 
    upper_dim_implies_orthonormal_family_axiom by fastforce

lemma par_reflexivity:
  assumes "A \<noteq> B"
  shows "A B Par A B"
  using Par_def assms not_col_distincts by blast

lemma par_strict_irreflexivity:
  "\<not> A B ParStrict A B"
  using ParStrict_def col_trivial_3 by blast

lemma not_par_strict_id:
  "\<not> A B ParStrict A C"
  using ParStrict_def col_trivial_1 by blast

lemma par_id:
  assumes "A B Par A C"
  shows "Col A B C"
  using Col_cases Par_def assms not_par_strict_id by auto

lemma par_strict_not_col_1:
  assumes "A B ParStrict C D"
  shows "\<not> Col A B C"
  using Col_perm ParStrict_def assms col_trivial_1 by blast

lemma par_strict_not_col_2:
  assumes "A B ParStrict C D"
  shows "\<not> Col B C D"
  using ParStrict_def assms col_trivial_3 by auto

lemma par_strict_not_col_3:
  assumes "A B ParStrict C D"
  shows "\<not> Col C D A"
  using Col_perm ParStrict_def assms col_trivial_1 by blast

lemma par_strict_not_col_4:
  assumes "A B ParStrict C D"
  shows "\<not> Col A B D"
  using Col_perm ParStrict_def assms col_trivial_3 by blast

lemma par_id_1:
  assumes "A B Par A C"
  shows "Col B A C"
  using Par_def assms not_par_strict_id by auto

lemma par_id_2:
  assumes "A B Par A C"
  shows "Col B C A"
  using Col_perm assms par_id_1 by blast

lemma par_id_3:
  assumes "A B Par A C"
  shows "Col A C B"
  using Col_perm assms par_id_2 by blast

lemma par_id_4:
  assumes "A B Par A C"
  shows "Col C B A"
  using Col_perm assms par_id_2 by blast

lemma par_id_5:
  assumes "A B Par A C"
  shows "Col C A B"
  using assms col_permutation_2 par_id by blast

lemma par_strict_symmetry:
  assumes "A B ParStrict C D"
  shows "C D ParStrict A B"
  using ParStrict_def assms coplanar_perm_16 by blast

lemma par_symmetry:
  assumes "A B Par C D"
  shows "C D Par A B"
proof -
  have "A B ParStrict C D \<or> (A \<noteq> B \<and> C \<noteq> D \<and> Col A C D \<and> Col B C D)" 
    using Par_def assms by auto
  thus ?thesis 
    by (metis (full_types) Col_cases Par_def col2__eq par_strict_symmetry)
qed

lemma par_left_comm:
  assumes "A B Par C D"
  shows "B A Par C D"
  by (metis ParStrict_def Par_def assms ncoplanar_perm_6 not_col_permutation_5)

lemma par_right_comm:
  assumes "A B Par C D"
  shows "A B Par D C"
  using assms par_left_comm par_symmetry by blast

lemma par_comm:
  assumes "A B Par C D"
  shows "B A Par D C"
  using assms par_left_comm par_right_comm by blast

lemma par_strict_left_comm:
  assumes "A B ParStrict C D"
  shows "B A ParStrict C D"
  using ParStrict_def assms ncoplanar_perm_6 not_col_permutation_5 by blast

lemma par_strict_right_comm:
  assumes "A B ParStrict C D"
  shows "A B ParStrict D C"
  using assms par_strict_left_comm par_strict_symmetry by blast

lemma par_strict_comm:
  assumes "A B ParStrict C D"
  shows "B A ParStrict D C"
  by (simp add: assms par_strict_left_comm par_strict_right_comm)

lemma par_strict_neq1:
  assumes "A B ParStrict C D"
  shows "A \<noteq> B"
  using assms col_trivial_1 par_strict_not_col_4 by blast

lemma par_strict_neq2:
  assumes "A B ParStrict C D"
  shows "C \<noteq> D"
  using assms col_trivial_2 par_strict_not_col_2 by blast

lemma par_neq1:
  assumes "A B Par C D"
  shows "A \<noteq> B"
  using Par_def assms par_strict_neq1 by blast

lemma par_neq2:
  assumes "A B Par C D"
  shows "C \<noteq> D"
  using assms par_neq1 par_symmetry by blast

lemma Par_cases:
  assumes "A B Par C D \<or> B A Par C D \<or> A B Par D C \<or> B A Par D C \<or> 
           C D Par A B \<or> C D Par B A \<or> D C Par A B \<or> D C Par B A"
  shows "A B Par C D"
  using assms par_right_comm par_symmetry by blast

lemma Par_perm:
  assumes "A B Par C D"
  shows "A B Par C D \<and> B A Par C D \<and> A B Par D C \<and> B A Par D C \<and> 
         C D Par A B \<and> C D Par B A \<and> D C Par A B \<and> D C Par B A"
  using Par_cases assms by blast

lemma Par_strict_cases:
  assumes "A B ParStrict C D \<or> B A ParStrict C D \<or> A B ParStrict D C \<or> 
           B A ParStrict D C \<or> C D ParStrict A B \<or> C D ParStrict B A \<or> 
           D C ParStrict A B \<or> D C ParStrict B A"
  shows "A B ParStrict C D"
  using assms par_strict_right_comm par_strict_symmetry by blast

lemma Par_strict_perm:
  assumes "A B ParStrict C D"
  shows "A B ParStrict C D \<and> B A ParStrict C D \<and> A B ParStrict D C \<and> 
         B A ParStrict D C \<and> C D ParStrict A B \<and> C D ParStrict B A \<and> 
         D C ParStrict A B \<and> D C ParStrict B A"
  using Par_strict_cases assms by blast

lemma l12_6:
  assumes "A B ParStrict C D"
  shows "A B OS C D"
  by (metis Col_def ParStrict_def Par_strict_perm TS_def assms cop_nts__os 
      par_strict_not_col_2)

lemma pars__os3412:
  assumes "A B ParStrict C D"
  shows "C D OS A B"
  by (simp add: assms l12_6 par_strict_symmetry)

lemma perp_dec:
  "A B Perp C D \<or> \<not> A B Perp C D"
  by simp

lemma col_cop2_perp2__col:
  assumes "X1 X2 Perp A B" and
    "Y1 Y2 Perp A B" and
    "Col X1 Y1 Y2" and
    "Coplanar A B X2 Y1" and
    "Coplanar A B X2 Y2"
  shows "Col X2 Y1 Y2"
proof cases
  assume "X1 = Y2"
  thus ?thesis
    using assms(1) assms(2) assms(4) cop_perp2__col not_col_permutation_2 
      perp_left_comm by blast
next
  assume "X1 \<noteq> Y2"
  hence "Y2 X1 Perp A B"
    by (metis Col_cases assms(2) assms(3) perp_col perp_comm perp_right_comm)
  hence P1: "X1 Y2 Perp A B"
    using Perp_perm by blast
  thus ?thesis
  proof cases
    assume "X1 = Y1"
    thus ?thesis
      using assms(1) assms(2) assms(5) cop_perp2__col not_col_permutation_4 by blast
  next
    assume "X1 \<noteq> Y1"
    hence "X1 Y1 Perp A B"
      using Col_cases P1 assms(3) perp_col by blast
    thus ?thesis
      using P1 assms(1) assms(4) assms(5) col_transitivity_2 cop_perp2__col 
        perp_not_eq_1 by blast
  qed
qed

lemma col_perp2_ncol_col:
  assumes "X1 X2 Perp A B" and
    "Y1 Y2 Perp A B" and
    "Col X1 Y1 Y2" and
    "\<not> Col X1 A B"
  shows "Col X2 Y1 Y2"
proof (cases "Y1 = Y2")
  case True
  thus ?thesis 
    using col_trivial_2 by blast
next
  case False
  hence "Y1 \<noteq> Y2"
    by blast
  show ?thesis
  proof (cases "X2 = Y1")
    case True
    thus ?thesis 
      by (simp add: col_trivial_1)
  next 
    case False
    hence "X2 \<noteq> Y1"
      by blast
    show ?thesis 
    proof (cases "X1 = Y1")
      case True
      hence "Coplanar A B X2 Y1" 
        using assms(1) coplanar_perm_17 perp__coplanar by blast
      hence "Coplanar A B X2 Y2" 
        using True assms(2) assms(4) coplanar_perm_18 coplanar_perm_3 
          coplanar_trans_1 perp__coplanar by blast
      thus ?thesis 
        using True assms(1) assms(2) col_permutation_4 cop_perp2__col by blast
    next
      case False
      hence "X1 \<noteq> Y1" 
        by blast
      hence "Y1 X1 Perp A B"
        by (metis Col_cases assms(2) assms(3) perp_col)
      hence "Coplanar A B X2 Y1" 
        by (meson assms(1) assms(4) coplanar_perm_3 coplanar_trans_1 
            ncoplanar_perm_18 perp__coplanar)
      hence "Coplanar A B X2 Y2" 
        by (meson False assms(1) assms(3) col_cop2__cop ncoplanar_perm_22 perp__coplanar)
      thus ?thesis 
        by (metis \<open>Coplanar A B X2 Y1\<close> \<open>Y1 X1 Perp A B\<close> assms(1) assms(2) 
            col_permutation_2 cop_perp2__col not_col_distincts perp_col2)
    qed
  qed
qed

lemma l12_9:
  assumes
    "Coplanar C1 C2 A1 B1" and
    "Coplanar C1 C2 A1 B2" and
    "Coplanar C1 C2 A2 B1" and
    "Coplanar C1 C2 A2 B2" and
    "A1 A2 Perp C1 C2" and
    "B1 B2 Perp C1 C2"
  shows "A1 A2 Par B1 B2"
proof (cases)
  assume "Col A1 B1 B2"
  moreover have "A1 \<noteq> A2"
    using assms(5) perp_distinct by auto
  moreover have "B1 \<noteq> B2"
    using assms(6) perp_distinct by auto
  ultimately show ?thesis
    using Par_def assms(3) assms(4) assms(5) assms(6) col_cop2_perp2__col by blast
next
  assume "\<not> Col A1 B1 B2"
  {
    assume "\<not> Col C1 C2 A1"
    hence "Coplanar A1 A2 B1 B2" 
    proof -
      have "Coplanar C1 C2 A1 A1" 
        using ncop_distincts by blast
      moreover have "C1 C2 Perp A1 A2" 
        using Perp_perm assms(5) by blast
      hence "Coplanar C1 C2 A1 A2" 
        using perp__coplanar by auto
      ultimately show ?thesis 
        by (meson coplanar_pseudo_trans \<open>\<not> Col C1 C2 A1\<close> assms(1) assms(2))
    qed
  }
  moreover
  {
    assume "\<not> Col C1 C2 A2"
    have "Coplanar A1 A2 B1 B2"
    proof -
      have "C1 C2 Perp A2 A1" 
        using Perp_perm assms(5) by blast
      hence "Coplanar C1 C2 A2 A1" 
        using perp__coplanar by auto
      moreover have "Coplanar C1 C2 A2 A2" 
        using ncop_distincts by blast
      ultimately show ?thesis 
        by (meson \<open>\<not> Col C1 C2 A2\<close> assms(3) assms(4) coplanar_perm_5 coplanar_pseudo_trans_lem1)
    qed
  }
  moreover
  {
    assume "\<exists> X. Col X A1 A2 \<and> Col X B1 B2"
    then obtain AB where "Col AB A1 A2" and "Col AB B1 B2" by auto
    have "False"
    proof cases
      assume "AB = A1"
      thus ?thesis
        using \<open>\<not> Col A1 B1 B2\<close> \<open>Col AB B1 B2\<close> by blast
    next
      assume "AB \<noteq> A1"
      hence "A1 AB Perp C1 C2"
        using assms(5) not_col_permutation_2 perp_col by (metis \<open>Col AB A1 A2\<close>)
      hence "AB A1 Perp C1 C2"
        by (simp add: perp_left_comm)
      thus ?thesis
        using assms(1) assms(2) assms(6) col_cop2_perp2__col
          \<open>Col AB B1 B2\<close> \<open>\<not> Col A1 B1 B2\<close> by blast
    qed
  }
  moreover
  have "C1 C2 Perp A1 A2"
    using Perp_cases assms(5) by blast
  hence "\<not> Col C1 C2 A1 \<or> \<not> Col C1 C2 A2" 
    using perp_not_col2 by blast
  ultimately show ?thesis 
    using ParStrict_def Par_def by blast
qed

lemma parallel_existence:
  assumes "A \<noteq> B"
  shows "\<exists> C D. C \<noteq> D \<and> A B Par C D \<and> Col P C D"
proof cases
  assume "Col A B P"
  thus ?thesis
    using Col_perm assms par_reflexivity by blast
next
  assume "\<not> Col A B P"
  then obtain P' where "Col A B P'" and "A B Perp P P'"
    using l8_18_existence by blast
  hence "P \<noteq> P'" 
    using \<open>\<not> Col A B P\<close> by auto
  show ?thesis
  proof cases
    assume "P' = A"
    have "\<exists> Q. Per Q P A \<and> Cong Q P A B \<and> A P OS Q B"
    proof -
      have "Col A P P"
        using not_col_distincts by auto
      moreover have "\<not> Col A P B" 
        using \<open>\<not> Col A B P\<close> not_col_permutation_5 by blast
      ultimately show ?thesis 
        using ex_per_cong not_col_distincts by presburger
    qed
    then obtain Q where "Per Q P A" and "Cong Q P A B" and "A P OS Q B" by auto
    hence "P \<noteq> Q"
      using os_distincts by auto
    moreover have "A B Par P Q" 
    proof -
      have "P Q Perp P A" 
        by (metis Perp_cases \<open>P \<noteq> P'\<close> \<open>P \<noteq> Q\<close> \<open>P' = A\<close> \<open>Per Q P A\<close> per_perp)
      moreover have "Coplanar P A A P"
        using ncop_distincts by auto
      moreover have "Coplanar P A B P"
        using ncop_distincts by auto
      moreover have "Coplanar P A B Q" 
        using \<open>A P OS Q B\<close> coplanar_perm_7 os__coplanar by blast
      moreover have "A B Perp P A" 
        using \<open>A B Perp P P'\<close> \<open>P' = A\<close> by auto
      ultimately show ?thesis 
        using l12_9 ncop_distincts by blast
    qed
    ultimately show ?thesis 
      using col_trivial_1 by auto
  next
    assume "P' \<noteq> A"
    have "\<exists> Q. Per Q P P' \<and> Cong Q P A B \<and> P' P OS Q A" 
    proof -
      have "P' \<noteq> P" 
        using \<open>P \<noteq> P'\<close> by auto
      moreover have "A \<noteq> B"
        by (simp add: assms)
      moreover have "Col P' P P"
        using not_col_distincts by blast
      moreover have "\<not> Col P' P A" 
        using Col_perm \<open>Col A B P'\<close> \<open>\<not> Col A B P\<close> col_transitivity_2 \<open>P' \<noteq> A\<close> by blast
      ultimately show ?thesis
        using ex_per_cong by blast
    qed
    then obtain Q where "Per Q P P'" and "Cong Q P A B" and "P' P OS Q A" by blast
    hence "P \<noteq> Q"
      using os_distincts by blast
    moreover have "A B Par P Q"
    proof -
      have "Coplanar P P' A P"
        using ncop_distincts by auto
      moreover have "Coplanar P P' A Q"
        using ncoplanar_perm_7 os__coplanar \<open>P' P OS Q A\<close> by blast
      moreover 
      have "Coplanar A P B Q" 
        using \<open>Coplanar P P' A Q\<close> col_cop2__cop coplanar_perm_9 
          ncop_distincts ncoplanar_perm_17 not_col_permutation_5 
        by (metis \<open>Col A B P'\<close> \<open>P' \<noteq> A\<close>)
      hence "Coplanar P P' B Q"
        using assms col2_cop__cop col_trivial_2 coplanar_perm_10 coplanar_perm_5 
          calculation(1) by (meson \<open>Col A B P'\<close>)
      moreover have "Coplanar P P' B P"
        using ncop_distincts by auto
      moreover have "P Q Perp P P'"
        using Perp_perm per_perp by (metis \<open>P \<noteq> P'\<close> \<open>P \<noteq> Q\<close> \<open>Per Q P P'\<close>)
      ultimately show ?thesis
        using l12_9 \<open>A B Perp P P'\<close> by blast
    qed
    moreover have "Col P P Q"
      by (simp add: col_trivial_1)
    ultimately show ?thesis
      by blast
  qed
qed

lemma par_col_par:
  assumes "C \<noteq> D'" and
    "A B Par C D" and
    "Col C D D'"
  shows "A B Par C D'"
proof -
  {
    assume P1: "A B ParStrict C D"
    have "Coplanar A B C D'"
      using assms(2) assms(3) col2__eq col2_cop__cop par__coplanar par_neq2 by blast
    {
      assume "\<exists> X. Col X A B \<and> Col X C D'"
      then obtain X where "Col X A B" and "Col X C D'"
        by blast
      hence "Col X C D" 
        using Col_cases assms(1) assms(3) col2__eq by blast
      hence False 
        using P1 ParStrict_def \<open>Col X A B\<close> by auto
    }
    hence "A B Par C D'" 
      using ParStrict_def Par_def \<open>Coplanar A B C D'\<close> by blast
  }
  {
    assume "A \<noteq> B \<and> C \<noteq> D \<and> Col A C D \<and> Col B C D"
    hence "A B Par C D'"
      using Par_def assms(1) assms(3) col2__eq col_permutation_2 by blast
  }
  thus ?thesis
    using Par_def \<open>A B ParStrict C D \<Longrightarrow> A B Par C D'\<close> assms(2) by auto
qed

lemma parallel_existence1:
  assumes "A \<noteq> B"
  shows "\<exists> Q. A B Par P Q"
proof -
  obtain C D where "C \<noteq> D \<and> A B Par C D \<and> Col P C D"
    using assms parallel_existence by blast
  thus ?thesis
    by (metis Col_cases Par_cases par_col_par)
qed

lemma par_not_col:
  assumes "A B ParStrict C D" and
    "Col X A B"
  shows "\<not> Col X C D"
  using ParStrict_def assms(1) assms(2) by blast

lemma not_strict_par1:
  assumes "A B Par C D" and
    "Col A B X" and
    "Col C D X"
  shows "Col A B C"
proof -
  have "A B ParStrict C D \<or> (A \<noteq> B \<and> C \<noteq> D \<and> Col A C D \<and> Col B C D)" 
    using Par_def assms(1) by blast
  thus ?thesis 
    by (meson assms(2) assms(3) col2__eq col_permutation_2 par_not_col)
qed

lemma not_strict_par2:
  assumes "A B Par C D" and
    "Col A B X" and
    "Col C D X"
  shows "Col A B D"
  using Par_cases assms(1) assms(2) assms(3) not_col_permutation_4 
    not_strict_par1 by blast

lemma not_strict_par:
  assumes "A B Par C D" and
    "Col A B X" and      
    "Col C D X"
  shows "Col A B C \<and> Col A B D"
  using assms(1) assms(2) assms(3) not_strict_par1 not_strict_par2 by blast

lemma not_par_not_col:
  assumes "A \<noteq> B" and
    "A \<noteq> C" and
    "\<not> A B Par A C"
  shows "\<not> Col A B C"
  using Par_def assms(1) assms(2) assms(3) not_col_distincts 
    not_col_permutation_4 by blast

lemma not_par_inter_uniqueness:
  assumes "A \<noteq> B" and
    "C \<noteq> D" and
    "\<not> A B Par C D" and
    "Col A B X" and
    "Col C D X" and
    "Col A B Y" and
    "Col C D Y"
  shows "X = Y"
proof cases
  assume P1: "C = Y"
  thus ?thesis
  proof cases
    assume P2: "C = X"
    thus ?thesis
      using P1 by auto
  next
    assume "C \<noteq> X"
    hence "\<not> Col C D A" 
      by (metis P1 Par_def assms(1) assms(2) assms(3) assms(4) assms(5) 
          assms(6) col_permutation_1 col_transitivity_2)
    thus ?thesis 
      using assms(1) assms(4) assms(5) assms(6) assms(7) l6_21 by blast
  qed
next
  assume "C \<noteq> Y"
  hence "\<not> Col A B C" 
    by (metis Par_def assms(1) assms(2) assms(3) assms(6) assms(7) 
        col_transitivity_1 col_transitivity_2 colx not_col_permutation_5)
  thus ?thesis 
    using assms(2) assms(4) assms(5) assms(6) assms(7) l6_21 by blast
qed

lemma inter_uniqueness_not_par:
  assumes  "\<not> Col A B C" and
    "Col A B P" and
    "Col C D P"
  shows "\<not> A B Par C D"
  using assms(1) assms(2) assms(3) not_strict_par1 by blast

lemma col_not_col_not_par:
  assumes "\<exists> P. Col A B P \<and> Col C D P" and
    "\<exists> Q. Col C D Q \<and> \<not>Col A B Q"
  shows "\<not> A B Par C D"
  using assms(1) assms(2) colx not_strict_par par_neq2 by blast

lemma par_distincts:
  assumes "A B Par C D"
  shows "A B Par C D \<and> A \<noteq> B \<and> C \<noteq> D"
  using assms par_neq1 par_neq2 by blast

lemma par_not_col_strict:
  assumes "A B Par C D" and
    "Col C D P" and
    "\<not> Col A B P"
  shows "A B ParStrict C D"
  using Col_cases Par_def assms(1) assms(2) assms(3) col3 by blast

lemma col_cop_perp2__pars:
  assumes "\<not> Col A B P" and
    "Col C D P" and
    "Coplanar A B C D" and
    "A B Perp P Q" and
    "C D Perp P Q"
  shows  "A B ParStrict C D"
proof -
  have P1: "C \<noteq> D"
    using assms(5) perp_not_eq_1 by auto
  hence P2: "Coplanar A B C P"
    using col_cop__cop assms(2) assms(3) by blast
  moreover have P3: "Coplanar A B D P" using col_cop__cop
    using P1 assms(2) assms(3) col2_cop__cop col_trivial_2 by blast
  have "A B Par C D"
  proof -
    have "Coplanar P A Q C"
    proof -
      have "\<not> Col B P A"
        by (simp add: assms(1) not_col_permutation_1)
      moreover have "Coplanar B P A Q"
        by (meson assms(4) ncoplanar_perm_12 perp__coplanar)
      moreover have "Coplanar B P A C"
        using P2 ncoplanar_perm_13 by blast
      ultimately show ?thesis
        using coplanar_trans_1 by auto
    qed
    hence P4: "Coplanar P Q A C"
      using ncoplanar_perm_2 by blast
    have "Coplanar P A Q D"
    proof -
      have "\<not> Col B P A"
        by (simp add: assms(1) not_col_permutation_1)
      moreover have "Coplanar B P A Q"
        by (meson assms(4) ncoplanar_perm_12 perp__coplanar)
      moreover have "Coplanar B P A D"
        using P3 ncoplanar_perm_13 by blast
      ultimately show ?thesis
        using coplanar_trans_1 by blast
    qed
    then moreover have "Coplanar P Q A D"
      using ncoplanar_perm_2 by blast
    moreover have "Coplanar P Q B C"
      using P2 assms(1) assms(4) coplanar_perm_1 coplanar_perm_10 coplanar_trans_1 
        perp__coplanar by blast
    moreover have "Coplanar P Q B D"
      by (meson P3 assms(1) assms(4) coplanar_trans_1 ncoplanar_perm_1 
          ncoplanar_perm_13 perp__coplanar)
    ultimately show ?thesis
      using assms(4) assms(5) l12_9 P4 by auto
  qed
  thus ?thesis
    using assms(1) assms(2) par_not_col_strict by auto
qed

lemma all_one_side_par_strict:
  assumes "C \<noteq> D" and
    "\<forall> P. Col C D P \<longrightarrow> A B OS C P"
  shows "A B ParStrict C D"
proof -
  have P1: "Coplanar A B C D"
    by (simp add: assms(2) col_trivial_2 os__coplanar)
  {
    assume "\<exists> X. Col X A B \<and> Col X C D"
    then obtain X where P2: "Col X A B \<and> Col X C D" by blast
    have "A B OS C X"
      by (simp add: P2 Col_perm assms(2))
    then obtain M where "A B TS C M \<and> A B TS X M"
      by (meson Col_cases P2 col124__nos)
    hence "False"
      using P2 TS_def by blast
  }
  thus ?thesis
    using P1 ParStrict_def by auto
qed

lemma par_col_par_2:
  assumes "A \<noteq> P" and                         
    "Col A B P" and
    "A B Par C D"
  shows "A P Par C D"
  using assms(1) assms(2) assms(3) par_col_par par_symmetry by blast

lemma par_col2_par:
  assumes "E \<noteq> F" and
    "A B Par C D" and
    "Col C D E" and
    "Col C D F"
  shows "A B Par E F"
  by (metis assms(1) assms(2) assms(3) assms(4) col_transitivity_2 not_col_permutation_4 
      par_col_par par_distincts par_right_comm)

lemma par_col2_par_bis:
  assumes "E \<noteq> F" and
    "A B Par C D" and
    "Col E F C" and
    "Col E F D"
  shows "A B Par E F"
  by (metis assms(1) assms(2) assms(3) assms(4) col_transitivity_1 not_col_permutation_2 
      par_col2_par)

lemma par_strict_col_par_strict:
  assumes "C \<noteq> E" and
    "A B ParStrict C D" and
    "Col C D E"
  shows "A B ParStrict C E"
proof -
  have P1: "C E Par A B"
    using Par_def Par_perm assms(1) assms(2) assms(3) par_col_par_2 by blast
  {
    assume "C E ParStrict A B"
    hence "A B ParStrict C E"
      by (metis par_strict_symmetry)
  }
  thus ?thesis
    using Col_cases Par_def P1 assms(2) par_strict_not_col_1 by blast
qed

lemma par_strict_col2_par_strict:
  assumes "E \<noteq> F" and
    "A B ParStrict C D" and
    "Col C D E" and
    "Col C D F"
  shows "A B ParStrict E F"
proof -
  have "Coplanar A B E F" 
    by (metis ParStrict_def Par_strict_perm col2_cop__cop assms(2) assms(3) 
        assms(4) par_strict_neq1)
  moreover 
  {
    assume "\<exists> X. Col X A B \<and> Col X E F"
    then obtain X where "Col X A B" and "Col X E F"
      by blast
    hence "Col X C D" 
      by (meson assms(1) assms(3) assms(4) colx not_col_permutation_2)
    hence False 
      using ParStrict_def \<open>Col X A B\<close> assms(2) by blast
  }
  ultimately show ?thesis 
    using ParStrict_def by blast
qed

lemma line_dec:
  "(Col C1 B1 B2 \<and> Col C2 B1 B2) \<or> \<not> (Col C1 B1 B2 \<and> Col C2 B1 B2)"
  by simp

lemma par_distinct:
  assumes "A B Par C D"
  shows "A \<noteq> B \<and> C \<noteq> D"
  using assms par_neq1 par_neq2 by auto

lemma par_col4__par:
  assumes "E \<noteq> F" and
    "G \<noteq> H" and
    "A B Par C D" and
    "Col A B E" and
    "Col A B F" and
    "Col C D G" and
    "Col C D H"
  shows "E F Par G H"
proof -
  have "C D Par E F"
    using Par_cases assms(1) assms(3) assms(4) assms(5) par_col2_par by blast
  hence "E F Par C D"
    by (simp add: \<open>C D Par E F\<close> par_symmetry)
  thus ?thesis
    using assms(2) assms(6) assms(7) par_col2_par by blast
qed

lemma par_strict_col4__par_strict:
  assumes "E \<noteq> F" and
    "G \<noteq> H" and
    "A B ParStrict C D" and
    "Col A B E" and
    "Col A B F" and
    "Col C D G" and
    "Col C D H"
  shows "E F ParStrict G H"
proof -
  have "C D ParStrict E F"
    using Par_strict_cases assms(1) assms(3) assms(4) assms(5) 
      par_strict_col2_par_strict by blast
  hence "E F ParStrict C D"
    by (simp add: \<open>C D ParStrict E F\<close> par_strict_symmetry)
  thus ?thesis
    using assms(2) assms(6) assms(7) par_strict_col2_par_strict by blast
qed

lemma par_strict_one_side:
  assumes "A B ParStrict C D" and
    "Col C D P"
  shows "A B OS C P"
proof cases
  assume "C = P"
  thus ?thesis
    using assms(1) assms(2) not_col_permutation_5 one_side_reflexivity 
      par_not_col by blast
next
  assume "C \<noteq> P"
  thus ?thesis
    using assms(1) assms(2) l12_6 par_strict_col_par_strict by blast
qed

lemma par_strict_all_one_side:
  assumes "A B ParStrict C D"
  shows "\<forall> P. Col C D P \<longrightarrow> A B OS C P"
  using assms par_strict_one_side by blast

lemma inter_trivial:
  assumes "\<not> Col A B X"
  shows "X Inter A X B X"
  by (metis Col_perm Inter_def assms col_trivial_1)

lemma inter_sym:
  assumes "X Inter A B C D"
  shows "X Inter C D A B"
proof -
  obtain P where P1: "Col P C D \<and> \<not> Col P A B"
    using Inter_def assms by auto
  have P2: "A \<noteq> B"
    using P1 col_trivial_2 by blast
  thus ?thesis
  proof cases
    assume "A = X"
    have "Col B A B"
      by (simp add: col_trivial_3)
    {
      assume P3: "Col B C D"
      have "Col P A B"
      proof -
        have "C \<noteq> D"
          using Inter_def assms by blast
        moreover have "Col C D P"
          using P1 not_col_permutation_2 by blast
        moreover have "Col C D A"
          using Inter_def \<open>A = X\<close> assms by auto
        moreover have "Col C D B"
          using P3 not_col_permutation_2 by blast
        ultimately show ?thesis
          using col3 by blast
      qed
      hence "False"
        by (simp add: P1)
    }
    hence "\<not> Col B C D" by auto
    thus ?thesis
      using Inter_def P2 assms by (meson col_trivial_3)
  next
    assume P5: "A \<noteq> X"
    have P6: "Col A A B"
      using not_col_distincts by blast
    {
      assume P7: "Col A C D"
      have "Col A P X"
      proof -
        have "C \<noteq> D"
          using Inter_def assms by auto
        moreover have "Col C D A"
          using Col_cases P7 by blast
        moreover have "Col C D P"
          using Col_cases P1 by auto
        moreover have "Col C D X"
          using Inter_def assms by auto
        ultimately show ?thesis
          using col3 by blast
      qed
      hence "Col P A B"
        by (metis Col_perm Inter_def P5 assms col_transitivity_2)
      hence "False"
        by (simp add: P1)
    }
    hence "\<not> Col A C D" by auto
    thus ?thesis
      by (meson Inter_def P2 assms col_trivial_1)
  qed
qed

lemma inter_left_comm:
  assumes "X Inter A B C D"
  shows "X Inter B A C D"
  using Col_cases Inter_def assms by auto

lemma inter_right_comm:
  assumes "X Inter A B C D"
  shows "X Inter A B D C"
  by (metis assms inter_left_comm inter_sym)

lemma inter_comm:
  assumes "X Inter A B C D"
  shows "X Inter B A D C"
  using assms inter_left_comm inter_right_comm by blast

(** In the proof given by Tarski on page 125 the distinction of cases is explicit.
This is a good example on the significance of decidability. *)

lemma l12_17:
  assumes "A \<noteq> B" and
    "P Midpoint A C" and
    "P Midpoint B D"
  shows "A B Par C D"
proof cases
  assume P1: "Col A B P"
  thus ?thesis
  proof cases
    assume "A = P"
    thus ?thesis
      using assms(1) assms(2) assms(3) cong_diff_2 is_midpoint_id 
        midpoint_col midpoint_cong not_par_not_col by blast
  next
    assume P2: "A \<noteq> P"
    thus ?thesis
    proof cases
      assume "B = P"
      thus ?thesis
        by (metis assms(1) assms(2) assms(3) midpoint_col midpoint_distinct_2 
            midpoint_distinct_3 not_par_not_col par_comm)
    next
      assume P3: "B \<noteq> P"
      have P4: "Col B P D"
        using assms(3) midpoint_col not_col_permutation_4 by blast
      have P5: "Col A P C"
        using assms(2) midpoint_col not_col_permutation_4 by blast
      hence P6: "Col B C P"
        using P1 P2 col_transitivity_2 not_col_permutation_3 not_col_permutation_5 by blast
      have "C \<noteq> D"
        using assms(1) assms(2) assms(3) l7_9 by blast
      moreover have "Col A C D"
        using P1 P3 P4 P6 col3 not_col_permutation_3 not_col_permutation_5 by blast
      moreover have "Col B C D"
        using P3 P4 P6 col_trivial_3 colx by blast
      ultimately show ?thesis
        by (simp add: Par_def assms(1))
    qed
  qed
next
  assume T1: "\<not> Col A B P"
  then obtain E where T2: "Col A B E \<and> A B Perp P E"
    using l8_18_existence by blast
  have T3: "A \<noteq> P"
    using T1 col_trivial_3 by blast
  thus ?thesis
  proof cases
    assume T4: "A = E"
    hence T5: "Per P A B"
      using T2 l8_2 perp_per_1 by blast
    obtain B' where T6: "Bet B A B' \<and> Cong A B' B A"
      using segment_construction by blast
    obtain D' where T7: "Bet B' P D' \<and> Cong P D' B' P"
      using segment_construction by blast
    have T8: "C Midpoint D D'"
      using T6 T7 assms(2) assms(3) midpoint_def not_cong_3412 
        symmetry_preserves_midpoint by blast
    have "Col A B B'"
      using Col_cases Col_def T6 by blast
    hence T9: "Per P A B'"
      using per_col T5 assms(1) by blast
    obtain B'' where T10: "A Midpoint B B'' \<and> Cong P B P B''"
      using Per_def T5 by auto
    hence "B' = B''"
      using T6 cong_symmetry midpoint_def symmetric_point_uniqueness by blast
    hence "Cong P D P D'"
      by (metis Cong_perm Midpoint_def T10 T7 assms(3) cong_inner_transitivity)
    hence T12: "Per P C D"
      using Per_def T8 by auto
    hence T13: "C PerpAt P C C D"
      by (metis T3 assms(1) assms(2) assms(3) l7_3_2 per_perp_in sym_preserve_diff)
    have T14: "P \<noteq> C"
      using T3 assms(2) is_midpoint_id_2 by auto
    have T15: "C \<noteq> D"
      using assms(1) assms(2) assms(3) l7_9 by auto
    have T15A: "C C Perp C D \<or> P C Perp C D"
      using T12 T14 T15 per_perp by auto
    {
      assume "C C Perp C D"
      hence "A B Par C D"
        using perp_distinct by auto
    }
    {
      assume "P C Perp C D"
      have "A B Par C D"
      proof -
        have "Coplanar P A A C"
          using ncop_distincts by blast
        moreover have "Coplanar P A A D"
          using ncop_distincts by blast
        moreover have "Coplanar P A B C"
          by (simp add: assms(2) coplanar_perm_1 midpoint__coplanar)
        moreover have "Coplanar P A B D"
          using assms(3) midpoint_col ncop__ncols by blast
        moreover have "A B Perp P A"
          using T2 T4 by auto
        moreover have "C D Perp P A"
        proof -
          have "P A Perp C D"
          proof -
            have "P \<noteq> A"
              using T3 by auto
            moreover have "P C Perp C D"
              using T14 T15 T12 per_perp by blast
            moreover have "Col P C A"
              by (simp add: assms(2) l7_2 midpoint_col)
            ultimately show ?thesis
              using perp_col by blast
          qed
          thus ?thesis
            using Perp_perm by blast
        qed
        ultimately show ?thesis using l12_9 by blast
      qed
    }
    thus ?thesis using T15A
      using \<open>C C Perp C D \<Longrightarrow> A B Par C D\<close> by blast
  next
    assume S1B: "A \<noteq> E"
    obtain F where S2: "Bet E P F \<and> Cong P F E P"
      using segment_construction by blast
    hence S2A: "P Midpoint E F"
      using midpoint_def not_cong_3412 by blast
    hence S3: "Col C D F"
      using T2 assms(2) assms(3) mid_preserves_col by blast
    obtain A' where S4: "Bet A E A' \<and> Cong E A' A E"
      using segment_construction by blast
    obtain C' where S5: "Bet A' P C' \<and> Cong P C' A' P"
      using segment_construction by blast
    have S6: "F Midpoint C C'"
      using S4 S5 S2A assms(2) midpoint_def not_cong_3412 
        symmetry_preserves_midpoint by blast
    have S7: "Per P E A"
      using T2 col_trivial_3 l8_16_1 by blast
    have S8: "Cong P C P C'"
    proof -
      have "Cong P C P A"
        using Cong_perm Midpoint_def assms(2) by blast
      moreover have "Cong P A P C'"
      proof -
        obtain A'' where S9: "E Midpoint A A'' \<and> Cong P A P A''"
          using Per_def S7 by blast
        have S10: "A' = A''"
          using Cong_perm Midpoint_def S4 S9 symmetric_point_uniqueness by blast
        hence "Cong P A P A'" using S9 by auto
        moreover have "Cong P A' P C'"
          using Cong_perm S5 by blast
        ultimately show ?thesis
          using cong_transitivity by blast
      qed
      ultimately show ?thesis
        using cong_transitivity by blast
    qed
    hence S9: "Per P F C"
      using S6 Per_def by blast
    hence "F PerpAt P F F C"
      by (metis S2 S2A T1 T2 S1B assms(2) cong_diff_3 l7_9 per_perp_in)
    hence "F PerpAt F P C F"
      using Perp_in_perm by blast
    hence S10: "F P Perp C F \<or> F F Perp C F"
      using l8_15_2 perp_in_col by blast
    {
      assume S11: "F P Perp C F"
      have "Coplanar P E A C"
      proof -
        have "Col P E P \<and> Col A C P"
          using assms(2) col_trivial_3 midpoint_col not_col_permutation_2 by blast
        thus ?thesis
          using Coplanar_def by blast
      qed
      moreover have "Coplanar P E A D"
      proof -
        have "Col P D B \<and> Col E A B"
          using Mid_cases T2 assms(3) midpoint_col not_col_permutation_1 by blast
        thus ?thesis
          using Coplanar_def by blast
      qed
      moreover have "Coplanar P E B C"
        by (metis S1B T2 calculation(1) col2_cop__cop col_transitivity_1 
            ncoplanar_perm_5 not_col_permutation_5)
      moreover have "Coplanar P E B D"
        by (metis S1B T2 calculation(2) col2_cop__cop col_transitivity_1 
            ncoplanar_perm_5 not_col_permutation_5)
      moreover have "P E Perp C D"
      proof -
        have "P \<noteq> E" 
          using T2 perp_distinct by blast
        moreover have "C \<noteq> D"
          using assms(1) assms(2) assms(3) sym_preserve_diff by blast
        moreover have "P F Perp C F"
          using Perp_perm S11 by blast
        hence "F P Perp C F"
          by (simp add: S11)
        moreover have "Col P F E"
          by (simp add: Col_def S2)
        moreover have "Col C F D"
          using Col_perm S3 by blast     
        ultimately show ?thesis 
          using T1 perp_col4 \<open>P F Perp C F\<close> col_trivial_3 by meson
      qed
      moreover hence "C D Perp P E" 
        using Perp_cases by blast
      ultimately have "A B Par C D"
        using T2 l12_9 by blast
    }
    {
      assume "F F Perp C F"
      hence "A B Par C D"
        using perp_distinct by blast
    }
    thus ?thesis
      using S10 \<open>F P Perp C F \<Longrightarrow> A B Par C D\<close> by blast
  qed
qed

lemma l12_18_a:
  assumes "Cong A B C D" and
    "Cong B C D A" and
    "\<not> Col A B C" and
    "B \<noteq> D" and
    "Col A P C" and
    "Col B P D"
  shows "A B Par C D"
proof -
  have "P Midpoint A C \<and> P Midpoint B D"
    using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) l7_21 by blast
  thus ?thesis
    using assms(3) l12_17 not_col_distincts by blast
qed

lemma l12_18_b:
  assumes "Cong A B C D" and
    "Cong B C D A" and
    "\<not> Col A B C" and
    "B \<noteq> D" and
    "Col A P C" and
    "Col B P D"
  shows "B C Par D A"
proof -
  have "P Midpoint A C" 
    using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) l7_21_R1 by blast
  have "Cong B C D A" 
    by (simp add: assms(2))
  moreover have "Cong C D A B" 
    using Cong_cases assms(1) by auto
  moreover 
  {
    assume "Col B C D"
    hence "Col B C P" 
      using NCol_perm assms(4) assms(6) col_trivial_3 colx by blast
    hence "Col C A B" 
    proof -
      {
        assume "C = P"
        hence False 
          using \<open>P Midpoint A C\<close> assms(3) col_trivial_3 is_midpoint_id_2 by blast
      }
      moreover have "Col C P A" 
        using Col_cases assms(5) by blast
      moreover have "Col C P B" 
        using Col_cases \<open>Col B C P\<close> by blast
      ultimately show ?thesis 
        using col_transitivity_1 by blast
    qed
    hence False 
      using NCol_cases assms(3) by blast
  }
  moreover have "C \<noteq> A" 
    using assms(3) not_col_distincts by blast
  moreover have "Col B P D" 
    by (simp add: assms(6))
  moreover have "Col C P A" 
    using Col_cases assms(5) by blast
  ultimately show ?thesis 
    using l12_18_a by blast
qed

lemma l12_18_c:
  assumes "Cong A B C D" and
    "Cong B C D A" and
    "\<not> Col A B C" and
    "B \<noteq> D" and
    "Col A P C" and
    "Col B P D"
  shows "B D TS A C"
proof -
  have "P Midpoint A C \<and> P Midpoint B D"
    using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) l7_21 by blast
  thus ?thesis
  proof -
    have "A C TS B D"
      by (metis Col_cases mid_two_sides \<open>P Midpoint A C \<and> P Midpoint B D\<close> assms(3))
    hence "\<not> Col B D A"
      by (meson Col_cases mid_preserves_col ts__ncol \<open>P Midpoint A C \<and> P Midpoint B D\<close> l7_2)
    thus ?thesis
      by (meson mid_two_sides \<open>P Midpoint A C \<and> P Midpoint B D\<close>)
  qed
qed

lemma l12_18_d:
  assumes "Cong A B C D" and
    "Cong B C D A" and
    "\<not> Col A B C" and
    "B \<noteq> D" and
    "Col A P C" and
    "Col B P D"
  shows "A C TS B D"
  by (metis Col_cases TS_def assms(1) assms(2) assms(3)
      assms(4) assms(5) assms(6) l12_18_c not_col_distincts not_cong_2143 not_cong_4321)

lemma l12_18:
  assumes "Cong A B C D" and
    "Cong B C D A" and
    "\<not> Col A B C" and
    "B \<noteq> D" and
    "Col A P C" and
    "Col B P D"
  shows "A B Par C D \<and> B C Par D A \<and> B D TS A C \<and> A C TS B D"
  using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) 
    l12_18_a l12_18_b l12_18_c l12_18_d by auto

lemma par_two_sides_two_sides:
  assumes "A B Par C D" and
    "B D TS A C"
  shows "A C TS B D"
  by (metis Par_def TS_def assms(1) assms(2) invert_one_side invert_two_sides 
      l12_6 l9_31 not_col_permutation_4 one_side_symmetry os_ts1324__os pars__os3412)

lemma par_one_or_two_sides:
  assumes "A B ParStrict C D"
  shows "(A C TS B D \<and> B D TS A C) \<or> (A C OS B D \<and> B D OS A C)"
proof (cases)
  assume "A C TS B D"
  hence "B D TS A C" 
    by (metis Par_strict_cases assms invert_one_side invert_two_sides l12_6 
        l9_31 os_ts1324__os)
  thus ?thesis
    using \<open>A C TS B D\<close> by blast
next
  assume "\<not> A C TS B D"
  hence "A C OS B D" 
    by (metis Col_perm ParStrict_def assms cop_nos__ts l9_2 ncoplanar_perm_4 
        one_side_symmetry par_strict_not_col_1 par_strict_not_col_3)
  hence "B D OS A C" 
    by (meson assms invert_one_side invert_two_sides l12_6 l9_31 one_side_symmetry 
        os_ts1324__os pars__os3412)
  thus ?thesis 
    using \<open>A C OS B D\<close> by auto
qed

lemma l12_21_b:
  assumes "A C TS B D" and
    "B A C CongA D C A"
  shows "A B Par C D"
proof -
  have P1: "\<not> Col A B C"
    using TS_def assms(1) not_col_permutation_4 by blast
  hence P2: "A \<noteq> B"
    using col_trivial_1 by auto
  have P3: "C \<noteq> D"
    using assms(1) ts_distincts by blast
  then obtain D' where P4: "C Out D D' \<and> Cong C D' A B"
    using P2 segment_construction_3 by blast
  have P5: "B A C CongA D' C A"
  proof -
    have "A Out B B"
      using P2 out_trivial by auto
    moreover have "A Out C C"
      using P1 col_trivial_3 out_trivial by force
    moreover have "C Out D' D"
      by (simp add: P4 l6_6)
    moreover have "C Out A A"
      using P1 not_col_distincts out_trivial by auto
    ultimately show ?thesis
      using assms(2) l11_10 by blast
  qed
  hence P6: "Cong D' A B C"
    using Cong_perm P4 cong_pseudo_reflexivity l11_49 by blast
  have P7: "A C TS D' B"
  proof -
    have "A C TS D B"
      by (simp add: assms(1) l9_2)
    moreover have "Col C A C"
      using col_trivial_3 by auto
    ultimately show ?thesis
      using P4 l9_5 by blast
  qed
  then obtain M where P8: "Col M A C \<and> Bet D' M B"
    using TS_def by blast
  have "B \<noteq> D'"
    using P7 not_two_sides_id by blast
  hence "M Midpoint A C \<and> M Midpoint B D'"
    by (metis Col_cases P1 P4 P6 P8 bet_col l7_21 not_cong_3412)
  hence "A B Par C D'"
    using P2 l12_17 by blast
  thus ?thesis
    by (meson P3 P4 par_col_par l6_6 out_col)
qed

lemma l12_22_aux:
  assumes "P \<noteq> A" and
    "A \<noteq> C" and
    "Bet P A C" and
    "P A OS B D" and
    "B A P CongA D C P"
  shows "A B Par C D"
proof -
  have P1: "P \<noteq> C"
    using CongA_def assms(5) by blast
  obtain B' where P2: "Bet B A B' \<and> Cong A B' B A"
    using segment_construction by blast
  have P3: "P A B CongA C A B'"
    by (metis CongA_def P2 assms(2) assms(3) assms(5) cong_reverse_identity l11_14)
  have P4: "D C A CongA D C P"
    by (metis Col_def assms(2) assms(3) assms(4) bet_out_1 col124__nos l6_6 
        out2__conga out_trivial)
  have P5: "A B' Par C D"
  proof -
    have "\<not> Col B P A"
      using assms(4) col123__nos not_col_permutation_2 by blast
    hence "P A TS B B'"
      by (metis P2 assms(4) bet__ts cong_reverse_identity invert_two_sides 
          not_col_permutation_3 os_distincts)
    hence "A C TS B' D"
      by (meson assms(2) assms(3) assms(4) bet_col bet_col1 col_preserves_two_sides 
          l9_2 l9_8_2)
    moreover have "B' A C CongA D C A"
    proof -
      have "B' A C CongA B A P"
        by (simp add: P3 conga_comm conga_sym)
      moreover have "B A P CongA D C A"
        using P4 assms(5) not_conga not_conga_sym by blast
      ultimately show ?thesis
        using not_conga by blast
    qed
    ultimately show ?thesis
      using l12_21_b by blast
  qed
  have "C D Par A B"
  proof -
    have "A \<noteq> B"
      using assms(4) os_distincts by blast
    moreover have "C D Par A B'"
      using P5 par_symmetry by blast
    moreover have "Col A B' B"
      by (simp add: Col_def P2)
    ultimately show ?thesis
      using par_col_par by blast
  qed
  thus ?thesis
    using Par_cases by blast
qed

lemma l12_22_b:
  assumes "P Out A C" and
    "P A OS B D" and
    "B A P CongA D C P"
  shows "A B Par C D"
proof cases
  assume "A = C"
  thus ?thesis
    using assms(2) assms(3) conga_comm conga_os__out not_par_not_col 
      os_distincts out_col by blast
next
  assume P1: "A \<noteq> C"
  {
    assume "Bet P A C"
    hence "A B Par C D"
      using P1 assms(2) assms(3) conga_diff2 l12_22_aux by blast
  }
  {
    assume P2: "Bet P C A"
    have "C D Par A B"
    proof -
      have "P C OS D B"
        using assms(1) assms(2) col_one_side one_side_symmetry 
          out_col out_diff2 by blast
      moreover have "D C P CongA B A P"
        using assms(3) not_conga_sym by blast
      thus ?thesis
        by (metis P1 P2 assms(1) calculation l12_22_aux out_distinct)
    qed
    hence "A B Par C D"
      using Par_cases by auto
  }
  thus ?thesis
    using Out_def \<open>Bet P A C \<Longrightarrow> A B Par C D\<close> assms(1) by blast
qed

lemma par_strict_par:
  assumes "A B ParStrict C D"
  shows "A B Par C D"
  using Par_def assms by auto

lemma par_strict_distinct:
  assumes "A B ParStrict C D"
  shows " A \<noteq> B \<and> C \<noteq> D"
  using assms par_strict_neq1 par_strict_neq2 by auto

lemma col_par:
  assumes "A \<noteq> B" and
    "B \<noteq> C" and
    "Col A B C"
  shows "A B Par B C"
  by (simp add: Par_def assms(1) assms(2) assms(3) col_trivial_1)

lemma acute_col_perp__out:
  assumes "Acute A B C" and
    "Col B C A'" and
    "B C Perp A A'"
  shows "B Out A' C"
proof -
  {
    assume P1: "\<not> Col B C A"
    then obtain B' where P2: "B C Perp B' B \<and> B C OS A B'"
      using assms(2) l10_15 os_distincts by blast
    have P3: "\<not> Col B' B C"
      using P2 col124__nos col_permutation_1 by blast
    {
      assume "Col B B' A"
      hence "A B C LtA A B C"
        using P2 acute_one_side_aux acute_sym assms(1) one_side_not_col124 by blast
      hence "False"
        by (simp add: nlta)
    }
    hence P4: "\<not> Col B B' A" by auto
    have P5: "B B' ParStrict A A'"
    proof -
      have "B B' Par A A'"
      proof -
        have "Coplanar B C B A"
          using ncop_distincts by blast
        moreover have "Coplanar B C B A'"
          using ncop_distincts by blast
        moreover have "Coplanar B C B' A"
          using P2 coplanar_perm_1 os__coplanar by blast
        moreover have "Coplanar B C B' A'"
          using assms(2) ncop__ncols by auto
        moreover have "B B' Perp B C"
          using P2 Perp_perm by blast
        moreover have "A A' Perp B C"
          using Perp_perm assms(3) by blast
        ultimately show ?thesis
          using l12_9 by auto
      qed
      moreover have "Col A A' A"
        by (simp add: col_trivial_3)
      moreover have "\<not> Col B B' A"
        by (simp add: P4)
      ultimately show ?thesis
        using par_not_col_strict by auto
    qed
    hence P6: "\<not> Col B B' A'"
      using P5 par_strict_not_col_4 by auto
    hence "B B' OS A' C"
    proof -
      have "B B' OS A' A"
        using P5 l12_6 one_side_symmetry by blast
      moreover have "B B' OS A C"
        using P2 acute_one_side_aux acute_sym assms(1) one_side_symmetry by blast
      ultimately show ?thesis
        using one_side_transitivity by blast
    qed
    hence "B Out A' C"
      using Col_cases assms(2) col_one_side_out by blast
  }
  thus ?thesis
    using assms(2) assms(3) perp_not_col2 by blast
qed

lemma acute_col_perp__out_1:
  assumes "Acute A B C" and
    "Col B C A'" and
    "B A Perp A A'"
  shows "B Out A' C"
proof -
  obtain A0 where P1: "Bet A B A0 \<and> Cong B A0 A B"
    using segment_construction by blast
  obtain C0 where P2: "Bet C B C0 \<and> Cong B C0 C B"
    using segment_construction by blast
  have P3: "\<not> Col B A A'"
    using assms(3) col_trivial_2 perp_not_col2 by blast
  have "Bet A' B C0"
  proof -
    have P4: "Col A' B C0"
      using P2 acute_distincts assms(1) assms(2) bet_col col_transitivity_2 
        not_col_permutation_4 by blast
    {
      assume P5: "B Out A' C0"
      have "B Out A A0"
      proof -
        have "Bet C B A'"
          using Bet_perm Col_def P2 P5 assms(2) between_exchange3 
            not_bet_and_out outer_transitivity_between2 by metis
        hence "A B C CongA A0 B A'"
          using P1 P3 acute_distincts assms(1) cong_diff_4 l11_14 not_col_distincts by blast
        hence "Acute A' B A0"
          using acute_conga__acute acute_sym assms(1) by blast
        moreover have "B A0 Perp A' A"
        proof -
          have "B \<noteq> A0"
            using P1 P3 col_trivial_1 cong_reverse_identity by blast
          moreover have "B A Perp A' A"
            using Perp_perm assms(3) by blast
          moreover have "Col B A A0"
            using P1 bet_col not_col_permutation_4 by blast
          ultimately show ?thesis
            using perp_col by blast
        qed
        ultimately show ?thesis
          using Col_cases P1 acute_col_perp__out bet_col by blast
      qed
      hence "False"
        using P1 not_bet_and_out by blast
    }
    thus ?thesis
      using l6_4_2 P4 by blast
  qed
  thus ?thesis
    by (metis P2 P3 acute_distincts assms(1) cong_diff_3 l6_2 not_col_distincts)
qed

lemma conga_inangle_per2__inangle:
  assumes "Per A B C" and
    "T InAngle A B C" and
    "P B A CongA P B C" and
    "Per B P T" and
    "Coplanar A B C P"
  shows "P InAngle A B C"
proof cases
  assume "P = T"
  thus ?thesis
    by (simp add: assms(2))
next
  assume P1: "P \<noteq> T"
  obtain P' where P2: "P' InAngle A B C \<and> P' B A CongA P' B C"
    using CongA_def angle_bisector assms(3) by presburger
  have P3: "Acute P' B A"
    using P2 acute_sym assms(1) conga_inangle_per__acute by blast
  have P4: "\<not> Col A B C"
    using assms(1) assms(3) conga_diff2 conga_diff56 l8_9 by blast
  have P5: "Col B P P'"
  proof -
    have "\<not> B Out A C"
      using Col_cases P4 out_col by blast
    moreover have "Coplanar A B P P'"
    proof -
      have T1: "\<not> Col C A B"
        using Col_perm P4 by blast
      moreover have "Coplanar C A B P"
        using assms(5) ncoplanar_perm_8 by blast
      moreover have "Coplanar C A B P'"
        using P2 inangle__coplanar ncoplanar_perm_21 by blast
      ultimately show ?thesis
        using coplanar_trans_1 by blast
    qed
    moreover have "Coplanar B C P P'"
    proof -
      have "Coplanar A B C P"
        by (meson P2 bet__coplanar calculation(1) calculation(2) col_in_angle_out 
            coplanar_perm_18 coplanar_trans_1 inangle__coplanar l11_21_a l6_6 l6_7 
            not_col_permutation_4 not_col_permutation_5)
      have "Coplanar A B C P'"
        using P2 inangle__coplanar ncoplanar_perm_18 by blast
      thus ?thesis
        using P4 \<open>Coplanar A B C P\<close> coplanar_trans_1 by blast
    qed
    ultimately show ?thesis using conga2_cop2__col P2 assms(3) by blast
  qed
  have "B Out P P'"
  proof -
    have "Acute T B P'"
      using P2 acute_sym assms(1) assms(2) conga_inangle2_per__acute by blast
    moreover have "B P' Perp T P"
      by (metis P1 P5 acute_distincts assms(3) assms(4) calculation 
          col_per_perp conga_distinct l8_2 not_col_permutation_4)
    ultimately show ?thesis
      using Col_cases P5 acute_col_perp__out by blast
  qed
  thus ?thesis
    using Out_cases P2 in_angle_trans inangle_distincts out341__inangle by blast
qed

lemma perp_not_par:
  assumes "A B Perp X Y"
  shows "\<not> A B Par X Y"
proof -
  obtain P where P1: "P PerpAt A B X Y"
    using Perp_def assms by blast
  {
    assume P2: "A B Par X Y"
    {
      assume P3: "A B ParStrict X Y"
      hence "False"
      proof -
        have "Col P A B"
          using Col_perm P1 perp_in_col by blast
        moreover have "Col P X Y"
          using P1 col_permutation_2 perp_in_col by blast
        ultimately show ?thesis
          using P3 par_not_col by blast
      qed
    }
    {
      assume P4: "A \<noteq> B \<and> X \<noteq> Y \<and> Col A X Y \<and> Col B X Y"
      hence "False"
      proof cases
        assume "A = Y"
        thus ?thesis
          using P4 assms not_col_permutation_1 perp_not_col by blast
      next
        assume "A \<noteq> Y"
        thus ?thesis
          using Col_perm P4 Perp_perm assms perp_not_col2 by blast
      qed
    }
    hence "False"
      using Par_def P2 \<open>A B ParStrict X Y \<Longrightarrow> False\<close> by auto
  }
  thus ?thesis by auto
qed

lemma cong_conga_perp:
  assumes "B P TS A C" and
    "Cong A B C B" and
    "A B P CongA C B P"
  shows "A C Perp B P"
proof -
  have P1: " \<not> Col A B P"
    using TS_def assms(1) by blast
  hence P2: "B \<noteq> P"
    using col_trivial_2 by blast
  have P3: "A \<noteq> B"
    using assms(1) ts_distincts by blast
  have P4: "C \<noteq> B"
    using assms(1) ts_distincts by auto
  have P5: "A \<noteq> C"
    using assms(1) not_two_sides_id by auto
  show ?thesis
  proof cases
    assume P6: "Bet A B C"
    hence "Per P B A"
      by (meson conga_comm Tarski_neutral_dimensionless_axioms assms(3) l11_18_2)
    thus ?thesis
      using P2 P3 P5 Per_perm P6 bet_col per_perp perp_col by blast
  next
    assume P7: "\<not> Bet A B C"
    obtain T where P7A: "Col T B P \<and> Bet A T C"
      using TS_def assms(1) by auto
    hence P8: "B \<noteq> T"
      using P7 by blast
    hence P9: "T B A CongA T B C"
      by (meson Col_cases P7A col_conga__conga conga_comm assms(3))
    hence P10: "Cong T A T C"
      using assms(2) cong2_conga_cong cong_reflexivity not_cong_2143 by blast
    hence P11: "T Midpoint A C"
      using P7A midpoint_def not_cong_2134 by blast
    have P12: "Per B T A"
      using P11 Per_def assms(2) not_cong_2143 by blast
    thus ?thesis
    proof -
      have "A C Perp B T"
        by (metis P11 P12 P5 P8 col_per_perp midpoint_col midpoint_distinct_1)
      moreover have "B \<noteq> T"
        by (simp add: P8)
      moreover have "T \<noteq> A"
        using P1 P7A by blast
      moreover have "C \<noteq> T"
        using P10 P5 cong_identity by blast
      moreover have "C \<noteq> A"
        using P5 by auto
      moreover have "Col T A C"
        by (meson P7A bet_col not_col_permutation_4)
      ultimately show ?thesis
        using P2 P7A not_col_permutation_4 perp_col1 by blast
    qed
  qed
qed

lemma perp_inter_exists:
  assumes "A B Perp C D"
  shows "\<exists> P. Col A B P \<and> Col C D P"
proof -
  obtain P where "P PerpAt A B C D"
    using Perp_def assms by auto
  thus ?thesis
    using perp_in_col by blast
qed

lemma perp_inter_perp_in:
  assumes "A B Perp C D"
  shows "\<exists> P. Col A B P \<and> Col C D P \<and> P PerpAt A B C D"
  by (meson Perp_def perp_in_col Tarski_neutral_dimensionless_axioms assms)

lemma cop_npars__inter_exists:
  assumes "Coplanar A1 B1 A2 B2" and
    "\<not> A1 B1 ParStrict A2 B2"
  shows "\<exists> X. Col X A1 B1 \<and> Col X A2 B2" 
  using ParStrict_def assms(1) assms(2) by blast

lemma cop_npar__inter_exists:
  assumes "Coplanar A1 B1 A2 B2" and
    "\<not> A1 B1 Par A2 B2"
  shows "\<exists> X. Col X A1 B1 \<and> Col X A2 B2" 
proof -
  have "\<not> A1 A2 ParStrict A2 B2" 
    using not_par_strict_id par_strict_left_comm by blast
  thus ?thesis 
    using assms(1) assms(2) cop_npars__inter_exists par_strict_par by blast
qed

lemma cop_npar__inter:
  assumes "A1 \<noteq> B1" and
    "A2 \<noteq> B2" and
    "Coplanar A1 B1 A2 B2" and
    "\<not> A1 B1 Par A2 B2"
  shows "\<exists> X. X Inter A1 B1 A2 B2" 
proof -
  obtain X where "Col X A1 B1" and "Col X A2 B2"
    using cop_npar__inter_exists assms(3) assms(4) by blast
  have "\<exists> P. Col P A2 B2 \<and> \<not> Col P A1 B1" 
  proof (cases "Col A2 A1 B1")
    case True
    have "Col B2 A2 B2" 
      by (simp add: col_trivial_3)
    moreover
    have "\<not> Col B2 A1 B1" 
      using Par_cases True assms(1) assms(2) assms(4) calculation col_par 
        not_col_permutation_3 par_col2_par_bis by metis
    ultimately
    show ?thesis 
      by blast
  next
    case False
    thus ?thesis 
      using col_trivial_1 by blast
  qed
  moreover
  have "Col A1 B1 X \<and> Col A2 B2 X" 
    using Col_cases \<open>Col X A1 B1\<close> \<open>Col X A2 B2\<close> by blast
  ultimately
  show ?thesis 
    using assms(2) Inter_def by auto
qed

lemma inter__npar: 
  assumes "X Inter A1 A2 B1 B2"
  shows "\<not> A1 A2 Par B1 B2" 
proof -
  {
    assume "A1 A2 ParStrict B1 B2"
    hence False 
      by (meson Inter_def assms not_strict_par par_strict_not_col_1 par_strict_par)
  }
  moreover
  {
    assume "A1 \<noteq> A2 \<and> B1 \<noteq> B2 \<and> Col A1 B1 B2 \<and> Col A2 B1 B2"
    hence "A1 A2 Par B1 B2" 
      using Par_def by auto
    hence False 
      by (meson Inter_def assms col_not_col_not_par not_col_permutation_2)
  }
  ultimately
  show ?thesis 
    using Par_def by presburger
qed

lemma cong_identity_inv:
  assumes "A \<noteq> B"  
  shows "\<not> Cong A B C C"
  using assms cong_diff by blast

lemma midpoint_midpoint_col:
  assumes "A \<noteq> B" and
    "M Midpoint A A'" and
    "M Midpoint B B'" and
    "Col A B B'"
  shows "A' \<noteq> B' \<and> Col A A' B' \<and> Col B A' B'"
proof -
  have f1: "\<forall>p. p Midpoint p p"
    by (metis l7_3_2)
  hence f2: "B = B' \<or> Col B' A M"
    by (metis (no_types) Col_perm assms(3) assms(4) col_transitivity_2 
        mid_two_sides not_two_sides_id)
  have f3: "\<forall>p. (M = A \<or> Col A p A') \<or> \<not> Col M A p"
    using f1 by (metis (no_types) Col_perm assms(2) col_transitivity_2 
        mid_two_sides not_two_sides_id)
  hence "(M = A \<or> B = B') \<or> Col B A' B'"
    using f1 by (metis Col_perm assms(3) assms(4) col_transitivity_2 
        mid_two_sides not_two_sides_id)
  thus ?thesis
    using f3 f2 f1 by (metis (full_types) Col_perm Mid_perm assms(1) assms(2) 
        assms(3) assms(4) col_transitivity_2 l7_17_bis mid_two_sides not_two_sides_id 
        symmetric_point_uniqueness)
qed

lemma midpoint_par_strict:
  assumes "\<not> Col A B B'" and
    "M Midpoint A A'" and
    "M Midpoint B B'" 
  shows "A B ParStrict A' B'"
  by (metis assms(1) assms(2) assms(3) col_trivial_1 col_trivial_2 l12_17 par_not_col_strict) 

lemma bet3_cong3_bet:
  assumes "A \<noteq> B" and
    "A \<noteq> C" and
    "A \<noteq> D" and
    "Bet D A C" and
    "Bet A C B" and
    "Bet D C D'" and
    "Cong A B C D" and
    "Cong A D B C" and
    "Cong D C C D'"
  shows "Bet C B D'"
proof -
  have "Bet D C B" 
    using assms(2) assms(4) assms(5) outer_transitivity_between2 by blast
  have "B \<noteq> C" 
    using assms(3) assms(8) cong_identity_inv by auto
  have "D' \<noteq> C" 
    using assms(2) assms(4) assms(9) bet_neq23__neq cong_diff by blast
  have "D \<noteq> C" 
    using \<open>D' \<noteq> C\<close> assms(9) cong_reverse_identity by blast
  have "C Out B D'" 
    using Out_def \<open>B \<noteq> C\<close> \<open>Bet D C B\<close> \<open>D \<noteq> C\<close> \<open>D' \<noteq> C\<close> assms(6) l5_2 by auto
  hence "D Out B D'" 
    using \<open>Bet D C B\<close> out_bet__out by blast
  have "D A Le D C" 
    using assms(4) cong_reflexivity l5_5_2 by blast
  hence "D B Le D D'" 
    by (meson \<open>Bet D C B\<close> assms(6) assms(8) assms(9) bet2_le2__le 
        cong_commutativity l5_6 le_reflexivity)
  thus ?thesis 
    by (metis l6_13_1 \<open>Bet D C B\<close> \<open>D Out B D'\<close> between_exchange3)
qed

lemma bet_double_bet:
  assumes "B' Midpoint A B" and 
    "C' Midpoint A C" and
    "Bet A B' C'" 
  shows "Bet A B C" 
proof -
  have "A B' Le A C'"
    by (simp add: assms(3) bet__le1213)
  hence "B' B Le C' C"
    using assms(1) assms(2) l5_6 midpoint_cong by blast 
  have "A B Le A C"
    using \<open>A B' Le A C'\<close> assms(1) assms(2) le_mid2__le13 by auto
  thus ?thesis 
  proof cases
    assume "A = B'"
    thus ?thesis
      using assms(1) between_trivial2 is_midpoint_id by blast 
  next
    assume "A \<noteq> B'"
    thus ?thesis
      by (metis Midpoint_def \<open>A B Le A C\<close> assms(1) assms(2) assms(3) bet_out 
          l6_13_1 l6_6 l6_7 out_diff1) 
  qed
qed

lemma bet_half_bet:
  assumes "Bet A B C" and
    "B' Midpoint A B" and
    "C' Midpoint A C" 
  shows "Bet A B' C'"
proof cases
  assume "A = B"
  thus ?thesis 
    using assms(2) l8_20_2 not_bet_distincts by blast
next
  assume "A \<noteq> B"
  thus ?thesis 
    by (metis midpoint_bet assms(1) assms(2) assms(3) bet_double_bet 
        between_equality_2 between_exchange4 l5_3 l7_17)
qed

lemma midpoint_preserves_bet:
  assumes "B' Midpoint A B" and 
    "C' Midpoint A C"
  shows "Bet A B C \<longleftrightarrow> Bet A B' C'"
  using assms(1) assms(2) bet_double_bet bet_half_bet by blast 

lemma symmetry_preseves_bet1:
  assumes "M Midpoint A A'" and 
    "M Midpoint B B'" and 
    "Bet M A B"
  shows "Bet M A' B'"
  using assms(1) assms(2) assms(3) l7_15 l7_3_2 by blast

lemma symmetry_preseves_bet2:
  assumes "M Midpoint A A'" and 
    "M Midpoint B B'" and
    "Bet M A' B'"
  shows "Bet M A B"
  using Mid_cases assms(1) assms(2) assms(3) symmetry_preseves_bet1 by blast 

lemma symmetry_preserves_bet:
  assumes "M Midpoint A A'" and 
    "M Midpoint B B'" 
  shows "Bet M A' B' \<longleftrightarrow> Bet M A B"
  using assms(1) assms(2) symmetry_preseves_bet1 symmetry_preseves_bet2 by blast 

lemma bet_cong_bet:
  assumes "A \<noteq> B" and
    "Bet A B C" and
    "Bet A B D" and
    "Cong A C B D"
  shows "Bet B C D"
  by (meson assms(1) assms(2) assms(3) assms(4) bet_col between_equality 
      col_cong_bet col_transitivity_1) 

lemma col_cong_mid :
  assumes "A B Par A' B'" and 
    "\<not>  A B ParStrict A' B'" and 
    "Cong A B A' B'"
  shows "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
(M Midpoint A B' \<and> M Midpoint B A'))" 
proof cases
  assume "A = A'"
  thus ?thesis
    by (metis Mid_perm Par_def midpoint_existence assms(1) assms(2) assms(3) l7_20 l7_3_2) 
next
  assume "A \<noteq> A'"
  thus ?thesis 
  proof cases
    assume "B = B'"
    thus ?thesis
      by (metis \<open>A \<noteq> A'\<close> assms(1) assms(3) bet_cong_eq between_trivial 
          l7_3_2 midpoint_def not_cong_1243 not_cong_3412 par_comm par_id_1 third_point) 
  next
    assume "B \<noteq> B'"
    thus ?thesis 
    proof cases
      assume "A = B'"
      thus ?thesis
        by (metis Cong_perm Mid_perm Midpoint_def Par_perm assms(1) assms(3) 
            between_cong_2 between_trivial l7_3_2 midpoint_existence par_id third_point) 
    next
      assume "A \<noteq> B'"
      thus ?thesis 
      proof cases
        assume "A' = B"
        thus ?thesis
          using Par_def \<open>A \<noteq> B'\<close> assms(1) assms(2) assms(3) cong_col_mid l7_3_2 by blast 
      next
        assume "A' \<noteq> B"
        have "Col A B A'"
          using Par_def assms(1) assms(2) col2__eq col_permutation_5 by blast 
        have "Col A B B'"
          using assms(1) assms(2) l8_18_existence not_strict_par2 par_not_col_strict by blast
        {
          assume "Bet A B B'" 
          {
            assume "Bet B A' B'"
            obtain M where "M Midpoint A' B"
              using midpoint_existence by blast
            hence "Cong A' M B M"
              using cong_right_commutativity midpoint_cong by blast 
            have "Bet B M B'"
              using Mid_cases \<open>Bet B A' B'\<close> \<open>M Midpoint A' B\<close> between_exchange4 
                midpoint_bet by blast 
            hence "Bet A M B'"
              using \<open>Bet A B B'\<close> between_exchange2 by blast 
            have "Cong A M B' M" using l2_11
              by (meson Midpoint_def \<open>Bet A B B'\<close> \<open>Bet B A' B'\<close> \<open>M Midpoint A' B\<close> 
                  assms(3) between_inner_transitivity between_symmetry not_cong_3412 
                  not_cong_3421)  
            hence "M Midpoint A B'"
              using \<open>Bet A M B'\<close> midpoint_def not_cong_1243 by blast 
            hence "M Midpoint A B' \<and> M Midpoint B A'"
              using Mid_perm \<open>M Midpoint A' B\<close> by blast 
            hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
    (M Midpoint A B' \<and> M Midpoint B A'))" by auto
          }
          {
            assume "Bet A' B' B"
            obtain M where P2: "M Midpoint B B'"
              using midpoint_existence by blast
            hence "Bet A' M B"
              using \<open>Bet A' B' B\<close> between_exchange4 between_symmetry midpoint_bet by blast 
            hence "Bet M B A"
              using \<open>B \<noteq> B'\<close> \<open>Bet A B B'\<close> \<open>Bet A' B' B\<close> between_inner_transitivity 
                between_symmetry outer_transitivity_between by blast 
            hence "Bet A' M A"
              using \<open>B \<noteq> B'\<close> \<open>Bet A B B'\<close> \<open>Bet A' B' B\<close> \<open>Bet A' M B\<close> between_symmetry 
                outer_transitivity_between by blast 
            have "Cong B M B' M"
              using Cong_perm P2 midpoint_cong by blast 
            hence "Cong A M A' M" 
              using l2_11 by (meson P2 \<open>Bet A' B' B\<close> \<open>Bet M B A\<close> assms(3) 
                  between_exchange3 between_symmetry midpoint_bet) 
            hence "M Midpoint A A'"
              using \<open>Bet A' M A\<close> between_symmetry midpoint_def not_cong_1243 by blast 
            hence "M Midpoint A A' \<and> M Midpoint B B'"
              using \<open>M Midpoint B B'\<close> by blast 
            hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
    (M Midpoint A B' \<and> M Midpoint B A'))" by auto
          }
          {
            assume "Bet B' B A'"
            hence "Bet B A' A"
              by (metis \<open>B \<noteq> B'\<close> \<open>Bet A B B'\<close> assms(3) bet_cong_eq between_symmetry l5_2) 
            obtain M where "M Midpoint A' B"
              using midpoint_existence by blast 
            hence "Bet A' M B"
              by (simp add: midpoint_bet) 
            have "Cong A' M B M"
              using Cong_perm \<open>M Midpoint A' B\<close> midpoint_cong by blast 
            have "M Midpoint A B'" 
            proof -
              have "Bet B M A"
                using Mid_cases \<open>Bet B A' A\<close> \<open>M Midpoint A' B\<close> between_exchange4 
                  midpoint_bet by blast 
              hence "Bet B' B M"
                using \<open>Bet A B B'\<close> between_inner_transitivity between_symmetry by blast 
              have "Bet M A' A"
                using \<open>Bet A' M B\<close> \<open>Bet B A' A\<close> between_inner_transitivity 
                  between_symmetry by blast 
              have "Bet B' M A'"
                using \<open>Bet A' M B\<close> \<open>Bet B' B A'\<close> bet3__bet between_trivial by blast 
              have "Cong A M B' M"
                by (meson \<open>Bet B A' A\<close> \<open>Bet B' B A'\<close> \<open>Bet B' B M\<close> \<open>Bet M A' A\<close> 
                    \<open>Col A B B'\<close> \<open>Cong A' M B M\<close> assms(3) between_symmetry 
                    col_bet2_cong1 l2_11_b) 
              thus ?thesis
                by (metis Bet_cases midpoint_distinct_1 \<open>A' \<noteq> B\<close> \<open>Bet B' M A'\<close> 
                    \<open>Bet M A' A\<close> \<open>M Midpoint A' B\<close> midpoint_def not_cong_1243 
                    outer_transitivity_between) 
            qed
            hence "M Midpoint A B' \<and> M Midpoint B A'"
              by (simp add: \<open>M Midpoint A B'\<close> Mid_perm \<open>M Midpoint A' B\<close>) 
            hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
    (M Midpoint A B' \<and> M Midpoint B A'))" by auto
          }
          have "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
  (M Midpoint A B' \<and> M Midpoint B A'))"
            using Par_def \<open>Bet A' B' B \<Longrightarrow> 
    \<exists>M. M Midpoint A A' \<and> M Midpoint B B' \<or> 
    M Midpoint A B' \<and> M Midpoint B A'\<close> 
              \<open>Bet B A' B' \<Longrightarrow> \<exists>M. M Midpoint A A' \<and> M Midpoint B B' \<or> 
    M Midpoint A B' \<and> M Midpoint B A'\<close> 
              \<open>Bet B' B A' \<Longrightarrow> \<exists>M. M Midpoint A A' \<and> M Midpoint B B' \<or> 
    M Midpoint A B' \<and> M Midpoint B A'\<close> 
              assms(1) assms(2) between_symmetry third_point by blast 
        }
        {
          assume "Bet B B' A" 
          have P3: "Col A A' B'"
            using Par_def assms(1) assms(2) by blast 
          {
            assume "Bet A A' B'"
            hence "A = A' \<and> B = B'"
              using \<open>Bet B B' A\<close> assms(3) bet_cong_eq between_symmetry cong_symmetry by blast 
            hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
    (M Midpoint A B' \<and> M Midpoint B A'))"
              using \<open>B \<noteq> B'\<close> by linarith 
          }
          {
            assume "Bet A' B' A"
            hence "Bet B' B A'"
              by (metis Bet_perm Cong_perm \<open>A \<noteq> B'\<close> \<open>Bet B B' A\<close> assms(3) 
                  bet_cong_eq between_inner_transitivity l5_1) 
            obtain M where "M Midpoint B B'"
              using midpoint_existence by blast
            hence "Bet B M B'"
              using midpoint_bet by blast
            have "M Midpoint A A' \<and> M Midpoint B B'" 
            proof -
              have "Bet A M A'"
                by (meson Bet_perm \<open>Bet A' B' A\<close> \<open>Bet B M B'\<close> \<open>Bet B' B A'\<close> between_exchange4) 
              moreover
              have "Cong M B M B'"
                using Midpoint_def \<open>M Midpoint B B'\<close> not_cong_2134 by blast 
              hence "Cong A M A' M"
                using Bet_cases between_inner_transitivity \<open>Bet B B' A\<close> \<open>Bet B M B'\<close>
                  assms(3) between_exchange2 calculation l4_3 by meson
              ultimately
              show ?thesis
                using Cong_cases \<open>M Midpoint B B'\<close> midpoint_def by blast 
            qed
            hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
    (M Midpoint A B' \<and> M Midpoint B A'))" by auto
          }
          {
            assume "Bet B' A A'"
            obtain M where "M Midpoint A B'"
              using midpoint_existence by blast
            hence "Bet A M B'"
              by (simp add: midpoint_bet)
            have "Cong A M B' M"
              using Cong_perm \<open>M Midpoint A B'\<close> midpoint_cong by blast
            have "M Midpoint A B' \<and> M Midpoint B A'"
            proof -
              have "Bet A' A M"
                using \<open>Bet A M B'\<close> \<open>Bet B' A A'\<close> between_inner_transitivity 
                  between_symmetry by blast 
              have "Bet B M A'"
                using \<open>A \<noteq> B'\<close> \<open>Bet A M B'\<close> \<open>Bet B B' A\<close> \<open>Bet B' A A'\<close> 
                  bet3__bet outer_transitivity_between outer_transitivity_between2 by blast 
              moreover
              have "Cong B M A' M" 
              proof -
                let ?C  = "A"
                let ?C' = "B'"
                have "Bet B M ?C" 
                  by (meson \<open>B \<noteq> B'\<close> \<open>Bet A M B'\<close> \<open>Bet B B' A\<close> bet3__bet l5_1)
                moreover have "Bet A' M ?C'" 
                  using Bet_cases \<open>Bet A M B'\<close> \<open>Bet A' A M\<close> \<open>Bet B' A A'\<close> outer_transitivity_between2 by blast
                moreover have "Cong B ?C A' ?C'" 
                  using assms(3) not_cong_2134 by blast
                moreover have "Cong M ?C M ?C'" 
                  using \<open>Cong A M B' M\<close> not_cong_2143 by blast
                ultimately show ?thesis 
                  using l4_3 by force
              qed
              ultimately
              show ?thesis
                by (simp add: \<open>M Midpoint A B'\<close> cong_right_commutativity midpoint_def) 
            qed
            have "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
    (M Midpoint A B' \<and> M Midpoint B A'))"
              using \<open>M Midpoint A B' \<and> M Midpoint B A'\<close> by blast 
          }
          hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
  (M Midpoint A B' \<and> M Midpoint B A'))"
            using Col_def P3 \<open>Bet A A' B' \<Longrightarrow> \<exists>M. M Midpoint A A' \<and> M Midpoint B B' \<or> 
    M Midpoint A B' \<and> M Midpoint B A'\<close> 
              \<open>Bet A' B' A \<Longrightarrow> \<exists>M. M Midpoint A A' \<and> M Midpoint B B' \<or> 
    M Midpoint A B' \<and> M Midpoint B A'\<close> by blast 
        }
        {
          assume "Bet B' A B" 
          {
            assume "Bet A B A'" 
            hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
    (M Midpoint A B' \<and> M Midpoint B A'))"
              using \<open>A' \<noteq> B\<close> \<open>Bet B' A B\<close> assms(1) assms(3) bet_cong_eq 
                not_cong_1243 outer_transitivity_between2 par_distincts by blast 
          }
          {
            assume "Bet B A' A" 
            obtain M where "M Midpoint A A'"
              using midpoint_existence by blast
            hence "Bet A M A'"
              by (simp add: midpoint_bet)
            have "Cong A M A' M"
              using Cong_perm \<open>M Midpoint A A'\<close> midpoint_cong by blast 
            have "M Midpoint B B'" 
            proof -
              have "Bet B M B'"
                by (meson Bet_perm \<open>Bet A M A'\<close> \<open>Bet B A' A\<close> \<open>Bet B' A B\<close> between_exchange4) 
              moreover
              have "Cong B M B' M"
              proof -
                let ?C  = "A"
                let ?C' = "A'"
                have "Bet B M ?C" 
                  using Bet_cases \<open>Bet A M A'\<close> \<open>Bet B A' A\<close> between_exchange2 by blast
                moreover have "Bet B' M ?C'" 
                  by (metis \<open>Bet A M A'\<close> \<open>Bet B' A B\<close> \<open>M Midpoint A A'\<close> 
                      between_inner_transitivity between_symmetry calculation 
                      midpoint_distinct_3 outer_transitivity_between2)
                moreover have "Cong B ?C B' ?C'" 
                  using assms(3) not_cong_2143 by blast
                moreover have "Cong M ?C M ?C'" 
                  using \<open>Cong A M A' M\<close> cong_commutativity by blast
                ultimately show ?thesis 
                  using l4_3 by blast
              qed
              ultimately
              show ?thesis
                using midpoint_def not_cong_1243 by blast
            qed
            hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
    (M Midpoint A B' \<and> M Midpoint B A'))"
              using \<open>M Midpoint A A'\<close> by blast 
          }
          {
            assume "Bet A' A B" 
            have "Bet A B' A' \<or> Bet A A' B'"
              by (metis \<open>Bet A' A B\<close> \<open>Bet B' A B\<close> assms(1) between_symmetry l5_2 par_distincts)
            { 
              assume "Bet A B' A'"
              obtain M where "M Midpoint A B'"
                using midpoint_existence by presburger
              have "M Midpoint B A'" 
              proof -
                have "Bet B M A'" 
                  using \<open>A \<noteq> B'\<close> \<open>Bet A B' A'\<close> \<open>Bet B' A B\<close> \<open>M Midpoint A B'\<close> 
                    bet3__bet between_symmetry midpoint_bet 
                    outer_transitivity_between by metis
                moreover
                have "Cong A M B' M"
                  using Midpoint_def \<open>M Midpoint A B'\<close> cong_right_commutativity by blast 
                hence "Cong B M A' M"
                  by (meson \<open>A \<noteq> B'\<close> \<open>Bet A B' A'\<close> \<open>Bet B' A B\<close> assms(3) cong_4312 
                      cong_reflexivity cong_symmetry five_segment) 
                ultimately
                show ?thesis
                  using midpoint_def not_cong_1243 by blast 
              qed
              hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
      (M Midpoint A B' \<and> M Midpoint B A'))"
                using \<open>M Midpoint A B'\<close> by blast 
            }
            { 
              assume "Bet A A' B'"
              obtain M where "M Midpoint A A'"
                using midpoint_existence by blast
              have "M Midpoint B B'" 
              proof -
                have "Bet B M B'"
                proof -
                  have "Bet B A B'" 
                    using Bet_cases \<open>Bet B' A B\<close> by blast
                  moreover have "Bet A M B'" 
                    using Midpoint_def \<open>Bet A A' B'\<close> \<open>M Midpoint A A'\<close> between_exchange4 
                    by blast
                  ultimately show ?thesis 
                    using between_exchange2 by blast
                qed
                moreover
                have "Cong A M A' M"
                  using Cong_perm \<open>M Midpoint A A'\<close> midpoint_cong by blast 
                hence "Cong B M B' M"
                  using Cong_cases \<open>A \<noteq> A'\<close> \<open>Bet A A' B'\<close> \<open>Bet A' A B\<close> assms(3) 
                    cong_reflexivity five_segment by blast 
                ultimately
                show ?thesis
                  using midpoint_def not_cong_1243 by blast
              qed
              hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
      (M Midpoint A B' \<and> M Midpoint B A'))"
                using \<open>M Midpoint A A'\<close> by blast 
            }
            hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
    (M Midpoint A B' \<and> M Midpoint B A'))"
              using \<open>Bet A B' A' \<Longrightarrow> \<exists>M. M Midpoint A A' \<and> M Midpoint B B' \<or> 
      M Midpoint A B' \<and> M Midpoint B A'\<close> 
                \<open>Bet A B' A' \<or> Bet A A' B'\<close> by blast 
          }
          hence "\<exists> M. ((M Midpoint A A' \<and> M Midpoint B B') \<or>
  (M Midpoint A B' \<and> M Midpoint B A'))"
            using Col_def \<open>Bet A B A' \<Longrightarrow> \<exists>M. M Midpoint A A' \<and> M Midpoint B B' \<or> 
    M Midpoint A B' \<and> M Midpoint B A'\<close> 
              \<open>Bet B A' A \<Longrightarrow> \<exists>M. M Midpoint A A' \<and> M Midpoint B B' \<or> 
    M Midpoint A B' \<and> M Midpoint B A'\<close> \<open>Col A B A'\<close> by blast 
        }
        thus ?thesis
          using Col_def \<open>Bet A B B' \<Longrightarrow> \<exists>M. M Midpoint A A' \<and> M Midpoint B B' \<or> 
  M Midpoint A B' \<and> M Midpoint B A'\<close> 
            \<open>Bet B B' A \<Longrightarrow> \<exists>M. M Midpoint A A' \<and> M Midpoint B B' \<or> 
  M Midpoint A B' \<and> M Midpoint B A'\<close> \<open>Col A B B'\<close> by blast 
      qed
    qed
  qed
qed

lemma mid_par_cong1:
  assumes "A \<noteq> B" and
    "M Midpoint A A'" and 
    "M Midpoint B B'" 
  shows "Cong A B A' B' \<and> A B Par A' B'"
  by (metis assms(1) assms(2) assms(3) cong_4321 cong_right_commutativity l12_17 
      l7_13_R1 midpoint_distinct_2)

lemma mid_par_cong2:
  assumes "A \<noteq> B'" and 
    "M Midpoint A A'" and
    "M Midpoint B B'" 
  shows "Cong A B' A' B \<and> A B' Par A' B"
  using assms(1) assms(2) assms(3) l7_2 mid_par_cong1 by blast

lemma mid_par_cong:
  assumes "A \<noteq> B" and 
    "A \<noteq> B'" and
    "M Midpoint A A'" and
    "M Midpoint B B'" 
  shows "Cong A B A' B' \<and> Cong A B' A' B \<and> A B Par A' B' \<and> A B' Par A' B"
  using assms(1) assms(2) assms(3) assms(4) mid_par_cong1 mid_par_cong2 by auto

lemma Parallelogram_strict_Parallelogram:
  assumes "ParallelogramStrict A B C D" 
  shows "Parallelogram A B C D"
  by (simp add: Parallelogram_def assms) 

lemma plgf_permut:
  assumes "ParallelogramFlat A B C D"
  shows "ParallelogramFlat B C D A" 
proof cases
  assume "A = B"
  thus ?thesis 
    by (metis ParallelogramFlat_def assms cong_diff_3 not_col_distincts)
next
  assume "A \<noteq> B"
  thus ?thesis
  proof -
    have f1: "A \<noteq> B"
      using \<open>A \<noteq> B\<close> by force
    have f2: "\<forall>p pa pb pc. \<not> Cong p pa pb pc \<or> Cong pc pb pa p"
      using not_cong_4321 by blast
    have "ParallelogramFlat A B C D"
      using assms by force
    then show ?thesis
      using f2 f1 by (smt (z3) NCol_cases ParallelogramFlat_def col_transitivity_2 not_cong_3412)
  qed 
qed

lemma plgf_sym:
  assumes "ParallelogramFlat A B C D"
  shows "ParallelogramFlat C D A B"
  by (simp add: assms plgf_permut) 

lemma plgf_irreflexive:
  shows "\<not> ParallelogramFlat A B A B"
  by (simp add: ParallelogramFlat_def) 

lemma plgs_irreflexive:
  shows "\<not> ParallelogramStrict A B A B"   
proof -
  {
    assume "ParallelogramStrict A B A B"
    hence "A A TS B B \<and> A B Par A B \<and> Cong A B A B" 
      using ParallelogramStrict_def by auto
    hence "False"
      using not_two_sides_id by blast 
  }
  thus ?thesis by blast
qed

lemma plg_irreflexive:
  shows "\<not> Parallelogram A B A B" 
proof -
  {
    assume "Parallelogram A B A B"
    hence "ParallelogramStrict A B A B \<or> ParallelogramFlat A B A B"
      by (simp add: Parallelogram_def) 
    hence "False"
      using plgf_irreflexive plgs_irreflexive by blast 
  }
  thus ?thesis by auto
qed

lemma plgf_mid:
  assumes "ParallelogramFlat A B C D"
  shows "\<exists> M. M Midpoint A C \<and> M Midpoint B D"
proof -
  have P1: "Col A B C \<and> Col A B D \<and> Cong A B C D \<and> Cong A D C B \<and> (A \<noteq> C \<or> B \<noteq> D)"
    using ParallelogramFlat_def assms by blast  
  have "\<exists> M. M Midpoint A C \<and> M Midpoint B D"
  proof cases
    assume "A = B"
    obtain M where "M Midpoint A C"
      using midpoint_existence by blast
    moreover
    have "M Midpoint B D" 
    proof -
      have "Bet A M C"
        by (simp add: calculation midpoint_bet) 
      hence "Bet B M D"
        using P1 \<open>A = B\<close> cong_reverse_identity by blast 
      moreover
      have "Cong B M D M"
        using P1 \<open>A = B\<close> \<open>M Midpoint A C\<close> cong_reverse_identity l7_13 l7_3_2 
          not_cong_4321 by blast 
      ultimately
      show ?thesis
        using Cong_perm midpoint_def by blast 
    qed
    ultimately
    show ?thesis by auto 
  next
    assume "A \<noteq> B"
    hence "C \<noteq> D"
      using P1 cong_identity by blast
    hence "A B Par C D"
      using P1 \<open>A \<noteq> B\<close> par_col2_par par_reflexivity by blast 
    have "\<not> A B ParStrict C D"
      using P1 par_strict_not_col_1 by blast
    then obtain M where P2: "M Midpoint A C \<and> M Midpoint B D  \<or> 
  M Midpoint A D \<and> M Midpoint B C"
      using P1 \<open>A B Par C D\<close> col_cong_mid by blast
    {
      assume "M Midpoint A D" and "M Midpoint B C" and "B \<noteq> C"
      have "Col A B M" 
      proof -
        have "A \<noteq> D" 
          using P1 \<open>B \<noteq> C\<close> cong_reverse_identity by blast
        have "Col A D B" 
          using P1 col_permutation_5 by blast
        moreover have "Col A D M" 
          by (metis Col_def Midpoint_def P2 between_equality_2 calculation 
              col2__eq not_bet_distincts)
        ultimately show ?thesis 
          using \<open>A \<noteq> D\<close> col_transitivity_1 by blast
      qed
      hence False
        by (metis (full_types) Cong_perm Mid_cases P1 \<open>A \<noteq> B\<close> \<open>Col A B M\<close> 
            \<open>M Midpoint A D\<close> \<open>M Midpoint B C\<close> 
            cong_col_mid cong_cong_half_1 l7_9 not_col_permutation_2)
    }
    moreover
    {
      assume "M Midpoint A D" and "M Midpoint B C" and "B = C"
      hence  "\<exists> M. M Midpoint A C \<and> M Midpoint B D" 
        using P1 P2 cong_identity l7_3 by blast 
    }
    ultimately show ?thesis 
      using P2 by auto
  qed
  thus ?thesis 
    by simp
qed

lemma mid_plgs:
  assumes "\<not> Col A B C" and
    "M Midpoint A C" and
    "M Midpoint B D" 
  shows "ParallelogramStrict A B C D" 
proof -
  have "A \<noteq> C"
    using assms(1) col_trivial_3 by auto
  hence "M \<noteq> A"
    using assms(2) is_midpoint_id by auto 
  have "B \<noteq> D"
    using assms(1) assms(2) assms(3) bet_col l8_20_2 midpoint_bet by blast
  hence "M \<noteq> D"
    using assms(3) is_midpoint_id_2 by auto 
  have "Bet A M C"
    by (simp add: assms(2) midpoint_bet)
  have "Bet B M D"
    by (simp add: assms(3) midpoint_bet) 
  thus ?thesis 
  proof -
    have "A C TS B D"
      using NCol_perm assms(1) assms(2) assms(3) mid_two_sides by blast 
    moreover
    have "A B Par C D"
      using assms(1) assms(2) assms(3) mid_par_cong1 not_col_distincts by blast 
    moreover
    have "Cong A B C D"
      using assms(1) assms(2) assms(3) mid_par_cong1 not_col_distincts by blast 
    ultimately
    show ?thesis
      using ParallelogramStrict_def by blast 
  qed
qed

lemma mid_plgf_aux:
  assumes "A \<noteq> C" and
    "Col A B C" and
    "M Midpoint A C" and 
    "M Midpoint B D" 
  shows "ParallelogramFlat A B C D"
proof cases
  assume "B = D"
  thus ?thesis 
    by (metis ParallelogramFlat_def Cong_perm col_trivial_2 midpoint_cong 
        assms(1) assms(2) assms(3) assms(4) l8_20_2)
next
  assume "B \<noteq> D"
  thus ?thesis
  proof -
    have "M Midpoint C A"
      by (simp add: assms(3) l7_2)
    then have "Cong A D C B"
      by (simp add: assms(4) l7_13)
    then show ?thesis
      by (metis Col_cases ParallelogramFlat_def \<open>B \<noteq> D\<close> assms(2,3,4) col_trivial_1 l7_13 l7_2 midpoint_midpoint_col)
  qed 
qed

lemma mid_plgf_1:
  assumes "A \<noteq> C"
    (*"B \<noteq> D" and*)
    "Col A B C" and
    "M Midpoint A C" and 
    "M Midpoint B D"
  shows "ParallelogramFlat A B C D"
  using assms(1) assms(2) assms(3) assms(4) mid_plgf_aux by auto 

lemma mid_plgf_2:
  assumes "B \<noteq> D" 
    "Col A B C" and
    "M Midpoint A C" and 
    "M Midpoint B D"
  shows "ParallelogramFlat A B C D"
  by (metis Col_def assms(1) assms(2) assms(3) assms(4) l7_3 mid_plgf_aux 
      midpoint_bet plgf_permut) 

lemma mid_plgf:
  assumes "A \<noteq> C \<or> B \<noteq> D" 
    "Col A B C" and
    "M Midpoint A C" and 
    "M Midpoint B D"
  shows "ParallelogramFlat A B C D"
  using assms(1) assms(2) assms(3) assms(4) mid_plgf_2 mid_plgf_aux by blast 

lemma mid_plg:
  assumes "A \<noteq> C \<or> B \<noteq> D" and
    "M Midpoint A C" and 
    "M Midpoint B D" 
  shows "Parallelogram A B C D"
  using Parallelogram_def assms(1) assms(2) assms(3) mid_plgf_2 mid_plgf_aux mid_plgs by blast 

lemma midpoint_cong_uniqueness:
  assumes "Col A B C" and
    "M Midpoint A B" and
    "M Midpoint C D" and
    "Cong A B C D" 
  shows "A = C \<and> B = D \<or> A = D \<and> B = C"
proof cases
  assume "A = B"
  thus ?thesis 
    using assms(2) assms(3) assms(4) cong_reverse_identity l7_3 by blast
next
  assume "A \<noteq> B"
  thus ?thesis 
  proof cases
    assume "A = C"
    thus ?thesis 
      using assms(2) assms(3) symmetric_point_uniqueness by blast
  next
    assume "A \<noteq> C"
    have "M Midpoint A C" 
    proof -
      have "Cong A M C M" 
        using assms(2) assms(3) assms(4) cong_cong_half_1 by blast
      moreover have "Col A M C" 
        by (meson l6_16_1 midpoint_col \<open>A \<noteq> B\<close> assms(1) assms(2) col_permutation_2) 
      hence "Bet A M C" 
        by (metis col_trivial_2 cong_right_commutativity \<open>A \<noteq> C\<close> bet_cong_eq 
            calculation col_cong2_bet2 not_bet_distincts third_point)
      ultimately show ?thesis 
        using Cong_cases midpoint_def by blast
    qed
    thus ?thesis 
      by (metis assms(2) assms(3) l7_9_bis)
  qed
qed

lemma plgf_not_comm1:
  assumes "A \<noteq> B" and
    "ParallelogramFlat A B C D" 
  shows  "\<not> ParallelogramFlat A B D C" 
proof -
  obtain M where P1: "M Midpoint A C \<and> M Midpoint B D"
    using assms(2) plgf_mid by blast
  thus ?thesis
    by (metis ParallelogramFlat_def assms(1) l7_2 midpoint_cong_uniqueness 
        midpoint_midpoint_col)
qed

lemma plgf_not_comm2:
  assumes "A \<noteq> B" and
    "ParallelogramFlat A B C D" 
  shows "\<not> ParallelogramFlat B A C D" 
proof -
  have "ParallelogramFlat C D A B"
    by (metis assms(2) plgf_sym)
  thus ?thesis
    by (metis assms(1) plgf_not_comm1 plgf_sym) 
qed

lemma plgf_not_comm:
  assumes "A \<noteq> B" and
    "ParallelogramFlat A B C D" 
  shows "\<not> ParallelogramFlat A B D C \<and> \<not> ParallelogramFlat B A C D"
  by (simp add: assms(1) assms(2) plgf_not_comm1 plgf_not_comm2)

lemma plgf_cong:
  assumes "ParallelogramFlat A B C D"
  shows "Cong A B C D \<and> Cong A D B C"
  using Cong_perm ParallelogramFlat_def assms by blast 

lemma plg_to_parallelogram:
  assumes "Plg A B C D" 
  shows "Parallelogram A B C D"
proof -
  obtain M where "M Midpoint A C \<and> M Midpoint B D"
    using Plg_def assms by auto
  thus ?thesis
    using Plg_def assms mid_plg by auto
qed

lemma plgs_one_side:
  assumes "ParallelogramStrict A B C D"
  shows "A B OS C D \<and> C D OS A B" 
proof -
  have "A C TS B D \<and> A B Par C D \<and> Cong A B C D"
    using ParallelogramStrict_def assms by auto
  show ?thesis using l12_6
    by (simp add: \<open>A C TS B D \<and> A B Par C D \<and> Cong A B C D\<close> 
        \<open>\<And>D C Ba A. A Ba ParStrict C D \<Longrightarrow> A Ba OS C D\<close> 
        Par_perm invert_two_sides l9_2 par_two_sides_two_sides ts_ts_os)
qed

lemma parallelogram_strict_not_col:
  assumes "ParallelogramStrict A B C D"
  shows "\<not> Col A B C"
  by (metis assms one_side_not_col123 plgs_one_side) 

lemma parallelogram_strict_not_col_2: 
  assumes "ParallelogramStrict A B C D"
  shows "\<not> Col B C D"
  using Col_perm assms col124__nos plgs_one_side by blast 

lemma parallelogram_strict_not_col_3:
  assumes "ParallelogramStrict A B C D"
  shows "\<not> Col C D A"
  using assms col123__nos plgs_one_side by blast 

lemma parallelogram_strict_not_col_4:
  assumes "ParallelogramStrict A B C D"
  shows "\<not> Col A B D"
  by (metis (no_types) assms col124__nos plgs_one_side) 

lemma plgs__pars:
  assumes "ParallelogramStrict A B C D" 
  shows "A B ParStrict C D" 
proof -
  have "A C TS B D \<and> A B Par C D \<and> Cong A B C D"
    using ParallelogramStrict_def assms by auto
  thus ?thesis
    using Par_def assms parallelogram_strict_not_col_2 by auto
qed

lemma plgs_sym:
  assumes "ParallelogramStrict A B C D"
  shows "ParallelogramStrict C D A B" 
proof -
  have "A C TS B D \<and> A B Par C D \<and> Cong A B C D"
    using ParallelogramStrict_def assms by auto
  thus ?thesis
    by (simp add: \<open>A C TS B D \<and> A B Par C D \<and> Cong A B C D\<close> 
        Cong_perm Par_perm ParallelogramStrict_def invert_two_sides l9_2) 
qed

lemma plg_sym:
  assumes "Parallelogram A B C D"
  shows "Parallelogram C D A B"
  using Parallelogram_def assms plgf_sym plgs_sym by blast 

lemma Rhombus_Plg:
  assumes "Rhombus A B C D" 
  shows "Plg A B C D"
  using Rhombus_def assms by auto 

lemma Rectangle_Plg:
  assumes "Rectangle A B C D"
  shows "Plg A B C D"
  using Rectangle_def assms by auto 

lemma Rectangle_Parallelogram:
  assumes "Rectangle A B C D" 
  shows "Parallelogram A B C D"
  using Rectangle_def assms plg_to_parallelogram by auto 

lemma plg_cong_rectangle:
  assumes "Plg A B C D" and
    "Cong A C B D" 
  shows "Rectangle A B C D"
  by (simp add: Rectangle_def assms(1) assms(2))

lemma plg_trivial:
  assumes "A \<noteq> B" 
  shows "Parallelogram A B B A"
  by (simp add: ParallelogramFlat_def Parallelogram_def assms col_trivial_2 
      col_trivial_3 cong_pseudo_reflexivity cong_trivial_identity) 

lemma plg_trivial1: 
  assumes "A \<noteq> B" 
  shows "Parallelogram A A B B"
  by (simp add: ParallelogramFlat_def Parallelogram_def assms col_trivial_1 
      cong_pseudo_reflexivity cong_trivial_identity) 

lemma col_not_plgs:
  assumes "Col A B C" 
  shows "\<not>ParallelogramStrict A B C D"
  using assms parallelogram_strict_not_col by auto 

lemma plg_col_plgf: 
  assumes "Col A B C" and
    "Parallelogram A B C D" 
  shows "ParallelogramFlat A B C D"
  using Parallelogram_def assms(1) assms(2) parallelogram_strict_not_col by blast 

lemma plg_bet1: 
  assumes "Parallelogram A B C D" and
    "Bet A C B" 
  shows "Bet D A C" 
proof -
  have "Col A B C"
    by (meson Bet_perm assms(2) bet_col1 between_exchange2 point_construction_different)
  moreover
  have " ParallelogramStrict A B C D \<or> ParallelogramFlat A B C D"
    using Parallelogram_def assms(1) by auto 
  ultimately 
  show ?thesis
    by (metis (no_types) Bet_perm ParallelogramFlat_def assms(2) col_cong2_bet2 
        col_not_plgs plgf_cong)
qed

lemma plgf_trivial1: 
  assumes "A \<noteq> B"
  shows "ParallelogramFlat A B B A"
  by (meson ParallelogramFlat_def not_col_distincts assms cong_pseudo_reflexivity 
      cong_trivial_identity) 

lemma plgf_trivial2:
  assumes "A \<noteq> B" 
  shows "ParallelogramFlat A A B B"
  by (meson assms plgf_permut plgf_trivial1) 

lemma plgf_not_point: 
  assumes "ParallelogramFlat A A B B"
  shows "A \<noteq> B"
  using assms plgf_irreflexive by blast 

lemma plgf_trivial_neq:
  assumes "ParallelogramFlat A A C D"
  shows "C = D \<and> A \<noteq> C"
  using assms cong_reverse_identity plgf_cong plgf_irreflexive by blast 

lemma plgf_trivial_trans: 
  assumes "ParallelogramFlat A A B B" and
    "ParallelogramFlat B B C C" 
  shows "ParallelogramFlat A A C C \<or> A = C"
  using plgf_trivial2 by auto 

lemma plgf_trivial:
  assumes "A \<noteq> B" 
  shows "ParallelogramFlat A B B A"
  using assms plgf_trivial1 by auto 

lemma plgf3_mid:
  assumes "ParallelogramFlat A B A C"
  shows "A Midpoint B C"
  by (metis assms midpoint_distinct_3 plgf_mid) 

lemma cong3_id: 
  assumes "A \<noteq> B" and
    "Col A B C" and 
    "Col A B D" and 
    "Cong A B C D" and
    "Cong A D B C" and 
    "Cong A C B D" 
  shows "A = D \<and> B = C \<or> A = C \<and> B = D"
proof cases
  assume "A = C"
  thus ?thesis 
    using assms(6) cong_reverse_identity by presburger
next
  assume "A \<noteq> C"
  have "\<exists> M. M Midpoint A B \<and> M Midpoint C D \<or> M Midpoint A D \<and> M Midpoint C B" 
  proof -
    have "A C Par B D" 
      by (metis col3 col_trivial_3 cong_identity par_col2_par par_reflexivity 
          \<open>A \<noteq> C\<close> assms(1) assms(2) assms(3) assms(6) not_col_permutation_5)
    moreover have "\<not> A C ParStrict B D" 
      using Col_cases assms(3) par_strict_not_col_3 by blast
    ultimately show ?thesis 
      using assms(6) col_cong_mid by blast
  qed
  then obtain M where "M Midpoint A B \<and> M Midpoint C D \<or> 
  M Midpoint A D \<and> M Midpoint C B"
    by blast
  {
    assume "M Midpoint A B \<and> M Midpoint C D"
    hence "A = D \<and> B = C \<or> A = C \<and> B = D" 
      using assms(2) assms(4) midpoint_cong_uniqueness by blast
  }
  moreover
  {
    assume "M Midpoint A D \<and> M Midpoint C B"
    hence "A = D \<and> B = C \<or> A = C \<and> B = D" 
      by (metis assms(1) assms(3) assms(5) l7_2 midpoint_cong_uniqueness 
          not_col_permutation_5)
  }
  ultimately show ?thesis 
    using \<open>M Midpoint A B \<and> M Midpoint C D \<or> M Midpoint A D \<and> M Midpoint C B\<close> 
    by fastforce
qed

lemma col_cong_mid1:
  assumes "A \<noteq> D" and 
    "Col A B C" and
    "Col A B D" and
    "Cong A B C D" and
    "Cong A C B D"
  shows "\<exists> M. M Midpoint A D \<and> M Midpoint B C" 
proof -
  have "\<exists> M.(M Midpoint A C \<and> M Midpoint B D \<or> 
  M Midpoint A D \<and> M Midpoint B C)" 
  proof cases
    assume "A = B"
    thus ?thesis
      using assms(4) cong_reverse_identity midpoint_existence by blast 
  next
    assume "A \<noteq> B"
    have "C \<noteq> D" 
      using \<open>A \<noteq> B\<close> assms(4) cong_identity by blast
    hence "A B Par C D" 
      by (meson \<open>A \<noteq> B\<close> assms(2) assms(3) not_col_distincts not_par_inter_uniqueness)
    moreover have "\<not> A B ParStrict C D" 
      using assms(3) par_strict_not_col_4 by blast
    ultimately show ?thesis 
      using assms(4) col_cong_mid by auto
  qed
  then obtain M where P3: "(M Midpoint A C \<and> M Midpoint B D) \<or>
  (M Midpoint A D \<and> M Midpoint B C)" by auto
  {
    assume "M Midpoint A C \<and> M Midpoint B D"
    hence "M Midpoint A D \<and> M Midpoint B C"
      using assms(1) assms(2) assms(5) col_permutation_5 midpoint_cong_uniqueness by blast 
  }
  thus ?thesis
    using P3 by blast 
qed

lemma col_cong_mid2: 
  assumes "A \<noteq> C" and
    "Col A B C" and
    "Col A B D" and
    "Cong A B C D" and
    "Cong A D B C" 
  shows "\<exists> M. M Midpoint A C \<and> M Midpoint B D"
  by (simp add: assms(1) assms(2) assms(3) assms(4) assms(5) col_cong_mid1 
      cong_right_commutativity) 

lemma plgs_not_col:
  assumes "ParallelogramStrict A B C D"
  shows "\<not> Col A B C \<and> \<not> Col B C D \<and> \<not> Col C D A \<and> \<not> Col A B D"
  using assms col_not_plgs parallelogram_strict_not_col_2 parallelogram_strict_not_col_3 
    parallelogram_strict_not_col_4 by blast 

lemma not_col_sym_not_col:
  assumes "\<not> Col A B C" and
    "A Midpoint B B'"
  shows "\<not> Col A B' C"
  by (metis assms(1) assms(2) col_transitivity_1 midpoint_col midpoint_distinct_2
      not_col_permutation_5) 

lemma plg_existence:
  assumes "A \<noteq> B" 
  shows "\<exists> D. Parallelogram A B C D" 
proof -
  obtain M where "M Midpoint A C"
    using midpoint_existence by auto
  thus ?thesis
    by (metis assms l7_17_bis l7_3_2 mid_plg symmetric_point_construction)
qed

lemma plgs_diff:
  assumes "ParallelogramStrict A B C D" 
  shows "ParallelogramStrict A B C D \<and> A \<noteq> B \<and> B \<noteq> C \<and> C \<noteq> D \<and> D \<noteq> A \<and> A \<noteq> C \<and> B \<noteq> D"
  using assms not_col_distincts parallelogram_strict_not_col_2 
    parallelogram_strict_not_col_3 parallelogram_strict_not_col_4 by blast 

lemma sym_par:
  assumes "A \<noteq> B" and
    "M Midpoint A A'" and 
    "M Midpoint B B'" 
  shows "A B Par A' B'" 
  using l12_17 assms(1) assms(2) assms(3) by auto 

lemma symmetry_preserves_two_sides:
  assumes "Col X Y M" and
    "X Y TS A B" and
    "M Midpoint A A'" and
    "M Midpoint B B'"
  shows "X Y TS A' B'" 
proof -
  have "Bet A M A'"
    by (simp add: assms(3) midpoint_bet) 
  have "Bet B M B'"
    by (simp add: assms(4) midpoint_bet)
  have "X \<noteq> Y"
    using assms(2) ts_distincts by blast 
  have "\<not> Col A X Y"
    using TS_def assms(2) by blast 
  have "\<not> Col B X Y"
    using TS_def assms(2) by blast 
  have "A \<noteq> M"
    using Col_perm \<open>\<not> Col A X Y\<close> assms(1) by blast
  hence "A' \<noteq> M"
    using assms(3) cong_identity midpoint_cong by blast 
  have "B \<noteq> M"
    using Col_perm \<open>\<not> Col B X Y\<close> assms(1) by blast
  hence "B' \<noteq> M"
    using assms(4) midpoint_not_midpoint by blast 
  have "X Y TS A A'"
    using NCol_cases \<open>A' \<noteq> M\<close> \<open>\<not> Col A X Y\<close> assms(1) assms(3) colx 
      l9_18_R2 midpoint_bet midpoint_col by meson
  have "\<not> Col B' X Y"
    using \<open>B' \<noteq> M\<close> \<open>Bet B M B'\<close> assms(1) bet_col bet_col1 \<open>\<not> Col B X Y\<close> 
      l6_21 not_col_permutation_3 not_col_permutation_4 by metis
  hence "X Y TS B B'"
    using TS_def \<open>\<not> Col B X Y\<close> \<open>Bet B M B'\<close> assms(1) not_col_permutation_1 by blast 
  moreover
  have "X Y OS A' B"
    by (meson \<open>X Y TS A A'\<close> assms(2) l9_2 l9_8_1) 
  ultimately
  show ?thesis
    using l9_8_2 one_side_symmetry by blast 
qed

lemma symmetry_preserves_one_side:
  assumes "Col X Y M" and
    "X Y OS A B" and
    "M Midpoint A A'" and
    "M Midpoint B B'"
  shows "X Y OS A' B'" 
proof -
  obtain P where "X Y TS A P" and "X Y TS B P"
    using OS_def assms(2) by blast
  have "X \<noteq> Y"
    using assms(2) os_distincts by blast 
  have "\<not> Col A X Y" 
    using TS_def \<open>X Y TS A P\<close> by blast
  have "\<not> Col B X Y"
    using assms(2) between_trivial one_side_chara by blast 
  have "\<not> Col P X Y" 
    using TS_def \<open>X Y TS A P\<close> by blast
  have "A \<noteq> M"
    using assms(1) assms(2) col123__nos by blast
  hence "A' \<noteq> M"
    using assms(3) midpoint_not_midpoint by blast 
  have "B \<noteq> M"
    using assms(1) assms(2) col124__nos by blast
  hence "B' \<noteq> M"
    using assms(4) midpoint_not_midpoint by blast
  have "X Y TS A A'"
  proof -
    {
      assume "Col A' X Y" 
      have "Col A M A'" 
        using Col_def Midpoint_def assms(3) by auto
      have "Col M A' X" 
        using \<open>Col A' X Y\<close> \<open>X \<noteq> Y\<close> assms(1) col_permutation_2 col_trivial_2 colx by blast
      hence "Col A X Y" 
        by (metis \<open>A' \<noteq> M\<close> \<open>Col A M A'\<close> \<open>Col A' X Y\<close> assms(1) col_permutation_3 
            col_permutation_4 col_trivial_2 l6_21)
      hence False 
        using \<open>\<not> Col A X Y\<close> by auto
    }
    moreover have "\<exists> T. Col T X Y \<and> Bet A T A'" 
    proof -
      have "Col M X Y" 
        using assms(1) not_col_permutation_1 by blast
      moreover have "Bet A M A'" 
        by (simp add: assms(3) midpoint_bet)
      ultimately show ?thesis 
        by blast
    qed
    ultimately show ?thesis 
      using TS_def \<open>\<not> Col A X Y\<close> by blast
  qed
  moreover
  have "X Y TS B B'"
    by (meson \<open>B' \<noteq> M\<close> \<open>\<not> Col B X Y\<close> assms(1) assms(4) colx l9_18_R2 
        midpoint_bet midpoint_col not_col_permutation_1)
  ultimately
  show ?thesis
    by (meson assms(2) l9_2 l9_8_1 l9_8_2) 
qed

lemma plgf_bet:
  assumes "ParallelogramFlat A B B' A'"
  shows "Bet A' B' A \<and> Bet B' A B 
  \<or> Bet A' A B' \<and> Bet A B' B
  \<or> Bet A A' B \<and> Bet A' B B'
  \<or> Bet A B A' \<and> Bet B A' B'" 
proof -
  let ?t = "Bet A' B' A \<and> Bet B' A B 
  \<or> Bet A' A B' \<and> Bet A B' B
  \<or> Bet A A' B \<and> Bet A' B B'
  \<or> Bet A B A' \<and> Bet B A' B'" 
  have "Col A B B' \<and> Col A B A' \<and> Cong A B B' A' \<and> Cong A A' B' B \<and> (A \<noteq> B' \<or> B \<noteq> A')"
    using ParallelogramFlat_def assms by blast
  show ?thesis
  proof cases
    assume "A = B"
    thus ?thesis
      using \<open>Col A B B' \<and> Col A B A' \<and> Cong A B B' A' \<and> Cong A A' B' B \<and> 
    (A \<noteq> B' \<or> B \<noteq> A')\<close> between_trivial between_trivial2 
        cong_reverse_identity by blast 
  next
    assume "A \<noteq> B"
    have "A' \<noteq> B'" 
      using \<open>A \<noteq> B\<close> assms plgf_not_comm1 by blast
    have "Col A' B' A" 
      by (meson \<open>A \<noteq> B\<close> 
          \<open>Col A B B' \<and> Col A B A' \<and> Cong A B B' A' \<and> Cong A A' B' B \<and> (A \<noteq> B' \<or> B \<noteq> A')\<close> 
          col_transitivity_1 not_col_permutation_3)
    have "Col A' B' B" 
      by (meson \<open>A \<noteq> B\<close> 
          \<open>Col A B B' \<and> Col A B A' \<and> Cong A B B' A' \<and> Cong A A' B' B \<and> (A \<noteq> B' \<or> B \<noteq> A')\<close> 
          col_transitivity_2 not_col_permutation_3)
    {
      assume "Bet A B A'"
      hence ?t 
        using \<open>Col A B B' \<and> Col A B A' \<and> Cong A B B' A' \<and> Cong A A' B' B \<and> (A \<noteq> B' \<or> B \<noteq> A')\<close> 
          col_cong2_bet4 cong_right_commutativity by blast
    }
    moreover
    {
      assume "Bet B A' A"
      hence ?t 
        by (metis Bet_cases 
            \<open>Col A B B' \<and> Col A B A' \<and> Cong A B B' A' \<and> Cong A A' B' B \<and> (A \<noteq> B' \<or> B \<noteq> A')\<close> 
            \<open>Col A' B' A\<close> col_cong2_bet4 cong_right_commutativity not_col_permutation_1)
    }
    moreover
    {
      assume "Bet A' A B"
      {
        assume "Bet A B B'"
        hence ?t 
          by (metis bet_cong_eq outer_transitivity_between2 \<open>A \<noteq> B\<close> \<open>Bet A' A B\<close> 
              \<open>Col A B B' \<and> Col A B A' \<and> Cong A B B' A' \<and> Cong A A' B' B \<and> (A \<noteq> B' \<or> B \<noteq> A')\<close> 
              cong_right_commutativity)
      }
      moreover
      {
        assume "Bet B B' A"
        hence ?t 
          by (metis between_symmetry \<open>Bet A' A B\<close> between_inner_transitivity)
      }
      moreover
      {
        assume "Bet B' A B"
        {
          assume "A = A'"
          hence ?t 
            using \<open>Bet B A' A \<Longrightarrow> 
          Bet A' B' A \<and> Bet B' A B \<or> 
          Bet A' A B' \<and> Bet A B' B \<or> 
          Bet A A' B \<and> Bet A' B B' \<or> Bet A B A' \<and> Bet B A' B'\<close> 
              not_bet_distincts by blast
        }
        moreover
        {
          assume "A \<noteq> A'"
          have "Bet A B' A' \<or> Bet A A' B'" 
            by (metis \<open>A \<noteq> B\<close> \<open>Bet A' A B\<close> \<open>Bet B' A B\<close> between_symmetry l5_2)
          hence "Bet A' B' A" 
            by (metis Cong_cases \<open>A \<noteq> B\<close> \<open>Bet A' A B\<close> \<open>Bet B' A B\<close> 
                \<open>Col A B B' \<and> Col A B A' \<and> Cong A B B' A' \<and> Cong A A' B' B \<and> (A \<noteq> B' \<or> B \<noteq> A')\<close> 
                bet_cong_bet between_symmetry)
          hence ?t 
            using \<open>Bet B' A B\<close> by blast
        }
        hence ?t 
          using calculation by fastforce
      }
      moreover
      have ?t 
        using Col_def 
          \<open>Col A B B' \<and> Col A B A' \<and> Cong A B B' A' \<and> Cong A A' B' B \<and> (A \<noteq> B' \<or> B \<noteq> A')\<close> 
          calculation(1) calculation(2) calculation(3) by force
    }
    ultimately show ?thesis 
      using Col_def 
        \<open>Col A B B' \<and> Col A B A' \<and> Cong A B B' A' \<and> Cong A A' B' B \<and> (A \<noteq> B' \<or> B \<noteq> A')\<close> 
      by blast
  qed
qed

lemma plgs_existence:
  assumes "A \<noteq> B"
  shows "\<exists> C. \<exists> D. ParallelogramStrict A B C D" 
proof -
  obtain C where "\<not> Col A B C"
    using assms not_col_exists by presburger
  obtain D where "Parallelogram A B C D"
    using assms plg_existence by blast
  {
    assume "ParallelogramFlat A B C D"
    have "False"
      using ParallelogramFlat_def \<open>ParallelogramFlat A B C D\<close> \<open>\<not> Col A B C\<close> by force 
  }
  thus ?thesis
    using Parallelogram_def \<open>Parallelogram A B C D\<close> by blast 
qed

lemma Rectangle_not_triv:
  shows "\<not> Rectangle A A A A"
  using Rectangle_Parallelogram plg_irreflexive by blast

lemma Plg_triv:
  assumes "A \<noteq> B"
  shows "Plg A A B B" 
proof -
  have "\<exists> M. M Midpoint A B \<and> M Midpoint A B"
    by (simp add: midpoint_existence) 
  thus ?thesis
    by (simp add: Plg_def assms)
qed

lemma Rectangle_triv:
  assumes "A \<noteq> B"
  shows "Rectangle A A B B" 
proof -
  have "Plg A A B B"
    by (simp add: Plg_triv assms) 
  moreover
  have "Cong A B A B"
    by (simp add: cong_reflexivity) 
  ultimately
  show ?thesis
    by (simp add: plg_cong_rectangle)
qed

lemma Rectangle_not_triv_2:
  shows "\<not> Rectangle A B A B"
  using Rectangle_Parallelogram plg_irreflexive by blast 

lemma Square_not_triv:
  shows "\<not> Square A A A A"
  by (simp add: Rectangle_not_triv_2 Square_def Tarski_neutral_dimensionless_axioms) 

lemma Square_not_triv_2:
  shows "\<not> Square A A B B"
  by (metis Square_def Square_not_triv cong_diff_3) 

lemma Square_not_triv_3:
  shows "\<not> Square A B A B"
  by (simp add: Rectangle_not_triv_2 Square_def) 

lemma Square_Rectangle:
  assumes "Square A B C D"
  shows "Rectangle A B C D"
  using Square_def assms by blast 

lemma Square_Parallelogram:
  assumes "Square A B C D"
  shows "Parallelogram A B C D"
  by (simp add: Rectangle_Parallelogram Square_Rectangle assms) 

lemma Rhombus_Rectangle_Square:
  assumes "Rhombus A B C D" and
    "Rectangle A B C D"
  shows "Square A B C D"
  using Rhombus_def Square_def assms(1) assms(2) by blast

lemma rhombus_cong_square:
  assumes "Rhombus A B C D" and
    "Cong A C B D" 
  shows "Square A B C D"
  by (simp add: Rhombus_Plg Rhombus_Rectangle_Square assms(1) assms(2) plg_cong_rectangle) 

lemma Kite_comm:
  assumes "Kite A B C D"
  shows "Kite C D A B"
  using Kite_def assms by auto 

lemma per2_col_eq:
  assumes "A \<noteq> P" and
    "A \<noteq> P'" and
    "Per A P B" and
    "Per A P' B" and
    "Col P A P'"
  shows "P = P'"
  by (metis assms(1) assms(2) assms(3) assms(4) assms(5) col_per2_cases 
      l6_16_1 l8_2 not_col_permutation_3)

lemma per2_preserves_diff:
  assumes "PO \<noteq> A'" and
    "PO \<noteq> B'" and
    "Col PO A' B'" and
    "Per PO A' A" and
    "Per PO B' B" and
    "A' \<noteq> B'"
  shows "A \<noteq> B"
  using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) not_col_permutation_4 
    per2_col_eq by blast

lemma per23_preserves_bet:
  assumes "Bet A B C" and
    "A \<noteq> B'" and "A \<noteq> C'" and
    "Col A B' C'" and
    "Per A B' B" and
    "Per A C' C"
  shows "Bet A B' C'"
proof -
  have P1: "Col A B C"
    by (simp add: assms(1) bet_col)
  show ?thesis
  proof cases
    assume P2: "B = B'"
    hence "Col A C' C"
      using P1 assms(2) assms(4) col_transitivity_1 by blast
    hence P4: "A = C' \<or> C = C'"
      by (simp add: assms(6) l8_9)
    {
      assume "A = C'"
      hence "Bet A B' C'"
        using assms(3) by auto
    }
    {
      assume "C = C'"
      hence "Bet A B' C'"
        using P2 assms(1) by auto
    }
    thus ?thesis
      using P4 assms(3) by auto
  next
    assume T1: "B \<noteq> B'"
    have T2: "A \<noteq> C"
      using assms(3) assms(6) l8_8 by auto
    have T3: "C \<noteq> C'"
      using P1 T1 assms(2) assms(3) assms(4) assms(5) col_trivial_3 colx l8_9 
        not_col_permutation_5 by blast
    have T3A: "A B' Perp B' B"
      using T1 assms(2) assms(5) per_perp by auto
    have T3B: "A C' Perp C' C"
      using T3 assms(3) assms(6) per_perp by auto
    have T4: "B B' Par C C'"
    proof -
      have "Coplanar A B' B C"
        using P1 ncop__ncols by blast
      moreover have "Coplanar A B' B C'"
        using assms(4) ncop__ncols by blast
      moreover have "Coplanar A B' B' C"
        using ncop_distincts by blast
      moreover have "B B' Perp A B'"
        using Perp_perm \<open>A B' Perp B' B\<close> by blast
      moreover have "C C' Perp A B'"
        using Col_cases Perp_cases T3B assms(2) assms(4) perp_col1 by blast
      ultimately show ?thesis
        using l12_9 bet__coplanar between_trivial by auto
    qed
    moreover have "Bet A B' C'"
    proof cases
      assume "B = C"
      thus ?thesis
        by (metis T1 per_col_eq assms(4) assms(5) calculation l6_16_1 l6_6 
            or_bet_out out_diff1 par_id)
    next
      assume T6: "B \<noteq> C"
      have T7: "\<not> Col A B' B"
        using T1 assms(2) assms(5) l8_9 by blast
      have T8: "\<not> Col A C' C"
        using T3 assms(3) assms(6) l8_9 by blast
      have T9: "B' \<noteq> C'"
        using P1 T6 assms(2) assms(5) assms(6) col_per2__per col_permutation_1 
          l8_2 l8_8 by blast
      have T10: "B B' ParStrict C C' \<or> (B \<noteq> B' \<and> C \<noteq> C' \<and> Col B C C' \<and> Col B' C C')"
        using Par_def calculation by blast
      {
        assume T11: "B B' ParStrict C C'"
        hence T12: "B B' OS C' C"
          using l12_6 one_side_symmetry by blast
        have "B B' TS A C"
          using Col_cases T6 T7 assms(1) bet__ts by blast
        hence "Bet A B' C'"
          using T12 assms(4) l9_5 l9_9 not_col_distincts or_bet_out by blast
      }
      {
        assume "B \<noteq> B' \<and> C \<noteq> C' \<and> Col B C C' \<and> Col B' C C'"
        hence "Bet A B' C'"
          using Col_def T6 T8 assms(1) col_transitivity_2 by blast
      }
      thus ?thesis
        using T10 \<open>B B' ParStrict C C' \<Longrightarrow> Bet A B' C'\<close> by blast
    qed
    ultimately show ?thesis
      using P1 Par_def T1 T2 assms(4) col_transitivity_2 not_col_permutation_1 
        par_strict_not_col_2 by blast
  qed
qed

lemma per23_preserves_bet_inv:
  assumes "Bet A B' C'" and
    "A \<noteq> B'" and
    "Col A B C" and
    "Per A B' B" and
    "Per A C' C"
  shows "Bet A B C"
proof cases
  assume T1: "B = B'"
  hence "Col A C' C"
    using Col_def assms(1) assms(2) assms(3) col_transitivity_1 by blast
  hence T2: "A = C' \<or> C = C'"
    by (simp add: assms(5) l8_9)
  {
    assume "A = C'"
    hence "Bet A B C"
      using assms(1) assms(2) between_identity by blast
  }
  {
    assume "C = C'"
    hence "Bet A B C"
      by (simp add: T1 assms(1))
  }
  thus ?thesis
    using T2 \<open>A = C' \<Longrightarrow> Bet A B C\<close> by auto
next
  assume P1: "B \<noteq> B'"
  hence P2: "A B' Perp B' B"
    using assms(2) assms(4) per_perp by auto
  have "Per A C' C"
    by (simp add: assms(5))
  hence P2: "C' PerpAt A C' C' C"
    by (metis (mono_tags, lifting) Col_cases P1 assms(1) assms(2) assms(3) assms(4) 
        bet_col bet_neq12__neq col_transitivity_1 l8_9 per_perp_in)
  hence P3: "A C' Perp C' C"
    using perp_in_perp by auto
  hence "C' \<noteq> C"
    using \<open>A C' Perp C' C\<close> perp_not_eq_2 by auto
  have "C' PerpAt C' A C C'"
    by (simp add: Perp_in_perm P2)
  hence "(C' A Perp C C') \<or> (C' C' Perp C C')"
    using Perp_def by blast
  have "A \<noteq> C'"
    using assms(1) assms(2) between_identity by blast
  {
    assume "C' A Perp C C'"
    have "Col A B' C'" using assms(1)
      by (simp add: Col_def)
    have "A B' Perp C' C"
      using Col_cases \<open>A C' Perp C' C\<close> \<open>Col A B' C'\<close> assms(2) perp_col by blast
    have P7: "B' B Par C' C"
    proof -
      have "Coplanar A B' B' C'"
        using ncop_distincts by blast
      moreover have "Coplanar A B' B' C"
        using ncop_distincts by auto
      moreover have "Coplanar A B' B C'"
        using Bet_perm assms(1) bet__coplanar ncoplanar_perm_20 by blast
      moreover have "Coplanar A B' B C"
        using assms(3) ncop__ncols by auto
      moreover have "B' B Perp A B'"
        by (metis P1 Perp_perm assms(2) assms(4) per_perp)
      moreover have "C' C Perp A B'"
        using Perp_cases \<open>A B' Perp C' C\<close> by auto
      ultimately show ?thesis using l12_9 by blast
    qed
    have "Bet A B C"
    proof cases
      assume "B = C"
      thus ?thesis
        by (simp add: between_trivial)
    next
      assume T1: "B \<noteq> C"
      have T2: "B' B ParStrict C' C \<or> (B' \<noteq> B \<and> C' \<noteq> C \<and> Col B' C' C \<and> Col B C' C)"
        using P7 Par_def by auto
      {
        assume T3: "B' B ParStrict C' C"
        hence "B' \<noteq> C'"
          using not_par_strict_id by auto
        have "\<exists> X. Col X B' B \<and> Col X B' C"
          using col_trivial_1 by blast
        have "B' B OS C' C"
          by (simp add: T3 l12_6)
        have "B' B TS A C'"
          by (metis Bet_cases T3 assms(1) assms(2) bet__ts l9_2 par_strict_not_col_1)
        hence T8: "B' B TS C A"
          using \<open>B' B OS C' C\<close> l9_2 l9_8_2 by blast
        then obtain T where T9: "Col T B' B \<and> Bet C T A"
          using TS_def by auto
        have "\<not> Col A C B'"
          using T8 assms(3) not_col_permutation_2 not_col_permutation_3 ts__ncol by blast
        hence "T = B"
          by (metis Col_def Col_perm T9 assms(3) colx)
        hence "Bet A B C"
          using Bet_cases T9 by auto
      }
      {
        assume "B' \<noteq> B \<and> C' \<noteq> C \<and> Col B' C' C \<and> Col B C' C"
        hence "Col A B' B"
          by (metis Col_perm T1 assms(3) l6_16_1)
        hence "A = B' \<or> B = B'"
          using assms(4) l8_9 by auto
        hence "Bet A B C"
          by (simp add: P1 assms(2))
      }
      thus ?thesis
        using T2 \<open>B' B ParStrict C' C \<Longrightarrow> Bet A B C\<close> by auto
    qed
  }
  thus ?thesis
    by (simp add: P3 perp_comm)
qed

lemma per13_preserves_bet:
  assumes "Bet A B C" and
    "B \<noteq> A'" and
    "B \<noteq> C'" and
    "Col A' B C'" and
    "Per B A' A" and
    "Per B C' C"
  shows "Bet A' B C'"
  by (smt (verit, ccfv_SIG) NCol_perm assms(1,3,4,5,6) bet_col1 bet_out_1 
      between_exchange2 l8_20_1_R1
      l8_7 not_bet_and_out per23_preserves_bet_inv point_construction_different third_point)

lemma per13_preserves_bet_inv:
  assumes "Bet A' B C'" and
    "B \<noteq> A'" and
    "B \<noteq> C'" and
    "Col A B C" and
    "Per B A' A" and
    "Per B C' C"
  shows "Bet A B C"
  by (meson Col_def assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) 
      between_equality between_symmetry per23_preserves_bet)

lemma per3_preserves_bet1:
  assumes "Col PO A B" and
    "Bet A B C" and
    "PO \<noteq> A'" and
    "PO \<noteq> B'" and
    "PO \<noteq> C'" and
    "Per PO A' A" and
    "Per PO B' B" and
    "Per PO C' C" and
    "Col A' B' C'" and
    "Col PO A' B'"
  shows "Bet A' B' C'"
proof cases
  assume "A = B"
  thus ?thesis
    using assms(10) assms(3) assms(4) assms(6) assms(7) between_trivial2 
      per2_preserves_diff by blast
next
  assume P1: "A \<noteq> B"
  show ?thesis
  proof cases
    assume P2: "A = A'"
    show ?thesis
    proof cases
      assume P3: "B = B'"
      hence "Col PO C C'" 
        using P1 P2 assms(10) assms(2) assms(9) bet_col col3 col_permutation_1 by blast
      hence "C = C'"
        using assms(5) assms(8) l8_9 not_col_permutation_5 by blast
      thus ?thesis
        using P2 P3 assms(2) by blast
    next
      assume P4: "B \<noteq> B'"
      show ?thesis
      proof cases
        assume "A = B'"
        thus ?thesis
          using P2 between_trivial2 by auto
      next
        assume "A \<noteq> B'"
        have "A \<noteq> C"
          using P1 assms(2) between_identity by blast
        have P7: "\<not> Col PO B' B"
          using P4 assms(4) assms(7) l8_9 by blast
        show ?thesis
          using P2 P7 assms(1) assms(10) assms(3) col_transitivity_1 by blast
      qed
    qed
  next
    assume R1: "A \<noteq> A'"
    show ?thesis
    proof cases
      assume R2: "A' = B'"
      thus ?thesis
        by (simp add: between_trivial2)
    next
      assume R3: "A' \<noteq> B'"
      show ?thesis
      proof cases
        assume "B = C"
        have "B' = C'"
          by (metis per2_col_eq \<open>A' \<noteq> B'\<close> \<open>B = C\<close> assms(10) assms(4) assms(5) 
              assms(7) assms(8) assms(9) col_transitivity_2 not_col_permutation_2)
        thus ?thesis
          by (simp add: between_trivial)
      next
        assume R4: "B \<noteq> C"
        show ?thesis
        proof cases
          assume "B = B'"
          thus ?thesis
            by (metis R1 assms(1) assms(10) assms(3) assms(4) assms(6) 
                l6_16_1 l8_9 not_col_permutation_2)
        next
          assume R5: "B \<noteq> B'"
          show ?thesis
          proof cases
            assume "A' = B"
            thus ?thesis
              using R5 assms(10) assms(4) assms(7) col_permutation_5 l8_9 by blast
          next
            assume R5A: "A' \<noteq> B"
            have R6: "C \<noteq> C'"
              by (metis P1 R1 R3 assms(1) assms(10) assms(2) assms(3) 
                  assms(5) assms(6) assms(9) bet_col col_permutation_1 
                  col_trivial_2 l6_21 l8_9)
            have R7: "A A' Perp PO A'"
              by (metis Perp_cases R1 assms(3) assms(6) per_perp)
            have R8: "C C' Perp PO A'"
            proof -
              have "C' C' Perp C' C \<or> PO C' Perp C' C" 
                using R6 assms(5) assms(8) per_perp by auto
              moreover
              {
                assume "C' C' Perp C' C"
                hence "C C' Perp PO A'" 
                  using \<open>C' C' Perp C' C\<close> perp_not_eq_1 by blast
              }
              moreover
              {
                assume "PO C' Perp C' C"
                hence "C C' Perp PO A'" 
                  by (metis Col_cases R3 Perp_cases assms(10) assms(3) assms(9) col3 
                      col_trivial_3 perp_col1)
              }
              ultimately show ?thesis 
                by blast
            qed
            have "A A' Par C C'"
            proof -
              have "Coplanar PO A' A C"
                using P1 assms(1) assms(2) bet_col col_trivial_2 colx ncop__ncols by blast
              moreover have "Coplanar PO A' A C'"
                using R3 assms(10) assms(9) col_trivial_2 colx ncop__ncols by blast
              moreover have "Coplanar PO A' A' C"
                using ncop_distincts by blast
              moreover have "Coplanar PO A' A' C'"
                using ncop_distincts by blast
              ultimately show ?thesis using l12_9 R7 R8 by blast
            qed
            have S1: "B B' Perp PO A'"
              by (metis Col_cases Per_cases Perp_perm R5 assms(10) assms(3) 
                  assms(4) assms(7) col_per_perp)
            have "A A' Par B B'"
            proof -
              have "Coplanar PO A' A B"
                using assms(1) ncop__ncols by auto
              moreover have "Coplanar PO A' A B'"
                using assms(10) ncop__ncols by auto
              moreover have "Coplanar PO A' A' B"
                using ncop_distincts by auto
              moreover have "Coplanar PO A' A' B'"
                using ncop_distincts by auto
              moreover have "A A' Perp PO A'"
                by (simp add: R7)
              moreover have "B B' Perp PO A'"
                by (simp add: S1)
              ultimately show ?thesis
                using l12_9 by blast
            qed
            {
              assume "A A' ParStrict B B'"
              hence "A A' OS B B'"
                by (simp add: l12_6)
              have "B B' TS A C"
                using R4 \<open>A A' ParStrict B B'\<close> assms(2) bet__ts par_strict_not_col_3 by auto
              have "B B' OS A A'"
                using \<open>A A' ParStrict B B'\<close> pars__os3412 by auto
              have "B B' TS A' C"
                using \<open>B B' OS A A'\<close> \<open>B B' TS A C\<close> l9_8_2 by blast
              have "Bet A' B' C'"
              proof cases
                assume "C = C'"
                thus ?thesis
                  using R6 by auto
              next
                assume "C \<noteq> C'"
                have "C C' Perp PO A'"
                  by (simp add: R8)
                have Q2: "B B' Par C C'"
                proof -
                  have "Coplanar PO A' B C"
                    by (metis P1 assms(1) assms(2) bet_col col_transitivity_1 
                        colx ncop__ncols not_col_permutation_5)
                  moreover have "Coplanar PO A' B C'"
                    using R3 assms(10) assms(9) col_trivial_2 colx ncop__ncols by blast
                  moreover have "Coplanar PO A' B' C"
                    by (simp add: assms(10) col__coplanar)
                  moreover have "Coplanar PO A' B' C'"
                    using assms(10) col__coplanar by auto
                  moreover have "B B' Perp PO A'"
                    by (simp add: S1)
                  moreover have "C C' Perp PO A'"
                    by (simp add: R8)
                  ultimately show ?thesis
                    using l12_9 by auto
                qed
                hence Q3: "(B B' ParStrict C C') \<or> (B \<noteq> B' \<and> C \<noteq> C' \<and> Col B C C' \<and> Col B' C C')"
                  by (simp add: Par_def)
                {
                  assume "B B' ParStrict C C'"
                  hence "B B' OS C C'"
                    using l12_6 by auto
                  hence "B B' TS C' A'"
                    using \<open>B B' TS A' C\<close> l9_2 l9_8_2 by blast
                  then obtain T where Q4: "Col T B B' \<and> Bet C' T A'"
                    using TS_def by blast
                  have "T = B'"
                  proof -
                    have "\<not> Col B B' A'"
                      using \<open>B B' OS A A'\<close> col124__nos by auto
                    moreover have "A' \<noteq> C'"
                      using \<open>B B' TS C' A'\<close> not_two_sides_id by auto
                    moreover have "Col B B' T"
                      using Col_cases Q4 by auto
                    moreover have "Col B B' B'"
                      using not_col_distincts by blast
                    moreover have "Col A' C' T"
                      by (simp add: Col_def Q4)
                    ultimately show ?thesis
                      by (meson assms(9) col_permutation_5 l6_21)
                  qed
                  hence "Bet A' B' C'"
                    using Q4 between_symmetry by blast
                }
                {
                  assume "B \<noteq> B' \<and> C \<noteq> C' \<and> Col B C C' \<and> Col B' C C'"
                  hence "Bet A' B' C'"
                    using TS_def \<open>B B' TS A C\<close> l6_16_1 not_col_permutation_2 by blast
                }
                thus ?thesis
                  using Q3 \<open>B B' ParStrict C C' \<Longrightarrow> Bet A' B' C'\<close> by blast
              qed
            }
            {
              assume R8: "A \<noteq> A' \<and> B \<noteq> B' \<and> Col A B B' \<and> Col A' B B'"
              have "A' A Perp PO A'"
                by (simp add: R7 perp_left_comm)
              have "\<not> Col A' A PO"
                using Col_cases R8 assms(3) assms(6) l8_9 by blast
              hence "Bet A' B' C'"
                using Col_perm P1 R8 assms(1) l6_16_1 by blast
            }
            thus ?thesis
              using Par_def \<open>A A' Par B B'\<close> \<open>A A' ParStrict B B' \<Longrightarrow> Bet A' B' C'\<close> by auto
          qed
        qed
      qed
    qed
  qed
qed

lemma per3_preserves_bet2_aux:
  assumes "Col PO A C" and
    "A \<noteq> C'" and
    "Bet A B' C'" and
    "PO \<noteq> A" and
    "PO \<noteq> B'" and
    "PO \<noteq> C'" and
    "Per PO B' B" and
    "Per PO C' C" and
    "Col A B C" and
    "Col PO A C'"
  shows "Bet A B C"
proof cases
  assume "A = B"
  thus ?thesis
    by (simp add: between_trivial2)
next
  assume P1: "A \<noteq> B"
  show ?thesis
  proof cases
    assume "B = C"
    thus ?thesis
      by (simp add: between_trivial)
  next
    assume P2: "B \<noteq> C"
    have P3: "Col PO A B'"
      by (metis Col_def assms(10) assms(2) assms(3) l6_16_1)
    hence P4: "Col PO B' C'"
      using assms(10) assms(4) col_transitivity_1 by blast
    show ?thesis
    proof cases
      assume "B = B'"
      thus ?thesis
        by (metis per_col_eq assms(1) assms(10) assms(3) assms(4) assms(6) 
            assms(8) col_transitivity_1)
    next
      assume P5: "B \<noteq> B'"
      have P6: "C = C'"
        using assms(1) assms(10) assms(4) assms(6) assms(8) col_transitivity_1 l8_9 by blast
      hence "False"
        by (metis P3 P5 P6 per_col_eq assms(1) assms(2) assms(4) assms(5) 
            assms(7) assms(9) col_transitivity_1 l6_16_1 not_col_permutation_4)
      thus ?thesis by blast
    qed
  qed
qed

lemma per3_preserves_bet2:
  assumes "Col PO A C" and
    "A' \<noteq> C'" and
    "Bet A' B' C'" and
    "PO \<noteq> A'" and
    "PO \<noteq> B'" and
    "PO \<noteq> C'" and
    "Per PO A' A" and
    "Per PO B' B" and
    "Per PO C' C" and
    "Col A B C" and
    "Col PO A' C'"
  shows "Bet A B C"
proof cases
  assume "A = A'"
  thus ?thesis
    using assms(1) assms(10) assms(11) assms(2) assms(3) assms(4) assms(5) 
      assms(6) assms(8) assms(9) per3_preserves_bet2_aux by blast
next
  assume P1: "A \<noteq> A'"
  show ?thesis
  proof cases
    assume "C = C'"
    thus ?thesis
      by (metis P1 assms(1) assms(11) assms(4) assms(6) assms(7) col_trivial_3 
          l6_21 l8_9 not_col_permutation_2)
  next
    assume P2: "C \<noteq> C'"
    hence P3: "PO A' Perp C C'"
      by (metis assms(11) assms(4) assms(6) assms(9) col_per_perp l8_2 not_col_permutation_1)
    have P4: "PO A' Perp A A'"
      using P1 assms(4) assms(7) per_perp perp_right_comm by auto
    have "A A' Par C C'"
    proof -
      have "Coplanar PO A' A C"
        using assms(1) ncop__ncols by blast
      moreover have "Coplanar PO A' A C'"
        by (meson assms(11) ncop__ncols)
      moreover have "Coplanar PO A' A' C"
        using ncop_distincts by blast
      moreover have "Coplanar PO A' A' C'"
        using ncop_distincts by blast
      moreover have "A A' Perp PO A'"
        using P4 Perp_cases by blast
      moreover have "C C' Perp PO A'"
        using P3 Perp_cases by auto
      ultimately show ?thesis
        using l12_9 by blast
    qed
    {
      assume P5: "A A' ParStrict C C'"
      hence P6: "A A' OS C C'"
        by (simp add: l12_6)
      have P7: "C C' OS A A'"
        by (simp add: P5 pars__os3412)
      have "Bet A B C"
      proof cases
        assume P8: "B = B'"
        hence "A' A OS B C'"
          by (metis P6 assms(10) assms(3) bet_out col123__nos col124__nos 
              invert_one_side out_one_side)
        hence "A A' OS B C'"
          by (simp add: invert_one_side)
        hence "A A' OS B C"
          using P6 one_side_symmetry one_side_transitivity by blast
        hence P12: "A Out B C"
          using assms(10) col_one_side_out by blast
        have "C' C OS B A'"
          by (metis Col_perm P5 P7 P8 assms(10) assms(3) bet_out_1 col123__nos 
              out_one_side par_strict_not_col_2)
        hence "C C' OS B A"
          by (meson P7 invert_one_side one_side_symmetry one_side_transitivity)
        hence "C C' OS A B"
          using one_side_symmetry by blast
        hence "C Out A B"
          using assms(10) col_one_side_out col_permutation_2 by blast
        thus ?thesis
          by (simp add: P12 out2__bet)
      next
        assume T1: "B \<noteq> B'"
        have T2: "PO A' Perp B B'"
        proof -
          have "Per PO B' B"
            by (simp add: assms(8))
          hence "B' PerpAt PO B' B' B"
            using T1 assms(5) per_perp_in by auto
          hence "B' PerpAt B' PO B B'"
            by (simp add: perp_in_comm)
          hence T4: "B' PO Perp B B' \<or> B' B' Perp B B'"
            using Perp_def by auto
          {
            assume T5: "B' PO Perp B B'"
            have "Col A' B' C'"
              by (simp add: assms(3) bet_col)
            hence "Col PO B' A'"
              using assms(11) assms(2) col2__eq col_permutation_4 
                col_permutation_5 by blast
            hence "PO A' Perp B B'"
              by (metis T5 assms(4) col_trivial_3 perp_col2 perp_comm)
          }
          {
            assume "B' B' Perp B B'"
            hence "PO A' Perp B B'"
              using perp_distinct by auto
          }
          thus ?thesis
            using T4 \<open>B' PO Perp B B' \<Longrightarrow> PO A' Perp B B'\<close> by linarith
        qed
        have T6: "B B' Par A A'"
        proof -
          have "Coplanar PO A' B A"
            by (metis Col_cases P7 assms(1) assms(10) col_transitivity_2 
                ncop__ncols os_distincts)
          moreover have "Coplanar PO A' B A'"
            using ncop_distincts by blast
          moreover have "Coplanar PO A' B' A"
          proof -
            have "(Bet PO A' C' \<or> Bet PO C' A') \<or> Bet C' PO A'"
              by (meson assms(11) third_point)
            thus ?thesis
              by (meson Bet_perm assms(3) bet__coplanar between_exchange2 
                  l5_3 ncoplanar_perm_8)
          qed
          moreover have "Coplanar PO A' B' A'"
            using ncop_distincts by auto
          moreover have "B B' Perp PO A'"
            using Perp_cases T2 by blast
          moreover have "A A' Perp PO A'"
           