(*
Authors: Tobias Nipkow, Akihisa Yamada
*)

section "Context-Free Grammars"

theory Context_Free_Grammar
imports
  "Fresh_Identifiers.Fresh_Nat"
  "Regular-Sets.Regular_Set"
begin

lemma append_Cons_eq_append_Cons:
  "y' \<notin> set xs \<Longrightarrow> y \<notin> set xs' \<Longrightarrow>
   xs @ y # zs = xs' @ y' # zs' \<longleftrightarrow> xs = xs' \<and> y = y' \<and> zs = zs'"
  by (induction xs arbitrary: xs'; force simp: Cons_eq_append_conv)

lemma insert_conc: "insert w W @@ V = {w @ v | v. v \<in> V} \<union> W @@ V"
  by auto

lemma conc_insert: "W @@ insert v V = {w @ v | w. w \<in> W} \<union> W @@ V"
  by auto

declare relpowp.simps(2)[simp del]

lemma bex_pair_conv: "(\<exists>(x,y) \<in> R. P x y) \<longleftrightarrow> (\<exists>x y. (x,y) \<in> R \<and> P x y)"
  by auto

lemma in_image_map_prod: "fgp \<in> map_prod f g ` R \<longleftrightarrow> (\<exists>(x,y)\<in>R. fgp = (f x,g y))"
  by auto


subsection "Symbols and Context-Free Grammars"

text \<open>Most of the theory is based on arbitrary sets of productions.
Finiteness of the set of productions is only required where necessary.
Finiteness of the type of terminal symbols is only required where necessary.
Whenever fresh nonterminals need to be invented, the type of nonterminals is assumed to be infinite.\<close>

datatype ('n,'t) sym = Nt 'n | Tm 't

type_synonym ('n,'t) syms = "('n,'t) sym list"

type_synonym ('n,'t) prod = "'n \<times> ('n,'t) syms"

type_synonym ('n,'t) prods = "('n,'t) prod list"
type_synonym ('n,'t) Prods = "('n,'t) prod set"

datatype ('n,'t) cfg = cfg (prods : "('n,'t) prods") (start : "'n")
datatype ('n,'t) Cfg = Cfg (Prods : "('n,'t) Prods") (Start : "'n")

definition isTm :: "('n, 't) sym \<Rightarrow> bool" where 
"isTm S = (\<exists>a. S = Tm a)"

definition isNt :: "('n, 't) sym \<Rightarrow> bool" where 
"isNt S = (\<exists>A. S = Nt A)"

fun destTm :: "('n, 't) sym  \<Rightarrow> 't" where 
\<open>destTm (Tm a) = a\<close>

lemma isNt_simps[simp,code]:
  \<open>isNt (Nt A) = True\<close>
  \<open>isNt (Tm a) = False\<close> 
by (simp_all add: isNt_def)

lemma isTm_simps[simp,code]:
  \<open>isTm (Nt A) = False\<close>
  \<open>isTm (Tm a) = True\<close> 
by (simp_all add: isTm_def)

lemma filter_isTm_map_Tm[simp]: \<open>filter isTm (map Tm xs) = map Tm xs\<close>
by(induction xs) auto

lemma destTm_o_Tm[simp]: \<open>destTm \<circ> Tm = id\<close>
by auto

definition Nts_syms :: "('n,'t)syms \<Rightarrow> 'n set" where
"Nts_syms w = {A. Nt A \<in> set w}"

definition Tms_syms :: "('n,'t)syms \<Rightarrow> 't set" where
"Tms_syms w = {a. Tm a \<in> set w}"

definition Nts :: "('n,'t)Prods \<Rightarrow> 'n set" where
  "Nts P = (\<Union>(A,w)\<in>P. {A} \<union> Nts_syms w)"

definition Tms :: "('n,'t)Prods \<Rightarrow> 't set" where 
  "Tms P = (\<Union>(A,w)\<in>P. Tms_syms w)"

definition Syms :: "('n,'t)Prods \<Rightarrow> ('n,'t) sym set" where 
  "Syms P = (\<Union>(A,w)\<in>P. {Nt A} \<union> set w)"

lemma Tms_mono: "P \<subseteq> P' \<Longrightarrow> Tms P \<subseteq> Tms P'"
unfolding Tms_def Tms_syms_def by blast

definition nts_syms_acc :: "('n,'t)syms \<Rightarrow> 'n list \<Rightarrow> 'n list" where
"nts_syms_acc = foldr (\<lambda>sy ns. case sy of Nt A \<Rightarrow> List.insert A ns | Tm _ \<Rightarrow> ns)"

definition nts_syms :: "('n,'t)syms \<Rightarrow> 'n list" where
"nts_syms sys = nts_syms_acc sys []"

definition nts :: "('n,'t)prods \<Rightarrow> 'n list" where
"nts ps = foldr (\<lambda>(A,sys) ns. List.insert A (nts_syms_acc sys ns)) ps []"

definition tms_syms_acc :: "('n,'t)syms \<Rightarrow> 't list \<Rightarrow> 't list" where
"tms_syms_acc = foldr (\<lambda>sy ts. case sy of Tm a \<Rightarrow> List.insert a ts | Nt _ \<Rightarrow> ts)"

definition tms_syms :: "('n,'t)syms \<Rightarrow> 't list" where
"tms_syms sys = tms_syms_acc sys []"

definition tms :: "('n,'t)prods \<Rightarrow> 't list" where
"tms ps = foldr (\<lambda>(_,sys). tms_syms_acc sys) ps []"

definition Lhss :: "('n, 't) Prods \<Rightarrow> 'n set" where
"Lhss P = (\<Union>(A,w) \<in> P. {A})"

abbreviation lhss :: "('n, 't) prods \<Rightarrow> 'n set" where
"lhss ps \<equiv> Lhss(set ps)"

definition Rhs_Nts :: "('n, 't) Prods \<Rightarrow> 'n set" where
"Rhs_Nts P = (\<Union>(_,w)\<in>P. Nts_syms w)"

definition Rhss :: "('n \<times> 'a) set \<Rightarrow> 'n \<Rightarrow> 'a set" where
"Rhss P A = {w. (A,w) \<in> P}"

lemma Rhss_code[code]: "Rhss P A = snd ` {Aw \<in> P. fst Aw = A}"
by(auto simp add: Rhss_def image_iff)

lemma inj_Nt: "inj Nt"
by (simp add: inj_def)

lemma map_Tm_inject_iff[simp]: "map Tm xs = map Tm ys \<longleftrightarrow> xs = ys"
by (metis sym.inject(2) list.inj_map_strong)

lemma map_Nt_eq_map_Nt_iff[simp]: "map Nt u = map Nt v \<longleftrightarrow> u = v"
by(rule inj_map_eq_map[OF inj_Nt])

lemma map_Nt_eq_map_Tm_iff[simp]: "map Nt u = map Tm v \<longleftrightarrow> u = [] \<and> v = []"
by (cases u) auto

lemmas map_Tm_eq_map_Nt_iff[simp] = eq_iff_swap[OF map_Nt_eq_map_Tm_iff]

lemma Nts_syms_Nil[simp,code]: "Nts_syms [] = {}"
unfolding Nts_syms_def by auto

lemma Nts_syms_Cons[simp,code]: "Nts_syms (s#ss) = (case s of Nt A \<Rightarrow> {A} | _ \<Rightarrow> {}) \<union> Nts_syms ss"
by (auto simp: Nts_syms_def split: sym.split)

lemma Tms_syms_Nil[simp,code]: "Tms_syms [] = {}"
unfolding Tms_syms_def by auto

lemma Tms_syms_Cons[simp,code]: "Tms_syms (s#ss) = (case s of Tm a \<Rightarrow> {a} | _ \<Rightarrow> {}) \<union> Tms_syms ss"
by (auto simp: Tms_syms_def split: sym.split)

lemma Nts_syms_append[simp]: "Nts_syms (u @ v) = Nts_syms u \<union> Nts_syms v"
by (auto simp: Nts_syms_def)

lemma Tms_syms_append[simp]: "Tms_syms (u @ v) = Tms_syms u \<union> Tms_syms v"
by (auto simp: Tms_syms_def)

lemma Nts_syms_map_Nt[simp]: "Nts_syms (map Nt w) = set w"
unfolding Nts_syms_def by auto

lemma Tms_syms_map_Tm[simp]: "Tms_syms (map Tm w) = set w"
unfolding Tms_syms_def by auto

lemma Nts_syms_map_Tm[simp]: "Nts_syms (map Tm w) = {}"
unfolding Nts_syms_def by auto

lemma Tms_syms_map_Nt[simp]: "Tms_syms (map Nt w) = {}"
unfolding Tms_syms_def by auto

lemma Nts_syms_rev: "Nts_syms (rev w) = Nts_syms w"
by(auto simp: Nts_syms_def)

lemma Tms_syms_rev: "Tms_syms (rev w) = Tms_syms w"
by(auto simp: Tms_syms_def)

lemma Nts_syms_empty_iff: "Nts_syms w = {} \<longleftrightarrow> (\<exists>u. w = map Tm u)"
by(induction w) (auto simp: ex_map_conv split: sym.split)

lemma Tms_syms_empty_iff: "Tms_syms w = {} \<longleftrightarrow> (\<exists>u. w = map Nt u)"
by(induction w) (auto simp: ex_map_conv split: sym.split)

text \<open>If a sentential form contains a \<open>Nt\<close>, it must have a last and a first \<open>Nt\<close>:\<close>

lemma non_word_has_last_Nt: "Nts_syms w \<noteq> {} \<Longrightarrow> \<exists>u A v. w = u @ [Nt A] @ map Tm v"
proof (induction w)
  case Nil
  then show ?case by simp
next
  case (Cons a list)
  then show ?case using Nts_syms_empty_iff[of list]
    by(auto simp: Cons_eq_append_conv split: sym.splits)
qed

lemma non_word_has_first_Nt: "Nts_syms w \<noteq> {} \<Longrightarrow> \<exists>u A v. w = map Tm u @ Nt A # v"
  using Nts_syms_rev non_word_has_last_Nt[of "rev w"]
  by (metis append.assoc append_Cons append_Nil rev.simps(2) rev_eq_append_conv rev_map)

lemma in_Nts_iff_in_Syms: "B \<in> Nts P \<longleftrightarrow> Nt B \<in> Syms P"
unfolding Nts_def Syms_def Nts_syms_def by (auto)

lemma Nts_mono: "G \<subseteq> H \<Longrightarrow> Nts G \<subseteq> Nts H"
by (auto simp add: Nts_def)

lemma Nts_Un: "Nts (P1 \<union> P2) = Nts P1 \<union> Nts P2"
by (simp add: Nts_def)

lemma Rhs_Nts_Un: "Rhs_Nts (P \<union> Q) = Rhs_Nts P \<union> Rhs_Nts Q"
  by (simp add: Rhs_Nts_def)

lemma Rhss_Un: "Rhss (P \<union> Q) A = Rhss P A \<union> Rhss Q A"
  by (auto simp: Rhss_def)

lemma Rhss_UN: "Rhss (\<Union>PP) A = \<Union>{Rhss P A | P. P \<in> PP}"
  by (auto simp: Rhss_def)

lemma Rhss_empty[simp]: "Rhss {} A = {}"
  by (auto simp: Rhss_def)

lemma Rhss_insert: "Rhss (insert (A,\<alpha>) P) B = (if A = B then insert \<alpha> (Rhss P B) else Rhss P B)"
  by (auto simp: Rhss_def)

lemma Nts_Lhss_Rhs_Nts: "Nts P = Lhss P \<union> Rhs_Nts P"
unfolding Nts_def Lhss_def Rhs_Nts_def by auto

lemma Nts_Nts_syms: "w \<in> Rhss P A \<Longrightarrow> Nts_syms w \<subseteq> Nts P"
unfolding Rhss_def Nts_def by blast

lemma Syms_simps[simp]:
  "Syms {} = {}"
  "Syms(insert (A,w) P) = {Nt A} \<union> set w \<union> Syms P"
  "Syms(P \<union> P') = Syms P \<union> Syms P'"
by(auto simp: Syms_def)

lemma Lhss_simps[simp]:
  "Lhss {} = {}"
  "Lhss(insert (A,w) P) = {A} \<union> Lhss P"
  "Lhss(P \<union> P') = Lhss P \<union> Lhss P'"
by(auto simp: Lhss_def)

lemma in_LhssI: "(A,\<alpha>) \<in> P \<Longrightarrow> A \<in> Lhss P"
by (auto simp: Lhss_def)

lemma Lhss_Collect: "Lhss {p. X p} = {A. \<exists>\<alpha>. X (A,\<alpha>)}"
  by (auto simp: Lhss_def)

lemma in_Rhs_NtsI: "(A,\<alpha>) \<in> P \<Longrightarrow> B \<in> Nts_syms \<alpha> \<Longrightarrow> B \<in> Rhs_Nts P"
by (auto simp: Rhs_Nts_def)

lemma set_nts_syms: "set(nts_syms_acc sys ns) = Nts_syms sys \<union> set ns"
unfolding nts_syms_acc_def
by(induction sys arbitrary: ns) (auto split: sym.split)

lemma set_nts: "set(nts ps) = Nts (set ps)"
by(induction ps) (auto simp: nts_def Nts_def set_nts_syms split: prod.splits)

lemma distinct_nts_syms_acc: "distinct(nts_syms_acc sys ns) = distinct ns"
unfolding nts_syms_acc_def
by(induction sys arbitrary: ns) (auto split: sym.split)

lemma distinct_nts_syms: "distinct(nts_syms sys)"
unfolding nts_syms_def by(simp add: distinct_nts_syms_acc)

lemma distinct_nts: "distinct(nts ps)"
by(induction ps) (auto simp: nts_def distinct_nts_syms_acc distinct_nts_syms)

lemma set_tms_syms_acc: "set(tms_syms_acc sys ts) = Tms_syms sys \<union> set ts"
unfolding tms_syms_acc_def
by(induction sys arbitrary: ts) (auto split: sym.split)

corollary set_tms_syms: "set(tms_syms sys) = Tms_syms sys"
unfolding tms_syms_def Tms_syms_def set_tms_syms_acc Tms_syms_def by (auto)

lemma set_tms: "set(tms ps) = Tms (set ps)"
by(induction ps) (auto simp: tms_def Tms_def set_tms_syms_acc split: prod.splits)

lemma distinct_tms_syms_acc: "distinct(tms_syms_acc sys ts) = distinct ts"
unfolding tms_syms_acc_def
by(induction sys arbitrary: ts) (auto split: sym.split)

lemma distinct_tms_syms: "distinct(tms_syms sys)"
unfolding tms_syms_def by(simp add: distinct_tms_syms_acc)

lemma distinct_tms: "distinct(tms ps)"
by(induction ps) (auto simp: tms_def distinct_tms_syms_acc split: sym.split)


subsubsection \<open>Finiteness Lemmas\<close>

lemma finite_Nts_syms: "finite (Nts_syms w)"
by (induction w) (auto split: sym.split)

lemma finite_Tms_syms: "finite (Tms_syms w)"
by (induction w) (auto split: sym.split)

lemma finite_nts: "finite(Nts (set ps))"
unfolding Nts_def by (simp add: finite_Nts_syms split_def)

lemma finite_tms: "finite(Tms (set ps))"
unfolding Tms_def by (simp add: finite_Tms_syms split_def)

lemma fresh0_nts: "fresh0(Nts (set ps)) \<notin> Nts (set ps)"
by(fact fresh0_notIn[OF finite_nts])

lemma finite_nts_prods_start: "finite(Nts(set(prods g)) \<union> {start g})"
unfolding Nts_def by (simp add: finite_Nts_syms split_def)

lemma fresh_nts_prods_start: "fresh0(Nts(set(prods g)) \<union> {start g}) \<notin> Nts(set(prods g)) \<union> {start g}"
by(fact fresh0_notIn[OF finite_nts_prods_start])

lemma finite_Nts: "finite P \<Longrightarrow> finite (Nts P)"
unfolding Nts_def by (simp add: case_prod_beta finite_Nts_syms)

lemma finite_Tms: "finite P \<Longrightarrow> finite (Tms P)"
unfolding Tms_def by (simp add: case_prod_beta finite_Tms_syms)

lemma finite_Rhss: "finite P \<Longrightarrow> finite (Rhss P A)"
unfolding Rhss_def by (metis Image_singleton finite_Image)


subsection "Derivations and Languages"

subsubsection \<open>The standard derivations \<open>\<Rightarrow>\<close>, \<open>\<Rightarrow>*\<close>, \<open>\<Rightarrow>(n)\<close>\<close>

inductive derive :: "('n,'t) Prods \<Rightarrow> ('n,'t) syms \<Rightarrow> ('n,'t)syms \<Rightarrow> bool"
  ("(2_ \<turnstile>/ (_ \<Rightarrow>/ _))" [50, 0, 50] 50) for P where
"(A,\<alpha>) \<in> P \<Longrightarrow> P \<turnstile> u @ [Nt A] @ v \<Rightarrow> u @ \<alpha> @ v"

abbreviation deriven ("(2_ \<turnstile>/ (_ /\<Rightarrow>'(_')/ _))" [50, 0, 0, 50] 50) where
"P \<turnstile> u \<Rightarrow>(n) v \<equiv> (derive P ^^ n) u v"

abbreviation derives ("(2_ \<turnstile>/ (_/ \<Rightarrow>*/ _))" [50, 0, 50] 50) where
"P \<turnstile> u \<Rightarrow>* v \<equiv> ((derive P) ^**) u v"

definition Ders :: "('n,'t)Prods \<Rightarrow> 'n \<Rightarrow> ('n,'t)syms set" where
"Ders P A = {w. P \<turnstile> [Nt A] \<Rightarrow>* w}"

abbreviation ders :: "('n,'t)prods \<Rightarrow> 'n \<Rightarrow> ('n,'t)syms set" where
"ders ps \<equiv> Ders (set ps)"

lemma DersI:
  assumes "P \<turnstile> [Nt A] \<Rightarrow>* w" shows "w \<in> Ders P A"
  using assms by (auto simp: Ders_def)

lemma DersD:
  assumes "w \<in> Ders P A" shows "P \<turnstile> [Nt A] \<Rightarrow>* w"
  using assms by (auto simp: Ders_def)

lemmas DersE = DersD[elim_format]

text \<open>The \emph{language} of a nonterminal is the set of the terminal words it derives.\<close>

definition Lang :: "('n,'t)Prods \<Rightarrow> 'n \<Rightarrow> 't list set" where
"Lang P A = {w. P \<turnstile> [Nt A] \<Rightarrow>* map Tm w}"

abbreviation lang :: "('n,'t)prods \<Rightarrow> 'n \<Rightarrow> 't list set" where
"lang ps A \<equiv> Lang (set ps) A"

abbreviation LangS :: "('n,'t) Cfg \<Rightarrow> 't list set" where
"LangS G \<equiv> Lang (Prods G) (Start G)"

abbreviation langS :: "('n,'t) cfg \<Rightarrow> 't list set" where
"langS g \<equiv> lang (prods g) (start g)"

text \<open>Language is extended over mixed words.\<close>

definition Lang_of :: "('n,'t) Prods \<Rightarrow> ('n,'t) syms \<Rightarrow> 't list set" where
"Lang_of P \<alpha> = {w. P \<turnstile> \<alpha> \<Rightarrow>* map Tm w}"

abbreviation Lang_of_set :: "('n,'t) Prods \<Rightarrow> ('n,'t) syms set \<Rightarrow> 't list set" where
"Lang_of_set P X \<equiv> \<Union>(Lang_of P ` X)"

lemma Lang_Ders: "map Tm ` (Lang P A) \<subseteq> Ders P A"
unfolding Lang_def Ders_def by auto

lemma Lang_subset_if_Ders_subset: "Ders R A \<subseteq> Ders R' A \<Longrightarrow> Lang R A \<subseteq> Lang R' A"
by (auto simp add: Lang_def Ders_def)

lemma Lang_eqI_derives:
  assumes "\<And>v. R \<turnstile> [Nt A] \<Rightarrow>* map Tm v \<longleftrightarrow> S \<turnstile> [Nt A] \<Rightarrow>* map Tm v"
  shows "Lang R A = Lang S A"
  by (auto simp: Lang_def assms)

lemma derive_iff: "R \<turnstile> u \<Rightarrow> v \<longleftrightarrow> (\<exists> (A,w) \<in> R. \<exists>u1 u2. u = u1 @ Nt A # u2 \<and> v = u1 @ w @ u2)"
  apply (rule iffI)
   apply (induction rule: derive.induct)
   apply (fastforce)
  using derive.intros by fastforce 

lemma not_derive_from_Tms: "\<not> P \<turnstile> map Tm as \<Rightarrow> w"
by(auto simp add: derive_iff map_eq_append_conv)

lemma deriven_from_TmsD: "P \<turnstile> map Tm as \<Rightarrow>(n) w \<Longrightarrow> w = map Tm as"
by (metis not_derive_from_Tms relpowp_E2)
 
lemma derives_from_Tms_iff: "P \<turnstile> map Tm as \<Rightarrow>* w \<longleftrightarrow> w = map Tm as"
by (meson deriven_from_TmsD rtranclp.rtrancl_refl rtranclp_power)

lemma Un_derive: "R \<union> S \<turnstile> y \<Rightarrow> z \<longleftrightarrow> R \<turnstile> y \<Rightarrow> z \<or> S \<turnstile> y \<Rightarrow> z"
  by (fastforce simp: derive_iff)

lemma derives_rule:
  assumes 2: "(A,w) \<in> R" and 1: "R \<turnstile> x \<Rightarrow>* y @ Nt A # z" and 3: "R \<turnstile> y@w@z \<Rightarrow>* v"
  shows "R \<turnstile> x \<Rightarrow>* v"
proof-
  note 1
  also have "R \<turnstile> y @ Nt A # z \<Rightarrow> y @ w @ z"
    using derive.intros[OF 2] by simp
  also note 3
  finally show ?thesis.
qed

lemma derives_Cons_rule:
  assumes 1: "(A,w) \<in> R" and 2: "R \<turnstile> w @ u \<Rightarrow>* v" shows "R \<turnstile> Nt A # u \<Rightarrow>* v"
  using derives_rule[OF 1, of "Nt A # u" "[]" u v] 2 by auto

lemma deriven_mono: "P \<subseteq> P' \<Longrightarrow> P \<turnstile> u \<Rightarrow>(n) v \<Longrightarrow> P' \<turnstile> u \<Rightarrow>(n) v"
by (metis Un_derive relpowp_mono subset_Un_eq)

lemma derives_mono: "P \<subseteq> P' \<Longrightarrow> P \<turnstile> u \<Rightarrow>* v \<Longrightarrow> P' \<turnstile> u \<Rightarrow>* v"
by (meson deriven_mono rtranclp_power)

lemma Lang_mono: "P \<subseteq> P' \<Longrightarrow> Lang P A \<subseteq> Lang P' A"
by (auto simp: Lang_def derives_mono)

lemma Lang_of_mono: "P \<subseteq> P' \<Longrightarrow> Lang_of P w \<subseteq> Lang_of P' w"
  using derives_mono by (auto simp: Lang_of_def)

lemma derive_set_subset:
  "P \<turnstile> u \<Rightarrow> v \<Longrightarrow> set v \<subseteq> set u \<union> Syms P"
by (auto simp: derive_iff Syms_def)

lemma deriven_set_subset:
  "P \<turnstile> u \<Rightarrow>(n) v \<Longrightarrow> set v \<subseteq> set u \<union> Syms P"
  by (induction n arbitrary: u)
    (auto simp: relpowp_Suc_left dest!: derive_set_subset)

lemma derives_set_subset:
  "P \<turnstile> u \<Rightarrow>* v \<Longrightarrow> set v \<subseteq> set u \<union> Syms P"
  by (auto simp: rtranclp_power dest!: deriven_set_subset)

lemma derive_Nts_syms_subset:
  "P \<turnstile> u \<Rightarrow> v \<Longrightarrow> Nts_syms v \<subseteq> Nts_syms u \<union> Rhs_Nts P"
by(auto simp: Rhs_Nts_def derive_iff)

lemma deriven_Nts_syms_subset:
  "P \<turnstile> u \<Rightarrow>(n) v \<Longrightarrow> Nts_syms v \<subseteq> Nts_syms u \<union> Rhs_Nts P"
  by (induction n arbitrary: u)
    (auto simp: relpowp_Suc_left dest!: derive_Nts_syms_subset)

lemma derives_Nts_syms_subset:
  "P \<turnstile> u \<Rightarrow>* v \<Longrightarrow> Nts_syms v \<subseteq> Nts_syms u \<union> Rhs_Nts P"
  by (auto simp: rtranclp_power dest!: deriven_Nts_syms_subset)

lemma derive_Tms_syms_subset:
  "P \<turnstile> u \<Rightarrow> v \<Longrightarrow> Tms_syms v \<subseteq> Tms_syms u \<union> Tms P"
by(auto simp: Tms_def derive_iff)

lemma deriven_Tms_syms_subset:
  "P \<turnstile> u \<Rightarrow>(n) v \<Longrightarrow> Tms_syms v \<subseteq> Tms_syms u \<union> Tms P"
  by (induction n arbitrary: u)
    (auto simp: relpowp_Suc_left dest!: derive_Tms_syms_subset)

lemma derives_Tms_syms_subset:
  "P \<turnstile> u \<Rightarrow>* v \<Longrightarrow> Tms_syms v \<subseteq> Tms_syms u \<union> Tms P"
  by (auto simp: rtranclp_power dest!: deriven_Tms_syms_subset)


subsubsection "Customized Induction Principles"

lemma deriven_induct[consumes 1, case_names 0 Suc]:
  assumes "P \<turnstile> xs \<Rightarrow>(n) ys"
  and "Q 0 xs"
  and "\<And>n u A v w. \<lbrakk> P \<turnstile> xs \<Rightarrow>(n) u @ [Nt A] @ v; Q n (u @ [Nt A] @ v); (A,w) \<in> P \<rbrakk> \<Longrightarrow> Q (Suc n) (u @ w @ v)"
  shows "Q n ys"
using assms(1) proof (induction n arbitrary: ys)
  case 0
  thus ?case using assms(2) by auto
next
  case (Suc n)
  from relpowp_Suc_E[OF Suc.prems]
  obtain xs' where n: "P \<turnstile> xs \<Rightarrow>(n) xs'" and 1: "P \<turnstile> xs' \<Rightarrow> ys" by auto
  from derive.cases[OF 1] obtain u A v w where "xs' = u @ [Nt A] @ v" "(A,w) \<in> P" "ys = u @ w @ v"
    by metis
  with Suc.IH[OF n] assms(3) n
  show ?case by blast
qed

lemma derives_induct[consumes 1, case_names base step]:
  assumes "P \<turnstile> xs \<Rightarrow>* ys"
  and "Q xs"
  and "\<And>u A v w. \<lbrakk> P \<turnstile> xs \<Rightarrow>* u @ [Nt A] @ v; Q (u @ [Nt A] @ v); (A,w) \<in> P \<rbrakk> \<Longrightarrow> Q (u @ w @ v)"
  shows "Q ys"
using assms
proof (induction rule: rtranclp_induct)
  case base
  from this(1) show ?case .
next
  case step
  from derive.cases[OF step(2)] step(1,3-) show ?case by metis
qed

lemma converse_derives_induct[consumes 1, case_names base step]:
  assumes "P \<turnstile> xs \<Rightarrow>* ys"
  and Base: "Q ys"
  and Step: "\<And>u A v w. \<lbrakk> P \<turnstile> u @ [Nt A] @ v \<Rightarrow>* ys; Q (u @ w @ v); (A,w) \<in> P \<rbrakk> \<Longrightarrow> Q (u @ [Nt A] @ v)"
  shows "Q xs"
  using assms(1)
  apply (induction rule: converse_rtranclp_induct)
  by (auto elim!: derive.cases intro!: Base Step intro: derives_rule)


subsubsection "(De)composing derivations"

lemma derive_append:
  "\<G> \<turnstile> u \<Rightarrow> v \<Longrightarrow> \<G> \<turnstile> u@w \<Rightarrow> v@w"
apply(erule derive.cases)
using derive.intros by fastforce

lemma derive_prepend:
  "\<G> \<turnstile> u \<Rightarrow> v \<Longrightarrow> \<G> \<turnstile> w@u \<Rightarrow> w@v"
apply(erule derive.cases)
by (metis append.assoc derive.intros)

lemma deriven_append:
  "P \<turnstile> u \<Rightarrow>(n) v \<Longrightarrow> P \<turnstile> u @ w \<Rightarrow>(n) v @ w"
  apply (induction n arbitrary: v)
   apply simp
  using derive_append by (fastforce simp: relpowp_Suc_right)

lemma deriven_prepend:
  "P \<turnstile> u \<Rightarrow>(n) v \<Longrightarrow> P \<turnstile> w @ u \<Rightarrow>(n) w @ v"
  apply (induction n arbitrary: v)
   apply simp
  using derive_prepend by (fastforce simp: relpowp_Suc_right)

lemma derives_append:
  "P \<turnstile> u \<Rightarrow>* v \<Longrightarrow> P \<turnstile> u@w \<Rightarrow>* v@w"
  by (metis deriven_append rtranclp_power)

lemma derives_prepend:
  "P \<turnstile> u \<Rightarrow>* v \<Longrightarrow> P \<turnstile> w@u \<Rightarrow>* w@v"
  by (metis deriven_prepend rtranclp_power)

lemma derive_append_decomp:
  "P \<turnstile> u@v \<Rightarrow> w \<longleftrightarrow>
  (\<exists>u'. w = u'@v \<and> P \<turnstile> u \<Rightarrow> u') \<or> (\<exists>v'. w = u@v' \<and> P \<turnstile> v \<Rightarrow> v')"
(is "?l \<longleftrightarrow> ?r")
proof
  assume ?l
  then obtain A r u1 u2
    where Ar: "(A,r) \<in> P"
      and uv: "u@v = u1 @ Nt A # u2"
      and w: "w = u1 @ r @ u2"
    by (auto simp: derive_iff)
  from uv have "(\<exists>s. u2 = s @ v \<and> u = u1 @ Nt A # s) \<or>
  (\<exists>s. u1 = u@s \<and> v = s @ Nt A # u2)"
    by (auto simp: append_eq_append_conv2 append_eq_Cons_conv)
  with Ar w show "?r" by (fastforce simp: derive_iff)
next
  show "?r \<Longrightarrow> ?l"
    by (auto simp add: derive_append derive_prepend)
qed

lemma deriven_append_decomp:
  "P \<turnstile> u @ v \<Rightarrow>(n) w \<longleftrightarrow>
  (\<exists>n1 n2 w1 w2. n = n1 + n2 \<and> w = w1 @ w2 \<and> P \<turnstile> u \<Rightarrow>(n1) w1 \<and> P \<turnstile> v \<Rightarrow>(n2) w2)"
  (is "?l \<longleftrightarrow> ?r")
proof
  show "?l \<Longrightarrow> ?r"
  proof (induction n arbitrary: u v)
    case 0
    then show ?case by simp
  next
    case (Suc n)
    from Suc.prems
    obtain u' v'
      where or: "P \<turnstile> u \<Rightarrow> u' \<and> v' = v \<or> u' = u \<and> P \<turnstile> v \<Rightarrow> v'"
        and n: "P \<turnstile> u'@v' \<Rightarrow>(n) w"
      by (fastforce simp: relpowp_Suc_left derive_append_decomp)
    from Suc.IH[OF n] or
    show ?case
      apply (elim disjE)
       apply (metis add_Suc relpowp_Suc_I2)
      by (metis add_Suc_right relpowp_Suc_I2)
  qed
next
  assume ?r
  then obtain n1 n2 w1 w2
    where [simp]: "n = n1 + n2" "w = w1 @ w2"
      and u: "P \<turnstile> u \<Rightarrow>(n1) w1" and v: "P \<turnstile> v \<Rightarrow>(n2) w2"
    by auto
  from u deriven_append
  have "P \<turnstile> u @ v \<Rightarrow>(n1) w1 @ v" by fastforce
  also from v deriven_prepend
  have "P \<turnstile> w1 @ v \<Rightarrow>(n2) w1 @ w2" by fastforce
  finally show ?l by auto
qed

lemma derives_append_decomp:
  "P \<turnstile> u @ v \<Rightarrow>* w \<longleftrightarrow> (\<exists>u' v'. P \<turnstile> u \<Rightarrow>* u' \<and> P \<turnstile> v \<Rightarrow>* v' \<and> w = u' @ v')"
  by (auto simp: rtranclp_power deriven_append_decomp)

lemma derives_concat:
  "\<forall>i \<in> set is. P \<turnstile> f i \<Rightarrow>* g i \<Longrightarrow> P \<turnstile> concat(map f is) \<Rightarrow>* concat(map g is)"
proof(induction "is")
  case Nil
  then show ?case by auto
next
  case Cons
  thus ?case by(auto simp: derives_append_decomp less_Suc_eq)
qed

lemma derives_concat1:
  "\<forall>i \<in> set is. P \<turnstile> [f i] \<Rightarrow>* g i \<Longrightarrow> P \<turnstile> map f is \<Rightarrow>* concat(map g is)"
using derives_concat[where f = "\<lambda>i. [f i]"] by auto

lemma derive_Cons:
"P \<turnstile> u \<Rightarrow> v \<Longrightarrow> P \<turnstile> a#u \<Rightarrow> a#v"
  using derive_prepend[of P u v "[a]"] by auto

lemma derives_Cons:
"R \<turnstile> u \<Rightarrow>* v \<Longrightarrow> R \<turnstile> a#u \<Rightarrow>* a#v"
  using derives_prepend[of _ _ _ "[a]"] by simp

lemma derive_from_empty[simp]:
  "P \<turnstile> [] \<Rightarrow> w \<longleftrightarrow> False"
  by (auto simp: derive_iff)

lemma deriven_from_empty[simp]:
  "P \<turnstile> [] \<Rightarrow>(n) w \<longleftrightarrow> n = 0 \<and> w = []"
  by (induct n, auto simp: relpowp_Suc_left)

lemma derives_from_empty[simp]:
  "\<G> \<turnstile> [] \<Rightarrow>* w \<longleftrightarrow> w = []"
  by (auto elim: converse_rtranclpE)

lemma deriven_start1:
  assumes "P \<turnstile> [Nt A] \<Rightarrow>(n) map Tm w"
  shows "\<exists>\<alpha> m. n = Suc m \<and> P \<turnstile> \<alpha> \<Rightarrow>(m) (map Tm w) \<and> (A,\<alpha>) \<in> P"
proof (cases n)
  case 0
  thus ?thesis
    using assms by auto
next
  case (Suc m)
  then obtain \<alpha> where *: "P \<turnstile> [Nt A] \<Rightarrow> \<alpha>" "P \<turnstile> \<alpha> \<Rightarrow>(m) map Tm w"
    using assms by (meson relpowp_Suc_E2)
  from  derive.cases[OF *(1)] have "(A, \<alpha>) \<in> P"
    by (simp add: Cons_eq_append_conv) blast
  thus ?thesis using *(2) Suc by auto
qed

lemma derives_start1: "P \<turnstile> [Nt A] \<Rightarrow>* map Tm w  \<Longrightarrow>  \<exists>\<alpha>. P \<turnstile> \<alpha> \<Rightarrow>* map Tm w \<and> (A,\<alpha>) \<in> P"
using deriven_start1 by (metis rtranclp_power)

lemma notin_Lhss_iff_Rhss: "A \<notin> Lhss P \<longleftrightarrow> Rhss P A = {}"
  by (auto simp: Lhss_def Rhss_def)

lemma Lang_empty_if_notin_Lhss: "A \<notin> Lhss P \<Longrightarrow> Lang P A = {}"
unfolding Lhss_def Lang_def
using derives_start1 by fastforce

lemma derive_Tm_Cons:
  "P \<turnstile> Tm a # u \<Rightarrow> v \<longleftrightarrow> (\<exists>w. v = Tm a # w \<and> P \<turnstile> u \<Rightarrow> w)"
  by (fastforce simp: derive_iff Cons_eq_append_conv)

lemma deriven_Tm_Cons:
  "P \<turnstile> Tm a # u \<Rightarrow>(n) v \<longleftrightarrow> (\<exists>w. v = Tm a # w \<and> P \<turnstile> u \<Rightarrow>(n) w)"
proof (induction n arbitrary: u)
  case 0
  show ?case by auto
next
  case (Suc n)
  then show ?case by (force simp: derive_Tm_Cons relpowp_Suc_left OO_def)
qed

lemma deriven_Tms_prepend: "R \<turnstile> map Tm t @ u \<Rightarrow>(n) v \<Longrightarrow> \<exists>v1. v = map Tm t @ v1 \<and> R \<turnstile> u \<Rightarrow>(n) v1"
  by (induction t arbitrary: v) (auto simp add: deriven_Tm_Cons)  

lemma derives_Tm_Cons:
  "P \<turnstile> Tm a # u \<Rightarrow>* v \<longleftrightarrow> (\<exists>w. v = Tm a # w \<and> P \<turnstile> u \<Rightarrow>* w)"
  by (metis deriven_Tm_Cons rtranclp_power)

lemma derives_Tm[simp]: "P \<turnstile> [Tm a] \<Rightarrow>* w \<longleftrightarrow> w = [Tm a]"
by(simp add: derives_Tm_Cons)

lemma derive_singleton: "P \<turnstile> [a] \<Rightarrow> u \<longleftrightarrow> (\<exists>A. (A,u) \<in> P \<and> a = Nt A)"
  by (auto simp: derive_iff Cons_eq_append_conv)

lemma deriven_singleton: "P \<turnstile> [a] \<Rightarrow>(n) u \<longleftrightarrow> (
  case n of 0 \<Rightarrow> u = [a]
   | Suc m \<Rightarrow> \<exists>(A,w) \<in> P. a = Nt A \<and> P \<turnstile> w \<Rightarrow>(m) u)"
  (is "?l \<longleftrightarrow> ?r")
proof
  show "?l \<Longrightarrow> ?r"
  proof (induction n)
    case 0
    then show ?case by simp
  next
    case (Suc n)
    then show ?case
      by (smt (verit, ccfv_threshold) case_prod_conv derive_singleton nat.simps(5) relpowp_Suc_E2)
  qed
  show "?r \<Longrightarrow> ?l"
    by (auto simp: derive_singleton relpowp_Suc_I2 split: nat.splits)
qed

lemma deriven_Cons_decomp:
  "P \<turnstile> a # u \<Rightarrow>(n) v \<longleftrightarrow>
  (\<exists>v2. v = a#v2 \<and> P \<turnstile> u \<Rightarrow>(n) v2) \<or>
  (\<exists>n1 n2 A w v1 v2. n = Suc (n1 + n2) \<and> v = v1 @ v2 \<and> a = Nt A \<and>
   (A,w) \<in> P \<and> P \<turnstile> w \<Rightarrow>(n1) v1 \<and> P \<turnstile> u \<Rightarrow>(n2) v2)"
(is "?l = ?r")
proof
  assume ?l
  then obtain n1 n2 v1 v2
    where [simp]: "n = n1 + n2" "v = v1 @ v2"
      and 1: "P \<turnstile> [a] \<Rightarrow>(n1) v1" and 2: "P \<turnstile> u \<Rightarrow>(n2) v2"
    unfolding deriven_append_decomp[of n P "[a]" u v,simplified]
    by auto
  show ?r
  proof (cases "n1")
    case 0
    with 1 2 show ?thesis by auto
  next
    case [simp]: (Suc m)
    with 1 obtain A w
      where [simp]: "a = Nt A" "(A,w) \<in> P" and w: "P \<turnstile> w \<Rightarrow>(m) v1"
      by (auto simp: deriven_singleton)
    with 2
    have "n = Suc (m + n2) \<and> v = v1 @ v2 \<and> a = Nt A \<and>
   (A,w) \<in> P \<and> P \<turnstile> w \<Rightarrow>(m) v1 \<and> P \<turnstile> u \<Rightarrow>(n2) v2"
      by auto
    then show ?thesis
      by (auto simp: append_eq_Cons_conv)
  qed
next
  assume "?r"
  then
  show "?l"
  proof (elim disjE exE conjE)
    fix v2
    assume [simp]: "v = a # v2" and u: "P \<turnstile> u \<Rightarrow>(n) v2"
    from deriven_prepend[OF u, of "[a]"]
    show ?thesis
      by auto
  next
    fix n1 n2 A w v1 v2
    assume [simp]: "n = Suc (n1 + n2)" "v = v1 @ v2" "a = Nt A"
      and Aw: "(A, w) \<in> P"
      and w: "P \<turnstile> w \<Rightarrow>(n1) v1"
      and u: "P \<turnstile> u \<Rightarrow>(n2) v2"
    have "P \<turnstile> [a] \<Rightarrow> w"
      by (simp add: Aw derive_singleton)
    with w have "P \<turnstile> [a] \<Rightarrow>(Suc n1) v1"
      by (metis relpowp_Suc_I2)
    from deriven_append[OF this]
    have 1: "P \<turnstile> a#u \<Rightarrow>(Suc n1) v1@u"
      by auto
    also have "P \<turnstile> \<dots> \<Rightarrow>(n2) v1@v2"
      using deriven_prepend[OF u].
    finally
    show ?thesis by simp
  qed
qed

lemma derives_Cons_decomp:
  "P \<turnstile> s # u \<Rightarrow>* v \<longleftrightarrow>
  (\<exists>v2. v = s#v2 \<and> P \<turnstile> u \<Rightarrow>* v2) \<or>
  (\<exists>A w v1 v2. v = v1 @ v2 \<and> s = Nt A \<and> (A,w) \<in> P \<and> P \<turnstile> w \<Rightarrow>* v1 \<and> P \<turnstile> u \<Rightarrow>* v2)" (is "?L \<longleftrightarrow> ?R")
proof
  assume ?L thus ?R using deriven_Cons_decomp[of _ P s u v] by (metis rtranclp_power)
next
  assume ?R thus ?L by (meson derives_Cons derives_Cons_rule derives_append_decomp)
qed

lemma deriven_Suc_decomp_left:
  "P \<turnstile> u \<Rightarrow>(Suc n) v \<longleftrightarrow> (\<exists>p A u2 w v1 v2 n1 n2.
  u = p @ Nt A # u2 \<and> v = p @ v1 @ v2 \<and> n = n1 + n2 \<and>
  (A,w) \<in> P \<and> P \<turnstile> w \<Rightarrow>(n1) v1 \<and>
  P \<turnstile> u2 \<Rightarrow>(n2) v2)" (is "?l \<longleftrightarrow> ?r")
proof
  show "?r \<Longrightarrow> ?l"
    by (auto intro!: deriven_prepend simp: deriven_Cons_decomp)
  show "?l \<Longrightarrow> ?r"
  proof(induction u arbitrary: v n)
    case Nil
    then show ?case by auto
  next
    case (Cons a u)
    from Cons.prems[unfolded deriven_Cons_decomp]
    show ?case
    proof (elim disjE exE conjE, goal_cases)
      case (1 v2)
      with Cons.IH[OF this(2)] show ?thesis
        by (metis append_Cons)
    next
      case (2 n1 n2 A w v1 v2)
      then show ?thesis by (fastforce simp:Cons_eq_append_conv)
    qed
  qed
qed

lemma derives_NilD: "P \<turnstile> w \<Rightarrow>* [] \<Longrightarrow> s \<in> set w \<Longrightarrow> P \<turnstile> [s] \<Rightarrow>* []"
proof(induction arbitrary: s rule: converse_derives_induct)
  case base
  then show ?case by simp
next
  case (step u A v w)
  then show ?case using derives_append_decomp[where u="[Nt A]" and v=v]
    by (auto simp: derives_append_decomp)
qed

lemma derives_append_append:
  "P \<turnstile> \<alpha> \<Rightarrow>* \<alpha>' \<Longrightarrow> P \<turnstile> \<beta> \<Rightarrow>* \<beta>' \<Longrightarrow> P \<turnstile> \<alpha> @ \<beta> \<Rightarrow>* \<alpha>' @ \<beta>'"
  using derives_append_decomp by blast

lemma derives_append_Nt_Cons:
"(B,\<beta>) \<in> P \<Longrightarrow>
  P \<turnstile> \<alpha> \<Rightarrow>* \<alpha>' \<Longrightarrow> P \<turnstile> \<beta> \<Rightarrow>* \<beta>' \<Longrightarrow> P \<turnstile> \<gamma> \<Rightarrow>* \<gamma>' \<Longrightarrow>
  P \<turnstile> \<alpha> @ Nt B # \<gamma> \<Rightarrow>* \<alpha>' @ \<beta>' @ \<gamma>'"
  by (metis derives_Cons_decomp derives_append_decomp)

lemma derives_simul_rules:
  assumes "\<And>A w. (A,w) \<in> P \<Longrightarrow> P' \<turnstile> [Nt A] \<Rightarrow>* w"
  shows "P \<turnstile> w \<Rightarrow>* w' \<Longrightarrow> P' \<turnstile> w \<Rightarrow>* w'"
proof(induction rule: derives_induct)
  case base
  then show ?case by simp
next
  case (step u A v w)
  then show ?case
    by (meson assms derives_append derives_prepend rtranclp_trans)
qed

subsubsection \<open>Derivations leading to terminal words\<close>

lemma derive_decomp_Tm: "P \<turnstile> \<alpha> \<Rightarrow>(n) map Tm \<beta> \<Longrightarrow>
  \<exists>\<beta>s ns. \<beta> = concat \<beta>s \<and> length \<alpha> = length \<beta>s \<and> length \<alpha> = length ns \<and> sum_list ns = n
          \<and> (\<forall>i < length \<beta>s. P \<turnstile> [\<alpha> ! i] \<Rightarrow>(ns!i) map Tm (\<beta>s ! i))"
  (is "_ \<Longrightarrow> \<exists>\<beta>s ns. ?G \<alpha> \<beta> n \<beta>s ns")
proof (induction \<alpha> arbitrary: \<beta> n)
  case (Cons s \<alpha>)
  from deriven_Cons_decomp[THEN iffD1, OF Cons.prems]
  show ?case
  proof (elim disjE exE conjE)
    fix \<gamma> assume as: "map Tm \<beta> = s # \<gamma>" "P \<turnstile> \<alpha> \<Rightarrow>(n) \<gamma>"
    then obtain s' \<gamma>' where "\<beta> = s' # \<gamma>'"  "P \<turnstile> \<alpha> \<Rightarrow>(n) map Tm \<gamma>'" "s = Tm s'" by force
    from Cons.IH[OF this(2)] obtain \<beta>s ns where *: "?G \<alpha> \<gamma>' n \<beta>s ns"
      by blast
    let ?\<beta>s = "[s']#\<beta>s"
    let ?ns = "0#ns"
    have "?G (s#\<alpha>) \<beta> n ?\<beta>s ?ns"
      using \<open>\<beta> = _\<close> as * by (auto simp: nth_Cons')
    then show ?thesis by blast
  next
    fix n1 n2 A w \<beta>1 \<beta>2
    assume *: "n = Suc (n1 + n2)" "map Tm \<beta> = \<beta>1 @ \<beta>2" "s = Nt A" "(A, w) \<in> P" "P \<turnstile> w \<Rightarrow>(n1) \<beta>1" "P \<turnstile> \<alpha> \<Rightarrow>(n2) \<beta>2"
    then obtain \<beta>1' \<beta>2' where **: "\<beta> = \<beta>1' @ \<beta>2'" "P \<turnstile> w \<Rightarrow>(n1) map Tm \<beta>1'" "P \<turnstile> \<alpha> \<Rightarrow>(n2) map Tm \<beta>2'"
      by (metis (no_types, lifting) append_eq_map_conv)
    from Cons.IH[OF this(3)] obtain \<beta>s ns
      where ***: "?G \<alpha> \<beta>2' n2 \<beta>s ns"
      by blast
    let ?\<beta>s = "\<beta>1'#\<beta>s"
    let ?ns = "Suc n1 # ns"
    from * ** have "P \<turnstile> [(s#\<alpha>) ! 0] \<Rightarrow>(?ns ! 0) map Tm (?\<beta>s ! 0)"
      by (metis derive_singleton nth_Cons_0 relpowp_Suc_I2)
    then have "?G (s#\<alpha>) \<beta> n ?\<beta>s ?ns"
      using * ** *** by (auto simp add: nth_Cons' derives_Cons_rule fold_plus_sum_list_rev)
    then show ?thesis by blast
  qed
qed simp

lemma word_decomp1: 
  "R \<turnstile> p @ [Nt A] @ map Tm ts \<Rightarrow>(n) map Tm q 
  \<Longrightarrow> \<exists>pt At w k m. R \<turnstile> p \<Rightarrow>(k) map Tm pt \<and> R \<turnstile> w \<Rightarrow>(m) map Tm At \<and> (A, w) \<in> R 
      \<and> q = pt @ At @ ts \<and> n = Suc(k + m)"
proof -
  assume assm: "R \<turnstile> p @ [Nt A] @ map Tm ts \<Rightarrow>(n) map Tm q"
  then obtain q1 where P: "R \<turnstile> p@[Nt A] \<Rightarrow>(n) q1 \<and> map Tm q = q1 @ map Tm ts"
    unfolding deriven_append_decomp
    by (metis add.commute add_0 append.assoc not_derive_from_Tms relpowp_E2)
  then obtain q1t where "q1 = map Tm q1t" "q = q1t @ ts"
    by (metis map_Tm_inject_iff map_eq_append_conv)
  with P obtain pt At w k m where P2: "R \<turnstile> p \<Rightarrow>(k) map Tm pt \<and> R \<turnstile> w \<Rightarrow>(m) map Tm At \<and> (A, w) \<in> R 
     \<and> q1t = pt @ At \<and> n = Suc(k + m)" 
    by(fastforce simp: deriven_append_decomp map_eq_append_conv dest: deriven_start1)
  then have "q = pt @ At @ ts" using \<open>q = _\<close> by simp
  then show ?thesis using P2 by blast
qed

lemma deriven_start_sent: 
  "R \<turnstile> u @ Nt V # w \<Rightarrow>(Suc n) map Tm x \<Longrightarrow> \<exists>v. (V, v) \<in> R \<and> R \<turnstile> u @ v @ w \<Rightarrow>(n) map Tm x"
proof -
  assume assm: "R \<turnstile> u @ Nt V # w \<Rightarrow>(Suc n) map Tm x"
  then obtain n1 n2 xu xvw 
    where P1: "Suc n = n1 + n2 \<and> map Tm x = xu @ xvw \<and> R \<turnstile> u \<Rightarrow>(n1) xu \<and> R \<turnstile> Nt V # w \<Rightarrow>(n2) xvw"
    by (auto simp add: deriven_append_decomp)
  then have t: "\<nexists>t. xvw = Nt V # t"
    by (metis append_eq_map_conv map_eq_Cons_D sym.distinct(1))
  then obtain n3 n4 v xv xw 
    where P2: "n2 = Suc (n3 + n4) \<and> xvw = xv @ xw \<and> (V,v) \<in> R \<and> R \<turnstile> v \<Rightarrow>(n3) xv \<and> R \<turnstile> w \<Rightarrow>(n4) xw"
    using P1 t by (auto simp add: deriven_Cons_decomp)
  then have "R \<turnstile> v @ w \<Rightarrow>(n3 + n4) xvw" using P2
    using deriven_append_decomp diff_Suc_1 by blast
  then have "R \<turnstile> u @ v @ w \<Rightarrow>(n1 + n3 + n4) map Tm x" using P1 P2 deriven_append_decomp
    using ab_semigroup_add_class.add_ac(1) by blast
  then have "R \<turnstile> u @ v @ w \<Rightarrow>(n) map Tm x" using P1 P2
    by (simp add: add.assoc)
  then show ?thesis using P2 by blast
qed

lemma deriven_Nt_Cons_map_Tm: "P \<turnstile> Nt A # \<beta> \<Rightarrow>(n) map Tm w \<longleftrightarrow>
(\<exists>\<alpha> m l v u. (A,\<alpha>) \<in> P \<and> P \<turnstile> \<alpha> \<Rightarrow>(m) map Tm v \<and> P \<turnstile> \<beta> \<Rightarrow>(l) map Tm u \<and>
  n = Suc (m + l) \<and> w = v @ u)"
  by (force simp: deriven_Cons_decomp map_eq_append_conv)

lemma deriven_Tm_Cons_map_Tm: "P \<turnstile> Tm a # \<beta> \<Rightarrow>(n) map Tm w \<longleftrightarrow>
(\<exists>v. P \<turnstile> \<beta> \<Rightarrow>(n) map Tm v \<and> w = a # v)"
  by (auto simp: deriven_Tm_Cons)

lemma deriven_Cons_map_Tm:
  "P \<turnstile> x # u \<Rightarrow>(n) map Tm w \<longleftrightarrow>
  (\<exists>a v2. x = Tm a \<and> w = a # v2 \<and> P \<turnstile> u \<Rightarrow>(n) map Tm v2) \<or>
  (\<exists>n1 n2 A \<alpha> v1 v2. n = Suc (n1 + n2) \<and> w = v1 @ v2 \<and> x = Nt A \<and>
   (A,\<alpha>) \<in> P \<and> P \<turnstile> \<alpha> \<Rightarrow>(n1) map Tm v1 \<and> P \<turnstile> u \<Rightarrow>(n2) map Tm v2)"
  apply (cases x)
   apply (force simp: deriven_Nt_Cons_map_Tm)
  by (force simp: deriven_Tm_Cons_map_Tm)

lemma deriven_append_map_Tm: "P \<turnstile> \<alpha> @ \<beta> \<Rightarrow>(n) map Tm w \<longleftrightarrow>
  (\<exists>m l v u. P \<turnstile> \<alpha> \<Rightarrow>(m) map Tm v \<and> P \<turnstile> \<beta> \<Rightarrow>(l) map Tm u \<and> n = m + l \<and> w = v @ u)"
proof (induction \<alpha> arbitrary: \<beta> n w)
  case Nil
  show ?case by simp
next
  case (Cons x \<alpha>)
  show ?case
  proof (cases x)
    case x: (Tm a)
    show ?thesis by (force simp: x deriven_Tm_Cons_map_Tm Cons)
  next
    case x: (Nt A)
    show ?thesis
    proof safe
      assume "P \<turnstile> (x # \<alpha>) @ \<beta> \<Rightarrow>(n) map Tm w"
      from this[unfolded x append.simps deriven_Nt_Cons_map_Tm]
      obtain \<gamma> m l v u where
        n: "n = Suc (m + l)" and A: "(A,\<gamma>) \<in> P" and w: "w = v @ u"
        and \<gamma>v: "P \<turnstile> \<gamma> \<Rightarrow>(m) map Tm v" and \<alpha>\<beta>: "P \<turnstile> \<alpha> @ \<beta> \<Rightarrow>(l) map Tm u"
        by auto
      from A \<gamma>v have Av: "P \<turnstile> [Nt A] \<Rightarrow>(Suc m) map Tm v"
        by (simp add: derive_singleton relpowp_Suc_I2)
      from \<alpha>\<beta>[unfolded Cons]
      obtain k j t s where l: "l = k + j" and u: "u = t @ s"
        and \<alpha>: "P \<turnstile> \<alpha> \<Rightarrow>(k) map Tm t" and \<beta>: "P \<turnstile> \<beta> \<Rightarrow>(j) map Tm s" by auto
      from Av \<alpha> have x\<alpha>: "P \<turnstile> x # \<alpha> \<Rightarrow>(Suc m + k) map Tm (v @ t)"
        by (force simp: x deriven_Nt_Cons_map_Tm simp del: map_append)
      show "\<exists>m l v u.
       P \<turnstile> x # \<alpha> \<Rightarrow>(m) map Tm v \<and>
       P \<turnstile> \<beta> \<Rightarrow>(l) map Tm u \<and> n = m + l \<and>
       w = v @ u"
        apply (intro exI conjI)
           apply (fact x\<alpha>)
          apply (fact \<beta>)
        by (auto simp: n l w u)
    next
      fix m l v u
      assume n: "n = m + l" and w: "w = v @ u"
        and x\<alpha>: "P \<turnstile> x # \<alpha> \<Rightarrow>(m) map Tm v"
        and \<beta>: "P \<turnstile> \<beta> \<Rightarrow>(l) map Tm u"
      from x\<alpha>[unfolded x deriven_Nt_Cons_map_Tm]
      obtain \<gamma> k j t s where
        m: "m = Suc (k + j)" and A: "(A,\<gamma>) \<in> P" and v: "v = t @ s"
        and \<gamma>: "P \<turnstile> \<gamma> \<Rightarrow>(k) map Tm t" and \<alpha>: "P \<turnstile> \<alpha> \<Rightarrow>(j) map Tm s"
        by auto
      show "P \<turnstile> (x # \<alpha>) @ \<beta> \<Rightarrow>(m + l) map Tm (v @ u)"
        apply (unfold x append.simps deriven_Nt_Cons_map_Tm)
      proof (intro exI conjI)
        show "m + l = Suc (k + (j + l))" by (simp add: m)
        show "(A,\<gamma>) \<in> P" using A.
        show "v @ u = t @ s @ u" by (simp add: v)
        show  "P \<turnstile> \<gamma> \<Rightarrow>(k) map Tm t" using \<gamma>.
        from \<alpha> \<beta> show \<alpha>\<beta>: "P \<turnstile> \<alpha> @ \<beta> \<Rightarrow>(j+l) map Tm (s @ u)"
          by (unfold Cons, auto)
      qed
    qed
  qed
qed

lemma deriven_Nt_map_Tm: "P \<turnstile> \<alpha> @ Nt B # \<gamma> \<Rightarrow>(n) map Tm w \<longleftrightarrow>
(\<exists>\<beta> m l k v u t. (B,\<beta>) \<in> P \<and>
  P \<turnstile> \<alpha> \<Rightarrow>(m) map Tm v \<and> P \<turnstile> \<beta> \<Rightarrow>(l) map Tm u \<and> P \<turnstile> \<gamma> \<Rightarrow>(k) map Tm t \<and>
  n = Suc (m + l + k) \<and> w = v @ u @ t)"
  by (force simp: deriven_append_map_Tm deriven_Nt_Cons_map_Tm)

lemma map_Tm_Nt_eq_map_Tm_Nt:
  "map Tm xs @ Nt y # zs = map Tm xs' @ Nt y' # zs' \<longleftrightarrow> xs = xs' \<and> y = y' \<and> zs = zs'"
  apply (subst append_Cons_eq_append_Cons)
  by auto

lemma deriven_Suc_map_Tm_decomp: "P \<turnstile> \<alpha> \<Rightarrow>(Suc n) map Tm w \<longleftrightarrow>
(\<exists>v B \<beta> \<gamma> u t m l. (B,\<beta>) \<in> P \<and> P \<turnstile> \<beta> \<Rightarrow>(m) map Tm u \<and> P \<turnstile> \<gamma> \<Rightarrow>(l) map Tm t \<and>
  n = m + l \<and> \<alpha> = map Tm v @ Nt B # \<gamma> \<and> w = v @ u @ t)"
  by (fastforce simp: deriven_Suc_decomp_left map_eq_append_conv map_Tm_Nt_eq_map_Tm_Nt append_eq_map_conv)

lemma derives_append_map_Tm:
  "P \<turnstile> \<alpha> @ \<beta> \<Rightarrow>* map Tm w \<longleftrightarrow>
  (\<exists>v u. P \<turnstile> \<alpha> \<Rightarrow>* map Tm v \<and> P \<turnstile> \<beta> \<Rightarrow>* map Tm u \<and> w = v @ u)"
  by (force simp: rtranclp_power deriven_append_map_Tm)

lemma derives_Nt_map_Tm:
 "P \<turnstile> \<alpha> @ Nt B # \<gamma> \<Rightarrow>* map Tm w \<longleftrightarrow>
  (\<exists>\<beta> v u t. (B,\<beta>) \<in> P \<and>
    P \<turnstile> \<alpha> \<Rightarrow>* map Tm v \<and> P \<turnstile> \<beta> \<Rightarrow>* map Tm u \<and> P \<turnstile> \<gamma> \<Rightarrow>* map Tm t \<and>
    w = v @ u @ t)"
  by (force simp: rtranclp_power deriven_Nt_map_Tm)

lemma derives_Nt_Cons_map_Tm:
 "P \<turnstile> Nt A # \<beta> \<Rightarrow>* map Tm w \<longleftrightarrow>
  (\<exists>\<alpha> v u. (A,\<alpha>) \<in> P \<and> P \<turnstile> \<alpha> \<Rightarrow>* map Tm v \<and> P \<turnstile> \<beta> \<Rightarrow>* map Tm u \<and> w = v @ u)"
  using derives_Nt_map_Tm[where \<alpha> = "[]"] by simp

lemma derives_Nt_Cons_Lang:
"P \<turnstile> Nt A # \<alpha> \<Rightarrow>* map Tm w \<longleftrightarrow> (\<exists>v u. v \<in> Lang P A \<and> P \<turnstile> \<alpha> \<Rightarrow>* map Tm u \<and> w = v @ u)"
  by (force simp: derives_Cons_decomp Lang_def map_eq_Cons_conv map_eq_append_conv)

lemma Lang_of_Nil[simp]: "Lang_of P [] = {[]}"
  by (auto simp: Lang_of_def)

lemma Lang_of_iff_derives: "w \<in> Lang_of P \<alpha> \<longleftrightarrow> P \<turnstile> \<alpha> \<Rightarrow>* map Tm w"
  by (auto simp: Lang_of_def)

lemma Lang_ofE_deriven:
  assumes "w \<in> Lang_of P \<alpha>" and "\<And>n. P \<turnstile> \<alpha> \<Rightarrow>(n) map Tm w \<Longrightarrow> thesis"
  shows thesis
  using assms by (auto simp: Lang_of_iff_derives rtranclp_power)

lemma Lang_of_Tm_Cons: "Lang_of P (Tm a # \<alpha>) = {[a]} @@ Lang_of P \<alpha>"
  by (auto simp: Lang_of_def derives_Tm_Cons conc_def)

lemma Lang_of_map_Tm: "Lang_of P (map Tm w) = {w}"
  by (induction w, simp_all add: Lang_of_Tm_Cons insert_conc)

lemma Lang_of_Nt_Cons: "Lang_of P (Nt A # \<alpha>) = Lang P A @@ Lang_of P \<alpha>"
  by (force simp add: Lang_of_def Lang_def derives_Cons_decomp map_eq_append_conv conc_def)

lemma Lang_of_Cons: "Lang_of P (x # \<alpha>) = (case x of Tm a \<Rightarrow> {[a]} | Nt A \<Rightarrow> Lang P A) @@ Lang_of P \<alpha>"
  by (simp add: Lang_of_Tm_Cons Lang_of_Nt_Cons split: sym.splits)

lemma Lang_of_append: "Lang_of P (\<alpha> @ \<beta>) = Lang_of P \<alpha> @@ Lang_of P \<beta>"
  by (induction \<alpha> arbitrary: \<beta>, simp_all add: Lang_of_Cons conc_assoc split: sym.splits)

lemma Lang_of_set_conc: "Lang_of_set P (X @@ Y) = Lang_of_set P X @@ Lang_of_set P Y"
  by (force simp: Lang_of_append elim!: concE)

lemma Lang_of_set_Rhss: "Lang_of_set P (Rhss P A) = Lang P A"
  by (auto simp: Lang_def Lang_of_def Rhss_def converse_rtranclp_into_rtranclp derive_singleton
      dest: derives_start1)

lemma Lang_of_prod_subset: "(A,\<alpha>) \<in> P \<Longrightarrow> Lang_of P \<alpha> \<subseteq> Lang P A"
  apply (fold Lang_of_set_Rhss) by (auto simp: Rhss_def)

lemma Lang_le_iff_Lang_of_le: "Lang P \<le> Lang P' \<longleftrightarrow> Lang_of P \<le> Lang_of P'"
proof (safe intro!: le_funI)
  fix \<alpha> w
  assume le: "Lang P \<le> Lang P'" and w: "w \<in> Lang_of P \<alpha>"
  from w show "w \<in> Lang_of P' \<alpha>"
    apply (induction \<alpha> arbitrary: w)
    using le[THEN le_funD, THEN subsetD]
    by (auto simp: Lang_of_Cons insert_conc split: sym.splits)
next
  fix A w
  assume le: "Lang_of P \<le> Lang_of P'" and w: "w \<in> Lang P A"
  from le[THEN le_funD, of "[Nt A]"] w
  show "w \<in> Lang P' A" by (auto simp: Lang_of_Cons)
qed

lemma Lang_eq_iff_Lang_of_eq: "Lang P = Lang P' \<longleftrightarrow> Lang_of P = Lang_of P'"
  apply (subst eq_iff) by (auto simp: Lang_le_iff_Lang_of_le)

lemma Lang_of_le_iff_derives:
  "Lang_of P \<le> Lang_of P' \<longleftrightarrow> (\<forall>\<alpha> w. P \<turnstile> \<alpha> \<Rightarrow>* map Tm w \<longrightarrow> P' \<turnstile> \<alpha> \<Rightarrow>* map Tm w)"
  by (auto simp: Lang_of_def le_fun_def)

lemma Lang_le_iff_derives:
  "Lang P \<le> Lang P' \<longleftrightarrow> (\<forall>\<alpha> w. P \<turnstile> \<alpha> \<Rightarrow>* map Tm w \<longrightarrow> P' \<turnstile> \<alpha> \<Rightarrow>* map Tm w)"
  by (simp only: Lang_le_iff_Lang_of_le Lang_of_le_iff_derives)

lemma Lang_eq_iff_derives:
  "Lang P = Lang P' \<longleftrightarrow> (\<forall>\<alpha> w. P \<turnstile> \<alpha> \<Rightarrow>* map Tm w \<longleftrightarrow> P' \<turnstile> \<alpha> \<Rightarrow>* map Tm w)"
  apply (subst eq_iff) by (auto simp: Lang_le_iff_derives)

lemma Rhss_le_Ders: "Rhss P \<le> Ders P"
  by (auto simp: le_fun_def Rhss_def Ders_def derive_singleton)

lemma Lang_of_set_pow: "Lang_of_set P (X ^^ n) = Lang_of_set P X ^^ n"
  by (induction n, simp_all add: Lang_of_set_conc)

lemma Lang_of_set_star: "Lang_of_set P (star X) = star (Lang_of_set P X)"
  by (auto simp: star_def Lang_of_set_pow)

text \<open>Bottom-up definition of \<open>\<Rightarrow>*\<close>. Single definition yields more compact inductions.
But \<open>derives_induct\<close> may already do the job.\<close>

inductive derives_bu :: "('n, 't) Prods \<Rightarrow> ('n,'t) syms \<Rightarrow> ('n,'t) syms \<Rightarrow> bool"
  ("(2_ \<turnstile>/ (_/ \<Rightarrow>bu/ _))" [50, 0, 50] 50) for P :: "('n, 't) Prods"
  where
bu_refl: "P \<turnstile> \<alpha> \<Rightarrow>bu \<alpha>" |
bu_prod: "(A,\<alpha>) \<in> P \<Longrightarrow> P \<turnstile> [Nt A] \<Rightarrow>bu \<alpha>" |
bu_embed: "\<lbrakk> P \<turnstile> \<alpha> \<Rightarrow>bu \<alpha>\<^sub>1 @ \<alpha>\<^sub>2 @ \<alpha>\<^sub>3; P \<turnstile> \<alpha>\<^sub>2 \<Rightarrow>bu \<beta> \<rbrakk> \<Longrightarrow> P \<turnstile> \<alpha> \<Rightarrow>bu \<alpha>\<^sub>1 @ \<beta> @ \<alpha>\<^sub>3"

lemma derives_if_bu: "P \<turnstile> \<alpha> \<Rightarrow>bu \<beta> \<Longrightarrow> P \<turnstile> \<alpha> \<Rightarrow>* \<beta>"
proof(induction rule: derives_bu.induct)
  case (bu_refl) then show ?case by simp
next
  case (bu_prod A \<alpha>) then show ?case by (simp add: derives_Cons_rule)
next
  case (bu_embed \<alpha> \<alpha>\<^sub>1 \<alpha>\<^sub>2 \<alpha>\<^sub>3 \<beta>) then show ?case
    by (meson derives_append derives_prepend rtranclp_trans)
qed

lemma derives_bu_if: "P \<turnstile> \<alpha> \<Rightarrow>* \<beta> \<Longrightarrow> P \<turnstile> \<alpha> \<Rightarrow>bu \<beta>"
proof(induction rule: derives_induct)
  case base
  then show ?case by (simp add: bu_refl)
next
  case (step u A v w)
  then show ?case
    by (meson bu_embed bu_prod)
qed

lemma derives_bu_iff: "P \<turnstile> \<alpha> \<Rightarrow>bu \<beta> \<longleftrightarrow> P \<turnstile> \<alpha> \<Rightarrow>* \<beta>"
by (meson derives_bu_if derives_if_bu)


subsubsection "Leftmost/Rightmost Derivations"

inductive derivel :: "('n,'t) Prods \<Rightarrow> ('n,'t) syms \<Rightarrow> ('n,'t)syms \<Rightarrow> bool"
  ("(2_ \<turnstile>/ (_ \<Rightarrow>l/ _))" [50, 0, 50] 50) where
"(A,\<alpha>) \<in> P \<Longrightarrow> P \<turnstile> map Tm u @ Nt A # v \<Rightarrow>l map Tm u @ \<alpha> @ v"

abbreviation derivels ("(2_ \<turnstile>/ (_ \<Rightarrow>l*/ _))" [50, 0, 50] 50) where
"P \<turnstile> u \<Rightarrow>l* v \<equiv> ((derivel P) ^**) u v"

abbreviation derivels1 ("(2_ \<turnstile>/ (_ \<Rightarrow>l+/ _))" [50, 0, 50] 50) where
"P \<turnstile> u \<Rightarrow>l+ v \<equiv> ((derivel P) ^++) u v"

abbreviation deriveln ("(2_ \<turnstile>/ (_ \<Rightarrow>l'(_')/ _))" [50, 0, 0, 50] 50) where
"P \<turnstile> u \<Rightarrow>l(n) v \<equiv> ((derivel P) ^^ n) u v"

inductive deriver :: "('n,'t) Prods \<Rightarrow> ('n,'t) syms \<Rightarrow> ('n,'t)syms \<Rightarrow> bool"
  ("(2_ \<turnstile>/ (_ \<Rightarrow>r/ _))" [50, 0, 50] 50) where
"(A,\<alpha>) \<in> P \<Longrightarrow> P \<turnstile> u @ Nt A # map Tm v \<Rightarrow>r u @ \<alpha> @ map Tm v"

abbreviation derivers ("(2_ \<turnstile>/ (_ \<Rightarrow>r*/ _))" [50, 0, 50] 50) where
"P \<turnstile> u \<Rightarrow>r* v \<equiv> ((deriver P) ^**) u v"

abbreviation derivers1 ("(2_ \<turnstile>/ (_ \<Rightarrow>r+/ _))" [50, 0, 50] 50) where
"P \<turnstile> u \<Rightarrow>r+ v \<equiv> ((deriver P) ^++) u v"

abbreviation derivern ("(2_ \<turnstile>/ (_ \<Rightarrow>r'(_')/ _))" [50, 0, 0, 50] 50) where
"P \<turnstile> u \<Rightarrow>r(n) v \<equiv> ((deriver P) ^^ n) u v"


lemma derivel_iff: "R \<turnstile> u \<Rightarrow>l v \<longleftrightarrow>
 (\<exists> (A,w) \<in> R. \<exists>u1 u2. u = map Tm u1 @ Nt A # u2 \<and> v = map Tm u1 @ w @ u2)"
  by (auto simp: derivel.simps)

lemma derivel_from_empty[simp]:
  "P \<turnstile> [] \<Rightarrow>l w \<longleftrightarrow> False" by (auto simp: derivel_iff)

lemma deriveln_from_empty[simp]:
  "P \<turnstile> [] \<Rightarrow>l(n) w \<longleftrightarrow> n = 0 \<and> w = []"
  by (induct n, auto simp: relpowp_Suc_left)

lemma derivels_from_empty[simp]:
  "\<G> \<turnstile> [] \<Rightarrow>l* w \<longleftrightarrow> w = []"
  by (auto elim: converse_rtranclpE)

lemma Un_derivel: "R \<union> S \<turnstile> y \<Rightarrow>l z \<longleftrightarrow> R \<turnstile> y \<Rightarrow>l z \<or> S \<turnstile> y \<Rightarrow>l z"
  by (fastforce simp: derivel_iff)

lemma derivel_append:
  "P \<turnstile> u \<Rightarrow>l v \<Longrightarrow> P \<turnstile> u @ w \<Rightarrow>l v @ w"
  by (force simp: derivel_iff)

lemma deriveln_append:
  "P \<turnstile> u \<Rightarrow>l(n) v \<Longrightarrow> P \<turnstile> u @ w \<Rightarrow>l(n) v @ w"
proof (induction n arbitrary: u)
  case 0
  then show ?case by simp
next
  case (Suc n)
  then obtain y where uy: "P \<turnstile> u \<Rightarrow>l y" and yv: "P \<turnstile> y \<Rightarrow>l(n) v"
    by (auto simp: relpowp_Suc_left)
  have "P \<turnstile> u @ w \<Rightarrow>l y @ w"
    using derivel_append[OF uy].
  also from Suc.IH yv have "P \<turnstile> \<dots> \<Rightarrow>l(n) v @ w" by auto
  finally show ?case.
qed

lemma derivels_append:
  "P \<turnstile> u \<Rightarrow>l* v \<Longrightarrow> P \<turnstile> u @ w \<Rightarrow>l* v @ w"
  by (metis deriveln_append rtranclp_power)

lemma derivels1_append:
  "P \<turnstile> u \<Rightarrow>l+ v \<Longrightarrow> P \<turnstile> u @ w \<Rightarrow>l+ v @ w"
  by (metis deriveln_append tranclp_power)

lemma derivel_Tm_Cons:
  "P \<turnstile> Tm a # u \<Rightarrow>l v \<longleftrightarrow> (\<exists>w. v = Tm a # w \<and> P \<turnstile> u \<Rightarrow>l w)"
apply (cases v)
 apply (simp add: derivel_iff)
apply (fastforce simp: derivel.simps Cons_eq_append_conv Cons_eq_map_conv)
done

lemma deriveln_Tm_Cons:
  "P \<turnstile> Tm a # u \<Rightarrow>l(n) v \<longleftrightarrow> (\<exists>w. v = Tm a # w \<and> P \<turnstile> u \<Rightarrow>l(n) w)"
  by (induction n arbitrary: u v;
      fastforce simp: derivel_Tm_Cons relpowp_Suc_right OO_def)

lemma derivels_Tm_Cons:
  "P \<turnstile> Tm a # u \<Rightarrow>l* v \<longleftrightarrow> (\<exists>w. v = Tm a # w \<and> P \<turnstile> u \<Rightarrow>l* w)"
  by (metis deriveln_Tm_Cons rtranclp_power)

lemma derivel_map_Tm_append:
  "P \<turnstile> map Tm w @ u \<Rightarrow>l v \<longleftrightarrow> (\<exists>x. v = map Tm w @ x \<and> P \<turnstile> u \<Rightarrow>l x)"
  apply (induction w arbitrary:v)
  by (auto simp: derivel_Tm_Cons Cons_eq_append_conv)

lemma deriveln_map_Tm_append:
  "P \<turnstile> map Tm w @ u \<Rightarrow>l(n) v \<longleftrightarrow> (\<exists>x. v = map Tm w @ x \<and> P \<turnstile> u \<Rightarrow>l(n) x)"
  by (induction n arbitrary: u;
      force simp: derivel_map_Tm_append relpowp_Suc_left OO_def)

lemma derivels_map_Tm_append:
  "P \<turnstile> map Tm w @ u \<Rightarrow>l* v \<longleftrightarrow> (\<exists>x. v = map Tm w @ x \<and> P \<turnstile> u \<Rightarrow>l* x)"
  by (metis deriveln_map_Tm_append rtranclp_power)

lemma derivel_Nt_Cons:
  "P \<turnstile> Nt A # u \<Rightarrow>l v \<longleftrightarrow> (\<exists>w. (A,w) \<in> P \<and> v = w @ u)"
  by (auto simp: derivel_iff Cons_eq_append_conv Cons_eq_map_conv)

lemma derivels1_Nt_Cons:
  "P \<turnstile> Nt A # u \<Rightarrow>l+ v \<longleftrightarrow> (\<exists>w. (A,w) \<in> P \<and> P \<turnstile> w @ u \<Rightarrow>l* v)"
  by (auto simp: tranclp_unfold_left derivel_Nt_Cons OO_def)

lemma derivels_Nt_Cons:
  "P \<turnstile> Nt A # u \<Rightarrow>l* v \<longleftrightarrow> v = Nt A # u \<or> (\<exists>w. (A,w) \<in> P \<and> P \<turnstile> w @ u \<Rightarrow>l* v)"
  by (auto simp: Nitpick.rtranclp_unfold derivels1_Nt_Cons)

lemma deriveln_Nt_Cons:
  "P \<turnstile> Nt A # u \<Rightarrow>l(n) v \<longleftrightarrow> (
  case n of 0 \<Rightarrow> v = Nt A # u
  | Suc m \<Rightarrow> \<exists>w. (A,w) \<in> P \<and> P \<turnstile> w @ u \<Rightarrow>l(m) v)"
  by (cases n) (auto simp: derivel_Nt_Cons relpowp_Suc_left OO_def)

lemma derivel_Cons:
  "P \<turnstile> x # u \<Rightarrow>l v \<longleftrightarrow>
  (case x of Nt A \<Rightarrow> \<exists>w. (A,w) \<in> P \<and> v = w @ u | Tm a \<Rightarrow> \<exists>w. v = Tm a # w \<and> P \<turnstile> u \<Rightarrow>l w)"
  by (auto simp: derivel_Nt_Cons derivel_Tm_Cons split: sym.splits)

lemma deriveln_Cons:
"P \<turnstile> x # u \<Rightarrow>l(n) v \<longleftrightarrow> (
  case n of 0 \<Rightarrow> v = x # u
  | Suc m \<Rightarrow> (
    case x of Nt A \<Rightarrow> \<exists>w. (A,w) \<in> P \<and> P \<turnstile> w @ u \<Rightarrow>l(m) v
    | Tm a \<Rightarrow> \<exists>w. v = Tm a # w \<and> P \<turnstile> u \<Rightarrow>l(n) w))"
  by (auto simp: deriveln_Nt_Cons deriveln_Tm_Cons split: nat.splits sym.splits)

lemma derivel_not_elim_Tm:
  "P \<turnstile> xs \<Rightarrow>l map Nt w \<Longrightarrow> \<exists>v. xs = map Nt v"
  by (cases xs)
    (auto simp: derivel_Cons Cons_eq_map_conv map_eq_append_conv split: sym.splits)

lemma deriveln_not_elim_Tm:
  assumes "P \<turnstile> xs \<Rightarrow>l(n) map Nt w"
  shows "\<exists>v. xs = map Nt v"
using assms
proof (induction n arbitrary: xs)
  case 0
  then show ?case by auto
next
  case (Suc n)
  then obtain z where "P \<turnstile> xs \<Rightarrow>l z" and "P \<turnstile> z \<Rightarrow>l(n) map Nt w"
    using relpowp_Suc_E2 by metis
  with Suc show ?case using derivel_not_elim_Tm
    by blast
qed

lemma decomp_derivel_map_Nts:
  assumes "P \<turnstile> map Nt Xs \<Rightarrow>l map Nt Zs"
  shows "\<exists>X Xs' Ys. Xs = X # Xs' \<and> P \<turnstile> [Nt X] \<Rightarrow>l map Nt Ys \<and> Zs = Ys @ Xs'"
using assms unfolding derivel_iff
by (fastforce simp: map_eq_append_conv split: prod.splits)

lemma derivel_imp_derive: "P \<turnstile> u \<Rightarrow>l v \<Longrightarrow> P \<turnstile> u \<Rightarrow> v"
  using derive.simps derivel.cases self_append_conv2 by fastforce

(* TODO: CFG? *)
lemma derivel_append_iff:
  "P \<turnstile> u@v \<Rightarrow>l w \<longleftrightarrow>
  (\<exists>u'. w = u'@v \<and> P \<turnstile> u \<Rightarrow>l u') \<or> (\<exists>u' v'. w = u@v' \<and> u = map Tm u' \<and> P \<turnstile> v \<Rightarrow>l v')"
(is "?l \<longleftrightarrow> ?r")
proof
  assume ?l
  then obtain A r u1 u2
    where Ar: "(A,r) \<in> P"
      and uv: "u@v = map Tm u1 @ Nt A # u2"
      and w: "w = map Tm u1 @ r @ u2"
    by (auto simp: derivel_iff)
  from uv have case_dist: "(\<exists>s. u2 = s @ v \<and> u = map Tm u1 @ Nt A # s) \<or>
  (\<exists>s. map Tm u1 = u @ s  \<and> v = s @ Nt A # u2)" (is "?h1 \<or> ?h2")
    by (auto simp: append_eq_append_conv2 append_eq_Cons_conv)
  show ?r proof (rule disjE[OF case_dist])
    assume ?h1
    with Ar w show ?thesis by (fastforce simp: derivel_iff)
  next
    assume ?h2
    then obtain s where map_u1_def: "map Tm u1 = u @ s" and v_def: "v = s @ Nt A # u2" by blast
    from map_u1_def obtain u' s' where u_def: "u = map Tm u'" and s_def: "s = map Tm s'"
      using append_eq_map_conv[of u s Tm u1] by auto

    from w map_u1_def s_def have "w = u @ (map Tm s' @ r @ u2)" by simp

    moreover from Ar v_def s_def have "P \<turnstile> v \<Rightarrow>l map Tm s' @ r @ u2"
      using derivel_iff[of P] by blast

    ultimately show ?thesis
      using u_def by blast
  qed
next
  show "?r \<Longrightarrow> ?l"
    by (auto simp add: derivel_append derivel_map_Tm_append)
qed

lemma deriveln_ConsD:
  assumes "P \<turnstile> x#v \<Rightarrow>l(n) u"
  shows "(\<exists>u'. u = u' @ v \<and> P \<turnstile> [x] \<Rightarrow>l(n) u') \<or> (\<exists>w\<^sub>1 u\<^sub>2 m\<^sub>1 m\<^sub>2. m\<^sub>1 + m\<^sub>2 = n \<and> u = map Tm w\<^sub>1 @ u\<^sub>2 
                                                    \<and> P \<turnstile> [x] \<Rightarrow>l(m\<^sub>1) map Tm w\<^sub>1 \<and> P \<turnstile> v \<Rightarrow>l(m\<^sub>2) u\<^sub>2)"
using assms proof (induction n arbitrary: u)
  case (Suc n)
  from Suc(2) obtain w where x_v_deriveln_w: "P \<turnstile> x # v \<Rightarrow>l(n) w" and w_derivel_u: "P \<turnstile> w \<Rightarrow>l u"
    by (metis relpowp_Suc_E)
  from Suc(1)[OF x_v_deriveln_w] have IH: "(\<exists>u'. w = u' @ v \<and> P \<turnstile> [x] \<Rightarrow>l(n) u') \<or>
  (\<exists>w\<^sub>1 u\<^sub>2 m\<^sub>1 m\<^sub>2. m\<^sub>1 + m\<^sub>2 = n \<and> w = map Tm w\<^sub>1 @ u\<^sub>2 \<and> P \<turnstile> [x] \<Rightarrow>l(m\<^sub>1) map Tm w\<^sub>1 \<and> P \<turnstile> v \<Rightarrow>l(m\<^sub>2) u\<^sub>2)" (is "?l \<or> ?r") .
  show ?case proof (rule disjE[OF IH])
    assume ?l
    then obtain u' where w_def: "w = u' @ v" and x_deriveln_u': "P \<turnstile> [x] \<Rightarrow>l(n) u'" by blast
    from w_def w_derivel_u have "P \<turnstile> u' @ v \<Rightarrow>l u" by simp
    hence case_dist: "(\<exists>u\<^sub>0. u = u\<^sub>0 @ v \<and> P \<turnstile> u' \<Rightarrow>l u\<^sub>0) \<or>
                  (\<exists>u\<^sub>1 u\<^sub>2. u = u' @ u\<^sub>2 \<and> u' = map Tm u\<^sub>1 \<and> P \<turnstile> v \<Rightarrow>l u\<^sub>2)" (is "?h1 \<or> ?h2")
      using derivel_append_iff[of P u' v u] by simp
    show ?thesis proof (rule disjE[OF case_dist])
      assume ?h1
      then obtain u\<^sub>0 where u_def: "u = u\<^sub>0 @ v" and u'_derivel_u0: "P \<turnstile> u' \<Rightarrow>l u\<^sub>0" by blast
      from x_deriveln_u' u'_derivel_u0 have "P \<turnstile> [x] \<Rightarrow>l(Suc n) u\<^sub>0" by (simp add: relpowp_Suc_I)
      with u_def show ?thesis by blast
    next
      assume ?h2
      then obtain u\<^sub>1 u\<^sub>2 where u_def: "u = u' @ u\<^sub>2" and u'_def: "u' = map Tm u\<^sub>1" and v_derivel_u2: "P \<turnstile> v \<Rightarrow>l u\<^sub>2" by blast
      from x_deriveln_u' u'_def have "P \<turnstile> [x] \<Rightarrow>l(n) map Tm u\<^sub>1" by simp
      with u_def u'_def v_derivel_u2 show ?thesis by fastforce
    qed
  next
    assume ?r
    then obtain w\<^sub>1 u\<^sub>2 m\<^sub>1 m\<^sub>2 where m1_m2_n: "m\<^sub>1 + m\<^sub>2 = n" and w_def: "w = map Tm w\<^sub>1 @ u\<^sub>2" and 
                                      x_derivelm1_w1: "P \<turnstile> [x] \<Rightarrow>l(m\<^sub>1) map Tm w\<^sub>1" and v_derivelm2_u2: "P \<turnstile> v \<Rightarrow>l(m\<^sub>2) u\<^sub>2" by blast
    from w_def w_derivel_u have "P \<turnstile> map Tm w\<^sub>1 @ u\<^sub>2 \<Rightarrow>l u" by simp
    then obtain u' where u_def: "u = map Tm w\<^sub>1 @ u'" and u2_derivel_u': "P \<turnstile> u\<^sub>2 \<Rightarrow>l u'"
      using derivel_map_Tm_append by blast

    from m1_m2_n have "m\<^sub>1 + Suc m\<^sub>2 = Suc n" by simp

    moreover from v_derivelm2_u2 u2_derivel_u' have "P \<turnstile> v \<Rightarrow>l(Suc m\<^sub>2) u'"
      by (simp add: relpowp_Suc_I)

    ultimately show ?thesis
      using u_def x_derivelm1_w1 by blast
  qed
qed simp

lemma deriveln_Cons_TmsD:
  assumes "P \<turnstile> x#v \<Rightarrow>l(n) map Tm w"
  shows "\<exists>w\<^sub>1 w\<^sub>2 m\<^sub>1 m\<^sub>2. m\<^sub>1 + m\<^sub>2 = n \<and> w = w\<^sub>1 @ w\<^sub>2 \<and> P \<turnstile> [x] \<Rightarrow>l(m\<^sub>1) map Tm w\<^sub>1 \<and> P \<turnstile> v \<Rightarrow>l(m\<^sub>2) map Tm w\<^sub>2"
proof -
  have case_dist: "(\<exists>u'. map Tm w = u' @ v \<and> P \<turnstile> [x] \<Rightarrow>l(n) u') \<or> (\<exists>w\<^sub>1 u\<^sub>2 m\<^sub>1 m\<^sub>2. m\<^sub>1 + m\<^sub>2 = n \<and> map Tm w = map Tm w\<^sub>1 @ u\<^sub>2 
                                                    \<and> P \<turnstile> [x] \<Rightarrow>l(m\<^sub>1) map Tm w\<^sub>1 \<and> P \<turnstile> v \<Rightarrow>l(m\<^sub>2) u\<^sub>2)" (is "?l \<or> ?r")
    using deriveln_ConsD[OF assms] by simp
  show ?thesis proof (rule disjE[OF case_dist])
    assume ?l
    then obtain u' where map_w_def: "map Tm w = u' @ v" and x_derives_u': "P \<turnstile> [x] \<Rightarrow>l(n) u'" by blast
    from map_w_def obtain w\<^sub>1 w\<^sub>2 where "w = w\<^sub>1 @ w\<^sub>2" and map_w\<^sub>1_def: "map Tm w\<^sub>1 = u'" and "map Tm w\<^sub>2 = v"
      using map_eq_append_conv[of Tm w u' v] by blast

    moreover from x_derives_u' map_w\<^sub>1_def have "P \<turnstile> [x] \<Rightarrow>l(n) map Tm w\<^sub>1" by simp

    moreover have "P \<turnstile> map Tm w\<^sub>2 \<Rightarrow>l(0) map Tm w\<^sub>2" by simp

    ultimately show ?thesis by force
  next
    assume ?r
    then obtain w\<^sub>1 u\<^sub>2 m\<^sub>1 m\<^sub>2 where m1_m2_n: "m\<^sub>1 + m\<^sub>2 = n" and map_w_def: "map Tm w = map Tm w\<^sub>1 @ u\<^sub>2" 
                                               and x_derivelm1_w1: "P \<turnstile> [x] \<Rightarrow>l(m\<^sub>1) map Tm w\<^sub>1" and v_derivelm2_u2: "P \<turnstile> v \<Rightarrow>l(m\<^sub>2) u\<^sub>2" by blast
    from map_w_def obtain w\<^sub>1' u\<^sub>2' where "w = w\<^sub>1' @ u\<^sub>2'" and "map (Tm) w\<^sub>1 = map Tm w\<^sub>1'" and "u\<^sub>2 = map (Tm) u\<^sub>2'"
      using map_eq_append_conv[of "Tm" w "map Tm w\<^sub>1" u\<^sub>2] by auto
    with m1_m2_n x_derivelm1_w1 v_derivelm2_u2 show ?thesis by auto
  qed                    
qed

lemma deriveln_imp_deriven:
  "P \<turnstile> u \<Rightarrow>l(n) v \<Longrightarrow> P \<turnstile> u \<Rightarrow>(n) v"
  using relpowp_mono derivel_imp_derive by metis

lemma derivels_imp_derives:
  "P \<turnstile> u \<Rightarrow>l* v \<Longrightarrow> P \<turnstile> u \<Rightarrow>* v"
  by (metis derivel_imp_derive mono_rtranclp)

lemma deriveln_iff_deriven:
  "P \<turnstile> u \<Rightarrow>l(n) map Tm v \<longleftrightarrow> P \<turnstile> u \<Rightarrow>(n) map Tm v"
  (is "?l \<longleftrightarrow> ?r")
proof
  show "?l \<Longrightarrow> ?r" using deriveln_imp_deriven.
  assume ?r
  then show "?l"
  proof (induction n arbitrary: u v rule: less_induct)
    case (less n)
    from \<open>P \<turnstile> u \<Rightarrow>(n) map Tm v\<close>
    show ?case
    proof (induction u arbitrary: v)
      case Nil
      then show ?case by simp
    next
      case (Cons a u)
      show ?case
        using Cons.prems(1) Cons.IH less.IH
        by (auto simp: deriven_Cons_decomp deriveln_Tm_Cons deriveln_Nt_Cons)
           (metis deriven_append_decomp lessI)
    qed
  qed
qed

lemma derivels_iff_derives: "P \<turnstile> u \<Rightarrow>l* map Tm v \<longleftrightarrow> P \<turnstile> u \<Rightarrow>* map Tm v"
  using deriveln_iff_deriven
  by (metis rtranclp_power)

lemma deriver_iff: "R \<turnstile> u \<Rightarrow>r v \<longleftrightarrow>
  (\<exists> (A,w) \<in> R. \<exists>u1 u2. u = u1 @ Nt A # map Tm u2 \<and> v = u1 @ w @ map Tm u2)"
  by (auto simp: deriver.simps)

lemma deriver_imp_derive: "R \<turnstile> u \<Rightarrow>r v \<Longrightarrow> R \<turnstile> u \<Rightarrow> v"
  by (auto simp: deriver_iff derive_iff)

lemma derivern_imp_deriven: "R \<turnstile> u \<Rightarrow>r(n) v \<Longrightarrow> R \<turnstile> u \<Rightarrow>(n) v"
  by (metis (no_types, lifting) deriver_imp_derive relpowp_mono)

lemma derivers_imp_derives: "R \<turnstile> u \<Rightarrow>r* v \<Longrightarrow> R \<turnstile> u \<Rightarrow>* v"
  by (metis deriver_imp_derive mono_rtranclp)

lemma deriver_iff_rev_derivel:
  "P \<turnstile> u \<Rightarrow>r v \<longleftrightarrow> map_prod id rev ` P \<turnstile> rev u \<Rightarrow>l rev v" (is "?l \<longleftrightarrow> ?r")
proof
  assume ?l
  then obtain A w u1 u2 where Aw: "(A,w) \<in> P"
    and u: "u = u1 @ Nt A # map Tm u2"
    and v: "v = u1 @ w @ map Tm u2" by (auto simp: deriver.simps)
  from bexI[OF _ Aw] have "(A, rev w) \<in> map_prod id rev ` P" by (auto simp: image_def)
  from derivel.intros[OF this, of "rev u2" "rev u1"] u v
  show ?r by (simp add: rev_map)
next
  assume ?r
  then obtain A w u1 u2 where Aw: "(A,w) \<in> P"
    and u: "u = u1 @ Nt A # map Tm u2"
    and v: "v = u1 @ w @ map Tm u2"
    by (auto simp: derivel_iff rev_eq_append_conv rev_map)
  then show ?l by (auto simp: deriver_iff)
qed

lemma rev_deriver_iff_derivel:
  "map_prod id rev ` P \<turnstile> u \<Rightarrow>r v \<longleftrightarrow> P \<turnstile> rev u \<Rightarrow>l rev v"
  by (simp add: deriver_iff_rev_derivel image_image prod.map_comp o_def)

lemma derivern_iff_rev_deriveln:
  "P \<turnstile> u \<Rightarrow>r(n) v \<longleftrightarrow> map_prod id rev ` P \<turnstile> rev u \<Rightarrow>l(n) rev v"
proof (induction n arbitrary: u)
  case 0
  show ?case by simp
next
  case (Suc n)
  show ?case
    apply (unfold relpowp_Suc_left OO_def)
    apply (unfold Suc deriver_iff_rev_derivel)
    by (metis rev_rev_ident)
qed

lemma rev_derivern_iff_deriveln:
  "map_prod id rev ` P \<turnstile> u \<Rightarrow>r(n) v \<longleftrightarrow> P \<turnstile> rev u \<Rightarrow>l(n) rev v"
  by (simp add: derivern_iff_rev_deriveln image_image prod.map_comp o_def)

lemma derivers_iff_rev_derivels:
  "P \<turnstile> u \<Rightarrow>r* v \<longleftrightarrow> map_prod id rev ` P \<turnstile> rev u \<Rightarrow>l* rev v"
  using derivern_iff_rev_deriveln
  by (metis rtranclp_power)

lemma rev_derivers_iff_derivels:
  "map_prod id rev ` P \<turnstile> u \<Rightarrow>r* v \<longleftrightarrow> P \<turnstile> rev u \<Rightarrow>l* rev v"
  by (simp add: derivers_iff_rev_derivels image_image prod.map_comp o_def)

lemma rev_derive:
  "map_prod id rev ` P \<turnstile> u \<Rightarrow> v \<longleftrightarrow> P \<turnstile> rev u \<Rightarrow> rev v"
  by (force simp: derive_iff rev_eq_append_conv bex_pair_conv in_image_map_prod intro: exI[of _ "rev _"])

lemma rev_deriven:
  "map_prod id rev ` P \<turnstile> u \<Rightarrow>(n) v \<longleftrightarrow> P \<turnstile> rev u \<Rightarrow>(n) rev v"
apply (induction n arbitrary: u)
 apply simp
by (auto simp: relpowp_Suc_left OO_def rev_derive intro: exI[of _ "rev _"])

lemma rev_derives:
  "map_prod id rev ` P \<turnstile> u \<Rightarrow>* v \<longleftrightarrow> P \<turnstile> rev u \<Rightarrow>* rev v"
  using rev_deriven
  by (metis rtranclp_power)

lemma derivern_iff_deriven: "P \<turnstile> u \<Rightarrow>r(n) map Tm v \<longleftrightarrow> P \<turnstile> u \<Rightarrow>(n) map Tm v"
  by (auto simp: derivern_iff_rev_deriveln rev_map deriveln_iff_deriven rev_deriven)

lemma derivers_iff_derives: "P \<turnstile> u \<Rightarrow>r* map Tm v \<longleftrightarrow> P \<turnstile> u \<Rightarrow>* map Tm v"
  by (simp add: derivern_iff_deriven rtranclp_power)

lemma derivern_prepend: "R \<turnstile> u \<Rightarrow>r(n) v \<Longrightarrow> R \<turnstile> p @ u \<Rightarrow>r(n) p @ v"
  by (fastforce simp: derivern_iff_rev_deriveln rev_map deriveln_append rev_eq_append_conv)

lemma deriver_append_map_Tm:
  "P \<turnstile> u @ map Tm w \<Rightarrow>r v \<longleftrightarrow> (\<exists>x. v = x @ map Tm w \<and> P \<turnstile> u \<Rightarrow>r x)"
  by (fastforce simp: deriver_iff_rev_derivel rev_map derivel_map_Tm_append rev_eq_append_conv)

lemma derivern_append_map_Tm:
  "P \<turnstile> u @ map Tm w \<Rightarrow>r(n) v \<longleftrightarrow> (\<exists>x. v = x @ map Tm w \<and> P \<turnstile> u \<Rightarrow>r(n) x)"
  by (fastforce simp: derivern_iff_rev_deriveln rev_map deriveln_map_Tm_append rev_eq_append_conv)

lemma deriver_snoc_Nt:
  "P \<turnstile> u @ [Nt A] \<Rightarrow>r v \<longleftrightarrow> (\<exists>w. (A,w) \<in> P \<and> v = u @ w)"
  by (force simp: deriver_iff_rev_derivel derivel_Nt_Cons rev_eq_append_conv)

lemma deriver_singleton:
  "P \<turnstile> [Nt A] \<Rightarrow>r v \<longleftrightarrow> (A,v) \<in> P"
  using deriver_snoc_Nt[of P "[]"] by auto

lemma derivers1_snoc_Nt:
  "P \<turnstile> u @ [Nt A] \<Rightarrow>r+ v \<longleftrightarrow> (\<exists>w. (A,w) \<in> P \<and> P \<turnstile> u @ w \<Rightarrow>r* v)"
  by (auto simp: tranclp_unfold_left deriver_snoc_Nt OO_def)

lemma derivers_snoc_Nt:
  "P \<turnstile> u @ [Nt A] \<Rightarrow>r* v \<longleftrightarrow> v = u @ [Nt A] \<or> (\<exists>w. (A,w) \<in> P \<and> P \<turnstile> u @ w \<Rightarrow>r* v)"
  by (auto simp: Nitpick.rtranclp_unfold derivers1_snoc_Nt)

lemma derivern_snoc_Tm:
  "P \<turnstile> u @ [Tm a] \<Rightarrow>r(n) v \<longleftrightarrow> (\<exists>w. v = w @ [Tm a] \<and> P \<turnstile> u \<Rightarrow>r(n) w)"
  by (force simp: derivern_iff_rev_deriveln deriveln_Tm_Cons)

lemma derivern_snoc_Nt:
  "P \<turnstile> u @ [Nt A] \<Rightarrow>r(n) v \<longleftrightarrow> (
  case n of 0 \<Rightarrow> v = u @ [Nt A]
  | Suc m \<Rightarrow> \<exists>w. (A,w) \<in> P \<and> P \<turnstile> u @ w \<Rightarrow>r(m) v)"
  by (cases n) (auto simp: relpowp_Suc_left deriver_snoc_Nt OO_def)

lemma derivern_singleton:
  "P \<turnstile> [Nt A] \<Rightarrow>r(n) v \<longleftrightarrow> (
  case n of 0 \<Rightarrow> v = [Nt A]
  | Suc m \<Rightarrow> \<exists>w. (A,w) \<in> P \<and> P \<turnstile> w \<Rightarrow>r(m) v)"
  using derivern_snoc_Nt[of n P "[]" A v] by (cases n, auto)

lemma derivern_snoc_Nt_Tms_decomp1: 
  "R \<turnstile> p @ [Nt A] \<Rightarrow>r(n) map Tm q 
   \<Longrightarrow> \<exists>pt At w k m. R \<turnstile> p \<Rightarrow>(k) map Tm pt \<and> R \<turnstile> w \<Rightarrow>(m) map Tm At \<and> (A, w) \<in> R 
        \<and> q = pt @ At \<and> n = Suc(k + m)"
proof-
  assume assm: "R \<turnstile> p @ [Nt A] \<Rightarrow>r(n) map Tm q"
  then have "R \<turnstile> p @ [Nt A] \<Rightarrow>(n) map Tm q" by (simp add: derivern_iff_deriven)
  then have "\<exists>n1 n2 q1 q2. n = n1 + n2 \<and> map Tm q = q1@q2 \<and> R \<turnstile> p \<Rightarrow>(n1) q1 \<and> R \<turnstile> [Nt A] \<Rightarrow>(n2) q2"
    using deriven_append_decomp by blast
  then obtain n1 n2 q1 q2 
    where decomp1: "n = n1 + n2 \<and> map Tm q = q1 @ q2 \<and> R \<turnstile> p \<Rightarrow>(n1) q1 \<and> R \<turnstile> [Nt A] \<Rightarrow>(n2) q2"
    by blast
  then have "\<exists>pt At. q1 = map Tm pt \<and> q2 = map Tm At \<and> q = pt @ At"
    by (meson map_eq_append_conv)
  then obtain pt At where decomp_tms: "q1 = map Tm pt \<and> q2 = map Tm At \<and> q = pt @ At" by blast
  then have "\<exists>w m. n2 = Suc m \<and> R \<turnstile> w \<Rightarrow>(m) (map Tm At) \<and> (A,w) \<in> R" 
    using decomp1 
    by (auto simp add: deriven_start1)
  then obtain w m where "n2 = Suc m \<and> R \<turnstile> w \<Rightarrow>(m) (map Tm At) \<and> (A,w) \<in> R" by blast
  then have "R \<turnstile> p \<Rightarrow>(n1) map Tm pt \<and> R \<turnstile> w \<Rightarrow>(m) map Tm At \<and> (A, w) \<in> R 
     \<and> q = pt @ At \<and> n = Suc(n1 + m)" 
    using decomp1 decomp_tms by auto
  then show ?thesis by blast
qed

subsection \<open>Redundant Productions\<close>

text \<open>Productions of the form \<open>A \<rightarrow> A\<close> are redundant.\<close>

lemma no_self_loops_derive:
  "reflclp (derive {(A,\<alpha>) \<in> P. \<alpha> \<noteq> [Nt A]}) = reflclp (derive P)"
  by (force simp: fun_eq_iff derive_iff)

lemma no_self_loops_derives:
  "{(A,\<alpha>) \<in> P. \<alpha> \<noteq> [Nt A]} \<turnstile> u \<Rightarrow>* v \<longleftrightarrow> P \<turnstile> u \<Rightarrow>* v"
  apply (subst rtranclp_reflclp[symmetric])
  by (simp add: no_self_loops_derive)

lemma Lang_of_no_self_loops:
  "Lang_of {(A,\<alpha>) \<in> P. \<alpha> \<noteq> [Nt A]} = Lang_of P"
  by (simp add: fun_eq_iff Lang_of_def no_self_loops_derives)

lemma Lang_no_self_loops:
  "Lang {(A,\<alpha>) \<in> P. \<alpha> \<noteq> [Nt A]} = Lang P"
  by (simp add: Lang_eq_iff_Lang_of_eq Lang_of_no_self_loops)

lemma Lang_eq_Rhss_no_self_loop:
  "Lang P A = Lang_of_set P (Rhss P A - {[Nt A]})"
proof-
  have "Lang P A = Lang {(A,\<alpha>) \<in> P. \<alpha> \<noteq> [Nt A]} A"
    by (simp add: Lang_no_self_loops)
  also have "\<dots> = Lang_of_set {(A,\<alpha>) \<in> P. \<alpha> \<noteq> [Nt A]} (Rhss P A - {[Nt A]})"
    by (auto simp: Lang_of_set_Rhss[symmetric] Rhss_def)
  finally show ?thesis by (simp add: Lang_of_no_self_loops)
qed

lemma no_self_loops_derivel:
  "reflclp (derivel {(A,\<alpha>) \<in> P. \<alpha> \<noteq> [Nt A]}) = reflclp (derivel P)"
  by (force simp: fun_eq_iff derivel_iff)

lemma no_self_loops_derivels:
  "{(A,\<alpha>) \<in> P. \<alpha> \<noteq> [Nt A]} \<turnstile> u \<Rightarrow>l* v \<longleftrightarrow> P \<turnstile> u \<Rightarrow>l* v"
  apply (subst rtranclp_reflclp[symmetric])
  by (simp add: no_self_loops_derivel)

text \<open>Rules that can be simulated by other rules are redundant.\<close>

lemma Rhss_le_Ders_imp_Lang_le: assumes "Rhss P \<le> Ders P'" shows "Lang P \<le> Lang P'"
  apply (unfold Lang_le_iff_derives)
proof (intro allI impI)
  fix \<alpha> w
  assume "P \<turnstile> \<alpha> \<Rightarrow>* map Tm w"
  then obtain n where "P \<turnstile> \<alpha> \<Rightarrow>(n) map Tm w" by (auto simp: rtranclp_power)
  then show "P' \<turnstile> \<alpha> \<Rightarrow>* map Tm w"
  proof (induction n arbitrary: \<alpha> w rule: less_induct)
    case (less n')
    show ?case
    proof (cases n')
      case 0
      with less.prems show ?thesis by simp
    next
      case [simp]: (Suc n)
      from less.prems[unfolded this deriven_Suc_map_Tm_decomp]
      obtain B \<beta> \<gamma> v u t m l where B: "(B,\<beta>) \<in> P"
        and lb: "P \<turnstile> \<beta> \<Rightarrow>(m) map Tm u" and lc: "P \<turnstile> \<gamma> \<Rightarrow>(l) map Tm t"
        and [simp]: "\<alpha> = map Tm v @ Nt B # \<gamma>" "w = v @ u @ t" "n = m+l"
        by blast
      from less.IH[OF _ lc] have c: "P' \<turnstile> \<gamma> \<Rightarrow>* map Tm t" by simp
      from assms[THEN le_funD, of B] B
      have "\<beta> \<in> Ders P' B" by (auto simp: Rhss_def)
      then have "P' \<turnstile> [Nt B] \<Rightarrow>* \<beta>" by (auto simp: Ders_def)
      from derives_prepend[OF derives_append[OF this]]
      have "P' \<turnstile> \<alpha> \<Rightarrow>* map Tm v @ \<beta> @ \<gamma>" by simp
      also from less.IH[OF _ lb] c have "P' \<turnstile> \<dots> \<Rightarrow>* map Tm w"
        by (auto intro!: derives_append_append)
      finally show ?thesis.
    qed
  qed
qed

lemma Lang_Un_redundant: assumes "Rhss R \<le> Ders P" shows "Lang (P \<union> R) = Lang P"
proof (rule antisym)
  show "Lang (P \<union> R) \<le> Lang P"
    apply (rule Rhss_le_Ders_imp_Lang_le)
    using assms Rhss_le_Ders[of P] by (simp add: le_fun_def Rhss_Un)
next
  show "Lang P \<le> Lang (P \<union> R)"
    apply (rule le_funI)
    apply (rule Lang_mono) by simp
qed

lemmas Lang_of_Un_redundant = Lang_Un_redundant[unfolded Lang_eq_iff_Lang_of_eq]

text \<open>Productions whose lhss do not appear in other rules are redundant.\<close>

lemma derive_Un_disj_Lhss:
  assumes \<alpha>: "Nts_syms \<alpha> \<inter> Lhss Q = {}"
  shows "P \<union> Q \<turnstile> \<alpha> \<Rightarrow> \<beta> \<longleftrightarrow> P \<turnstile> \<alpha> \<Rightarrow> \<beta>"
  using \<alpha> by (auto simp: Lhss_def derive_iff)

lemma deriven_Un_disj_Lhss:
  assumes PQ: "Rhs_Nts P \<inter> Lhss Q = {}" and \<alpha>: "Nts_syms \<alpha> \<inter> Lhss Q = {}"
  shows "P \<union> Q \<turnstile> \<alpha> \<Rightarrow>(n) \<beta> \<longleftrightarrow> P \<turnstile> \<alpha> \<Rightarrow>(n) \<beta>" (is "?l \<longleftrightarrow> ?r")
proof
  show "?l \<Longrightarrow> ?r"
  proof (induction n arbitrary: \<beta>)
    case 0
    then show ?case by simp
  next
    case (Suc n)
    from Suc.prems obtain \<beta>' where 1: "P \<union> Q \<turnstile> \<alpha> \<Rightarrow>(n) \<beta>'" and 2: "P \<union> Q \<turnstile> \<beta>' \<Rightarrow> \<beta>"
      by (auto simp: relpowp_Suc_right)
    from Suc.IH[OF 1] have P1: "P \<turnstile> \<alpha> \<Rightarrow>(n) \<beta>'".
    from deriven_Nts_syms_subset[OF P1] \<alpha> PQ
    have "Nts_syms \<beta>' \<inter> Lhss Q = {}" by auto
    from P1 2[unfolded derive_Un_disj_Lhss[OF this]]
    show ?case by (auto simp: relpowp_Suc_right)
  qed
next
  assume ?r
  from deriven_mono[OF _ this]
  show ?l by auto
qed

lemma derives_Un_disj_Lhss:
  assumes "Rhs_Nts P \<inter> Lhss Q = {}" and "Nts_syms \<alpha> \<inter> Lhss Q = {}"
  shows "P \<union> Q \<turnstile> \<alpha> \<Rightarrow>* \<beta> \<longleftrightarrow> P \<turnstile> \<alpha> \<Rightarrow>* \<beta>"
  using deriven_Un_disj_Lhss[OF assms] by (simp add: rtranclp_power)

lemma Lang_Un_disj_Lhss:
  assumes disj: "Rhs_Nts P \<inter> Lhss Q = {}" and A: "A \<notin> Lhss Q"
  shows "Lang (P \<union> Q) A = Lang P A"
  apply (rule Lang_eqI_derives)
  apply (rule derives_Un_disj_Lhss)
  using assms by auto

lemma Lang_disj_Lhss_Un:
  assumes disj: "Lhss P \<inter> Rhs_Nts Q = {}" and A: "A \<notin> Lhss P"
  shows "Lang (P \<union> Q) A = Lang Q A"
  using Lang_Un_disj_Lhss[of Q P A] assms by (simp add: ac_simps)

lemma Lang_of_Un_disj_Lhss:
  assumes "Rhs_Nts P \<inter> Lhss Q = {}" and "Nts_syms \<alpha> \<inter> Lhss Q = {}"
  shows "Lang_of (P \<union> Q) \<alpha> = Lang_of P \<alpha>"
  using derives_Un_disj_Lhss[OF assms] by (simp add: Lang_of_def)

lemma Lang_of_disj_Lhss_Un:
  assumes disj: "Lhss P \<inter> Rhs_Nts Q = {}" "Nts_syms \<alpha> \<inter> Lhss P = {}"
  shows "Lang_of (P \<union> Q) \<alpha> = Lang_of Q \<alpha>"
  using Lang_of_Un_disj_Lhss[of Q P \<alpha>] assms by (simp add: ac_simps)

lemma Lang_of_set_Un_disj_Lhss:
  assumes PQ: "Rhs_Nts P \<inter> Lhss Q = {}" and VQ: "\<Union>(Nts_syms ` V) \<inter> Lhss Q = {}"
  shows "Lang_of_set (P \<union> Q) V = Lang_of_set P V"
proof-
  { fix v assume "v \<in> V"
    with VQ have "Nts_syms v \<inter> Lhss Q = {}" by auto
    note Lang_of_Un_disj_Lhss[OF PQ this]
  }
  then show ?thesis by auto
qed

lemma Lang_of_set_disj_Lhss_Un:
  assumes disj: "Lhss P \<inter> Rhs_Nts Q = {}" "\<Union>(Nts_syms ` V) \<inter> Lhss P = {}"
  shows "Lang_of_set (P \<union> Q) V = Lang_of_set Q V"
  using Lang_of_set_Un_disj_Lhss[of Q P V] assms by (simp add: ac_simps)

subsection \<open>Substitution in Lists\<close>

text \<open>Function \<open>substs y ys xs\<close> replaces every occurrence of \<open>y\<close> in \<open>xs\<close> with the list \<open>ys\<close>\<close>

fun substs :: "'a \<Rightarrow> 'a list \<Rightarrow> 'a list \<Rightarrow> 'a list" where
"substs y ys [] = []" |
"substs y ys (x#xs) = (if x = y then ys @ substs y ys xs else x # substs y ys xs)"

text \<open>Alternative definition, but apparently no simpler to use:
@{prop "substs y ys xs = concat (map (\<lambda>x. if x=y then ys else [x]) xs)"}\<close>

abbreviation "substsNt A \<equiv> substs (Nt A)"

lemma substs_append[simp]: "substs y ys (xs @ xs') = substs y ys xs @ substs y ys xs'"
by (induction xs) auto

lemma substs_skip: "y \<notin> set xs \<Longrightarrow> substs y ys xs = xs"
by (induction xs) auto

lemma susbstsNT_map_Tm[simp]: "substsNt A \<alpha> (map Tm w) = map Tm w"
by(rule substs_skip) auto

lemma substs_len: "length (substs y [y'] xs) = length xs"
by (induction xs) auto

lemma substs_rev: "y' \<notin> set xs \<Longrightarrow> substs y' [y] (substs y [y'] xs) = xs"
by (induction xs) auto

lemma substs_der:
  "(B,v) \<in> P \<Longrightarrow> P \<turnstile> u \<Rightarrow>* substs (Nt B) v u"
proof (induction u)
  case Nil
  then show ?case by simp
next
  case (Cons a u)
  then show ?case
    by (auto simp add: derives_Cons_rule derives_prepend derives_Cons)
qed


subsection \<open>Epsilon-Freeness\<close>

text \<open>Some facts about \<open>\<epsilon>\<close>-derivations:\<close>

lemma deriven_Cons_Nil: "P \<turnstile> x # xs \<Rightarrow>(n) [] \<longleftrightarrow>
  (\<exists>A \<alpha> l m. P \<turnstile> \<alpha> \<Rightarrow>(l) [] \<and> P \<turnstile> xs \<Rightarrow>(m) [] \<and> x = Nt A \<and> (A,\<alpha>) \<in> P \<and> n = Suc (l+m))"
  using deriven_Nt_Cons_map_Tm[where w=Nil,simplified]
  by (cases x, auto simp add: deriven_Nt_Cons_map_Tm[where w=Nil,simplified]
      deriven_Tm_Cons)

lemma derives_Cons_Nil: "P \<turnstile> x # xs \<Rightarrow>* [] \<longleftrightarrow>
  (\<exists>A \<alpha>. P \<turnstile> \<alpha> \<Rightarrow>* [] \<and> P \<turnstile> xs \<Rightarrow>* [] \<and> x = Nt A \<and> (A,\<alpha>) \<in> P)"
  by (auto simp: derives_Cons_decomp)

text \<open>Adding production whose rhs does not derive \<open>\<epsilon>\<close> by other rules
does not change the \<open>\<epsilon>\<close>-derivations.\<close>

lemma insert_derives_Nil:
  assumes \<alpha>0: "\<not> P \<turnstile> \<alpha> \<Rightarrow>* []"
  shows "insert (A,\<alpha>) P \<turnstile> \<alpha>' \<Rightarrow>* [] \<longleftrightarrow> P \<turnstile> \<alpha>' \<Rightarrow>* []" (is "?l \<longleftrightarrow> ?r")
proof
  assume ?l
  then obtain n where "insert (A,\<alpha>) P \<turnstile> \<alpha>' \<Rightarrow>(n) []" by (auto simp: rtranclp_power)
  then show "P \<turnstile> \<alpha>' \<Rightarrow>* []"
  proof (induction n arbitrary: \<alpha>' rule: less_induct)
    case (less n)
    show ?case
    proof (cases \<alpha>')
      case Nil
      then show ?thesis by simp
    next
      case \<alpha>': (Cons x xs)
      from less.prems[unfolded \<alpha>' deriven_Cons_Nil]
      obtain B \<beta> l m where \<beta>: "insert (A,\<alpha>) P \<turnstile> \<beta> \<Rightarrow>(l) []"
        and xs: "insert (A,\<alpha>) P \<turnstile> xs \<Rightarrow>(m) []"
        and x: "x = Nt B"
        and B: "(B,\<beta>) \<in> insert (A,\<alpha>) P"
        and n: "n = Suc (l + m)"
        by auto
      from less.IH[OF _ \<beta>] have P\<beta>: "P \<turnstile> \<beta> \<Rightarrow>* []" by (simp add: n)
      from less.IH[OF _ xs] have Pxs: "P \<turnstile> xs \<Rightarrow>* []" by (simp add: n)
      show ?thesis
      proof (cases "(B,\<beta>) \<in> P")
        case True
        with P\<beta> Pxs show ?thesis by (auto simp: \<alpha>' x derives_Cons_Nil)
      next
        case False
        with B have "B = A" "\<beta> = \<alpha>" by auto
        with P\<beta> \<alpha>0 show ?thesis by simp
      qed
    qed
  qed
next
  assume r: ?r show "?l" by (rule derives_mono[OF _ r], auto) 
qed

definition Eps_free where "Eps_free R = (\<forall>(_,r) \<in> R. r \<noteq> [])"

abbreviation "eps_free rs == Eps_free(set rs)"

lemma Eps_freeI:
  assumes "\<And>A r. (A,r) \<in> R \<Longrightarrow> r \<noteq> []" shows "Eps_free R"
  using assms by (auto simp: Eps_free_def)

lemma Eps_free_Nil: "Eps_free R \<Longrightarrow> (A,[]) \<notin> R"
  by (auto simp: Eps_free_def)

lemma Eps_freeE_Cons: "Eps_free R \<Longrightarrow> (A,w) \<in> R \<Longrightarrow> \<exists>a u. w = a#u"
  by (cases w, auto simp: Eps_free_def)

lemma Eps_free_derives_Nil:
  assumes R: "Eps_free R" shows "R \<turnstile> l \<Rightarrow>* [] \<longleftrightarrow> l = []" (is "?l \<longleftrightarrow> ?r")
proof
  show "?l \<Longrightarrow> ?r"
  proof (induction rule: converse_derives_induct)
    case base
    show ?case by simp
  next
    case (step u A v w)
    then show ?case by (auto simp: Eps_free_Nil[OF R])
  qed
qed auto

lemma Eps_free_deriven_Nil:
  "\<lbrakk> Eps_free R; R \<turnstile> l \<Rightarrow>(n) [] \<rbrakk> \<Longrightarrow> l = []"
by (metis Eps_free_derives_Nil relpowp_imp_rtranclp)

lemma Eps_free_derivels_Nil: "Eps_free R \<Longrightarrow> R \<turnstile> l \<Rightarrow>l* [] \<longleftrightarrow> l = []"
by (meson Eps_free_derives_Nil derivels_from_empty derivels_imp_derives)

lemma Eps_free_deriveln_Nil: "Eps_free R \<Longrightarrow> R \<turnstile> l \<Rightarrow>l(n) [] \<Longrightarrow> l = []"
by (metis Eps_free_derivels_Nil relpowp_imp_rtranclp)

lemma decomp_deriveln_map_Nts:
  assumes "Eps_free P"
  shows "P \<turnstile> Nt X # map Nt Xs \<Rightarrow>l(n) map Nt Zs \<Longrightarrow>
     \<exists>Ys'. Ys' @ Xs = Zs \<and> P \<turnstile> [Nt X] \<Rightarrow>l(n) map Nt Ys'"
proof (induction n arbitrary: Zs)
  case 0
  then show ?case
    by (auto)
next
  case (Suc n)
  then obtain ys where n: "P \<turnstile> Nt X # map Nt Xs \<Rightarrow>l(n) ys" and "P \<turnstile> ys \<Rightarrow>l map Nt Zs"
    using relpowp_Suc_E by metis
  from \<open>P \<turnstile> ys \<Rightarrow>l map Nt Zs\<close> obtain Ys where "ys = map Nt Ys"
    using derivel_not_elim_Tm by blast
  from Suc.IH[of Ys] this n
  obtain Ys' where "Ys = Ys' @ Xs \<and> P \<turnstile> [Nt X] \<Rightarrow>l(n) map Nt Ys'" by auto
  moreover from \<open>ys = _\<close> \<open>P \<turnstile> ys \<Rightarrow>l map Nt Zs\<close> decomp_derivel_map_Nts[of P Ys Zs]
  obtain Y Xs' Ysa where "Ys = Y # Xs' \<and> P \<turnstile> [Nt Y] \<Rightarrow>l map Nt Ysa \<and> Zs = Ysa @ Xs'" by auto
  ultimately show ?case using Eps_free_deriveln_Nil[OF assms, of n "[Nt X]"]
    by (auto simp: Cons_eq_append_conv derivel_Nt_Cons relpowp_Suc_I)
qed

end